import { LogLevel } from './constants.js';
import { parse as parse$1 } from './cst/parse.js';
export { parse as parseCST } from './cst/parse.js';
import { Document } from './doc/Document.js';
export { Document } from './doc/Document.js';
import { YAMLSemanticError } from './errors.js';
import { warn } from './log.js';
export { defaultOptions, scalarOptions } from './options.js';
export { visit } from './visit.js';

function parseAllDocuments(src, options) {
  const stream = [];
  let prev;

  for (const cstDoc of parse$1(src)) {
    const doc = new Document(undefined, null, options);
    doc.parse(cstDoc, prev);
    stream.push(doc);
    prev = doc;
  }

  return stream;
}
function parseDocument(src, options) {
  const cst = parse$1(src);
  const doc = new Document(cst[0], null, options);

  if (cst.length > 1 && LogLevel.indexOf(doc.options.logLevel) >= LogLevel.ERROR) {
    const errMsg = 'Source contains multiple documents; please use YAML.parseAllDocuments()';
    doc.errors.unshift(new YAMLSemanticError(cst[1], errMsg));
  }

  return doc;
}
function parse(src, reviver, options) {
  if (options === undefined && reviver && typeof reviver === 'object') {
    options = reviver;
    reviver = undefined;
  }

  const doc = parseDocument(src, options);
  doc.warnings.forEach(warning => warn(doc.options.logLevel, warning));

  if (doc.errors.length > 0) {
    if (LogLevel.indexOf(doc.options.logLevel) >= LogLevel.ERROR) throw doc.errors[0];else doc.errors = [];
  }

  return doc.toJS({
    reviver
  });
}
function stringify(value, replacer, options) {
  if (typeof options === 'string') options = options.length;

  if (typeof options === 'number') {
    const indent = Math.round(options);
    options = indent < 1 ? undefined : indent > 8 ? {
      indent: 8
    } : {
      indent
    };
  }

  if (value === undefined) {
    const {
      keepUndefined
    } = options || replacer || {};
    if (!keepUndefined) return undefined;
  }

  return new Document(value, replacer, options).toString();
}

export { parse, parseAllDocuments, parseDocument, stringify };
