'use strict';

var _rollupPluginBabelHelpers = require('../_virtual/_rollupPluginBabelHelpers.js');
var constants = require('../constants.js');
var errors = require('../errors.js');
var Alias = require('./Alias.js');
var BlockValue = require('./BlockValue.js');
var Collection = require('./Collection.js');
var CollectionItem = require('./CollectionItem.js');
var FlowCollection = require('./FlowCollection.js');
var Node = require('./Node.js');
var PlainValue = require('./PlainValue.js');
var QuoteDouble = require('./QuoteDouble.js');
var QuoteSingle = require('./QuoteSingle.js');
var Range = require('./Range.js');

function createNewNode(type, props) {
  switch (type) {
    case constants.Type.ALIAS:
      return new Alias.Alias(type, props);

    case constants.Type.BLOCK_FOLDED:
    case constants.Type.BLOCK_LITERAL:
      return new BlockValue.BlockValue(type, props);

    case constants.Type.FLOW_MAP:
    case constants.Type.FLOW_SEQ:
      return new FlowCollection.FlowCollection(type, props);

    case constants.Type.MAP_KEY:
    case constants.Type.MAP_VALUE:
    case constants.Type.SEQ_ITEM:
      return new CollectionItem.CollectionItem(type, props);

    case constants.Type.COMMENT:
    case constants.Type.PLAIN:
      return new PlainValue.PlainValue(type, props);

    case constants.Type.QUOTE_DOUBLE:
      return new QuoteDouble.QuoteDouble(type, props);

    case constants.Type.QUOTE_SINGLE:
      return new QuoteSingle.QuoteSingle(type, props);

    /* istanbul ignore next */

    default:
      return null;
    // should never happen
  }
}
/**
 * @param {boolean} atLineStart - Node starts at beginning of line
 * @param {boolean} inFlow - true if currently in a flow context
 * @param {boolean} inCollection - true if currently in a collection context
 * @param {number} indent - Current level of indentation
 * @param {number} lineStart - Start of the current line
 * @param {Node} parent - The parent of the node
 * @param {string} src - Source of the YAML document
 */


class ParseContext {
  static parseType(src, offset, inFlow) {
    switch (src[offset]) {
      case '*':
        return constants.Type.ALIAS;

      case '>':
        return constants.Type.BLOCK_FOLDED;

      case '|':
        return constants.Type.BLOCK_LITERAL;

      case '{':
        return constants.Type.FLOW_MAP;

      case '[':
        return constants.Type.FLOW_SEQ;

      case '?':
        return !inFlow && Node.Node.atBlank(src, offset + 1, true) ? constants.Type.MAP_KEY : constants.Type.PLAIN;

      case ':':
        return !inFlow && Node.Node.atBlank(src, offset + 1, true) ? constants.Type.MAP_VALUE : constants.Type.PLAIN;

      case '-':
        return !inFlow && Node.Node.atBlank(src, offset + 1, true) ? constants.Type.SEQ_ITEM : constants.Type.PLAIN;

      case '"':
        return constants.Type.QUOTE_DOUBLE;

      case "'":
        return constants.Type.QUOTE_SINGLE;

      default:
        return constants.Type.PLAIN;
    }
  }

  constructor(orig = {}, {
    atLineStart,
    inCollection,
    inFlow,
    indent,
    lineStart,
    parent
  } = {}) {
    _rollupPluginBabelHelpers.defineProperty(this, "parseNode", (overlay, start) => {
      if (Node.Node.atDocumentBoundary(this.src, start)) return null;
      const context = new ParseContext(this, overlay);
      const {
        props,
        type,
        valueStart
      } = context.parseProps(start);
      const node = createNewNode(type, props);
      let offset = node.parse(context, valueStart);
      node.range = new Range.Range(start, offset);
      /* istanbul ignore if */

      if (offset <= start) {
        // This should never happen, but if it does, let's make sure to at least
        // step one character forward to avoid a busy loop.
        node.error = new Error(`Node#parse consumed no characters`);
        node.error.parseEnd = offset;
        node.error.source = node;
        node.range.end = start + 1;
      }

      if (context.nodeStartsCollection(node)) {
        if (!node.error && !context.atLineStart && context.parent.type === constants.Type.DOCUMENT) {
          node.error = new errors.YAMLSyntaxError(node, 'Block collection must not have preceding content here (e.g. directives-end indicator)');
        }

        const collection = new Collection.Collection(node);
        offset = collection.parse(new ParseContext(context), offset);
        collection.range = new Range.Range(start, offset);
        return collection;
      }

      return node;
    });

    this.atLineStart = atLineStart != null ? atLineStart : orig.atLineStart || false;
    this.inCollection = inCollection != null ? inCollection : orig.inCollection || false;
    this.inFlow = inFlow != null ? inFlow : orig.inFlow || false;
    this.indent = indent != null ? indent : orig.indent;
    this.lineStart = lineStart != null ? lineStart : orig.lineStart;
    this.parent = parent != null ? parent : orig.parent || {};
    this.root = orig.root;
    this.src = orig.src;
  }

  nodeStartsCollection(node) {
    const {
      inCollection,
      inFlow,
      src
    } = this;
    if (inCollection || inFlow) return false;
    if (node instanceof CollectionItem.CollectionItem) return true; // check for implicit key

    let offset = node.range.end;
    if (src[offset] === '\n' || src[offset - 1] === '\n') return false;
    offset = Node.Node.endOfWhiteSpace(src, offset);
    return src[offset] === ':';
  } // Anchor and tag are before type, which determines the node implementation
  // class; hence this intermediate step.


  parseProps(offset) {
    const {
      inFlow,
      parent,
      src
    } = this;
    const props = [];
    let lineHasProps = false;
    offset = this.atLineStart ? Node.Node.endOfIndent(src, offset) : Node.Node.endOfWhiteSpace(src, offset);
    let ch = src[offset];

    while (ch === constants.Char.ANCHOR || ch === constants.Char.COMMENT || ch === constants.Char.TAG || ch === '\n') {
      if (ch === '\n') {
        const lineStart = offset + 1;
        const inEnd = Node.Node.endOfIndent(src, lineStart);
        const indentDiff = inEnd - (lineStart + this.indent);
        const noIndicatorAsIndent = parent.type === constants.Type.SEQ_ITEM && parent.context.atLineStart;
        if (!Node.Node.nextNodeIsIndented(src[inEnd], indentDiff, !noIndicatorAsIndent)) break;
        this.atLineStart = true;
        this.lineStart = lineStart;
        lineHasProps = false;
        offset = inEnd;
      } else if (ch === constants.Char.COMMENT) {
        const end = Node.Node.endOfLine(src, offset + 1);
        props.push(new Range.Range(offset, end));
        offset = end;
      } else {
        let end = Node.Node.endOfIdentifier(src, offset + 1);

        if (ch === constants.Char.TAG && src[end] === ',' && /^[a-zA-Z0-9-]+\.[a-zA-Z0-9-]+,\d\d\d\d(-\d\d){0,2}\/\S/.test(src.slice(offset + 1, end + 13))) {
          // Let's presume we're dealing with a YAML 1.0 domain tag here, rather
          // than an empty but 'foo.bar' private-tagged node in a flow collection
          // followed without whitespace by a plain string starting with a year
          // or date divided by something.
          end = Node.Node.endOfIdentifier(src, end + 5);
        }

        props.push(new Range.Range(offset, end));
        lineHasProps = true;
        offset = Node.Node.endOfWhiteSpace(src, end);
      }

      ch = src[offset];
    } // '- &a : b' has an anchor on an empty node


    if (lineHasProps && ch === ':' && Node.Node.atBlank(src, offset + 1, true)) offset -= 1;
    const type = ParseContext.parseType(src, offset, inFlow);
    return {
      props,
      type,
      valueStart: offset
    };
  }
  /**
   * Parses a node from the source
   * @param {ParseContext} overlay
   * @param {number} start - Index of first non-whitespace character for the node
   * @returns {?Node} - null if at a document boundary
   */


}

exports.ParseContext = ParseContext;
