'use strict';

var constants = require('./constants.js');
var parse$1 = require('./cst/parse.js');
var Document = require('./doc/Document.js');
var errors = require('./errors.js');
var log = require('./log.js');
var options = require('./options.js');
var visit = require('./visit.js');

function parseAllDocuments(src, options) {
  const stream = [];
  let prev;

  for (const cstDoc of parse$1.parse(src)) {
    const doc = new Document.Document(undefined, null, options);
    doc.parse(cstDoc, prev);
    stream.push(doc);
    prev = doc;
  }

  return stream;
}
function parseDocument(src, options) {
  const cst = parse$1.parse(src);
  const doc = new Document.Document(cst[0], null, options);

  if (cst.length > 1 && constants.LogLevel.indexOf(doc.options.logLevel) >= constants.LogLevel.ERROR) {
    const errMsg = 'Source contains multiple documents; please use YAML.parseAllDocuments()';
    doc.errors.unshift(new errors.YAMLSemanticError(cst[1], errMsg));
  }

  return doc;
}
function parse(src, reviver, options) {
  if (options === undefined && reviver && typeof reviver === 'object') {
    options = reviver;
    reviver = undefined;
  }

  const doc = parseDocument(src, options);
  doc.warnings.forEach(warning => log.warn(doc.options.logLevel, warning));

  if (doc.errors.length > 0) {
    if (constants.LogLevel.indexOf(doc.options.logLevel) >= constants.LogLevel.ERROR) throw doc.errors[0];else doc.errors = [];
  }

  return doc.toJS({
    reviver
  });
}
function stringify(value, replacer, options) {
  if (typeof options === 'string') options = options.length;

  if (typeof options === 'number') {
    const indent = Math.round(options);
    options = indent < 1 ? undefined : indent > 8 ? {
      indent: 8
    } : {
      indent
    };
  }

  if (value === undefined) {
    const {
      keepUndefined
    } = options || replacer || {};
    if (!keepUndefined) return undefined;
  }

  return new Document.Document(value, replacer, options).toString();
}

exports.parseCST = parse$1.parse;
exports.Document = Document.Document;
exports.defaultOptions = options.defaultOptions;
exports.scalarOptions = options.scalarOptions;
exports.visit = visit.visit;
exports.parse = parse;
exports.parseAllDocuments = parseAllDocuments;
exports.parseDocument = parseDocument;
exports.stringify = stringify;
