'use strict';

var Collection = require('../ast/Collection.js');
var Scalar = require('../ast/Scalar.js');
var constants = require('../constants.js');
var errors = require('../errors.js');
var resolveMap = require('./resolveMap.js');
var resolveScalar = require('./resolveScalar.js');
var resolveSeq = require('./resolveSeq.js');

function resolveByTagName({
  knownTags,
  tags
}, tagName, value, onError) {
  const matchWithTest = [];

  for (const tag of tags) {
    if (tag.tag === tagName) {
      if (tag.test) {
        if (typeof value === 'string') matchWithTest.push(tag);else onError(`The tag ${tagName} cannot be applied to a collection`);
      } else {
        const res = tag.resolve(value, onError);
        return res instanceof Collection.Collection ? res : new Scalar.Scalar(res);
      }
    }
  }

  if (matchWithTest.length > 0) return resolveScalar.resolveScalar(value, matchWithTest);
  const kt = knownTags[tagName];

  if (kt) {
    tags.push(Object.assign({}, kt, {
      default: false,
      test: undefined
    }));
    const res = kt.resolve(value, onError);
    return res instanceof Collection.Collection ? res : new Scalar.Scalar(res);
  }

  return null;
}

function resolveTag(doc, node, tagName) {
  const {
    MAP,
    SEQ,
    STR
  } = constants.defaultTags;
  let value, fallback;

  const onError = message => doc.errors.push(new errors.YAMLSemanticError(node, message));

  try {
    switch (node.type) {
      case constants.Type.FLOW_MAP:
      case constants.Type.MAP:
        value = resolveMap.resolveMap(doc, node);
        fallback = MAP;
        if (tagName === SEQ || tagName === STR) onError(`The tag ${tagName} cannot be applied to a mapping`);
        break;

      case constants.Type.FLOW_SEQ:
      case constants.Type.SEQ:
        value = resolveSeq.resolveSeq(doc, node);
        fallback = SEQ;
        if (tagName === MAP || tagName === STR) onError(`The tag ${tagName} cannot be applied to a sequence`);
        break;

      default:
        value = node.strValue || '';

        if (typeof value !== 'string') {
          value.errors.forEach(error => doc.errors.push(error));
          value = value.str;
        }

        if (tagName === MAP || tagName === SEQ) onError(`The tag ${tagName} cannot be applied to a scalar`);
        fallback = STR;
    }

    const res = resolveByTagName(doc.schema, tagName, value, onError);

    if (res) {
      if (tagName && node.tag) res.tag = tagName;
      return res;
    }
  } catch (error) {
    /* istanbul ignore if */
    if (!error.source) error.source = node;
    doc.errors.push(error);
    return null;
  }

  try {
    if (!fallback) throw new Error(`The tag ${tagName} is unavailable`);
    const msg = `The tag ${tagName} is unavailable, falling back to ${fallback}`;
    doc.warnings.push(new errors.YAMLWarning(node, msg));
    const res = resolveByTagName(doc.schema, fallback, value, onError);
    res.tag = tagName;
    return res;
  } catch (error) {
    const refError = new errors.YAMLReferenceError(node, error.message);
    refError.stack = error.stack;
    doc.errors.push(refError);
    return null;
  }
}

exports.resolveTag = resolveTag;
