'use strict';

var Alias = require('./ast/Alias.js');
var Node = require('./ast/Node.js');
var Pair = require('./ast/Pair.js');
var Scalar = require('./ast/Scalar.js');
var YAMLMap = require('./ast/YAMLMap.js');
var YAMLSeq = require('./ast/YAMLSeq.js');
var Document = require('./doc/Document.js');

const BREAK = Symbol('break visit');
const SKIP = Symbol('skip children');
const REMOVE = Symbol('remove node');

function _visit(key, node, visitor, path) {
  let ctrl = undefined;
  if (typeof visitor === 'function') ctrl = visitor(key, node, path);else if (node instanceof YAMLMap.YAMLMap) {
    if (visitor.Map) ctrl = visitor.Map(key, node, path);
  } else if (node instanceof YAMLSeq.YAMLSeq) {
    if (visitor.Seq) ctrl = visitor.Seq(key, node, path);
  } else if (node instanceof Pair.Pair) {
    if (visitor.Pair) ctrl = visitor.Pair(key, node, path);
  } else if (node instanceof Scalar.Scalar) {
    if (visitor.Scalar) ctrl = visitor.Scalar(key, node, path);
  } else if (node instanceof Alias.Alias) {
    if (visitor.Scalar) ctrl = visitor.Alias(key, node, path);
  }

  if (ctrl instanceof Node.Node) {
    const parent = path[path.length - 1];

    if (parent instanceof YAMLMap.YAMLMap || parent instanceof YAMLSeq.YAMLSeq) {
      parent.items[key] = ctrl;
    } else if (parent instanceof Pair.Pair) {
      if (key === 'key') parent.key = ctrl;else parent.value = ctrl;
    } else if (parent instanceof Document.Document) {
      parent.contents = ctrl;
    } else {
      const pt = parent && parent.type;
      throw new Error(`Cannot replace node with ${pt} parent`);
    }

    return _visit(key, ctrl, visitor, path);
  }

  if (typeof ctrl !== 'symbol') {
    if (node instanceof YAMLMap.YAMLMap || node instanceof YAMLSeq.YAMLSeq) {
      path = Object.freeze(path.concat(node));

      for (let i = 0; i < node.items.length; ++i) {
        const ci = _visit(i, node.items[i], visitor, path);

        if (typeof ci === 'number') i = ci - 1;else if (ci === BREAK) return BREAK;else if (ci === REMOVE) {
          node.items.splice(i, 1);
          i -= 1;
        }
      }
    } else if (node instanceof Pair.Pair) {
      path = Object.freeze(path.concat(node));

      const ck = _visit('key', node.key, visitor, path);

      if (ck === BREAK) return BREAK;else if (ck === REMOVE) node.key = null;

      const cv = _visit('value', node.value, visitor, path);

      if (cv === BREAK) return BREAK;else if (cv === REMOVE) node.value = null;
    }
  }

  return ctrl;
}

function visit(node, visitor) {
  if (node instanceof Document.Document) {
    const cd = _visit(null, node.contents, visitor, Object.freeze([node]));

    if (cd === REMOVE) node.contents = null;
  } else _visit(null, node, visitor, Object.freeze([]));
}
visit.BREAK = BREAK;
visit.SKIP = SKIP;
visit.REMOVE = REMOVE;

exports.visit = visit;
