import { isNode, isMap } from '../nodes/Node.js';
import { Scalar } from '../nodes/Scalar.js';
import { resolveBlockMap } from './resolve-block-map.js';
import { resolveBlockSeq } from './resolve-block-seq.js';
import { resolveFlowCollection } from './resolve-flow-collection.js';

function composeCollection(CN, doc, token, anchor, tagName, onError) {
    let coll;
    switch (token.type) {
        case 'block-map': {
            coll = resolveBlockMap(CN, doc, token, anchor, onError);
            break;
        }
        case 'block-seq': {
            coll = resolveBlockSeq(CN, doc, token, anchor, onError);
            break;
        }
        case 'flow-collection': {
            coll = resolveFlowCollection(CN, doc, token, anchor, onError);
            break;
        }
    }
    if (!tagName)
        return coll;
    // Cast needed due to: https://github.com/Microsoft/TypeScript/issues/3841
    const Coll = coll.constructor;
    if (tagName === '!' || tagName === Coll.tagName) {
        coll.tag = Coll.tagName;
        return coll;
    }
    const expType = isMap(coll) ? 'map' : 'seq';
    let tag = doc.schema.tags.find(t => t.collection === expType && t.tag === tagName);
    if (!tag) {
        const kt = doc.schema.knownTags[tagName];
        if (kt && kt.collection === expType) {
            doc.schema.tags.push(Object.assign({}, kt, { default: false }));
            tag = kt;
        }
        else {
            onError(coll.range[0], `Unresolved tag: ${tagName}`, true);
            coll.tag = tagName;
            return coll;
        }
    }
    const res = tag.resolve(coll, msg => onError(coll.range[0], msg), doc.options);
    const node = isNode(res)
        ? res
        : new Scalar(res);
    node.range = coll.range;
    node.tag = tagName;
    if (tag === null || tag === void 0 ? void 0 : tag.format)
        node.format = tag.format;
    return node;
}

export { composeCollection };
