import { Directives } from '../doc/directives.js';
import { Document } from '../doc/Document.js';
import { YAMLWarning, YAMLParseError } from '../errors.js';
import { isCollection } from '../nodes/Node.js';
import { defaultOptions } from '../options.js';
import { composeDoc } from './compose-doc.js';
import { resolveEnd } from './resolve-end.js';

function parsePrelude(prelude) {
    let comment = '';
    let atComment = false;
    let afterEmptyLine = false;
    for (let i = 0; i < prelude.length; ++i) {
        const source = prelude[i];
        switch (source[0]) {
            case '#':
                comment +=
                    (comment === '' ? '' : afterEmptyLine ? '\n\n' : '\n') +
                        source.substring(1);
                atComment = true;
                afterEmptyLine = false;
                break;
            case '%':
                if (prelude[i + 1][0] !== '#')
                    i += 1;
                atComment = false;
                break;
            default:
                // This may be wrong after doc-end, but in that case it doesn't matter
                if (!atComment)
                    afterEmptyLine = true;
                atComment = false;
        }
    }
    return { comment, afterEmptyLine };
}
/**
 * Compose a stream of CST nodes into a stream of YAML Documents.
 *
 * ```ts
 * const options: Options = { ... }
 * const docs: Document.Parsed[] = []
 * const composer = new Composer(doc => docs.push(doc), options)
 * const parser = new Parser(composer.next)
 * parser.parse(source)
 * composer.end()
 * ```
 */
class Composer {
    constructor(onDocument, options = {}) {
        this.doc = null;
        this.atDirectives = false;
        this.prelude = [];
        this.errors = [];
        this.warnings = [];
        this.onError = (offset, message, warning) => {
            if (warning)
                this.warnings.push(new YAMLWarning(offset, message));
            else
                this.errors.push(new YAMLParseError(offset, message));
        };
        /**
         * Advance the composed by one CST token. Bound to the Composer
         * instance, so may be used directly as a callback function.
         */
        this.next = (token) => {
            switch (token.type) {
                case 'directive':
                    this.directives.add(token.source, this.onError);
                    this.prelude.push(token.source);
                    this.atDirectives = true;
                    break;
                case 'document': {
                    const doc = composeDoc(this.options, this.directives, token, this.onError);
                    this.decorate(doc, false);
                    if (this.doc)
                        this.onDocument(this.doc);
                    this.doc = doc;
                    this.atDirectives = false;
                    break;
                }
                case 'byte-order-mark':
                case 'space':
                    break;
                case 'comment':
                case 'newline':
                    this.prelude.push(token.source);
                    break;
                case 'error': {
                    const msg = token.source
                        ? `${token.message}: ${JSON.stringify(token.source)}`
                        : token.message;
                    const error = new YAMLParseError(-1, msg);
                    if (this.atDirectives || !this.doc)
                        this.errors.push(error);
                    else
                        this.doc.errors.push(error);
                    break;
                }
                case 'doc-end': {
                    if (!this.doc) {
                        const msg = 'Unexpected doc-end without preceding document';
                        this.errors.push(new YAMLParseError(token.offset, msg));
                        break;
                    }
                    const end = resolveEnd(token.end, token.offset + token.source.length, this.doc.options.strict, this.onError);
                    this.decorate(this.doc, true);
                    if (end.comment) {
                        const dc = this.doc.comment;
                        this.doc.comment = dc ? `${dc}\n${end.comment}` : end.comment;
                    }
                    this.doc.range[1] = end.offset;
                    break;
                }
                default:
                    this.errors.push(new YAMLParseError(-1, `Unsupported token ${token.type}`));
            }
        };
        this.directives = new Directives({
            version: (options === null || options === void 0 ? void 0 : options.version) || defaultOptions.version
        });
        this.onDocument = onDocument;
        this.options = options;
    }
    decorate(doc, afterDoc) {
        const { comment, afterEmptyLine } = parsePrelude(this.prelude);
        //console.log({ dc: doc.comment, prelude, comment })
        if (comment) {
            const dc = doc.contents;
            if (afterDoc) {
                doc.comment = doc.comment ? `${doc.comment}\n${comment}` : comment;
            }
            else if (afterEmptyLine || doc.directives.marker || !dc) {
                doc.commentBefore = comment;
            }
            else if (isCollection(dc) && !dc.flow && dc.items.length > 0) {
                const it = dc.items[0];
                const cb = it.commentBefore;
                it.commentBefore = cb ? `${comment}\n${cb}` : comment;
            }
            else {
                const cb = dc.commentBefore;
                dc.commentBefore = cb ? `${comment}\n${cb}` : comment;
            }
        }
        if (afterDoc) {
            Array.prototype.push.apply(doc.errors, this.errors);
            Array.prototype.push.apply(doc.warnings, this.warnings);
        }
        else {
            doc.errors = this.errors;
            doc.warnings = this.warnings;
        }
        this.prelude = [];
        this.errors = [];
        this.warnings = [];
    }
    /**
     * Current stream status information.
     *
     * Mostly useful at the end of input for an empty stream.
     */
    streamInfo() {
        return {
            comment: parsePrelude(this.prelude).comment,
            directives: this.directives,
            errors: this.errors,
            warnings: this.warnings
        };
    }
    end(forceDoc = false, offset = -1) {
        if (this.doc) {
            this.decorate(this.doc, true);
            this.onDocument(this.doc);
            this.doc = null;
        }
        else if (forceDoc) {
            const opts = Object.assign({ directives: this.directives }, this.options);
            const doc = new Document(undefined, opts);
            if (this.atDirectives)
                this.onError(offset, 'Missing directives-end indicator line');
            doc.range = [0, offset];
            this.decorate(doc, false);
            this.onDocument(doc);
        }
    }
}

export { Composer };
