'use strict';

var Node = require('../nodes/Node.js');
var Scalar = require('../nodes/Scalar.js');
var resolveBlockMap = require('./resolve-block-map.js');
var resolveBlockSeq = require('./resolve-block-seq.js');
var resolveFlowCollection = require('./resolve-flow-collection.js');

function composeCollection(CN, doc, token, anchor, tagName, onError) {
    let coll;
    switch (token.type) {
        case 'block-map': {
            coll = resolveBlockMap.resolveBlockMap(CN, doc, token, anchor, onError);
            break;
        }
        case 'block-seq': {
            coll = resolveBlockSeq.resolveBlockSeq(CN, doc, token, anchor, onError);
            break;
        }
        case 'flow-collection': {
            coll = resolveFlowCollection.resolveFlowCollection(CN, doc, token, anchor, onError);
            break;
        }
    }
    if (!tagName)
        return coll;
    // Cast needed due to: https://github.com/Microsoft/TypeScript/issues/3841
    const Coll = coll.constructor;
    if (tagName === '!' || tagName === Coll.tagName) {
        coll.tag = Coll.tagName;
        return coll;
    }
    const expType = Node.isMap(coll) ? 'map' : 'seq';
    let tag = doc.schema.tags.find(t => t.collection === expType && t.tag === tagName);
    if (!tag) {
        const kt = doc.schema.knownTags[tagName];
        if (kt && kt.collection === expType) {
            doc.schema.tags.push(Object.assign({}, kt, { default: false }));
            tag = kt;
        }
        else {
            onError(coll.range[0], `Unresolved tag: ${tagName}`, true);
            coll.tag = tagName;
            return coll;
        }
    }
    const res = tag.resolve(coll, msg => onError(coll.range[0], msg), doc.options);
    const node = Node.isNode(res)
        ? res
        : new Scalar.Scalar(res);
    node.range = coll.range;
    node.tag = tagName;
    if (tag === null || tag === void 0 ? void 0 : tag.format)
        node.format = tag.format;
    return node;
}

exports.composeCollection = composeCollection;
