'use strict';

var Alias = require('../nodes/Alias.js');
var composeCollection = require('./compose-collection.js');
var composeScalar = require('./compose-scalar.js');
var resolveEnd = require('./resolve-end.js');
var utilEmptyScalarPosition = require('./util-empty-scalar-position.js');

const CN = { composeNode, composeEmptyNode };
function composeNode(doc, token, props, onError) {
    const { spaceBefore, comment, anchor, tagName } = props;
    let node;
    switch (token.type) {
        case 'alias':
            node = composeAlias(doc, token, onError);
            if (anchor || tagName)
                onError(token.offset, 'An alias node must not specify any properties');
            break;
        case 'scalar':
        case 'single-quoted-scalar':
        case 'double-quoted-scalar':
        case 'block-scalar':
            node = composeScalar.composeScalar(doc, token, anchor, tagName, onError);
            break;
        case 'block-map':
        case 'block-seq':
        case 'flow-collection':
            node = composeCollection.composeCollection(CN, doc, token, anchor, tagName, onError);
            break;
        default:
            console.log(token);
            throw new Error(`Unsupporten token type: ${token.type}`);
    }
    if (spaceBefore)
        node.spaceBefore = true;
    if (comment) {
        if (token.type === 'scalar' && token.source === '')
            node.comment = comment;
        else
            node.commentBefore = comment;
    }
    return node;
}
function composeEmptyNode(doc, offset, before, pos, { spaceBefore, comment, anchor, tagName }, onError) {
    const token = {
        type: 'scalar',
        offset: utilEmptyScalarPosition.emptyScalarPosition(offset, before, pos),
        indent: -1,
        source: ''
    };
    const node = composeScalar.composeScalar(doc, token, anchor, tagName, onError);
    if (spaceBefore)
        node.spaceBefore = true;
    if (comment)
        node.comment = comment;
    return node;
}
function composeAlias(doc, { offset, source, end }, onError) {
    const name = source.substring(1);
    const src = doc.anchors.getNode(name);
    if (!src)
        onError(offset, `Aliased anchor not found: ${name}`);
    const alias = new Alias.Alias(src);
    const re = resolveEnd.resolveEnd(end, offset + source.length, doc.options.strict, onError);
    alias.range = [offset, re.offset];
    if (re.comment)
        alias.comment = re.comment;
    return alias;
}

exports.composeEmptyNode = composeEmptyNode;
exports.composeNode = composeNode;
