'use strict';

var Node = require('../nodes/Node.js');
var Scalar = require('../nodes/Scalar.js');
var resolveBlockScalar = require('./resolve-block-scalar.js');
var resolveFlowScalar = require('./resolve-flow-scalar.js');

function composeScalar(doc, token, anchor, tagName, onError) {
    const { offset } = token;
    const { value, type, comment, length } = token.type === 'block-scalar'
        ? resolveBlockScalar.resolveBlockScalar(token, doc.options.strict, onError)
        : resolveFlowScalar.resolveFlowScalar(token, doc.options.strict, onError);
    const tag = tagName
        ? findScalarTagByName(doc.schema, value, tagName, onError)
        : findScalarTagByTest(doc.schema, value, token.type === 'scalar');
    let scalar;
    try {
        const res = tag
            ? tag.resolve(value, msg => onError(offset, msg), doc.options)
            : value;
        scalar = Node.isScalar(res) ? res : new Scalar.Scalar(res);
    }
    catch (error) {
        onError(offset, error.message);
        scalar = new Scalar.Scalar(value);
    }
    scalar.range = [offset, offset + length];
    scalar.source = value;
    if (type)
        scalar.type = type;
    if (tagName)
        scalar.tag = tagName;
    if (tag === null || tag === void 0 ? void 0 : tag.format)
        scalar.format = tag.format;
    if (comment)
        scalar.comment = comment;
    if (anchor)
        doc.anchors.setAnchor(scalar, anchor);
    return scalar;
}
const defaultScalarTag = (schema) => schema.tags.find(tag => !tag.collection && tag.tag === 'tag:yaml.org,2002:str');
function findScalarTagByName(schema, value, tagName, onError) {
    var _a;
    if (tagName === '!')
        return defaultScalarTag(schema); // non-specific tag
    const matchWithTest = [];
    for (const tag of schema.tags) {
        if (!tag.collection && tag.tag === tagName) {
            if (tag.default && tag.test)
                matchWithTest.push(tag);
            else
                return tag;
        }
    }
    for (const tag of matchWithTest)
        if ((_a = tag.test) === null || _a === void 0 ? void 0 : _a.test(value))
            return tag;
    const kt = schema.knownTags[tagName];
    if (kt && !kt.collection) {
        // Ensure that the known tag is available for stringifying,
        // but does not get used by default.
        schema.tags.push(Object.assign({}, kt, { default: false, test: undefined }));
        return kt;
    }
    onError(0, `Unresolved tag: ${tagName}`, tagName !== 'tag:yaml.org,2002:str');
    return defaultScalarTag(schema);
}
function findScalarTagByTest(schema, value, apply) {
    var _a;
    if (apply) {
        for (const tag of schema.tags) {
            if (tag.default && ((_a = tag.test) === null || _a === void 0 ? void 0 : _a.test(value)))
                return tag;
        }
    }
    return defaultScalarTag(schema);
}

exports.composeScalar = composeScalar;
