'use strict';

var Pair = require('../nodes/Pair.js');
var YAMLMap = require('../nodes/YAMLMap.js');
var resolveProps = require('./resolve-props.js');
var utilContainsNewline = require('./util-contains-newline.js');

const startColMsg = 'All mapping items must start at the same column';
function resolveBlockMap({ composeNode, composeEmptyNode }, doc, { indent, items, offset }, anchor, onError) {
    var _a;
    const start = offset;
    const map = new YAMLMap.YAMLMap(doc.schema);
    if (anchor)
        doc.anchors.setAnchor(map, anchor);
    for (const { start, key, sep, value } of items) {
        // key properties
        const keyProps = resolveProps.resolveProps(doc, start, true, 'explicit-key-ind', offset, onError);
        const implicitKey = !keyProps.found;
        if (implicitKey) {
            if (key) {
                if (key.type === 'block-seq')
                    onError(offset, 'A block sequence may not be used as an implicit map key');
                else if ('indent' in key && key.indent !== indent)
                    onError(offset, startColMsg);
            }
            if (!keyProps.anchor && !keyProps.tagName && !sep) {
                // TODO: assert being at last item?
                if (keyProps.comment) {
                    if (map.comment)
                        map.comment += '\n' + keyProps.comment;
                    else
                        map.comment = keyProps.comment;
                }
                continue;
            }
        }
        else if (((_a = keyProps.found) === null || _a === void 0 ? void 0 : _a.indent) !== indent)
            onError(offset, startColMsg);
        offset += keyProps.length;
        if (implicitKey && utilContainsNewline.containsNewline(key))
            onError(offset, 'Implicit keys need to be on a single line');
        // key value
        const keyStart = offset;
        const keyNode = key
            ? composeNode(doc, key, keyProps, onError)
            : composeEmptyNode(doc, offset, start, null, keyProps, onError);
        offset = keyNode.range[1];
        // value properties
        const valueProps = resolveProps.resolveProps(doc, sep || [], !key || key.type === 'block-scalar', 'map-value-ind', offset, onError);
        offset += valueProps.length;
        if (valueProps.found) {
            if (implicitKey) {
                if ((value === null || value === void 0 ? void 0 : value.type) === 'block-map' && !valueProps.hasNewline)
                    onError(offset, 'Nested mappings are not allowed in compact mappings');
                if (doc.options.strict &&
                    keyProps.start < valueProps.found.offset - 1024)
                    onError(offset, 'The : indicator must be at most 1024 chars after the start of an implicit block mapping key');
            }
            // value value
            const valueNode = value
                ? composeNode(doc, value, valueProps, onError)
                : composeEmptyNode(doc, offset, sep, null, valueProps, onError);
            offset = valueNode.range[1];
            map.items.push(new Pair.Pair(keyNode, valueNode));
        }
        else {
            // key with no value
            if (implicitKey)
                onError(keyStart, 'Implicit map keys need to be followed by map values');
            if (valueProps.comment) {
                if (keyNode.comment)
                    keyNode.comment += '\n' + valueProps.comment;
                else
                    keyNode.comment = valueProps.comment;
            }
            map.items.push(new Pair.Pair(keyNode));
        }
    }
    map.range = [start, offset];
    return map;
}

exports.resolveBlockMap = resolveBlockMap;
