import type { YAMLError, YAMLWarning } from '../errors.js';
import { Node, NODE_TYPE, ParsedNode } from '../nodes/Node.js';
import { Pair } from '../nodes/Pair.js';
import { CreateNodeOptions, DocumentOptions, Options, ParseOptions, SchemaOptions, ToJSOptions, ToStringOptions } from '../options.js';
import { Anchors } from './Anchors.js';
import { Schema } from './Schema.js';
import { Directives } from './directives.js';
export declare type Replacer = any[] | ((key: any, value: any) => unknown);
export declare namespace Document {
    interface Parsed<T extends ParsedNode = ParsedNode> extends Document<T> {
        range: [number, number];
        /** The schema used with the document. */
        schema: Schema;
    }
}
export declare class Document<T = unknown> {
    readonly [NODE_TYPE]: symbol;
    /**
     * Anchors associated with the document's nodes;
     * also provides alias & merge node creators.
     */
    anchors: Anchors;
    /** A comment before this Document */
    commentBefore: string | null;
    /** A comment immediately after this Document */
    comment: string | null;
    /** The document contents. */
    contents: T | null;
    directives: Directives;
    /** Errors encountered during parsing. */
    errors: YAMLError[];
    options: Required<Omit<ParseOptions & DocumentOptions, 'lineCounter' | 'directives' | 'version'>>;
    /** The schema used with the document. Use `setSchema()` to change. */
    schema: Schema;
    /** Warnings encountered during parsing. */
    warnings: YAMLWarning[];
    /**
     * @param value - The initial value for the document, which will be wrapped
     *   in a Node container.
     */
    constructor(value?: any, options?: Options);
    constructor(value: any, replacer: null | Replacer, options?: Options);
    /** Adds a value to the document. */
    add(value: any): void;
    /** Adds a value to the document. */
    addIn(path: Iterable<unknown>, value: unknown): void;
    /**
     * Convert any value into a `Node` using the current schema, recursively
     * turning objects into collections.
     */
    createNode(value: unknown, { flow, keepUndefined, onTagObj, replacer, tag }?: CreateNodeOptions): Node;
    /**
     * Convert a key and a value into a `Pair` using the current schema,
     * recursively wrapping all values as `Scalar` or `Collection` nodes.
     */
    createPair<K extends Node = Node, V extends Node = Node>(key: unknown, value: unknown, options?: CreateNodeOptions): Pair<K, V>;
    /**
     * Removes a value from the document.
     * @returns `true` if the item was found and removed.
     */
    delete(key: any): boolean;
    /**
     * Removes a value from the document.
     * @returns `true` if the item was found and removed.
     */
    deleteIn(path: Iterable<unknown>): boolean;
    /**
     * Returns item at `key`, or `undefined` if not found. By default unwraps
     * scalar values from their surrounding node; to disable set `keepScalar` to
     * `true` (collections are always returned intact).
     */
    get(key: unknown, keepScalar?: boolean): unknown;
    /**
     * Returns item at `path`, or `undefined` if not found. By default unwraps
     * scalar values from their surrounding node; to disable set `keepScalar` to
     * `true` (collections are always returned intact).
     */
    getIn(path: Iterable<unknown>, keepScalar?: boolean): unknown;
    /**
     * Checks if the document includes a value with the key `key`.
     */
    has(key: unknown): boolean;
    /**
     * Checks if the document includes a value at `path`.
     */
    hasIn(path: Iterable<unknown>): boolean;
    /**
     * Sets a value in this document. For `!!set`, `value` needs to be a
     * boolean to add/remove the item from the set.
     */
    set(key: any, value: unknown): void;
    /**
     * Sets a value in this document. For `!!set`, `value` needs to be a
     * boolean to add/remove the item from the set.
     */
    setIn(path: Iterable<unknown>, value: unknown): void;
    /**
     * Change the YAML version and schema used by the document.
     *
     * Overrides all previously set schema options
     */
    setSchema(version: '1.1' | '1.2', options?: SchemaOptions): void;
    /** A plain JavaScript representation of the document `contents`. */
    toJS(opt?: ToJSOptions & {
        [ignored: string]: unknown;
    }): any;
    /**
     * A JSON representation of the document `contents`.
     *
     * @param jsonArg Used by `JSON.stringify` to indicate the array index or
     *   property name.
     */
    toJSON(jsonArg?: string | null, onAnchor?: ToJSOptions['onAnchor']): any;
    /** A YAML representation of the document. */
    toString(options?: ToStringOptions): string;
}
