'use strict';

var createNode = require('../doc/createNode.js');
var log = require('../log.js');
var addComment = require('../stringify/addComment.js');
var stringify = require('../stringify/stringify.js');
var Scalar = require('./Scalar.js');
var toJS = require('./toJS.js');
var Node = require('./Node.js');

function createPair(key, value, ctx) {
    const k = createNode.createNode(key, undefined, ctx);
    const v = createNode.createNode(value, undefined, ctx);
    return new Pair(k, v);
}
const isMergeKey = (key) => key === Pair.MERGE_KEY ||
    (Node.isScalar(key) &&
        key.value === Pair.MERGE_KEY &&
        (!key.type || key.type === Scalar.Scalar.PLAIN));
// If the value associated with a merge key is a single mapping node, each of
// its key/value pairs is inserted into the current mapping, unless the key
// already exists in it. If the value associated with the merge key is a
// sequence, then this sequence is expected to contain mapping nodes and each
// of these nodes is merged in turn according to its order in the sequence.
// Keys in mapping nodes earlier in the sequence override keys specified in
// later mapping nodes. -- http://yaml.org/type/merge.html
function mergeToJSMap(ctx, map, value) {
    if (!Node.isAlias(value) || !Node.isMap(value.source))
        throw new Error('Merge sources must be map aliases');
    const srcMap = value.source.toJSON(null, ctx, Map);
    for (const [key, value] of srcMap) {
        if (map instanceof Map) {
            if (!map.has(key))
                map.set(key, value);
        }
        else if (map instanceof Set) {
            map.add(key);
        }
        else if (!Object.prototype.hasOwnProperty.call(map, key)) {
            Object.defineProperty(map, key, {
                value,
                writable: true,
                enumerable: true,
                configurable: true
            });
        }
    }
    return map;
}
class Pair extends Node.NodeBase {
    constructor(key, value = null) {
        super(Node.PAIR);
        this.key = key;
        this.value = value;
    }
    // @ts-ignore This is fine.
    get commentBefore() {
        return Node.isNode(this.key) ? this.key.commentBefore : undefined;
    }
    set commentBefore(cb) {
        if (this.key == null)
            this.key = new Scalar.Scalar(null); // FIXME
        if (Node.isNode(this.key))
            this.key.commentBefore = cb;
        else {
            const msg = 'Pair.commentBefore is an alias for Pair.key.commentBefore. To set it, the key must be a Node.';
            throw new Error(msg);
        }
    }
    // @ts-ignore This is fine.
    get spaceBefore() {
        return Node.isNode(this.key) ? this.key.spaceBefore : undefined;
    }
    set spaceBefore(sb) {
        if (this.key == null)
            this.key = new Scalar.Scalar(null); // FIXME
        if (Node.isNode(this.key))
            this.key.spaceBefore = sb;
        else {
            const msg = 'Pair.spaceBefore is an alias for Pair.key.spaceBefore. To set it, the key must be a Node.';
            throw new Error(msg);
        }
    }
    addToJSMap(ctx, map) {
        if (ctx && ctx.doc.schema.merge && isMergeKey(this.key)) {
            if (Node.isSeq(this.value))
                for (const it of this.value.items)
                    mergeToJSMap(ctx, map, it);
            else if (Array.isArray(this.value))
                for (const it of this.value)
                    mergeToJSMap(ctx, map, it);
            else
                mergeToJSMap(ctx, map, this.value);
        }
        else {
            const key = toJS.toJS(this.key, '', ctx);
            if (map instanceof Map) {
                const value = toJS.toJS(this.value, key, ctx);
                map.set(key, value);
            }
            else if (map instanceof Set) {
                map.add(key);
            }
            else {
                const stringKey = stringifyKey(this.key, key, ctx);
                const value = toJS.toJS(this.value, stringKey, ctx);
                if (stringKey in map)
                    Object.defineProperty(map, stringKey, {
                        value,
                        writable: true,
                        enumerable: true,
                        configurable: true
                    });
                else
                    map[stringKey] = value;
            }
        }
        return map;
    }
    toJSON(_, ctx) {
        const pair = ctx && ctx.mapAsMap ? new Map() : {};
        return this.addToJSMap(ctx, pair);
    }
    toString(ctx, onComment, onChompKeep) {
        if (!ctx || !ctx.doc)
            return JSON.stringify(this);
        const { allNullValues, doc, indent, indentStep, options: { indentSeq, simpleKeys } } = ctx;
        let { key, value } = this;
        let keyComment = (Node.isNode(key) && key.comment) || null;
        if (simpleKeys) {
            if (keyComment) {
                throw new Error('With simple keys, key nodes cannot have comments');
            }
            if (Node.isCollection(key)) {
                const msg = 'With simple keys, collection cannot be used as a key value';
                throw new Error(msg);
            }
        }
        let explicitKey = !simpleKeys &&
            (!key ||
                (keyComment && value == null) ||
                Node.isCollection(key) ||
                (Node.isScalar(key)
                    ? key.type === Scalar.Scalar.BLOCK_FOLDED || key.type === Scalar.Scalar.BLOCK_LITERAL
                    : typeof key === 'object'));
        ctx = Object.assign({}, ctx, {
            allNullValues: false,
            implicitKey: !explicitKey && (simpleKeys || !allNullValues),
            indent: indent + indentStep
        });
        let chompKeep = false;
        let str = stringify.stringify(key, ctx, () => (keyComment = null), () => (chompKeep = true));
        if (!explicitKey && !ctx.inFlow && str.length > 1024) {
            if (simpleKeys)
                throw new Error('With simple keys, single line scalar must not span more than 1024 characters');
            explicitKey = true;
        }
        if ((allNullValues && (!simpleKeys || ctx.inFlow)) ||
            (value == null && (explicitKey || ctx.inFlow))) {
            str = addComment.addComment(str, ctx.indent, keyComment);
            if (this.comment) {
                if (keyComment && !this.comment.includes('\n'))
                    str += `\n${ctx.indent || ''}#${this.comment}`;
                else
                    str = addComment.addComment(str, ctx.indent, this.comment);
                if (onComment)
                    onComment();
            }
            else if (chompKeep && !keyComment && onChompKeep)
                onChompKeep();
            return ctx.inFlow && !explicitKey ? str : `? ${str}`;
        }
        str = explicitKey
            ? `? ${addComment.addComment(str, ctx.indent, keyComment)}\n${indent}:`
            : addComment.addComment(`${str}:`, ctx.indent, keyComment);
        if (this.comment) {
            if (keyComment && !explicitKey && !this.comment.includes('\n'))
                str += `\n${ctx.indent || ''}#${this.comment}`;
            else
                str = addComment.addComment(str, ctx.indent, this.comment);
            if (onComment)
                onComment();
        }
        let vcb = '';
        let valueComment = null;
        if (Node.isNode(value)) {
            if (value.spaceBefore)
                vcb = '\n';
            if (value.commentBefore) {
                const cs = value.commentBefore.replace(/^/gm, `${ctx.indent}#`);
                vcb += `\n${cs}`;
            }
            valueComment = value.comment;
        }
        else if (value && typeof value === 'object') {
            value = doc.createNode(value);
        }
        ctx.implicitKey = false;
        if (!explicitKey && !keyComment && !this.comment && Node.isScalar(value))
            ctx.indentAtStart = str.length + 1;
        chompKeep = false;
        if (!indentSeq &&
            indentStep.length >= 2 &&
            !ctx.inFlow &&
            !explicitKey &&
            Node.isSeq(value) &&
            !value.flow &&
            !value.tag &&
            !doc.anchors.getName(value)) {
            // If indentSeq === false, consider '- ' as part of indentation where possible
            ctx.indent = ctx.indent.substr(2);
        }
        const valueStr = stringify.stringify(value, ctx, () => (valueComment = null), () => (chompKeep = true));
        let ws = ' ';
        if (vcb || keyComment || this.comment) {
            ws = `${vcb}\n${ctx.indent}`;
        }
        else if (!explicitKey && Node.isCollection(value)) {
            const flow = valueStr[0] === '[' || valueStr[0] === '{';
            if (!flow || valueStr.includes('\n'))
                ws = `\n${ctx.indent}`;
        }
        else if (valueStr[0] === '\n')
            ws = '';
        if (chompKeep && !valueComment && onChompKeep)
            onChompKeep();
        return addComment.addComment(str + ws + valueStr, ctx.indent, valueComment);
    }
}
Pair.MERGE_KEY = '<<';
function stringifyKey(key, jsKey, ctx) {
    if (jsKey === null)
        return '';
    if (typeof jsKey !== 'object')
        return String(jsKey);
    if (Node.isNode(key) && ctx && ctx.doc) {
        const strCtx = stringify.createStringifyContext(ctx.doc, {});
        strCtx.inFlow = true;
        strCtx.inStringifyKey = true;
        const strKey = key.toString(strCtx);
        if (!ctx.mapKeyWarned) {
            let jsonStr = JSON.stringify(strKey);
            if (jsonStr.length > 40)
                jsonStr = jsonStr.substring(0, 36) + '..."';
            log.warn(ctx.doc.options.logLevel, `Keys with collection values will be stringified due to JS Object restrictions: ${jsonStr}. Set mapAsMap: true to use object keys.`);
            ctx.mapKeyWarned = true;
        }
        return strKey;
    }
    return JSON.stringify(jsKey);
}

exports.Pair = Pair;
exports.createPair = createPair;
