import { Token } from './tokens.js';
/**
 * A YAML concrete syntax tree (CST) parser
 *
 * While the `parse()` method provides an API for parsing a source string
 * directly, the parser may also be used with a user-provided lexer:
 *
 * ```ts
 * const cst: Token[] = []
 * const parser = new Parser(tok => cst.push(tok))
 * const src: string = ...
 *
 * // The following would be equivalent to `parser.parse(src, false)`
 * const lexer = new Lexer(parser.next)
 * lexer.lex(src, false)
 * parser.end()
 * ```
 */
export declare class Parser {
    private push;
    private onNewLine?;
    /** If true, space and sequence indicators count as indentation */
    private atNewLine;
    /** If true, next token is a scalar value */
    private atScalar;
    /** Current indentation level */
    private indent;
    /** Current offset since the start of parsing */
    offset: number;
    /** On the same line with a block map key */
    private onKeyLine;
    /** Top indicates the node that's currently being built */
    stack: Token[];
    /** The source of the current token, set in parse() */
    private source;
    /** The type of the current token, set in parse() */
    private type;
    /**
     * @param push - Called separately with each parsed token
     * @param onNewLine - If defined, called separately with the start position of
     *   each new line (in `parse()`, including the start of input).
     * @public
     */
    constructor(push: (token: Token) => void, onNewLine?: (offset: number) => void);
    /**
     * Parse `source` as a YAML stream, calling `push` with each directive,
     * document and other structure as it is completely parsed. If `incomplete`,
     * a part of the last line may be left as a buffer for the next call.
     *
     * Errors are not thrown, but pushed out as `{ type: 'error', message }` tokens.
     * @public
     */
    parse(source: string, incomplete?: boolean): void;
    /**
     * Advance the parser by the `source` of one lexical token. Bound to the
     * Parser instance, so may be used directly as a callback function.
     */
    next: (source: string) => void;
    private lexer;
    /** Call at end of input to push out any remaining constructions */
    end(): void;
    private get sourceToken();
    private step;
    private peek;
    private pop;
    private stream;
    private document;
    private scalar;
    private blockScalar;
    private blockMap;
    private blockSequence;
    private flowCollection;
    private flowScalar;
    private startBlockValue;
    private documentEnd;
    private lineEnd;
}
