'use strict';

var composer = require('./compose/composer.js');
var Document = require('./doc/Document.js');
var errors = require('./errors.js');
var log = require('./log.js');
var lineCounter = require('./parse/line-counter.js');
var parser = require('./parse/parser.js');

function parseOptions(options) {
    const prettyErrors = !options || options.prettyErrors !== false;
    const lineCounter$1 = (options && options.lineCounter) ||
        (prettyErrors && new lineCounter.LineCounter()) ||
        null;
    return { lineCounter: lineCounter$1, prettyErrors };
}
/**
 * Parse the input as a stream of YAML documents.
 *
 * Documents should be separated from each other by `...` or `---` marker lines.
 *
 * @returns If an empty `docs` array is returned, it will be of type
 *   EmptyStream and contain additional stream information. In
 *   TypeScript, you should use `'empty' in docs` as a type guard for it.
 */
function parseAllDocuments(source, options) {
    const { lineCounter, prettyErrors } = parseOptions(options);
    const docs = [];
    const composer$1 = new composer.Composer(doc => docs.push(doc), options);
    const parser$1 = new parser.Parser(composer$1.next, lineCounter === null || lineCounter === void 0 ? void 0 : lineCounter.addNewLine);
    parser$1.parse(source);
    composer$1.end();
    if (prettyErrors && lineCounter)
        for (const doc of docs) {
            doc.errors.forEach(errors.prettifyError(source, lineCounter));
            doc.warnings.forEach(errors.prettifyError(source, lineCounter));
        }
    if (docs.length > 0)
        return docs;
    return Object.assign([], { empty: true }, composer$1.streamInfo());
}
/** Parse an input string into a single YAML.Document */
function parseDocument(source, options) {
    const { lineCounter, prettyErrors } = parseOptions(options);
    // `doc` is always set by compose.end(true) at the very latest
    let doc = null;
    const composer$1 = new composer.Composer(_doc => {
        if (!doc)
            doc = _doc;
        else if (doc.options.logLevel !== 'silent') {
            const errMsg = 'Source contains multiple documents; please use YAML.parseAllDocuments()';
            doc.errors.push(new errors.YAMLParseError(_doc.range[0], errMsg));
        }
    }, options);
    const parser$1 = new parser.Parser(composer$1.next, lineCounter === null || lineCounter === void 0 ? void 0 : lineCounter.addNewLine);
    parser$1.parse(source);
    composer$1.end(true, source.length);
    if (prettyErrors && lineCounter) {
        doc.errors.forEach(errors.prettifyError(source, lineCounter));
        doc.warnings.forEach(errors.prettifyError(source, lineCounter));
    }
    return doc;
}
function parse(src, reviver, options) {
    let _reviver = undefined;
    if (typeof reviver === 'function') {
        _reviver = reviver;
    }
    else if (options === undefined && reviver && typeof reviver === 'object') {
        options = reviver;
    }
    const doc = parseDocument(src, options);
    if (!doc)
        return null;
    doc.warnings.forEach(warning => log.warn(doc.options.logLevel, warning));
    if (doc.errors.length > 0) {
        if (doc.options.logLevel !== 'silent')
            throw doc.errors[0];
        else
            doc.errors = [];
    }
    return doc.toJS(Object.assign({ reviver: _reviver }, options));
}
function stringify(value, replacer, options) {
    let _replacer = null;
    if (typeof replacer === 'function' || Array.isArray(replacer)) {
        _replacer = replacer;
    }
    else if (options === undefined && replacer) {
        options = replacer;
    }
    if (typeof options === 'string')
        options = options.length;
    if (typeof options === 'number') {
        const indent = Math.round(options);
        options = indent < 1 ? undefined : indent > 8 ? { indent: 8 } : { indent };
    }
    if (value === undefined) {
        const { keepUndefined } = options || replacer || {};
        if (!keepUndefined)
            return undefined;
    }
    return new Document.Document(value, _replacer, options).toString(options);
}

exports.parse = parse;
exports.parseAllDocuments = parseAllDocuments;
exports.parseDocument = parseDocument;
exports.stringify = stringify;
