'use strict';

var anchors = require('../doc/anchors.js');
var Node = require('../nodes/Node.js');
var stringifyString = require('./stringifyString.js');

const createStringifyContext = (doc, options) => ({
    anchors: new Set(),
    doc,
    indent: '',
    indentStep: typeof options.indent === 'number' ? ' '.repeat(options.indent) : '  ',
    options: Object.assign({
        blockQuote: true,
        defaultKeyType: null,
        defaultStringType: 'PLAIN',
        directives: null,
        doubleQuotedAsJSON: false,
        doubleQuotedMinMultiLineLength: 40,
        falseStr: 'false',
        indentSeq: true,
        lineWidth: 80,
        minContentWidth: 20,
        nullStr: 'null',
        simpleKeys: false,
        singleQuote: null,
        trueStr: 'true',
        verifyAliasOrder: true
    }, options)
});
function getTagObject(tags, item) {
    if (item.tag) {
        const match = tags.filter(t => t.tag === item.tag);
        if (match.length > 0)
            return match.find(t => t.format === item.format) || match[0];
    }
    let tagObj = undefined;
    let obj;
    if (Node.isScalar(item)) {
        obj = item.value;
        const match = tags.filter(t => t.identify && t.identify(obj));
        tagObj =
            match.find(t => t.format === item.format) || match.find(t => !t.format);
    }
    else {
        obj = item;
        tagObj = tags.find(t => t.nodeClass && obj instanceof t.nodeClass);
    }
    if (!tagObj) {
        // @ts-ignore
        const name = obj && obj.constructor ? obj.constructor.name : typeof obj;
        throw new Error(`Tag not resolved for ${name} value`);
    }
    return tagObj;
}
// needs to be called before value stringifier to allow for circular anchor refs
function stringifyProps(node, tagObj, { anchors: anchors$1, doc }) {
    const props = [];
    const anchor = (Node.isScalar(node) || Node.isCollection(node)) && node.anchor;
    if (anchor && anchors.anchorIsValid(anchor)) {
        anchors$1.add(anchor);
        props.push(`&${anchor}`);
    }
    if (node.tag) {
        props.push(doc.directives.tagString(node.tag));
    }
    else if (!tagObj.default) {
        props.push(doc.directives.tagString(tagObj.tag));
    }
    return props.join(' ');
}
function stringify(item, ctx, onComment, onChompKeep) {
    if (Node.isPair(item))
        return item.toString(ctx, onComment, onChompKeep);
    if (Node.isAlias(item))
        return item.toString(ctx);
    let tagObj = undefined;
    const node = Node.isNode(item)
        ? item
        : ctx.doc.createNode(item, { onTagObj: o => (tagObj = o) });
    if (!tagObj)
        tagObj = getTagObject(ctx.doc.schema.tags, node);
    const props = stringifyProps(node, tagObj, ctx);
    if (props.length > 0)
        ctx.indentAtStart = (ctx.indentAtStart || 0) + props.length + 1;
    const str = typeof tagObj.stringify === 'function'
        ? tagObj.stringify(node, ctx, onComment, onChompKeep)
        : Node.isScalar(node)
            ? stringifyString.stringifyString(node, ctx, onComment, onChompKeep)
            : node.toString(ctx, onComment, onChompKeep);
    if (!props)
        return str;
    return Node.isScalar(node) || str[0] === '{' || str[0] === '['
        ? `${props} ${str}`
        : `${props}\n${ctx.indent}${str}`;
}

exports.createStringifyContext = createStringifyContext;
exports.stringify = stringify;
