/**
 * @package      npmjs.com/package/@yandeu/tap (tap.min.js)
 *
 * @author       Yannick Deubel (https://github.com/yandeu)
 * @copyright    Copyright (c) 2021 Yannick Deubel; Project Url: https://github.com/yandeu/tap
 * @license      {@link https://github.com/yandeu/tap/blob/master/LICENSE|MIT}
 * @description  Inspired by tapjs (https://www.npmjs.com/package/tapjs)
 */
import { Events } from '@yandeu/events';
import { eventMatrix } from './eventMatrix';
import { VERSION } from './version';
export class Tap {
    constructor(domElement) {
        this._events = new Events();
        this.domElement = null;
        this._isDown = false;
        this._isPaused = false;
        this.active = {
            touch: false,
            mouse: false,
            pointer: false
        };
        this.registered = {
            touch: false,
            mouse: false,
            pointer: false
        };
        this._currentPosition = { x: -1, y: -1 };
        this._lastPosition = { x: -1, y: -1 };
        this._isPointerLockAvailable = 'onpointerlockchange' in document;
        this._add(domElement);
    }
    static get VERSION() {
        return VERSION;
    }
    get isDown() {
        return this._isDown;
    }
    set _position(position) {
        if (position.x === this._currentPosition.x && position.y == this._currentPosition.y)
            return;
        this._lastPosition = this._currentPosition;
        this._currentPosition = position;
    }
    get currentPosition() {
        return this._currentPosition;
    }
    get lastPosition() {
        return this._lastPosition;
    }
    get isPaused() {
        return this._isPaused;
    }
    pause() {
        this._isPaused = true;
    }
    resume() {
        this._isPaused = false;
    }
    // listen to pointer lock change events
    _onPointerLockChange() {
        return new Promise(resolve => {
            document.addEventListener('pointerlockchange', e => {
                resolve(e);
            }, { once: true });
        });
    }
    get pointerLock() {
        return {
            onceChange: this._onPointerLockChange,
            request: () => {
                return new Promise((resolve, reject) => {
                    if (!this._isPointerLockAvailable)
                        return reject('PointerLock is not available!');
                    if (this.pointerLock.isLocked)
                        return reject('Pointer is already locked!');
                    // listen to pointer lock change events
                    this._onPointerLockChange().then(e => {
                        resolve(e);
                    });
                    this.once.down(() => {
                        var _a;
                        (_a = this.domElement) === null || _a === void 0 ? void 0 : _a.requestPointerLock();
                    });
                });
            },
            exit: () => {
                return new Promise((resolve, reject) => {
                    if (!this.pointerLock.isLocked)
                        return reject('Pointer is not locked!');
                    // listen to pointer lock change events
                    this._onPointerLockChange().then(e => {
                        resolve(e);
                    });
                    document.exitPointerLock();
                });
            },
            available: this._isPointerLockAvailable,
            isLocked: !!document.pointerLockElement
        };
    }
    /** (once ignores paused) */
    get once() {
        return {
            down: (callback) => {
                this._events.once('down', data => {
                    callback(data);
                });
            },
            move: (callback) => {
                this._events.once('move', data => {
                    callback(data);
                });
            },
            up: (callback) => {
                this._events.once('up', data => {
                    callback(data);
                });
            }
        };
    }
    get on() {
        return {
            down: (callback) => {
                this._events.on('down', data => {
                    if (!this._isPaused)
                        callback(data);
                });
            },
            move: (callback) => {
                this._events.on('move', data => {
                    if (!this._isPaused)
                        callback(data);
                });
            },
            up: (callback) => {
                this._events.on('up', data => {
                    if (!this._isPaused)
                        callback(data);
                });
            }
        };
    }
    _add(element) {
        const el = (this.domElement = element !== null && element !== void 0 ? element : window);
        if (!el)
            console.warn('[tap] No domElement found!');
        this._onDown = this._onDown.bind(this);
        this._onMove = this._onMove.bind(this);
        this._onUp = this._onUp.bind(this);
        eventMatrix.forEach(input => {
            if (input.test && input.enabled) {
                this.active[input.name] = true;
                el.addEventListener(input.events.down, this._onDown, false);
                el.addEventListener(input.events.move, this._onMove, false);
                el.addEventListener(input.events.up, this._onUp, false);
            }
        });
    }
    _remove(type) {
        if (!this.active[type])
            return;
        const el = this.domElement;
        if (!el)
            console.warn('[tap] No domElement found!');
        eventMatrix.forEach(input => {
            if (input.name === type) {
                el.removeEventListener(input.events.down, this._onDown, false);
                el.removeEventListener(input.events.move, this._onMove, false);
                el.removeEventListener(input.events.up, this._onUp, false);
            }
        });
        this.active[type] = false;
    }
    destroy() {
        this.pause();
        Object.keys(this.active).forEach(key => {
            this._remove(key);
        });
        this._events.removeAllListeners();
        // @ts-ignore
        this._events = null;
        // @ts-ignore
        this.domElement = null;
        // @ts-ignore
        this._isDown = null;
        // @ts-ignore
        this._isPaused = null;
        // @ts-ignore
        this.active = null;
        // @ts-ignore
        this.registered = null;
        // @ts-ignore
        this._currentPosition = null;
        // @ts-ignore
        this._lastPosition = null;
    }
    _calcPosition(e) {
        let x;
        let y;
        if (e.touches && e.touches[0]) {
            x = e.touches[0].pageX;
            y = e.touches[0].pageY;
        }
        else if (e.clientX) {
            x = e.clientX;
            y = e.clientY;
        }
        else {
            x = this._currentPosition.x;
            y = this._currentPosition.y;
        }
        if (this.pointerLock.isLocked) {
            x = e.movementX;
            y = e.movementY;
        }
        this._position = { x, y };
        return { x, y };
    }
    _removeDuplicates(e) {
        if (e.type === 'pointerdown')
            this.registered.pointer = true;
        if (e.type === 'touchstart')
            this.registered.touch = true;
        if (e.type === 'mousedown')
            this.registered.mouse = true;
        if (e.type === 'touchstart' && this.active.touch && this.registered.pointer) {
            this._remove('touch');
            return false;
        }
        if (e.type === 'mousedown' && this.active.mouse && (this.registered.pointer || this.registered.touch)) {
            this._remove('mouse');
            return false;
        }
        return true;
    }
    _onDown(e) {
        const proceed = this._removeDuplicates(e);
        if (!proceed)
            return;
        this._isDown = true;
        this._events.emit('down', { position: this._calcPosition(e), event: e });
    }
    _onMove(e) {
        this._events.emit('move', { position: this._calcPosition(e), event: e, dragging: this._isDown });
    }
    _onUp(e) {
        this._isDown = false;
        this._events.emit('up', { position: this._calcPosition(e), event: e });
    }
}
// test section
// const test = new Tap(document.body)
// test.pointerLock.onceChange()
// test.on.down(({ event, position }) => {
//   console.log(event, position)
// })
// const requestPointerLockLoop = () => {
//   console.log('request')
//   test.pointerLock.request().then(() => {
//     console.log('locked')
//     test.pointerLock.onceChange().then(() => {
//       console.log('onceChange', !test.pointerLock.isLocked)
//       if (!test.pointerLock.isLocked) requestPointerLockLoop()
//     })
//   })
// }
// requestPointerLockLoop()
// setTimeout(() => {
//   test.pointerLock.request()
// }, 2000)
//# sourceMappingURL=index.js.map