export function getMovedCoords(moved: ContentMove | {
    start: RelativePosition;
    end: RelativePosition;
}, tr: Transaction, split: boolean): {
    start: Item;
    end: Item;
};
export function findMoveLoop(tr: Transaction, moved: ContentMove, movedItem: Item, trackedMovedItems: Set<Item>): boolean;
/**
 * @private
 */
export class ContentMove {
    /**
     * @param {RelativePosition} start
     * @param {RelativePosition} end
     * @param {number} priority if we want to move content that is already moved, we need to assign a higher priority to this move operation.
     */
    constructor(start: RelativePosition, end: RelativePosition, priority: number);
    start: RelativePosition;
    end: RelativePosition;
    priority: number;
    /**
     * We store which Items+ContentMove we override. Once we delete
     * this ContentMove, we need to re-integrate the overridden items.
     *
     * This representation can be improved if we ever run into memory issues because of too many overrides.
     * Ideally, we should probably just re-iterate the document and re-integrate all moved items.
     * This is fast enough and reduces memory footprint significantly.
     *
     * @type {Set<Item>}
     */
    overrides: Set<Item>;
    /**
     * @return {number}
     */
    getLength(): number;
    /**
     * @return {Array<any>}
     */
    getContent(): Array<any>;
    /**
     * @return {boolean}
     */
    isCountable(): boolean;
    /**
     * @return {ContentMove}
     */
    copy(): ContentMove;
    /**
     * @param {number} offset
     * @return {ContentMove}
     */
    splice(offset: number): ContentMove;
    /**
     * @param {ContentMove} right
     * @return {boolean}
     */
    mergeWith(right: ContentMove): boolean;
    /**
     * @param {Transaction} transaction
     * @param {Item} item
     */
    integrate(transaction: Transaction, item: Item): void;
    /**
     * @param {Transaction} transaction
     * @param {Item} item
     */
    delete(transaction: Transaction, item: Item): void;
    /**
     * @param {StructStore} store
     */
    gc(store: StructStore): void;
    /**
     * @param {UpdateEncoderV1 | UpdateEncoderV2} encoder
     * @param {number} offset
     */
    write(encoder: UpdateEncoderV1 | UpdateEncoderV2, offset: number): void;
    /**
     * @return {number}
     */
    getRef(): number;
    isCollapsed(): boolean;
}
export function readContentMove(decoder: UpdateDecoderV1 | UpdateDecoderV2): ContentMove;
import { RelativePosition } from "../utils/RelativePosition.js";
import { Transaction } from "../utils/Transaction.js";
import { Item } from "./Item.js";
import { StructStore } from "../utils/StructStore.js";
import { UpdateEncoderV1 } from "../utils/UpdateEncoder.js";
import { UpdateEncoderV2 } from "../utils/UpdateEncoder.js";
import { UpdateDecoderV1 } from "../utils/UpdateDecoder.js";
import { UpdateDecoderV2 } from "../utils/UpdateDecoder.js";
