/**
 * Structure that helps to iterate through list-like structures. This is a useful abstraction that keeps track of move operations.
 */
export class ListCursor {
    /**
     * @param {AbstractType<any>} type
     */
    constructor(type: AbstractType<any>);
    type: AbstractType<any>;
    /**
     * Current index-position
     */
    index: number;
    /**
     * Relative position to the current item (if item.content.length > 1)
     */
    rel: number;
    /**
     * This refers to the current right item, unless reachedEnd is true. Then it refers to the left item.
     *
     * @public
     * @type {Item | null}
     */
    public nextItem: Item | null;
    reachedEnd: boolean;
    /**
     * @type {Item | null}
     */
    currMove: Item | null;
    /**
     * @type {Item | null}
     */
    currMoveStart: Item | null;
    /**
     * @type {Item | null}
     */
    currMoveEnd: Item | null;
    /**
     * @type {Array<{ start: Item | null, end: Item | null, move: Item }>}
     */
    movedStack: {
        start: Item | null;
        end: Item | null;
        move: Item;
    }[];
    clone(): ListCursor;
    /**
     * @type {Item | null}
     */
    get left(): Item | null;
    /**
     * @type {Item | null}
     */
    get right(): Item | null;
    /**
     * @param {Transaction} tr
     * @param {number} index
     */
    moveTo(tr: Transaction, index: number): void;
    /**
     * When using skipUncountables=false within a "useSearchMarker" call, it is recommended
     * to move the marker to the end. @todo do this after each useSearchMarkerCall
     *
     * @param {Transaction} tr
     * @param {number} len
     * @param {boolean} skipUncountables Iterate as much as possible iterating over uncountables until we find the next item.
     */
    forward(tr: Transaction, len: number, skipUncountables: boolean): ListCursor;
    /**
     * We prefer to insert content outside of a moved range.
     * Try to escape the moved range by walking to the left over deleted items.
     *
     * @param {Transaction} tr
     */
    reduceMoveDepth(tr: Transaction): void;
    /**
     * @param {Transaction} tr
     * @param {number} len
     * @return {ListCursor}
     */
    backward(tr: Transaction, len: number): ListCursor;
    /**
     * @template {{length: number}} T
     * @param {Transaction} tr
     * @param {number} len
     * @param {T} value the initial content
     * @param {function(AbstractContent, number, number):T} slice
     * @param {function(T, T): T} concat
     */
    _slice<T extends {
        length: number;
    }>(tr: Transaction, len: number, value: T, slice: (arg0: AbstractContent, arg1: number, arg2: number) => T, concat: (arg0: T, arg1: T) => T): T;
    /**
     * @param {Transaction} tr
     * @param {number} len
     */
    delete(tr: Transaction, len: number): void;
    /**
     * @param {Transaction} tr
     */
    _splitRel(tr: Transaction): void;
    /**
     * Important: you must update markers after calling this method!
     *
     * @param {Transaction} tr
     * @param {Array<AbstractContent>} content
     */
    insertContents(tr: Transaction, content: Array<AbstractContent>): void;
    /**
     * @param {Transaction} tr
     * @param {Array<{ start: RelativePosition, end: RelativePosition }>} ranges
     */
    insertMove(tr: Transaction, ranges: Array<{
        start: RelativePosition;
        end: RelativePosition;
    }>): void;
    /**
     * @param {Transaction} tr
     * @param {Array<Object<string,any>|Array<any>|boolean|number|null|string|Uint8Array>} values
     */
    insertArrayValue(tr: Transaction, values: Array<{
        [x: string]: any;
    } | Array<any> | boolean | number | null | string | Uint8Array>): void;
    /**
     * @param {Transaction} tr
     * @param {number} len
     */
    slice(tr: Transaction, len: number): any[];
    /**
     * @param {Transaction} tr
     * @param {function(any, number, any):void} f
     */
    forEach(tr: Transaction, f: (arg0: any, arg1: number, arg2: any) => void): void;
    /**
     * @template T
     * @param {Transaction} tr
     * @param {function(any, number, any):T} f
     * @return {Array<T>}
     */
    map<T_1>(tr: Transaction, f: (arg0: any, arg1: number, arg2: any) => T_1): T_1[];
    /**
     * @param {Transaction} tr
     */
    values(tr: Transaction): {
        [Symbol.iterator](): any;
        next: () => {
            done: boolean;
            value?: undefined;
        } | {
            done: boolean;
            value: any;
        };
    };
}
export function getMinimalListViewRanges(tr: Transaction, walker: ListCursor, len: number): Array<{
    start: RelativePosition;
    end: RelativePosition;
}>;
import { AbstractType } from "../types/AbstractType.js";
import { Item } from "../structs/Item.js";
import { Transaction } from "./Transaction.js";
import { AbstractContent } from "../structs/Item.js";
import { RelativePosition } from "./RelativePosition.js";
