/*
 * Wrapped stacktrace
 *
 * We need this because in some implementations, constructing a trace is slow
 * and so we want to defer accessing the trace for as long as possible
 */
'use strict';
function _Stacktrace(e) {
    this._e = e;
}
_Stacktrace.prototype.get = function () {
    if (global.zone.stackFramesFilter && this._e.stack) {
        return this._e.stack
            .split('\n')
            .filter(global.zone.stackFramesFilter)
            .join('\n');
    }
    return this._e.stack;
};
function _getStacktraceWithUncaughtError() {
    return new _Stacktrace(new Error());
}
function _getStacktraceWithCaughtError() {
    try {
        throw new Error();
    }
    catch (e) {
        return new _Stacktrace(e);
    }
}
// Some implementations of exception handling don't create a stack trace if the exception
// isn't thrown, however it's faster not to actually throw the exception.
var stack = _getStacktraceWithUncaughtError();
var _getStacktrace = stack && stack._e.stack
    ? _getStacktraceWithUncaughtError
    : _getStacktraceWithCaughtError;
exports.longStackTraceZone = {
    getLongStacktrace: function (exception) {
        var traces = [];
        var currentZone = this;
        if (exception) {
            if (currentZone.stackFramesFilter && exception.stack) {
                traces.push(exception.stack.split('\n')
                    .filter(currentZone.stackFramesFilter)
                    .join('\n'));
            }
            else {
                traces.push(exception.stack);
            }
        }
        var now = Date.now();
        while (currentZone && currentZone.constructedAtException) {
            traces.push('--- ' + (new Date(currentZone.constructedAtTime)).toString() +
                ' - ' + (now - currentZone.constructedAtTime) + 'ms ago', currentZone.constructedAtException.get());
            currentZone = currentZone.parent;
        }
        return traces.join('\n');
    },
    stackFramesFilter: function (line) {
        return !/zone(-microtask)?(\.min)?\.js/.test(line);
    },
    onError: function (exception) {
        var reporter = this.reporter || console.log.bind(console);
        reporter(exception.toString());
        reporter(this.getLongStacktrace(exception));
    },
    '$fork': function (parentFork) {
        return function () {
            var newZone = parentFork.apply(this, arguments);
            newZone.constructedAtException = _getStacktrace();
            newZone.constructedAtTime = Date.now();
            return newZone;
        };
    }
};
//# sourceMappingURL=data:application/json;base64,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