/*
 * check that document.registerElement(name, { prototype: proto });
 * is properly patched
 */
var util_1 = require('../util');
function registerElement() {
    return ('registerElement' in document);
}
registerElement.message = 'document.registerElement';
describe('document.registerElement', util_1.ifEnvSupports(registerElement, function () {
    // register a custom element for each callback
    var callbackNames = [
        'created',
        'attached',
        'detached',
        'attributeChanged'
    ];
    var callbacks = {};
    var testZone = global.zone.fork();
    var customElements;
    customElements = testZone.run(function () {
        callbackNames.map(function (callbackName) {
            var fullCallbackName = callbackName + 'Callback';
            var proto = Object.create(HTMLElement.prototype);
            proto[fullCallbackName] = function (arg) {
                callbacks[callbackName](arg);
            };
            return document.registerElement('x-' + callbackName.toLowerCase(), {
                prototype: proto
            });
        });
    });
    it('should work with createdCallback', function (done) {
        callbacks.created = function () {
            expect(global.zone).toBeDirectChildOf(testZone);
            done();
        };
        document.createElement('x-created');
    });
    it('should work with attachedCallback', function (done) {
        callbacks.attached = function () {
            expect(global.zone).toBeDirectChildOf(testZone);
            done();
        };
        var elt = document.createElement('x-attached');
        document.body.appendChild(elt);
        document.body.removeChild(elt);
    });
    it('should work with detachedCallback', function (done) {
        callbacks.detached = function () {
            expect(global.zone).toBeDirectChildOf(testZone);
            done();
        };
        var elt = document.createElement('x-detached');
        document.body.appendChild(elt);
        document.body.removeChild(elt);
    });
    it('should work with attributeChanged', function (done) {
        callbacks.attributeChanged = function () {
            expect(global.zone).toBeDirectChildOf(testZone);
            done();
        };
        var elt = document.createElement('x-attributechanged');
        elt.id = 'bar';
    });
    it('should work with non-writable, non-configurable prototypes created with defineProperty', function (done) {
        testZone.run(function () {
            var proto = Object.create(HTMLElement.prototype);
            Object.defineProperty(proto, 'createdCallback', {
                writeable: false,
                configurable: false,
                value: checkZone
            });
            document.registerElement('x-prop-desc', { prototype: proto });
            function checkZone() {
                expect(global.zone).toBeDirectChildOf(testZone);
                done();
            }
        });
        var elt = document.createElement('x-prop-desc');
    });
    it('should work with non-writable, non-configurable prototypes created with defineProperties', function (done) {
        testZone.run(function () {
            var proto = Object.create(HTMLElement.prototype);
            Object.defineProperties(proto, {
                createdCallback: {
                    writeable: false,
                    configurable: false,
                    value: checkZone
                }
            });
            document.registerElement('x-props-desc', { prototype: proto });
            function checkZone() {
                expect(global.zone).toBeDirectChildOf(testZone);
                done();
            }
        });
        var elt = document.createElement('x-props-desc');
    });
    it('should check bind callback if not own property', function (done) {
        testZone.run(function () {
            var originalProto = {
                createdCallback: checkZone
            };
            var secondaryProto = Object.create(originalProto);
            expect(secondaryProto.createdCallback).toBe(originalProto.createdCallback);
            document.registerElement('x-inherited-callback', { prototype: secondaryProto });
            expect(secondaryProto.createdCallback).not.toBe(originalProto.createdCallback);
            function checkZone() {
                expect(global.zone).toBeDirectChildOf(testZone);
                done();
            }
            var elt = document.createElement('x-inherited-callback');
        });
    });
    it('should not throw if no options passed to registerElement', function () {
        expect(function () {
            document.registerElement('x-no-opts');
        }).not.toThrow();
    });
}));
//# sourceMappingURL=data:application/json;base64,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