import * as har_format from 'har-format';
import { Request as Request$1, PostDataCommon, Param } from 'har-format';
import { ReducedHelperObject } from './helpers/reducer.js';
import { CodeBuilderOptions } from './helpers/code-builder.js';
import { UrlWithParsedQuery } from 'url';
import { Merge } from 'type-fest';

type TargetId = keyof typeof targets;
type ClientId = string;
interface ClientInfo {
    description: string;
    extname: Extension;
    installation?: string;
    key: ClientId;
    link: string;
    title: string;
}
type Converter<T extends Record<string, any>> = (request: Request, options?: Merge<CodeBuilderOptions, T>) => string;
interface Client<T extends Record<string, any> = Record<string, any>> {
    convert: Converter<T>;
    info: ClientInfo;
}
interface ClientPlugin<T extends Record<string, any> = Record<string, any>> {
    client: Client<T>;
    target: TargetId;
}
type Extension = `.${string}` | null;
interface TargetInfo {
    cli?: string;
    default: string;
    key: TargetId;
    title: string;
}
interface Target {
    clientsById: Record<ClientId, Client>;
    info: TargetInfo;
}
declare const targets: {
    c: Target;
    clojure: Target;
    csharp: Target;
    go: Target;
    http: Target;
    java: Target;
    javascript: Target;
    json: Target;
    kotlin: Target;
    node: Target;
    objc: Target;
    ocaml: Target;
    php: Target;
    powershell: Target;
    python: Target;
    r: Target;
    ruby: Target;
    shell: Target;
    swift: Target;
};
declare const isTarget: (target: Target) => target is Target;
declare const addTarget: (target: Target) => void;
declare const isClient: (client: Client) => client is Client;
declare const addClientPlugin: (plugin: ClientPlugin) => void;
declare const addTargetClient: (targetId: TargetId, client: Client) => void;

interface AvailableTarget extends TargetInfo {
    clients: ClientInfo[];
}
declare const availableTargets: () => AvailableTarget[];
declare const extname: (targetId: TargetId, clientId: ClientId) => "" | `.${string}`;

/** is this wrong?  yes.  according to the spec (http://www.softwareishard.com/blog/har-12-spec/#postData) it's technically wrong since `params` and `text` are (by the spec) mutually exclusive.  However, in practice, this is not what is often the case.
 *
 * In general, this library takes a _descriptive_ rather than _perscriptive_ approach (see https://amyrey.web.unc.edu/classes/ling-101-online/tutorials/understanding-prescriptive-vs-descriptive-grammar/).
 *
 * Then, in addition to that, it really adds to complexity with TypeScript (TypeScript takes this constraint very very seriously) in a way that's not actually super useful.  So, we treat this object as though it could have both or either of `params` and/or `text`.
 */
type PostDataBase = PostDataCommon & {
    params?: Param[];
    text?: string;
};
type HarRequest = Omit<Request$1, 'postData'> & {
    postData: PostDataBase;
};
interface RequestExtras {
    allHeaders: ReducedHelperObject;
    cookiesObj: ReducedHelperObject;
    fullUrl: string;
    headersObj: ReducedHelperObject;
    postData: PostDataBase & {
        boundary?: string;
        jsonObj?: ReducedHelperObject;
        paramsObj?: ReducedHelperObject;
    };
    queryObj: ReducedHelperObject;
    uriObj: UrlWithParsedQuery;
}
type Request = HarRequest & RequestExtras;
interface Entry {
    request: Partial<HarRequest>;
}
interface HarEntry {
    log: {
        creator: {
            name: string;
            version: string;
        };
        entries: Entry[];
        version: string;
    };
}
interface HTTPSnippetOptions {
    harIsAlreadyEncoded?: boolean;
}
declare class HTTPSnippet {
    initCalled: boolean;
    entries: Entry[];
    requests: Request[];
    options: HTTPSnippetOptions;
    constructor(input: HarEntry | HarRequest, opts?: HTTPSnippetOptions);
    init(): this;
    prepare(harRequest: HarRequest, options: HTTPSnippetOptions): {
        allHeaders: {
            [x: string]: string | string[];
        };
        fullUrl: string;
        url: string;
        uriObj: {
            query: ReducedHelperObject;
            search: string;
            path: string | null;
            auth: string | null;
            hash: string | null;
            host: string | null;
            hostname: string | null;
            href: string;
            pathname: string | null;
            protocol: string | null;
            slashes: boolean | null;
            port: string | null;
        };
        method: string;
        httpVersion: string;
        cookies: har_format.Cookie[];
        headers: har_format.Header[];
        queryString: har_format.QueryString[];
        headersSize: number;
        bodySize: number;
        comment?: string | undefined;
        postData: PostDataCommon & {
            params?: Param[];
            text?: string;
        } & {
            boundary?: string;
            jsonObj?: ReducedHelperObject;
            paramsObj?: ReducedHelperObject;
        };
        cookiesObj: ReducedHelperObject;
        headersObj: ReducedHelperObject;
        queryObj: ReducedHelperObject;
    };
    convert(targetId: TargetId, clientId?: ClientId, options?: any): false | string[];
}

export { type ClientId as C, type Extension as E, type HarRequest as H, type RequestExtras as R, type TargetId as T, type ClientInfo as a, type Converter as b, type Client as c, type ClientPlugin as d, type TargetInfo as e, type Target as f, addTarget as g, isClient as h, isTarget as i, addClientPlugin as j, addTargetClient as k, type Request as l, type HTTPSnippetOptions as m, HTTPSnippet as n, availableTargets as o, extname as p, targets as t };
