/*!
 * OpenUI5
 * (c) Copyright 2009-2024 SAP SE or an SAP affiliate company.
 * Licensed under the Apache License, Version 2.0 - see LICENSE.txt.
 */
sap.ui.define(
	[
		"sap/m/library",
		"sap/ui/mdc/filterbar/FilterBarBase",
		"sap/ui/mdc/filterbar/FilterBarBaseRenderer",
		"sap/ui/mdc/filterbar/aligned/FilterItemLayout",
		"sap/ui/mdc/filterbar/vh/FilterContainer",
		"sap/m/Button",
		"sap/m/p13n/enum/PersistenceMode"
	],
	function (
		mLibrary,
		FilterBarBase,
		FilterBarBaseRenderer,
		FilterItemLayout,
		FilterContainer,
		Button,
		PersistenceMode
	) {
		"use strict";
		/**
		 * Modules for value help dialog {@link sap.ui.mdc.filterbar.vh.FilterBar FilterBar}
		 * @namespace
		 * @name sap.ui.mdc.filterbar.vh
		 * @since 1.84.0
		 * @public
		 */

		/**
		 * Constructor for a new <code>FilterBar</code> for a value help dialog.
		 * @param {string} [sId] ID for the new control, generated automatically if no ID is given
		 * @param {object} [mSettings] initial settings for the new control
		 * @class The <code>FilterBar</code> control is used to display filter properties in a user-friendly manner to populate values for a query.
		 * The filters are arranged in a logical row that is divided depending on the space available and the width of the filters.
		 * The Go button triggers the search event, and the Show Filters button shows the additional filter field.<br>
		 * The <code>FilterBar</code> control creates and handles the filters based on the provided metadata information.
		 * The metadata information is provided via the {@link sap.ui.mdc.FilterBarDelegate FilterBarDelegate} implementation. This implementation has to be provided by the application.<br>
		 * <b>Note:</b> The <code>FilterBar</code> can only be used for a {@link sap.ui.mdc.valuehelp.Dialog Dialog} and not on its own.
		 * @extends sap.ui.mdc.filterbar.FilterBarBase
		 * @author SAP SE
		 * @version 1.120.10
		 * @constructor
		 * @public
		 * @since 1.84.0
		 * @alias sap.ui.mdc.filterbar.vh.FilterBar
		 */
		const FilterBar = FilterBarBase.extend(
			"sap.ui.mdc.filterbar.vh.FilterBar",
			{
				metadata: {
					library: "sap.ui.mdc",
					properties: {
						/**
						 * Path to the <code>Delegate</code> module that provides the required APIs to execute model-specific logic.<br>
						 * <b>Note:</b> Ensure that the related file can be requested (any required library has to be loaded before that).<br>
						 * Do not bind or modify the module. This property can only be configured during control initialization.
						 *
						 * @experimental
						 */
						delegate: {
							type: "object",
							defaultValue: {
								/**
								 * Contains the class name which implements the {@link sap.ui.mdc.FilterBarDelegate FilterBarDelegate} class.
								 */
								name: "sap/ui/mdc/filterbar/vh/FilterBarDelegate",
								/**
								 * Contains the mandatory information about the metamodel name <code>modelName</code> and the main data part in its <code>collectionName</code>.<br>
								 * <b>Note:</b> Additional information relevant for the specific {@link sap.ui.mdc.FilterBarDelegate FilterBarDelegate} implementation might be included but is of no relevance for the {@link sap.ui.mdc.FilterBar FilterBar} control itself.
								 */
								payload: {
									modelName: undefined,
									collectionName: ""
								}}
						},
						/**
						 * Determines whether the Show/Hide Filters button is in the state show or hide.<br>
						 */
						expandFilterFields: {
							type: "boolean",
							defaultValue: true
						},
						/**
						 * Number of FilterItems which will be shown via Show Filters.<br>
						 */
						filterFieldThreshold: {
							type: "int",
							defaultValue: 8
						}
					},
					aggregations : {
					}
				},

				renderer: FilterBarBaseRenderer
			}
		);

		const ButtonType = mLibrary.ButtonType;

		FilterBar.prototype._createInnerLayout = function () {
			this._cLayoutItem = FilterItemLayout;

			this._oFilterBarLayout = new FilterContainer(this.getId() + "-innerLayout", {});
			this.setAggregation("layout", this._oFilterBarLayout, true);


			this._oBtnFilters = new Button(this.getId() + "-btnShowFilters", {
				type: ButtonType.Transparent,
				press: this._onToggleFilters.bind(this)
			});
			this._oBtnFilters.bindProperty("text", {
				model : FilterBarBase.INNER_MODEL_NAME,
				path: "/expandFilterFields",
				formatter: function(bExpandFilterFields) {
					return this._oRb.getText("valuehelp." + (bExpandFilterFields ? "HIDE" : "SHOW") + "ADVSEARCH");
				}.bind(this)
			});
			this._oBtnFilters.bindProperty("visible", {
				model : FilterBarBase.INNER_MODEL_NAME,
				path: "/filterItems",
				formatter: function(aItems) {
					return aItems.length > 0;
				}
			});

			this._oFilterBarLayout.addControl(
				this._getSearchButton().bindProperty("visible", {
					parts: [
						{
							path: '/showGoButton',
							model: FilterBarBase.INNER_MODEL_NAME
						}, {
							path: "/liveMode",
							model: FilterBarBase.INNER_MODEL_NAME
						}
					],
					formatter: function(bShowGoButton, bLiveMode) {
						return bShowGoButton && ((this._isPhone()) ? true : !bLiveMode);
					}.bind(this)
				})
			);

			this._oFilterBarLayout.addControl(this._oBtnFilters);

			this._oShowAllFiltersBtn = new Button(this.getId() + "-btnShowAllFilters", {
				type: ButtonType.Transparent,
				press: this._onShowAllFilters.bind(this),
				text: this._oRb.getText("valuehelp.SHOWALLFILTERS"),
				visible: false
			});

			this._oFilterBarLayout.addEndContent(this._oShowAllFiltersBtn);
		};


		FilterBar.prototype.init = function() {
			FilterBarBase.prototype.init.apply(this, arguments);
			this.getEngine().defaultProviderRegistry.attach(this, PersistenceMode.Transient);
		};

		FilterBar.prototype.exit = function() {
			this.getEngine().defaultProviderRegistry.detach(this);
			if (this._oCollectiveSearch) { // do not destroy CollectiveSearch as it is owned by value help and might be reused there
				this._oFilterBarLayout.removeControl(this._oCollectiveSearch);
				this._oCollectiveSearch = null;
			}
			FilterBarBase.prototype.exit.apply(this, arguments);
			this._oBasicSearchField = null;
			this._oBtnFilters = null;
			this._oShowAllFiltersBtn = null;
		};

		FilterBar.prototype._onToggleFilters = function (oEvent) {
			this.setExpandFilterFields(!this.getExpandFilterFields());
		};

		FilterBar.prototype._onShowAllFilters = function (oEvent) {
			this._oFilterBarLayout._updateFilterBarLayout(true);
		};

		/**
		 * Sets the {@link sap.ui.mdc.filterbar.vh.CollectiveSearchSelect CollectiveSearchSelect} control.
		 *
		 * <b>Note:</b> This must only be done by the corresponding value help, not from outside.
		 *
		 * @param {sap.ui.mdc.filterbar.vh.CollectiveSearchSelect} oCollectiveSearch Instance of the {@link sap.ui.mdc.filterbar.vh.CollectiveSearchSelect CollectiveSearchSelect} control
		 * @returns {this} Reference to <code>this</code> to allow method chaining
		 * @protected
		 */
		FilterBar.prototype.setCollectiveSearch = function (oCollectiveSearch) {
			if (this._oCollectiveSearch) {
				if (this._oFilterBarLayout) {
					this._oFilterBarLayout.removeControl(this._oCollectiveSearch);
				}
			}
			this._oCollectiveSearch = oCollectiveSearch;
			if (this._oFilterBarLayout && this._oCollectiveSearch) {
				this._oFilterBarLayout.insertControl(this._oCollectiveSearch, 0);
			}

			return this;
		};

		/**
		 * Gets the {@link sap.ui.mdc.filterbar.vh.CollectiveSearchSelect CollectiveSearchSelect} control
		 *
		 * <b>Note:</b> This must only be used by the corresponding value help, not from outside.
		 *
		 * @returns {sap.ui.mdc.filterbar.vh.CollectiveSearchSelect} Instance of the {@link sap.ui.mdc.filterbar.vh.CollectiveSearchSelect CollectiveSearchSelect} control
		 * @protected
		 */
		FilterBar.prototype.getCollectiveSearch = function () {
			return this._oCollectiveSearch;
		};

		/**
		 * Destroys the {@link sap.ui.mdc.filterbar.vh.CollectiveSearchSelect CollectiveSearchSelect} control.
		 *
		 * <b>Note:</b> This must only be used by the corresponding value help, not from outside.
		 *
		 * @returns {this} Reference to <code>this</code> to allow method chaining
		 * @protected
		 */
		 FilterBar.prototype.destroyCollectiveSearch = function () {
			if (this._oCollectiveSearch && this._oFilterBarLayout) {
				this._oFilterBarLayout.removeControl(this._oCollectiveSearch);
				this._oCollectiveSearch.destroy();
				this._oCollectiveSearch = undefined;
			}

			return this;
		};

		FilterBar.prototype.setBasicSearchField = function (oBasicSearchField) {
			if (this._oBasicSearchField) {
				if (this._oFilterBarLayout) {
					this._oFilterBarLayout.removeControl(this._oBasicSearchField);
				}
				this._oBasicSearchField.detachSubmit(this._handleFilterItemSubmit, this);
			}
			this._oBasicSearchField = oBasicSearchField;

			if (oBasicSearchField) {
				if (this.isPropertyInitial("expandFilterFields")) {
					this.setExpandFilterFields(false);
				}

				if (this._oFilterBarLayout) {
					this._oFilterBarLayout.insertControl(oBasicSearchField, this._oCollectiveSearch ? 1 : 0);
				}

				oBasicSearchField.attachSubmit(this._handleFilterItemSubmit, this);
				if (!this._oObserver.isObserved(oBasicSearchField, {properties: ["visible"]})) {
					this._oObserver.observe(oBasicSearchField, {properties: ["visible"]});
				}
			}

			return this;
		};

		FilterBar.prototype.getBasicSearchField = function () {
			return this._oBasicSearchField;
		};

		FilterBar.prototype.destroyBasicSearchField = function () {
			if (this._oBasicSearchField && this._oFilterBarLayout) {
				this._oFilterBarLayout.removeControl(this._oBasicSearchField);
				this._oBasicSearchField.detachSubmit(this._handleFilterItemSubmit, this);
				if (this._oObserver.isObserved(this._oBasicSearchField, {properties: ["visible"]})) {
					this._oObserver.unobserve(this._oBasicSearchField);
				}
				this._oBasicSearchField.destroy();
				this._oBasicSearchField = undefined;
			}

			return this;
		};

		/**
		 * Getter for the initial focusable <code>control</code> on the <code>FilterBar</code>.
		 *
		 * @returns {sap.ui.core.Control} Control instance which could get the focus.
		 *
		 * @private
		 * @ui5-restricted sap.ui.mdc
		 */
		FilterBar.prototype.getInitialFocusedControl = function() {
			let oCtrl = this.getBasicSearchField();
			if (!oCtrl && this.getShowGoButton()) {
				oCtrl = this._btnSearch;
			}
			return oCtrl;
		};

		return FilterBar;
	}
);
