/*!
 * OpenUI5
 * (c) Copyright 2025 SAP SE or an SAP affiliate company.
 * Licensed under the Apache License, Version 2.0 - see LICENSE.txt.
 */

sap.ui.define([
	"sap/m/library",
	"./QuickAction",
	"./QuickActionItem",
	"sap/m/Switch"
], function (
	library,
	QuickAction,
	QuickActionItem,
	Switch
) {
	"use strict";

	/**
	 * Constructor for a new <code>QuickTotalItem</code>.
	 *
	 * @param {string} [sId] ID for the new <code>QuickTotalItem</code>, generated automatically if no ID is given
	 * @param {object} [mSettings] Initial settings for the new <code>QuickTotalItem</code>
	 *
	 * @class
	 * The <code>QuickTotalItem</code> class is used for items for the <code>sap.m.table.columnmenu.QuickTotal</code>.
	 * It can be used to specify control- and application-specific items for totaling.
	 *
	 * @extends sap.m.table.columnmenu.QuickActionItem
	 *
	 * @author SAP SE
	 * @version 1.136.0
	 *
	 * @public
	 * @since 1.110
	 *
	 * @alias sap.m.table.columnmenu.QuickTotalItem
	 */
	var QuickTotalItem = QuickActionItem.extend("sap.m.table.columnmenu.QuickTotalItem", {

		metadata: {
			library: "sap.m",
			properties: {
				/**
				 * Specifies whether a total for the respective column is shown.
				 */
				totaled: { type: "boolean", defaultValue: false }
			},
			aggregations: {
				/**
				 * Defines the quick action of the quick total item.
				 */
				quickAction: { type: "sap.m.table.columnmenu.QuickAction", multiple: false, visibility: "hidden" }
			}
		}
	});

	QuickTotalItem.prototype._getAction = function() {
		var oQuickAction = this.getAggregation("quickAction");
		var sLabel = this.getLabel();

		if (oQuickAction) {
			oQuickAction.setLabel(sLabel);
		} else {
			oQuickAction = new QuickAction({
				label: sLabel,
				content: [this._createContent()],
				category: library.table.columnmenu.Category.Aggregate,
				contentSize: library.InputListItemContentSize.S
			});
		}

		this.setAggregation("quickAction", oQuickAction, true);
		return oQuickAction;
	};

	QuickTotalItem.prototype._createContent = function() {
		return new Switch({
			state: this.getTotaled(),
			customTextOn: " ",
			customTextOff: " ",
			change: [{item: this}, this._onTotalChange, this]
		});
	};

	/*
	 * @see JSDoc generated by SAPUI5 control API generator
	 */
	QuickTotalItem.prototype.setTotaled = function(bTotaled) {
		this.setProperty("totaled", bTotaled, true);

		var oQuickAction = this.getAggregation("quickAction");
		if (oQuickAction) {
			var oSwitch = oQuickAction.getContent()[0];
			oSwitch.setState(bTotaled);
		}
		return this;
	};

	QuickTotalItem.prototype._onTotalChange = function (oEvent, mTotalInfo) {
		var bTotaled = oEvent.getSource().getState();
		this.setTotaled(bTotaled);
		this.getParent().onChange(mTotalInfo.item);
	};

	return QuickTotalItem;
});