/*!
 * OpenUI5
 * (c) Copyright 2025 SAP SE or an SAP affiliate company.
 * Licensed under the Apache License, Version 2.0 - see LICENSE.txt.
 */

// Provides the default renderer for control sap.tnt.NavigationList
sap.ui.define([
	"sap/ui/core/Lib"
], function (Lib) {
	"use strict";

	/**
	 * NavigationListRenderer renderer.
	 *
	 * @author SAP SE
	 * @namespace
	 */
	const NavigationListRenderer = {
		apiVersion: 2
	};

	/**
	 * Renders the HTML for the given control, using the provided {@link sap.ui.core.RenderManager}.
	 *
	 * @param {sap.ui.core.RenderManager} oRM The RenderManager that can be used for writing to the renderer output buffer
	 * @param {sap.tnt.NavigationList} oControl An object representation of the control that should be rendered
	 */
	NavigationListRenderer.render = function (oRM, oControl) {
		const bExpanded = oControl.getExpanded(),
			bHasItemWithIcon = oControl._containsIcon(),
			aVisibleItems = oControl.getItems().filter((oItem) => oItem.getVisible());

		oRM.openStart("ul", oControl)
			.class("sapTntNL");

		if (!bExpanded) {
			oRM.class("sapTntNLCollapsed");
		}

		if (!bHasItemWithIcon) {
			oRM.class("sapTntNLNoIcons");
		}

		var sWidth = oControl.getWidth();
		if (sWidth && bExpanded) {
			oRM.style("width", sWidth);
		}

		// ARIA
		const sRole = !bExpanded && !oControl.hasStyleClass("sapTntNLPopup") ? "menubar" : "tree";
		oRM.accessibilityState(oControl, {
			role: sRole,
			orientation: sRole === "menubar" ? "vertical" : undefined,
			roledescription: Lib.getResourceBundleFor("sap.tnt").getText(
				sRole === "menubar" ?
					"NAVIGATION_LIST_ITEM_ROLE_DESCRIPTION_MENUBAR" :
					"NAVIGATION_LIST_ITEM_ROLE_DESCRIPTION_TREE")
		});

		oRM.openEnd();

		const oFirstGroup = aVisibleItems.find((oItem) => oItem.isA("sap.tnt.NavigationListGroup"));
		aVisibleItems.forEach((oItem) => {
			oItem.render(oRM, oControl, oItem === oFirstGroup);
		});

		if (!bExpanded) {
			oControl._getOverflowItem().render(oRM, oControl);
		}

		oRM.close("ul");
	};

	return NavigationListRenderer;
}, /* bExport= */ true);