/*
 * ! OpenUI5
 * (c) Copyright 2009-2020 SAP SE or an SAP affiliate company.
 * Licensed under the Apache License, Version 2.0 - see LICENSE.txt.
 */

sap.ui.define([
	"sap/base/util/merge",
	"sap/ui/fl/write/connectors/BaseConnector",
	"sap/ui/fl/initial/_internal/connectors/BackendConnector",
	"sap/ui/fl/initial/_internal/connectors/Utils",
	"sap/ui/fl/write/_internal/connectors/Utils",
	"sap/base/util/restricted/_pick"
], function(
	merge,
	BaseConnector,
	InitialConnector,
	InitialUtils,
	WriteUtils,
	_pick
) {
	"use strict";

	/**
	 * Send request to a back end to write or update flex data.
	 *
	 * @param {object} mPropertyBag - Property bag
	 * @param {string} mPropertyBag.method - POST for writing new data and PUT for update an existing data
	 * @param {object[]} [mPropertyBag.flexObjects] - Objects to be written (i.e. change definitions, variant definitions etc.)
	 * @param {object} [mPropertyBag.flexObject] - Object to be updated
	 * @param {string} mPropertyBag.url Configured - url for the connector
	 * @param {number} [mPropertyBag.parentVersion] - Indicates if changes should be written as a draft and on which version the changes should be based on
	 * @returns {Promise} Promise resolves as soon as the writing was completed
	 */
	function _doWrite(mPropertyBag) {
		var mParameters = {};
		if (mPropertyBag.parentVersion !== undefined) {
			mParameters.parentVersion = mPropertyBag.parentVersion;
		}
		if (this.isLanguageInfoRequired) {
			InitialUtils.addLanguageInfo(mParameters);
		}
		var sWriteUrl = InitialUtils.getUrl(this.ROUTES.CHANGES, mPropertyBag, mParameters);
		delete mPropertyBag.fileName;
		delete mParameters["sap-language"];
		var sTokenUrl = InitialUtils.getUrl(this.ROUTES.TOKEN, mPropertyBag, mParameters);

		var oRequestOption = WriteUtils.getRequestOptions(
			this.initialConnector,
			sTokenUrl,
			mPropertyBag.flexObjects || mPropertyBag.flexObject,
			"application/json; charset=utf-8",
			"json"
		);
		return WriteUtils.sendRequest(sWriteUrl, mPropertyBag.method, oRequestOption);
	}

	/**
	 * Send request to a back end to write or update a single flex data.
	 *
	 * @param {object} mPropertyBag - Property bag
	 * @param {string} mPropertyBag.method - POST for writing new data and PUT for update an existing data
	 * @param {object} mPropertyBag.flexObject - Object to be updated
	 * @param {string} mPropertyBag.url - Configured url for the connector
	 * @returns {Promise} Promise resolves as soon as the writing was completed
	 */
	function _doSingleWrite(mPropertyBag) {
		mPropertyBag.fileName = mPropertyBag.flexObject.fileName;
		return _doWrite.call(this, mPropertyBag);
	}

	/**
	 * Base connector for saving and deleting data flexibility data from an back end.
	 *
	 * @namespace sap.ui.fl.write._internal.connectors.BackendConnector
	 * @since 1.72
	 * @version 1.84.17
	 * @private
	 * @ui5-restricted sap.ui.fl.write._internal.connectors
	 */
	var BackendConnector = merge({}, BaseConnector, /** @lends sap.ui.fl.write._internal.connectors.BackendConnector */ {

		xsrfToken: null,

		/**
		 * Resets flexibility files for a given application.
		 *
		 * @param {object} mPropertyBag - Property bag
		 * @param {string} mPropertyBag.reference - Flex reference of the application
		 * @param {string} mPropertyBag.url - Configured url for the connector
		 * @param {string} [mPropertyBag.generator] - Generator with which the changes were created
		 * @param {string} [mPropertyBag.selectorIds] - Selector IDs of controls for which the reset should filter (comma-separated list)
		 * @param {string} [mPropertyBag.changeTypes] - Change types of the changes which should be reset (comma-separated list)
		 * @returns {Promise} Promise resolves as soon as the reset has completed
		 */
		reset: function (mPropertyBag) {
			var aParameters = ["reference", "generator"];
			var mParameters = _pick(mPropertyBag, aParameters);
			if (mPropertyBag.selectorIds) {
				mParameters.selector = mPropertyBag.selectorIds;
			}
			if (mPropertyBag.changeTypes) {
				mParameters.changeType = mPropertyBag.changeTypes;
			}
			delete mPropertyBag.reference;

			var sResetUrl = InitialUtils.getUrl(this.ROUTES.CHANGES, mPropertyBag, mParameters);

			var sTokenUrl = InitialUtils.getUrl(this.ROUTES.TOKEN, mPropertyBag);

			var oRequestOption = WriteUtils.getRequestOptions(
				this.initialConnector,
				sTokenUrl
			);
			return WriteUtils.sendRequest(sResetUrl, "DELETE", oRequestOption);
		},

		/**
		 * Write flex data into a back end; This method is called with a list of entities like changes, variants,
		 * control variants, variant changes and variant management changes.
		 *
		 * @param {object} mPropertyBag - Property bag
		 * @param {object[]} mPropertyBag.flexObjects - Objects to be written (i.e. change definitions, variant definitions etc.)
		 * @param {string} mPropertyBag.url - Configured url for the connector
		 * @param {number} [mPropertyBag.parentVersion] - Indicates if changes should be written as a draft and on which version the changes should be based on
		 * @returns {Promise} Promise resolves as soon as the writing was completed
		 */
		write: function (mPropertyBag) {
			mPropertyBag.method = "POST";
			return _doWrite.call(this, mPropertyBag).then(function (oResponse) {
				//Single save of Personalization and Keyuser service return single JSON object
				//It needs to be put into an array so output of this function is the same format for both saving of single or multiple changes
				if (oResponse.response && !Array.isArray(oResponse.response)) {
					oResponse.response = [oResponse.response];
				}
				return oResponse;
			});
		},

		/**
		 * Update an existing flex data stored in the back end.
		 *
		 * @param {object} mPropertyBag - Property bag
		 * @param {object} mPropertyBag.flexObject - Flex Object to be updated
		 * @param {string} mPropertyBag.url - Configured url for the connector
		 * @returns {Promise} Resolves as soon as the writing is completed without data
		 */
		update: function (mPropertyBag) {
			mPropertyBag.method = "PUT";
			return _doSingleWrite.call(this, mPropertyBag);
		},

		/**
		 * Delete an existing flex data stored in the back end.
		 *
		 * @param {object} mPropertyBag - Property bag
		 * @param {object} mPropertyBag.flexObject - Flex Object to be deleted
		 * @param {string} mPropertyBag.url - Configured url for the connector
		 * @returns {Promise} Resolves as soon as the deletion is completed without data
		 */
		remove: function (mPropertyBag) {
			var mParameters = {
				namespace: mPropertyBag.flexObject.namespace
			};
			mPropertyBag.fileName = mPropertyBag.flexObject.fileName;
			var sDeleteUrl = InitialUtils.getUrl(this.ROUTES.CHANGES, mPropertyBag, mParameters);
			delete mPropertyBag.fileName;
			var sTokenUrl = InitialUtils.getUrl(this.ROUTES.TOKEN, mPropertyBag);

			var oRequestOption = WriteUtils.getRequestOptions(
				this.initialConnector,
				sTokenUrl,
				undefined,
				"application/json; charset=utf-8",
				"json"
			);
			return WriteUtils.sendRequest(sDeleteUrl, "DELETE", oRequestOption);
		},

		/**
		 * Called to get the flex features.
		 *
		 * @returns {Promise<object>} Promise resolves with an object containing the flex features
		 */
		loadFeatures: function (mPropertyBag) {
			if (this.initialConnector.settings) {
				return Promise.resolve({response: this.initialConnector.settings});
			}
			var sFeaturesUrl = InitialUtils.getUrl(this.ROUTES.SETTINGS, mPropertyBag);
			return InitialUtils.sendRequest(sFeaturesUrl).then(function (oResult) {
				return oResult.response;
			});
		}
	});

	BackendConnector.initialConnector = InitialConnector;
	return BackendConnector;
}, true);
