/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
(function() {
    oscar._imgPath = null;
    oscar._scriptName = null;
    oscar._getScriptLocation = function() {
        return this._scriptLocation;
    };
    oscar.getImagePath = function() {
        return oscar._getScriptLocation() + "images/";
    }

    oscar.jQuery = jQuery.noConflict();
    window.$$ = oscar.jQuery;
    oscar.DownloadHost = null;

    // Proxy to use for viewing iso metadata. Used by
    // oscar.Util.Plugin.Download.GetRecordByIdView
    oscar.ISOMetadataViewProxy = null;

    // Proxy to use for viewing iso metadata. Used by
    // oscar.Util.Plugin.Download.GetRecordById
    oscar.ISOMetadataDownloadProxy = null;

    // Proxy to use for previewing bathymetry coverages.
    oscar.PreviewCoverageProxy = null;
})();
/**
 * Constant: VERSION_NUMBER
 */
oscar.VERSION_NUMBER = "15.0.2-RC1";

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.debug
 */
oscar.debug = {
    canOutput : function() {
        var canOutput = false;
        try {
            canOutput = (window.console) ? true : false;
        } catch (err) {
        }
        
        return canOutput
    },
    error : function(obj) {
        if (this.canOutput()) {
            window.console.error(obj.message);
        }
    }
};

/**
 * Class: oscar.Util
 */
oscar.Util = {};

/**
 * APIMethod: buildUrl This method takes a string and object of parameters and
 * returns a url with query string.
 */
oscar.Util.buildUrl = function(url, parameters) {
    if (!url) {
        url = "";
    }
    var paramString = OpenLayers.Util.getParameterString(parameters);
    paramString = unescape(paramString);
    if (paramString.length > 0) {
        var separator = (url.indexOf('?') > -1) ? '&' : '?';
        url += separator + paramString;
    }
    return url;
}

/**
 * APIMethod: extractGeometriesFromFeatures This method takes an array of
 * features and extracts the geometry objects from them and returns them in an
 * array.
 * 
 * Returns an array of geometries.
 */
oscar.Util.extractGeometriesFromFeatures = function(features) {
    var geometryArray = [];
    while (features.length > 0) {
        var feature = features.shift();
        var geometry = feature.geometry;
        geometryArray.push(geometry);
    }
    
    return geometryArray;
}

/**
 * APIMethod: combineGeometries This method take an array of
 * <OpenLayers.Geometry> objects and attempts to combine them to fewer geometry
 * objects.
 * 
 * Returns an array of geometry objects.
 */
oscar.Util.combineGeometries = function(geoms) {
    var merged = []
    while (geoms.length > 0) {
        var geometry = geoms.shift();
        if (!oscar.Util.mergeToExistingGeometry(merged, geometry)) {
            merged.push(geometry);
        }
    }
    return merged;
}

/**
 * APIMethod: mergeGeometries Takes two <OpenLayers.Geometry> objects and merges
 * them to a single <OpenLayers.Geometry> object.
 * 
 * Returns the new geometry object.
 */
oscar.Util.mergeGeometries = function(geomA, geomB) {
    var reader = new jsts.io.WKTReader();
    var gom, strFeatB, union;
    strFeatA = reader.read(geomA.toString());
    strFeatB = reader.read(geomB.toString());
    union = strFeatA.union(strFeatB);
    var parser = new jsts.io.OpenLayersParser();
    return parser.write(union);
}

/**
 * APIMethod: mergeToExistingGeometry Takes a geometry and attempts to merge it
 * to a list of existing geometry objects. Returns true or false of the merge
 * was successful.
 */
oscar.Util.mergeToExistingGeometry = function(geometries, geometry) {
    for ( var g in geometries) {
        var existingGeometry = geometries[g];
        if (existingGeometry.intersects(geometry)) {
            geometries[g] = oscar.Util.mergeGeometries(existingGeometry, geometry);
            return true;
        }
    }
    return false;
    
}

/**
 * APIMethod:getMetersConversionFactor Uses the projection to obtain a
 * conversion factor value to display units in meters. Parameters: - projection (
 * OpenLayers.Projection)
 * 
 */
oscar.Util.getMetersConversionFactor = function(projection) {
    var proj = projection.proj;
    if (proj.projName == "longlat") {
        return OpenLayers.INCHES_PER_UNIT.degrees * OpenLayers.METERS_PER_INCH;
    } else if (proj.to_meter) {
        return proj.to_meter;
    } else {
        return 1;
    }
}

/**
 * 
 * APIMethod: isFeatureInArray Returns true of a feature name is found in the
 * array Parameters: - featureName {String} - selectedFeatures - array of
 * feature names {String}
 * 
 */
oscar.Util.isFeatureInArray = function(featureName, selectedFeatures) {
    for ( var i in selectedFeatures) {
        if (featureName == selectedFeatures[i])
            return true;
    }
    return false;
};

/**
 * Method: createFormElement
 * 
 * @deprecated
 * 
 * {Note: I cannot find any usage of this function anywhere}
 */
oscar.Util.createFormElement = function(element, type, name, id, value, parent) {
    var e = document.createElement(element);
    e.setAttribute("name", name);
    e.setAttribute("type", type);
    e.setAttribute("id", id);
    e.setAttribute("value", value);
    parent.appendChild(e);
};

/**
 * APIMethod: checkAll
 * 
 * Checks all checkboxes in a form except a checkbox using the name "allbox"
 */
oscar.Util.checkAll = function(theForm) {
    for (var i = 0; i < theForm.elements.length; i++) {
        var e = theForm.elements[i];
        var eName = e.name;
        if (eName != 'allbox' && e.type != null && (e.type.indexOf("checkbox") == 0)) {
            e.checked = theForm.allbox.checked;
        }
    }
};

/**
 * APIMethod: isOneChecked
 * 
 * Takes an array of checkbox elements and checks to see if at least one is
 * checked.
 */
oscar.Util.isOneChecked = function(checkbox) {
    if (checkbox == null) {
        return false;
    }
    if (checkbox.length) {
        for (var i = 0; i < checkbox.length; i++) {
            if (checkbox[i].checked) {
                return true;
            }
        }
    } else {
        if (checkbox.checked) {
            return true;
        }
    }
    return false;
};

/**
 * APIMethod: isSphericalMercator
 * 
 * Returns true if the coordinate system is EPSG:3857 or EPSG:900913
 * 
 * Parameters:
 * 
 * srs - {String} A coordiante system reference such as "EPSG:4326" or
 * "EPSG:3857"
 */
oscar.Util.isSphericalMercator = function(srs) {
    if ((srs == "EPSG:900913") || (srs == "EPSG:3857")) {
        return true;
    }
    return false;
};

/**
 * APIMethod: moveSelectedOptions
 * 
 * Moves selected options from one select box to another
 * 
 * Parameters:
 * 
 * fromSelectName - {String} Source select reference
 * 
 * toSelectName - {String} destination select reference
 */
oscar.Util.moveSelectedOptions = function(fromSelectName, toSelectName) {
    var fromSelect = document.getElementsByName(fromSelectName)[0];
    var toSelect = document.getElementsByName(toSelectName)[0];
    if (fromSelect != null && toSelect != null) {
        var i;
        for (i = 0; i < fromSelect.options.length; i++) {
            if (fromSelect.options[i].selected) {
                var opt = fromSelect.options[i];
                toSelect.options[toSelect.options.length] = new Option(opt.text, opt.value);
                fromSelect.remove(i);
                i--;
            }
        }
    }
};

// This function moves all the options from the fromSelect to the toSelect
/**
 * APIMethod: moveAllSelectOptions
 * 
 * Moves all options from one select box to another
 * 
 * Parameters:
 * 
 * fromSelectName - {String} Source select reference
 * 
 * toSelectName - {String} destination select reference
 */
oscar.Util.moveAllSelectOptions = function(fromSelectName, toSelectName) {
    var fromSelect = document.getElementsByName(fromSelectName)[0];
    var toSelect = document.getElementsByName(toSelectName)[0];
    
    if (fromSelect != null && toSelect != null) {
        var i;
        var numOfFromSelectOptions = fromSelect.options.length;
        for (i = 0; i < numOfFromSelectOptions; i++) {
            var opt = fromSelect.options[0];
            toSelect.options[toSelect.options.length] = new Option(opt.text, opt.value);
            fromSelect.remove(0);
        }
    }
};

/**
 * Submits the given form. If an action other than the one already associated
 * with the given form is supplied, it applies that one instead.
 * 
 * Input: form - The form to be submitted. By default it is submitted to the URI
 * defined by the form's action property. action - (Optional) The name of the
 * struts action to be invoked.
 */
oscar.Util.submitForm = function(form, action) {
    if (action != null)
        form.action = action;
    form.submit();
};

/**
 * APIMethod: getBrowserName
 * 
 * Returns: {String} A string which specifies which is the current browser in
 * which we are running.
 * 
 * Currently-supported browser detection and codes: * 'opera' -- Opera * 'msie' --
 * Internet Explorer * 'safari' -- Safari * 'firefox' -- FireFox * 'mozilla' --
 * Mozilla
 * 
 * If we are unable to property identify the browser, we return an empty string.
 */
oscar.Util.getBrowserName = function() {
    var browserName = "";
    
    var ua = navigator.userAgent.toLowerCase();
    if (ua.indexOf("opera") != -1) {
        browserName = "opera";
    } else if (ua.indexOf("msie") != -1) {
        browserName = "msie";
    } else if (ua.indexOf("safari") != -1) {
        browserName = "safari";
    } else if (ua.indexOf("mozilla") != -1) {
        if (ua.indexOf("firefox") != -1) {
            browserName = "firefox";
        } else {
            browserName = "mozilla";
        }
    }
    
    return browserName;
};

/**
 * APIMethod: buildWFSFilterForRequest
 * 
 * This method will generate a WFSFilter for WFS requests. geometry -
 * OpenLayers.LonLat || OpenLayers.Bounds
 * 
 * returns a WFSFilter
 * 
 */
oscar.Util.buildWFSFilterForRequest = function(geometry, srs, wfsVersion, propertyName) {
    var reqType = OpenLayers.Filter.Spatial.BBOX;
    spatialFilter = new OpenLayers.Filter.Spatial({
        type : reqType,
        value : geometry,
        projection : srs,
        property : propertyName
    });
    var filterFormat = new OpenLayers.Format.Filter({
        version : wfsVersion
    });
    var xml = new OpenLayers.Format.XML();
    return xml.write(filterFormat.write(spatialFilter));
};

/**
 * APIMethod: getDataFormatter
 * 
 * returns a data formatter for the passed-in data format
 * 
 * Parameters:
 * 
 * dataFormatType - Type of the data format options - An optional object whose
 * properties will be set on this instance
 */

oscar.Util.getDataFormatter = function(dataFormatType, options) {
    var dft = dataFormatType.toLowerCase();
    if (dft.indexOf("gml2") > -1 || dft.indexOf("gml/2") > -1)
        return new OpenLayers.Format.GML(options);
    else if (dft.indexOf("kml") > -1)
        return new OpenLayers.Format.KML(options);
    else if (dft.indexOf("json") > -1)
        return new OpenLayers.Format.GeoJSON(options);
    else if (dft == "georss_simple" || dft == "georss_geo")
        return new OpenLayers.Format.GeoRSS(options);
    return null;
};

/**
 * Maintain existing definition of $.
 */
if (typeof window.$ === "undefined") {
    window.$ = OpenLayers.Util.getElement;
}

/**
 * Fix the getElementById method for IE 7 and lower use browser sniffing to
 * determine if using an affected version
 */
var isOldIE = false;
var userAgent = navigator.userAgent;
if (userAgent.indexOf("MSIE") != -1) {
    var userAgentVersion = parseFloat(navigator.userAgent.split("MSIE")[1]);
    if (userAgentVersion < 8.0) {
        isOldIE = true;
    }
}

if (isOldIE) {
    // overwrite the existing getElementById method
    document.nativeGetElementById = document.getElementById;
    document.getElementById = function(id) {
        var elem = document.nativeGetElementById(id);
        if (elem) {
            // verify it is a valid match
            if (elem.attributes['id'] && elem.attributes['id'].value == id) {
                return elem;
            } else {
                // the non-standard, document.all array has keys
                // for all name'd, and id'd elements
                // start at one, because we know the first match
                // is wrong
                for (var i = 1; i < document.all[id].length; i++) {
                    if (document.all[id][i].attributes['id'] && document.all[id][i].attributes['id'].value == id) {
                        return document.all[id][i];
                    }
                }
            }
        }
        return null;
    };
}
/**
 * Method: getTimeStamp
 * 
 * Used to append to a URL to prevent caching
 * 
 * Returns: {String} Returns a key value pair.
 * 
 * Example
 * 
 * timestamp=421564215
 * 
 */
oscar.Util.getTimeStamp = function() {
    var now = new Date().getTime();
    return "timestamp=" + now;
};

/**
 * Method: getToolTipId
 * 
 * returns the toolTipId
 */

oscar.Util.getToolTipId = function(objWithClassName) {
    return toolTipId = objWithClassName.CLASS_NAME.toLowerCase().replaceAll(".", "_");
};

/**
 * Class: oscar.Util.EpsgConversion
 * 
 * Converts epsg codes to urns and vice versa
 */
oscar.Util.EpsgConversion = {
    CRS : "CRS",
    OGC : "OGC",
    EPSG : "EPSG",
    EPSG_URN : "urn:ogc:def:crs:EPSG::",
    CRS_URN : "urn:ogc:def:crs:OGC:2:",
    
    /**
     * APIMethod: oscar.Util.EpsgConversion.urnToEpsg
     * 
     * Convert urn to crs.
     * 
     * Parameters:
     * 
     * urn - {String} academic urn
     * 
     * Returns:
     * 
     * epsg - {String} epsg crs.
     * 
     * Usage: > var epsg =
     * oscar.Util.EpsgConversion.urnToEpsg("urn:ogc:def:crs:EPSG::4326"); > >
     * Returns "EPSG:4326"
     */
    urnToEpsg : function(urn) {
        var tokens = urn.split(":");
        if (tokens.length == 7) {
            if (tokens[4] == oscar.Util.EpsgConversion.OGC) {
                return oscar.Util.EpsgConversion.CRS + ':' + tokens[6];
            } else {
                return tokens[4] + ':' + tokens[6];
            }
        } else {
            return urn;
        }
    },
    /**
     * APIMethod: oscar.Util.EpsgConversion.epsgToUrn
     * 
     * Convert espg crs to urn.
     * 
     * Parameters:
     * 
     * epsg - {String}
     * 
     * Returns:
     * 
     * urn - {String} academic urn
     * 
     * Usage: > var urn = oscar.Util.EpsgConversion.epsgToUrn("EPSG:4326"); > >
     * Returns "urn:ogc:def:crs:EPSG::4326"
     */
    epsgToUrn : function(epsg) {
        var s = epsg.split(":");
        if (s[0] == oscar.Util.EpsgConversion.EPSG) {
            return oscar.Util.EpsgConversion.EPSG_URN + s[1];
        } else if (s[0] == oscar.Util.EpsgConversion.CRS) {
            return oscar.Util.EpsgConversion.CRS_URN + s[1];
        } else {
            return epsg;
        }
    }
};

/**
 * Add additional functionality to existing JavaScript Objects
 */

String.prototype.replaceAll = function(stringToFind, stringToReplace) {
    var temp = this;
    var index = temp.indexOf(stringToFind);
    while (index != -1) {
        
        temp = temp.replace(stringToFind, stringToReplace);
        
        index = temp.indexOf(stringToFind);
    }
    return temp;
};

/**
 * Adds the format function to the String class.
 */

String.prototype.format = function() {
    var str = this;
    for (var i = 0; i < arguments.length; i++) {
        var reg = new RegExp("\\{" + i + "\\}", "gm");
        str = str.replace(reg, arguments[i]);
    }
    return str;
}

/**
 * Prototype functions for storing objects in HTML5 storage objects.
 */

Storage.prototype.setObj = function(key, obj) {
    return this.setItem(key, JSON.stringify(obj));
}

Storage.prototype.getObj = function(key) {
    return JSON.parse(this.getItem(key))
};

/**
 * APIMethod: oscar.Util.getServerGeometryPropertyName
 * 
 * returns the geometry property name of the current server
 * 
 * Parameters: schema - {JSON} the DescribeFeatureType schema of current server
 * 
 * Returns: {String} Returns server GeometryPropertyName as a String.
 * 
 */
oscar.Util.getServerGeometryPropertyName = function(schema) {
    var elements = schema.ComplexType.ComplexContent.Extension.Sequence.Elements;
    for (var i = 0; i < elements.length; i++) {
        if (elements[i].type.indexOf("gml:") != -1) {
            return geoPropName = elements[i].name;
        }
    }
    return "";
};
/**
 * Method: oscar.Util.convertFormat
 * 
 * returns the user-friendly format text name of the current server
 * 
 * Parameters: format -(String) the format description.
 * 
 * Returns: {String} Returns the format text as a String.
 * 
 */
oscar.Util.convertFormat = function(format) {
    if (format != null || format.length > 0) {
        if (format.indexOf("gml/3") != -1 || format.indexOf("gml3") != -1) {
            return "gml3";
        } else if (format.indexOf("gml/2") != -1 || format.indexOf("gml2") != -1) {
            return "gml2";
        } else if (format.indexOf("kml") != -1) {
            return "kml"
        } else if (format.indexOf("json") != -1) {
            return "json";
        } else if (format.indexOf("tiff") != -1) {
            return "tiff";
        } else if (format.indexOf("xyz") != -1) {
            return "XYZ";
        } else if (format.indexOf("bag") != -1) {
            return "BAG";
        } else if (format.indexOf("shapefile") != -1) {
            return "ShapeFile";
        }
    }
    return format;
};

/**
 * APIMethod: oscar.alert
 * 
 * Displays an alert dialog box to the user.
 */
oscar.alert = function(props, content, callbacks) {
    var _defaults = {
        modal : true,
        resizable : false,
        closeOnEscape : false,
        show : "fade",
        hide : "fade",
        draggable : false,
        open : function(event, ui) {
            $$(this).parent().children().children(".ui-dialog-titlebar-close").hide();
        }
    }
    var options = OpenLayers.Util.extend(props, _defaults);
    var $dlg = $$("<div></div>").dialog(options).html(content);
    var $buttonPanel = $$("<div></div>").addClass("dlgButtonPanel");
    var $ok = $$("<a></a>").html("Ok").button({
        icons : {
            primary : "ui-icon-check"
        },
        text : false
    }).click(function() {
        $dlg.dialog("close");
        if (callbacks && callbacks.complete) {
            callbacks.complete.apply(callbacks.scope, [ true ]);
        }
    });
    $buttonPanel.append($ok);
    $dlg.append($buttonPanel);
}

/**
 * APIMethod: oscar.confirm
 * 
 * Displays an confirm dialog box.
 */
oscar.confirm = function(props, content, callbacks) {
    var _defaults = {
        modal : true,
        resizable : false,
        closeOnEscape : false,
        show : "fade",
        hide : "fade",
        draggable : false,
        open : function(event, ui) {
            $$(this).parent().children().children(".ui-dialog-titlebar-close").hide();
        }
    }

    var options = OpenLayers.Util.extend(props, _defaults);
    
    var $dlg = $$("<div></div>").dialog(options).html(content);
    var $buttonPanel = $$("<div></div>").addClass("dlgButtonPanel");
    var $ok = $$("<a></a>").html("Ok").button({
        icons : {
            primary : "ui-icon-check"
        },
        text : false
    }).click(function() {
        $dlg.dialog("close");
        if (callbacks && callbacks.complete) {
            callbacks.complete.apply(callbacks.scope, [ true ]);
        }
    });
    var $cancel = $$("<a></a>").html("Cancel").button({
        icons : {
            primary : "ui-icon-cancel"
        },
        text : false
    }).click(function() {
        $dlg.dialog("close");
        if (callbacks && callbacks.complete) {
            callbacks.complete.apply(callbacks.scope, [ false ]);
        }
    });
    $buttonPanel.append($ok);
    $buttonPanel.append($cancel);
    $dlg.append($buttonPanel);
}

/**
 * Method: parseText This method will take a String argument and look for URLs,
 * email links and images and return anchor or image tags.
 * 
 * Usage: oscar.Util.parseText(String)
 */

oscar.Util.parseText = function(input) {
    if (input == null)
        return "";
    
    /**
     * internal functions for parsing images, email links and urls.
     */
    var fns = {
        makeTag : function(tag) {
            return document.createElement(tag);
        },
        url : function(input) {
            if (input.indexOf("http://") == 0 || input.indexOf("https://") == 0) {
                var elem = this.makeTag("span");
                var anchor = this.makeTag("a");
                elem.appendChild(anchor);
                anchor.href = input;
                anchor.innerHTML = input;
                anchor.target = "_new";
                input = elem.innerHTML;
            }
            return input;
        },
        
        email : function(input) {
            if (input.indexOf("@") > -1) {
                var elem = this.makeTag("span");
                var anchor = this.makeTag("a");
                elem.appendChild(anchor);
                anchor.href = "mailto:" + input;
                anchor.innerHTML = input;
                input = elem.innerHTML;
            }
            return input;
        },
        image : function(input) {
            if (input.indexOf(".png") > -1 || input.indexOf(".gif") > -1 || input.indexOf(".jpg") > -1 || input.indexOf(".jpeg") > -1) {
                var elem = this.makeTag("span");
                var image = this.makeTag("img");
                elem.appendChild(image);
                image.src = input;
                input = elem.innerHTML;
            }
            return input;
        }
    
    };
    var arr = input.split(" ");
    for (var i = 0; i < arr.length; i++) {
        var index = arr[i];
        arr[i] = fns.email(arr[i]);
        arr[i] = fns.image(arr[i]);
        arr[i] = fns.url(arr[i]);
    }
    return arr.join(" ");
}

/**
 * APIMethod: getGridOffsets Returns the correct offsets from a WCS
 * DescribeCoverage operation
 */
oscar.Util.getGridOffsets = function(offsetsAsString) {
    var validOffsets = [];
    var offsets = offsetsAsString.split(" ");
    // x offset
    validOffsets.push(parseFloat(offsets[0]));
    
    // y offset
    var yOffsetIndex = (offsets.length == 2) ? 1 : 3;
    validOffsets.push(parseFloat(offsets[yOffsetIndex]));
    
    return validOffsets;
}

/**
 * APIMethod: isGeographicCRS Returns true / false if a CRS is geographic. If
 * false then the CRS is assumed to be projected
 * 
 */
oscar.Util.isGeographicCRS = function(projection) {
    return ($$.trim(projection.proj.projName) == "longlat") ? true : false;
}

/**
 * APIMethod: boundsToFeature This method taks an OpenLayers.Bounds object and
 * convert it to a feature. Parameters: - bounds <OpenLayers.Bounds> -
 * srcProjection <OpenLayers.Projection> - map <OpenLayers.Map>
 */
oscar.Util.boundsToFeature = function(bbox, srcProjection, map) {
    // multipolygons will not draw next to each other when crossing
    // the date line, they draw on opposite sides of the map.
    
    var feature = null;
    var mapMaxExtent = map.getMaxExtent();
    var featureBounds = bbox.clone();
    
    if (map.getProjectionObject().getCode() != srcProjection.getCode()) {
        featureBounds.transform(srcProjection, map.getProjectionObject());
    }
    
    if (bbox.left > bbox.right) {
        var geomA = new OpenLayers.Bounds(featureBounds.left, featureBounds.bottom, map.getMaxExtent().right, featureBounds.top).toGeometry();
        
        var geomB = new OpenLayers.Bounds(map.getMaxExtent().left, featureBounds.bottom, featureBounds.right, featureBounds.top).toGeometry();
        var multi_polygon = new OpenLayers.Geometry.MultiPolygon([ geomA, geomB ]);
        return new OpenLayers.Feature.Vector(multi_polygon);
    } else {
        if (bbox.left < 0 && featureBounds.left > 0) {
            featureBounds.left *= -1;
        }
        
        if (bbox.right < 0 && featureBounds.right > 0) {
            featureBounds.right *= -1;
        }
        
        return new OpenLayers.Feature.Vector(featureBounds.toGeometry());
    }
}

/**
 * Override the default OL pink color for broken images
 */
OpenLayers.Util.onImageLoadErrorColor = "transparent";

oscar.Util.WizardFactory = function(protocol, link, options) {
    switch (protocol) {
        case "OGC:WCS-1.1.0-http-get-capabilities":
            return new oscar.Gui.Wizard.WebCoverageServiceDownload(protocol, link.url, options);
        case "IENC":
        case "Shapefile":
        case "KML":
            // return new
            // oscar.Gui.Wizard.DirectDownload(protocol,link.url);
        default:
            return null;
    }
};

/**
 * APIMethod: oscar.Util.renderEntities Takes a string and renders the character
 * entities. Parameters: - str String object. Returns: Rendered entity string
 * 
 */
oscar.Util.renderEntities = function(str) {
    var field = document.createElement("textarea");
    field.innerHTML = str;
    return field.value;
};

oscar.Util.downloadFromService = function(url, filename, proxyOverride) {
    if (proxyOverride || oscar.DownloadHost) {
        var form = document.createElement("form");
        $$(form).css("display", "none");
        var formName = OpenLayers.Util.createUniqueID("randomForm");
        form.name = formName;
        form.action = proxyOverride || oscar.DownloadHost;
        form.method = "POST";
        var iUrl = document.createElement("input");
        iUrl.name = "url";
        iUrl.type = "text";
        iUrl.value = url;
        form.appendChild(iUrl);
        var iFilename = document.createElement("input");
        iFilename.name = "identifier";
        iFilename.type = "text";
        iFilename.value = filename;
        form.appendChild(iFilename);
        document.body.appendChild(form);
        form.submit();
    } else {
        window.open(url, "abc123", "width=640,height=480,menuBar=yes,location=false,scrollbars=yes");
    }
}

oscar.Util.getGrammarSymbol = function(grammar) {
    grammar = grammar.replace(/'/g, '');
    switch (grammar) {
        case "FIELD":
            return "FIELD"
        case "LIKE":
            return "LIKE";
        case "EQUALS":
            return "=";
        case "LT":
            return "<";
        case "LTE":
            return "<=";
        case "GT":
            return ">";
        case "GTE":
            return ">=";
        case "NEQ":
            return "<>";
        case "OPEN":
            return "(";
        case "CLOSE":
            return ")";
        case "WORD":
            return "''";
        case "AND":
            return "AND";
        case "OR":
            return "OR";
            break;
        case "EOF":
        default:
            return null;
    }
}
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.BaseClass
 * 
 * The base class of all Oscar objects.
 * 
 */
oscar.BaseClass = function() {
    var Class = function() {
        this.initialize.apply(this, arguments);

    };
    var extended = {};
    var parent;
    for (var i = 0, len = arguments.length; i < len; ++i) {

        if (typeof arguments[i] == "function") {
            parent = arguments[i].prototype;
        } else {

            parent = arguments[i];
        }
        OpenLayers.Util.extend(extended, parent);
    }
    Class.prototype = extended;
    return Class;
};

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.QueryType = new oscar.BaseClass({
    TEXT : 1,
    SPATIAL : 2,
    query : null,
    type : null,
    initialize : function(type, query, options) {
        if (this.options) {
            OpenLayers.Util.extend(this, options);
        }
        this.type = type;
        this.query = query;
    },
    toFilter : function() {
        switch (this.type) {
            case 1:
                return this._buildTextualFilter();
            case 2:
                return this._buildSpatialFilter();
            default:
                return null;
        }
    },
    _buildTextualFilter : function() {
        var formatter = new OpenLayers.Format.CQL();
        return formatter.read(this.query);
    },
    _buildSpatialFilter : function() {
        var spatial_filter = new OpenLayers.Filter.Comparison({
            type : OpenLayers.Filter.Spatial.BBOX,
            property : "ows:BoundingBox",
            value : this.query.spatial,
            projection : this.query.projection
        });
        return spatial_filter;
    },
    CLASS_NAME : "oscar.Query"
});

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.ox
 * 
 * Namespace object for Oscar eXchange object.
 */
oscar.ox = oscar.BaseClass({
    /**
     * Property: version
     */
    version : null,

    /**
     * Constructor: oscar.ox
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        OpenLayers.Util.extend(this, options);
        this.themes = [];
    },

    /**
     * Method: addTheme
     * 
     * Parameters: oxTheme - {<oscar.ox.Theme>} a theme object.
     */
    addTheme : function(oxTheme) {
        this.themes.push(oxTheme);
    },
    /**
     * Constant: CLASS_NAME - oscar.ox
     */
    CLASS_NAME : "oscar.ox"
});

/**
 * Class: oscar.ox.Theme
 * 
 * This class represents a Theme from Oscar eXchange Format
 * 
 */
oscar.ox.Theme = oscar.BaseClass({
    /**
     * APIProperty: layers
     */
    layers : null,
    /**
     * APIProperty: selectionLayers
     */
    selectionLayers : null,
    /**
     * APIProperty: displayOrder
     */
    displayOrder : -1,
    /**
     * Property: id
     */
    id : 0,
    /**
     * APIProperty: name
     */
    name : null,
    /**
     * APIProperty: srs
     */
    srs : null,
    /**
     * APIProperty: projection
     */
    projection : null,
    /**
     * APIProperty: isActive
     */
    isActive : false,
    /**
     * APIProperty: services
     */
    services : null,
    /**
     * APIProperty: covers
     */
    covers : null,
    /**
     * Constructor: oscar.ox.Theme
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        this.layers = [];
        this.selectionLayers = [];
        this.covers = [];
        OpenLayers.Util.extend(this, options);
    },
    /**
     * APIMethod: addLayer
     * 
     * Adds an {<oscar.ox.Layer>} to the theme object.
     * 
     * Parameters: oxLayer - {<oscar.ox.Layer>} the layer to be added.
     */
    addLayer : function(oxLayer) {
        this.layers.push(oxLayer);
        if (oxLayer.layerType == "SELECTION")
            this.selectionLayers.push(oxLayer);
    },
    /**
     * APIMethod: hasSelectionService
     * 
     * Returns: {Boolean} whether a selection service is available.
     */
    hasSelectionService : function() {
        return (this.services && this.services.selection) ? true : false;
    },
    /**
     * APIMethod: getSelectionService
     * 
     * Returns: {Object} *serviceEntries* the selection service.
     */
    getSelectionService : function() {
        if (this.hasSelectionService())
            return this.services.selection.serviceEntries;
        else
            return null;
    },
    /**
     * APIMethod: getSelectionService
     * 
     * Returns: {Array} *oscar.ogc.CatalogueService* An array of catalogue
     * services.
     */
    getCatalogueService : function() {
        if (this.hasCatalogueService()) {
            var services = [];
            for (var i = 0; i < this.services.catalogue.serviceEntries.length; i++) {
                services.push(new oscar.ogc.CatalogueService(this.services.catalogue.serviceEntries[i]));
            }
            return services;
        } else {
            return [];
        }
    },
    /**
     * APIMethod: hasCatalogueService
     * 
     * Returns: {Boolean} whether a catalogue service is available.
     */
    hasCatalogueService : function() {
        return (this.services && this.services.catalogue) ? true : false;
    },
    /**
     * APIMethod: hasExtractionService
     * 
     * Returns: {Boolean} whether an extraction service is available.
     */
    hasExtractionService : function() {
        return (this.services && this.services.extraction) ? true : false;
    },
    /**
     * APIMethod: getExtractionService
     * 
     * Returns: {Object} *serviceEntries* the extraction service.
     */
    getExtractionService : function() {
        if (this.hasExtractionService())
            return this.services.extraction.serviceEntries;
        else
            return null;
    },
    getProjection : function() {
        return this.projection;
    },
    /**
     * APIMethod: getDefaultCover
     * 
     * Returns: {Object} a default cover or null.
     */
    getDefaultCover : function() {
        return this.getCover("DEFAULT");
    },

    /**
     * APIMethod: getMaxCover
     * 
     * Returns: {Object} the max extent cover.
     */
    getMaxCover : function() {
        return this.getCover("MAX");
    },
    /**
     * Method: getCover
     * 
     * Parameters: type - {String} Type of cover: MAX or DEFAULT
     * 
     * Returns: {<OpenLayers.Bounds>} the cover object or null.
     */
    getCover : function(type) {
        for (var i = 0; i < this.covers.length; i++) {
            if (this.covers[i].type == type) {
                return new OpenLayers.Bounds(this.covers[i].minX, this.covers[i].minY, this.covers[i].maxX, this.covers[i].maxY)
            }
        }
        return null;
    },

    /**
     * Method: buildMap Calls the toMapLayer for each of the layers in the theme
     * to be added to the map.
     * 
     * Parameters: map - OpenLayers.Map
     */
    buildMap : function(map) {
        if (this.displaySRS) {
            try {
                var mPosition = map.getControlsByClass("OpenLayers.Control.MousePosition")[0];
                mPosition.displayProjection = new OpenLayers.Projection(this.displaySRS);

            } catch (err) {

            }
        }

        var numLayers = this.layers.length;
        for (var lIndex = 0; lIndex < numLayers; lIndex++) {
            var tLayer = this.layers[lIndex];
            if (!tLayer.baseLayer) {
                tLayer.baseLayer = (lIndex == 0) ? true : false;
            }
            var objType = tLayer.toMapLayer(map);

        }

    },
    /**
     * Constant: CLASS_NAME - oscar.ox.Theme
     */
    CLASS_NAME : "oscar.ox.Theme"
});

/**
 * Class: oscar.ox.Layer
 * 
 * This class represents a Layer from Oscar eXchange Format.
 * 
 */
oscar.ox.Layer = oscar.BaseClass({
    /**
     * APIProperty: baseLayer
     */
    baseLayer : false,
    /**
     * APIProperty: displayOrder
     */
    displayOrder : null,
    /**
     * APIProperty: dataLayers
     */
    dataLayers : null,
    /**
     * APIProperty: layerId
     */
    layerId : null,
    /**
     * APIProperty: layerType
     */
    layerType : null,
    /**
     * APIProperty: name
     */
    name : null,
    /**
     * APIProperty: parameters
     */
    parameters : null,
    /**
     * APIProperty: urls
     */
    urls : null,
    /**
     * Constructor: oscar.ox.Layer
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        this.dataLayers = [];
        this.parameters = {};
        this.urls = [];
        OpenLayers.Util.extend(this, options);
    },
    /**
     * APIMethod: addDataLayer Adds an {<oscar.ox.DataLayer>} object to the
     * dataLayers array.
     * 
     * Parameters: dataLayer - an {<oscar.ox.DataLayer>} object.
     */
    addDataLayer : function(dataLayer) {
        if (!this.dataLayers)
            this.dataLayers = [];

        this.dataLayers.push(dataLayer);
    },
    /**
     * APIMethod: addUrl Adds a url source to the urls object.
     * 
     * Parameters: url - {String} a url String.
     * 
     */
    addUrl : function(url) {
        if (!this.urls)
            this.urls = [];
        this.urls.push(url)
    },
    /**
     * APIMethod: toMapLayer
     * 
     * Returns: {Object} *olLayer* which inherits from <OpenLayers.Layer> and is
     * able to be added to an <OpenLayers.Map> object.
     */
    toMapLayer : function(map) {
        layerNamesArr = [];
        for (var i = 0; i < this.dataLayers.length; i++) {
            layerNamesArr.push(this.dataLayers[i].layerName);
        }
        var olLayer = null;
        var onEvents = null;
		var aliasproj = new OpenLayers.Projection("EPSG:3857");
        switch (this.layerType) {
		case "GOOGLE_PHYSICAL":
			olLayer = new OpenLayers.Layer.Google("Google Physical", {
				type :google.maps.MapTypeId.TERRAIN,
				sphericalMercator :true
			});
			olLayer.projection = aliasproj;
			break;
		case "GOOGLE_STREETS":
			olLayer = new OpenLayers.Layer.Google("Google Streets", {
				sphericalMercator :true
			});
			olLayer.projection = aliasproj;
			break;

		case "GOOGLE_HYBRID":
			olLayer = new OpenLayers.Layer.Google("Google Hybrid", {
				type :google.maps.MapTypeId.HYBRID,
				sphericalMercator :true
			});
			olLayer.projection = aliasproj;
			break;
		case "GOOGLE_SATELLITE":
			olLayer = new OpenLayers.Layer.Google("Google Satellite", {
				type :google.maps.MapTypeId.SATELLITE,
				sphericalMercator :true
			});
			olLayer.projection = aliasproj;
			break;
        case "MARKER":
            var externalProjection = null;
            if (this.parameters.srs) {
                externalProjection = new OpenLayers.Projection(this.parameters.srs);
            } else {
                externalProjection = new OpenLayers.Projection("EPSG:4326");
            }
            var olLayer = null;
            var layerOptions = {
                projection : map.getProjectionObject(),
                formatOptions : {
                    externalProjection : externalProjection,
                    internalProjection : map.getProjectionObject()
                }
            };
            var formatType = this.parameters.formatType.toLowerCase();
            switch (formatType) {

            case "georss_simple":
                var style = new OpenLayers.Style({
                    externalGraphic : this.parameters.iconUri
                });
                var rule = new OpenLayers.Rule({
                    symbolizer : {
                        pointRadius : 30
                    },
                    filter : new OpenLayers.Filter.Comparison({
                        type : OpenLayers.Filter.Comparison.LIKE,
                        property : "title",
                        value : "*powder*"
                    })
                });
                rule.filter.value2regex("*");

                var elseRule = new OpenLayers.Rule({
                    elseFilter : true,
                    symbolizer : {
                        pointRadius : 20
                    }
                });

                style.addRules([ elseRule ]);

                olLayer = null;

                olLayer = new OpenLayers.Layer.Vector(this.name, {
                    setMap : function() {
                        OpenLayers.Layer.Vector.prototype.setMap.apply(this, arguments);
                        this.protocol.format.internalProjection = this.map.projection;
                    },
                    strategies : [ new OpenLayers.Strategy.Fixed() ],
                    protocol : new OpenLayers.Protocol.HTTP({
                        url : this.urls[0],
                        format : new OpenLayers.Format.GeoRSS({
                            extractStyles : false,
                            externalProjection : new OpenLayers.Projection("EPSG:4326"),
                            createFeatureFromItem : function(item) {
                                var feature = OpenLayers.Format.GeoRSS.prototype.createFeatureFromItem.apply(this, arguments);
                                return feature;
                            }
                        })
                    }),
                    styleMap : new OpenLayers.StyleMap({
                        "default" : style,
                        "select" : new OpenLayers.Style({
                            pointRadius : 35
                        })
                    })
                });

                olLayer.events.on({
                    "featureselected" : function(evt) {
                        var feature = evt.feature;
                        var pHolder = document.createElement("div");
                        var header = document.createElement("div");
                        header.innerHTML = feature.attributes['title'];
                        var p = document.createElement("p");
                        p.innerHTML = oscar.Util.parseText(feature.attributes['description']);
                        pHolder.appendChild(header);
                        pHolder.appendChild(p);
                        html = p.innerHTML;
                        var popup = new oscar.FramedCloud("id", feature.geometry.getBounds().getCenterLonLat(), null, pHolder.innerHTML, null, true);
                        popup.autoSize = true;
                        feature.popup = popup;
                        feature.layer.map.addPopup(popup);
                    },
                    "featureunselected" : function(evt) {
                        var feature = evt.feature;
                        feature.layer.map.removePopup(feature.popup);
                        feature.popup = null;
                    },

                    scope : this
                });

                break;
            case "kml":
                olLayer = null;
                var style = new OpenLayers.Style({
                    externalGraphic : this.parameters.iconUri,
                    pointRadius : 20
                });
                olLayer = new OpenLayers.Layer.Vector(this.name, {
                    setMap : function() {
                        OpenLayers.Layer.Vector.prototype.setMap.apply(this, arguments);
                        this.protocol.format.internalProjection = this.map.projection;
                    },
                    strategies : [ new OpenLayers.Strategy.Fixed() ],
                    protocol : new OpenLayers.Protocol.HTTP({
                        url : this.urls[0],
                        format : new OpenLayers.Format.KML({
                            extractStyles : true,
                            externalProjection : new OpenLayers.Projection("EPSG:4326")
                        })
                    }),
                    styleMap : new OpenLayers.StyleMap({
                        "default" : style,
                        "select" : new OpenLayers.Style({
                            pointRadius : 35
                        })
                    })
                });

                olLayer.events.on({
                    "beforefeaturesadded" : function(evt) {
                        for ( var f in evt.features) {
                            var feat = evt.features[f];
                            if (feat.geometry.CLASS_NAME != "OpenLayers.Geometry.Point") {
                                var lonlat = feat.geometry.getBounds().getCenterLonLat();
                                var point = new OpenLayers.Geometry.Point(lonlat.lon, lonlat.lat);
                                feat.geometry = point;
                            }
                        }
                    },
                    "featureselected" : function(evt) {
                        var feature = evt.feature;
                        var pHolder = document.createElement("div");
                        var header = document.createElement("div");
                        header.innerHTML = oscar.Util.parseText(feature.attributes['name']);
                        var p = document.createElement("p");
                        p.innerHTML = oscar.Util.parseText(feature.attributes['description']);
                        pHolder.appendChild(header);
                        pHolder.appendChild(p);

                        html = oscar.Util.parseText(feature.attributes['description']);
                        var popup = new oscar.FramedCloud("id", feature.geometry.getBounds().getCenterLonLat(), null, pHolder.innerHTML, null, true);
                        popup.autoSize = true;
                        feature.popup = popup;
                        feature.layer.map.addPopup(popup);
                    },
                    "featureunselected" : function(evt) {
                        var feature = evt.feature;
                        feature.layer.map.removePopup(feature.popup);
                        feature.popup = null;
                    },

                    scope : this
                });
                break;
            case "youtube":
                onEvents = {
                    "featureselected" : function(evt) {
                        var feature = evt.feature;
                        var pHolder = document.createElement("div");
                        var header = document.createElement("div");
                        header.innerHTML = feature.attributes['title'];
                        var youtubeString = 'http://www.youtube.com/embed/';
                        var fid = feature.fid.substring(feature.fid.lastIndexOf("/") + 1);
                        html = '<iframe width="480" height="390" src="' + youtubeString + fid + '" frameborder="0" allowfullscreen></iframe>';
                        var p = document.createElement("p");
                        p.innerHTML = html;
                        pHolder.appendChild(header);
                        pHolder.appendChild(p);

                        html = feature.attributes['description'];
                        var popup = new oscar.FramedCloud("id", feature.geometry.getBounds().getCenterLonLat(), null, pHolder.innerHTML, null, true);
                        popup.autoSize = true;
                        feature.popup = popup;
                        feature.layer.map.addPopup(popup);
                    },
                    "featureunselected" : function(evt) {
                        var feature = evt.feature;
                        feature.layer.map.removePopup(feature.popup);
                        feature.popup = null;
                    },
                    scope : this
                };
            case "flickr":
                if (onEvents == null) {
                    onEvents = null;
                }
            case "picasa":
                var style = new OpenLayers.Style({
                    externalGraphic : "${thumbnail}"
                });
                var rule = new OpenLayers.Rule({
                    symbolizer : {
                        pointRadius : 30
                    },
                    filter : new OpenLayers.Filter.Comparison({
                        type : OpenLayers.Filter.Comparison.LIKE,
                        property : "title",
                        value : "*powder*"
                    })
                });
                rule.filter.value2regex("*");

                var elseRule = new OpenLayers.Rule({
                    elseFilter : true,
                    symbolizer : {
                        pointRadius : 20
                    }
                });

                style.addRules([ elseRule ]);

                olLayer = new OpenLayers.Layer.Vector(this.name, {
                    setMap : function() {
                        OpenLayers.Layer.Vector.prototype.setMap.apply(this, arguments);
                        this.protocol.format.internalProjection = this.map.projection;
                    },
                    strategies : [ new OpenLayers.Strategy.Fixed() ],
                    protocol : new OpenLayers.Protocol.HTTP({
                        url : this.urls[0],
                        format : new OpenLayers.Format.GeoRSS({
                            externalProjection : new OpenLayers.Projection("EPSG:4326"),
                            createFeatureFromItem : function(item) {
                                var feature = OpenLayers.Format.GeoRSS.prototype.createFeatureFromItem.apply(this, arguments);
                                var thumbnail = this.getElementsByTagNameNS(item, "*", "thumbnail")[0].getAttribute("url");
                                feature.attributes.thumbnail = thumbnail;
                                return feature;
                            }
                        })
                    }),
                    styleMap : new OpenLayers.StyleMap({
                        "default" : style,
                        "select" : new OpenLayers.Style({
                            pointRadius : 35
                        })
                    })
                });

                if (onEvents == null) {
                    onEvents = {
                        "featureselected" : function(evt) {
                            var feature = evt.feature;
                            html = feature.attributes['description'];
                            var popup = new oscar.FramedCloud("id", feature.geometry.getBounds().getCenterLonLat(), null, html, null, true);
                            popup.autoSize = true;
                            feature.popup = popup;
                            feature.layer.map.addPopup(popup);
                        },
                        "featureunselected" : function(evt) {
                            var feature = evt.feature;
                            feature.layer.map.removePopup(feature.popup);
                            feature.popup = null;

                        },

                        scope : this
                    };
                }
                olLayer.events.on(onEvents);
                break;
            }
            break;
        case "OSM":
            olLayer = new OpenLayers.Layer.OSM(this.name);
			olLayer.projection = aliasproj;
            break;
        case "WMS":
            var params = {};
            params.layers = layerNamesArr;
            params.format = (this.format) ? this.format : "image/png";
            params.transparent = true;
            if (this.parameters.version)
                params.version = this.parameters.version;
            olLayer = new OpenLayers.Layer.WMS(this.name, this.urls, params, {
                wrapDateLine : false,
                buffer : 0,
                isBaseLayer : this.baseLayer
            });
            break;
        case "WMTS":
            olLayer = new OpenLayers.Layer.WMTS({
                name : this.name,
                url : this.urls[0],
                layer : this.dataLayers[0].layerName,
                matrixSet : this.tileMatrixSet,
                tileOrigin : new OpenLayers.LonLat(this.tileOrigin[0], this.tileOrigin[1]),
                tileFullExtent : new OpenLayers.Bounds(this.tileFullExtent[0], this.tileFullExtent[1], this.tileFullExtent[2], this.tileFullExtent[3]),
                style : "default",
                requestEncoding : "REST",
                format : this.format,
                isBaseLayer : this.baseLayer,
                buffer : 0
            });
            break;
        };

        if (olLayer != null) {
            map.addLayer(olLayer);
        }
    },
    /**
     * Constant: CLASS_NAME - oscar.ox.Layer
     */
    CLASS_NAME : "oscar.ox.Layer"
});

/**
 * Class: oscar.ox.DataLayer
 * 
 * This class represents a DataLayer from Oscar eXchange Format.
 */
oscar.ox.DataLayer = oscar.BaseClass({
    /**
     * APIProperty: id
     */
    id : null,
    /**
     * APIProperty: index
     */
    index : null,
    /**
     * APIProperty: layerName
     */
    layerName : null,
    /**
     * Constructor: oscar.ox.DataLayer
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        OpenLayers.Util.extend(this, options);
    },

    /**
     * Constant: CLASS_NAME - oscar.ox.DataLayer
     */
    CLASS_NAME : "oscar.ox.DataLayer"
});

/**
 * Class: oscar.ox.Services
 */
oscar.ox.Services = oscar.BaseClass({
    /**
     * Property: selection
     */
    selection : null,
    /**
     * Property: extraction
     */
    extraction : null,

    catalogue : null,

    /**
     * Constructor: oscar.ox.Services
     */
    initialize : function() {

    },
    /**
     * Method: addSelectionService
     * 
     * returns: {Object} the newly created selection service.
     */
    addSelectionService : function() {
        this.selection = new oscar.ox.SelectionService();
        return this.selection;
    },
    /**
     * Method: addExtractionService
     * 
     * returns: {Object} the newly created extraction service.
     */
    addExtractionService : function() {
        this.extraction = new oscar.ox.ExtractionService();
        return this.extraction;
    },
    /**
     * Method: addCatalogueService
     * 
     * returns: {Object} the newly created catalogue service.
     */
    addCatalogueService : function() {
        this.catalogue = new oscar.ox.CatalogueService();
        return this.catalogue;
    },

    /**
     * Constant: CLASS_NAME - oscar.ox.Services
     */
    CLASS_NAME : "oscar.ox.Services"
});

/**
 * Class: oscar.ox.Service
 * 
 * Represents a type of Service containing an array of ServiceEntry objects.
 */
oscar.ox.Service = oscar.BaseClass({

    /**
     * APIProperty: serviceEntries
     * 
     * An array of <oscar.ox.ServiceEntry> objects.
     */
    serviceEntries : null,

    /**
     * Constructor: oscar.ox.Service
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        this.serviceEntries = [];
    },

    /**
     * Method: addServiceEntry
     * 
     * Parameters: serviceEntry - {oscar.ox.ServiceEntry} object.
     */
    addServiceEntry : function(serviceEntry) {
        if (serviceEntry.CLASS_NAME && serviceEntry.CLASS_NAME == "oscar.ox.ServiceEntry")
            this.serviceEntries.push(serviceEntry.clone());
    },
    /**
     * APIMethod: getServiceEntries
     * 
     * Returns an array of service entry objects.
     * 
     * Returns: {Array} - An array of <oscar.ox.ServiceEntry> objects.
     */
    getServiceEntries : function() {
        return this.serviceEntries
    },

    /**
     * APIMethod: getServiceEntry
     * 
     * Returns a specified service entry.
     * 
     * Parameters: index - {Number} The index of the <oscar.ox.ServiceEntry> to
     * return.
     * 
     * Returns: {oscar.ox.ServiceEntry} - A ServiceEntry object.
     */
    getServiceEntry : function(index) {
        return this.serviceEntries[index];
    },
    /**
     * Constant: CLASS_NAME - oscar.ox.Service
     */
    CLASS_NAME : "oscar.ox.Service"
});

/**
 * Class: oscar.ox.SelectionService
 * 
 * Represents the selection service from Oscar eXchange Format.
 * 
 * Inherits from: <oscar.ox.Service>
 */
oscar.ox.SelectionService = oscar.BaseClass(oscar.ox.Service, {
    /**
     * Constructor: oscar.ox.SelectionService
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        oscar.ox.Service.prototype.initialize.apply(this, [ options ]);
    },
    /**
     * Constant: CLASS_NAME - oscar.ox.SelectionService
     */
    CLASS_NAME : "oscar.ox.SelectionService"
});

oscar.ox.CatalogueService = oscar.BaseClass(oscar.ox.Service, {
    /**
     * Constructor: oscar.ox.SelectionService
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        oscar.ox.Service.prototype.initialize.apply(this, [ options ]);
    },
    /**
     * Constant: CLASS_NAME - oscar.ox.CatalogueService
     */
    CLASS_NAME : "oscar.ox.CatalogueService"
});

/**
 * Class: oscar.ox.ExtractionService
 * 
 * Represents the extraction service from Oscar eXchange Format.
 * 
 * Inherits from: <oscar.ox.Service>
 */
oscar.ox.ExtractionService = oscar.BaseClass(oscar.ox.Service, {
    /**
     * Constructor: oscar.ox.ExtractionService
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        oscar.ox.Service.prototype.initialize.apply(this, [ options ]);
    },
    /**
     * Constant: CLASS_NAME - oscar.ox.ExtractionService
     */
    CLASS_NAME : "oscar.ox.ExtractionService"
});

/**
 * Class: oscar.ox.ServiceEntry
 * 
 * An object containing connection information for a service.
 * 
 */
oscar.ox.ServiceEntry = oscar.BaseClass({
    /**
     * APIProperty: url
     * 
     * {String} The URL of the service.
     */
    url : null,
    /**
     * APIProperty: version
     * 
     * {String} The version of the service.
     */
    version : null,
    /**
     * APIProperty: serviceType
     * 
     * {String} Type of the service.
     */
    serviceType : null,
    /**
     * APIProperty: format
     * 
     * {String}
     */
    format : null,
    /**
     * APIProperty: identifiers
     * 
     * {Array{String}}
     * 
     */
    identifiers : null,
    /**
     * APIProperty: geometryName
     * 
     * {String} the geometry property name of the current server.
     */
    geometryName : null,
    /**
     * Constructor: oscar.ox.ServiceEntry
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        this.identifiers = [];
        this.geometryName = "";
        if (options) {
            OpenLayers.Util.extend(this, options);
        }
    },

    addUrl : function(url) {
        this.url = url;
    },

    /**
     * Method: clone
     */
    clone : function() {
        var props = {};
        OpenLayers.Util.extend(props, this);
        return new oscar.ox.ServiceEntry(props);
    },
    /**
     * Constant: CLASS_NAME - oscar.ox.ServiceEntry
     */
    CLASS_NAME : "oscar.ox.ServiceEntry"
});

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Gui
 * 
 * The Oscar GUI package offers a set of utilities for building richly
 * interactive web applications.This oscar.Gui object defines the Oscar GUI
 * namespace.
 * 
 */

oscar.Gui = oscar.BaseClass({
    /**
     * APIProperty: EVENT_TYPES
     * 
     */
    EVENT_TYPES : [ "afterDraw", "afterAppend", "beforeDraw" ],
    /**
     * APIProperty: events
     */
    events : null,
    /**
     * Property: div
     * 
     * The div element of the GUI object.
     */
    div : null,
    initialize : function(options) {
        OpenLayers.Util.extend(this, options);
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
    },
    /**
     * APIMethod: draw
     * 
     * Creates the div element for the GUI Class
     */
    draw : function() {
        this.div = document.createElement("div");
        this.div.id = OpenLayers.Util.createUniqueID(this.CLASS_NAME);
        var className = this.CLASS_NAME.replace(/\./g, "");
        oscar.jQuery(this.div).addClass(className);
    },
    /**
     * APIMethod: appendTo
     * 
     * Appends the div to a parent container.
     */
    appendTo : function(parent) {
        if (this.div == null) {
            this.draw();
        }
        $$(parent).append(this.div);
    },

    /**
     * Constant: CLASS_NAME
     */
    CLASS_NAME : "oscar.Gui"
});

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Gui.CatalogueResults
 * 
 * Inherits from: - <OpenLayers.Gui>
 * 
 */

oscar.Gui.CatalogueResults = new oscar.BaseClass(oscar.Gui, {
    EVENT_TYPES : [ "next", "previous", "jumpTo", "recordFocus" ],
    features : [],
    map : null,
    catalogueServices : null,
    searchHandler : null,
    resultsList : null,
    buttons : [],
    initialize : function(options) {
        oscar.Gui.prototype.initialize.apply(this, [ options ]);
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
        this.appendTo(this.parent);
        if (this.searchHandler) {
            this.searchHandler.events.on({
                "success" : this.showResults,
                "beforeSearch" : this.preSearch,
                "afterSearch" : this.postSearch,
                scope : this
            });
            this.events.on({
                "next" : this.searchHandler.next,
                "previous" : this.searchHandler.previous,
                "jumpTo" : this.searchHandler.jumpTo,
                scope : this.searchHandler
            });
        }
    },
    draw : function() {
        oscar.Gui.prototype.draw.apply(this);
        var scope = this;
    },
    _buildLayout : function() {
        this.clearResults();
        var build_ele = function(name, classArray) {
            var element = $$("<" + name + "></" + name + ">");
            if (classArray) {
                for (var i = 0; i < classArray.length; i++) {
                    element.addClass(classArray[i]);
                }
            }
            return element;
        };
        /**
         * This panel is used to display initial results from the query
         */
        var styleMap = {
            "height" : "100%",
            "width" : "100%",
            "backgroundColor" : "#fff",
            "position" : "absolute",
            "top" : "0px",
            "left" : "0px"
        };
        this.results_panel = build_ele("div", [ "results-panel" ]);
        this.results_panel.css(styleMap);
        this.modes_panel = build_ele("div", [ "modes-panel" ]);
        this.modes_panel.hide();
        this.modes_panel.css(styleMap);
        /**
         * These panels are used to facilitate in the display of the query
         * results
         */
        var pagination_panel = build_ele("div", [ "ui-layout-north" ]);
        var results_list_panel = build_ele("div", [ "ui-layout-center" ]);
        
        $$(this.div).append(this.modes_panel);
        $$(this.div).append(this.results_panel);
        
        this.results_panel.append(pagination_panel);
        this.results_panel.append(results_list_panel);
        this.layout = this.results_panel.layout({
            spacing_open : 5,
            spacing_closed : 5,
            resizable : false,
            north : {
                closable : true,
            },
            center : {},
            center__showOverflow : true,
        });
        
        this.$searchInfo = $$("<div>&nbsp;</div>");
        this.$previous = $$("<button>Previous</button>");
        this.$next = $$("<button>Next</button>");
        
        this.$buttons = $$("<div></div>");
        this.$buttons.css({
            "position" : "relative"
        });
        this.$buttons.append(this.$searchInfo);
        this.$next.button({
            icons : {
                primary : "ui-icon-oux-next"
            },
            text : false,
            disabled : true
        }).click($$.proxy(function() {
            this.events.triggerEvent("next");
        }, this));
        
        this.$next.css({
            "position" : "absolute",
            "right" : "0px",
            "top" : "0px"
        });
        
        this.$previous.button({
            icons : {
                primary : "ui-icon-oux-prev"
            },
            text : false,
            disabled : true
        }).click($$.proxy(function() {
            this.events.triggerEvent("previous");
        }, this));
        
        this.$previous.css({
            "position" : "absolute",
            "left" : "0px",
            "top" : "0px"
        });
        
        this.$searchInfo.addClass("searchResultInformation");
        this.$buttons.append(this.$next);
        this.$buttons.append(this.$previous);
        pagination_panel.append(this.$buttons);
        this.$buttons.css("visibility", "hidden");
        
        this.results = $$("<div></div>");
        
        results_list_panel.append(this.results);
        
        $$(window).resize($$.proxy(function() {
            this.layout.resizeAll();
        }, this));
    },
    clearResults : function() {
        // should we clear the results here?
        // do we need to cycle through the plugins to close any "modes"
        $$(this.div).empty();
        var feat_layer = map.getLayersByName("results")[0];
        if (feat_layer) {
            feat_layer.removeAllFeatures();
        }
    },
    showResults : function(results) {
        this._buildLayout();
        this.layout.resizeAll();
        var records = results.records;
        
        this.showSearchInfo(results.SearchResults);
        
        if (records.length == 0) {
            this.$searchInfo.html(oscar.i18n("map.information.no.records.found"));
            return;
        }
        var features_array = [];
        for ( var r in records) {
            var record = records[r];
            if (record.bounds) {
                var recordProjection = record.projection || "EPSG:4326";
                var crs = oscar.Util.CoordinateReferences.getReference(recordProjection);
                
                var projection = new OpenLayers.Projection(crs.code);
                var bbox = record.bounds.toArray(oscar.Util.isGeographicCRS(projection))

                if (oscar.Util.isGeographicCRS(projection)) {
                    record.bounds = OpenLayers.Bounds.fromArray(bbox);
                }
                
                projection = new OpenLayers.Projection("EPSG:4326");
                
                var feature = oscar.Util.boundsToFeature(record.bounds, projection, this.map);
                record.feature = feature.clone();
                features_array.push(record.feature);
            }
            this.addRecordToResultList(record);
        }
        
        features_array.sort(function(feature_a, feature_b) {
            if (feature_a.geometry.getArea() < feature_b.geometry.getArea()) {
                return 1;
            } else if (feature_a.geometry.getArea() > feature_b.geometry.getArea()) {
                return -1;
            } else {
                return 0;
            }
        });
        
        this.renderFeaturesToMap(features_array);
        this.results.slimScroll({
            height : "100%",
            scrollTo : 0
        });
        $$(window).resize($$.proxy(function() {
            this.results.slimScroll({
                height : "100%",
                scrollTo : 0
            });
        }, this));
    },
    showSearchInfo : function(info) {
        var matched = info.numberOfRecordsMatched;
        var returned = info.numberOfRecordsReturned;
        var next = info.nextRecord;
        this.$buttons.css("visibility", "visible");
        
        var str = "";
        var start = (info.nextRecord != 0) ? info.nextRecord - info.numberOfRecordsReturned : 1;
        if (next > 0) {
            start = next - returned;
        } else {
            start = matched - returned + 1;
        }
        str += start;
        to = start + returned - 1;
        if (to > info.numberOfRecordsMatched) {
            to = info.numberOfRecordsMatched
        }
        
        str += " - " + to;
        str += " of " + info.numberOfRecordsMatched;
        
        if (to != matched) {
            this.$next.button("enable");
        } else {
            this.$next.button("disable");
        }
        
        if (start != 1) {
            this.$previous.button("enable");
        } else {
            this.$previous.button("disable");
        }
        
        var select = $$("<select></select>").change($$.proxy(function() {
            this.events.triggerEvent("jumpTo", select.val());
        }, this));
        
        var numPages = Math.ceil(matched / 10);
        var startAt = 1;
        var endAt = (matched > 10) ? 10 : matched;
        for (var i = 1; i <= numPages; i++) {
            var option = $$("<option></option>");
            option.val(startAt);
            if (startAt == start) {
                option.attr("selected", "selected");
            }
            var string = "";
            string += startAt + " - " + endAt;
            string += " of " + info.numberOfRecordsMatched;
            startAt += 10;
            endAt += 10;
            if (endAt > info.numberOfRecordsMatched) {
                endAt = info.numberOfRecordsMatched;
            }
            option.html(string);
            select.append(option);
        }
        this.$searchInfo.append(select);
    },
    enablePagination : function(info) {
        
    },
    postSearch : function() {
        $$(this.div).fadeTo("fast", 1);
        this._addFeatureLayer();
        this._addSelectFeature();
    },
    preSearch : function() {
        var selFeatures = this.map.getControlsByClass("OpenLayers.Control.SelectFeature");
        for (var i = 0; i < selFeatures.length; i++) {
            selFeatures[i].deactivate();
        }
        
        if (this.activePlugin) {
            this.activePlugin.destroy();
        }
        $$(this.div).fadeTo("fast", 0.2);
    },
    
    _addFeatureLayer : function() {
        var feat_layer = this.map.getLayersByName("results")[0];
        
        if (!feat_layer) {
            var styleMap = new OpenLayers.StyleMap({
                "default" : new OpenLayers.Style({
                    fillColor : "#0f0f0f",
                    fillOpacity : 0.01,
                    strokeWidth : 0,
                    strokeColor : "#0f0f0f",
                    strokeOpacity : 0.3
                }),
                "select" : new OpenLayers.Style({
                    strokeColor : "#0f0f0f",
                    fillColor : "#0f0f0f",
                    fillOpacity : 0.01,
                    fillOpacity : 0.01,
                    strokeOpacity : 0.3,
                    strokeWidth : 3
                })
            });
            
            feat_layer = new OpenLayers.Layer.Vector("results", {
                styleMap : styleMap,
                renderers : [ 'Canvas', 'VML' ],
                wrapDateLine : true,
                projection : map.getProjectionObject()
            });
            var scope = this;
            feat_layer.events.on({
                "featureselected" : function(event) {
                    var feature = event.feature;
                    if (feature.record_div) {
                        scope.overRecord($$(feature.record_div).data("record"));
                    }
                },
                
                "featureunselected" : function(event) {
                    var feature = event.feature;
                    if (feature.record_div) {
                        scope.blurRecord($$(feature.record_div).data("record"));
                    }
                }
            });
            map.addLayer(feat_layer);
        }
    },
    _addSelectFeature : function() {
        var selectFeature = this.map.getControlsByClass("OpenLayers.Control.SelectFeature")[0];
        if (!selectFeature) {
            var scope = this;
            selectFeature = new OpenLayers.Control.SelectFeature(this.map.getLayersByName("results")[0], {
                click : true,
                hover : true,
                callbacks : {
                    "click" : $$.proxy(function(feature) {
                        if (feature.record_div) {
                            this.focusRecord($$(feature.record_div).data("record"));
                        }
                    }, scope)
                }
            });
            map.addControl(selectFeature);
        }
        selectFeature.activate();
        selectFeature.handlers.feature.stopDown = false;
    },
    renderFeaturesToMap : function(features) {
        var feat_layer = this.map.getLayersByName("results")[0];
        feat_layer.addFeatures(features);
        map.zoomToExtent(feat_layer.getDataExtent());
    },
    addRecordToResultList : function(record) {
        var $result = $$("<div class='result'></div");
        $result.css("position", "relative");
        var $title = $$("<div class='title'></div>");
        var $abs = $$("<div class='abs'></div>");
        $result.append($title);
        $result.append($abs);
        var title = cswget.title(record) || cswget.identifier(record);
        var abs = cswget.abs(record);
        $title.html(title);
        $title.attr("title", title);
        $abs.html(abs);
        $abs.attr("title", abs);
        this.results.append($result);
        $result.data("record", record);
        record.container = $result;
        if (record.feature) {
            if (!record.feature.record_div) {
                record.feature.record_div = $result;
            }
        }
        /*
         * Display the link data here. We should create wizards to handle the
         * different protocol cases.
         */
        $result.append(this.createLinksPanel(record));
        
        this.attachMouseEvents($result, record);
        
    },
    createLinksPanel : function(record) {
        var $div = $$("<div></div>");
        
        if (record.links == null || record.links.length == 0) {
            record.links = [];
        }
        for (var i = 0; i < record.links.length; i++) {
            var link = record.links[i];
            var plugin = oscar.getPluginManager().getPluginFor(link.protocol);
            if (plugin) {
                plugin.setOptions({
                    link : link,
                    map : this.map,
                    record : record,
                    catalogueService : this.catalogueServices[0],
                    mode_container : this.modes_panel
                });
                plugin.drawTo($div);
                if (plugin.events) {
                    plugin.events.on({
                        "enterMode" : function(plugin) {
                            this.activePlugin = plugin;
                            this.results_panel.hide("slide", {
                                direction : "left"
                            }, 500);
                        },
                        "exitMode" : function() {
                            this.activePlugin = null;
                            this.results_panel.show("slide", {
                                direction : "left"
                            }, 500);
                        },
                        scope : this
                    });
                }
            }
        }
        return $div;
    },
    overRecord : function(record) {
        record.container.switchClass("", "over", 100);
        record.container.children().each(function() {
            var $this = $$(this);
            $this.addClass("over");
        });
    },
    blurRecord : function(record) {
        record.container.switchClass("over", "", 100), record.container.children().each(function() {
            var $this = $$(this);
            $this.removeClass("over");
            
        });
        
    },
    focusRecord : function(record) {
        this.events.triggerEvent("recordFocus", record);
        var $this = record.container;
        $this.parent().children().each(function() {
            var $local = $$(this);
            $local.removeClass("active");
        });
        
        $this.addClass("active");
        if (record.feature) {
            map.zoomToExtent(record.feature.geometry.getBounds());
        }
        var selection_layer = null;
        try {
            selection_layer = this.map.getLayersByName("selected-feature")[0]
            selection_layer.removeAllFeatures();
        } catch (err) {
            var styleMap = new OpenLayers.StyleMap({
                "default" : new OpenLayers.Style({
                    strokeColor : "#004580",
                    fillColor : "#004580",
                    fillOpacity : 0.1,
                    strokeOpacity : 1,
                    strokeWidth : 2
                })
            });
            selection_layer = new OpenLayers.Layer.Vector("selected-feature", {
                styleMap : styleMap,
                renderers : [ 'Canvas', 'VML' ],
                wrapDateLine : true,
                projection : map.getProjectionObject()
            });
            this.map.addLayer(selection_layer);
        }
        selection_layer.addFeatures([ record.feature.clone() ]);
    },
    attachMouseEvents : function($resultDiv, record) {
        var scope = this;
        $resultDiv.mouseenter(function(e) {
            if (record.feature) {
                var ctrl = scope.map.getControlsByClass(OpenLayers.Control.SelectFeature.prototype.CLASS_NAME)[0];
                ctrl.select(record.feature);
            }
        });
        
        $resultDiv.mouseleave(function(e) {
            if (record.feature) {
                var ctrl = scope.map.getControlsByClass(OpenLayers.Control.SelectFeature.prototype.CLASS_NAME)[0];
                ctrl.unselect(record.feature);
                
            }
        });
        var scope = this;
        $resultDiv.click(function(e) {
            scope.focusRecord(record);
        });
        
        this.map.events.on({
            "mouseout" : function(evt) {
                try {
                    var ctrl = map.getControlsByClass(OpenLayers.Control.SelectFeature.prototype.CLASS_NAME)[0];
                    ctrl.unselectAll();
                } catch (err) {
                    oscar.debug.error(err);
                }
            }
        });
    },
    CLASS_NAME : "oscar.Gui.CatalogueResults"
});

var cswget = {
    title : function(r) {
        if (r.title) {
            try {
                if ($$.isArray(r.title)) {
                    return r.title[0].value;
                } else {
                    return r.title;
                }
                
            } catch (err) {
                return null;
            }
        }
    },
    abs : function(r) {
        if (r["abstract"]) {
            try {
                if ($$.isArray(r["abstract"])) {
                    if (r["abstract"][0].value)
                        return r["abstract"][0].value;
                    else
                        return r["abstract"][0];
                } else {
                    return r["abstract"];
                }
            } catch (err) {
                return null;
            }
        }
    },
    projection : function(r) {
        
    },
    bounds : function(r) {
        
    },
    identifier : function(r) {
        if (r.identifier) {
            try {
                return r.identifier[0].value;
            } catch (err) {
                return null;
            }
        }
        
    }
}
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * @requires oscar.Gui
 */
/**
 * Class: oscar.Gui.Download
 * 
 * Parent class for the Download gui elements.
 * 
 * Inherits from: - <oscar.Gui>
 */

oscar.Gui.Download = oscar.BaseClass(oscar.Gui, {
    /**
     * Constructor: Not to be called directly.
     */
    initialize : function() {
        oscar.Gui.prototype.initialize.apply(this);
    },
    /**
     * Method: downloadFromService
     * 
     * Creates a hidden iframe to post a url to allow the Save As dialog to be
     * used when downloading from a service.
     */
    downloadFromService : function(url, filename, proxyOverride) {
        if (proxyOverride || oscar.DownloadHost) {
            var form = document.createElement("form");
            $$(form).css("display", "none");
            var formName = OpenLayers.Util.createUniqueID("randomForm");
            form.name = formName;
            form.action = proxyOverride || oscar.DownloadHost;
            form.method = "POST";
            var iUrl = document.createElement("input");
            iUrl.name = "url";
            iUrl.type = "text";
            iUrl.value = url;
            form.appendChild(iUrl);
            var iFilename = document.createElement("input");
            iFilename.name = "identifier";
            iFilename.type = "text";
            iFilename.value = filename;
            form.appendChild(iFilename);
            document.body.appendChild(form);
            form.submit();
        } else {
            window.open(url, "abc123", "width=640,height=480,menuBar=yes,location=false,scrollbars=yes");
        }
    },
    /**
     * Constant: CLASS_NAME
     */
    CLASS_NAME : "oscar.Gui.Download"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * @requires oscar.Gui.Download
 */

/**
 * Class: oscar.Gui.Download.WCS
 * 
 * This is a small GUI widget for executing AJAX requests for downloading from a
 * web coverage service.
 * 
 * Inherits from: - <oscar.Gui.Download>
 */

oscar.Gui.Download.WCS = oscar.BaseClass(oscar.Gui.Download, {
    /**
     * APIProperty: autoStart {Boolean} - Should the requests start immediately
     * after the draw method call. Defaults to true.
     * 
     */
    autoStart : true,
    /**
     * APIProperty: title The text to be displayed for the download reference.
     */
    title : null,
    /**
     * APIProperty: events {<OpenLayers.Events>} An events object that handles
     * all events on the label.
     */
    events : null,
    /**
     * Constant: EVENT_TYPES {Array(String)} Supported application event types.
     * 
     * Register a listener for a particular event with the following syntax: >
     * wcs.events.register(type, obj, listener);
     * 
     * Supported ClickableLabel event types:
     * 
     * xmlReceived - Triggered when the xml is received from the service.
     * 
     * xslReceived - Triggered when the xsl for transformation is received.
     */
    EVENT_TYPES : [ "xmlReceived", "xslReceived" ],

    /**
     * Property: wcsRequest {OpenLayers.Request.GET} - Handles the GetCoverage
     * request.
     * 
     */
    wcsRequest : null,
    /**
     * Property: hasXML {Boolean} - If the GetCoverage request returns
     * successfully this is set to true.
     */
    hasXML : false,
    /**
     * Property: xslRequest {OpenLayers.Request.GET} - Handles the xslt request.
     * 
     */
    xslRequest : null,
    /**
     * Property: hasXSL {Boolean} - If the xslt request returns successfully
     * this is set to true.
     */
    hasXSL : false,

    /**
     * Constructor: oscar.Gui.Download.WCS
     * 
     * Parameters: url - The url of the web coverage service. parameters -
     * {Object} Key:Value pair parameters for a GetCoverage request. options -
     * {Object} An optional object whose properties will be set on this
     * instance.
     */
    initialize : function(url, parameters, options) {
        OpenLayers.Util.extend(this, options);
        this.url = url;
        this.parameters = parameters;
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
        this.events.register("xmlReceived", this, this.checkExternalRequests);
        this.events.register("xslReceived", this, this.checkExternalRequests);
        this.isEMLDownload = (this.url.indexOf("store=false") != -1) ? true : false;

    },

    /**
     * APIMethod: draw
     * 
     * Creates the DOM elmements to display the widget on screen.
     * 
     * Returns: A DOM element.
     */
    draw : function() {
        this.div = document.createElement("div");
        this.contentContainer = document.createElement("div");
        this.div.id = OpenLayers.Util.createUniqueID(this.CLASS_NAME);
        var className = this.CLASS_NAME.replace(/\./g, "");
        oscar.jQuery(this.div).addClass(className);

        this.content = document.createElement("div");
        oscar.jQuery(this.content).addClass("downloadContainer");
        this.grfx = document.createElement("div");
        this.content.appendChild(this.grfx);
        oscar.jQuery(this.grfx).addClass("animatedDownload");
        this.txt = document.createElement("div");
        oscar.jQuery(this.txt).addClass("dmTextContainer");
        this.transformedDiv = document.createElement("div");
        oscar.jQuery(this.transformedDiv).addClass("dContainer");
        this.content.appendChild(this.txt);
        this.content.appendChild(this.transformedDiv);

        if (this.title == null) {
            this.title = prompt("Download Name:", "default");
        }
        this.txt.innerHTML = this.title;
        this.txt.title = this.title;
        this.div.appendChild(this.content);
        this.disableClick();

        if (this.autoStart) {
            var scope = this;
            var f = function() {
                scope.initRequests();
            }
            setTimeout(f, 500);
        }

        return this.div;
    },

    /**
     * Method: disableClick
     * 
     * Called from the draw method. This method disables calls through the
     * element to the map.
     */
    disableClick : function() {
        OpenLayers.Event.observe(this.div, "mousedown", function(e) {
            OpenLayers.Event.stop(e, true);
        });
        return;
        oscar.jQuery(this.div).draggable({
            containment : "parent",
            start : function(event, ui) {
                oscar.jQuery(this).addClass("olDragDown");
            },
            stop : function(event, ui) {
                oscar.jQuery(this).removeClass("olDragDown");
            }
        });

    },

    /**
     * APIMethod: initRequests
     * 
     * Called from the constructor. Sends off the requests for GetCoverage and
     * the stylesheet to for transformation.
     */
    initRequests : function() {
        if (!this.isEMLDownload) {
            this.wcsRequest = OpenLayers.Request.GET({
                url : this.url,
                async : true,
                parameters : this.parameters,
                success : this.xmlObtained,
                failure : this.requestFail,
                scope : this
            });

            this.xslRequest = OpenLayers.Request.GET({
                url : oscar._getScriptLocation() + "resources/GetCoverage.xsl",
                async : true,
                success : this.xslObtained,
                failure : this.requestFail,
                scope : this
            });
        } else {
            // fake a download
            var span = document.createElement("span");
            $$(span).addClass("dCoverage");
            $$(this.grfx).addClass("downloadReady");
            this.transformedDiv.appendChild(span);
            var scope = this;
            OpenLayers.Event.observe(span, "click", function(e) {
                scope.gotoUrl("dCoverage", scope.url);
            });
        }
    },

    /**
     * Method: xslObtained Called when the xsl request has completed
     * successfully.
     */
    xslObtained : function(resp) {
        this.xsl = resp.responseXML;
        this.hasXSL = true;
        this.events.triggerEvent("xslReceived");
    },
    /**
     * Method: xmlObtained Called when the GetCoverage request has completed
     * successfully.
     */
    xmlObtained : function(resp) {
        this.xml = resp.responseXML;
        this.hasXML = true;
        this.events.triggerEvent("xmlReceived");
    },
    /**
     * Method: checkExternalRequests Called from xslObtained and xmlObtained
     * methods. Will call the transform method if both responses are valid.
     */
    checkExternalRequests : function(e) {
        if (this.hasXSL && this.hasXML) {
            this.transform();
        }
    },

    /**
     * Method: transform
     * 
     * Called from the checkExternalRequests method. This method transforms the
     * received xml against the received xsl to display in the widget.
     */
    transform : function() {
        var results = oscar.Util["Transform"].transform(this.xml, this.xsl);
        var pHolder = document.createElement("div");
        if (typeof results == "string") {
            pHolder.innerHTML = results;
        } else {
            pHolder.appendChild(results);
        }
        var scope = this;
        var fadeIn = function() {
            scope.transformedDiv.innerHTML = pHolder.innerHTML;
            $$(scope.transformedDiv).find("span").each(function() {
                var $this = $$(this);
                var href = $this.attr("href").trim();
                var cls = $this.attr("class").trim();
                $this.click(function() {
                    scope.gotoUrl(cls, href);
                });
            });
            
            oscar.jQuery(scope.grfx).addClass("downloadReady");
            oscar.jQuery(this.content).addClass("ready");
            oscar.jQuery(scope.content).fadeIn("fast");
        }
        oscar.jQuery(this.content).fadeOut("fast", fadeIn);
    },
    /**
     * Method: gotoUrl Called when the coverage and metadata icons are clicked.
     * Will trigger a window or iframe to spawn allowing download of a coverage
     * or viewing of the metadata.
     * 
     */
    gotoUrl : function(classType, url) {
        switch (classType) {
        case "dCoverage":
            var filename = url.substring(url.lastIndexOf("/") + 1);
            this.downloadFromService(url, filename);
            break;
        case "dMetadata":
            this.downloadFromService(url, "metadata.xml");
            break;
        }
    },
    /**
     * Method: requestFail Called when the GetCoverage or xslt requests have
     * failed.
     */
    requestFail : function(resp) {
        if (this.xslRequest != null) {
            this.xslRequest.abort();
        }
        if (this.wcsRequest != null) {
            this.wcsRequest.abort();
        }
        var scope = this;
        var fadeIn = function() {
            scope.txt.innerHTML = "Download Cancelled";
            oscar.jQuery(scope.grfx).addClass("downloadCancelled");
            oscar.jQuery(scope.content).fadeIn("fast");
        }
        oscar.jQuery(this.content).fadeOut("fast", fadeIn);
    },

    CLASS_NAME : "oscar.Gui.Download.WCS"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Gui.Download.WFS
 * 
 * Download widget for executing WFS GetFeature requests for download. from a
 * web coverage service.
 * 
 * Inherits from: - <oscar.Gui.Download>
 */

oscar.Gui.Download.WFS = oscar.BaseClass(oscar.Gui.Download, {
    /**
     * APIProperty: autoStart {Boolean} - Should the requests start immediately
     * after the draw method call. Defaults to true.
     * 
     */
    autoStart : true,
    /**
     * APIProperty: title The text to be displayed for the download reference.
     */
    title : null,

    /**
     * Constructor: oscar.Gui.Download.WFS
     * 
     * Parameters: url - The url of the web coverage service. parameters -
     * {Object} Key:Value pair parameters for a GetCoverage request. options -
     * {Object} An optional object whose properties will be set on this
     * instance.
     */
    initialize : function(url, parameters, options) {
        OpenLayers.Util.extend(this, options);
        this.url = url;
        this.parameters = parameters;
        var paramString = OpenLayers.Util.getParameterString(this.parameters);
        paramString = unescape(paramString);
        if (paramString.length > 0) {
            var separator = (this.url.indexOf('?') > -1) ? '&' : '?';
            this.url += separator + paramString;
        }

    },

    /**
     * APIMethod: draw
     * 
     * Creates the DOM elmements to display the widget on screen.
     * 
     * Returns: A DOM element.
     */
    draw : function() {
        var scope = this;
        this.div = document.createElement("div");
        this.contentContainer = document.createElement("div");
        this.div.id = OpenLayers.Util.createUniqueID(this.CLASS_NAME);
        var className = this.CLASS_NAME.replace(/\./g, "");
        oscar.jQuery(this.div).addClass(className);

        this.content = document.createElement("div");
        oscar.jQuery(this.content).addClass("downloadContainer");
        this.grfx = document.createElement("div");
        this.content.appendChild(this.grfx);
        oscar.jQuery(this.grfx).addClass("animatedDownload");
        this.txt = document.createElement("div");
        oscar.jQuery(this.txt).addClass("dmTextContainer");

        this.transformedDiv = document.createElement("div");
        oscar.jQuery(this.transformedDiv).addClass("dContainer");

        downloadButton = $$("<span></span>").addClass("ui-icon-disk-save").attr("title", oscar.i18n("saveButtonLabel"));

        $$(this.transformedDiv).append(downloadButton);

        downloadButton.click(function(e) {
            scope.prepDownloadParams();
        });

        this.content.appendChild(this.txt);
        this.content.appendChild(this.transformedDiv);

        if (this.title == null) {
            this.title = prompt("Download Name:", "default");
        }
        this.txt.innerHTML = this.title;
        this.txt.title = this.title;
        this.txt.alt = this.title;
        this.div.appendChild(this.content);
        this.disableClick();

        if (this.autoStart) {
            var scope = this;
            var f = function() {
                scope.initRequests();
            }
            setTimeout(f, 500);
        }

        return this.div;
    },

    /**
     * Method: prepDownloadParams
     * 
     * Method for preparing the parameters for the download functionality.
     */
    prepDownloadParams : function() {
        var filename = this.title + ".";
        var format = new String(this.parameters["outputFormat"]);

        if (format.toLowerCase().indexOf("gml") > -1) {
            filename += "xml";
        } else if (format.toLowerCase().indexOf("kml") > -1) {
            filename += "kml";
        } else if (format.toLowerCase().indexOf("json") > -1) {
            filename += "json";
        } else if (format.toLowerCase().indexOf("shape") > -1) {
            filename += "zip";
        } else { // anything else
            filename += format.toLowerCase();
        }

        this.downloadFromService(this.url, filename);
    },

    /**
     * Method: disableClick
     * 
     * Called from the draw method. This method disables calls through the
     * element to the map.
     */
    disableClick : function() {
        OpenLayers.Event.observe(this.div, "mousedown", function(e) {
            OpenLayers.Event.stop(e, true);
        });
        return;
        oscar.jQuery(this.div).draggable({
            containment : "parent",
            start : function(event, ui) {
                oscar.jQuery(this).addClass("olDragDown");
            },
            stop : function(event, ui) {
                oscar.jQuery(this).removeClass("olDragDown");
            }
        });

    },

    /**
     * APIMethod: initRequests
     * 
     * Called from the constructor. Sends off the requests for GetCoverage and
     * the stylesheet to for transformation.
     */
    initRequests : function() {
        this.transform();
        return;
    },

    transform : function() {
        var scope = this;
        var fadeIn = function() {
            oscar.jQuery(scope.grfx).addClass("downloadReady");
            oscar.jQuery(scope.content).addClass("ready");
            oscar.jQuery(scope.content).fadeIn("fast");
        }
        oscar.jQuery(this.content).fadeOut("fast", fadeIn);
    },

    /**
     * Method: gotoUrl Called when the coverage and metadata icons are clicked.
     * Will trigger a window or iframe to spawn allowing download of a coverage
     * or viewing of the metadata.
     * 
     */
    gotoUrl : function(classType, url) {
        switch (classType) {
        case "dCoverage":
            var filename = url.substring(url.lastIndexOf("/") + 1);
            this.downloadFromService(url, filename);
            break;
        case "dMetadata":
            window.open(url);
            break;
        }
    },
    /**
     * Method: requestFail Called when the GetCoverage or xslt requests have
     * failed.
     */
    requestFail : function(resp) {
        var scope = this;
        var fadeIn = function() {
            scope.txt.innerHTML = "Download Cancelled";
            oscar.jQuery(scope.grfx).addClass("downloadCancelled");
            oscar.jQuery(scope.content).fadeIn("fast");
        }
        oscar.jQuery(this.content).fadeOut("fast", fadeIn);
    },

    CLASS_NAME : "oscar.Gui.Download.WFS"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Gui.CollapsiblePanel
 * 
 * A user interface widget that contains a header with a collapsible content
 * panel.
 */

oscar.Gui.CollapsiblePanel = oscar.BaseClass({
    /**
     * APIProperty: collapsed
     * 
     * Boolean - Whether the content panel is open or closed during
     * initialization
     * 
     * Default - false
     */
    collapsed : false,
    /**
     * APIProperty: title
     * 
     * String - The value of the header
     */
    title : "",
    /**
     * APIProperty: panel
     * 
     * HTMLElement - The div element for the object
     */
    panel : null,
    /**
     * APIProperty: container
     * 
     * HTMLElement - Reference to the container div
     */
    container : null,
    /**
     * APIProperty: contentPane
     * 
     * Reference to the content pane.
     */
    contentPane : null,
    /**
     * APIProperty: headerPanel
     * 
     * Reference to the header panel.
     */
    headerPanel : null,
    /**
     * APIProperty: css
     * 
     * Object - Contains the names of the css class references.
     */
    css : {
        container : "_collapsiblePanel",
        headerPanel : "_headerPanel",
        contentPanel : "_contentPanel",
        contentOpen : "contentOpen",
        contentClosed : "contentClosed",
        headerText : "_headerText"
    },
    /**
     * Constructor: oscar.Gui.CollapsiblePanel
     * 
     * Parameters: container - the div container for this CollapsiblePanel
     * options - {Object} An optional object whose properties will be set on
     * this instance.
     */
    initialize : function(container, options) {
        OpenLayers.Util.extend(this, options);
        this.container = container;
        this.panel = document.createElement('div');
        this.panel.id = OpenLayers.Util.createUniqueID(this.CLASS_NAME);
        oscar.jQuery(this.panel).addClass(this.CLASS_NAME.replace(/\./g, "_") + "_collapsiblePanel");
        this.contentPane = document.createElement('div');
        this.contentPane.id = OpenLayers.Util.createUniqueID(this.CLASS_NAME + "_ContentPane");
        oscar.jQuery(this.contentPane).addClass(this.CLASS_NAME.replace(/\./g, "_") + this.css.contentPanel);
        this.createHeaderPanel();
        this.panel.appendChild(this.contentPane);
        $(this.container).appendChild(this.panel);
    },
    /**
     * APIMethod: createHeaderPanel
     * 
     * Creates the header panel
     */
    createHeaderPanel : function() {
        var headerPanel = document.createElement("div");
        oscar.jQuery(headerPanel).addClass(this.CLASS_NAME.replace(/\./g, "_") + this.css.headerPanel);
        var ctx = this;
        this.titleElement = document.createElement("div");
        oscar.jQuery(this.titleElement).addClass(this.CLASS_NAME.replace(/\./g, "_") + this.css.headerText);
        headerPanel.appendChild(this.titleElement);
        oscar.jQuery(this.titleElement).click(function(event) {
            ctx.collapsed = (!ctx.collapsed);
            if (ctx.collapsed) {
                oscar.jQuery(ctx.titleElement).removeClass(ctx.css.contentOpen);
                oscar.jQuery(ctx.titleElement).addClass(ctx.css.contentClosed);
            } else {
                oscar.jQuery(ctx.titleElement).removeClass(ctx.css.contentClosed);
                oscar.jQuery(ctx.titleElement).addClass(ctx.css.contentOpen);
            }
            oscar.jQuery(ctx.contentPane).slideToggle('slow');
        });
        if (this.collapsed) {
            oscar.jQuery(this.titleElement).addClass(this.css.contentClosed);
            oscar.jQuery(this.contentPane).css("display", "none");
        } else {
            oscar.jQuery(this.titleElement).addClass(this.css.contentOpen);
        }

        this.panel.appendChild(headerPanel);

    },
    /**
     * APIMethod: setTitle
     * 
     * Parameters:
     * 
     * title - <String> The title of the panel
     */
    setTitle : function(title) {
        this.title = title;
        this.titleElement.innerHTML = this.title;
    },
    /**
     * APIMethod: getPanel
     * 
     * Returns the panel reference
     */
    getPanel : function() {
        return this.panel;
    },
    /**
     * APIMethod: getContentPane
     * 
     * Returns the contentPane reference
     */
    getContentPane : function() {
        return this.contentPane;
    },
    /**
     * APIMethod: appendContent
     * 
     * Parameters:
     * 
     * node - <HTMLElement> - Adds content to the content pane.
     */
    appendContent : function(node) {
        this.contentPane.appendChild(node);
    },
    /**
     * Constant: CLASS_NAME - oscar.Gui.CollapsiblePanel
     */
    CLASS_NAME : "oscar.Gui.CollapsiblePanel"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 * 
 * Copyright 2014 CARIS <http://www.caris.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Gui.DownloadOptions
 * 
 * This GUI class is used to display download options for WFS and WCS contents.
 * 
 */

oscar.Gui.DownloadOptions = oscar.BaseClass(oscar.Gui, {
    /**
     * APIProperty: db The database containing the object references
     */
    db : null,
    /**
     * Property: capabilities The current capabilities object
     */
    capabilities : null,
    /**
     * Property: serviceType The type of service
     */
    serviceType : null,
    
    /**
     * Property: showAbstract Show the abstract of the feature selected.
     * Defaults to true.
     */
    showAbstract : true,
    /**
     * Property: feature
     * 
     */
    feature : null,
    /**
     * Property: defaultOptions
     */
    defaultOptions : null,
    /**
     * APIProperty: EVENT_TYPES
     */
    EVENT_TYPES : [ "serviceReady" ],
    /**
     * APIProperty: events
     */
    events : null,
    /**
     * Constructor: new oscar.Gui.DownloadOptions
     */
    initialize : function(options) {
        oscar.Gui.prototype.initialize.apply(this, arguments);
        this.defaultOptions = {};
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : false
        });
        
    },
    /**
     * Method: draw Creates the interface panel.
     */
    draw : function() {
        oscar.Gui.prototype.draw.apply(this);
        oscar.jQuery(this.div).addClass("options");
        OpenLayers.Event.observe(this.div, "mousedown", function(e) {
            OpenLayers.Event.stop(e, true);
        });
        OpenLayers.Event.observe(this.div, "click", function(e) {
            OpenLayers.Event.stop(e, true);
        });
        
        return this.div;
    },
    /**
     * Method: redraw Created the gui elements to be displayed.
     */
    redraw : function() {
        this.div.innerHTML = "";
        var infoPanel = document.createElement("div");
        this.div.appendChild(infoPanel);
        oscar.jQuery(infoPanel).addClass("info");
        var featureTitle = document.createElement("span");
        var featureAbstract = document.createElement("p");
        infoPanel.appendChild(featureTitle);
        infoPanel.appendChild(featureAbstract);
        featureTitle.innerHTML = this.feature.div.data("title") || this.feature.div.data("id");
        if (!this.feature.div.data("abstract") || this.feature.div.data("abstract").length == 0) {
            featureAbstract.innerHTML = "";
        } else {
            featureAbstract.innerHTML = this.feature.div.data("abstract");
        }
        
        if (!this.showAbstract) {
            $$(infoPanel).css("display", "none");
        }
        
        this.buildDownloadOptions();
        
    },
    /**
     * Method: deactivate
     * 
     */
    deactivate : function() {
        if (this.cropTool) {
            this.cropTool.deactivate();
        }
    },
    
    buildDownloadOptions : function() {
        var scope = this;
        this.defaultOptions = {};
        
        var userConfigPanel = document.createElement("div");
        userConfigPanel.id = "userConfigPanel";
        this.div.appendChild(userConfigPanel);
        
        var buttonsPanel = document.createElement("div");
        oscar.jQuery(buttonsPanel).addClass("buttonsPanel");
        this.div.appendChild(buttonsPanel);
        
        this.cropDiv = document.createElement("span");
        oscar.jQuery(this.cropDiv).addClass("dOption crop");
        this.cropDiv.setAttribute("title", oscar.i18n("Crop"));
        buttonsPanel.appendChild(this.cropDiv);
        
        var downloadDiv = document.createElement("span");
        downloadDiv.setAttribute("title", oscar.i18n("AddToQueue"));
        oscar.jQuery(downloadDiv).addClass("dOption queueDownload");
        buttonsPanel.appendChild(downloadDiv);
        
        OpenLayers.Event.observe(downloadDiv, "click", function(e) {
            scope.queueDownload();
        });
        
        OpenLayers.Event.observe(this.cropDiv, "click", function(e) {
            scope.cropTool = new oscar.Control.Box();
            scope.map.addControl(scope.cropTool);
            $$(scope.cropDiv).addClass("active");
            scope.cropTool.activate();
            scope.cropTool.events.on({
                'done' : function(geom) {
                    var layer = scope.feature.layer;
                    var featureBounds = scope.feature.geometry.getBounds();
                    var newBounds = geom.getBounds();
                    
                    if (!featureBounds.intersectsBounds(newBounds) && !featureBounds.containsBounds(newBounds)) {
                        alert("Crop area outside of the bounds");
                        return;
                    }
                    
                    if (scope.cropFeature) {
                        try {
                            layer.removeFeatures(scope.cropFeature);
                            scope.cropFeature = null;
                        } catch (err) {
                        }
                    }
                    
                    scope.cropFeature = newFeature = scope.feature.clone();
                    scope.cropFeature.geometry = geom;
                    layer.addFeatures(scope.cropFeature);
                    layer.drawFeature(scope.cropFeature, "temporary");
                    scope.cropTool.deactivate();
                    $$(scope.cropDiv).removeClass("active");
                    scope.defaultOptions.bbox = geom.getBounds();
                },
                scope : scope
            });
            
        });
        var proc = this["getOptions_" + this.serviceType];
        if (proc) {
            proc.call(this, userConfigPanel);
        }
        
    },
    
    /**
     * This function will build the download options for a Web Feature Service
     * Supports: - Format
     */
    getOptions_WFS : function(div) {
        this.defaultOptions.service = "WFS";
        var GetFeatureOp = oscar.Util.Metadata.getOperation(this.capabilities.capabilities, "GetFeature");
        
        outputFormats = oscar.Util.Metadata.getParameters(this.capabilities.capabilities, "GetFeature", [ "outputFormat", "formats" ])

        var formats = [];
        if ($$.isArray(outputFormats)) {
            formats = outputFormats;
        } else {
            for ( var prop in outputFormats) {
                formats.push(prop);
            }
        }
        
        var id = oscar.Util.Metadata.getFeatureTypesById(this.capabilities.capabilities, this.feature.div.data("id"));
        
        this.makeFormatList(div, formats);
        this.makeCRSList(div, id.srss || [ id.srs ]);
        var opURL = null
        try {
            opUrl = GetFeatureOp.dcp.http.get[0].url || GetFeatureOp.dcp.http.get;
        } catch (e) { // wfs 1.0.0 doesn't have dcp so this will throw a null
                        // error...
            opUrl = GetFeatureOp.href.get;
        }
        this.defaultOptions.operationUrl = opUrl;
        this.defaultOptions.id = this.feature.div.data("id");
        this.defaultOptions.bbox = this.feature.div.data("bbox");
        this.defaultOptions.title = this.feature.div.data("title") || this.feature.div.data("id");
        
    },
    
    /**
     * Method: makeFormatList
     * 
     * Creates a combo box list for selecting formats.
     */
    makeFormatList : function(div, outputFormats) {
        var scope = this;
        var formatDiv = document.createElement("div");
        var inArray = function(format, arr) {
            for (var i = 0; i < arr.length; i++) {
                if (format.toLowerCase() == arr[i].label.toLowerCase()) {
                    return true;
                }
            }
            return false;
        }

        var usedFormats = [];
        for ( var f in outputFormats) {
            
            var simpleFormat = oscar.Util.convertFormat(outputFormats[f]);
            if (inArray(simpleFormat, usedFormats)) {
                continue;
            }
            var formatObj = {
                label : simpleFormat,
                value : outputFormats[f]
            }
            usedFormats.push(formatObj);
        }
        $$(div).append($$("<label></label>").html(oscar.i18n("Format") + ":").addClass("heading"));
        var input = document.createElement("input");
        input.type = "text";
        formatDiv.appendChild(input);
        $$(input).autocomplete({
            minLength : 0,
            source : usedFormats,
            select : function(event, ul) {
                this.value = ul.item.label;
                scope.defaultOptions.format = ul.item.value;
                if (ul.item.value.indexOf("bag") > -1) {
                    var ref = oscar.Util.CoordinateReferences.getReference(scope.gridBaseCRS);
                    $$(".crsInput").attr("disabled", "disabled");
                    $$(".crsButton").attr("disabled", "disabled");
                    $$(".crsInput").val(ref.description);
                    scope.defaultOptions.crs = scope.gridBaseCRS;
                } else {
                    $$(".crsInput").removeAttr("disabled");
                    $$(".crsButton").removeAttr("disabled");
                }
                return false;
            },
            focus : function(event, ui) {
                this.value = ui.item.label;
                scope.defaultOptions.format = ui.item.value;
                return false;
                
            }
        }).css("width", "190px");
        
        oscar.jQuery(input).addClass("foramtListWidth");
        
        var button = document.createElement("button");
        button.innerHTML = oscar.i18n("Format");
        formatDiv.appendChild(button);
        oscar.jQuery(button).insertAfter(input).button({
            icons : {
                primary : "ui-icon-triangle-1-s"
            },
            text : false
        }).removeClass("ui-corner-all").addClass("ui-corner-right ui-button-icon").click(function() {
            // close if already visible
            if (oscar.jQuery(input).autocomplete("widget").is(":visible")) {
                oscar.jQuery(input).autocomplete("close");
                return;
            }
            // pass empty string as value to search for, displaying all results
            oscar.jQuery(input).autocomplete("search", "");
            oscar.jQuery(input).focus();
        });
        input.value = usedFormats[0].label;
        this.defaultOptions.format = usedFormats[0].value
        div.appendChild(formatDiv);
        
    },
    
    /**
     * Method: makeCRSList
     * 
     * Creates a combo box list for selecting coordinate reference systems.
     */
    makeCRSList : function(div, crss) {
        var scope = this;
        var crsDiv = document.createElement("div");
        $$(crsDiv).css("margin-top", "5px");
        var crsReferences = []
        for (var i = 0; i < crss.length; i++) {
            var crs = oscar.Util.CoordinateReferences.getReference(crss[i]);
            // create an instance of the projection so it is ready for use
            // later.
            new OpenLayers.Projection(crs.code);
            crsReferences.push(crs);
        }
        $$(crsDiv).append($$("<label></label>").html(oscar.i18n("srsCodeColumnLabel") + ":").addClass("heading").css("display","block"));

        var input = document.createElement("input");
        input.type = "text";
        $$(input).addClass("crsInput");
        crsDiv.appendChild(input);
        oscar.jQuery(input).autocomplete({
            minLength : 0,
            source : function(request, response) {
                var term = request.term;
                var match = [];
                for ( var i in crsReferences) {
                    if (crsReferences[i].code.toLowerCase().indexOf(term.toLowerCase()) != -1) {
                        match.push(crsReferences[i]);
                    }
                }
                response(match);
                return;
            },
            select : function(event, ul) {
                this.value = ul.item.description;
                scope.defaultOptions.crs = ul.item.code;
                return false;
            },
            focus : function(event, ui) {
                this.value = ui.item.description;
                return false;
                
            }
        }).data("uiAutocomplete")._renderItem = function(ul, item) {
            var li = document.createElement("li");
            return oscar.jQuery(li).data("ui-autocomplete-item", item).append("<a>" + item.code + "<br>" + item.description + "</a>").appendTo(ul);
            
        };
        
        $$(input).css("width", "190px");
        
        var button = document.createElement("button");
        button.innerHTML = oscar.i18n("srsCodeColumnLabel");
        $$(button).addClass("crsButton");
        crsDiv.appendChild(button);
        oscar.jQuery(button).insertAfter(input).button({
            icons : {
                primary : "ui-icon-triangle-1-s"
            },
            text : false
        }).removeClass("ui-corner-all").addClass("ui-corner-right ui-button-icon").click(function() {
            // close if already visible
            if (oscar.jQuery(input).autocomplete("widget").is(":visible")) {
                oscar.jQuery(input).autocomplete("close");
                return;
            }
            // pass empty string as value to search for, displaying all results
            oscar.jQuery(input).autocomplete("search", "");
            oscar.jQuery(input).focus();
        })
        input.value = crsReferences[0].description;
        this.defaultOptions.crs = crsReferences[0].code
        div.appendChild(crsDiv);
        
    },
    /**
     * Method: _createFieldCheckout This method creates the checkbox to display
     * the available fields for the coverage. Parameters: - field - a JSON
     * object representing a field
     */
    _createFieldCheckbox : function(field) {
        var scope = this;
        var $input = $$("<input type='checkbox'>");
        var id = OpenLayers.Util.createUniqueID(field.identifier);
        $input.attr("id", id);
        $input.attr("value", field.identifier);
        $input.data("field", field.identifier);
        $input.click(function() {
            var $this = $$(this);
            var currentFields = scope.defaultOptions.field;
            var fieldFound = false;
            for (var i = 0; i < currentFields.length; i++) {
                var selectedField = currentFields[i];
                if ($this.attr("id") == selectedField.attr("id")) {
                    fieldFound = true;
                    currentFields.splice(i, 1);
                    break;
                }
            }
            if (!fieldFound) {
                currentFields.push($this);
            }
            scope.defaultOptions.field = currentFields
        });
        return $input;
    },
    /**
     * Method: _createInterpolationMethodList This method creates the selection
     * dropdown lists showing the interpolation methods available for the field.
     * Parameters: - field - a JSON object representing a field
     */
    _createInterploationMethodList : function(field) {
        var $selection = $$("<select></select>");
        var id = OpenLayers.Util.createUniqueID(field.identifier);
        $selection.attr("id", id);
        var defaultMethod = "";
        if (field.interpolationMethods.defaultMethod) {
            defaultMethod = field.interpolationMethods.defaultMethod;
        }
        for (m in field.interpolationMethods.methods) {
            var method = field.interpolationMethods.methods[m];
            var $option = $$("<option></option").html(method);
            if (method == defaultMethod) {
                $option.attr("selected", true);
            }
            $selection.append($option);
        }
        // hide the interpolation method list if none are available.
        if (field.interpolationMethods.methods.length == 0) {
            $selection.css("display", "none");
        }
        
        return $selection;
    },
    /**
     * Method: makeFieldList
     * 
     * Creates a button to display a list to select fields for extraction from a
     * Web Coverage Service.
     */
    makeFieldList : function(div, fields) {
        var scope = this;
        // create the div to contain the field list
        var $fieldDiv = $$("<div></div>");
        var $fieldButton = $$("<button></button>").html(oscar.i18n("Fields"));
        $fieldButton.addClass("heading");
        
        var $table = $$("<table></table>");
        $table.css("display", "none");
        for (i in fields) {
            var field = fields[i];
            var $row = $$("<tr></tr>");
            var $inputCell = $$("<td></td>");
            var $fieldCell = $inputCell.clone().html(field.identifier);
            $fieldCell.addClass("heading");
            var $selectionCell = $inputCell.clone();
            
            var $selection = this._createInterploationMethodList(field);
            
            var $input = this._createFieldCheckbox(field);
            
            $input.data("selection", $selection);
            
            if (i == 0) {
                $input.attr("checked", true);
                this.defaultOptions.field = new Array($input);
            }
            
            $row.append($inputCell);
            $inputCell.append($input);
            $row.append($fieldCell);
            $row.append($selectionCell);
            $selectionCell.append($selection);
            
            $table.append($row);
        }
        
        $fieldDiv.append($table);
        $$(div).append($fieldButton);
        $fieldButton.click(function() {
            $table.slideToggle("slow");
        });
        
        $$(div).append($fieldDiv);
    },
    /**
     * Method: makeResolutionFields This method creates the gui elements to
     * display resolution values in the download options panel.
     */
    makeResolutionFields : function(div) {
        var offsets = oscar.Util.getGridOffsets(this.gridOffsets);
        
        var $resolutionDiv = $$("<div></div>");
        var $xLabel = $$("<label>").html(oscar.i18n("resolution-x") + ":&nbsp;");
        $xLabel.addClass("heading");
        this.$xText = $$("<input type='text' id='' size='5'>");
        var $yLabel = $$("<label>").html(oscar.i18n("resolution-y") + ":&nbsp;");
        $yLabel.addClass("heading");
        this.$yText = $$("<input type='text' id='' size='5'>");
        var $meters = $$("<label></label>").html("&nbsp;" + oscar.i18n("units:meters"));
        $meters.addClass("heading");
        $resolutionDiv.append($xLabel);
        $resolutionDiv.append(this.$xText);
        
        $resolutionDiv.append($meters);
        $resolutionDiv.append($$("<br/>"));
        $resolutionDiv.append($yLabel);
        $resolutionDiv.append(this.$yText);
        $resolutionDiv.append($meters.clone());
        $$(div).append($resolutionDiv);
        // variable names are just labels; the offsets array could be ordered XY
        // or YX, depending on the CRS
        var offsetX = parseFloat(offsets[0]);
        var offsetY = parseFloat(offsets[1]);
        var projection = new OpenLayers.Projection(this.gridBaseCRS);
        offsetX *= oscar.Util.getMetersConversionFactor(projection);
        offsetY *= oscar.Util.getMetersConversionFactor(projection);
        var isGeographicCRS = oscar.Util.isGeographicCRS(projection);
        if (isGeographicCRS) {
            this.$xText.val(offsetY);
            this.$yText.val(offsetX);
        } else {
            this.$xText.val(offsetX);
            this.$yText.val(offsetY);
        }
    },
    /**
     * This function will build the download options for a Web Coverage Service
     * Supports: - Format - CRS - Fields
     */
    
    getOptions_WCS : function(div) {
        this.defaultOptions.service = "WCS";
        var loading = document.createElement("div");
        oscar.jQuery(loading).addClass("optionsLoading");
        div.appendChild(loading);
        
        var GetCoverageOp = oscar.Util.Metadata.getOperation(this.capabilities.capabilities, "GetCoverage");
        
        var isServiceStorageAllowed = function(op) {
            for (var i = 0; i < op.parameters.length; i++) {
                var param = op.parameters[i];
                if (param.name.toLowerCase("store")) {
                    for (var j = 0; j < param.values.length; j++) {
                        if (param.values[j].toLowerCase() == "true") {
                            return true;
                        }
                    }
                    
                }
            }
            return false;
        }
        this.defaultOptions.store = isServiceStorageAllowed(GetCoverageOp);
        
        var url = oscar.Util.Metadata.getOperationHref(this.capabilities.capabilities, "DescribeCoverage");
        
        var params = {
            service : "WCS",
            request : "DescribeCoverage",
            identifiers : this.feature.div.data("id"),
            version : this.capabilities.capabilities.version
        }
        var scope = this;
        var success = function(resp) {
            oscar.jQuery(loading).fadeOut();
            var version = this.capabilities.capabilities.version;
            var reader = new oscar.Format.WCSDescribeCoverage({
                "version" : version
            });
            var coverageDescription = reader.read(resp.responseXML);
            var fields = null;
            
            try {
                this.gridBaseCRS = coverageDescription.coverageDescription.domain.spatialDomain.gridCRS.gridBaseCRS;
                
                this.gridType = "urn:ogc:def:method:WCS:1.1:2dSimpleGrid";
                if (coverageDescription.coverageDescription.domain.spatialDomain.gridCRS.gridType) {
                    this.gridType = coverageDescription.coverageDescription.domain.spatialDomain.gridCRS.gridType;
                }
                
                if (coverageDescription.coverageDescription.domain.spatialDomain.gridCRS.gridOrigin) {
                    this.gridOrigin = coverageDescription.coverageDescription.domain.spatialDomain.gridCRS.gridOrigin;
                } else {
                    this.gridOrigin = "0 0";
                }
                if (coverageDescription.coverageDescription.domain.spatialDomain.gridCRS.gridOffsets) {
                    this.gridOffsets = coverageDescription.coverageDescription.domain.spatialDomain.gridCRS.gridOffsets;
                } else {
                    this.gridOffsets = "0 0";
                }
                var fields = coverageDescription.coverageDescription.range.fields;
                supportedCRSs = coverageDescription.coverageDescription.supportedCRS;
                supportedFormats = coverageDescription.coverageDescription.supportedFormats;
                this.makeFormatList(div, supportedFormats);
                this.makeCRSList(div, supportedCRSs);
                this.makeFieldList(div, fields);
                this.makeResolutionFields(div);
            } catch (err) {
                alert(err.message);
                alert("error in response");
            }
            
        };
        
        var request = new OpenLayers.Request.GET({
            url : url,
            params : params,
            async : true,
            success : success,
            failure : function(resp) {
            },
            scope : this
        });
        
        this.defaultOptions.operationUrl = GetCoverageOp.dcp.http.get;
        this.defaultOptions.id = this.feature.div.data("id");
        this.defaultOptions.bbox = this.feature.div.data("bbox");
        this.defaultOptions.title = this.feature.div.data("title") || this.feature.div.data("id");
        
    },
    /**
     * Method: setFeature
     * 
     * Sets the current selected feature.
     * 
     * Parameters:
     *  - OpenLayers.Feature
     */
    setFeature : function(feature) {
        this.feature = feature;
        this.getCapabilities();
        this.redraw();
    },
    
    /**
     * Method: getCapabilites
     * 
     * Uses the curent feature to obtain the reference in the capabilities
     * document.
     * 
     */
    getCapabilities : function() {
        this.capabilities = this.db.search("capabilities", this.feature.div.data("fk_capabilities"), function(table, query) {
            return table.records[query];
        });
        this.serviceType = oscar.Util.Metadata.getServiceType(this.capabilities.capabilities);
    },
    
    /**
     * Method: queueDownload
     * 
     * Adds an item to the download queue panel.
     */
    queueDownload : function() {
        var serviceType = this.defaultOptions.service;
        var downloadService = null;
        var params = {
            service : serviceType,
            version : this.capabilities.capabilities.version
        }

        var buildUrl = function(url, params) {
            if (!url) {
                url = "";
            }
            var paramString = OpenLayers.Util.getParameterString(params);
            paramString = unescape(paramString);
            if (paramString.length > 0) {
                var separator = (url.indexOf('?') > -1) ? '&' : '?';
                url += separator + paramString;
            }
            
            return url;
        }

        switch (serviceType) {
            case "WFS":
                var buildWFSFilter = function(bbox, projection, version, typeNames, crs) {
                    var options = {
                        version : version,
                        featureType : typeNames
                    };
                    if (version != "1.0.0") {
                        options.xy = (projection.proj.projName == "longlat") ? false : true;
                    }
                    if (crs) {
                        options.srsName = crs;
                    }
                    var format = new OpenLayers.Format.WFST(options);
                    var sFilter = new OpenLayers.Filter.Spatial({
                        type : OpenLayers.Filter.Spatial.BBOX,
                        value : bbox,
                        projection : projection.projCode
                    });
                    var data = OpenLayers.Format.XML.prototype.write.apply(format, [ format.writeNode("wfs:GetFeature", {
                        filter : sFilter
                    }) ]);
                    return data;
                };
                
                var filter = buildWFSFilter(this.defaultOptions.bbox, this.map.projection, this.capabilities.capabilities.version, this.defaultOptions.id, this.defaultOptions.crs);
                var params = {
                    service : "WFS",
                    request : "GetFeature",
                    srsName : this.defaultOptions.crs,
                    filter : encodeURIComponent(filter),
                    version : this.capabilities.capabilities.version,
                    typename : this.defaultOptions.id,
                    outputFormat : encodeURIComponent(this.defaultOptions.format)
                }

                var url = this.defaultOptions.operationUrl;
                var paramString = OpenLayers.Util.getParameterString(params);
                paramString = unescape(paramString);
                if (paramString.length > 0) {
                    var separator = (url.indexOf('?') > -1) ? '&' : '?';
                    url += separator + paramString;
                }
                downloadService = new oscar.Gui.Download.WFS(this.defaultOptions.operationUrl, params, {
                    title : this.defaultOptions.title
                });
                break;
            case "WCS":
                var bounds = this.defaultOptions.bbox;
                var projection = new OpenLayers.Projection(this.defaultOptions.crs);
                var urn = oscar.Util.EpsgConversion.epsgToUrn(projection.projCode);
                var isGeographicCRS = oscar.Util.isGeographicCRS(projection);
                // do I need to transform the boundingbox
                if (projection.projCode != this.map.getProjectionObject().projCode) { // perform
                                                                                        // transformation
                    bounds = bounds.clone().transform(this.map.getProjectionObject(), projection);
                }
                
                var localBBOX = bounds.toArray(isGeographicCRS);
                
                // get the requested fields / bands to build the range subset
                var fields = this.defaultOptions.field;
                var fieldsArray = new Array();
                for (f in fields) {
                    var field = fields[f];
                    var $input = $$(field);
                    var fieldName = encodeURI($input.val());
                    var select = $$($input.data("selection"));
                    var selectValue = select.val();
                    
                    if (select.val() != null) {
                        fieldsArray.push(fieldName + ":" + select.val());
                    } else {
                        fieldsArray.push(fieldName);
                    }
                }
                
                var rngSubset = "";
                if (fieldsArray.length > 1) {
                    rngSubset = fieldsArray.join(";");
                } else {
                    rngSubset = fieldsArray.join(" ");
                }
                
                var localparams = {
                    request : "GetCoverage",
                    store : this.defaultOptions.store,
                    GridBaseCRS : urn,
                    identifier : this.defaultOptions.id,
                    BoundingBox : localBBOX + "," + urn,
                    format : this.defaultOptions.format,
                    gridType : this.gridType
                }

                if (fieldsArray.length > 0) {
                    localparams.RangeSubset = rngSubset;
                }
                
                /*
                 * If the urn value is the same as the gridBaseCRS value then
                 * include the grid origin
                 */
                if (urn == this.gridBaseCRS) {
                    localparams.GridOrigin = this.gridOrigin.split(" ").join(",")
                }
                
                // inject the new grid offset values.
                var resX = parseFloat(this.$xText.val());
                var resY = parseFloat(this.$yText.val());
                
                if (resY > 0) {
                    resY *= -1;
                }
                
                resX /= oscar.Util.getMetersConversionFactor(projection);
                resY /= oscar.Util.getMetersConversionFactor(projection);
                var offsetArray = [];
                if (isGeographicCRS) {
                    offsetArray.push(resY);
                    offsetArray.push(resX);
                } else {
                    offsetArray.push(resX);
                    offsetArray.push(resY);
                }
                
                if (this.gridType == "urn:ogc:def:method:WCS:1.1:2dGridIn2dCrs") {
                    offsetArray.splice(1, 0, 0);
                    offsetArray.splice(1, 0, 0);
                }
                localparams.GridOffsets = offsetArray.toString();
                
                OpenLayers.Util.extend(localparams, params);
                var url = buildUrl(this.defaultOptions.operationUrl, localparams);
                
                downloadService = new oscar.Gui.Download.WCS(url, null, {
                    title : this.defaultOptions.title
                });
                break;
        }
        if (downloadService) {
            this.events.triggerEvent("serviceReady", downloadService);
        }
        
    },
    /**
     * Constant: CLASS_NAME
     */
    CLASS_NAME : "oscar.Gui.DownloadOptions"

});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 * 
 * Copyright 2014 CARIS <http://www.caris.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Gui.Info = new oscar.BaseClass(oscar.Gui, {
    EVENT_TYPES : [ 'whatsthis' ],
    context : null,
    initialize : function(options) {
        this.context = oscar.i18n("Help");
        this.EVENT_TYPES = this.EVENT_TYPES.concat(oscar.Gui.prototype.EVENT_TYPES);
        oscar.Gui.prototype.initialize.apply(this, [ options ]);
        this.id = "oscar-context-help";
        this.iframeId = "oscar-context-help-iframe";
    },
    draw : function() {
        oscar.Gui.prototype.draw.apply(this);
        this.btn = $$("<div></div>")
        this.btn.attr("title", this.context);
        this.btn.addClass("ui-button");
        this.btn.addClass("ico");
        this.btn.click($$.proxy(this.buildDlg, this));
        $$(this.div).append(this.btn);
    },
    buildDlg : function() {
        this.close();
        
        var dlg = $$("<div></div>");
        dlg.attr("id", this.id);
        dlg.addClass("whatsthis-dlg");
        
        $$('body').append(dlg);
        
        dlg.position({
            my : "right top",
            at : "right bottom",
            of : this.btn
        });
        
        var content = $$("<div></div>").html("");
        content.addClass("wt-content-container");
        
        var titleBar = $$("<div></div>").html(this.context);
        titleBar.addClass("wt-titleBar");
        var closeBox = $$("<div></div>")
        var contentBox = $$("<div></div>");
        contentBox.addClass("wt-content");
        closeBox.addClass("closeBox");
        closeBox.click($$.proxy(this.close, this));
        closeBox.attr("title",oscar.i18n("Close"));
        titleBar.append(closeBox);
        content.append(titleBar);
        content.append(contentBox);
        var iframe = $$("<iframe></iframe>");
        iframe.attr("id", this.iframeId);
        iframe.attr("frameBorder", "0");
        iframe.css({
            "width" : "100%",
            "height" : "100%"
        });
        contentBox.append(iframe);
        
        dlg.append(content);
        var containerHeight = content.height();
        var titleHeight = titleBar.height();
        var remainingHeight = containerHeight - titleHeight;
        contentBox.css({
            "height" : remainingHeight
        });
        
        dlg.draggable({
            containment : "parent",
            handle : titleBar
        });
        
        this.events.triggerEvent("whatsthis", this);
    },
    show : function(loc) {
        
        if (loc) {
            if (loc.indexOf("{0}") != -1) {
                loc = loc.format(OpenLayers.Lang.getCode());
            }
            $$("#" + this.iframeId).attr("src", loc);
        } else {
            this.close();
        }
    },
    close : function() {
        try {
            $$("#" + this.id).remove();
        } catch (err) {
        }
    },
    CLASS_NAME : "oscar.Gui.Info"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Gui.ServicePanel
 * 
 * A panel to display metadata of a service.
 */

oscar.Gui.ServicePanel = oscar.BaseClass(oscar.Gui.CollapsiblePanel, {
    /**
     * APIProperty: serviceMetadata {oscar.Gui.CollapsiblePanel} Displays
     * information about the service.
     */
    serviceMetadata : null,
    /**
     * APIProperty: serviceContent {oscar.Gui.CollapsiblePanel} Displays
     * information about the service contents.
     */
    serviceContent : null,
    /**
     * Constructor: oscar.Gui.ServicePanel
     * 
     * Parameters: div - {HTMLDivElement} the div container for this widget.
     */
    initialize : function(div) {
        oscar.Gui.CollapsiblePanel.prototype.initialize.apply(this, [ div ]);
        this.css.contentClass = "servicePanelContent";
        this.serviceContents = new oscar.Gui.CollapsiblePanel(this.contentPane);
        this.serviceContents.setTitle(oscar.i18n("servicepanel_content_header"));

        this.serviceMetadata = new oscar.Gui.CollapsiblePanel(this.contentPane, {
            collapsed : true
        });
        this.serviceMetadata.setTitle(oscar.i18n("servicepanel_info_header"));
        this.panel.appendChild(this.contentPane);
        $(this.container).appendChild(this.panel);
    },
    /**
     * APIMethod: appendMetadataContent Adds content to the service metadata
     * panel.
     * 
     * Parameters: node - {HTMLDivElement} div container for the metadata
     * content added.
     * 
     */
    appendMetadataContent : function(node) {
        this.serviceMetadata.appendContent(node);
    },
    /**
     * APIMethod: appendServiceContents Adds content to the service contents
     * panel.
     * 
     * Parameters: node - {HTMLDivElement} div container for the service content
     * added.
     * 
     */
    appendServiceContents : function(node) {
        this.serviceContents.appendContent(node);
    },
    /**
     * Constant: CLASS_NAME - oscar.Gui.ServicePanel
     */
    CLASS_NAME : "oscar.Gui.ServicePanel"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Gui.ClickableLabel
 * 
 * This ClickableLabel widget provides a label that, when clicked, fires a
 * labelClicked event with the object reference.
 * 
 * Inherits from: - <oscar.Gui>
 */
oscar.Gui.ClickableLabel = oscar.BaseClass(oscar.Gui, {
    /**
     * APIProperty: styles
     * 
     * styles.BLOCK - Creates a <div> container
     * 
     * styles.INLINE - Creates a <span> container
     */
    styles : {
        "BLOCK" : "block",
        "INLINE" : "inline"
    },
    /**
     * Constant: EVENT_TYPES {Array(String)} Supported application event types.
     * 
     * Register a listener for a particular event with the following syntax: >
     * clickablelabel.events.register(type, obj, listener);
     * 
     * Supported ClickableLabel event types:
     * 
     * labelClicked - Triggered when the label is clicked. Passes the object
     * reference to the listeners. labelTextChanged - Triggered when the label
     * text is changed.
     */
    EVENT_TYPES : [ "labelClicked", "labelTextChanged" ],

    /**
     * APIProperty: events {<OpenLayers.Events>} An events object that handles
     * all events on the label.
     */
    events : null,
    /**
     * APIProperty: container
     * 
     * Reference to the label container.
     */
    container : null,
    /**
     * APIProperty: style
     * 
     * block or inline
     */
    style : "inline",
    /**
     * APIProperty: ref Holds a reference to an object.
     */
    ref : null,
    /**
     * Property: count Counter for the number of times the label has been
     * clicked.
     */
    count : 0,
    /**
     * Property: isToggled True or False if the label is in a toggled state.
     */
    isToggled : false,
    /**
     * Property: labelText Holds the value of the label text.
     * 
     */
    labelText : "",
    /**
     * Property: previousText
     * 
     * Holds the previous text of the label before the text change.
     */
    previousText : "",

    /**
     * Constructor: oscar.Gui.ClickableLabel
     * 
     * Parameters: labelText - text of the label. options - {Object} An optional
     * object whose properties will be set on this instance.
     */
    initialize : function(labelText, options) {
        if (options)
            OpenLayers.Util.extend(this, options);
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
        this.events.register("labelTextChanged", this, this.updateText);
        this.createContainer();
        this.setLabelText(labelText);
    },
    /**
     * Method: createContainer
     * 
     * Creates the container div || span based on the style attribute.
     */
    createContainer : function() {
        if (this.style == this.styles.BLOCK)
            this.container = document.createElement("div");
        else if (this.style == this.styles.INLINE)
            this.container = document.createElement("span");
        else {
            throw "Invalid style: " + this.style;
            return;
        }
        var scope = this;
        $$(this.container).mouseup(function() {
            scope.clicked();
        });
    },
    /**
     * Method: clicked
     * 
     * Triggers the labelClicked event passing the object reference to the
     * listeners.
     */
    clicked : function(e) {
        this.count++;
        this.isToggled = !this.isToggled;
        this.events.triggerEvent("labelClicked", this)
    },

    /**
     * APIMethod: setLabelText
     * 
     * Sets the label text and triggers the labelTextChanged event.
     */
    setLabelText : function(labelText) {
        this.previousText = this.labelText;
        this.labelText = labelText;
        this.events.triggerEvent("labelTextChanged");
    },
    /**
     * Method: updateText
     * 
     * Updates the text of the label.
     */
    updateText : function(e) {
        this.container.innerHTML = this.labelText;
    },

    /**
     * APIMethod: appendTo
     * 
     * Appends the the container to the element provided.
     */
    appendTo : function(elem) {
        elem.appendChild(this.container);
    },

    /**
     * Method: revert Reverts the label to the previous text.
     */
    revert : function() {
        if (this.previousText.length > 0)
            this.setLabelText(this.previousText);
    },

    /**
     * APIMethod: toggle
     * 
     * Returns true or false if the label has been toggled
     */
    toggle : function() {
        return this.isToggled;
    },

    /**
     * APIMethod: applyClass
     * 
     * Takes a single CSS class or an array of CSS classes and adds them to the
     * container.
     * 
     * Parameters: cls - the CSS class or array of CSS classes should be
     * applied.
     */
    applyClass : function(cls) {
        if (!(cls instanceof Array)) {
            cls = [ cls ];
        }
        for (var i = 0; i < cls.length; i++) {
            oscar.jQuery(this.container).addClass(cls[i]);
        }
    },
    /**
     * APIMethod: removeClass
     * 
     * Takes a single CSS class or an array of CSS classes and removes them from
     * the container.
     * 
     * Parameters: cls - the CSS class for array of CSS classes should be
     * removed.
     */
    removeClass : function(cls) {
        if (!(cls instanceof Array)) {
            cls = [ cls ];
        }
        for (var i = 0; i < cls.length; i++) {
            oscar.jQuery(this.container).removeClass(cls[i]);
        }
    },

    /**
     * APIMethod: reset
     * 
     * Resets the counter to 0
     */
    reset : function() {
        this.count = 0;
    },

    /**
     * APIMethod: setTooltip
     * 
     * Parameters:
     * 
     * tooltipText - <String> Adds a tooltip to the label element
     */
    setTooltip : function(tooltipText) {
        oscar.jQuery(this.container).attr("title", tooltipText);
    },

    /**
     * Constant: CLASS_NAME - oscar.Gui.ClickableLabel
     */
    CLASS_NAME : "oscar.Gui.ClickableLabel"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Gui.ComboBox
 * 
 * A GUI ComboBox widget that user can directly type in the input field
 * 
 * or select from the drop down list.
 */
oscar.Gui.ComboBox = oscar.BaseClass(oscar.Gui, {
    /**
     * Properties:
     */
    /**
     * APIProperty: data
     * 
     * Reference to the list that user can select.
     */
    data : null,
    /**
     * APIProperty: container
     * 
     * Reference to the input default value.
     */
    defaultValue : "",
    /**
     * APIProperty: source
     * 
     * Reference to the select source based on the user input.
     */
    source : null,

    EVENT_TYPES : [ "onSelect", "onRender", "onChange" ],
    /**
     * Constructor: oscar.Gui.ComboBox
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */

    initialize : function(options) {
        this.EVENT_TYPES = oscar.Gui.ComboBox.prototype.EVENT_TYPES.concat(oscar.Gui.prototype.EVENT_TYPES);
        oscar.Gui.prototype.initialize.apply(this, [ options ]);
        this.draw();

    },
    /**
     * Method: draw
     * 
     * Creates the input field with a drop-down list .
     */

    draw : function() {
        oscar.Gui.prototype.draw.apply(this);
        this.input = document.createElement("input");
        this.input.setAttribute("class", "comboBoxInput");
        if (this.defaultValue != "") {
            this.input.value = this.defaultValue;
        }
        this.div.appendChild(this.input);
        var scope = this;
        var input = oscar.jQuery(this.input).autocomplete({
            delay : 0,
            minLength : 0,
            source : scope.source.bind(scope),
            select : scope.select.bind(scope),
            change : scope.change.bind(scope)
        });
        input.addClass("ui-widget ui-widget-content ui-corner-left");
        input.data("uiAutocomplete")._renderItem = scope.render.bind(scope);
        this.button = document.createElement("button");
        this.button.setAttribute("type", "button");
        this.button.innerHTML = "&nbsp";

        oscar.jQuery(this.button).attr("tabIndex", -1).attr("title", oscar.i18n("showall")).insertAfter(this.input).button({
            icons : {
                primary : "ui-icon-triangle-1-s"
            },
            text : false
        }).removeClass("ui-corner-all").addClass("ui-corner-right ui-button-icon").click(function() {
            // close if already visible
            if (oscar.jQuery(scope.input).autocomplete("widget").is(":visible")) {
                oscar.jQuery(scope.input).autocomplete("close");
                return;
            }
            // pass empty string as value to search for, displaying all results
            oscar.jQuery(scope.input).autocomplete("search", "");
            oscar.jQuery(scope.input).focus();
        });
    },
    /**
     * Method:select
     * 
     * Triggers the select event passing the object reference to the listeners.
     */
    select : function(event, ui) {
        if (this.onSelect) {
            this.onSelect.call(event.target, event, ui);
        }
        this.events.triggerEvent("onSelect", event.target.value);

    },
    /**
     * Method onSelect When the item selected, set the div element value as the
     * item value
     * 
     */
    onSelect : function(value) {
        var div = document.createElement("div")
        div.value = value;
    },
    /**
     * Method:render
     * 
     * Triggers the list item render event passing the item reference to the
     * listeners.
     */
    render : function(ul, item) {
        this.events.triggerEvent("onRender", ul, item);
        return this.onRender(ul, item);
    },

    /**
     * Method:onRender
     * 
     * Add the item to the ul as "li" element.
     * 
     */
    onRender : function(ul, item) {
        var li = document.createElement("li")
        return oscar.jQuery(li).data("ui-autocomplete-item", item).append("<a>" + item.label + "</a>").appendTo(ul);
    },

    /**
     * Method:change
     * 
     * Triggers the change event passing the input value reference to the
     * listeners.
     */
    change : function(event, ui) {
        if (this.onChange) {
            this.onChange.call(event.target, event, ui);
        }
        this.events.triggerEvent("onChange", event.target.value);
    },

    /**
     * Method onChange When the input value changed, set the div element value
     * as new value
     * 
     */
    onChange : function(value) {
        var div = document.createElement("div")
        div.value = value;
    },
    /**
     * Constant: CLASS_NAME - oscar.Gui.ComboBox
     */
    CLASS_NAME : "oscar.Gui.ComboBox"

});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 * 
 * Copyright 2014 CARIS <http://www.caris.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Gui.LinkedLists = new oscar.BaseClass(oscar.Gui, {
    displayClass : "oscarGuiLinkedLists",
    availableText : "",
    selectedText : "",
    EVENT_TYPES : [ "sourceReceived", "destinationReceived" ],
    initialize : function(options) {
        this.EVENT_TYPES = this.EVENT_TYPES.concat(oscar.Gui.prototype.EVENT_TYPES);
        oscar.Gui.prototype.initialize.apply(this, [ options ]);
        this.connectionClass = OpenLayers.Util.createUniqueID("connection");
        this.events.register("sourceReceived", this, this.sourceReceived);
        this.events.register("destinationReceived", this, this.destinationReceived);
        
    },
    filter : function() {
        return true;
    },
    destinationFilter : function() {
        return true;
    },
    draw : function() {
        var sourceId = OpenLayers.Util.createUniqueID("source");
        var destinationId = OpenLayers.Util.createUniqueID("destination");
        
        oscar.Gui.prototype.draw.apply(this);
        var $this = $$(this.div);
        this.sourceDiv = $$("<div></div>");
        this.sourceDiv.addClass("sourceDiv");
        this.sourceTextDiv = $$("<div></div>").html(oscar.i18n(this.availableText));
        this.sourceTextDiv.addClass("sourceText");
        this.navTextDiv = $$("<div></div>").html('&nbsp;');
        this.navTextDiv.addClass("navText");
        this.selectedTextDiv = $$("<div></div>").html(oscar.i18n(this.selectedText));
        this.selectedTextDiv.addClass("selectedText");
        
        this.sourceList = $$("<ul></ul>");
        this.sourceList.addClass(this.connectionClass);
        this.sourceList.attr("id", sourceId);
        this.sourceDiv.append(this.sourceList);
        
        this.buttonsDiv = $$("<div></div>");
        this.buttonsDiv.addClass("buttonsDiv");
        
        this.destinationDiv = $$("<div></div>");
        this.destinationDiv.addClass("selectedDiv");
        
        this.destinationList = $$("<ul></ul>");
        this.destinationList.attr("id", destinationId);
        this.destinationList.addClass(this.connectionClass);
        
        this.destinationDiv.append(this.destinationList);
        
        $this.append(this.sourceTextDiv);
        $this.append(this.navTextDiv);
        $this.append(this.selectedTextDiv);
        $this.append("<br clear='left'>");
        $this.append(this.sourceDiv);
        $this.append(this.sourceDiv);
        $this.append(this.sourceDiv);
        $this.append(this.buttonsDiv);
        $this.append(this.destinationDiv);
        
        $this.addClass(this.displayClass);
        this.buildButtons();
        $this.append("<br clear='left'>");
        $this.disableSelection();
        
    },
    sourceReceived : function(li) {
        this.sourceList.append(li);
        if (this.sourceDecorator) {
            this.sourceDecorator(li);
        }
        this.filter(li);
    },
    destinationReceived : function(li) {
        // check to see if this already exists
        var scope = this;
        var alreadyExists = false;
        this.destinationList.children().each(function() {
            var $this = $$(this);
            if (scope.compare(li, $$(this))) {
                alreadyExists = true;
            }
        });
        if (!alreadyExists) {
            this.destinationList.append(li);
        }
        if (this.destinationDecorator) {
            this.destinationDecorator(li);
        }
        this.destinationFilter(li);
    },
    
    buildButtons : function() {
        var scope = this;
        this.toTheRight = $$("<a onclick='return false;'></a>").html(oscar.i18n("MoveAllToTheRight"));
        
        this.toTheRight.button({
            icons : {
                primary : "ui-icon-seek-next"
            },
            text : false
        });
        this.toTheRight.click(function() {
            var available = scope.sourceList.children();
            available.each(function() {
                scope.events.triggerEvent("destinationReceived", $$(this));
            });
            scope.destinationList.sortable('refresh');
            return false;
        });
        this.toTheRight.css("float", "center");
        
        this.toTheLeft = $$("<a onclick='return false;'></a>").html(oscar.i18n("MoveAllToTheLeft"));
        
        this.toTheLeft.button({
            icons : {
                primary : "ui-icon-seek-prev"
            },
            text : false
        });
        this.toTheLeft.click(function() {
            var available = scope.destinationList.children();
            available.each(function() {
                scope.events.triggerEvent("sourceReceived", $$(this));
            });
            return false;
        });
        this.toTheLeft.css("float", "center");
        this.buttonsDiv.append(this.toTheRight);
        this.buttonsDiv.append(this.toTheLeft);
    },
    addToSourceList : function(listItem) {
        var li = $$("<li></li>").html(listItem.text);
        li.attr("title", listItem.text);
        var data = listItem.getData();
        
        for ( var p in data) {
            li.data(p, data[p]);
        }
        
        li.addClass("ui-state-default");
        if (!this.isSelected(li)) {
            this.events.triggerEvent("sourceReceived", li);
        }
        this.sortable();
        return li;
    },
    addToDestinationList : function(listItem) {
        var li = $$("<li></li>").html(listItem.text);
        li.attr("title", listItem.text);
        var data = listItem.getData();
        
        for ( var p in data) {
            li.data(p, data[p]);
        }
        li.addClass("ui-state-default");
        this.events.triggerEvent("destinationReceived", li);
        this.sortable();
        
        return li;
    },
    sortable : function() {
        var scope = this;
        $$(this.sourceList).sortable({
            connectWith : "." + this.connectionClass,
            dropOnEmpty : true,
            cursor : "move",
            receive : function(event, ui) {
                scope.events.triggerEvent("sourceReceived", ui.item);
            }
        }).disableSelection();
        
        $$(this.destinationList).sortable({
            connectWith : "." + this.connectionClass,
            dropOnEmpty : true,
            cursor : "move",
            receive : function(event, ui) {
                scope.events.triggerEvent("destinationReceived", ui.item);
            }
        }).disableSelection();
    },
    getAvailable : function() {
        return this.sourceList.children();
    },
    getSelected : function() {
        return this.destinationList.children();
    },
    isSelected : function(li) {
        var isFound = false;
        var scope = this;
        this.destinationList.children().each(function() {
            if (scope.compare(li, $$(this))) {
                isFound = true;
            }
        });
        return isFound;
    },
    clearSourceList : function() {
        this.sourceList.empty();
    },
    clearDestinationList : function() {
        this.destinationList.empty();
    },
    compare : function(a, b) {
        if (a.html() == b.html())
            return true;
        else
            return false;
    },
    showHelp : function(str) {
        if (!this.helpDiv) {
            this.helpDiv = $$("<div></div>");
            this.helpDiv.addClass("help2");
            $$(this.div).after(this.helpDiv);
        }
        this.helpDiv.html("");
        this.helpDiv.html(oscar.i18n(str));
        
    },
    CLASS_NAME : "oscar.Gui.LinkedLists"

});
oscar.ListItem = new oscar.BaseClass({
    text : "",
    data : {},
    initialize : function(text, data) {
        this.text = text;
        if (data) {
            this.data = data;
        }
    },
    setData : function(data) {
        this.data = data;
    },
    getData : function(data) {
        return this.data;
    },
    CLASS_NAME : "oscar.ListItem"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Gui.Wizard = new oscar.BaseClass(oscar.Gui, {
    initialize : function(options) {
        oscar.Gui.prototype.initialize.apply(this, [ options ]);
    },
    launch : function() {
        if (console) {
            console.log("override this method for your wizard");
        }
    },
    previous : function() {
    },
    next : function() {
        this.step++;
        var fn = this["steps"]["step_" + this.step];
        if (fn)
            fn.apply(this);
    },
    CLASS_NAME : "oscar.Gui.Wizard"
});

oscar.Gui.Wizard.DirectDownload = new oscar.BaseClass(oscar.Gui.Wizard, {
    protocol : null,
    url : null,
    initialize : function(protocol, url, options) {
        oscar.Gui.Wizard.prototype.initialize.apply(this, [ options ]);
        this.protocol = protocol;
        this.url = url;
    },

    launch : function() {
    },
    CLASS_NAME : "oscar.Gui.Wizard.DirectDownload"
});

oscar.Gui.Wizard.WebCoverageServiceDownload = new oscar.BaseClass(oscar.Gui.Wizard, {
    icon : "ui-icon-wcs-download",
    protocol : null,
    url : null,
    initialize : function(protocol, url, options) {
        oscar.Gui.Wizard.prototype.initialize.apply(this, [ options ]);
        this.protocol = protocol;
        this.url = $$.trim(url);
    },

    launch : function() {
        var database = new oscar.Util.Database();
        database.addTable("sources", [ "id", "title", "bbox", "abstract", "fk_capabilities", "dataType" ]);
        database.addTable("capabilities", [ "capabilities" ]);
        var params = {
            request : "GetCapabilities",
            service : "WCS",
            version : "1.1.0"
        }
        var success = function(response) {
            var reader = new oscar.Format.WCSCapabilities();
            var capabilities = reader.read(response.responseXML);
            var capIndex = database.addRecord("capabilities", {
                capabilities : capabilities
            });
            coverages = oscar.Util.Metadata.getCoverages(capabilities);
            for ( var c in coverages) {
                var coverage = coverages[c];
                var bbox = new OpenLayers.Bounds(coverage.wgs84BoundingBox.west, coverage.wgs84BoundingBox.south, coverage.wgs84BoundingBox.east, coverage.wgs84BoundingBox.north);
                var transformedBounds = bbox.clone();
                transformedBounds.transform(new OpenLayers.Projection("EPSG:4326"), this.map.getProjectionObject());
                var record = {
                    "id" : coverage.identifier,
                    "title" : coverage.title,
                    "abstract" : coverage["abstract"],
                    "bbox" : transformedBounds,
                    "fk_capabilities" : capIndex,
                    "dataType" : "wcs"
                }
                var pk = database.addRecord("sources", record);
            }
        };

        this.wcsRequest = OpenLayers.Request.GET({
            url : this.url,
            params : params,
            async : false,
            success : success,
            scope : this
        });
        // remove an existing data discovery control
        try {
            this.map.getControlsByClass("oscar.Control.DataDiscovery")[0].events.triggerEvent("closed");
        } catch (err) {
        }

        var title = cswget.title(this.record) || cswget.identifier(this.record);

        this.ctrl = new oscar.Control.DataDiscovery(database, {
            closable : true,
            showAbstract : false,
            query : title
        });
        this.ctrl.events.on({
            "closed" : function() {
                this.map.removeControl(this.ctrl);
            },
            scope : this
        });
        this.map.addControl(this.ctrl);

    },

    CLASS_NAME : "oscar.Gui.Wizard.WebCoverageServiceDownload"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Handler
 * 
 * The oscar.Handler object defines the Oscar Handler namespace. Handler objects
 * are used to execute actions (such as WFS AJAX requests) using a {<OpenLayers.Geometry>}
 * object and a {<oscar.ox.Theme>} object.
 * 
 */

oscar.Handler = oscar.BaseClass({
    /**
     * Constructor: oscar.Handler
     * 
     * Initialize the handler object.
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     * 
     */
    initialize : function(options) {

    },
    /**
     * APIMethod: execute
     * 
     * The *execute* method encapsulates the actual job to be done. All
     * subclasses of the Oscar Handler should override this method to complete
     * the its own business logic.
     * 
     * Parameters: geom - {<OpenLayers.Geometry>} object. theme - {<oscar.ox.Theme>}
     * object.
     */
    execute : function(geom, theme) {

    },
    /**
     * Constant: CLASS_NAME - oscar.Handler
     */
    CLASS_NAME : "oscar.Handler"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 * 
 * Copyright 2014 CARIS <http://www.caris.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Handler.CSW
 * 
 * Executes CSW requests
 * 
 * Inherits from: - <oscar.Handler>
 * 
 */

oscar.Handler.CSW = new oscar.BaseClass(oscar.Handler, {
    DEFAULT_SEARCH_PARAMETERS : {
        request : "GetRecords",
        service : "csw",
        version : "2.0.2",
        resultType : "results",
        outputSchema : "http://www.opengis.net/cat/csw/2.0.2",
        maxRecords : 10,
        startPosition : 1
    },
    EVENT_TYPES : [ "beforeSearch", "afterSearch", "advancedSearch", "displayfilter", "success", "failure" ],
    catalogue : null,
    initialize : function(catalogueService, options) {
        if ($$.isArray(catalogueService)) {
            this.catalogue = catalogueService[0];
        } else {
            this.catalogue = catalogueService;
        }
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
        if (this.catalogue.isSetDefaultOutputSchema()) {
            this.DEFAULT_SEARCH_PARAMETERS.outputSchema = this.catalogue.getDefaultOutputSchema();
        }
        
        if (options) {
            if (options.DEFAULT_SEARCH_PARAMETERS) {
                OpenLayers.Util.extend(this.DEFAULT_SEARCH_PARAMETERS, options.DEFAULT_SEARCH_PARAMETERS);
            }
        }
    },
    setMap : function(map) {
        this.map = map;
    },
    search : function(query) {
        this.parameters = OpenLayers.Util.extend({}, this.DEFAULT_SEARCH_PARAMETERS);

        var csw_query = {
            "ElementSetName" : {
                "value" : "full"
            }
        }

        var filter = filter = query.toFilter();;
        if (filter) {
            csw_query["Constraint"] = {
                "version" : "1.1.0",
                "Filter" : filter
            }
        }
        
        this.parameters.Query = csw_query;
        this.createRequest();
    },
    createRequest : function() {
        this.events.triggerEvent("beforeSearch");
        var formatter = new OpenLayers.Format.CSWGetRecords();
        var data = formatter.write(this.parameters);
        OpenLayers.Request.POST({
            url : this.catalogue.getUrl(oscar.ogc.CatalogueService.prototype.GETRECORDS, "POST"),
            success : this.success,
            failure : this.failure,
            data : data,
            scope : this
        });
    },
    success : function(resp) {
        this.events.triggerEvent("afterSearch");
        var formatter = new OpenLayers.Format.CSWGetRecords();
        this.results = formatter.read(resp.responseXML);
        this.results.SearchParameters = this.parameters;
        this.events.triggerEvent("success", this.results);
    },
    failure : function() {
        this.events.triggerEvent("failure");
    },
    next : function() {
        var info = this.results.SearchResults;
        this.parameters.startPosition = info.nextRecord;
        var newStartPosition = info.nextRecord;
        if (newStartPosition > info.numberOfRecordsMatched) {
            return;
        }
        this.parameters.startPosition = newStartPosition;
        this.createRequest();
    },
    previous : function() {
        var info = this.results.SearchResults;
        var matched = info.numberOfRecordsMatched;
        var returned = info.numberOfRecordsReturned;
        var next = info.nextRecord;
        
        if (next > 0) {
            this.parameters.startPosition = next - returned - this.DEFAULT_SEARCH_PARAMETERS.maxRecords;
        } else {
            this.parameters.startPosition = matched - returned + 1 - this.DEFAULT_SEARCH_PARAMETERS.maxRecords;
        }
        if (this.parameters.startPosition < 0)
            this.parameters.startPosition = 1;
        
        this.createRequest();
    },
    
    jumpTo : function(record) {
        this.parameters.startPosition = record;
        this.createRequest();
    },
    CLASS_NAME : "oscar.Handler.CSW"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Handler.WFS
 * 
 * Executes WFS requests based on selection layers of a theme.
 * 
 * Inherits from: - <oscar.Handler>
 * 
 */

oscar.Handler.WFS = oscar.BaseClass(oscar.Handler, {
    /**
     * Constant: EVENT_TYPES
     * 
     * complete - Triggered when all requests have been completed. Passes the
     * layer object reference. requestComplete - Triggered when a request has
     * completed. Used internally. beforeStart - Triggered when the requests are
     * about to start. clean - Triggered by this.clean() method.
     */
    EVENT_TYPES : [ 'complete', 'requestComplete', 'beforeStart', 'clean' ],
    /**
     * Property: events {<OpenLayers.Events>) An events object that handles all
     * events in the handler.
     */
    events : null,
    /**
     * Property: singleRequest
     * 
     * Specifies if a source will send a request per feature type or combine
     * them for a single request.
     */
    singleRequest : false,
    /**
     * APIProperty: requestCounter
     * 
     * {Integer} Keeps track of the number of requests being executed.
     */
    requestCounter : 0,
    /**
     * Property: feats
     * 
     * An array container to hold the features from all requests.
     */
    feats : null,
    /**
     * Property: map
     * 
     * Reference to the map object.
     */
    map : null,

    /**
     * Property: dialog
     * 
     * Reference to the dialog object.
     */
    dialog : null,

    /**
     * Constructor: oscar.Handler.WFS
     * 
     * Parameters:
     * 
     * map - <Oscar.Map> A reference to a <Oscar.Map> object. options - {Object}
     * An optional object whose properties will be set on this instance.
     * 
     */
    initialize : function(map, options) {
        this.feats = [];
        this.map = map;
        OpenLayers.Util.extend(this, options);
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
        this.events.register("requestComplete", this, this.requestComplete);
        this.events.register("clean", this, this.clean);
    },
    /**
     * APIMethod: execute
     * 
     * Launches a GetFeature request to obtain features from a WFS service.
     * 
     * Parameters: geom - <OpenLayers.Geometry> object. theme - <oscar.ox.Theme>
     * object.
     */
    execute : function(geom, theme) {
        this.events.triggerEvent('beforeStart');
        this.events.triggerEvent("clean");
        var selectionService = theme.getSelectionService();
        for (var i = 0; i < selectionService.length; i++) {
            var serviceEntry = selectionService[i];
            if (serviceEntry.geometryName == "") {
                serviceEntry.geometryName = this.doDescribeFeatureTypeRequest(serviceEntry, geom, theme);
            } else {
                this.doGetFeatureRequest(serviceEntry, geom, theme);
            }
        }
    },
    /**
     * Method: doDescribeFeatureTypeRequest
     * 
     * This method launches an AJAX DescribeFeatureType request against a WFS
     * service, parses the returned document, retrieves the service geometry
     * property name and assign it to the ServiceEntry object.
     * 
     * Parameters: serviceEntry - reference to a {<oscar.ox.ServiceEntry>}
     * object. geom - reference to a {<OpenLayers.Geometry>} object. theme -
     * reference to a {<oscar.ox.Theme>} object.
     */
    doDescribeFeatureTypeRequest : function(serviceEntry, geom, theme) {
        var params = {
            service : "WFS",
            request : "DescribeFeatureType",
            version : serviceEntry.version,
			typename:serviceEntry.identifiers[0]
        };
        var onSuccess = function(resp) {
            serviceEntry.schema = new OpenLayers.Format.WFSDescribeFeatureType().read(resp.responseXML);
            this.doGetFeatureRequest(serviceEntry, geom, theme);
        }
        var onFail = function(resp) {
            this.events.triggerEvent("requestComplete");
        }
        OpenLayers.Request.GET({
            url : serviceEntry.url,
            params : params,
            success : onSuccess,
            fail : onFail,
            scope : this
        });
    },


    /**
     * Method: doGetFeatureRequest
     * 
     * Executes an AJAX GetFeature request to a WFS service.
     * 
     * Parameters: serviceEntry - reference to a {<oscar.ox.ServiceEntry>}
     * object. geom - reference to a {<OpenLayers.Geometry>} object. theme
     * -reference to a {<oscar.ox.Theme>} object.
     */
    doGetFeatureRequest : function(serviceEntry, geom, theme) {
		
        var sFilter = new OpenLayers.Filter.Spatial({
            type : OpenLayers.Filter.Spatial.BBOX,
            value : this.getBounds(geom),
            projection : theme.srs
        });

        var formatOptions = {
            extractAttributes : true,
			autoConfig:true,
			singleFeatureType:false,
			srsName:theme.srs
        };

        if (serviceEntry.version != "1.0.0") {
            formatOptions.xy = (this.map.projection.proj.projName == "longlat") ? false : true;
        }
		
		var numRequests = serviceEntry.identifiers.length;
		var completedRequests = 0;
		
		var featureList=[];
		
		var callbackfn = function(evt) {
			featureList = featureList.concat(evt.features);
			completedRequests++;
		}
		var me = this;
		
		var reqCounter = window.setInterval(function() {
			if(completedRequests>=numRequests) {
				window.clearInterval(reqCounter);
				me.drawFeatures(featureList);
			}
		},500);
		 this.showDialog();
		
		for (var i in serviceEntry.identifiers) {
			var protocol = new OpenLayers.Protocol.WFS({
				url : serviceEntry.url,
				version : serviceEntry.version,
				featureType : serviceEntry.identifiers[i],
				singleFeatureType:false,
				geometryName : serviceEntry.schema.featureTypes[0].properties[0].name,
				formatOptions : formatOptions,
				filter : sFilter,
				scope : this
			});
			
			protocol.read({
				callback:callbackfn
			});
		}
    },
	
	drawFeatures:function(features) {
	
	    var now = new Date();
        var hour = now.getHours();
        var minutes = now.getMinutes();
        var ampm = (hour > 12) ? "am" : "pm";

        this.layer = new OpenLayers.Layer.Vector("@ " + hour + ":" + minutes + " " + ampm, {
            temporary : true,
            displayInLayerSwitcher : false
        });

        this.layer.events.on({
            "featureselected" : function(evt) {
                var feature = evt.feature;
                var str = "<table>";
                for ( var key in feature.attributes) {
                    var attr = feature.attributes[key];
                    str += "<tr><td>" + key;
                    str += "</td><td>" + attr + "</td></tr>";
                }
                str += "</table>";
                var popup = new oscar.FramedCloud("id", feature.geometry.getBounds().getCenterLonLat(), null, str, null, true);
                popup.autoSize = false;
                feature.popup = popup;
                feature.layer.map.addPopup(popup);

            },
            "featureunselected" : function(evt) {
                var feature = evt.feature;
                feature.layer.map.removePopup(feature.popup);
                feature.popup = null;
            },
            "beforefeatureremoved" : function(evt) {
                feature = evt.feature;
                if (feature.popup) {
                    feature.layer.map.removePopup(feature.popup);
                    feature.popup = null;
                }
            },
            "loadstart" : function() {
                this.showDialog();
            },
            "loadend" : function() {
                this.events.triggerEvent("requestComplete");
            },
            scope : this
        });

        this.map.addLayer(this.layer);
		this.layer.addFeatures(features);
		this.events.triggerEvent("requestComplete");
	},
	
	
    /**
     * Method: getBounds
     * 
     * Takes a {<OpenLayers.Geometry>} object and returns the bounds for it. If
     * the Geometry is a Point object it will create a small Bounds object from
     * a 2x2 pixel box.
     * 
     * Parameters: geom - {<OpenLayers.Geometry>} The geometry object to get
     * the bounds for.
     * 
     * Returns: {<OpenLayers.Bounds>} Bounds object of the geometry.
     */
    getBounds : function(geom) {
        if (geom.CLASS_NAME !== "OpenLayers.Geometry.Point") {
            return geom.getBounds();
        }

        // Get the pixel location of the point geometry and build a bbox
        // around it.
        var pixel = this.map.getPixelFromLonLat(new OpenLayers.LonLat(geom.x, geom.y));
        var topRight = this.map.getLonLatFromPixel(new OpenLayers.Pixel(pixel.x + 2, pixel.y + 2));
        var botLeft = this.map.getLonLatFromPixel(new OpenLayers.Pixel(pixel.x - 2, pixel.y - 2));

        var lRing = new OpenLayers.Geometry.LinearRing();
        lRing.addComponent(new OpenLayers.Geometry.Point(botLeft.lon, botLeft.lat));
        lRing.addComponent(new OpenLayers.Geometry.Point(topRight.lon, topRight.lat));
        var poly = new OpenLayers.Geometry.Polygon([ lRing ]);
        return poly.getBounds();

    },

    /**
     * Method: showDialog Shows the selection dialog.
     */
    showDialog : function() {
        this.dialog = new oscar.Gui.Dialog("Selection", {
            draggable : true,
            modal : false,
            width : 250,
            height : 100,
            zIndex : 2001,
            fixedcenter : false
        });
        this.dialog.setHeader(oscar.i18n("Loading"));
        this.dialog.setContent(oscar.i18n("Please Wait"));
        this.dialog.show();
    },
    requestComplete : function() {
        if (this.dialog) {
            this.dialog.hide();
            this.dialog = null;
        }
        try {
            if (this.layer.features.length > 0) {
                this.events.triggerEvent("complete", this.layer);
            }
        } catch (err) {
        }

    },

    clean : function() {
        if (this.layer) {
            this.layer.removeAllFeatures();
        }
    },

    /**
     * Constant: CLASS_NAME - oscar.Handler.WFS
     */
    CLASS_NAME : "oscar.Handler.WFS"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control
 * 
 * The components under *oscar.Control* package add extra effects and behaviors
 * to user's maps. This oscar.Control object defines the Oscar Control
 * namespace.
 * 
 * Inherits from: - <OpenLayers.Control>
 * 
 */
oscar.Control = oscar.BaseClass(OpenLayers.Control, {
    /**
     * APIMethod: deactivate
     * 
     * Called when the object is deactivated.
     */
    deactivate : function() {
        OpenLayers.Control.prototype.deactivate.apply(this, [ arguments ]);
    },
    /**
     * Method: draw
     * 
     * Draws the Control onto the map at the position specified.
     * 
     * Parameters: px - the position where the Control should be drawn.
     * 
     * Returns: div - container of the Control.
     */
    draw : function(px) {
        OpenLayers.Control.prototype.draw.apply(this, [ px ]);

        return this.div;
    },

    /**
     * Constant: CLASS_NAME - oscar.Control
     */
    CLASS_NAME : "oscar.Control"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.ArgParser The Oscar ArgParser extends the
 * <OpenLayers.Control.ArgParser> and parses the parameters from href string for
 * values of lon, lat, zoom and other layers information.
 * 
 * Inherits from: - <OpenLayers.Control.ArgParser>
 * 
 */
oscar.Control.ArgParser = oscar.BaseClass(OpenLayers.Control.ArgParser, {
    /**
     * Property: args Stores the map property.
     */
    args : null,

    /**
     * Constructor: oscar.Control.ArgParser
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function() {
        OpenLayers.Control.ArgParser.prototype.initialize.apply(this, []);
    },

    /**
     * APIMethod: setMap Set the map property for the control.
     * 
     * Parameters: map - {<oscar.Map>}
     */
    setMap : function(map) {
        this.map = map;
        this.args = OpenLayers.Util.getParameters();
    },

    /**
     * Constant: CLASS_NAME - oscar.Control.ArgParser
     */
    CLASS_NAME : 'oscar.Control.ArgParser'

});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.DragPanel
 * 
 * This control is a base control for creating draggable planels.
 * 
 * 
 * Inherits from: - <oscar.Control>
 * 
 */
oscar.Control.DragPanel = oscar.BaseClass(oscar.Control, {
    /**
     * APIProperty: resizable {Boolean} allows the panel to be resizable,
     * default is true.
     */
    resizable : true,
    /**
     * APIProperty: drawPanel {Boolean} whether the theme switcher should be
     * drawn, default is true.
     */
    drawPanel : true,

    /**
     * APIProperty: closed {Boolean} initial display of the panel, default is
     * true
     */
    closed : true,

    /**
     * APIProperty: closable {Boolean} Should the panel be able to be closed.
     * Defaults to false.
     */
    closable : false,

    /**
     * APIProperty: collapsible {Boolean} Should the panel be collapsible.
     * Defaults to true.
     */
    collapsible : true,
    /**
     * APIProperty: iconClass {String} class to usee for the icon of the panel.
     * Default is oDragPanel.
     */
    iconClass : "",
    /**
     * APIProperty: tooltipText {String} Key or text to display as a tool tip.
     */
    tooltipText : "",
    /**
     * APIProperty: titleText {String} Key or text to be used for the title of
     * the panel.
     */
    titleText : "",

    EVENT_TYPES : [ "closed", "minimized", "maximized" ],
    events : null,
    /**
     * Constructor: Creates a new instance of the DragPanel control.
     * 
     * Parameters: - options {Object} Override default property values
     * 
     * Usage: (code) var dragPanel= new
     * oscar.Control.DragPanel({resizable:true,closed:false}); (end)
     */
    initialize : function(options) {
        this.EVENT_TYPES = oscar.Control.DragPanel.prototype.EVENT_TYPES.concat(OpenLayers.Control.prototype.EVENT_TYPES);
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
        oscar.Control.prototype.initialize.apply(this, [ options ]);
    },
    /**
     * APIMethod: setMap Sets a reference to the map object.
     * 
     * Parameters: - map <OpenLayers.Map>
     */
    setMap : function(map) {
        this.map = map;
    },
    /**
     * APIMethod: draw Creates the panel to be rendered in the map.
     * 
     */
    draw : function() {
        oscar.Control.prototype.draw.apply(this);
        oscar.jQuery(this.div).addClass("o-widget");
        this.handle = document.createElement("div");
        oscar.jQuery(this.handle).addClass("handle");
        this.handle.setAttribute("title", oscar.i18n(this.tooltipText));
        this.icon = document.createElement("div");
        oscar.jQuery(this.icon).addClass("oIcon");
        if (this.iconClass && this.iconClass.length > 0) {
            oscar.jQuery(this.icon).addClass(this.iconClass);
        }
        this.title = document.createElement("div");
        var tSpan = document.createElement("span");
        oscar.jQuery(tSpan).addClass("panelTitleText");
        tSpan.innerHTML = oscar.i18n(this.titleText);
        this.title.appendChild(tSpan)
        oscar.jQuery(this.title).addClass("panelTitle");
        this.handle.appendChild(this.icon);
        this.handle.appendChild(this.title);

        if (this.closable) {
            this.closeBox = document.createElement("div");
            oscar.jQuery(this.closeBox).addClass("closeBox");
            var context = this;
            OpenLayers.Event.observe(this.closeBox, "click", function(e) {
                oscar.jQuery(context.div).fadeOut();
                context.events.triggerEvent("closed");
                // OpenLayers.Event.stop(e, true);
            });
            this.handle.appendChild(this.closeBox);
        }

        if (this.collapsible) {
            this.minMax = document.createElement("div");
            oscar.jQuery(this.minMax).addClass("minMax");
            oscar.jQuery(this.minMax).addClass("contentOpen");
            var context = this;
            OpenLayers.Event.observe(this.minMax, "click", function(e) {
                context.toggleContentDisplay();

                // OpenLayers.Event.stop(e, true);
            });
            oscar.jQuery(this.clicker).addClass("contentOpen");
            this.handle.appendChild(this.minMax);
        }

        this.div.appendChild(this.handle);
        this.content = document.createElement("div");

        oscar.jQuery(this.content).addClass("content");
        this.div.appendChild(this.content);
        OpenLayers.Event.observe(this.div, "mousedown", function(e) {
            OpenLayers.Event.stop(e, true);
        });

        oscar.jQuery(this.div).draggable({
            containment : "parent",
            start : function(event, ui) {
                oscar.jQuery(this).addClass("olDragDown");
                event.stopPropagation();
            },
            stop : function(event, ui) {
                oscar.jQuery(this).removeClass("olDragDown");
                event.stopPropagation();
            },
            drag : function(event, u) {
                event.stopPropagation();
            },
            cancel : "div.content"
        });
        if (this.resizable) {
            oscar.jQuery(this.div).resizable();
        }

        if (this.closed) {
            this.toggleContentDisplay();
        }

        oscar.jQuery(this.div).hide();

        var scope = this;
        if (this.drawPanel) {
            setTimeout(function() {
                oscar.jQuery(scope.div).fadeIn("fast");
            }, 0);
        }

        return this.div;
    },
    /**
     * APIMethod: setTitle Sets the title of the panel.
     * 
     * Parameters: - key {String} Translation key or text to use for the title.
     */
    setTitle : function(key) {
        this.title.innerHTML = oscar.i18n(key);
    },
    /**
     * APIMethod: toggleContentDisplay Opens or closes the display of the
     * content pane.
     */
    toggleContentDisplay : function() {
        if (oscar.jQuery(this.minMax).hasClass("contentOpen")) {
            oscar.jQuery(this.minMax).removeClass("contentOpen");
            oscar.jQuery(this.minMax).addClass("contentClosed");
            oscar.jQuery(this.div).addClass("small");
            oscar.jQuery(this.content).hide();
            if (this.resizable) {
                oscar.jQuery(this.div).resizable("destroy");
            }
        } else {
            oscar.jQuery(this.minMax).removeClass("contentClosed");
            oscar.jQuery(this.minMax).addClass("contentOpen");
            oscar.jQuery(this.div).removeClass("small");
            oscar.jQuery(this.content).show();
            if (this.resizable) {
                oscar.jQuery(this.div).resizable();
            }
        }
    },
    /**
     * Constant: CLASS_NAME
     */
    CLASS_NAME : "oscar.Control.DragPanel"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 * 
 * Copyright 2014 CARIS <http://www.caris.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.DataDiscovery
 * 
 * This control is used to allow users to search for data on the map and
 * provided a way for them to download or extract the data.
 * 
 */

oscar.Control.DataDiscovery = oscar.BaseClass(oscar.Control.DragPanel, {
    autoActivate : true,
    /**
     * APIProperty: tooltipText
     * 
     * {String} - translation key or text to use for the Theme Switcher tool
     * tip. Default is ThemeSwitcherToolTip.
     */
    tooltipText : "DataDiscoverySearchToolTip",
    /**
     * APIProperty: titleText
     * 
     * {String} - translation key or text to use for the Theme Switcher title.
     * Default is ThemeSwitcherTitle.
     */
    titleText : "DataDiscoverySearchTitle",
    /**
     * APIProperty: iconClass
     * 
     * {String} - css class to use for the icon of the widget.
     */
    iconClass : "",
    /**
     * APIProperty: closed
     * 
     * {Boolean} - Chooses if the widget should be open or closed initially.
     * Default is false.
     */
    closed : false,
    /**
     * APIProperty: resizable
     * 
     * {Boolean} - Chooses if the widget is allowed to be resized. Default is
     * true.
     */
    resizable : false,
    /**
     * APIProperty: collapsible
     * 
     * {Boolean} - Chooses if the widget can be collapsed. Default is true.
     */
    collapsible : true,
    
    /**
     * APIProperty: database {oscar.Util.Database} - The database object to use
     * for searching.
     */
    
    database : null,
    
    showAbstract : true,
    
    /**
     * APIProperty: styles
     * 
     * {Object} - Used to select and crop style.
     */
    
    styles : {
        select : {
            fillColor : "#fff",
            strokeColor : "#000",
            strokeWidth : 1,
            strokeDashstyle : 'solid',
            opacity : 0.1
        },
        temporary : {
            fillColor : "#ace",
            strokeColor : "#000",
            strokeWidth : 1,
            strokeDashstyle : 'dash',
            opacity : 0.1
        }
    },
    
    /**
     * Constructor: oscar.Control.DataDiscovery
     * 
     * Parameters:
     * 
     * db - {oscar.Util.Database} The database of results for searching.
     */
    initialize : function(db, options) {
        oscar.Control.DragPanel.prototype.initialize.apply(this, [ options ]);
        this.database = db;
        this.database.events.on({
            "dbupdated" : this.dbupdated,
            scope : this
        });
        this.events.on({
            "closed" : function() {
                this.deactivate();
            },
            scope : this
        });
    },
    dbupdated : function(e) {
    },
    /**
     * Method: setMap
     * 
     * Parameters:
     * 
     * map - {OpenLayers.Map}
     * 
     */
    setMap : function(map) {
        oscar.Control.prototype.setMap.apply(this, [ map ]);
        this.map.events.on({
            "moveend" : this.displayResults,
            scope : this
        });
        this.checkLayer();
    },
    
    /**
     * Method: draw
     * 
     * Draws the control.
     */
    draw : function() {
        oscar.Control.DragPanel.prototype.draw.apply(this);
        this.buildDiscoverPanels();
        return this.div;
    },
    
    /**
     * Method: buildDiscoveryPanel
     * 
     * Creates the different panels for the control.
     */
    buildDiscoverPanels : function() {
        this.discoverPanel = $$("<div></div>").addClass("discoverPanel");
        
        /* build the search header and panel */
        var searchHeader = $$("<h3></h3>");
        var searchHeaderAnchor = $$("<a></a>").attr("href", "#").html(oscar.i18n("Search"));
        searchHeader.append(searchHeaderAnchor);
        var searchPanel = $$("<div></div>").addClass("searchPanel");
        
        /* build the download options header and panel */
        var optionsHeader = $$("<h3></h3>");
        var optionsHeaderAnchor = $$("<a></a>").attr("href", "#").html(oscar.i18n("DownloadOptions"));
        optionsHeader.append(optionsHeaderAnchor);
        this.optionsPanel = $$("<div></div>").addClass("optionsPanel");
        
        /* build the download queue header and panel */
        this.queueHeader = $$("<h3></h3>");
        var queueHeaderAnchor = $$("<a></a>").attr("href", "#").html(oscar.i18n("DownloadQueue"));
        this.queueHeader.append(queueHeaderAnchor);
        this.queuePanel = $$("<div></div>").addClass("queuePanel");
        
        /* add everything to the main panel */
        $$(this.discoverPanel).append(searchHeader);
        $$(this.discoverPanel).append(searchPanel);
        $$(this.discoverPanel).append(optionsHeader);
        $$(this.discoverPanel).append(this.optionsPanel);
        $$(this.discoverPanel).append(this.queueHeader);
        $$(this.discoverPanel).append(this.queuePanel);
        
        var textEntryPanel = $$("<div></div>");
        textEntryPanel.css({
            'padding' : '2px 2px 2px 2px'
        });
        var reset = $$("<div></div>");
        this.resultsPanel = $$("<div></div>").addClass("resultsPanel");
        reset.addClass("oscar_Gui_MultiItemChooserTable_resetTable_disabled");
        reset.addClass("reset");
        var $mGlass = $$("<span></span>");
        $mGlass.addClass("magnifyingGlass");
        searchPanel.append($mGlass);
        searchPanel.append(textEntryPanel);
        searchPanel.append(this.resultsPanel);
        
        this.txt = $$("<input>").attr("type", "text").addClass("search");
        this.txt.css("width", "200px");
        textEntryPanel.append(this.txt);
        textEntryPanel.append(reset);
        
        var scope = this;
        
        reset.click(function() {
            scope.reset();
        });
        
        var br = $$("<br>").attr("clear", "left");
        textEntryPanel.append(br);
        
        this.txt.focus(function() {
            this.value = "";
        });
        
        this.txt.keyup(function(e) {
            switch (e.keyCode) {
                case 13:
                case 16:
                case 16:
                case 17:
                case 18:
                    return;
            }
            scope.displayResults();
        });
        
        var icons = {
            header : "ui-icon-circle-arrow-e",
            headerSelected : "ui-icon-circle-arrow-s"
        };
        oscar.jQuery(this.discoverPanel).accordion({
            collapsible : false,
            animated : false,
            icons : icons
        });
        OpenLayers.Event.observe(this.discoverPanel, "mousedown", function(e) {
            OpenLayers.Event.stop(e, true);
        });
        
        OpenLayers.Event.observe(this.discoverPanel, "click", function(e) {
            OpenLayers.Event.stop(e, true);
        });
        $$(this.content).append(this.discoverPanel);
        if (this.query) {
            this.txt.val(this.query);
        }
        this.getResults();
        
    },
    
    /**
     * Method: unselectFeature Removes the feature from the map and unselects
     * any feature from the list.
     */
    unselectFeature : function() {
        this.resultsPanel.children().each(function() {
            $$(this).removeClass("selected");
        });
        this.layer.removeAllFeatures();
    },
    /**
     * Method: reset Resets the search options and zooms the map to the max
     * extent.
     */
    reset : function() {
        this.unselectFeature();
        this.txt.val("");
        this.map.zoomToMaxExtent();
        this.displayResults();
    },
    
    /**
     * Method: getResults
     * 
     * Uses the database object to obtain an array of the available results.
     */
    
    getResults : function() {
        var scope = this;
        var columns = this.database.tables["sources"].columns;
        var records = this.database.tables["sources"].records;
        for (var r = 0; r < records.length; r++) {
            var record = records[r];
            var $recDiv = $$("<div></div>");
            $recDiv.html(record.title || record.id).addClass("result");
            var $dataType = $$("<div></div>");
            $dataType.addClass(record.dataType);
            $recDiv.prepend($dataType);
            for (var c = 0; c < columns.length; c++) {
                var column = columns[c];
                $recDiv.data(column, record[column]);
            }
            $recDiv.click(function() {
                var $this = $$(this);
                scope.unselectFeature();
                $this.addClass("selected");
                scope.discoverPanel.accordion("option", "active", 1);
                scope.drawFeature($this);
                
            });
            $recDiv.hide();
            this.resultsPanel.append($recDiv);
        }
        this.displayResults();
    },
    /**
     * Method: displayResults
     * 
     * Uses the array of results and validates it against any text entry in the
     * text field and the current extent of the map to decide if a result should
     * be displayed.
     */
    displayResults : function(e) {
        var scope = this;
        var query = this.txt.val().trim();
        var validResults = [];
        this.resultsPanel.children().each(function() {
            var $this = $$(this);
            var mapViewPort = scope.map.getExtent();
            var isInRange = (mapViewPort.containsBounds($this.data("bbox")) || mapViewPort.intersectsBounds($this.data("bbox")));
            
            var textFound = (query.length == 0 || $this.data("id").toLowerCase().indexOf(query.toLowerCase()) != -1 || $this.data("title").toLowerCase().indexOf(query.toLowerCase()) != -1);
            
            if (isInRange && textFound) {
                $this.show();
                validResults.push($this);
            } else {
                $this.hide();
            }
        });
        setTimeout(function() {
            if (validResults.length == 1) {
                validResults[0].click();
            }
        }, 500);
    },
    /**
     * Method: drawFeature Once the feature is selected in the results panel
     * this will display the feature on the map while activating the Download
     * Options panel.
     * 
     */
    drawFeature : function($div) {
        var bbox = $div.data("bbox");
        if (this.layer && this.layer.features.length > 0) {
            this.layer.removeAllFeatures();
        }
        var feat = new OpenLayers.Feature.Vector(bbox.toGeometry());
        feat.div = $div;
        this.layer.addFeatures(feat);
        this.layer.events.triggerEvent("loadend");
        
        var scope = this;
        setTimeout(function() {
            var selectFeature = scope.map.getControlsByClass("oscar.Control.SelectFeature")[0];
            if (selectFeature) {
                selectFeature.ctrl.unselectAll();
                selectFeature.ctrl.select(feat);
            }
        }, 0);
        
        var viewPort = this.map.getExtent();
        if (viewPort.containsBounds(bbox)) {
            this.map.zoomToExtent(bbox);
        }
        if (this.downloadOptions == null) {
            this.downloadOptions = new oscar.Gui.DownloadOptions({
                db : this.database,
                map : this.map,
                showAbstract : this.showAbstract
            });
            this.downloadOptions.events.on({
                "serviceReady" : this.queueDownload,
                scope : this
            });
            this.downloadOptions.appendTo(this.optionsPanel);
        }
        this.downloadOptions.setFeature(feat);
    },
    /**
     * Method: activate Called when activated.
     */
    activate : function() {
        oscar.Control.prototype.activate.apply(this);
        if (this.div) {
            oscar.jQuery(this.div).fadeIn();
        }
        this.checkLayer();
        this.displayResults();
    },
    /**
     * Method: checkLayer This will check to see if a layer has already been
     * created to display features.
     */
    checkLayer : function() {
        if (this.layer)
            return;
        this.layer = new OpenLayers.Layer.Vector("Results", {
            displayInLayerSwitcher : false
        });
        this.layer.hidden = true;
        var selectStyle = OpenLayers.Util.applyDefaults(this.styles.select, OpenLayers.Feature.Vector.style["select"]);
        selectStyle.cursor = "";
        
        var temporaryStyle = OpenLayers.Util.applyDefaults(this.styles.temporary, OpenLayers.Feature.Vector.style["temporary"]);
        this.layer.styleMap = new OpenLayers.StyleMap({
            "select" : selectStyle,
            "temporary" : temporaryStyle
        });
        
        this.map.addLayer(this.layer);
    },
    /**
     * Method: deactivate Called when the control is deactivated
     */
    deactivate : function() {
        this.map.events.un({
            "moveend" : this.displayResults,
            scope : this
        });
        
        if (this.layer && this.layer.map != null) {
            this.map.removeLayer(this.layer);
            this.layer = null;
        }
        if (this.div) {
            oscar.jQuery(this.div).fadeOut();
        }
        oscar.jQuery(this.div).empty();
        
        oscar.Control.prototype.deactivate.apply(this);
    },
    /**
     * Method: queueDownload This adds a download service to the download queue
     * panel.
     */
    queueDownload : function(service) {
        
        this.discoverPanel.accordion("option", "active", 2);
        
        if (this.queuePanel.children().length > 0) {
            this.queuePanel.prepend(service.draw());
        } else {
            this.queuePanel.append(service.draw());
        }
        
    },
    /**
     * Constant: CLASS_NAME
     */
    CLASS_NAME : "oscar.Control.DataDiscovery"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.LoadingBar A loading bar panel.
 * 
 * Inherits from: - <oscar.Control>
 * 
 */

oscar.Control.LoadingBar = oscar.BaseClass(oscar.Control, {
    initialize : function(options) {
        oscar.Control.prototype.initialize.apply(this, [ options ]);
    },

    draw : function(px) {
        OpenLayers.Control.prototype.draw.apply(this, arguments);
        $$(this.div);
        this.hide();
        return this.div;
    },

    show : function() {
        $$(this.div).show();
    },

    hide : function() {
        $$(this.div).hide();
    },

    CLASS_NAME : "oscar.Control.LoadingBar"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.Multi This control is the base class for controls which
 * contain a sub-menu with controls.
 * 
 * Inherits from: - <oscar.Control>
 * 
 */

oscar.Control.MultiControl = oscar.BaseClass(oscar.Control, {
    parent : null,
    initialize : function(options) {
        OpenLayers.Control.prototype.initialize.apply(this, [ options ]);

    },
    draw : function() {
        OpenLayers.Control.prototype.draw.apply(this, arguments);

        $$(this.div).dblclick(function(evt) {
            evt.stopPropagation();
        })
        $$(this.div).mousedown(function(evt) {
            evt.stopPropagation();
        })
        $$(this.div).hide();

        $$(this.div).addClass("subMenu");

        var scope = this;
        setTimeout(function() {
            scope.updateSize();
        }, 0)

        var leftBookEnd = $$("<div></div>");
        leftBookEnd.addClass("subMenuBookEnd");
        $$(this.div).append(leftBookEnd);

        return this.div;
    },
    /**
     * Method: updateSize
     */
    updateSize : function() {
        var $parent = $$(this.parent.panel_div);
        var $toolbar = $parent.parent();

        var $this = $$(this.div);
        $this.fadeIn();
        var width = 0;
        $this.children().each(function() {
            width += $$(this).outerWidth() + 2;
        })
        $this.width(width);

        var left = $parent.width() / 2;
        left += $toolbar.position().left - width + $parent.position().left;
        $this.css("left", left);
        $this.css("top", $toolbar.position().top + $toolbar.height() + 2);

    },

    addMuliControl : function(elem) {
        var scope = this;
        elem.data("active", false);
        elem.toggleClass("toolInactive");
        elem.click(function() {

            var $this = $$(this);
            $this.parent().children().each(function() {
                var $child = $$(this);
                if ($child.hasClass("toolActive")) {
                    $child.removeClass("toolActive");
                    $child.addClass("toolInactive");
                }

            })
            $this.removeClass("toolInactive");
            $this.addClass("toolActive");
            var tool = $this.data("tool");
            var ctrl = scope.controls[tool];
            scope.toggleControl(ctrl)
        });
        $$(this.div).append(elem);
    },

    /**
     * Method: toggleControl
     * 
     * Toggles the active control
     */
    toggleControl : function(ctrl) {
        if (this.popup) {
            this.map.removePopup(this.popup);
        }
        for ( var key in this.controls) {
            var control = this.controls[key];
            control.deactivate();
        }
        ctrl.activate();
        this.activeControl = ctrl;
    },
    CLASS_NAME : "oscar.Control.MultiControl"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Control.MeasurementTools = oscar.BaseClass(oscar.Control.MultiControl, {
    autoActivate : true,

    /**
     * Property: defaultUnit {String} The default display unit of the
     * measurement. Set to metric.
     */
    defaultUnit : 'metric',

    /**
     * Property: currentUnit {String} The current display unit.
     */
    currentUnit : null,

    /**
     * Property: dlg
     */
    dlg : null,

    sketchSymbolizers : {
        "Point" : {
            pointRadius : 4,
            graphicName : "square",
            fillColor : "white",
            fillOpacity : 1,
            strokeWidth : 1,
            strokeOpacity : 1,
            strokeColor : "#333333"
        },
        "Line" : {
            strokeWidth : 3,
            strokeOpacity : 1,
            strokeColor : "#f00",
            strokeDashstyle : "solid"
        },
        "Polygon" : {
            strokeWidth : 2,
            strokeOpacity : 1,
            strokeColor : "#f00",
            fillColor : "white",
            fillOpacity : 0.3
        }
    },
    initialize : function(options) {
        oscar.Control.MultiControl.prototype.initialize.apply(this, [ options ]);

        // Setup the sketchSymbolisers
        if (options && options.sketchSymbolizers) {
            for ( var key in this.sketchSymbolizers) {
                this._setupSketchSymbolizers(this.sketchSymbolizers[key], options.sketchSymbolizers[key]);
            }
        }
        this.style = new OpenLayers.Style();
        this.style.addRules([ new OpenLayers.Rule({
            symbolizer : this.sketchSymbolizers
        }) ]);
        this.styleMap = new OpenLayers.StyleMap({
            "default" : this.style
        });

        this.controls = {
            line : new OpenLayers.Control.Measure(OpenLayers.Handler.Path, {
                persist : true,
                handlerOptions : {
                    layerOptions : {
                        styleMap : this.styleMap
                    }
                }
            }, {
                displaySystem : this.defaultUnit
            }),
            polygon : new OpenLayers.Control.Measure(OpenLayers.Handler.Polygon, {
                persist : true,
                handlerOptions : {
                    layerOptions : {
                        styleMap : this.styleMap
                    }
                }
            }, {
                displaySystem : this.defaultUnit
            })
        };
        this.controls.line.events.includeXY = true;
    },
    draw : function(options) {
        var scope = this;
        oscar.Control.MultiControl.prototype.draw.apply(this, [ options ]);

        var line = $$("<div></div>");
        line.addClass("subTool");
        line.addClass("line");
        line.data("tool", "line");
        this.addMuliControl(line);
        var polygon = $$("<div></div>");
        polygon.addClass("subTool");
        polygon.addClass("polygon");
        polygon.data("tool", "polygon");

        this.addMuliControl(polygon);

        this._createUnits();

        return this.div;
    },

    /**
     * Method: _createUnits
     * 
     * Description: This method creates the dropdown list for the available
     * units.
     */

    _createUnits : function() {
        var units = [ 'english', 'metric' ];
        var container = document.createElement("span");
        oscar.jQuery(container).css("float", "left");
        oscar.jQuery(container).css("background", "transparent");
        oscar.jQuery(container).css("display", "block");
        oscar.jQuery(container).css("padding-top", "7px");

        var select = document.createElement("select");
        select.id = "oscarMeasureUnits";
        oscar.jQuery(select).addClass("measurementUnits");
        var option = document.createElement("option");
        select.options[select.options.length]

        for (var unit = 0; unit < units.length; unit++) {
            var option = document.createElement("option");
            select.options[select.options.length]
            option.value = units[unit];
            option.text = oscar.i18n(units[unit]);
            if (units[unit] == this.defaultUnit)
                option.selected = true;
            select.options[select.options.length] = option;
        }
        var scope = this;

        $$(select).change(function(evt) {
            scope.changeUnits();
        });

        $$(container).append($$(select));
        $$(this.div).append($$(container));
    },
    /**
     * Method: changeUnits This method is called when there is a change in units
     * drop down menu and updates the current units
     * 
     * Parameters: evt - {<OpenLayers.Event>}
     */

    changeUnits : function(evt) {
        var select = $('oscarMeasureUnits');
        unit = select.options[select.selectedIndex].value;

        if (unit == -1)
            return;
        this.currentUnit = unit;

        for ( var key in this.controls) {
            this.controls[key].displaySystem = unit;
        }

    },

    activate : function() {
        oscar.Control.MultiControl.prototype.activate.apply(this);
        var control;
        for ( var key in this.controls) {
            control = this.controls[key];
            control.events.on({
                "measure" : this.handleMeasurements,
                "measurepartial" : this.handleMeasurements,
                scope : this
            });
            this.map.addControl(control);
        }
    },
    deactivate : function() {

        if (this.dlg) {
            this.dlg.remove();
        }
        var control;
        for ( var key in this.controls) {
            control = this.controls[key];

            control.events.un({
                "measure" : this.handleMeasurements,
                "measurepartial" : this.handleMeasurements,
                scope : this
            });
            control.deactivate();
            this.map.removeControl(control);
        }
        oscar.Control.MultiControl.prototype.deactivate.apply(this);

    },
    /**
     * Method: _setupSketchSymbolizers This method will setup the customise
     * sketch symbolisers.
     * 
     * Parameter: defaultSketchSymbolizersOpts - an object to represent the
     * default sketch symbolisers options that are defined in the Measure
     * classes. sketchSymbolizersOpts - an object that represent the new
     * customise sketch symbolisers, and they will replace the default sketch
     * symbolisers.
     */
    _setupSketchSymbolizers : function(defaultSketchSymbolisersOpts, sketchSymbolisersOpts) {
        if (defaultSketchSymbolisersOpts && sketchSymbolisersOpts) {
            OpenLayers.Util.extend(defaultSketchSymbolisersOpts, sketchSymbolisersOpts);
        }
    },
    /**
     * Method: handleMeasurements This method displays the measurement on
     * screen.
     * 
     * Parameters: event - {Object}
     */
    handleMeasurements : function(event) {
        var units = event.units;
        var order = event.order;
        var measure = event.measure;
        var out = "";
        if (measure.toFixed(2) == 0.00) {
            return;
        }

        if (order == 1) {
            out += measure.toFixed(2) + " " + units;
        } else {
            out += measure.toFixed(2) + " " + units + "<sup>2</" + "sup>";
        }
        var scope = this;
        if (!this.dlg) {
            this.dlg = $$("<div></div>").html(out);
            this.dlg.dialog({
                height : 75,
                width : 150,
                close : function(e, ui) {
                    scope.dlg = null;
                },
                position : {
                    at : "left"
                }
            });
        } else {
            this.dlg.html(out);
        }
    },
    CLASS_NAME : "oscar.Control.MeasurementTools"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Control.SelectionTools = oscar.BaseClass(oscar.Control.MultiControl, {
    autoActivate : true,
    processor : null,

    sketchSymbolizers : {
        "Point" : {
            pointRadius : 4,
            graphicName : "square",
            fillColor : "white",
            fillOpacity : 1,
            strokeWidth : 1,
            strokeOpacity : 1,
            strokeColor : "#333333"
        },
        "Line" : {
            strokeWidth : 3,
            strokeOpacity : 1,
            strokeColor : "#666666",
            strokeDashstyle : "dash"
        },
        "Polygon" : {
            strokeWidth : 2,
            strokeOpacity : 1,
            strokeColor : "#666666",
            fillColor : "white",
            fillOpacity : 0.3
        }
    },
    initialize : function(options) {
        oscar.Control.MultiControl.prototype.initialize.apply(this, [ options ]);

        if (options && options.sketchSymbolizers) {
            for ( var key in this.sketchSymbolizers) {
                this._setupSketchSymbolizers(this.sketchSymbolizers[key], options.sketchSymbolizers[key]);
            }
        }
        this.style = new OpenLayers.Style();
        this.style.addRules([ new OpenLayers.Rule({
            symbolizer : this.sketchSymbolizers
        }) ]);
        this.styleMap = new OpenLayers.StyleMap({
            "default" : this.style
        });

        this.controls = {
            point : new oscar.Control.Point(),
            area : new oscar.Control.Box()
        };
    },
    draw : function(options) {
        var scope = this;
        oscar.Control.MultiControl.prototype.draw.apply(this, [ options ]);

        var line = $$("<div></div>");
        line.addClass("subTool");
        line.addClass("point");
        line.data("tool", "point");
        this.addMuliControl(line);

        var polygon = $$("<div></div>");
        polygon.addClass("subTool");
        polygon.addClass("range");
        polygon.data("tool", "area");
        this.addMuliControl(polygon);

        return this.div;

    },

    activate : function() {
        oscar.Control.MultiControl.prototype.activate.apply(this);
        var control;
        for ( var key in this.controls) {
            control = this.controls[key];
            control.events.on({
                'done' : this.done,
                scope : this
            });
            this.map.addControl(control);
        }
    },
    deactivate : function() {
        if (this.popup) {
            this.map.removePopup(this.popup);
        }
        var control;
        for ( var key in this.controls) {
            control = this.controls[key];

            control.events.un({
                'done' : this.done,
                scope : this
            });
            control.deactivate();
            this.map.removeControl(control);
        }
        oscar.Control.MultiControl.prototype.deactivate.apply(this);

    },
    /**
     * Method: _setupSketchSymbolizers This method will setup the customise
     * sketch symbolisers.
     * 
     * Parameter: defaultSketchSymbolizersOpts - an object to represent the
     * default sketch symbolisers options that are defined in the Measure
     * classes. sketchSymbolizersOpts - an object that represent the new
     * customise sketch symbolisers, and they will replace the default sketch
     * symbolisers.
     */
    _setupSketchSymbolizers : function(defaultSketchSymbolisersOpts, sketchSymbolisersOpts) {
        if (defaultSketchSymbolisersOpts && sketchSymbolisersOpts) {
            OpenLayers.Util.extend(defaultSketchSymbolisersOpts, sketchSymbolisersOpts);
        }
    },
    /**
     * Method: done Called when one of the selection handlers have been
     * completed.
     * 
     * Parameters: geom - {OpenLayers.Geometry} The resulting geometry from the
     * selection.
     */
    done : function(geom) {
        if (this.ignoreDblClick)
            return;
        var idcFn = function() {
            ctx.ignoreDblClick = null;
        }
        var ctx = this;
        this.ignoreDblClick = window.setTimeout(idcFn, this.threshold);
        if (this.parent.processor)
            this.parent.processor.execute(geom, this.parent.theme);
        else if (this.callback) {
            this.callback(geom);
        }
    },
    setProcessor : function(p) {
        this.processor = p;
    },
    callback : function(geom) {
        if (console) {
            console.log(geom);
        }
    },
    CLASS_NAME : "oscar.Control.SelectionTools"
});

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.PanZoom
 * 
 * This Oscar PanZoom control extends the <OpenLayers.Control.PanZoom> and adds
 * zoom controls to your map.
 * 
 * Inherits from: - <OpenLayers.Control.PanZoom>
 * 
 */

oscar.Control.PanZoom = oscar.BaseClass(OpenLayers.Control.PanZoom, {
    /**
     * Constructor: oscar.Control.PanZoom
     * 
     * Parameters: options - {object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        OpenLayers.Control.PanZoom.prototype.initialize.apply(this, [ options ]);
    },
    /**
     * APIMethod: draw Draws the controls on screen.
     * 
     * Parameters: px - position of the PanZoom control.
     */
    draw : function(px) {
        // initialize our internal div
        OpenLayers.Control.prototype.draw.apply(this, arguments);
        px = this.position;

        // place the controls
        this.buttons = [];

        var sz = new OpenLayers.Size(22, 30);
        var centered = new OpenLayers.Pixel(px.x, px.y);

        this._addButton("zoomin", "sliderTop.png", centered, sz);
        this._addButton("zoomout", "sliderBottom.png", centered.add(0, sz.h), sz);
        return this.div;
    },

    /**
     * Method: _addButton
     * 
     * Parameters: id - {String} img - {String} xy - {<OpenLayers.Pixel>} sz - {<OpenLayers.Size>}
     * 
     * Returns: {DOMElement} A Div (an alphaImageDiv, to be precise) that
     * contains the image of the button, and has all the proper event handlers
     * set.
     */
    _addButton : function(id, img, xy, sz) {
        // var imgLocation = OpenLayers.Util.getImageLocation(img);
        var imgLocation = oscar.getImagePath() + "panzoom/" + img;
        var btn = OpenLayers.Util.createAlphaImageDiv(this.id + "_" + id, xy, sz, imgLocation, "absolute");
        btn.style.cursor = "pointer";
        // we want to add the outer div
        this.div.appendChild(btn);
        btn.action = id;
        btn.className = "olButton";

        // we want to remember/reference the outer div
        this.buttons.push(btn);
        return btn;
    },

    /**
     * Constant: CLASS_NAME - oscar.Control.PanZoom
     */
    CLASS_NAME : "oscar.Control.PanZoom"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * @requires OpenLayers/Control/PanZoomBar.js
 */

/**
 * Class: oscar.Control.PanZoomBar The Oscar PanZoomBar extends
 * <OpenLayers.Control.PanZoomBar> and adds a PanZoomBar to your map.
 * 
 * Inherits from: - <OpenLayers.Control.PanZoomBar>
 */

oscar.Control.PanZoomBar = oscar.BaseClass(OpenLayers.Control.PanZoomBar, {
    /**
     * APIProperty: zoomStopWidth {Integer}
     */
    zoomStopWidth : 22,

    /**
     * APIProperty: zoomStopHeight {Integer}
     */
    zoomStopHeight : 11,

    /**
     * Property: slider
     */
    slider : null,

    /**
     * Property: sliderEvents {<OpenLayers.Events>}
     */
    sliderEvents : null,

    /**
     * Property: zoomBarDiv {DOMElement}
     */
    zoomBarDiv : null,

    /**
     * Property: divEvents {<OpenLayers.Events>}
     */
    divEvents : null,

    /**
     * APIProperty: zoomWorldIcon {Boolean}
     */
    zoomWorldIcon : false,

    /**
     * APIMethod: draw Draws the pan zoom bar to the screen.
     * 
     * Parameters: px - position of the pan zoom bar.
     */
    draw : function(px) {
        // initialize our internal div
        OpenLayers.Control.prototype.draw.apply(this, arguments);
        px = this.position.clone();

        // place the controls
        this.buttons = [];

        var sz = new OpenLayers.Size(21, 21);
        var loc = new OpenLayers.Pixel(13, 10);

        if (this.zoomWorldIcon) {
            this._addButton("zoomworld", "overviewIcon.png", loc, new OpenLayers.Size(28, 30));
            loc = loc.add(2, 32);
        }
        this._addButton("zoomin", "sliderTop.png", loc, new OpenLayers.Size(22, 30));
        loc = loc.add(0, 30);
        loc = this._addZoomBar(loc);
        this._addButton("zoomout", "sliderBottom.png", loc, new OpenLayers.Size(22, 30));

        return this.div;
    },

    /**
     * Method: _addButton
     * 
     * Parameters: id - {String} img - {String} xy - {<OpenLayers.Pixel>} sz - {<OpenLayers.Size>}
     * 
     * Returns: {DOMElement} A Div (an alphaImageDiv, to be precise) that
     * contains the image of the button, and has all the proper event handlers
     * set.
     */
    _addButton : function(id, img, xy, sz) {
        // var imgLocation = OpenLayers.Util.getImageLocation(img);
        var imgLocation = oscar.getImagePath() + "panzoom/" + img;
        var btn = OpenLayers.Util.createAlphaImageDiv(this.id + "_" + id, xy, sz, imgLocation, "absolute");
        btn.style.cursor = "pointer";
        // we want to add the outer div
        this.div.appendChild(btn);
        btn.action = id;
        btn.className = "olButton";

        // we want to remember/reference the outer div
        this.buttons.push(btn);
        return btn;
    },

    /**
     * Method: _addZoomBar
     * 
     * Parameters: centered - {<OpenLayers.Pixel>} where zoom bar drawing is to
     * start.
     * 
     * Returns: centered - {<OpenLayers.Pixel>}
     */
    _addZoomBar : function(centered) {
        var imgLocation = oscar.getImagePath() + "panzoom/sliderHandle.png"
        var id = "OpenLayers_Control_PanZoomBar_Slider" + this.map.id;
        var zoomsToEnd = this.map.getNumZoomLevels() - 1 - this.map.getZoom();
        var slider = OpenLayers.Util.createAlphaImageDiv(id, centered.add(-1, zoomsToEnd * this.zoomStopHeight), {
            w : 22,
            h : 9
        }, imgLocation, "absolute");
        slider.style.cursor = "move";
        this.slider = slider;

        this.sliderEvents = new OpenLayers.Events(this, slider, null, true, {
            includeXY : true
        });
        this.sliderEvents.on({
            "touchstart" : this.zoomBarDown,
            "touchmove" : this.zoomBarDrag,
            "touchend" : this.zoomBarUp,
            "mousedown" : this.zoomBarDown,
            "mousemove" : this.zoomBarDrag,
            "mouseup" : this.zoomBarUp
        });

        var sz = {
            w : this.zoomStopWidth,
            h : this.zoomStopHeight * this.map.getNumZoomLevels()
        };
        var imgLocation = oscar.getImagePath() + "panzoom/sliderCenter.png";
        var div = null;

        if (OpenLayers.Util.alphaHack()) {
            var id = this.id + "_" + this.map.id;
            div = OpenLayers.Util.createAlphaImageDiv(id, centered, {
                w : sz.w,
                h : this.zoomStopHeight
            }, imgLocation, "absolute", null, "crop");
            div.style.height = sz.h + "px";
        } else {
            div = OpenLayers.Util.createDiv('OpenLayers_Control_PanZoomBar_Zoombar' + this.map.id, centered, sz, imgLocation);
        }
        div.style.cursor = "pointer";
        div.className = "olButton";
        this.zoombarDiv = div;

        this.div.appendChild(div);

        this.startTop = parseInt(div.style.top);
        this.div.appendChild(slider);

        this.map.events.register("zoomend", this, this.moveZoomBar);

        centered = centered.add(0, this.zoomStopHeight * this.map.getNumZoomLevels());
        return centered;
    },
    /**
     * Constant: CLASS_NAME - oscar.Control.PanZoomBar
     */
    CLASS_NAME : "oscar.Control.PanZoomBar"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.ToolBar
 * 
 * The Oscar ToolBar control is a container of other mapping tools which are
 * also controls.
 * 
 * Inherits from: - <OpenLayers.Control.Panel>
 */
oscar.Control.Toolbar = oscar.BaseClass(OpenLayers.Control.Panel, {
    displayMeasurements : true,
    usePreviousView : true,
    EVENT_TYPES : [ "selection", "extraction", "measurement", "activate", "deactivate" ],
    initialize : function(options) {
        OpenLayers.Control.Panel.prototype.initialize.apply(this, [ options ]);
        this.leftCap = $$("<div></div>");
        this.leftCap.addClass("toolBarBookendLeft");
        this.rightCap = $$("<div></div>");
        this.rightCap.addClass("toolBarBookendRight");
    },
    /**
     * Method: redraw
     */
    redraw : function() {
        for (var l = this.div.childNodes.length, i = l - 1; i >= 0; i--) {
            this.div.removeChild(this.div.childNodes[i]);
        }
        this.div.innerHTML = "";
        if (this.active) {
            if (!this.usePreviousView) {
                $$(this.div).append(this.leftCap);
            }
            for (var i = 0, len = this.controls.length; i < len; i++) {
                this.div.appendChild(this.controls[i].panel_div);
                var tooltipText = oscar.i18n("tooltip_" + this.controls[i].CLASS_NAME.toLowerCase());

                var $panelDiv = $$(this.controls[i].panel_div);
                $panelDiv.attr("title", tooltipText);
                $panelDiv.data("ctrl", this.controls[i]);
                $panelDiv.mouseover(function() {
                    var $this = $$(this);
                    var ctrl = $this.data("ctrl");

                    if (ctrl.active)
                        return;

                    var displayClass = ctrl.displayClass;
                    var active = displayClass + "ItemActive";
                    var inactive = displayClass + "ItemInactive";
                    $this.removeClass(inactive);
                    $this.addClass(active);
                });

                $panelDiv.mouseout(function() {
                    var $this = $$(this);
                    var ctrl = $this.data("ctrl");

                    if (ctrl.active)
                        return;

                    var displayClass = ctrl.displayClass;
                    var active = displayClass + "ItemActive";
                    var inactive = displayClass + "ItemInactive";

                    $this.addClass(inactive);
                    $this.removeClass(active);
                });

            }
            $$(this.div).append(this.rightCap);
        }
    },
    activateControl : function(control) {
        OpenLayers.Control.Panel.prototype.activateControl.apply(this, [ control ]);

        for ( var key in this.controls) {
            if (this.controls[key].active && this.controls[key] != control) {
                this.controls[key].deactivate();
            }
        }

    },
    applyTheme : function(theme) {
        this.removeExistingControls();
        var ctrls = [];
        if (this.usePreviousView) {
            ctrls.push(new oscar.Control.PreviousView());
        }
        if (this.displayMeasurements) {
            ctrls.push(new oscar.Control.Measure());
        }
        if (theme.hasSelectionService()) {
            var selection = new oscar.Control.Select({
                theme : theme
            });
            ctrls.push(selection);
            this.events.triggerEvent("selection", selection);
        }
        if (theme.hasExtractionService()) {
            ctrls.push(new oscar.Control.DataExtractor(theme));
        }

        this.addControls(ctrls);
    },
    removeExistingControls : function() {
        while (this.controls.length != 0) {
            var control = this.controls.pop();
            control.deactivate();
            this.map.removeControl(control);
        }

    },
    CLASS_NAME : "oscar.Control.ToolBar"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.Box A box control for handling range selections.
 * 
 * Inherits from: - <oscar.Control>
 * 
 */

oscar.Control.Box = oscar.BaseClass(oscar.Control, {
    /**
     * Constant: EVENT_TYPES
     * 
     * {Array(String)} Supported application event types.
     * 
     * Supported event types: done - triggered when the *selection* event has
     * been completed. Will pass a geometry object to any listeners.
     * 
     */
    EVENT_TYPES : [ "done" ],
    
    /**
     * Property: type - OpenLayers.Control.TYPE_TOGGLE
     */
    type : OpenLayers.Control.TYPE_TOGGLE,
    
    /**
     * Property: allowPanning - Allows for panning.
     */
    allowPanning : true,
    
    /**
     * Property: keyMask - When panning is disabled, what key should the control
     * work on.
     */
    keyMask : OpenLayers.Handler.MOD_CTRL,
    
    /**
     * Constructor: oscar.Control.Box Creates a box control for handling range
     * selection.
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        OpenLayers.Util.extend(this, options);
        this.EVENT_TYPES = oscar.Control.Box.prototype.EVENT_TYPES.concat(OpenLayers.Control.prototype.EVENT_TYPES);
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
        this.handlers = {};
    },
    
    /**
     * APIMethod: draw
     * 
     * This method activates the handler and draws the selection area on screen.
     */
    draw : function() {
        var rngSelectOptions = {
            irregular : true
        };
        
        if (this.allowPanning) {
            var callbacks = {
                "move" : this.panMap,
                "done" : this.panMapDone
            };
            
            this.handlers.dragPan = new OpenLayers.Handler.Drag(this, callbacks, {
                keyMask : OpenLayers.Handler.MOD_CTRL
            });
        } else {
            rngSelectOptions.keyMask = this.keyMask;
        }
        
        this.handlers.rangeSelect = new OpenLayers.Handler.RegularPolygon(this, {
            done : this.done,
            down : function(e) {
                OpenLayers.Element.addClass(this.map.viewPortDiv, "olDrawBox");
            },
            up : function(e) {
                OpenLayers.Element.removeClass(this.map.viewPortDiv, "olDrawBox");
            }
        }, rngSelectOptions);
        
    },
    
    panMap : function(xy) {
        this.panned = true;
        this.map.pan(this.handlers.dragPan.last.x - xy.x, this.handlers.dragPan.last.y - xy.y, {
            dragging : this.handlers.dragPan.dragging,
            animate : false
        });
        
    },
    panMapDone : function(xy) {
        if (this.panned) {
            this.panMap(xy);
            this.panned = false;
        }
    },
    
    activate : function() {
        this.handlers.rangeSelect.activate();
        if (this.allowPanning) {
            this.handlers.dragPan.activate();
        }
        return oscar.Control.prototype.activate.apply(this, arguments);
        
    },
    deactivate : function() {
        for ( var handler in this.handlers) {
            this.handlers[handler].deactivate();
        }
        
        return oscar.Control.prototype.deactivate.apply(this.arguments);
    },
    
    /**
     * Method: done
     * 
     * Parameters: geom - {OpenLayers.Geometry} The geometry object of the
     * selection area.
     */
    done : function(geom) {
        this.events.triggerEvent("done", geom);
    },
    /**
     * Constant: CLASS_NAME - oscar.Control.Box
     */
    CLASS_NAME : "oscar.Control.Box"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.RangeSelect
 * 
 * The Oscar Range selection control adds range selection ability to user's map.
 * 
 * Inherits from: - <oscar.Control.Box>
 * 
 */
oscar.Control.RangeSelect = oscar.BaseClass(oscar.Control.Box, {
    /**
     * Property: container
     */
    container : null,

    /**
     * Property: displayClass
     */
    displayClass : "rangeSelect",

    /**
     * Property: div
     */
    div : null,

    /**
     * Constructor: oscar.Control.RangeSelect
     * 
     * Parameters: div - the container of the control.
     */
    initialize : function(div) {
        oscar.Control.Box.prototype.initialize.apply(this, []);
        this.container = div;
        this.div = document.createElement("span");
        this.div.ref = "area";
        oscar.jQuery(this.div).addClass('subTool');
        oscar.jQuery(this.div).addClass('range');
        oscar.jQuery(this.div).addClass('toolInactive');
    },

    /**
     * Constant: CLASS_NAME - oscar.Control.RangeSelect
     */
    CLASS_NAME : "oscar.Control.RangeSelect"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.Point
 * 
 * The Oscar Point creates a new control on your map for doing points selection.
 * 
 * Inherits from: - <oscar.Control>
 * 
 */

oscar.Control.Point = oscar.BaseClass(oscar.Control, {
    /**
     * Constant: EVENT_TYPES
     * 
     * done - {Array(String)} triggered when the selection event has been
     * completed. Will pass a geometry object to any listeners.
     * 
     */
    EVENT_TYPES : [ "done" ],
    type : OpenLayers.Control.TYPE_TOOL,
    events : null,

    /**
     * Constructor: oscar.Control.Point
     * 
     * Parameters:
     * 
     * options - {Object} An optional object whose properties will be set on
     * this instance.
     * 
     */
    initialize : function(options) {
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
    },
    /**
     * APIMethod: draw
     * 
     * The *draw* method activates the handler used to draw the selection point
     * on screen.
     */
    draw : function() {
        this.handler = new OpenLayers.Handler.Point(this, {
            done : this.done
        });
    },

    /**
     * Method: done
     * 
     * Parameters: geom - {OpenLayers.Geometry} The geometry object of the
     * selection point.
     */
    done : function(geom) {
        this.events.triggerEvent("done", geom);
    },
    /**
     * Constant: CLASS_NAME - oscar.Control.Point
     */
    CLASS_NAME : "oscar.Control.Point"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.Select
 * 
 * This control will listen for a point click or a bounding box drawn on the map
 * and send the result to a specified callback function.
 * 
 * Two modes will be available, Point (result of a click action) and Range
 * (result of a bounding box action).
 * 
 * Inherits from: - <oscar.Control.Multi>
 * 
 */

oscar.Control.Select = oscar.BaseClass(oscar.Control, {
    /**
     * Property: type {Integer} OpenLayers.Control.TYPE_TOGGLE
     * 
     * When added to a <Control.Panel>, 'type' is used by the panel to determine
     * how to handle our events.
     */
    type : OpenLayers.Control.TYPE_TOGGLE,

    /**
     * Constructor: oscar.Control.Measure
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        OpenLayers.Control.prototype.initialize.apply(this, [ options ]);

    },
    /**
     * APIMethod: activate Called with the control is activated.
     */
    activate : function() {
        OpenLayers.Control.prototype.activate.apply(this, arguments);
        this.tools = new oscar.Control.SelectionTools({
            parent : this
        });
        this.map.addControl(this.tools);
    },
    /**
     * APIMethod: deactivate Called when the control is deactivated. It will
     * also deactivate and remove any sub controls.
     */
    deactivate : function() {
        if (this.tools) {
            this.tools.deactivate();

            this.map.removeControl(this.tools);
        }
        if (this.processor) {
            this.processor.events.triggerEvent("clean");
        }
        OpenLayers.Control.prototype.deactivate.apply(this, arguments);
    },
    /**
     * Constant: CLASS_NAME - oscar.Control.Select
     */
    CLASS_NAME : "oscar.Control.Select"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.CatalogueSearchForm
 * 
 * Inherits from: - <oscar.Control>
 * 
 */

oscar.Control.CatalogueSearchForm = oscar.BaseClass(oscar.Control, {
    EVENT_TYPES : [ "search" ],
    events : null,
    initialize : function(options) {
        this.EVENT_TYPES = this.EVENT_TYPES.concat(oscar.Control.prototype.EVENT_TYPES);
        oscar.Control.prototype.initialize.apply(this, [ options ]);
        this.catalogueService = this.catalogueServices[0];
    },
    performSearch : function(queryString) {
        this.searchHandler.search(new oscar.QueryType(oscar.QueryType.prototype.TEXT, queryString));
    },
    setMap : function(map) {
        oscar.Control.prototype.setMap.apply(this, [ map ]);
    },
    draw : function(options) {
        oscar.Control.prototype.draw.apply(this, arguments);
        this.form = $$("<form onSubmit='return false;'></form>");
        $$(this.div).append(this.form);
    },
    addHelp : function(context, container) {
        whatsThis = new oscar.Gui.Info({
            context : context
        });
        whatsThis.appendTo(container);
        whatsThis.events.on({
            'whatsthis' : this.showHelp,
            scope : this
        });
    },
    helpLocation : function() {
        var str = oscar._getScriptLocation() + this.help;
        return str;
    },
    showHelp : function(info) {
        info.show(this.helpLocation());
        
    },
    CLASS_NAME : 'oscar.Control.CatalogueSearchForm'
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 * 
 * Copyright 2014 CARIS <http://www.caris.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Control.SimpleCatalogueSearch = oscar.BaseClass(oscar.Control.CatalogueSearchForm, {
    BANG : /!e/,
    form : null,
    basicSearch : true,
    handler : null,
    help : "help/simple-search-{0}.html",
    spatialSearch : null,
    initialize : function(options) {
        oscar.Control.CatalogueSearchForm.prototype.initialize.apply(this, arguments);
        this.spatialSearch = new oscar.Control.Box({
            allowPanning : false,
            keyMask : OpenLayers.Handler.MOD_CTRL
        });
        this.spatialSearch.events.on({
            "done" : this.buildSpatial,
            scope : this
        });
        
        this.events.on({
            "search" : this.preprocessQuery,
            scope : this
        });
    },
    helpLocation : function() {
        return oscar._getScriptLocation() + this.help;
    },
    preprocessQuery : function(query) {
        // check for advanced search flag.
        if (this.bangLookup(query) || query.trim().length === 0) {
            return;
        }
        
        // build the query filter string
        // add wildcards to the start and end of the query
        if (!query.startsWith("*")) {
            query = "*" + query;
        }
        if (!query.endsWith("*")) {
            query += "*";
        }
        
        var queryArray = [];
        for (var i = 0; i < this.catalogueService.defaultSearchFields.length; i++) {
            var queryString = "";
            queryString += this.catalogueService.defaultSearchFields[i] + " LIKE ";
            queryString += "'" + query + "'";
            queryArray.push(queryString);
        }
        this.searchHandler.events.triggerEvent("displayfilter", queryArray.join(" OR "));
        this.performSearch(queryArray.join(" OR "));
    },
    bangLookup : function(query) {
        var hasMatch = function(text, pattern) {
            if (pattern instanceof RegExp) {
                return pattern.exec(text);
            } else {
                return pattern(text);
            }
        };
        var hasBang = false;
        while ((matches = hasMatch(query, this.BANG))) {
            if (!hasBang) {
                hasBang = true;
            }
            var match = matches[0];
            query = query.replace(match, "").trim();
        }
        if (hasBang) {
            this.searchHandler.events.triggerEvent("advancedSearch", query.trim());
            return true;
        }
        return false;
    },
    draw : function() {
        oscar.Control.CatalogueSearchForm.prototype.draw.apply(this, arguments);
        
        this.input = $$("<input type='search' id='q' name='q' size='50' value=''>");
        this.input.attr("placeholder", oscar.i18n("simple-search-placeholder"));
        this.input.addClass("ui-widget ui-corner-all");
        this.input.css({
            'text-align' : 'left',
            'outline' : 'none',
            'cursor' : 'text',
            'border' : '1px solid #2a3583',
            'line-height' : '1.9em',
        });
        if (this.defaultText) {
            this.input.val(this.defaultText);
        }
        this.map.addControl(this.spatialSearch);
        this.spatialSearch.activate();
        var inputContainer = $$("<div></div>");
        inputContainer.addClass("oscar-inline-container");
        inputContainer.append(this.input);
        
        this.form.append(inputContainer);
        this.button = $$("<button></button").html(oscar.i18n("Search"));
        
        var search_btn_container = $$("<div></div>");
        search_btn_container.addClass("oscar-inline-container");
        search_btn_container.append(this.button);
        this.form.append(search_btn_container);
        var help_container = $$("<div></div>");
        help_container.addClass("oscar-inline-container");
        
        this.addHelp(oscar.i18n("Help"), help_container);
        this.form.append(help_container);
        
        var scope = this;
        this.button.button({
            icons : {
                primary : "ui-icon-oux-search"
            },
            text : true
        }).click($$.proxy(function() {
            this.events.triggerEvent("search", this.input.val());
        }, this));
        var args = OpenLayers.Util.getParameters();
        if (args.q) {
            this.input.val(args.q);
            setTimeout($$.proxy(function() {
                this.button.click();
            }, this), 500);
        }
        
        return this.div;
    },
    toggleFeatureSelection : function(forceDeactivate) {
        var selFeatures = this.map.getControlsByClass("OpenLayers.Control.SelectFeature");
        for (var i = 0; i < selFeatures.length; i++) {
            if (forceDeactivate) {
                selFeatures[i].deactivate();
            } else {
                selFeatures[i].activate();
            }
        }
    },
    buildSpatial : function(geom) {
        var query = {
            spatial : geom.getBounds(),
            projection : this.map.getProjection(),
            type : oscar.Handler.CSW.prototype.SPATIAL
        }
        var criteria = new oscar.QueryType(oscar.QueryType.prototype.SPATIAL, query);
        this.searchHandler.search(criteria);
    },
    CLASS_NAME : "oscar.Control.SimpleCatalogueSearch"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 * 
 * Copyright 2014 CARIS <http://www.caris.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Control.AdvancedCatalogueSearch = new oscar.BaseClass(oscar.Control.CatalogueSearchForm, {
    EVENT_TYPES : [ 'close' ],
    autoSearch : false,
    errorId : null,
    help : "help/advanced-search-{0}.html",
    initialize : function(options) {
        oscar.Control.prototype.initialize.apply(this, [ options ]);
        this.searchHandler.events.on({
            "displayfilter" : this.displayFilter,
            scope : this
        });
        
        this.parser = new oscar.Util.Parser(CQL_Parser);
        this.parser.events.on({
            "parsesuccess" : this.parseSuccess,
            scope : this
        });
        
        this._getSupportedFieldList();
        
        this.events.on({
            "close" : function() {
                $$("#ac-matches").remove();
            },
            scope : this
        
        });
    },
    _getSupportedFieldList : function() {
        this.supportedFieldList = [];
        var service = this.catalogueServices[0];
        
        var op = oscar.Util.Metadata.getOperation(service.capabilities, "GetRecords");
        if (!op.constraints)
            return;
        var getConstraint = function(name, constraints) {
            for (var i = 0; i < constraints.length; i++) {
                if (name.toLowerCase() == constraints[i].name.toLowerCase()) {
                    return constraints[i].values;
                }
            }
            
        }
        var supportedQueryables = getConstraint("SupportedISOQueryables", op.constraints);
        var additionalQueryables = getConstraint("AdditionalQueryables", op.constraints);
        this.supportedFieldList = this.supportedFieldList.concat(supportedQueryables);
        this.supportedFieldList = this.supportedFieldList.concat(additionalQueryables);
    },
    draw : function(options) {
        oscar.Control.CatalogueSearchForm.prototype.draw.apply(this, arguments);
        var layout = $$("<div></div>");
        layout.addClass(this.displayClass);
        this.form.append(layout);
        this.getAdvancedUI(layout);
    },
    parseSuccess : function(query) {
        $$("#ac-matches").remove();
        $$("#" + this.errorId).empty();
        
        if (this.autoSearch) {
            this.searchHandler.search(query);
        }
    },
    getAdvancedUI : function(container) {
        var closediv = $$("<div></div>");
        closediv.css({
            'width' : '25px',
            'height' : '25px',
            'position' : 'absolute',
            'right' : '0px',
            'top' : '0px'
        });
        container.append(closediv);
        
        var closeBtn = $$("<button></button>").html(oscar.i18n("Close"));
        closediv.append(closeBtn);
        closeBtn.button({
            icons : {
                primary : 'ui-icon-oux-close'
            },
            text : false
        }).click($$.proxy(function() {
            this.events.triggerEvent("close");
        }, this));
        var containerDiv = $$("<div></div>");
        containerDiv.css({
            "width" : "95%"
        });
        container.append(containerDiv);
        this.textarea = $$("<textarea></textarea>");
        this.textarea.addClass('advSearchTextArea');
        var scope = this;
        
        this.textarea.autocomplete({
            minLength : 0,
            source : function(req, resp) {
                resp(scope.getSuggestions(req));
            },
            // eliminate the focus change from the menu
            focus : function(e, ui) {
                return false;
            },
            select : $$.proxy(this._injectText, this)
        }).focus(function() {
            $$(this).autocomplete("search");
        }).bind('keypress', $$.proxy(function(event) {
            if (event.keyCode == 13) {
                try {
                    this.parser.parse(this.textarea.val());
                    this.performSearch(this.textarea.val());
                } catch (err) {
                }
                return false;
            }
        }, this));
        
        containerDiv.append(this.textarea);
        var btn = $$("<button></button>").html(oscar.i18n("Search"));
        btn.button({
            icons : {
                primary : "ui-icon-oux-search"
            },
            text : true
        }).click($$.proxy(this.search, this));
        
        var ctrls = $$("<div></div>");
        ctrls.addClass("advSearchControls");
        containerDiv.append(ctrls);
        var parserErrorDiv = $$("<span></span>");
        this.errorId = OpenLayers.Util.createUniqueID(this.CLASS_NAME + ".errorDiv.");
        parserErrorDiv.attr("id", this.errorId);
        parserErrorDiv.addClass("errorDiv");
        
        ctrls.append(parserErrorDiv);
        var btns = $$("<div></div>");
        btns.addClass("advSearchButtons");
        var search_btn_container = $$("<div></div>");
        search_btn_container.addClass("oscar-inline-container");
        var help_container = $$("<div></div>");
        help_container.addClass("oscar-inline-container");
        
        search_btn_container.append(btn);
        this.addHelp(oscar.i18n("Syntax Help"), help_container);
        btns.append(search_btn_container);
        btns.append(help_container);
        
        ctrls.append(btns);
    },
    search : function() {
        try {
            this.parser.parse(this.textarea.val());
            this.performSearch(this.textarea.val());
        } catch (parseError) {
            this.displayError(parseError);
        }
    },
    displayError : function(err) {
        var errString = "Suggestion: ";
        if (err.hash && err.hash.loc) {
            errString += " @";
            var errorLn = err.hash.loc.last_line;
            var errorCol = err.hash.loc.last_column;
            errString += errorLn + ":" + errorCol + ".";
        }
        
        if (err.hash && err.hash.expected) {
            var symbols = [];
            for (var i = 0; i < err.hash.expected.length; i++) {
                var symbol = null;
                if ((symbol = oscar.Util.getGrammarSymbol(err.hash.expected[i]))) {
                    symbols.push(symbol);
                }
            }
            if (symbols) {
                errString += " Expected " + symbols.join(" or ") + "."
            }
        }
        
        $$("#" + this.errorId).html(errString);
        
    },
    _injectText : function(e, ui) {
        // the query string
        var query = this.textarea.val();
        // injection value
        var injection = ui.item.value;
        
        // get the tokens
        var tokens = query.split(" ");
        var lastToken = tokens.pop();
        if (lastToken.indexOf("(") === 0) {
            lastToken = lastToken.substring(1, lastToken.length);
            tokens.push("(");
        }
        if (injection.toLowerCase().indexOf(lastToken.toLowerCase()) === -1) {
            tokens.push(lastToken);
        }
        tokens.push(injection);
        var tokenString = tokens.join(" ");
        this.textarea.val(tokenString);
        
        if (injection === "''") {
            setTimeout($$.proxy(function() {
                var query = this.textarea.val();
                var word_pos = query.lastIndexOf("'");
                var textarea = this.textarea[0];
                if (textarea.setSelectionRange) {
                    textarea.setSelectionRange(word_pos, word_pos);
                } else if (this.textarea[0].createTextRange) {
                    // because of IE
                }
            }, this), 100);
        }
        
        try {
            this.parser.parse(this.textarea.val());
        } catch (parseError) {
            this.displayError(parseError);
        }
        return false;
    },
    setQuery : function(query) {
        this.textarea.val(query);
        if (query.length > 0) {
            try {
                this.parser.parse(this.textarea.val());
                this.performSearch(this.textarea.val());
                
            } catch (parseError) {
                this.displayError(parseError);
                setTimeout($$.proxy(function() {
                    this.textarea.focus();
                }, this), 500);
            }
            
        }
    },
    getSuggestions : function(req) {
        var scope = this;
        
        // get supported field suggestions
        var getFieldSuggestions = function(partial) {
            var suggestions = [];
            for (var i = 0; i < scope.supportedFieldList.length; i++) {
                var field = scope.supportedFieldList[i];
                if (partial) {
                    if (field.indexOf(partial) === 0) {
                        suggestions.push(field);
                    }
                } else {
                    suggestions.push(field);
                }
            }
            return suggestions
        };
        
        // get parser suggestions
        var getParserSuggestions = function(expectations, partial) {
            if (!expectations)
                return;
            var suggestions = [];
            for (var i = 0; i < expectations.length; i++) {
                var expectation = expectations[i].replace(/'/g, '');
                switch (expectation) {
                    case "FIELD":
                        suggestions = suggestions.concat(getFieldSuggestions(partial));
                        break;
                    default:
                        var symbol = oscar.Util.getGrammarSymbol(expectation);
                        if (symbol) {
                            if (partial) {
                                if (symbol.toLowerCase().indexOf(partial.toLowerCase()) === 0) {
                                    suggestions.push(symbol);
                                }
                            } else {
                                suggestions.push(symbol);
                            }
                        }
                }
            }
            return suggestions;
        };
        
        var term = req.term;
        var matches = [];
        
        term = term.split(" ").pop();
        if (term.indexOf("(") === 0 && term.length > 1) {
            term = term.substring(1, term.length);
        }
        
        if (term.length === 0) {
            try {
                this.parser.parse(this.textarea.val());
            } catch (err) {
                this.displayError(err);
                matches = getParserSuggestions(err.hash.expected);
            }
            
        } else {
            matches = matches.concat(getFieldSuggestions(term));
            if (matches.length === 0) {
                try {
                    this.parser.parse(this.textarea.val());
                } catch (err) {
                    this.displayError(err);
                    matches = getParserSuggestions(err.hash.expected, term);
                }
            }
        }
        if (matches) {
            matches.sort();
            return matches;
        }
    },
    displayFilter : function(filter) {
        this.textarea.val(filter.toString());
    },
    CLASS_NAME : "oscar.Control.AdvancedSearchForm"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.Print
 * 
 * The Oscar Print control adds a *Print* link to user's map.
 * 
 * Inherits from: - <OpenLayers.Control.Permalink>
 */

oscar.Control.Print = oscar.BaseClass(OpenLayers.Control.Permalink, {
    /**
     * Property: callback
     * 
     * Function to go to when the link is clicked.
     */
    callback : null,

    /**
     * Constructor: oscar.Control.Print
     * 
     * Parameters: element - {DOMElement} base - {String} options - {Object} An
     * optional object whose properties will be set on this instance.
     */
    initialize : function(element, base, options) {
        var newArgs = [];
        newArgs.push(element);
        newArgs.push(base);
        newArgs.push(options);
        OpenLayers.Control.Permalink.prototype.initialize.apply(this, newArgs);
    },

    /**
     * APIMethod: destroy
     * 
     * Destroy the control.
     */
    destroy : function() {
        OpenLayers.Control.Permalink.prototype.destroy.apply(this, arguments);
    },

    /**
     * APIMethod: draw
     * 
     * Draw the control onto user's map.
     */
    draw : function() {
        OpenLayers.Control.prototype.draw.apply(this, arguments);

        if (!this.element) {
            var ctx = this;
            var oClick = function() {
                ctx.createPrintParams();
            };
            this.div.className = this.displayClass;
            this.element = document.createElement("a");
            this.element.innerHTML = OpenLayers.i18n("Print");
            this.element.href = "#";
            this.element.onclick = oClick;
            this.div.appendChild(this.element);
        }
        return this.div;
    },

    /**
     * Method: createPrintParams
     */
    createPrintParams : function() {
        var activeTheme = this.map.currentThemeName;
        var bbox = this.map.getExtent().toBBOX();
        if (this.callback)
            this.callback(activeTheme, bbox);
    },

    /**
     * Method: updateLink
     * 
     * Overridden to remove functionality
     */
    updateLink : function() {

    },

    /**
     * Constant: CLASS_NAME - oscar.Control.Print
     */
    CLASS_NAME : "oscar.Control.Print"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.DataExtractor
 * 
 * This control loads the capabilities documents for the extraction services and
 * will launch the DataDiscovery control when activated.
 * 
 * 
 */

oscar.Control.DataExtractor = oscar.BaseClass(oscar.Control, {
    /**
     * Property: type The type of the control
     */
    type : OpenLayers.Control.TYPE_TOGGLE,
    /**
     * Property: database {oscar.Util.Database} - contains the results of the
     * capabilities documents for quick searching.
     */
    database : null,
    /**
     * Constructor: new oscar.Control.DataExtractor
     */
    initialize : function(theme) {
        OpenLayers.Control.prototype.initialize.apply(this);
        this.theme = theme;
        this.database = new oscar.Util.Database();
        this.database.addTable("sources", [ "id", "title", "bbox", "abstract", "fk_capabilities", "dataType" ]);
        this.database.addTable("capabilities", [ "capabilities" ]);
        this.extractionServices = this.theme.getExtractionService();
        for ( var s in this.extractionServices) {
            var service = this.extractionServices[s];
            var proc = this["load_" + service.serviceType.toLowerCase()];
            if (proc) {
                proc.apply(this, [ service ]);
            } else {

            }
        }
    },
    /**
     * Method: activate: Called when the control is activated
     */
    activate : function() {
        oscar.Control.prototype.activate.apply(this);
        if (this.ctrl) {
            this.ctrl.activate();
        } else {
            this.ctrl = new oscar.Control.DataDiscovery(this.database)
            this.map.addControl(this.ctrl);
        }
    },
    /**
     * Method: deactivate Called when the control is deactivated.
     */
    deactivate : function() {
        if (this.ctrl) {
            this.ctrl.deactivate();
            this.ctrl = null;
        }
        oscar.Control.prototype.deactivate.apply(this);

    },
    /**
     * Method: load_wcs This method will load a wcs capabilities document and
     * store the results in the database.
     */
    load_wcs : function(service) {
        var params = {
            request : "GetCapabilities",
            service : "WCS",
            version : "1.1.0"
        }

        if (service.version) {
            params.version = service.version;
        }
        var success = function(response) {
            var reader = new oscar.Format.WCSCapabilities();
            var capabilities = reader.read(response.responseXML);
            var capIndex = this.database.addRecord("capabilities", {
                capabilities : capabilities
            });
            coverages = oscar.Util.Metadata.getCoverages(capabilities);
            for ( var c in coverages) {
                var coverage = coverages[c];
                if (service.identifiers.length > 0 && !oscar.Util.isFeatureInArray(coverage.identifier, service.identifiers))
                    continue;
                var bbox = this.getBoundingBox(coverage.wgs84BoundingBox, "EPSG:4326");
                var record = {
                    "id" : coverage.identifier,
                    "title" : coverage.title,
                    "abstract" : coverage["abstract"],
                    "bbox" : bbox,
                    "fk_capabilities" : capIndex,
                    "dataType" : "wcs"
                }
                var pk = this.database.addRecord("sources", record);
            }
        };
        this.wcsRequest = OpenLayers.Request.GET({
            url : service.url,
            params : params,
            async : true,
            success : success,
            failure : this.requestFail,
            scope : this
        });
    },
    /**
     * Method: load_wfs This method will load a wfs capabilities document and
     * store the results in the database.
     */
    load_wfs : function(service) {
        var success = function(response) {
            var reader = new OpenLayers.Format.WFSCapabilities();
            var capabilities = reader.read(response.responseXML);
            var capIndex = this.database.addRecord("capabilities", {
                capabilities : capabilities
            });
            features = oscar.Util.Metadata.getFeatureTypes(capabilities);
            for ( var f in features) {
                var feature = features[f];
                if (service.identifiers.length > 0 && !oscar.Util.isFeatureInArray(feature.name, service.identifiers))
                    continue;
                /*
                 * if the service version is 1.1.0 or higher then the
                 * wgs84boundingbox element is in EPSG:4326
                 */
                var srs = feature.srs || "EPSG:4326";
                if (service.version != "1.0.0") {
                    srs = "EPSG:4326";
                }
                var bounds = {
                    west : feature.bounds.left,
                    east : feature.bounds.right,
                    south : feature.bounds.bottom,
                    north : feature.bounds.top
                }
                var bbox = this.getBoundingBox(bounds, srs);
                var record = {
                    "id" : feature.name,
                    "title" : feature.title,
                    "abstract" : feature["abstract"],
                    "bbox" : bbox,
                    "fk_capabilities" : capIndex,
                    "dataType" : "wfs"
                }
                var pk = this.database.addRecord("sources", record);
            }
        };
        var params = {
            "service" : service.serviceType,
            "version" : service.version,
            "request" : "GetCapabilities"
        }
        this.wfsRequest = OpenLayers.Request.GET({
            url : service.url,
            params : params,
            async : true,
            success : success,
            failure : this.requestFail,
            scope : this
        });
    },
    /**
     * Method: getBoundingBox Takes a bbox object and returns a projected
     * bounding box if need be.
     */
    getBoundingBox : function(bbox, srs) {
        var objBbox = new OpenLayers.Bounds(bbox.west, bbox.south, bbox.east, bbox.north);
        if (this.map.projection.projCode != srs) {
            var proj = new OpenLayers.Projection(srs);
            objBbox.transform(proj, this.map.projection);
        }
        return objBbox;
    },
    /**
     * Method: requestFail Called when a capabilities request fails to load.
     */
    requestFail : function(resp) {
    },
    /**
     * Constant: CLASS_NAME
     */
    CLASS_NAME : "oscar.Control.DataExtractor"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.Measure The Oscar Measure control is used for measuring
 * distance or area.
 * 
 * Inherits from: - <oscar.Control.Multi>
 * 
 */

oscar.Control.Measure = oscar.BaseClass(oscar.Control, {
    /**
     * Property: type {Integer} OpenLayers.Control.TYPE_TOGGLE
     * 
     * When added to a <Control.Panel>, 'type' is used by the panel to determine
     * how to handle our events.
     */
    type : OpenLayers.Control.TYPE_TOGGLE,

    /**
     * Constructor: oscar.Control.Measure
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        OpenLayers.Control.prototype.initialize.apply(this, [ options ]);

    },
    /**
     * APIMethod: activate Called with the control is activated.
     */
    activate : function() {
        OpenLayers.Control.prototype.activate.apply(this, arguments);
        this.tools = new oscar.Control.MeasurementTools({
            parent : this
        });
        this.map.addControl(this.tools);
    },
    /**
     * APIMethod: deactivate Called when the control is deactivated. It will
     * also deactivate and remove any sub controls.
     */
    deactivate : function() {

        if (this.tools) {
            this.tools.deactivate();

            this.map.removeControl(this.tools);
        }
        OpenLayers.Control.prototype.deactivate.apply(this, arguments);
    },

    /**
     * Constant: CLASS_NAME - oscar.Control.Measure
     */
    CLASS_NAME : "oscar.Control.Measure"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * @requires OpenLayers.Control.Button
 */

/**
 * Class: oscar.Control.PreviousView
 * 
 * The Oscar PreviousView control keeps track of map movements and allows user
 * to go back to a previous view.
 * 
 * Inherits from: - <OpenLayers.Control.Button>
 * 
 */
oscar.Control.PreviousView = oscar.BaseClass(OpenLayers.Control.Button, {

    /**
     * Property: type {String}
     * 
     * Note that this control is not intended to be added directly to a control
     * panel. Instead, add the sub-controls previous and next. These
     * sub-controls are button type controls that activate and deactivate
     * themselves. If this parent control is added to a panel, it will act as a
     * toggle.
     */
    type : OpenLayers.Control.TYPE_BUTTON,

    /**
     * APIProperty: previous {<OpenLayers.Control>} A button type control whose
     * trigger method restores the previous state managed by this control.
     */
    previous : null,

    /**
     * APIProperty: limit
     * 
     * {Integer} Optional limit on the number of history items to retain. If
     * null, there is no limit. Default is 50.
     */
    limit : 50,

    /**
     * Property: activateOnDraw
     * 
     * {Boolean} Activate the control when it is first added to the map. Default
     * is true.
     */
    activateOnDraw : true,

    /**
     * Property: registry
     * 
     * {Object} An object with keys corresponding to event types. Values are
     * functions that return an object representing the current state.
     */
    registry : null,

    /**
     * Property: nextStack
     * 
     * {Array} Array of items in the history.
     */
    nextStack : null,

    /**
     * Property: previousStack
     * 
     * {Array} List of items in the history. First item represents the current
     * state.
     */
    previousStack : null,

    /**
     * Property: listeners
     * 
     * {Object} An object containing properties corresponding to event types.
     * This object is used to configure the control and is modified on
     * construction.
     */
    listeners : null,

    /**
     * Property: restoring
     * 
     * {Boolean} Currently restoring a history state. This is set to true before
     * calling restore and set to false after restore returns.
     */
    restoring : false,

    /**
     * Constructor: oscar.Control.PreviousView
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be used to extend the control.
     */
    initialize : function(options) {
        OpenLayers.Control.Button.prototype.initialize.apply(this, [ options ]);

        this.registry = OpenLayers.Util.extend({
            "moveend" : function() {
                return {
                    center : this.map.getCenter(),
                    resolution : this.map.getResolution()
                };
            }
        }, this.registry);

        this.clear();

    },
    /**
     * APIMethod: trigger Restore the previous state. If no items are in the
     * previous history stack, this has no effect.
     * 
     * Returns: state - {Object} Item representing state that was restored.
     * Undefined if no items are in the previous history stack.
     */
    trigger : function() {
        var current = this.previousStack.shift();
        var state = this.previousStack.shift();
        if (state != undefined) {
            this.previousStack.unshift(state);
            this.restoring = true;
            this.restore(state);
            this.restoring = false;
            this.onPreviousChange(this.previousStack[1], this.previousStack.length - 1);
        } else {
            this.previousStack.unshift(current);
        }
        return state;
    },

    /**
     * Method: onPreviousChange
     * 
     * Called when the previous history stack changes.
     * 
     * Parameters: state - {Object} An object representing the state to be
     * restored if previous is triggered again or null if no previous states
     * remain.
     * 
     * length - {Integer} The number of remaining previous states that can be
     * restored.
     */
    onPreviousChange : function(state, length) {
        if (state && !this.active) {
            this.activate();
        } else if (!state && this.active) {
            this.deactivate();
        }
    },

    /**
     * APIMethod: destroy
     * 
     * Destroy the control.
     */
    destroy : function() {
        OpenLayers.Control.prototype.destroy.apply(this);
    },

    /**
     * APIMethod: setMap
     * 
     * Set the map property for the control and <previous>
     * and &lt;next&gt; child controls.
     * 
     * Parameters:
     * map - {&lt;OpenLayers.Map&gt;}
     * 
     */
    setMap : function(map) {
        this.map = map;
        /* apply listeners here */
        var moveEndFunction = function() {
            if (!this.restoring) {
                var state = this.registry["moveend"].apply(this, arguments);
                this.previousStack.unshift(state);
                if (this.previousStack.length > 1) {
                    this.onPreviousChange(this.previousStack[1], this.previousStack.length - 1);
                }
                if (this.previousStack.length > (this.limit + 1)) {
                    this.previousStack.pop();
                }
            }
            return true;
        };

        this.map.events.register("moveend", this, moveEndFunction);
    },

    /**
     * APIMethod: draw
     * 
     * Called when the control is added to the map.
     */
    draw : function() {
        OpenLayers.Control.Button.prototype.draw.apply(this, arguments);
        if (this.activateOnDraw) {
            this.activate();
        }
    },
    /**
     * APIMethod: clear
     * 
     * Clear history.
     */
    clear : function() {
        this.previousStack = [];
    },

    /**
     * Method: restore
     * 
     * Update the state with the given object.
     * 
     * Parameters: state - {Object} An object representing the state to restore.
     */
    restore : function(state) {
        var zoom = this.map.getZoomForResolution(state.resolution);
        this.map.setCenter(state.center, zoom);
    },

    /**
     * APIMethod: activate Activate the control.
     * 
     * Returns: {Boolean} Control successfully activated.
     */
    activate : function() {
        return OpenLayers.Control.Button.prototype.activate.apply(this)
    },

    /**
     * Method: initStack
     * 
     * Called after the control is activated if the previous history stack is
     * empty.
     */
    initStack : function() {
        if (this.map.getCenter()) {
            this.listeners.moveend();
        }
    },

    /**
     * APIMethod: deactivate Deactivate the control. This unregisters any
     * listeners.
     */
    deactivate : function() {
        if (this.map)
            this.map.events.unregister("moveend", this);
    },

    /**
     * Constant: CLASS_NAME - oscar.Control.PreviousView
     */
    CLASS_NAME : "oscar.Control.PreviousView"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.OverviewMap
 * 
 * The Oscar OverviewMap control creates a small overview map which displays a
 * zoomed extent of the main map.
 * 
 * Inherits from: - <OpenLayers.Control.OverviewMap>
 */

oscar.Control.OverviewMap = oscar.BaseClass(OpenLayers.Control.OverviewMap, {
    /**
     * Constructor: oscar.Control.OverviewMap
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        OpenLayers.Control.OverviewMap.prototype.initialize.apply(this, arguments);
    },
    /**
     * APIMethod: draw
     * 
     * Draws the control on screen.
     */
    draw : function() {
        var layers = [];
        for ( var l in this.map.layers) {
            if (this.map.layers[l].renderer)
                continue;
            if (this.map.layers[l].clone) {
                layers.push(this.map.layers[l].clone());
            }
        }
        this.layers = layers;
        OpenLayers.Control.OverviewMap.prototype.draw.apply(this, arguments);
        $$(this.maximizeDiv).empty();
        $$(this.minimizeDiv).empty();
        return this.div;
    },

    /**
     * Constant: CLASS_NAME - oscar.Control.OverviewMap
     */
    CLASS_NAME : "oscar.Control.OverviewMap"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.SelectFeature
 * 
 * This control handles the selection of features from multiple layers. When
 * added to the map it registers for the addlayer and removelayer functions to
 * monitor when layers are added and adjusts an internal control to handle the
 * selection events.
 * 
 * 
 * 
 * Inherits from: - <oscar.Control>
 * 
 */

oscar.Control.SelectFeature = oscar.BaseClass(oscar.Control, {
    /**
     * APIProperty: autoActivate {Boolean} defaults to true
     */
    autoActivate : true,

    /**
     * Constructor: new oscar.Control.SelectFeature Creates a new instance of
     * this controls.
     */
    initialize : function(options) {
        OpenLayers.Control.prototype.initialize.apply(this, [ options ]);
        this.layers = [];
        this.handlers = {};

    },
    /**
     * Method: setMap Parameters: map (OpenLayers.Map)
     * 
     * Registers the map with the control. Registers the addlayer and
     * removelayer events.
     * 
     */
    setMap : function(map) {
        oscar.Control.prototype.setMap.apply(this, [ map ]);
        this.map.events.on({
            "addlayer" : this.addLayer,
            scope : this
        });
        this.map.events.on({
            "removelayer" : this.layerRemoved,
            scope : this
        });
    },
    /**
     * Method: activate Activates the control
     */
    activate : function() {
        this.handlers.click.activate();
        var scope = this;
        this.ctrl = new OpenLayers.Control.SelectFeature(this.layers);
        this.map.addControl(this.ctrl);
    },
    /**
     * Method: draw Resgisters any handlers. Since this is not a visible control
     * no div is returned.
     */
    draw : function() {
        var clickCallbacks = {
            'click' : this.clicked
        };
        this.handlers.click = new OpenLayers.Handler.Click(this, clickCallbacks);
    },

    /**
     * Method: clicked Called when the map is clicked
     * 
     * Parameters: e - MouseEvent
     * 
     */
    clicked : function(e) {
        var lonlat = this.map.getLonLatFromPixel(e.xy);

    },
    /**
     * Method: layerRemoved Called when a layer is removed from the map.
     * 
     * Parameters: e - Event
     */
    layerRemoved : function(e) {
        var layer = e.layer;
        var currLen = this.layers.length;
        for (var i = 0; i < this.layers.length; i++) {
            if (layer.id == this.layers[i].id) {
                this.layers.splice(i, 1);
                this.ctrl.setLayer(this.layers);
                break;
            }
        }

    },
    /**
     * Method: addLayer Called when a layer is added.
     * 
     * Parameters: e - Event
     */
    addLayer : function(e) {
        var layer = e.layer;
        if (layer.renderers) {
            layer.events.on({
                "loadend" : this.layerLoaded,
                scope : this
            });
        }
    },
    /**
     * Method: layerLoaded Called when a layer has finished loading.
     * 
     * Parameters: e - Event
     */
    layerLoaded : function(e) {
        var layer = e.object;
        if (layer.features.length == 0) {
            this.map.removeLayer(layer)
        } else {
            if (layer.hidden)
                return;
            var isNew = function(layer, existing) {
                for (var i = 0; i < existing.length; i++) {
                    if (layer.id == existing[i].id) {
                        return false;
                    }
                }
                return true;
            }
            if (isNew(layer, this.layers)) {
                try {
                    this.layers.push(layer);
                    this.ctrl.setLayer(this.layers);
                } catch (err) {
                }
                this.ctrl.activate();
            }
        }
    },
    /**
     * Method: clearControl Clears the internal control and removes it from the
     * map.
     */
    clearControl : function() {
        if (this.ctrl) {
            this.ctrl.deactivate();
            this.map.removeControl(this.ctrl);
            this.ctrl = null;
        }
    },
    /**
     * Constant: CLASS_NAME
     */
    CLASS_NAME : "oscar.Control.SelectFeature"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.ThemeManager
 * 
 * The Oscar ThemeManager control handles the drawing of themes onto a map
 * object.
 * 
 * Inherits from: - <OpenLayers.Control>
 * 
 */

oscar.Control.ThemeManager = oscar.BaseClass(OpenLayers.Control, {
    /**
     * Constant: EVENT_TYPES
     * 
     * themeAdded - Called when there is a *theme* added. afterDraw - Called
     * when the *doDraw* method is completed.
     */
    EVENT_TYPES : [ 'themeAdded', 'afterDraw' ],

    /**
     * Property: events
     * 
     * The *events* object holds registered events for this control.
     */
    events : null,

    /**
     * Property: defaultThemeIndex
     */
    defaultThemeIndex : 0,

    /**
     * Property: themes
     * 
     * {Array} List of themes <oscar.ox.Theme> of the map.
     */
    themes : null,

    /**
     * Property: themeLbls The theme labels
     */
    themeLbls : null,

    /**
     * Property: themeDlg The theme dialog.
     */
    themeDlg : null,

    /**
     * Property: autoDraw
     */
    autoDraw : false,

    /**
     * Property: layers
     */
    layers : null,

    /**
     * Property: IS_SPHERICALMERCATOR
     * 
     * {Boolean} Should the map act as a mercator-projected map.
     */
    IS_SPHERICALMERCATOR : null,

    /**
     * Property: guiManager
     */
    guiManager : null,

    /**
     * Property: controlManager
     */
    controlManager : null,

    /**
     * Property: activeTheme
     */
    activeTheme : null,

    /**
     * Property: argParserClass
     */
    argParserClass : oscar.Control.ArgParser,

    /**
     * Property: argParser
     */
    argParser : null,

    /**
     * Constructor: oscar.Control.ThemeManager
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
	OpenLayers.Control.prototype.initialize.apply(this, [ options ]);
	OpenLayers.Util.extend(this, options);
	this.themes = [];
	this.themeLbls = [];
	this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, true, {
	    includeXY : true
	});
	this.events.register('themeAdded', this, this.themeAdded);
    },

    /**
     * APIMethod: ignoreEvent
     * 
     * Ignores an event.
     * 
     * Parameters: evt - {<OpenLayers.Event>}
     */
    ignoreEvent : function(evt) {
	OpenLayers.Event.stop(evt);
    },

    /**
     * APIMethod: setMap
     * 
     * Set the map property for the control. This is done through an accessor so
     * that subclasses can override this and take special action once they have
     * their map variable set.
     * 
     * Parameters: map - {<OpenLayers.Map>}
     */
    setMap : function(map) {
	OpenLayers.Control.prototype.setMap.apply(this, [ map ]);
	if (this.guiManager) {
	    if (this.map.addToToolBox && this.guiManager.getToggleButton) {
		this.map.addToToolBox(this.guiManager.getToggleButton());
	    }
	}
	if (this.argParser == null) {
	    this.argParser = new this.argParserClass();
	    this.map.addControl(this.argParser);
	}
    },

    /**
     * APIMethod: setConfigManager
     * 
     * Sets the configManager property used to load an OXF (Oscar eXchange
     * Format) source. This will register event calls to *applyThemes* function
     * when the configManager triggers a *haveThemes* event.
     * 
     * Parameters: cm - {<oscar.Control.OXFConfigManager>}
     */
    setConfigManager : function(cm) {
	this.configManager = cm;
	this.configManager.events.on({
	    "haveThemes" : this.applyThemes,
	    scope : this
	});
    },

    /**
     * APIMethod: setGuiManager
     * 
     * Sets the GUI manager to use if there is to be a visual display of themes.
     * This method will trigger the *themeActivated* event when the GUI manager
     * triggers its themeActivated event.
     * 
     * Parameters: gui - GUI manager.
     */
    setGuiManager : function(gui) {
	this.guiManager = gui;
	this.guiManager.events.on({
	    "themeActivated" : this.themeActivated,
	    scope : this
	});
    },

    /**
     * Method: applyThemes
     * 
     * Gets triggered from the configuration manager object. This method loops
     * through the themes and calls addTheme.
     */
    applyThemes : function() {
	if (this.configManager) {
	    for (var i = 0; i < this.configManager.ox.themes.length; i++) {
		this.addTheme(this.configManager.ox.themes[i]);
	    }
	}
    },

    /**
     * Method: addTheme
     * 
     * This method takes a theme object and adds it to the theme manager.
     * 
     * Parameters: themeObj - {<oscar.ox.Theme>}
     */
    addTheme : function(themeObj) {
	this.themes.push(themeObj);
	this.events.triggerEvent("themeAdded", {
	    theme : themeObj
	});
	if (this.argParser) {
	    if (this.argParser.args && this.argParser.args.theme == themeObj.name) {
		this.autoDraw = true;
		this.drawTheme(themeObj);
		var center = new OpenLayers.LonLat(parseFloat(this.argParser.args.lon), parseFloat(this.argParser.args.lat));
		this.map.setCenter(center, parseInt(this.argParser.args.zoom));
		this.autoDraw = false;
		return;
	    }
	}
	if (this.autoDraw) {
	    /** draw the first theme * */
	    if (this.activeTheme == null) {
		this.drawTheme(this.themes[0]);
	    }
	}
    },

    /**
     * Method: themeAdded
     * 
     * This method gets triggered when a theme is added. If a GUI manager is
     * available it will call the addTheme function of the GUI manager passing
     * the theme object.
     * 
     * Parameters: themeObj - {<oscar.ox.Theme>}
     */
    themeAdded : function(themeObj) {
	if (this.guiManager) {
	    this.guiManager.setGuiContainer(this.div);
	    this.guiManager.addTheme(themeObj.theme);
	}
    },

    /**
     * Method: themeActivated
     * 
     * This method is triggered when a theme is activated and calls the
     * drawTheme method.
     * 
     * Parameters: e - {<OpenLayers.Event>}
     */
    themeActivated : function(e) {
	if (e.theme.isActive)
	    return;
	if (this.activeTheme) {
	    this.activeTheme.isActive = false;

	}
	e.theme.isActive = true;
	this.drawTheme(e.theme);
    },

    /**
     * Method: drawTheme
     * 
     * This method will take in a theme object and draw the layer of the theme
     * on to the map.
     * 
     * Parameters: theme - {<oscar.ox.Theme>}
     */
    drawTheme : function(theme, callback) {
	var mapLegendId = "mapLegend";
	$$("#"+mapLegendId).remove();
	 if(theme.legend) {
		var legend= $$("<img>");
		legend.attr('src',theme.legend);
		legend.attr("id",mapLegendId);
		legend.css({
			"position":"absolute",
			"zIndex":1000,
			"cursor":"move",
			"right":'5px',
			"bottom":'75px'
		});
		legend.appendTo(this.map.div);
		legend.draggable({containment: "parent"});
	}	

	// Look for an overview map and destroy it.
	try {
	    var overviewControl = this.map.getControlsByClass("oscar.Control.OverviewMap")[0] || this.map.getControlsByClass("OpenLayers.Control.OverviewMap")[0];
	    if (overviewControl) {
		overviewControl.destroy();
	    }
	} catch (err) {
	}

	// Lets get the current view port so we can show the new theme in the
	// same location.
	var viewPort = {
	    project : false
	}
	if (this.map.getExtent() != null) {
	    viewPort.previousView = this.map.getExtent();
	    viewPort.previousProjection = this.map.projection;
	    viewPort.project = true;
	}

	this.activeTheme = theme;

	$$(this.map.div).css({
	    "backgroundColor" : this.activeTheme.backgroundColor
	});

	this.layers = this.activeTheme.layers;
	var numZoomLevels = 16;
	try {
	    numZoomLevels = parseInt(this.activeTheme.parameters["numzoomlevels"]);
	} catch (err) {
	    numZoomLevels = 16;
	}

	var selectFeature = this.map.getControlsByClass("oscar.Control.SelectFeature")[0];
	if (selectFeature) {
	    selectFeature.ctrl.deactivate();
	}
	/** clear the map * */
	while (this.map.layers.length > 0) {
	    try {
		this.map.removeLayer(this.map.layers[this.map.layers.length - 1]);
	    } catch (err) {
	    }
	}

	/** check for spherical mercator srs * */
	var options = {};
	var maxExtent = null

	if ((maxExtent = this.activeTheme.getMaxCover()) != null) {
	    if (oscar.Util.isSphericalMercator(this.activeTheme.srs)) {
		options.maxResolution = 156543.0339;
	    } else {
		options.maxResolution = "auto";
	    }
	    options.maxExtent = maxExtent;
	} else {
	    if (oscar.Util.isSphericalMercator(this.activeTheme.srs)) {
		options = {
		    maxExtent : new OpenLayers.Bounds(-128 * 156543.0339, -128 * 156543.0339, 128 * 156543.0339, 128 * 156543.0339),
		    maxResolution : 156543.0339
		};
	    } else {
		options = {
		    maxResolution : 1.40625,
		    maxExtent : new OpenLayers.Bounds(-180, -90, 180, 90)
		};
	    }
	}
	options.projection = new OpenLayers.Projection(this.activeTheme.srs);
	options.numZoomLevels = numZoomLevels;

	OpenLayers.Util.extend(this.map, options);

	var scope = this;

	var waitTillReady = function() {

	    if (scope.map.projection.proj.readyToUse) {
		window.clearTimeout(scope.interval);
		scope.activeTheme.buildMap(scope.map);
		if (callback) {
		    callback.call();
		}

		var defaultCover = null;
		if (!scope.autoDraw) {
		    return
		}
		;
		if ((defaultCover = scope.activeTheme.getDefaultCover())) {
		    scope.doDraw(defaultCover);
		} else if (viewPort.project) {
		    var newExtent = viewPort.previousView.transform(viewPort.previousProjection, new OpenLayers.Projection(scope.activeTheme.srs));
		    scope.doDraw(newExtent);
		} else {
		    scope.doDraw();
		}
	    }

	}

	this.interval = setInterval(waitTillReady, 500);
    },

    /**
     * APIMethod: doDraw
     * 
     * Draws the active theme.
     * 
     * Parameters: extent - {<OpenLayers.Bounds>} or {<OpenLayers.LonLat>}
     * options - {Object} optional.
     * 
     */
    doDraw : function(extent, options) {

	if (extent) {
	    if (extent.CLASS_NAME == "OpenLayers.LonLat") {
		var x = extent.clone();
		var bounds = new OpenLayers.Bounds();
		bounds.extend(extent);
		if (options && options.zoom) {
		    this.map.setCenter(x, options.zoom);
		} else {
		    extent = bounds;
		    this.map.zoomToExtent(extent);
		}
	    } else {
		this.map.zoomToExtent(extent);
	    }
	} else {
	    this.map.zoomToMaxExtent();
	}
	this.events.triggerEvent('afterDraw', this);
    },

    /**
     * Constant: CLASS_NAME - oscar.Control.ThemeManager
     */
    CLASS_NAME : "oscar.Control.ThemeManager"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.OXFConfigManager
 * 
 * The Oscar OXFConfigManager control offers a <load> method that background
 * loads a remote document and builds <ox>, the Oscar eXchange container object,
 * in accordance with the Oscar eXchange Format <oscar.Format.OXF.v2>.
 * 
 * Inherits from: - <OpenLayers.Control>
 * 
 */
oscar.Control.OXFConfigManager = oscar.BaseClass(OpenLayers.Control, {
    /**
     * APIProperty: ox
     * 
     * The Oscar eXchange objects container. Please see <oscar.ox>.
     */
    ox : null,

    /**
     * Constant: EVENT_TYPES
     * 
     * haveThemes - triggered when the <ox> Oscar eXchange container is returned
     * successfully.
     */
    EVENT_TYPES : [ "haveThemes" ],

    /**
     * Property: events
     */
    events : null,

    /**
     * Property: themeManager
     */
    themeManager : null,

    /**
     * Property: defaultThemeId
     */
    defaultThemeId : -1,

    /**
     * Constructor: oscar.Control.OXFConfigManager
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        OpenLayers.Control.prototype.initialize.apply(this, arguments);
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
        this.events.register('haveThemes', this, this.applyThemes);
    },

    /**
     * APIMethod: load
     * 
     * Takes a URL as parameter and background load the document from the URL.
     * 
     * If load successfully, then read the document and build the Oscar eXchange
     * container object <ox>.
     * 
     * If load failed, then generates a error alert dialog.
     * 
     * Parameters: source - {String} the URI of the source document
     * 
     */
    load : function(source) {
        OpenLayers.Request.GET({
            url : source,
            success : this.success,
            fail : this.failure,
            scope : this
        });
    },

    /**
     * Method: success
     * 
     * Builds the <ox> (Oscar eXchange container) object using the Oscar
     * eXchange objects such as ox.Theme and ox.Layer etc, and then returns the
     * ox object container.
     * 
     * Parameters: transport - the returned document from <load>.
     * 
     */
    success : function(transport) {
        // lets seee if we have an xml file
        var cType = transport.getResponseHeader("Content-Type");
        var reader = null;
        var contentType = new String(transport.getResponseHeader("Content-Type"));
        if (contentType.indexOf("xml") > -1) {
            reader = new oscar.Format.OXF.XML();
            try {
                this.ox = reader.read(transport.responseXML);
            } catch (err) {
                this.failure(null);
            }
        } else {
            reader = new oscar.Format.OXF();
            try {
                this.ox = reader.read(transport.responseText);
                if (this.ox.themes.length == 0) {
                    this.failure(null);
                }
            } catch (err) {
                this.failure(null);
                return;
            }
        }

        this.events.triggerEvent("haveThemes", this.ox);
    },

    /**
     * Method: failure
     * 
     * This method is called if the request is unsuccessful.
     * 
     * Parameters:
     * 
     * transport - the returned document from <load>.
     * 
     */
    failure : function(transport) {
        new oscar.Gui.AlertDialog(oscar.i18n("Error"), oscar.i18n("NoDataAvailable"), {
            width : 300,
            height : 100,
            draggable : true
        });
    },

    /**
     * Method: applyThemeToMap
     * 
     * Sets a default themeID.
     * 
     * Parameters:
     * 
     * themeID - the ID of a theme.
     */
    applyThemeToMap : function(themeId) {
        this.defaultThemeId = themeId;
    },

    /**
     * Constant: CLASS_NAME - oscar.Control.OXFConfigManager
     */
    CLASS_NAME : "oscar.Control.OXFConfigManager"
})
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Control.ThemeSwitcher
 * 
 * The Oscar ThemeSwitcher control allows users to switch between themes.
 * 
 * 
 * Translation Keys: - ThemeSwitcherToolTip - ThemeSwitcherTitle
 * 
 * Inherits from: - <oscar.Control.DragPanel>
 * 
 */

oscar.Control.ThemeSwitcher = oscar.BaseClass(oscar.Control.DragPanel, {
    /**
     * APIProperty: tooltipText
     * 
     * {String} - translation key or text to use for the Theme Switcher tool
     * tip. Default is ThemeSwitcherToolTip.
     */
    tooltipText : "ThemeSwitcherToolTip",
    /**
     * APIProperty: titleText
     * 
     * {String} - translation key or text to use for the Theme Switcher title.
     * Default is ThemeSwitcherTitle.
     */
    titleText : "ThemeSwitcherTitle",
    /**
     * APIProperty: iconClass
     * 
     * {String} - css class to use for the icon of the widget.
     */
    iconClass : "ThemesIcon",
    /**
     * Property: closed
     * 
     * {Boolean} - Chooses if the widget should be open or closed initially.
     * Default is false.
     */
    closed : false,
    /**
     * Property: collapsible
     * 
     * {Boolean} - Chooses if the widget can be collapsed. Default is true.
     */
    collapsible : true,
    /**
     * Property: themesPanel
     * 
     * {Array[HTMLDivElements} - Array containing the div elements for the
     * different themes.
     */
    themePanels : null,
    /**
     * APIProperty: themeManager
     * 
     * <oscar.Control.ThemeManager> - The theme manager to use to draw the
     * themes.
     */
    themeManager : new oscar.Control.ThemeManager,
    /**
     * APIProperty: configManager
     * 
     * <oscar.Control.OXFConfigManager> - The configuration manager to use to
     * load theme configurations.
     */
    configManager : new oscar.Control.OXFConfigManager,
    /**
     * Property: toolbar
     * 
     */
    toolbar : null,

    /**
     * APIProperty: activeTheme
     * 
     * Keeps track of the active theme.
     */
    activeTheme : null,

    /**
     * APIProperty: events
     */

    events : null,
    /**
     * APIProperty: EVENT_TYPES
     */
    EVENT_TYPES : [ "switchthemes" ],
    /**
     * Constructor: new oscar.Control.ThemeSwitcher() Creates a new instance of
     * the oscar.Control.ThemeSwitcher control.
     */
    initialize : function(options) {
        this.EVENT_TYPES = oscar.Control.ThemeSwitcher.prototype.EVENT_TYPES.concat(OpenLayers.Control.prototype.EVENT_TYPES);
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
        oscar.Control.DragPanel.prototype.initialize.apply(this, [ options ]);
        if (this.configManager) {
            this.registerConfigManagerEvents();
        }
        this.themePanels = [];
        this.layerGroups = [];
        this.timers = [];
        this.layerCaps = [];
        this.resizable = false;
    },
    /**
     * APIMethod: setConfigManager Sets the configuration manager to use.
     * 
     * Parameters: - configManager <oscar.Control.OXFConfigManager>
     */
    setConfigManager : function(configManager) {
        this.configManager = configManager;
        this.registerConfigManagerEvents();
    },
    /**
     * APIMethod: getConfigManager Returns the configuration manager.
     * 
     * Returns: - <oscar.Control.OXFConfigManager
     */
    getConfigManager : function() {
        return this.configManager;
    },
    /**
     * Method: registerConfigManagerEvents Called from setConfigManager. Sets
     * the listeners on the events of the configuration manager.
     */
    registerConfigManagerEvents : function() {
        this.configManager.events.on({
            "haveThemes" : this.addThemes,
            scope : this
        });
    },
    /**
     * Method: setMap Sets the map object reference
     * 
     * Parameters: - map <OpenLayers.Map>
     */
    setMap : function(map) {
        this.map = map;
        if (this.configManager)
            this.map.addControl(this.configManager);
        if (this.themeManager)
            this.map.addControl(this.themeManager);
        this.toolbar = this.map.getControlsByClass("oscar.Control.ToolBar")[0];

        if ((this.toolbar = this.map.getControlsByClass("oscar.Control.ToolBar")[0])) {
            this.events.on({
                "switchthemes" : this.updateToolbar
            });
        }

        this.map.events.on({
            "addlayer" : this.addLayer,
            scope : this
        });
        this.map.events.on({
            "removelayer" : this.removeLayer,
            scope : this
        });

    },

    /**
     * APIMethod: setThemeManager Sets the themeManager to use.
     * 
     * Parameters: - themeManager <oscar.Control.ThemeManager>
     */
    setThemeManager : function(themeManager) {
        this.themeManager = themeManager;
    },
    /**
     * APIMethod: getThemeManager Returns the theme manager object.
     * 
     * Returns: - <oscar.Control.ThemeManager>
     */
    getThemeManager : function() {
        return this.themeManager;
    },

    /**
     * Method: addThemes Called when the configuration manager has completed its
     * request.
     * 
     * Parameters: - ox <oscar.ox>
     */
    addThemes : function(ox) {
        this.ox = ox;
        this.buildMenu();

        /*
         * Check to see if there is a specific theme being requested from a
         * permalink
         */
        if (this.themeManager.argParser) {
            var requestedThemeName = this.themeManager.argParser.args["theme"];
            for ( var item in this.ox.themes) {
                var theme = this.ox.themes[item];
                if (theme.name == requestedThemeName) {
                    themeFound = true;
                    this.themeSelect[0].selectedIndex = item;
                    this.themeManager.autoDraw = false;
                    var scope = this;
                    this.changeTheme(item, function() {
                        var center = new OpenLayers.LonLat(parseFloat(scope.themeManager.argParser.args.lon), parseFloat(scope.themeManager.argParser.args.lat));
                        var zoom = parseInt(scope.themeManager.argParser.args.zoom);
                        var options = {
                            zoom : zoom
                        };
                        scope.themeManager.doDraw(center, options);
                    });
                    return;
                }
            }
        }
        this.changeTheme(0);
    },

    /**
     * Method: changeTheme Draws a new theme
     * 
     * Parameters: - val <Number> array index of the theme to draw.
     */
    changeTheme : function(val, callback) {
        if (this.activeTheme != null) { // check to see if it's already active
            if (this.activeTheme == val)
                return;
        }
        this.activeTheme = val;

        this.events.triggerEvent("switchthemes", this.ox.themes[val]);
        this.themeManager.drawTheme(this.ox.themes[val], callback);

    },
    updateToolbar : function(theme) {
        this.toolbar.applyTheme(theme);
    },
    /**
     * Method: buildMenu Creates the theme switcher menu.
     */
    buildMenu : function() {
        var scope = this;

        var customContainer = document.createElement("div");
        var $customContainer = $$(customContainer);
        oscar.jQuery(customContainer).addClass("themeContainer");
        $$(this.content).append(customContainer);

        this.$themeDiv = $$("<div></div>").css({
            "position" : "relative"
        });
        this.themeSelect = $$("<select></select>").change(function() {
            scope.changeTheme(scope.themeSelect[0].selectedIndex);
        });
        this.$themeDiv.append(this.themeSelect);
        $customContainer.append(this.$themeDiv);

        for ( var item in this.ox.themes) {
            var theme = this.ox.themes[item];
            var opt = $$("<option></option").html(theme.name);
            this.themeSelect.append(opt);

        }

        this.$layersPanel = $$("<div></div>").addClass("themeLayers").css({
            "height" : "110px",
            "overflow-y" : "auto"
        });
        $$(this.content).append(this.$layersPanel);
    },

    /**
     * Method: draw Draws the control on the map.
     */
    draw : function() {
        oscar.Control.DragPanel.prototype.draw.apply(this);
        return this.div;
    },
    /**
     * Method: addLayer *Parameters* - evt <OpenLayers.Event> The event object
     * passed in when the addlayer event is fired from the map object.
     */
    addLayer : function(evt) {
        if (evt.layer.isBaseLayer) { // first theme or switched theme
            this.layerGroups = [];
            this.timers = [];
            this.layerCaps = [];
        }
        if (evt.layer.displayInLayerSwitcher && evt.layer.isBaseLayer == false) {
            this.getLayerCapabilities(evt.layer);
        }
    },
    /**
     * Method: removeLayer *Parameters* - evt <OpenLayers.Event> The event
     * object passed in when the removelayer event is fired from the map object.
     */
    removeLayer : function(evt) {
        var node = $("LT_" + evt.layer.id);
        if (node != null) {
            $$(node).remove();
        }
    },
    /**
     * Method: getLayerCapabilities Called when a layer is added. This method
     * will create a capabilities request based off of the layer properties.
     * Parameters: - layer <OpenLayers.Layer>
     * 
     */
    getLayerCapabilities : function(layer) {
        this.setPlaceHolder(layer);
        var params = {
            request : "GetCapabilities",
            service : "WMS"
        }

        switch (layer.CLASS_NAME) {
        case "OpenLayers.Layer.WMS":
            var success = function(resp) {
                var formatter = new OpenLayers.Format.WMSCapabilities();
                var capabilities = null;
                try {
                    capabilities = formatter.read(resp.responseXML);
                } catch (err) {
                    capabilities = formatter.read(resp.responseText);
                }
                this.layerCaps[layer.url] = capabilities;
                this.displayLayer(layer);
            };
            var fail = function(resp) {
            };
            OpenLayers.Request.GET({
                url : layer.url[0],
                params : params,
                success : success,
                fail : fail,
                scope : this
            });
            break;
        case "OpenLayers.Layer.WMTS":
            var baseUrl = layer.url;
            var capabilitiesUrl = baseUrl + "/" + layer.version + "/WMTSCapabilities.xml";
            var success = function(resp) {
                var formatter = new oscar.Format.WMTSCapabilities();
                var capabilities = null;
                try {
                    capabilities = formatter.read(resp.responseXML);
                } catch (err) {
                    capabilities = formatter.read(resp.responseText);
                }
                this.layerCaps[layer.url] = capabilities;
                this.displayLayer(layer);
            };
            var fail = function(resp) {
            };

            // clear out the parameters object as it is not needed.
            params = null;

            OpenLayers.Request.GET({
                url : capabilitiesUrl,
                success : success,
                fail : fail,
                scope : this
            });
            break;
        case "OpenLayers.Layer.Vector":
            this.displayLayer(layer);
            break;
        }
    },
    /**
     * Method: setPlaceholder Creates a DIV node to hold the the checkboxes and
     * labels of the service layers. Parameters: - layer <OpenLayers.Layer>
     * 
     */
    setPlaceHolder : function(layer) {
        var layerDiv = document.createElement("div");
        layerDiv.setAttribute("class", "layerToggle");
        id = "LT_" + layer.id;
        layerDiv.id = id;
        if (this.content.hasChildNodes()) {
            this.$layersPanel.prepend(layerDiv);
        } else {
            this.$layersPanel.append(layerDiv);
        }
    },
    /**
     * Method: showWMSLayer Displays the checkboxes and labels for the service
     * layers of the OL.Layer.WMS object.
     * 
     * Parameters: - layer <OpenLayers.Layer.WMS>
     */
    showWMSLayer : function(layer) {
        try {
            var layerDiv = $("LT_" + layer.id);
            var layerGroupName = OpenLayers.Util.createUniqueID(layer.name);
            this.layerGroups[layerGroupName] = {
                layer : layer,
                layers : []
            };
            if (layer.params.LAYERS.length > 0) {
                var CapabilitiesLayers = oscar.Util.Metadata.getLayers(this.layerCaps[layer.url]);
                var NameToTitle = function(name) {
                    for ( var sLayer in CapabilitiesLayers) {
                        var ServiceLayer = CapabilitiesLayers[sLayer];
                        if (name == ServiceLayer.name)
                            return ServiceLayer.title || ServiceLayer.name
                    }
                }
                var serviceLayers = layer.params.LAYERS.slice(0);
                serviceLayers = serviceLayers.reverse();
                for ( var wmsLayer in serviceLayers) {
                    var layerTitle = NameToTitle(serviceLayers[wmsLayer]);
                    var randomId = OpenLayers.Util.createUniqueID(layer.name);
                    var inputDiv = document.createElement("div");
                    oscar.jQuery(inputDiv).addClass("serviceLayer");
                    var input = document.createElement("input");
                    input.type = "checkbox";
                    input.title = layerTitle;
                    input.value = serviceLayers[wmsLayer];
                    var inputId = OpenLayers.Util.createUniqueID("checkbox");
                    input.id = inputId;
                    inputDiv.appendChild(input);
                    input.checked = true;
                    var label = document.createElement("label");
                    label.setAttribute("for", inputId);
                    oscar.jQuery(label).addClass("serviceLayerLabel");
                    // label.setAttribute("className","serviceLayerLabel");
                    label.innerHTML = layerTitle;
                    var btn = document.createElement("button");
                    btn.value = serviceLayers[wmsLayer];
                    inputDiv.appendChild(label);
                    // inputDiv.appendChild(btn);
                    layerDiv.appendChild(inputDiv);
                    oscar.jQuery(btn).button({
                        icons : {
                            primary : "ui-icon-gear"
                        }
                    });

                    var getClickFn = function(context, layerGroup) {
                        return function(e) {
                            context.adjustLayerGroup(layerGroup);
							e.stopPropagation();
                        };
                    };
                    this.layerGroups[layerGroupName].layers.push(input);
                    OpenLayers.Event.observe(input, "click", OpenLayers.Function.bindAsEventListener(getClickFn(this, layerGroupName)));
                }
            }
        } catch (err) {
        }

    },
    /**
     * Method: showWMTSLayer Displays the checkboxes and labels for the service
     * layers of the OL.Layer.WMTS object.
     * 
     * Parameters: - layer <OpenLayers.Layer.WMTS>
     */

    showWMTSLayer : function(layer) {
        var CapabilitiesLayers = oscar.Util.Metadata.getContent(this.layerCaps[layer.url]);
        var CapabilitiesThemes = oscar.Util.Metadata.getThemes(this.layerCaps[layer.url]);
        var items = CapabilitiesLayers.concat(CapabilitiesThemes);
        var NameToTitle = function(identifier) {
            for ( var sLayer in items) {
                var ServiceLayer = items[sLayer];
                if (identifier == ServiceLayer.identifier)
                    return ServiceLayer.title || ServiceLayer.identifier
            }
        }
        var layerDiv = $("LT_" + layer.id);
        var layerTitle = NameToTitle(layer.layer);
        var randomId = OpenLayers.Util.createUniqueID(layerTitle);
        var inputDiv = document.createElement("div");
        var input = document.createElement("input");
        input.type = "checkbox";
        input.title = layerTitle;
        input.value = layer.layer;
        inputDiv.appendChild(input);
        input.checked = true;
        var inputId = OpenLayers.Util.createUniqueID("checkbox");
        input.id = inputId;
        var label = document.createElement("label");
        oscar.jQuery(label).addClass("serviceLayerLabel");
        label.setAttribute("for", inputId);
        label.innerHTML = layerTitle;
        inputDiv.appendChild(label);
        layerDiv.appendChild(inputDiv);
        var obj = {
            layer : layer
        }
        var getClickFn = function(layer) {
            return function(e) {
                layer.setVisibility(!layer.getVisibility());
				e.stopPropagation();
            };
        };
        OpenLayers.Event.observe(input, "click", OpenLayers.Function.bindAsEventListener(getClickFn(layer)));

    },
    /**
     * Method: showVectorLayer
     * 
     * This method allows the display of the POI layers which are based on the
     * OpenLayers.Layer.Vector layer.
     */
    showVectorLayer : function(layer) {
        var layerDiv = $("LT_" + layer.id);
        var layerTitle = layer.name;
        var randomId = OpenLayers.Util.createUniqueID(layerTitle);
        var inputDiv = document.createElement("div");
        var input = document.createElement("input");
        input.type = "checkbox";
        input.title = layerTitle;
        input.value = layer.layer;
        inputDiv.appendChild(input);
        input.checked = layer.visibility;
        var inputId = OpenLayers.Util.createUniqueID("checkbox");
        input.id = inputId;
        var label = document.createElement("label");
        oscar.jQuery(label).addClass("serviceLayerLabel");
        label.setAttribute("for", inputId);
        label.innerHTML = layerTitle;
        inputDiv.appendChild(label);
        layerDiv.appendChild(inputDiv);
        var obj = {
            layer : layer
        }
        var getClickFn = function(layer) {
            return function(e) {
                layer.setVisibility(!layer.getVisibility());
				e.stopPropagation();
            };
        };
        OpenLayers.Event.observe(input, "click", OpenLayers.Function.bindAsEventListener(getClickFn(layer)));

    },

    /**
     * Method: displayLayer Called after the capabilities object has been
     * created. Parameters: - layer <OpenLayers.Layer>
     */
    displayLayer : function(layer) {
        if (layer.isBaseLayer) {
            return;
        }
        switch (layer.CLASS_NAME) {
        case "OpenLayers.Layer.WMS":
            this.showWMSLayer(layer);
            break;
        case "OpenLayers.Layer.WMTS":
            this.showWMTSLayer(layer);
            break;
        case "OpenLayers.Layer.Vector":
            this.showVectorLayer(layer);
            break;
        }

    },
    /**
     * Method: adjustLayerGroup This method will trigger the redraw method of a
     * layer when a checkbox has been clicked. Parameters: - layerGroup {String} -
     * name of the layer group to adjust.
     */
    adjustLayerGroup : function(layerGroup) {
        if (this.timers[this.layerGroups[layerGroup].layer.id]) {
            window.clearTimeout(this.timers[this.layerGroups[layerGroup].layer.id]);
            this.timers[this.layerGroups[layerGroup].layer.id] = null;
        }
        ;

        var inputs = this.layerGroups[layerGroup].layers;
        var layer = this.layerGroups[layerGroup].layer;
        var layers = [];
        for (var i = 0; i < inputs.length; i++) {
            if (inputs[i].checked) {
                layers.push(inputs[i].value);
            }
            ;
        }
        if (layers.length == 0) {
            layer.setVisibility(false);

        } else {
            var redraw = function(layer, layers) {
                return function() {
                    var reversedLayers = layers.reverse();
                    layer.mergeNewParams({
                        layers : reversedLayers
                    });
                    layer.setVisibility(true);
                }
            };
            this.timers[this.layerGroups[layerGroup].layer.id] = window.setTimeout(redraw(layer, layers), this.threshold);
        }
    },
    CLASS_NAME : "oscar.Control.ThemeSwitcher"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format
 * 
 * The Definition for the Oscar Format namespace.
 * 
 * Inherits from: - <OpenLayers.Format>
 * 
 */
oscar.Format = oscar.BaseClass(OpenLayers.Format, {
    /**
     * Constant: CLASS_NAME - oscar.Format
     */
    CLASS_NAME : "oscar.Format"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.OGC
 * 
 * Formatter base class for OGC XML documents.
 * 
 */
oscar.Format.OGC = {
    /**
     * Constant: CLASS_NAME - oscar.Format.OGC
     */
    CLASS_NAME : "oscar.Format.OGC"
};
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.OGC.ows
 * 
 * Methods used for reading elements defined in the OWS (OGC Web Services)
 * namespace.
 */

oscar.Format.OGC.ows = {};

oscar.Format.OGC.ows.v1_0_0 = {
    /**
     * Method: read_cap_ServiceIdentification
     * 
     * Reads the ServiceIdentification node.
     */
    read_cap_ServiceIdentification : function(capabilities, node) {
        capabilities.serviceIdentification = {};
        this.runChildNodes(capabilities.serviceIdentification, node);
    },
    /**
     * Method: read_cap_ServiceProvider
     * 
     * Reads the ServiceProvider node.
     */
    read_cap_ServiceProvider : function(capabilities, node) {
        capabilities.serviceProvider = {};
        this.runChildNodes(capabilities.serviceProvider, node);
    },
    /**
     * Method: read_cap_ProviderName
     * 
     * Reads the ProviderName node.
     */
    read_cap_ProviderName : function(serviceProvider, node) {
        serviceProvider.providerName = this.getChildValue(node);
    },
    /**
     * Method: read_cap_ProviderSite
     * 
     * Reads the ProviderSite node.
     */
    read_cap_ProviderSite : function(serviceProvider, node) {
        var href = this.getAttributeNS(node, "http://www.w3.org/1999/xlink", "href");
        if (href) {
            serviceProvider.providerSite = href;
        }
    },
    /**
     * Method: read_cap_ServiceContact
     * 
     * Reads the ServiceContact node.
     */
    read_cap_ServiceContact : function(serviceProvider, node) {
        serviceProvider.serviceContact = {};
        this.runChildNodes(serviceProvider.serviceContact, node);
    },
    /**
     * Method: read_cap_IndividualName
     * 
     * Reads the IndividualName node.
     */
    read_cap_IndividualName : function(serviceContact, node) {
        serviceContact.individualName = this.getChildValue(node);
    },
    /**
     * Method: read_cap_PositionName
     * 
     * Reads the PositionName node.
     */
    read_cap_PositionName : function(serviceContact, node) {
        serviceContact.positionName = this.getChildValue(node);
    },
    /**
     * Method: read_cap_ContactInfo
     * 
     * Reads the ContactInfo node.
     */
    read_cap_ContactInfo : function(serviceContact, node) {
        serviceContact.contactInfo = {};
        this.runChildNodes(serviceContact.contactInfo, node);
    },
    /**
     * Method: read_cap_Phone
     * 
     * Reads the Phone node.
     */
    read_cap_Phone : function(contactInfo, node) {
        if (!contactInfo.phone) {
            contactInfo.phone = {};
        }
        this.runChildNodes(contactInfo.phone, node);
    },
    /**
     * Method: read_cap_Voice
     * 
     * Reads the Voice node.
     */
    read_cap_Voice : function(phone, node) {
        phone.voice = this.getChildValue(node);
    },
    /**
     * Method: read_cap_Facsimile
     * 
     * Reads the Voice node.
     */
    read_cap_Facsimile : function(phone, node) {
        phone.facsimile = this.getChildValue(node);
    },

    /**
     * Method: read_cap_Address
     * 
     * Reads the Address node.
     */
    read_cap_Address : function(contactInfo, node) {
        contactInfo.address = {};
        this.runChildNodes(contactInfo.address, node);
    },

    /**
     * Method: read_cap_DeliveryPoint
     * 
     * Reads the DeliveryPoint node, which is the street address.
     */
    read_cap_DeliveryPoint : function(contactInfo, node) {
        contactInfo.deliveryPoint = this.getChildValue(node);
    },

    /**
     * Method: read_cap_City
     * 
     * Reads the City node.
     */
    read_cap_City : function(address, node) {
        address.city = this.getChildValue(node);
    },
    /**
     * Method: read_cap_AdministrativeArea
     * 
     * Reads the AdministrativeArea node.
     */
    read_cap_AdministrativeArea : function(address, node) {
        address.administrativeArea = this.getChildValue(node);
    },
    /**
     * Method: read_cap_Country
     * 
     * Reads the Country node.
     */
    read_cap_Country : function(address, node) {
        address.country = this.getChildValue(node);
    },
    /**
     * Method: read_cap_ElectronicMailAddress
     * 
     * Reads the ElectronicMailAddress node.
     */
    read_cap_ElectronicMailAddress : function(address, node) {
        address.electronicMailAddress = this.getChildValue(node);
    },
    /**
     * Method: read_cap_OnlineResource
     * 
     * Reads the OnlineResource node.
     */
    read_cap_OnlineResource : function(contactInfo, node) {
        var href = this.getAttributeNS(node, "http://www.w3.org/1999/xlink", "href");
        if (href) {
            contactInfo.onlineResource = href;
        }
    },
    /**
     * Method: read_cap_Title
     * 
     * Reads the Title node.
     */
    read_cap_Title : function(obj, node) {
        obj.title = this.getChildValue(node);
    },

    /**
     * Method: read_cap_Identifier
     * 
     * Reads the Identifier node.
     */
    read_cap_Identifier : function(obj, node) {
        obj.identifier = this.getChildValue(node);
    },

    /**
     * Method: read_cap_Abstract
     * 
     * Reads the Abstract node.
     */
    read_cap_Abstract : function(obj, node) {
        obj['abstract'] = this.getChildValue(node);
    },
    /**
     * Method: read_cap_Keywords
     * 
     * Reads the Keywords node.
     */
    read_cap_Keywords : function(obj, node) {
        obj.keywords = [];
        this.runChildNodes(obj.keywords, node);
    },
    /**
     * Method: read_cap_Keyword
     * 
     * Reads the Keyword node.
     */
    read_cap_Keyword : function(obj, node) {
        var keyword = this.getChildValue(node);
        obj.push(keyword);
    },
    /**
     * Method: read_cap_ServiceType
     * 
     * Reads the ServiceType node.
     */
    read_cap_ServiceType : function(obj, node) {
        obj.serviceType = this.getChildValue(node);
    },
    /**
     * Method: read_cap_OperationsMetadata
     * 
     * Reads the OperationsMetadata node.
     */
    read_cap_OperationsMetadata : function(capabilities, node) {
        capabilities.operationsMetadata = {};
        this.runChildNodes(capabilities.operationsMetadata, node);
    },
    /**
     * Method: read_cap_Operation
     * 
     * Reads the Operation node.
     */
    read_cap_Operation : function(obj, node) {
        var operation = this.getAttributeNS(node, "", "name");
        obj[operation] = {};
        this.runChildNodes(obj[operation], node);
    },
    /**
     * Method: read_cap_DCP
     * 
     * Reads the DCP node.
     */
    read_cap_DCP : function(obj, node) {
        if (!obj.dcp)
            obj.dcp = {};
        this.runChildNodes(obj.dcp, node);
    },
    /**
     * Method: read_cap_HTTP
     * 
     * Reads the HTTP node.
     */
    read_cap_HTTP : function(obj, node) {
        if (!obj.http)
            obj.http = {};
        this.runChildNodes(obj.http, node);
    },
    /**
     * Method: read_cap_Get
     * 
     * Reads the Get node.
     */
    read_cap_Get : function(obj, node) {
        var href = this.getAttributeNS(node, "http://www.w3.org/1999/xlink", "href");
        if (href) {
            obj.get = href;
        }
    },
    /**
     * Method: read_cap_Post
     * 
     * Reads the Post node.
     */
    read_cap_Post : function(obj, node) {
        var href = this.getAttributeNS(node, "http://www.w3.org/1999/xlink", "href");
        if (href) {
            obj.post = href;
        }
    },

    /**
     * Method:read_cap_Parameter
     * 
     * Reads the Parameter node.
     */
    read_cap_Parameter : function(obj, node) {
        if (!obj.parameters)
            obj.parameters = [];

        var parameter = {};
        parameter.name = this.getAttributeNS(node, "", "name");
        this.runChildNodes(parameter, node);
        obj.parameters.push(parameter);
    },

    /**
     * Method:reaed_cap_Value
     * 
     * Reads the Value node.
     */
    read_cap_Value : function(obj, node) {
        if (!obj.values)
            obj.values = [];
        var value = this.getChildValue(node);
        obj.values.push(value);
    },

    /**
     * Method: read_cap_AllowedValues
     * 
     * Reads the AllowedValues node.
     */
    read_cap_AllowedValues : function(obj, node) {
        this.runChildNodes(obj, node);
    },
    /**
     * Method: read_cap_WGS84BoundingBox
     * 
     * Reads the WGS84BoundingBox node.
     */
    read_cap_WGS84BoundingBox : function(obj, node) {
        obj.wgs84BoundingBox = {};
        this.runChildNodes(obj.wgs84BoundingBox, node);

    },
    /**
     * Method: read_cap_BoundingBox
     * 
     * Reads the BoundingBox node.
     */
    read_cap_BoundingBox : function(obj, node) {
        if (!obj.boundingBoxes) {
            obj.boundingBoxes = [];
        }
        var bbox = {};
        var crs = this.getAttributeNS(node, "", "crs");
        bbox.crs = crs;
        this.runChildNodes(bbox, node);
        obj.boundingBoxes.push(bbox);
    },
    /**
     * Method: read_cap_LowerCorner
     * 
     * Reads the LowerCorner node.
     */
    read_cap_LowerCorner : function(obj, node) {
        var leftBottom = this.getChildValue(node).split(' ');
        obj.west = leftBottom[0];
        obj.south = leftBottom[1];
    },
    /**
     * Method: read_cap_UpperCorner
     * 
     * Reads the UpperCorner node.
     */
    read_cap_UpperCorner : function(obj, node) {
        var rightTop = this.getChildValue(node).split(' ');
        obj.east = rightTop[0];
        obj.north = rightTop[1];
    },

    /**
     * Method: read_cap_Metadata
     * 
     * Reads the Metadata node
     */
    read_cap_Metadata : function(obj, node) {
        if (!obj.metadataUrls)
            obj.metadataUrls = [];
        var url = this.getAttributeNS(node, "http://www.w3.org/1999/xlink", "href");
        if (url.length == 0)
            return;
        var metadata = {
            url : url,
            type : "",
            format : ""
        };
        obj.metadataUrls.push(metadata);

    },
    readers : {
        "ows" : {
            "Constraint" : function(obj, node) {
                if (!obj.constraints) {
                    obj.constraints = [];
                }
                var constraint = {};
                constraint.name = this.getAttributeNS(node, "", "name");
                this.runChildNodes(constraint, node);
                obj.constraints.push(constraint);
            },

            "Value" : function(obj, node) {
                if (!obj.values) {
                    obj.values = [];
                }
                obj.values.push(this.getChildValue(node));
            }
        }

    },

    /**
     * Constant: CLASS_NAME - oscar.Format.OGC.ows
     */
    CLASS_NAME : "oscar.Format.OGC.ows.v1_0_0"
};
oscar.Format.OGC.ows.v1_1_0 = {};

OpenLayers.Util.extend(oscar.Format.OGC.ows.v1_1_0, oscar.Format.OGC.ows.v1_0_0);

OpenLayers.Util.extend(oscar.Format.OGC.ows.v1_1_0, {
    read_cap_Title : function(obj, node) {
        if (!obj.title)
            obj.title = {};
        var lang = null;
        lang = this.getAttributeNS(node, "http://www.w3.org/XML/1998/namespace", "lang");
        if (lang != null && lang.length > 0) {
            obj.title[lang] = this.getChildValue(node);
        } else {
            obj.title = this.getChildValue(node);
        }
    },
    CLASS_NAME : "oscar.Format.OGC.ows.v1_1_0"
});

oscar.Format.OGC.owcs = oscar.Format.OGC.ows.v1_0_0;
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.OGC.wcs
 * 
 * Methods used for reading elements defined in the OGC WCS namespace.
 */
oscar.Format.OGC.wcs = {

    /**
     * Method: read_cap_Contents
     * 
     * Reads the Contents node.
     */
    read_cap_Contents : function(obj, node) {
        obj.contents = {};
        this.runChildNodes(obj.contents, node);
    },

    /**
     * Method: read_cap_CoverageSummary
     * 
     * Reads the CoverageSummary node.
     */
    read_cap_CoverageSummary : function(contents, node) {
        if (!contents.coverages)
            contents.coverages = [];
        var coverageSummary = {};
        this.runChildNodes(coverageSummary, node);
        contents.coverages.push(coverageSummary);
    },
    /**
     * Method: read_cap_Identifier
     * 
     * Reads the Identifier node.
     */
    read_cap_Identifier : function(obj, node) {
        obj.identifier = this.getChildValue(node);
    },
    /**
     * Method: read_cap_SupportedFormat
     * 
     * Reads the SupportedFormat node.
     */
    read_cap_SupportedFormat : function(contents, node) {
        if (!contents.supportedFormats)
            contents.supportedFormats = [];
        var format = this.getChildValue(node);
        contents.supportedFormats.push(format);
    },
    /**
     * Method: read_cap_Domain
     * 
     * Reads the Domain node.
     */
    read_cap_Domain : function(coverDesc, node) {
        if (!coverDesc.domain) {
            coverDesc.domain = {};
        }

        this.runChildNodes(coverDesc.domain, node)
    },
    /**
     * Method: read_cap_SpatialDomain
     * 
     * Reads the SpatialDomain node.
     */
    read_cap_SpatialDomain : function(domain, node) {
        if (!domain.spatialDomain)
            domain.spatialDomain = {};

        this.runChildNodes(domain.spatialDomain, node);
    },

    /**
     * Method: read_cap_SupportedCRS
     * 
     * Reads the SupportedCRS node
     */
    read_cap_SupportedCRS : function(obj, node) {
        if (!obj.supportedCRS) {
            obj.supportedCRS = [];
        }
        var crs = this.getChildValue(node);
        obj.supportedCRS.push(crs);
    },
    /**
     * Method: read_cap_OtherSource
     * 
     * Reads the OtherSource node
     */
    read_cap_OtherSource : function(obj, node) {
        if (!obj.OtherSource) {
            obj.OtherSource = [];
        }
        obj.OtherSource.push({
            href : this.getAttributeNS(node,"http://www.w3.org/1999/xlink", "href")
        });
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.OGC.wcs
     */
    CLASS_NAME : "oscar.Format.OGC.wcs"
};
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.OGC.wfs
 * 
 * Methods used for reading elements defined in the OGC WFS namespace.
 */
oscar.Format.OGC.wfs = {

    /**
     * Method: read_cap_OutputFormats
     * 
     * Reads the OutputFormats node.
     */
    read_cap_OutputFormats : function(featureType, node) {
        var outputFormats = {
            formats : []
        }
        this.runChildNodes(outputFormats, node);
        featureType.outputFormats = outputFormats;
    },

    /**
     * Method: read_cap_Format
     * 
     * Reads the Format node.
     */
    read_cap_Format : function(obj, node) {
        var format = this.getChildValue(node);
        if (format) {
            obj.formats.push(format);
        }
    },
    /**
     * Method: read_cap_MetadataURL
     * 
     * Reads the MetadataURL node.
     */
    read_cap_MetadataURL : function(obj, node) {
        if (!obj.metadataUrls)
            obj.metadataUrls = [];
        var url = this.getChildValue(node);
        if (url.length == 0)
            return;
        var metadata = {
            url : url,
            type : this.getAttributeNS(node, "", "type"),
            format : this.getAttributeNS(node, "", "format")
        };
        obj.metadataUrls.push(metadata);
    },
    /**
     * Method: read_cap_DefaultSRS
     * 
     * Reads the DefaultSRS node.
     */
    read_cap_DefaultSRS : function(obj, node) {
        if (!obj.srss) {
            obj.srss = [];
        }
        obj.srss.push(this.getChildValue(node))
    },

    /**
     * Method: read_cap_OtherSRS
     * 
     * Reads the OtherSRS node.
     */
    read_cap_OtherSRS : function(obj, node) {
        if (!obj.srss) {
            obj.srss = [];
        }
        obj.srss.push(this.getChildValue(node))
    },

    /**
     * Constant: CLASS_NAME - oscar.Format.OGC.wfs
     */
    CLASS_NAME : "oscar.Format.OGC.wfs"
};
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.OGC.wmts
 * 
 * Methods used for reading elements defined in the OGC WMTS namespace
 */
oscar.Format.OGC.wmts = {
    /**
     * Constant: CLASS_NAME - oscar.Format.OGC.wmts
     */
    CLASS_NAME : "oscar.Format.OGC.wmts"
};
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.OXF
 * 
 * Reads an OXF (Oscar eXchange Format) configuration resource.
 * 
 * Inherits from: - <OpenLayers.Format.JSON>
 */

oscar.Format.OXF = oscar.BaseClass(OpenLayers.Format.JSON, {

    /**
     * Proptery: defaultVersion
     * 
     * Current version of the format.
     */
    defaultVersion : 2.0,

    /**
     * Property: version
     * 
     * Version to use when parsing
     */
    version : null,

    /**
     * Constructor: oscar.Format.OXF
     * 
     * Parameters: options - {Object} An optional object whose properties will
     * be set on this instance.
     */
    initialize : function(options) {
        OpenLayers.Format.JSON.prototype.initialize.apply(this, [ options ]);
    },

    /**
     * APIMethod: read
     * 
     * Reads an OXF document file.
     * 
     * Parameters: data - the document in OXF document file to be read.
     * 
     * Returns: configuration - the OXF configuration object.
     */
    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.JSON.prototype.read.apply(this, [ data ]);
        }
        var version = "" + (this.version || data.version || this.defaultVersion);
        var constr = oscar.Format.OXF["v" + version.replace(/\./g, "_")];
        if (!constr) {
            throw "Can't find a OXF parser for version " + version;
        }
        var parser = new constr(this.options);
        var configuration = parser.read(data);
        return configuration;
    },

    /**
     * Constant: CLASS_NAME - oscar.Format.OXF
     */
    CLASS_NAME : "oscar.Format.OXF"

});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.OXF.v2
 * 
 * Methods for reading OXF (Oscar eXchange Format) configuration objects. This
 * class is not to be instantiated directly.
 * 
 */

oscar.Format.OXF.v2 = oscar.BaseClass( {
	/**
         * Property: filters Used to parse different components of an OXF
         * string.
         */
	filters : {
		themes : function(obj, data) {
			if (!obj.themes)
				obj.themes = [];
			for ( var i = 0; i < data.length; i++) {
				var theme = new oscar.ox.Theme();
				this.runProps(theme, data[i]);
				obj.themes.push(theme);
			}
		},
		layers : function(obj, data) {
			if (!obj.layers)
				obj.layers = [];
			for ( var i = 0; i < data.length; i++) {
				var layer = new oscar.ox.Layer();
				this.runProps(layer, data[i]);
				obj.layers.push(layer);
			}
		},
		services : function(obj, data) {
			if (!obj.services) {
				obj.services = new oscar.ox.Services();
			}
			this.runProps(obj.services, data);
		},
		selection : function(obj, data) {
			if (data.length == 0)
				return
			var selectionService = obj.addSelectionService();
			this.runProps(selectionService, data[0]);
		},
		extraction : function(obj, data) {
			if (data.length == 0)
				return
			var extractionService = obj.addExtractionService();
			this.runProps(extractionService, data[0]);
		},
		catalogue : function(obj, data) {
			if (data.length == 0)
				return
			var catalogueService = obj.addCatalogueService();
			this.runProps(catalogueService, data[0]);
		},
		serviceEntries : function(obj, data) {
			for ( var i = 0; i < data.length; i++) {
				var se = new oscar.ox.ServiceEntry(data[i]);
				obj.addServiceEntry(se);
			}
		}
	},
	/**
         * Constructor: oscar.Format.OXF.v2
         * 
         * Creates an instance of one of the subclasses.
         * 
         * options - {Object} An optional object whose properties will be set on
         * this instance.
         */
	initialize : function(options) {

	},

	/**
         * APIMethod: read
         * 
         * Reads an OXF configuration object.
         */
	read : function(data) {
		var obj = {};
		this.runProps(obj, data);
		return obj;
	},
	/**
         * Method: runProps
         * 
         * Iterates through the properties of an object to see if there are any
         * parsing filters to be applied.
         */
	runProps : function(obj, data) {
		for ( var props in data) {
			var filter = this.filters[props];
			if (filter) {
				filter.apply(this, [ obj, data[props] ]);
			} else {
				obj[props] = data[props];
			}
		}
	},
	/**
         * Constant: CLASS_NAME - oscar.Format.OXF.v2
         */
	CLASS_NAME :"oscar.Format.OXF.v2"

});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Format.OXF.XML = oscar.BaseClass(OpenLayers.Format.XML, {
    /**
     * Property: defaultPrefix
     * 
     * default namespace for the xml nodes.
     */
    defaultPrefix : "ox",
    /**
     * Constructor: new oscar.Format.OXF.XML()
     */
    initialize : function(options) {
        OpenLayers.Format.XML.prototype.initialize.apply(this, [ options ]);
    },
    /**
     * APIMethod: read
     * 
     * Parameters: - {String} or {XMLDocument}
     * 
     * Returns: - {oscar.ox} configuration object.
     */
    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.XML.prototype.read.apply(this, [ data ]);
        }
        if (data && data.nodeType == 9) {
            data = data.documentElement;
        }
        var obj = new oscar.ox();
        this.readNode(data, obj);
        return obj;
    },

    /**
     * List of node readers.
     */
    readers : {
        "ox" : {
            "configuration" : function(node, obj) {
                obj.version = node.getAttribute("version");
                this.readChildNodes(node, obj);
            },
            "themes" : function(node, obj) {
                this.readChildNodes(node, obj);
            },
            "theme" : function(node, obj) {
                var theme = new oscar.ox.Theme();

                theme.parameters = {};
                var nZoom = parseInt(node.getAttribute("numzoomlevels"));
                theme.parameters.numzoomlevels = (isNaN(nZoom)) ? 16 : nZoom;
                var bColor = node.getAttribute("bgColor");
                theme.parameters.backgroundColor = (bColor == null) ? "#fff" : bColor;

                obj.addTheme(theme);
                this.readChildNodes(node, theme);

            },
            "srs" : function(node, obj) {
                var srs = this.getChildValue(node);
                obj.srs = srs;
            },
            "name" : function(node, obj) {
                obj.name = this.getChildValue(node);
            },

            "covers" : function(node, obj) {
                var covers = [];
                this.readChildNodes(node, covers)
                obj.covers = covers;
            },
            "cover" : function(node, cArr) {
                var cover = {};
                cover.type = node.getAttribute("type");
                cover.minX = parseFloat(node.getAttribute("minX"));
                cover.minY = parseFloat(node.getAttribute("minY"));
                cover.maxX = parseFloat(node.getAttribute("maxX"));
                cover.maxY = parseFloat(node.getAttribute("maxY"));
                cArr.push(cover);
            },
            "layers" : function(node, obj) {
                this.readChildNodes(node, obj);
            },
            "layer" : function(node, obj) {
                var layer = new oscar.ox.Layer();
                obj.addLayer(layer);
                this.readChildNodes(node, layer);
            },
            "urls" : function(node, obj) {
                this.readChildNodes(node, obj);
            },
            "url" : function(node, obj) {
                var url = this.getChildValue(node);
                obj.addUrl($$.trim(url));
            },
            "format" : function(node, obj) {
                obj.format = this.getChildValue(node);
            },
            "baseLayer" : function(node, obj) {

            },
            "layerType" : function(node, obj) {
                obj.layerType = this.getChildValue(node);
            },
            "dataLayers" : function(node, obj) {
                this.readChildNodes(node, obj);
            },
            "dataLayer" : function(node, obj) {
                var dLayer = new oscar.ox.DataLayer();
                obj.addDataLayer(dLayer);
                this.readChildNodes(node, dLayer);
            },
            "layerName" : function(node, obj) {
                var layerName = this.getChildValue(node);
                obj.layerName = layerName;
            },
            "tileOrigin" : function(node, obj) {
                var tOrigin = this.getChildValue(node);
                obj.tileOrigin = tOrigin.split(",");
            },
            "tileFullExtent" : function(node, obj) {
                var fullExtent = this.getChildValue(node);
                obj.tileFullExtent = fullExtent.split(",");
            },
            "requestType" : function(node, obj) {
                obj.requestType = this.getChildValue(node);
            },
            "tileMatrixSet" : function(node, obj) {
                obj.tileMatrixSet = this.getChildValue(node);
            },
            "parameter" : function(node, obj) {
                var pName = node.getAttribute("name");
                var pValue = node.getAttribute("value");
                obj.parameters[pName] = pValue;
            },
            "services" : function(node, obj) {
                obj.services = new oscar.ox.Services();
                this.readChildNodes(node, obj.services);
            },
            "service" : function(node, obj) {
                var serviceType = node.getAttribute("type");
                var service = null;
                switch (serviceType) {
                case "extraction":
                    service = obj.addExtractionService();
                    break;
                case "selection":
                    service = obj.addSelectionService();
                    break;
                }

                this.readChildNodes(node, service);
            },
            "serviceEntry" : function(node, obj) {
                var serviceEntry = new oscar.ox.ServiceEntry();
                serviceEntry.serviceType = node.getAttribute("type");
                serviceEntry.version = node.getAttribute("version");
                serviceEntry.format = node.getAttribute("format");
                this.readChildNodes(node, serviceEntry);
                obj.addServiceEntry(serviceEntry);
            },
            "identifiers" : function(node, obj) {

                this.readChildNodes(node, obj);
            },
            "identifier" : function(node, obj) {
                var identifier = this.getChildValue(node);
                obj.identifiers.push(identifier);
            }
        }
    },

    /**
     * Constant: CLASS_NAME
     */
    CLASS_NAME : "oscar.Format.OXF.XML"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.OGC.Filter
 * 
 * Formatter base class for OGC Filter documents.
 * 
 */
oscar.Format.OGC.Filter = {
    /**
     * Constant: CLASS_NAME - oscar.Format.OGC
     */
    CLASS_NAME : "oscar.Format.OGC.Filter"
};
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Format.OGC.Filter.v1_1_0 = {
    readers : OpenLayers.Util.applyDefaults({
        "ogc" : {
            "Filter_Capabilities" : function(obj, node) {
                obj["filterCapabilities"] = {};
                this.runChildNodes(obj["filterCapabilities"], node);
            },
            "Spatial_Capabilities" : function(obj, node) {
                obj["spatialCapabilities"] = {};
                this.runChildNodes(obj["spatialCapabilities"], node);
            },
            "GeometryOperands" : function(obj, node) {
                obj["geometryOperands"] = {};
                this.runChildNodes(obj["geometryOperands"], node);
            },
            "GeometryOperand" : function(obj, node) {
                if (!obj.operands) {
                    obj.operands = [];
                }
                obj.operands.push(this.getChildValue(node));

            },
            "SpatialOperators" : function(obj, node) {
                obj["spatialOperators"] = {};
                this.runChildNodes(obj["spatialOperators"], node);
            },
            "SpatialOperator" : function(obj, node) {
                if (!obj.operators) {
                    obj.operators = [];
                }
                var operator = {};
                operator.name = this.getAttributeNS(node, "", "name");
                obj.operators.push(operator);
            },
            "Scalar_Capabilities" : function(obj, node) {
                obj["scalarCapabilities"] = {};
                this.runChildNodes(obj["scalarCapabilities"], node);
            },
            "LogicalOperators" : function(obj, node) {
            },
            "LocicalOperator" : function(obj, node) {
            },
            "ComparisonOperators" : function(obj, node) {
                obj["comparisonOperators"] = {};
                this.runChildNodes(obj["comparisonOperators"], node);
            },
            "ComparisonOperator" : function(obj, node) {
                if (!obj.comparisonOperator) {
                    obj.comparisonOperator = [];
                }
                obj.comparisonOperator.push(this.getChildValue(node));
            },
            "ArithmeticOperators" : function(obj, node) {
            },
            "SimpleArithmetic" : function(obj, node) {
            },
            "Functions" : function(obj, node) {
            },
            "FunctionNames" : function(obj, node) {
            },
            "FunctionName" : function(obj, node) {
            },
            "Id_Capabilities" : function(ob, node) {
            },
        }
    }, oscar.Format.OGC.ows.v1_0_0.readers),
    /**
     * Constant: CLASS_NAME - oscar.Format.OGC,Filter.v1_1_0
     */
    CLASS_NAME : "oscar.Format.OGC.Filter.v1_1_0"
};
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.CSW
 * 
 * The base CSW format class.
 */

oscar.Format.CSW = new oscar.BaseClass(OpenLayers.Format.XML, {
    initialize : function(options) {
        OpenLayers.Format.prototype.initialize.apply(this, [ options ]);
        this.options = options;
    },

    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.XML.prototype.read.apply(this, [ data ]);
        }
        var root = data.documentElement;
        var version = this.version || root.getAttribute("version") || this.defaultVersion;
        var rootNodeName = root.nodeName;
        var constr = this.findConstructor(root.nodeName, version);
        if (!constr) {
            throw "Cannot find a CSW " + root.nodeName + " parser for version " + version;
        }
        var parser = new constr(this.options);
        var obj = parser.read(data);
        obj.version = version;
        return obj;
    },
    findConstructor : function(operationType, version) {
        try {
            return oscar.Format.CSW["v" + version.replace(/\./g, "_")][operationType];
        } catch (err) {
            return null;
        }

    },
    CLASS_NAME : "oscar.Format.CSW"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.CSW.v2_0_2
 */

oscar.Format.CSW.v2_0_2 = {};
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.CSW.v2_0_2.Capabilities
 * 
 * Formatter for reader the capabilities response for a Catalogue service.
 */
oscar.Format.CSW.v2_0_2.Capabilities = new oscar.BaseClass(OpenLayers.Format.XML, oscar.Format.OGC.ows.v1_0_0, oscar.Format.OGC.Filter.v1_1_0, {
    initialize : function() {
    },
    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.XML.prototype.apply(this, [ data ]);
        }
        var obj = {};
        var root = data.documentElement;
        this.runChildNodes(obj, root);
        return obj;
    },
    runChildNodes : function(obj, node) {
        var children = node.childNodes;
        var child, processor;
        for (var child=0;child<children.length;child++) {
            childNode = children[child];
            if (childNode.nodeType == 1) {
                processor = this.getProcessor(childNode);
                if (processor) {
                    processor.apply(this, [ obj, childNode ]);
                }
            }
        }
    },
    getProcessor : function(childNode) {
        var proc = null;
        var localName = (childNode.localName) ? childNode.localName : childNode.baseName;
        try {
            proc = this.readers[childNode.prefix][localName];
        } catch (err) {
            proc = this["read_cap_" + localName]
        }
        return proc || this["read_cap_" + localName];
    },
    CLASS_NAME : "oscar.Format.CSW.v2_0_2.Capabilities"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSCapabilities
 * 
 * Read WCS Capabilities file.
 * 
 * Inherits from: - <OpenLayers.Format.XML>
 */
oscar.Format.WCSCapabilities = oscar.BaseClass(OpenLayers.Format.XML, {

    /**
     * APIProperty: defaultVersion
     * 
     * {String} Version number to assume if none found. Default is "1.1.0".
     */
    defaultVersion : "1.1.0",

    /**
     * APIProperty: version
     * 
     * {String} Specify a version string if one is known.
     */
    version : null,

    /**
     * Property: parser {<OpenLayers.Format>} A cached versioned format used
     * for reading.
     */
    parser : null,

    /**
     * Constructor: oscar.Format.WMTSCapabilities
     * 
     * Create a new parser for WMTS capabilities.
     * 
     * Parameters:
     * 
     * options - {Object} An optional object whose properties will be set on
     * this instance.
     */
    initialize : function(options) {
        OpenLayers.Format.prototype.initialize.apply(this, [ options ]);
        this.options = options;
    },

    /**
     * APIMethod: read
     * 
     * Read capabilities data from a string, and return a list of layers.
     * 
     * Parameters: data - {String} or {DOMElement} data to read/parse.
     * 
     * Returns: capabilities - {Array} List of named layers.
     */
    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.XML.prototype.read.apply(this, [ data ]);
        }
        var root = data.documentElement;
        var version = this.version || root.getAttribute("version") || this.defaultVersion;
        if (!this.parser || this.parser.version !== version) {
            var constr = oscar.Format.WCSCapabilities["v" + version.replace(/\./g, "_")];
            if (!constr) {
                throw "Can't find a WCS capabilities parser for version " + version;
            }
            var parser = new constr(this.options);
        }
        var capabilities = parser.read(data);
        capabilities.version = version;
        return capabilities;
    },

    /**
     * Constant: CLASS_NAME - oscar.Format.WCSCapabilities
     */

    CLASS_NAME : "oscar.Format.WCSCapabilities"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSCapabilities.v1
 * 
 * Reads a WCS Capabilities document version 1
 * 
 * Inherits from:
 * 
 * <OpenLayers.Format.XML>
 * 
 */
oscar.Format.WCSCapabilities.v1 = oscar.BaseClass(OpenLayers.Format.XML, {
    /**
     * Constructor: oscar.Format.WCSCapabilities.v1
     * 
     * options - {Object} An optional object whose properties will be set on
     * this instance.
     */
    initialize : function(options) {
        OpenLayers.Format.XML.prototype.initialize.apply(this, [ options ]);
    },
    /**
     * Method: read
     */
    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.XML.prototype.read.apply(this, [ data ]);
        }
        var capabilities = {};
        var root = data.documentElement;
        this.runChildNodes(capabilities, root);
        return capabilities;
    },
    /**
     * Method: runChildNodes
     */
    runChildNodes : function(obj, node) {
        var children = node.childNodes;
        var childNode, processor;
        for (var i = 0; i < children.length; ++i) {
            childNode = children[i];
            if (childNode.nodeType == 1) {
                processor = this.getProcessor(childNode);
                if (processor) {
                    processor.apply(this, [ obj, childNode ]);
                }
            }
        }
    },
    /**
     * Method: getProcessor
     * 
     * Parameters: <XMLNode> Uses the node to obtain the correct method used to
     * read.
     */
    getProcessor : function(childNode) {
        processor = this["read_cap_" + childNode.nodeName.split(":").pop()];
        return processor;
    },

    /**
     * Constant: CLASS_NAME - oscar.Format.WCSCapabilities.v1
     */
    CLASS_NAME : "oscar.Format.WCSCapabilities.v1"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSCapabilities.v1_0_0
 * 
 * Reads a WCS Capabilities document version 1.0.0
 * 
 * This class will call methods from the <oscar.Format.OGC.ows> and
 * <oscar.Format.OGC.wcs> classes to keep capabilities objects consistent.
 * 
 * Inherits from:
 * 
 * <oscar.Format.WCSCapabilities.v1>
 * 
 */
oscar.Format.WCSCapabilities.v1_0_0 = oscar.BaseClass(

oscar.Format.WCSCapabilities.v1, oscar.Format.OGC.ows.v1_0_0, oscar.Format.OGC.wcs, {
    /**
     * Constructor: oscar.Format.WCSCapabilities.v1_0_0
     */
    initialize : function(options) {

    },
    /**
     * Method: nsRef
     * 
     * This method will call the 'methodName' provided by the 'namespace'
     * passing along the object and node. Allows for the code to use methods
     * already defined in their respective namespaces to read these nodes.
     */
    nsRef : function(namespace, methodName, obj, node) {
        // var processor = oscar.Format.OGC[namespace]["read_cap_" +
        // methodName];
        var processor = this["read_cap_" + methodName];
        if (processor)
            processor.apply(this, [ obj, node ]);
    },
    /**
     * Method: read_cap_Service
     * 
     * Reads the Service node.
     */
    read_cap_Service : function(obj, node) {
        this.nsRef("ows", "ServiceIdentification", obj, node);
    },
    /**
     * Method: read_cap_metadataLink
     * 
     * Reads the metadataLink node.
     */
    read_cap_metadataLink : function(obj, node) {
    },
    /**
     * Method: read_cap_description
     * 
     * Reads the description node.
     */
    read_cap_description : function(obj, node) {
        this.nsRef("ows", "Abstract", obj, node);
    },
    /**
     * Method: read_cap_name
     * 
     * Reads the name node.
     */
    read_cap_name : function(obj, node) {
        this.nsRef("ows", "ServiceType", obj, node);
    },
    /**
     * Method: read_cap_label
     * 
     * Reads the label node.
     */
    read_cap_label : function(obj, node) {
        this.nsRef("ows", "Title", obj, node);
    },
    /**
     * Method: read_cap_keywords
     * 
     * Reads the keywords node.
     */
    read_cap_keywords : function(obj, node) {
        this.nsRef("ows", "Keywords", obj, node);
    },
    /**
     * Method: read_cap_keyword
     * 
     * Reads the keyword node.
     */
    read_cap_keyword : function(obj, node) {
        this.nsRef("ows", "Keyword", obj, node);
    },
    /**
     * Method: read_cap_responsibleParty
     * 
     * Reads the responsibleParty node.
     */
    read_cap_responsibleParty : function(obj, node) {
        this.nsRef("ows", "ServiceProvider", obj, node);
    },
    /**
     * Method: read_cap_individualName
     * 
     * Reads the individualName node.
     */
    read_cap_individualName : function(obj, node) {
        this.nsRef("ows", "IndividualName", obj, node);
    },
    /**
     * Method: read_cap_positionName
     * 
     * Reads the positionName node.
     */
    read_cap_positionName : function(obj, node) {
        this.nsRef("ows", "PositionName", obj, node);
    },
    /**
     * Method: read_cap_organisationName
     * 
     * Reads the organisationName node.
     */
    read_cap_organisationName : function(obj, node) {
        this.nsRef("ows", "ProviderName", obj, node);
    },
    /**
     * Method: read_cap_contactInfo
     * 
     * Reads the contactInfo node.
     */
    read_cap_contactInfo : function(obj, node) {
        this.nsRef("ows", "ServiceContact", obj, node);
    },
    /**
     * Method: read_cap_phone
     * 
     * Reads the phone node.
     */
    read_cap_phone : function(obj, node) {
        this.nsRef("ows", "Phone", obj, node);
    },
    /**
     * Method: read_cap_address
     * 
     * Reads the address node.
     */
    read_cap_address : function(obj, node) {
        this.nsRef("ows", "Address", obj, node);
    },
    /**
     * Method: read_cap_city
     * 
     * Reads the city node.
     */
    read_cap_city : function(obj, node) {
        this.nsRef("ows", "City", obj, node);
    },
    /**
     * Method: read_cap_country
     * 
     * Reads the country node.
     */
    read_cap_country : function(obj, node) {
        this.nsRef("ows", "Country", obj, node);
    },
    /**
     * Method: read_cap_electronicMailAddress
     * 
     * Reads the electronicMailAddress node.
     */
    read_cap_electronicMailAddress : function(obj, node) {
        this.nsRef("ows", "ElectronicMailAddress", obj, node);
    },
    /**
     * Method: read_cap_onlineResource
     * 
     * Reads the onlineResource node.
     */
    read_cap_onlineResource : function(obj, node) {
        this.nsRef("ows", "OnlineResource", obj, node);
    },
    /**
     * Method: read_cap_fees
     * 
     * Reads the fees node.
     */
    read_cap_fees : function(obj, node) {
    },
    /**
     * Method: read_cap_accessConstraints
     * 
     * Reads the accessConstraints node.
     */
    read_cap_accessConstraints : function(obj, node) {
    },
    /**
     * Method: read_cap_Capability
     * 
     * Reads the Capability node.
     */
    read_cap_Capability : function(obj, node) {
        this.nsRef("ows", "OperationsMetadata", obj, node);
    },
    /**
     * Method: read_cap_Request
     * 
     * Reads the Request node.
     */
    read_cap_Request : function(obj, node) {
        this.runChildNodes(obj, node);
    },
    /**
     * Method: read_cap_GetCapabilities
     * 
     * Reads the GetCapabilities node.
     */
    read_cap_GetCapabilities : function(obj, node) {
        obj[node.nodeName] = {};
        this.runChildNodes(obj[node.nodeName], node);
    },
    /**
     * Method: read_cap_GetCoverage
     * 
     * Reads the GetCoverage node.
     */
    read_cap_GetCoverage : function(obj, node) {
        obj[node.nodeName] = {};
        this.runChildNodes(obj[node.nodeName], node);
    },

    /**
     * Method: read_cap_Get
     * 
     * Reads the Get node.
     */
    read_cap_Get : function(obj, node) {
        var temp = {};
        this.runChildNodes(temp, node);
        obj["Get"] = temp.OnlineResource;
    },
    /**
     * Method: read_cap_OnlineResource
     * 
     * Reads the OnlineResource node.
     */
    /**
     * read_cap_OnlineResource : function(obj, node) { this.nsRef("ows",
     * "OnlineResource", obj, node); },
     */
    /**
     * Method: read_cap_DescribeCoverage
     * 
     * Reads the DescribeCoverage node.
     */
    read_cap_DescribeCoverage : function(obj, node) {
        obj[node.nodeName] = {};
        this.runChildNodes(obj[node.nodeName], node);
    },
    /**
     * Method: read_cap_Post
     * 
     * Reads the Post node.
     */
    read_cap_Post : function(obj, node) {
        var temp = {};
        this.runChildNodes(temp, node);
        obj["Post"] = temp.OnlineResource;
    },
    /**
     * Method: read_cap_Exception
     * 
     * Reads the Exception node.
     */
    read_cap_Exception : function(obj, node) {
    },
    /**
     * Method: read_cap_Format
     * 
     * Reads the Format node.
     */
    read_cap_Format : function(obj, node) {
    },
    /**
     * Method: read_cap_ContentMetadata
     * 
     * Reads the ContentMetadata node.
     */
    read_cap_ContentMetadata : function(obj, node) {
        this.nsRef("wcs", "Contents", obj, node);
    },
    /**
     * Method: read_cap_CoverageOfferingBrief
     * 
     * Reads the CoverageOfferingBrief node.
     */
    read_cap_CoverageOfferingBrief : function(obj, node) {
        this.nsRef("wcs", "CoverageSummary", obj, node);
    },
    /**
     * Method: read_cap_lonLatEnvelope
     * 
     * Reads the lonLatEnvelope node.
     */
    read_cap_lonLatEnvelope : function(obj, node) {
        obj.wgs84BoundingBox = {};
        var bbox = [];
        this.runChildNodes(bbox, node);
        obj.wgs84BoundingBox.west = bbox[0];
        obj.wgs84BoundingBox.south = bbox[1];
        obj.wgs84BoundingBox.east = bbox[2];
        obj.wgs84BoundingBox.north = bbox[3];
    },
    /**
     * Method: read_cap_pos
     * 
     * Reads the pos node.
     */
    read_cap_pos : function(bbox, node) {
        var lonLat = this.getChildValue(node).split(" ");
        bbox.push(lonLat[0]);
        bbox.push(lonLat[1]);
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WCSCapabilities.v1_0_0
     */
    CLASS_NAME : "oscar.Format.WCSCapabilities.v1_0_0"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSCapabilities.v1_1_0
 * 
 * Reads a WCS Capabilities document version 1.1.0
 * 
 * Inherits from:
 * 
 * <oscar.Format.WCSCapabilities.v1>
 */
oscar.Format.WCSCapabilities.v1_1_0 = oscar.BaseClass(oscar.Format.WCSCapabilities.v1, oscar.Format.OGC.ows.v1_0_0, oscar.Format.OGC.wcs, {
    /**
     * Constructor: oscar.Format.WCSCapabilities.v1_1_0
     * 
     */
    initialize : function(options) {
    },
    /**
     * Method: getProcessor
     */
    getProcessor : function(childNode) {
        var prefix = childNode.prefix || "wcs";
        var localName = childNode.localName || childNode.nodeName.split(":").pop();
        var nsObj = this[prefix] || this;
        var processor = nsObj["read_cap_" + localName] || this["read_cap_" + localName];
        return processor;
    },

    /**
     * Constant: CLASS_NAME - oscar.Format.WCSCapabilities.v1_1_0
     */
    CLASS_NAME : "oscar.Format.WCSCapabilities.v1_1_0"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSCapabilities.v1_1_1
 * 
 * Reads a WCS Capabilities document version 1.1.1
 * 
 * Inherits from:
 * 
 * <oscar.Format.WCSCapabilities.v1_1_0>
 * 
 */
oscar.Format.WCSCapabilities.v1_1_1 = oscar.BaseClass(oscar.Format.WCSCapabilities.v1_1_0, oscar.Format.OGC.ows.v1_1_0, oscar.Format.OGC.wcs, {
    /**
     * Constructor: oscar.Format.WCSCapabilities.v1_1_1
     */
    initialize : function(options) {
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WCSCapabilities.v1_1_1
     */
    CLASS_NAME : "oscar.Format.WCSCapabilities.v1_1_1"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSCapabilities.v1_1_2
 * 
 * Reads a WCS Capabilities document version 1.1.2
 * 
 * Inherits from:
 * 
 * <oscar.Format.WCSCapabilities.v1_1_1>
 * 
 */
oscar.Format.WCSCapabilities.v1_1_2 = oscar.BaseClass(oscar.Format.WCSCapabilities.v1_1_1, oscar.Format.OGC.wcs, {
    /**
     * Constructor: oscar.Format.WCSCapabilities.v1_1_2
     */
    initialize : function(options) {
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WCSCapabilities.v1_1_2
     */
    CLASS_NAME : "oscar.Format.WCSCapabilities.v1_1_2"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSDescribeCoverage
 * 
 * Formatter base class for WCS DescribeCoverage documents.
 * 
 * Inherits from: - <OpenLayers.Format.XML>
 * 
 */
oscar.Format.WCSDescribeCoverage = oscar.BaseClass(OpenLayers.Format.XML, {
    /**
     * APIProperty: defaultVersion
     * 
     * The version number to use if no version parameter is supplied. Default is
     * 1.1.0.
     */
    defaultVersion : "1.1.0",
    /**
     * APIMethod: read
     * 
     * Read capabilities data from a string, and return a list of layers.
     * 
     * Parameters: data - {String} or {DOMElement} data to read/parse.
     * 
     * Returns: capabilities - {Array} List of named layers.
     */
    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.XML.prototype.read.apply(this, [ data ]);
        }
        var root = data.documentElement;
        var version = this.version;
        if (!version) {
            version = root.getAttribute("version");
            if (!version) {
                version = this.defaultVersion;
            }
        }
        var constr = oscar.Format.WCSDescribeCoverage["v" + version.replace(/\./g, "_")];
        if (!constr) {
            throw "Can't find a WCS DescribeCoverage parser for version " + version;
        }
        var parser = new constr(this.options);
        var capabilities = parser.read(data);
        capabilities.version = version;
        return capabilities;
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WCSDescribeCoverage
     */
    CLASS_NAME : "oscar.Format.WCSDescribeCoverage"
});

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSDescribeCoverage.v1
 * 
 * Reads a WCS DescribeCoverage document version 1.0
 * 
 * Inherits from:
 * 
 * <OpenLayers.Format.XML>
 */
oscar.Format.WCSDescribeCoverage.v1 = oscar.BaseClass(OpenLayers.Format.XML, {
    /**
     * Constructor: oscar.Format.WCSDescribeCoverage.v1
     */
    initialize : function(options) {
        OpenLayers.Format.XML.prototype.initialize.apply(this, [ options ]);
        this.options = options;
    },
    /**
     * Method:read
     * 
     */
    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.XML.prototype.read.apply(this, [ data ]);
        }
        var capabilities = {};
        var root = data.documentElement;
        this.runChildNodes(capabilities, root);
        return capabilities;
    },
    /**
     * Method: runChildNodes
     */
    runChildNodes : function(obj, node) {
        var children = node.childNodes;
        var childNode, processor;
        for (var i = 0; i < children.length; ++i) {
            childNode = children[i];
            if (childNode.nodeType == 1) {
                processor = this["read_cap_" + childNode.nodeName.split(":").pop()];
                if (processor) {
                    processor.apply(this, [ obj, childNode ]);
                }
            }
        }
    },

    /**
     * Constant: CLASS_NAME - oscar.Format.WCSDescribeCoverage.v1
     */
    CLASS_NAME : "oscar.Format.WCSDescribeCoverage.v1"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSDescribeCoverage.v1_0_0
 * 
 * Reads a WCS DescribeCoverage document version 1.0.0
 * 
 * Inherits from:
 * 
 * <oscar.Format.WCSDescribeCoverage.v1>
 */
oscar.Format.WCSDescribeCoverage.v1_0_0 = oscar.BaseClass(oscar.Format.WCSDescribeCoverage.v1, {
    /**
     * Constructor: oscar.Format.WCSDescribeCoverage.v1_0_0
     */
    initialize : function(options) {
    },

    /**
     * Method: read_cap_CoverageDescription
     * 
     * Reads the CoverageDescription node.
     */
    read_cap_CoverageOffering : function(obj, node) {
        if (!obj.coverageDescription)
            obj.coverageDescription = {};
        this.runChildNodes(obj.coverageDescription, node);
    },

    /**
     * Method: read_cap_Title
     * 
     * Reads the Title node.
     */
    read_cap_Title : function(obj, node) {
        obj.title = this.getChildValue(node);
    },
    /**
     * Method: read_cap_Abstract
     * 
     * Reads the Abstract node.
     */
    read_cap_Abstract : function(obj, node) {
        obj['abstract'] = this.getChildValue(node);
    },
    /**
     * Method: read_cap_Keywords
     * 
     * Reads the Keywords node.
     */
    read_cap_Keywords : function(obj, node) {
        obj.keywords = [];
        this.runChildNodes(obj.keywords, node);
    },
    /**
     * Method: read_cap_Keyword
     * 
     * Reads the keyword node.
     */
    read_cap_Keyword : function(obj, node) {
        var keyword = this.getChildValue(node);
        obj.push(keyword);
    },
    /**
     * Method: read_cap_Identifier
     * 
     * Reads the Identifier node.
     */
    read_cap_Identifier : function(obj, node) {
        obj.identifier = this.getChildValue(node);
    },
    /**
     * Method:read_cap_Domain
     * 
     * Reads the Domain node.
     */
    read_cap_Domain : function(coverDesc, node) {
        if (!coverDesc.domain) {
            coverDesc.domain = {};
        }

        this.runChildNodes(coverDesc.domain, node)
    },
    /**
     * Method:read_cap_SpatialDomain
     * 
     * Reads the SpatialDomain node.
     */
    read_cap_SpatialDomain : function(domain, node) {
        if (!domain.spatialDomain)
            domain.spatialDomain = {};

        this.runChildNodes(domain.spatialDomain, node);
    },
    /**
     * Method:read_cap_GridCRS
     * 
     * Reads the GridCRS node.
     */
    read_cap_GridCRS : function(spatialDomain, node) {
    },
    /**
     * Method:read_GridBaseCRS
     * 
     * Reads the GridBaseCRS node.
     */
    read_cap_GridBaseCRS : function(gridCRS, node) {
    },
    /**
     * Method: read_cap_GridType
     * 
     * Reads the GridType node.
     */
    read_cap_GridType : function(gridCRS, node) {
    },
    /**
     * Method: read_cap_GridOffsets
     * 
     * Reads the GridOffsets node.
     */
    read_cap_GridOffsets : function(gridCRS, node) {
    },
    /**
     * Method: read_cap_GridCS
     * 
     * Reads the GridCS node.
     */
    read_cap_GridCS : function(gridCRS, node) {
    },

    /**
     * Method: read_cap_AllowedValues
     * 
     * Reads the AllowedValues node.
     */
    read_cap_AllowedValues : function(obj, node) {
    },
    /**
     * Method: read_cap_WGS84BoundingBox
     * 
     * Reads the WG84BoundingBox node.
     */
    read_cap_WGS84BoundingBox : function(obj, node) {
        this.read_cap_BoundingBox(obj, node);
    },
    /**
     * Method: read_cap_BoundingBox
     * 
     * Reads the BoundingBox node.
     */
    read_cap_BoundingBox : function(obj, node) {
        if (!obj.boundingBoxes) {
            obj.boundingBoxes = [];
        }
        var bbox = {};
        var crs = this.getAttributeNS(node, "", "crs");
        bbox.crs = crs;

        this.runChildNodes(bbox, node);
        obj.boundingBoxes.push(bbox);
    },
    /**
     * Method: read_cap_LowerCorner
     * 
     * Reads the LowerCorner node.
     */
    read_cap_LowerCorner : function(obj, node) {
        var leftBottom = this.getChildValue(node).split(' ');
        obj.west = leftBottom[0];
        obj.south = leftBottom[1];
    },
    /**
     * Method: read_cap_UpperCorner
     * 
     * Reads the UpperCorner node.
     */
    read_cap_UpperCorner : function(obj, node) {
        var rightTop = this.getChildValue(node).split(' ');
        obj.east = rightTop[0];
        obj.north = rightTop[1];
    },
    /**
     * Method: read_cap_SupportedFormat
     * 
     * Reads the SupportedFormat node.
     */
    read_cap_SupportedFormat : function(contents, node) {
        if (!contents.supportedFormats)
            contents.supportedFormats = [];
        var format = this.getChildValue(node);
        contents.supportedFormats.push(format);
    },
    /**
     * Method: read_cap_SupportedCRS
     * 
     * Reads the SupportedCRS node.
     */
    read_cap_SupportedCRS : function(obj, node) {
        if (!obj.supportedCRS) {
            obj.supportedCRS = [];
        }
        var crs = this.getChildValue(node);
        obj.supportedCRS.push(crs);
    },
    /**
     * Method: read_cap_Range
     * 
     * Reads the Range node.
     */
    read_cap_Range : function(obj, node) {
        obj.range = {};
        this.runChildNodes(obj.range, node);
    },
    /**
     * Method: read_cap_Field
     * 
     * Reads the Field node.
     */
    read_cap_Field : function(obj, node) {
        if (!obj.fields)
            obj.fields = [];
        var field = {};
        this.runChildNodes(field, node);
        obj.fields.push(field);
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WCSDescribeCoverage.v1_0_0
     */
    CLASS_NAME : "oscar.Format.WCSDescribeCoverage.v1_0_0"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSDescribeCoverage.v1_1_0
 * 
 * Reads a WCS DescribeCoverage document version 1.1.0
 * 
 * Inherits from:
 * 
 * <oscar.Format.WCSDescribeCoverage.v1>
 */
oscar.Format.WCSDescribeCoverage.v1_1_0 = oscar.BaseClass(oscar.Format.WCSDescribeCoverage.v1, {
    /**
     * Constructor
     */
    initialize : function(options) {
    },

    /**
     * Method: read_cap_CoverageDescription
     * 
     * Reads the CoverageDescription node.
     */
    read_cap_CoverageDescription : function(obj, node) {
        if (!obj.coverageDescription)
            obj.coverageDescription = {};
        this.runChildNodes(obj.coverageDescription, node);
    },

    /**
     * Method: read_cap_Title
     * 
     * Reads the Title node.
     */
    read_cap_Title : function(obj, node) {
        obj.title = this.getChildValue(node);
    },
    /**
     * Method: read_cap_Abstract
     * 
     * Reads the Abstract node.
     */
    read_cap_Abstract : function(obj, node) {
        obj['abstract'] = this.getChildValue(node);
    },
    /**
     * Method: read_cap_Keywords
     * 
     * Reads the Keywords node.
     */
    read_cap_Keywords : function(obj, node) {
        obj.keywords = [];
        this.runChildNodes(obj.keywords, node);
    },
    /**
     * Method: read_cap_Keyword
     * 
     * Reads the keyword node.
     */
    read_cap_Keyword : function(obj, node) {
        var keyword = this.getChildValue(node);
        obj.push(keyword);
    },
    /**
     * Method: read_cap_Identifier
     * 
     * Reads the Identifier node.
     */
    read_cap_Identifier : function(obj, node) {
        obj.identifier = this.getChildValue(node);
    },
    /**
     * Method:read_cap_Domain
     * 
     * Reads the Domain node.
     */
    read_cap_Domain : function(coverDesc, node) {
        if (!coverDesc.domain) {
            coverDesc.domain = {};
        }

        this.runChildNodes(coverDesc.domain, node)
    },
    /**
     * Method:read_cap_SpatialDomain
     * 
     * Reads the SpatialDomain node.
     */
    read_cap_SpatialDomain : function(domain, node) {
        if (!domain.spatialDomain)
            domain.spatialDomain = {};

        this.runChildNodes(domain.spatialDomain, node);
    },
    /**
     * Method:read_cap_GridCRS
     * 
     * Reads the GridCRS node.
     */
    read_cap_GridCRS : function(spatialDomain, node) {
        if (!spatialDomain.gridCRS)
            spatialDomain.gridCRS = {};
        this.runChildNodes(spatialDomain.gridCRS, node);
    },
    /**
     * Method:read_GridBaseCRS
     * 
     * Reads the GridBaseCRS node.
     */
    read_cap_GridBaseCRS : function(gridCRS, node) {
        gridCRS.gridBaseCRS = this.getChildValue(node);
    },
    /**
     * Method: read_cap_GridType
     * 
     * Reads the GridType node.
     */
    read_cap_GridType : function(gridCRS, node) {
        gridCRS.gridType = this.getChildValue(node);
    },
    /**
     * Method: read_cap_GridOffsets
     * 
     * Reads the GridOffsets node.
     */
    read_cap_GridOffsets : function(gridCRS, node) {
        gridCRS.gridOffsets = this.getChildValue(node);
    },
    /**
     * Method: read_cap_GridCS
     * 
     * Reads the GridCS node.
     */
    read_cap_GridCS : function(gridCRS, node) {
        gridCRS.gridCS = this.getChildValue(node);
    },

    /**
     * Method: read_cap_GridOrigin
     * 
     * Reads the GridOrigin node.
     */
    read_cap_GridOrigin : function(gridCRS, node) {
        gridCRS.gridOrigin = this.getChildValue(node);
    },

    /**
     * Method: read_cap_AllowedValues
     * 
     * Reads the AllowedValues node.
     */
    read_cap_AllowedValues : function(obj, node) {
    },
    /**
     * Method: read_cap_WGS84BoundingBox
     * 
     * Reads the WG84BoundingBox node.
     */
    read_cap_WGS84BoundingBox : function(obj, node) {
        this.read_cap_BoundingBox(obj, node);
    },
    /**
     * Method: read_cap_BoundingBox
     * 
     * Reads the BoundingBox node.
     */
    read_cap_BoundingBox : function(obj, node) {
        if (!obj.boundingBoxes) {
            obj.boundingBoxes = [];
        }
        var bbox = {};
        var crs = this.getAttributeNS(node, "", "crs");
        bbox.crs = crs;

        this.runChildNodes(bbox, node);
        obj.boundingBoxes.push(bbox);
    },
    /**
     * Method: read_cap_LowerCorner
     * 
     * Reads the LowerCorner node.
     */
    read_cap_LowerCorner : function(obj, node) {
        var leftBottom = this.getChildValue(node).split(' ');
        obj.west = leftBottom[0];
        obj.south = leftBottom[1];
    },
    /**
     * Method: read_cap_UpperCorner
     * 
     * Reads the UpperCorner node.
     */
    read_cap_UpperCorner : function(obj, node) {
        var rightTop = this.getChildValue(node).split(' ');
        obj.east = rightTop[0];
        obj.north = rightTop[1];
    },
    /**
     * Method: read_cap_SupportedFormat
     * 
     * Reads the SupportedFormat node.
     */
    read_cap_SupportedFormat : function(contents, node) {
        if (!contents.supportedFormats)
            contents.supportedFormats = [];
        var format = this.getChildValue(node);
        contents.supportedFormats.push(format);
    },
    /**
     * Method: read_cap_SupportedCRS
     */
    read_cap_SupportedCRS : function(obj, node) {
        if (!obj.supportedCRS) {
            obj.supportedCRS = [];
        }
        var crs = this.getChildValue(node);
        obj.supportedCRS.push(crs);
    },
    /**
     * Method: read_cap_Range
     */
    read_cap_Range : function(obj, node) {
        obj.range = {};
        this.runChildNodes(obj.range, node);
    },
    /**
     * Method: read_cap_Field
     */
    read_cap_Field : function(obj, node) {
        if (!obj.fields)
            obj.fields = [];
        var field = {};
        this.runChildNodes(field, node);
        obj.fields.push(field);
    },
    read_cap_InterpolationMethods : function(obj, node) {
        if (!obj.interpolationMethods) {
            obj.interpolationMethods = {
                methods : [],
                defaultMethod : ""
            };
        }
        this.runChildNodes(obj.interpolationMethods, node);
    },
    read_cap_InterpolationMethod : function(obj, node) {

        var value = this.getChildValue(node);
        obj.methods.push(value);
    },
    read_cap_Default : function(obj, node) {
        obj.defaultMethod = this.getChildValue(node);
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WCSDescribeCoverage.v1_1_0
     */
    CLASS_NAME : "oscar.Format.WCSDescribeCoverage.v1_1_0"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSDescribeCoverage.v1_1_1
 * 
 * Reads a WCS DescribeCoverage document version 1.1.1
 * 
 * Inherits from:
 * 
 * <oscar.Format.WCSDescribeCoverage.v1_1_0>
 */
oscar.Format.WCSDescribeCoverage.v1_1_1 = oscar.BaseClass(oscar.Format.WCSDescribeCoverage.v1_1_0, {
    /**
     * Constructor
     */
    initialize : function(options) {
    },

    /**
     * Constant: CLASS_NAME - oscar.Format.WCSDescribeCoverage.v1_1_1
     */
    CLASS_NAME : "oscar.Format.WCSDescribeCoverage.v1_1_1"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WCSDescribeCoverage.v1_1_2
 * 
 * Reads a WCS DescribeCoverage document version 1.1.2
 * 
 * Inherits from:
 * 
 * <oscar.Format.WCSDescribeCoverage.v1_1_1>
 */
oscar.Format.WCSDescribeCoverage.v1_1_2 = oscar.BaseClass(oscar.Format.WCSDescribeCoverage.v1_1_1, {
    /**
     * Constructor
     */
    initialize : function(options) {
    },

    /**
     * Constant: CLASS_NAME - oscar.Format.WCSDescribeCoverage.v1_1_2
     */
    CLASS_NAME : "oscar.Format.WCSDescribeCoverage.v1_1_2"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WMTSCapabilities
 * 
 * Reads WMTS Capabilities document.
 * 
 * Inherits from: - <OpenLayers.Format.XML>
 */
oscar.Format.WMTSCapabilities = oscar.BaseClass(OpenLayers.Format.XML, {

    /**
     * APIProperty: defaultVersion
     * 
     * 
     * {String} Version number to assume if none found. Default is "1.0.0".
     */
    defaultVersion : "1.0.0",

    /**
     * APIProperty: version
     * 
     * 
     * {String} Specify a version string if one is known.
     */
    version : null,

    /**
     * Property: parser
     * 
     */
    parser : null,

    /**
     * Constructor: oscar.Format.WMTSCapabilities
     * 
     * Create a new parser for WMTS capabilities.
     * 
     * Parameters:
     * 
     * options - {Object} An optional object whose properties will be set on
     * this instance.
     */
    initialize : function(options) {
        OpenLayers.Format.prototype.initialize.apply(this, [ options ]);
        this.options = options;
    },

    /**
     * APIMethod: read
     * 
     * Read capabilities data from a string, and return a list of layers.
     * 
     * Parameters:
     * 
     * data - {String} or {DOMElement} data to read/parse.
     * 
     * Returns: capabilities - {Array} List of named layers.
     */
    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.XML.prototype.read.apply(this, [ data ]);
        }
        var root = data.documentElement;
        var version = this.version || root.getAttribute("version") || this.defaultVersion;
        if (!this.parser || this.parser.version !== version) {
            var constr = oscar.Format.WMTSCapabilities["v" + version.replace(/\./g, "_")];
            if (!constr) {
                throw "Can't find a WMTS capabilities parser for version " + version;
            }
            var parser = new constr(this.options);
        }
        var capabilities = parser.read(data);
        capabilities.version = version;
        return capabilities;
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WMTSCapabilities
     */
    CLASS_NAME : "oscar.Format.WMTSCapabilities"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WMTSCapabilities.v1
 * 
 * Read WMTS Capabilities document version 1.0
 * 
 * Inherits from: - <OpenLayers.Format.XML>
 */

oscar.Format.WMTSCapabilities.v1 = oscar.BaseClass(OpenLayers.Format.XML, {
    /**
     * Constructor: oscar.Format.WMTSCapabilities.v1
     */
    initialize : function(options) {
        OpenLayers.Format.XML.prototype.initialize.apply(this, [ options ]);
        this.options = options;
    },
    /**
     * Method: read
     */
    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.XML.prototype.read.apply(this, [ data ]);
        }
        var capabilities = {};
        var root = data.documentElement;
        this.runChildNodes(capabilities, root);
        return capabilities;
    },
    /**
     * Method: runChildNodes
     */
    runChildNodes : function(obj, node) {
        var children = node.childNodes;
        var childNode, processor;
        for (var i = 0; i < children.length; ++i) {
            childNode = children[i];
            if (childNode.nodeType == 1) {
                processor = this.getProcessor(childNode);
                if (processor) {
                    processor.apply(this, [ obj, childNode ]);
                }
            }
        }
    },
    /**
     * Method: getProcessor
     * 
     * Parameters:
     * 
     * childNode - <XMLNode> Uses the node to obtain the correct method used to
     * read.
     */
    getProcessor : function(childNode) {
        processor = this["read_cap_" + childNode.nodeName.split(":").pop()];
        return processor;
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WMTSCapabilities.v1
     */
    CLASS_NAME : "oscar.Format.WMTSCapabilities.v1"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WMTSCapabilities.v1_0_0
 * 
 * Read WMTS Capabilities version 1.0.0.
 * 
 * Inherits from: - <OpenLayers.Format.XML>
 */

oscar.Format.WMTSCapabilities.v1_0_0 = oscar.BaseClass(oscar.Format.WMTSCapabilities.v1, oscar.Format.OGC.ows.v1_1_0, {
    /**
     * Constructor: oscar.Format.WMTSCapabilities.v1_0_0
     */
    initialize : function(options) {
        oscar.Format.WMTSCapabilities.v1.prototype.initialize.apply(this, [ options ]);
        this.options = options;
    },
    /**
     * Method: getProcessor
     */
    getProcessor : function(childNode) {
        var local = childNode.nodeName.split(":").pop();
        var nsObj = this[childNode.prefix] || this;
        var processor = nsObj["read_cap_" + local] || this["read_cap_" + local];
        return processor;
    },
    /**
     * Method: read_cap_Contents
     * 
     * Reads the Contents node.
     */
    read_cap_Contents : function(capabilities, node) {
        capabilities.contents = {};
        this.runChildNodes(capabilities.contents, node);
    },
    /**
     * Method: read_cap_Layer
     * 
     * Reads the Layer node.
     */
    read_cap_Layer : function(contents, node) {
        if (!contents.layers)
            contents.layers = [];
        var layer = {};
        this.runChildNodes(layer, node);
        contents.layers.push(layer);
    },

    /**
     * Method: read_cap_Format
     * 
     * Reads the Format node.
     */
    read_cap_Format : function(obj, node) {
        obj.format = this.getChildValue(node);
    },
    /**
     * Method: read_cap_ResourceURL
     * 
     * Reads the ResourceURL ndoe.
     */
    read_cap_ResourceURL : function(obj, node) {
        if (!obj.resources)
            obj.resources = [];
        var resource = {};
        resource.format = this.getAttributeNS(node, "", "format");
        resource.resourceType = this.getAttributeNS(node, "", "resourceType");
        resource.template = this.getAttributeNS(node, "", "template");

        obj.resources.push(resource);
    },
    /**
     * Method: read_cap_Themes
     * 
     * Reads the Themes node.
     */
    read_cap_Themes : function(capabilities, node) {
        capabilities.themes = [];
        this.runChildNodes(capabilities.themes, node);
    },
    /**
     * Method: read_cap_Theme
     * 
     * Reads the Theme node.
     */
    read_cap_Theme : function(obj, node) {
        var theme = {};
        this.runChildNodes(theme, node);
        obj.push(theme);
    },
    /**
     * Method: read_cap_LayerRef
     * 
     * Reads the LayerRef node.
     */
    read_cap_LayerRef : function(obj, node) {
        if (!obj.layerRef)
            obj.layerRef = [];
        var val = this.getChildValue(node);
        obj.layerRef.push(val);
    },
    /**
     * Method: read_cap_TileMatrixSet
     * 
     * Reads the TileMatrixSet node.
     */
    read_cap_TileMatrixSet : function(contents, node) {
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WMTSCapabilities.v1_0_0
     */

    CLASS_NAME : "oscar.Format.WMTSCapabilities.v1_0_0"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WFSDescribeFeatureType
 * 
 * Formatter base class for WFS DescribeFeatureType.
 * 
 * Inherits from: - <OpenLayers.Format.XML>
 * 
 */
oscar.Format.WFSDescribeFeatureType = oscar.BaseClass(OpenLayers.Format.XML, {

    /**
     * APIProperty: defaultVersion
     * 
     * 
     * {String} Version number to assume if none found. Default is "1.0.0".
     */
    defaultVersion : "1.0.0",

    /**
     * APIProperty: version
     * 
     * 
     * {String} Specify a version string if one is known.
     */
    version : null,

    /**
     * Property: parser
     * 
     */
    parser : null,

    /**
     * Constructor: oscar.Format.WFSDescribeFeatureType
     * 
     * Parameters:
     * 
     * options - {Object} An optional object whose properties will be set on
     * this instance.
     */
    initialize : function(options) {
        OpenLayers.Format.prototype.initialize.apply(this, [ options ]);
        this.options = options;
    },

    /**
     * APIMethod: read
     * 
     * Read WFSDescribeFeatureType data from a string, and return a list of
     * nodes.
     * 
     * Parameters: data - {String} or {DOMElement} data to read/parse.
     * 
     * Returns: featureTypes - {Array} List of named nodes.
     */
    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.XML.prototype.read.apply(this, [ data ]);
        }
        var root = data.documentElement;
        var version = this.version;
        if (!version) {
            version = root.getAttribute("version");
            if (!version) {
                version = this.defaultVersion;
            }
        }
        var constr = oscar.Format.WFSDescribeFeatureType["v" + version.replace(/\./g, "_")];
        if (!constr) {
            throw "Can't find a WFSDescribeFeatureType parser for version " + version;
        }
        var parser = new constr(this.options);
        var featureTypes = parser.read(data);
        featureTypes.version = version;
        return featureTypes;
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WFSDescribeFeatureType
     */
    CLASS_NAME : "oscar.Format.WFSDescribeFeatureType"
});

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WFSDescribeFeatureType.v1
 * 
 * Reads OGC WFSDescribeFeatureType document version 1.0
 * 
 * Inherits from: - <OpenLayers.Format.XML>
 * 
 */
oscar.Format.WFSDescribeFeatureType.v1 = oscar.BaseClass(OpenLayers.Format.XML, {
    /**
     * Constructor
     */
    initialize : function(options) {
        OpenLayers.Format.XML.prototype.initialize.apply(this, [ options ]);
    },
    /**
     * Method: read
     * 
     */
    read : function(data) {
        if (typeof data == "string") {
            data = OpenLayers.Format.XML.prototype.read.apply(this, [ data ]);
        }
        var featureTypes = {};
        var root = data.documentElement;
        this.runChildNodes(featureTypes, root);
        return featureTypes;
    },
    /**
     * Method: runChildNodes
     */
    runChildNodes : function(obj, node) {
        var children = node.childNodes;
        var childNode, processor;
        for (var i = 0; i < children.length; ++i) {
            childNode = children[i];
            if (childNode.nodeType == 1) {
                processor = this.getProcessor(childNode);
                if (processor) {
                    processor.apply(this, [ obj, childNode ]);
                }
            }
        }
    },
    /**
     * Method: getProcessor
     * 
     * Parameters:
     * 
     * childNode - <XMLNode> Uses the node to obtain the correct method used to
     * read.
     */
    getProcessor : function(childNode) {
        processor = this["read_DesFeatureType_" + childNode.nodeName.split(":").pop()];
        return processor;
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WFSDescribeFeatureType.v1
     */
    CLASS_NAME : "oscar.Format.WFSDescribeFeatureType.v1"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WFSDescribeFeatureType.v1_0_0
 * 
 * Read WFS DescribeFeatureType version 1.0.0
 * 
 * Inherits from:
 * 
 * <OpenLayers.Format.XML>
 */

oscar.Format.WFSDescribeFeatureType.v1_0_0 = oscar.BaseClass(oscar.Format.WFSDescribeFeatureType.v1, {
    /**
     * Constructor: oscar.Format.WFSDescribeFeatureType.v1_0_0
     */
    initialize : function(options) {
        oscar.Format.WFSDescribeFeatureType.v1.prototype.initialize.apply(this, [ options ]);
        this.options = options;
    },
    /**
     * Method: getProcessor
     */
    getProcessor : function(childNode) {
        var local = childNode.nodeName.split(":").pop();
        var processor = this["read_DesFeatureType_" + local];
        return processor;
    },
    /**
     * Method: read_DesFeatureType_schema
     * 
     * Reads the schema node.
     */
    read_DesFeatureType_schema : function(FeatureTypes, node) {
        FeatureTypes.Schema = {};
        this.runChildNodes(FeatureTypes.Schema, node);
    },
    /**
     * Method: read_DesFeatureType_import
     * 
     * Reads the import node.
     */
    read_DesFeatureType_import : function(Schema, node) {
        Schema.Import = {};
        var importNode = {
            "namespace" : "",
            "schemaLocation" : ""
        };
        importNode.namespace = node.getAttribute("namespace");
        importNode.schemaLocation = node.getAttribute("schemaLocation");
        Schema.Import = importNode;
    },
    /**
     * Method: read_DesFeatureType_complexType
     * 
     * Reads the complexType node.
     */
    read_DesFeatureType_complexType : function(Schema, node) {
        if (!Schema.ComplexType)
            Schema.ComplexType = {};
        var complexType = {};
        this.runChildNodes(complexType, node);
        Schema.ComplexType = complexType;
    },
    /**
     * Method: read_DesFeatureType_complexContent
     * 
     * Reads the complexContent node.
     */
    read_DesFeatureType_complexContent : function(ComplexType, node) {
        if (!ComplexType.ComplexContent)
            ComplexType.ComplexContent = {};
        var complexContent = {};
        this.runChildNodes(complexContent, node);
        ComplexType.ComplexContent = complexContent;
    },
    /**
     * Method: read_DesFeatureType_extension
     * 
     * Reads the extension node.
     */
    read_DesFeatureType_extension : function(obj, node) {
        if (!obj.Extension)
            obj.Extension = {};
        var extension = {};
        this.runChildNodes(extension, node);
        obj.Extension = extension;
    },
    /**
     * Method: read_DesFeatureType_sequence
     * 
     * Reads the sequence node.
     */
    read_DesFeatureType_sequence : function(obj, node) {
        if (!obj.Sequence)
            obj.Sequence = {};
        var sequence = {};
        this.runChildNodes(sequence, node);
        obj.Sequence = sequence;
    },
    /**
     * Method: read_DesFeatureType_element
     * 
     * Reads the element node.
     */
    read_DesFeatureType_element : function(obj, node) {
        if (!obj.Elements)
            obj.Elements = [];
        var element = {
            "name" : "",
            "type" : ""
        };
        element.name = node.getAttribute("name");
        element.type = node.getAttribute("type");
        obj.Elements.push(element);
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WFSDescribeFeatureType.v1_0_0
     */

    CLASS_NAME : "oscar.Format.WFSDescribeFeatureType.v1_0_0"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Format.WFSDescribeFeatureType.v1_1_0
 * 
 * Read WFS DescribeFeatureType version 1.1.0
 * 
 * Inherits from: - <OpenLayers.Format.XML>
 */

oscar.Format.WFSDescribeFeatureType.v1_1_0 = oscar.BaseClass(oscar.Format.WFSDescribeFeatureType.v1, {
    /**
     * Constructor: oscar.Format.WFSDescribeFeatureType.v1_1_0
     */
    initialize : function(options) {
        oscar.Format.WFSDescribeFeatureType.v1.prototype.initialize.apply(this, [ options ]);
        this.options = options;
    },
    /**
     * Method: getProcessor
     */
    getProcessor : function(childNode) {
        var local = childNode.nodeName.split(":").pop();
        var processor = this["read_DesFeatureType_" + local];
        return processor;
    },
    /**
     * Method: read_DesFeatureType_schema
     * 
     * Reads the schema node.
     */
    read_DesFeatureType_schema : function(FeatureTypes, node) {
        FeatureTypes.Schema = {};
        this.runChildNodes(FeatureTypes.Schema, node);
    },
    /**
     * Method: read_DesFeatureType_import
     * 
     * Reads the import node.
     */
    read_DesFeatureType_import : function(Schema, node) {
        Schema.Import = {};
        var importNode = {
            "namespace" : "",
            "schemaLocation" : ""
        };
        importNode.namespace = node.getAttribute("namespace");
        importNode.schemaLocation = node.getAttribute("schemaLocation");
        Schema.Import = importNode;
    },
    /**
     * Method: read_DesFeatureType_complexType
     * 
     * Reads the complexType node.
     */
    read_DesFeatureType_complexType : function(Schema, node) {
        if (!Schema.ComplexType)
            Schema.ComplexType = {};
        var complexType = {};
        this.runChildNodes(complexType, node);
        Schema.ComplexType = complexType;
    },
    /**
     * Method: read_DesFeatureType_complexContent
     * 
     * Reads the complexContent node.
     */
    read_DesFeatureType_complexContent : function(ComplexType, node) {
        if (!ComplexType.ComplexContent)
            ComplexType.ComplexContent = {};
        var complexContent = {};
        this.runChildNodes(complexContent, node);
        ComplexType.ComplexContent = complexContent;
    },
    /**
     * Method: read_DesFeatureType_extension
     * 
     * Reads the extension node.
     */
    read_DesFeatureType_extension : function(obj, node) {
        if (!obj.Extension)
            obj.Extension = {};
        var extension = {};
        this.runChildNodes(extension, node);
        obj.Extension = extension;
    },
    /**
     * Method: read_DesFeatureType_sequence
     * 
     * Reads the sequence node.
     */
    read_DesFeatureType_sequence : function(obj, node) {
        if (!obj.Sequence)
            obj.Sequence = {};
        var sequence = {};
        this.runChildNodes(sequence, node);
        obj.Sequence = sequence;
    },
    /**
     * Method: read_DesFeatureType_element
     * 
     * Reads the element node.
     */
    read_DesFeatureType_element : function(obj, node) {
        if (!obj.Elements)
            obj.Elements = [];
        var element = {
            "name" : "",
            "type" : ""
        };
        element.name = node.getAttribute("name");
        element.type = node.getAttribute("type");
        obj.Elements.push(element);
    },
    /**
     * Constant: CLASS_NAME - oscar.Format.WFSDescribeFeatureType.v1_1_0
     */

    CLASS_NAME : "oscar.Format.WFSDescribeFeatureType.v1_1_0"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Util.SelectControl Class representing a Select Control for
 * moving selected/unselected options between two oscar.DivSelect containers.
 * 
 * @deprecated
 */
oscar.Util.SelectControl = oscar.BaseClass({
    available : null,
    selected : null,

    /**
     * Constructor.
     * 
     * @param avialableOptions
     *                oscar.DivSelect containing the available or unselected
     *                options.
     * @param selectedOptions
     *                oscar.DivSelect containing the selected options.
     */
    initialize : function(availableOptions, selectedOptions) {
        this.available = availableOptions;
        this.selected = selectedOptions;

    },

    /**
     * Moves options from the available oscar.DivSelect to the selected
     * oscar.DivSelect.
     * 
     * @param selectedOnly
     *                If true will only move options the user has selected. If
     *                false will move all the options.
     * @param showListName
     *                If true will show the listName as part of the option name
     *                when the option is moved to the selected DivSelect.
     * @param toListId
     *                The id of the div containing the oscar.DivSelect for
     *                selected options.
     */
    selectOptions : function(selectedOnly, showListName, toListId) {
        this.moveOptions(selectedOnly, showListName, this.available, this.selected, toListId, '');
    },

    /**
     * Moves options from the selected oscar.DivSelect to the available
     * oscar.DivSelect.
     * 
     * @param selectedOnly
     *                If true will only move options the user has selected. If
     *                false will move all the options.
     * @param listName
     *                The listName of the available list options.
     * @param toListId
     *                The id of the div containing the oscar.DivSelect for
     *                available options.
     * @param toListName
     *                The name of the currently active available list.
     */
    unSelectOptions : function(selectedOnly, toListId, toListName) {
        this.moveOptions(selectedOnly, false, this.selected, this.available, toListId, toListName);
    },

    /**
     * Generic function for moving an oscar.Util.DivSelectOption from one
     * oscar.DivSelect to another oscar.DivSelect.
     * 
     * @param selectedOnly
     *                If true will only move options the user has selected. If
     *                false will move all the options.
     * @param showListName
     * @param fromDiv
     *                The oscar.DivSelect the options will be removed from.
     * @param toDiv
     *                The oscar.DivSelect the options will be moved to.
     * @param toListId
     *                The id of the div containing the oscar.DivSelect options
     *                are being moved to.
     * @param toListName
     *                The name of the list options are being moved to.
     */
    moveOptions : function(selectedOnly, showListName, fromDiv, toDiv, toListId, toListName) {
        var options = null;
        if (selectedOnly === true) {
            options = fromDiv.getSelectedOptions();
        } else {
            options = fromDiv.getEnabledOptions();
        }

        for (var i = 0; i < options.length; i++) {
            var option = options[i];
            if (toListName === '' || toListName === option.listName) {
                toDiv.addOption(new oscar.Util.DivSelectOption(option.id, option.name, toListId, option.listName, showListName));
            }
        }

        if (selectedOnly === true) {
            fromDiv.removeSelectedOptions();
        } else {
            fromDiv.removeEnabledOptions();
        }
    },

    CLASS_NAME : "oscar.Util.SelectControl"
});

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Util.Database
 * 
 * This is a utility class to simulate a database.
 * 
 */

oscar.Util.Database = oscar.BaseClass({
    /**
     * APIProperty: events
     * 
     * {OpenLayers.Events)
     * 
     */
    events : null,
    /**
     * APIProperty: EVENT_TYPES {Array} - List of events this object will fire.
     */
    EVENT_TYPES : [ "dbupdated" ],
    /**
     * Property: tables {Array} - Contains the tables created.
     */
    tables : null,

    /**
     * Constructor: new oscar.Util.Database
     */
    initialize : function(options) {
        this.tables = [];
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : false
        });
    },
    /**
     * APIMethod: addTable
     * 
     * Adds a new table to the database object
     * 
     * Usage:
     * 
     * database.addTable('developers', ['index','name','email']);
     */
    addTable : function(tableName, columns) {
        this.tables[tableName] = {
            columns : columns,
            records : []
        };
    },

    /**
     * APIMethod: addRecord
     * 
     * Adds a record to a table
     * 
     * Parameters: - {String} tableName - {Array} records
     * 
     * Returns: - The number of records in the table.
     * 
     * Usage:
     * 
     * database.addRecord('developers',['0,'developer1',
     * 'developer1@caris.com']);
     * database.addRecord('developers',['1,'developer2',
     * 'developer2@caris.com']);
     */
    addRecord : function(tableName, data) {
        var table = this.tables[tableName];
        if (table) {
            table.records.push(data);
            this.events.triggerEvent("dbupdated");
            return table.records.length - 1;
        }
        return -1;
    },
    /**
     * APIMethod: search
     * 
     * Searchs a table based on a user defined query.
     * 
     * Parameters: - {String} tableName - {String} query - {Function} execute
     * 
     * Usage:
     * 
     * database.search('developers','developer1', function(table,query) { var
     * results = []; for(var r in table.records) { var record =
     * table.records[r]; if(record.name == query) { results.push(record); } }
     * return results;
     * 
     * });
     * 
     */
    search : function(tableName, query, execute) {
        var table = this.tables[tableName];
        return execute.call(this, table, query);

    },

    /**
     * Constant: CLASS_NAME
     */
    CLASS_NAME : "oscar.Util.Database"
});

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Util.Metadata
 * 
 * Methods to extract items from a capabilities document.
 */
oscar.Util.Metadata = {
    WFS : "WFS",
    WMS : "WMS",
    WMTS : "WMTS",
    WCS : "WCS",
    /**
     * APIMethod: getServiveTitle
     * 
     * Returns the title of a service
     * 
     * Parameter:
     * 
     * capabilities - {Object}
     */
    getServiceTitle : function(capabilities) {
        var service = this.getService(capabilities);
        var title = service.title || service.Title || service.serviceType || oscar.i18n("NotAvailable");
        if (title.length == 0) {
            title = oscar.i18n("NotAvailable");
        }
        if (typeof title == 'object') {
            var defaultTitle = title[OpenLayers.Lang.getCode()] || title['en'];
            title = defaultTitle;
        }
        return title;
    },
    /**
     * APIMethod: getServiceAbstract
     * 
     * Returns the abstract of a service
     * 
     * Parameter:
     * 
     * capabilities - {Object}
     */
    getServiceAbstract : function(capabilities) {
        var serviceAbstract = null;
        if (capabilities.serviceIdentification) {
            serviceAbstract = capabilities.serviceIdentification['abstract'] || oscar.i18n("NotAvailable");
        } else {
            serviceAbstract = capabilities.service['abstract'] || oscar.i18n("NotAvailable");
        }
        return serviceAbstract;
    },
    /**
     * APIMethod: getServiceKeywords
     * 
     * Returns the keywords of a service
     * 
     * Parameter:
     * 
     * capabilities - {Object}
     */
    getServiceKeywords : function(capabilities) {
        var serviceKeywords = null;
        if (capabilities.serviceIdentification) {
            serviceKeywords = capabilities.serviceIdentification.keywords || oscar.i18n("NotAvailable");
        }
        return serviceKeywords;

    },
    /**
     * APIMethod: getServiceType
     * 
     * Returns the type of service
     * 
     * Parameter:
     * 
     * capabilities - {Object}
     */
    getServiceType : function(capabilities) {
        var service = this.getService(capabilities);
        var serviceType = service.name || service.serviceType.value || service.serviceType;
        if (serviceType.indexOf("WMS") != -1) {
            return "WMS"
        } else if (serviceType.indexOf("WFS") != -1) {
            return "WFS";
        } else if (serviceType.indexOf("WMTS") != -1) {
            return "WMTS";
        } else if (serviceType.indexOf("WCS") != -1) {
            return "WCS";
        } else
            return serviceType;

    },
    /**
     * APIMethod: getService
     * 
     * Returns the service
     * 
     * Parameter:
     * 
     * capabilities - {Object}
     */
    getService : function(capabilities) {
        return capabilities.service || capabilities.serviceIdentification;
    },
    /**
     * APIMethod: getLayers
     * 
     * Parameter:
     * 
     * capabilities Returns the layers of a service
     */
    getLayers : function(capabilities) {
        try {
            var layers = capabilities.capability.layers;
        } catch (err) {
        }
        return layers;
    },
    /**
     * APIMethod: getFeatureTypes
     * 
     * Returns the feature types of a service
     * 
     * Parameter:
     * 
     * capabilities
     */
    getFeatureTypes : function(capabilities) {
        return capabilities.featureTypeList.featureTypes;

    },
    /**
     * APIMethod: getFeatureTypesById
     * 
     * Returns the feature types of a service with a specific id
     * 
     * Parameter:
     * 
     * capabilities id
     */
    getFeatureTypesById : function(capabilities, id) {
        for ( var f in capabilities.featureTypeList.featureTypes) {
            var feature = capabilities.featureTypeList.featureTypes[f];
            if (feature.name == id) {
                return feature;
            }
        }
        return null;

    },

    /**
     * APIMethod: getCoverages
     * 
     * Returns the coverages node of a wcs service
     * 
     * Parameter:
     * 
     * capabilities - {Object}
     * 
     */

    getCoverages : function(capabilities) {
        return capabilities.contents.coverages;
    },

    /**
     * APIMethod: getParameters
     * 
     * Parameters: capabilities - Capabilities object. operationString - Name of
     * the operation string, i.e. "GetFeature". parameterName - Name of the
     * parameter, i.e. "outputFormat".
     */
    getParameters : function(capabilities, operationString, parameterName) {
        var operation = null;
        if (capabilities.operationsMetadata) {
            operation = capabilities.operationsMetadata[operationString];
        } else if (capabilities.capability.request) {
            operation = capabilities.capability.request[operationString] || capabilities.capability.request[operationString.toLowerCase()];
        }
        if (operation.parameters) { // ows 1.1.0 or greater
            for ( var prop in operation.parameters) {
                if ($$.inArray(prop, parameterName) != -1) {
                    return operation.parameters[prop];
                }
            }
            for (var i = 0; i < operation.parameters.length; i++) {
                var op = operation.parameters[i];
                for (var j = 0; j < parameterName.length; j++) {
                    if (op.name.toLowerCase() == parameterName[j].toLowerCase()) {
                        return op.values;
                    }
                }
                if (operation.parameters[i].name == parameterName) {
                    return operation.parameters[i].values;
                }
            }
        } else { // ows 1.0.0 support
            for ( var p in operation) {
                for (var i = 0; i < parameterName.length; i++) {
                    if (p.toLowerCase() == parameterName[i].toLowerCase()) {
                        return operation[p];
                    }
                }
            }
        }

        return [];
    },

    /**
     * 
     * 
     * 
     */
    getIdentifierMetadataUrls : function(identifier) {
        var metadataUrls = identifier.metadataURLs || identifier.metadataUrls || [];
        return metadataUrls;
    },
    /**
     * APIMethod: getContent
     * 
     * Returns the content of a service
     * 
     * Parameter:
     * 
     * capabilities - {Object}
     */
    getContent : function(capabilities) {
        return capabilities.contents.layers;
    },
    /**
     * APIMethod: getThemes
     * 
     * Returns the Themes object of a WMTS Service
     * 
     * Parameter:
     * 
     * capabilities - {Object}
     */
    getThemes : function(capabilities) {
        return capabilities.themes;
    },

    /**
     * APIMethod: getOperation
     * 
     * Returns a service operation
     * 
     * Parameters:
     * 
     * capabilities - {object} Capabilities Object
     * 
     * operation - {String} Name of the Operation (GetFeature, GetMap, GetTile,
     * etc.)
     */
    getOperation : function(capabilities, operation) {
        if (capabilities.operationsMetadata) {
            return capabilities.operationsMetadata[operation];
        } else if (capabilities.capability.request) {
            operation = operation.toLowerCase();
            return capabilities.capability.request[operation];
        }
        return null;
    },
    /**
     * Method: getServiceHref
     * 
     * @deprecated - Use getOperationHref
     */

    getServiceHref : function(serviceType, capabilities) {
        return oscar.Util.Metadata.getOperationHref(serviceType, capabilities);
    },
    /**
     * APIMethod: getOperationHref
     * 
     * Returns the url of the service operations
     * 
     * Return Type: {String}
     * 
     * Parameters:
     * 
     * capabilities - {Object}
     * 
     * operation - {String} Name of the Operation (GetFeature, GetMap, GetTile,
     * etc.)
     * 
     */
    getOperationHref : function(capabilities, operation) {
        var op = oscar.Util.Metadata.getOperation(capabilities, operation);
        if (!op)
            return "";
        if (op.href) {
            if (typeof op.href == "object") {
                return op.href["get"];
            } else {
                return op.href;
            }
        } else {
            try {
                return op.dcp.http["get"];
            } catch (err) {
                return "";
            }
        }
    },
    /**
     * APIMethod: getContactInformation
     * 
     * Parameter:
     * 
     * capabilities - {Object}
     */
    getContactInformation : function(capabilities) {
        var contactInformation = null;
        var find = function(obj) {
            var keyArray = [ "serviceProvider", "contactInformation" ];
            var reference = null;
            for ( var prop in obj) {
                for (var i = 0; i < keyArray.length; i++) {
                    if (prop == keyArray[i])
                        return reference = obj[prop];
                }
                if (typeof obj[prop] == "object") {
                    reference = find(obj[prop]);
                    if (reference != null)
                        break;
                }
            }
            return reference;
        };
        var contactInformation = find(capabilities, "serviceProvider");
        if (capabilities.serviceProvider) {
            contactInformation = {};
            OpenLayers.Util.extend(contactInformation, capabilities.serviceProvider);
        }
        return contactInformation;
    },
    /**
     * Constant CLASS_NAME
     */
    CLASS_NAME : "oscar.Util.Metadata"
};
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Util.CoordinateReferences
 * 
 * This class is a static class to obtain a coordinate object by passing in
 * either an EPSG code or a URN value.
 */
oscar.Util.CoordinateReferences = {
    /**
     * APIProperty: coords
     * 
     * Array<Object> Each coordinate item contains a description of the
     * coordinate system and it's EPSG code.
     * 
     */
    coords : [ {
        "description" : "HD1909",
        "code" : "EPSG:3819"
    }, {
        "description" : "TWD67",
        "code" : "EPSG:3821"
    }, {
        "description" : "TWD97",
        "code" : "EPSG:3824"
    }, {
        "description" : "IGRS",
        "code" : "EPSG:3889"
    }, {
        "description" : "MGI 1901",
        "code" : "EPSG:3906"
    }, {
        "description" : "Unknown datum based upon the Airy 1830 ellipsoid",
        "code" : "EPSG:4001"
    }, {
        "description" : "Unknown datum based upon the Airy Modified 1849 ellipsoid",
        "code" : "EPSG:4002"
    }, {
        "description" : "Unknown datum based upon the Australian National Spheroid",
        "code" : "EPSG:4003"
    }, {
        "description" : "Unknown datum based upon the Bessel 1841 ellipsoid",
        "code" : "EPSG:4004"
    }, {
        "description" : "Unknown datum based upon the Bessel Modified ellipsoid",
        "code" : "EPSG:4005"
    }, {
        "description" : "Unknown datum based upon the Bessel Namibia ellipsoid",
        "code" : "EPSG:4006"
    }, {
        "description" : "Unknown datum based upon the Clarke 1858 ellipsoid",
        "code" : "EPSG:4007"
    }, {
        "description" : "Unknown datum based upon the Clarke 1866 ellipsoid",
        "code" : "EPSG:4008"
    }, {
        "description" : "Unknown datum based upon the Clarke 1866 Michigan ellipsoid",
        "code" : "EPSG:4009"
    }, {
        "description" : "Unknown datum based upon the Clarke 1880 (Benoit) ellipsoid",
        "code" : "EPSG:4010"
    }, {
        "description" : "Unknown datum based upon the Clarke 1880 (IGN) ellipsoid",
        "code" : "EPSG:4011"
    }, {
        "description" : "Unknown datum based upon the Clarke 1880 (RGS) ellipsoid",
        "code" : "EPSG:4012"
    }, {
        "description" : "Unknown datum based upon the Clarke 1880 (Arc) ellipsoid",
        "code" : "EPSG:4013"
    }, {
        "description" : "Unknown datum based upon the Clarke 1880 (SGA 1922) ellipsoid",
        "code" : "EPSG:4014"
    }, {
        "description" : "Unknown datum based upon the Everest 1830 (1937 Adjustment) ellipsoid",
        "code" : "EPSG:4015"
    }, {
        "description" : "Unknown datum based upon the Everest 1830 (1967 Definition) ellipsoid",
        "code" : "EPSG:4016"
    }, {
        "description" : "Unknown datum based upon the Everest 1830 Modified ellipsoid",
        "code" : "EPSG:4018"
    }, {
        "description" : "Unknown datum based upon the GRS 1980 ellipsoid",
        "code" : "EPSG:4019"
    }, {
        "description" : "Unknown datum based upon the Helmert 1906 ellipsoid",
        "code" : "EPSG:4020"
    }, {
        "description" : "Unknown datum based upon the Indonesian National Spheroid",
        "code" : "EPSG:4021"
    }, {
        "description" : "Unknown datum based upon the International 1924 ellipsoid",
        "code" : "EPSG:4022"
    }, {
        "description" : "MOLDREF99",
        "code" : "EPSG:4023"
    }, {
        "description" : "Unknown datum based upon the Krassowsky 1940 ellipsoid",
        "code" : "EPSG:4024"
    }, {
        "description" : "Unknown datum based upon the NWL 9D ellipsoid",
        "code" : "EPSG:4025"
    }, {
        "description" : "Unknown datum based upon the Plessis 1817 ellipsoid",
        "code" : "EPSG:4027"
    }, {
        "description" : "Unknown datum based upon the Struve 1860 ellipsoid",
        "code" : "EPSG:4028"
    }, {
        "description" : "Unknown datum based upon the War Office ellipsoid",
        "code" : "EPSG:4029"
    }, {
        "description" : "Unknown datum based upon the WGS 84 ellipsoid",
        "code" : "EPSG:4030"
    }, {
        "description" : "Unknown datum based upon the GEM 10C ellipsoid",
        "code" : "EPSG:4031"
    }, {
        "description" : "Unknown datum based upon the OSU86F ellipsoid",
        "code" : "EPSG:4032"
    }, {
        "description" : "Unknown datum based upon the OSU91A ellipsoid",
        "code" : "EPSG:4033"
    }, {
        "description" : "Unknown datum based upon the Clarke 1880 ellipsoid",
        "code" : "EPSG:4034"
    }, {
        "description" : "Unknown datum based upon the Authalic Sphere",
        "code" : "EPSG:4035"
    }, {
        "description" : "Unknown datum based upon the GRS 1967 ellipsoid",
        "code" : "EPSG:4036"
    }, {
        "description" : "Unknown datum based upon the Average Terrestrial System 1977 ellipsoid",
        "code" : "EPSG:4041"
    }, {
        "description" : "Unknown datum based upon the Everest (1830 Definition) ellipsoid",
        "code" : "EPSG:4042"
    }, {
        "description" : "Unknown datum based upon the WGS 72 ellipsoid",
        "code" : "EPSG:4043"
    }, {
        "description" : "Unknown datum based upon the Everest 1830 (1962 Definition) ellipsoid",
        "code" : "EPSG:4044"
    }, {
        "description" : "Unknown datum based upon the Everest 1830 (1975 Definition) ellipsoid",
        "code" : "EPSG:4045"
    }, {
        "description" : "RGRDC 2005",
        "code" : "EPSG:4046"
    }, {
        "description" : "Unspecified datum based upon the GRS 1980 Authalic Sphere",
        "code" : "EPSG:4047"
    }, {
        "description" : "Unspecified datum based upon the Clarke 1866 Authalic Sphere",
        "code" : "EPSG:4052"
    }, {
        "description" : "Unspecified datum based upon the International 1924 Authalic Sphere",
        "code" : "EPSG:4053"
    }, {
        "description" : "Unspecified datum based upon the Hughes 1980 ellipsoid",
        "code" : "EPSG:4054"
    }, {
        "description" : "Popular Visualisation CRS",
        "code" : "EPSG:4055"
    }, {
        "description" : "SREF98",
        "code" : "EPSG:4075"
    }, {
        "description" : "REGCAN95",
        "code" : "EPSG:4081"
    }, {
        "description" : "Greek",
        "code" : "EPSG:4120"
    }, {
        "description" : "GGRS87",
        "code" : "EPSG:4121"
    }, {
        "description" : "ATS77",
        "code" : "EPSG:4122"
    }, {
        "description" : "KKJ",
        "code" : "EPSG:4123"
    }, {
        "description" : "RT90",
        "code" : "EPSG:4124"
    }, {
        "description" : "Samboja",
        "code" : "EPSG:4125"
    }, {
        "description" : "LKS94 (ETRS89)",
        "code" : "EPSG:4126"
    }, {
        "description" : "Tete",
        "code" : "EPSG:4127"
    }, {
        "description" : "Madzansua",
        "code" : "EPSG:4128"
    }, {
        "description" : "Observatario",
        "code" : "EPSG:4129"
    }, {
        "description" : "Moznet",
        "code" : "EPSG:4130"
    }, {
        "description" : "Indian 1960",
        "code" : "EPSG:4131"
    }, {
        "description" : "FD58",
        "code" : "EPSG:4132"
    }, {
        "description" : "EST92",
        "code" : "EPSG:4133"
    }, {
        "description" : "PSD93",
        "code" : "EPSG:4134"
    }, {
        "description" : "Old Hawaiian",
        "code" : "EPSG:4135"
    }, {
        "description" : "St. Lawrence Island",
        "code" : "EPSG:4136"
    }, {
        "description" : "St. Paul Island",
        "code" : "EPSG:4137"
    }, {
        "description" : "St. George Island",
        "code" : "EPSG:4138"
    }, {
        "description" : "Puerto Rico",
        "code" : "EPSG:4139"
    }, {
        "description" : "NAD83(CSRS98)",
        "code" : "EPSG:4140"
    }, {
        "description" : "Israel",
        "code" : "EPSG:4141"
    }, {
        "description" : "Locodjo 1965",
        "code" : "EPSG:4142"
    }, {
        "description" : "Abidjan 1987",
        "code" : "EPSG:4143"
    }, {
        "description" : "Kalianpur 1937",
        "code" : "EPSG:4144"
    }, {
        "description" : "Kalianpur 1962",
        "code" : "EPSG:4145"
    }, {
        "description" : "Kalianpur 1975",
        "code" : "EPSG:4146"
    }, {
        "description" : "Hanoi 1972",
        "code" : "EPSG:4147"
    }, {
        "description" : "Hartebeesthoek94",
        "code" : "EPSG:4148"
    }, {
        "description" : "CH1903",
        "code" : "EPSG:4149"
    }, {
        "description" : "CH1903+",
        "code" : "EPSG:4150"
    }, {
        "description" : "CHTRF95",
        "code" : "EPSG:4151"
    }, {
        "description" : "NAD83(HARN)",
        "code" : "EPSG:4152"
    }, {
        "description" : "Rassadiran",
        "code" : "EPSG:4153"
    }, {
        "description" : "ED50(ED77)",
        "code" : "EPSG:4154"
    }, {
        "description" : "Dabola 1981",
        "code" : "EPSG:4155"
    }, {
        "description" : "S-JTSK",
        "code" : "EPSG:4156"
    }, {
        "description" : "Mount Dillon",
        "code" : "EPSG:4157"
    }, {
        "description" : "Naparima 1955",
        "code" : "EPSG:4158"
    }, {
        "description" : "ELD79",
        "code" : "EPSG:4159"
    }, {
        "description" : "Chos Malal 1914",
        "code" : "EPSG:4160"
    }, {
        "description" : "Pampa del Castillo",
        "code" : "EPSG:4161"
    }, {
        "description" : "Korean 1985",
        "code" : "EPSG:4162"
    }, {
        "description" : "Yemen NGN96",
        "code" : "EPSG:4163"
    }, {
        "description" : "South Yemen",
        "code" : "EPSG:4164"
    }, {
        "description" : "Bissau",
        "code" : "EPSG:4165"
    }, {
        "description" : "Korean 1995",
        "code" : "EPSG:4166"
    }, {
        "description" : "NZGD2000",
        "code" : "EPSG:4167"
    }, {
        "description" : "Accra",
        "code" : "EPSG:4168"
    }, {
        "description" : "American Samoa 1962",
        "code" : "EPSG:4169"
    }, {
        "description" : "SIRGAS 1995",
        "code" : "EPSG:4170"
    }, {
        "description" : "RGF93",
        "code" : "EPSG:4171"
    }, {
        "description" : "POSGAR",
        "code" : "EPSG:4172"
    }, {
        "description" : "IRENET95",
        "code" : "EPSG:4173"
    }, {
        "description" : "Sierra Leone 1924",
        "code" : "EPSG:4174"
    }, {
        "description" : "Sierra Leone 1968",
        "code" : "EPSG:4175"
    }, {
        "description" : "Australian Antarctic",
        "code" : "EPSG:4176"
    }, {
        "description" : "Pulkovo 1942(83)",
        "code" : "EPSG:4178"
    }, {
        "description" : "Pulkovo 1942(58)",
        "code" : "EPSG:4179"
    }, {
        "description" : "EST97",
        "code" : "EPSG:4180"
    }, {
        "description" : "Luxembourg 1930",
        "code" : "EPSG:4181"
    }, {
        "description" : "Azores Occidental 1939",
        "code" : "EPSG:4182"
    }, {
        "description" : "Azores Central 1948",
        "code" : "EPSG:4183"
    }, {
        "description" : "Azores Oriental 1940",
        "code" : "EPSG:4184"
    }, {
        "description" : "Madeira 1936",
        "code" : "EPSG:4185"
    }, {
        "description" : "OSNI 1952",
        "code" : "EPSG:4188"
    }, {
        "description" : "REGVEN",
        "code" : "EPSG:4189"
    }, {
        "description" : "POSGAR 98",
        "code" : "EPSG:4190"
    }, {
        "description" : "Albanian 1987",
        "code" : "EPSG:4191"
    }, {
        "description" : "Douala 1948",
        "code" : "EPSG:4192"
    }, {
        "description" : "Manoca 1962",
        "code" : "EPSG:4193"
    }, {
        "description" : "Qornoq 1927",
        "code" : "EPSG:4194"
    }, {
        "description" : "Scoresbysund 1952",
        "code" : "EPSG:4195"
    }, {
        "description" : "Ammassalik 1958",
        "code" : "EPSG:4196"
    }, {
        "description" : "Garoua",
        "code" : "EPSG:4197"
    }, {
        "description" : "Kousseri",
        "code" : "EPSG:4198"
    }, {
        "description" : "Egypt 1930",
        "code" : "EPSG:4199"
    }, {
        "description" : "Pulkovo 1995",
        "code" : "EPSG:4200"
    }, {
        "description" : "Adindan",
        "code" : "EPSG:4201"
    }, {
        "description" : "AGD66",
        "code" : "EPSG:4202"
    }, {
        "description" : "AGD84",
        "code" : "EPSG:4203"
    }, {
        "description" : "Ain el Abd",
        "code" : "EPSG:4204"
    }, {
        "description" : "Afgooye",
        "code" : "EPSG:4205"
    }, {
        "description" : "Agadez",
        "code" : "EPSG:4206"
    }, {
        "description" : "Lisbon",
        "code" : "EPSG:4207"
    }, {
        "description" : "Aratu",
        "code" : "EPSG:4208"
    }, {
        "description" : "Arc 1950",
        "code" : "EPSG:4209"
    }, {
        "description" : "Arc 1960",
        "code" : "EPSG:4210"
    }, {
        "description" : "Batavia",
        "code" : "EPSG:4211"
    }, {
        "description" : "Barbados 1938",
        "code" : "EPSG:4212"
    }, {
        "description" : "Beduaram",
        "code" : "EPSG:4213"
    }, {
        "description" : "Beijing 1954",
        "code" : "EPSG:4214"
    }, {
        "description" : "Belge 1950",
        "code" : "EPSG:4215"
    }, {
        "description" : "Bermuda 1957",
        "code" : "EPSG:4216"
    }, {
        "description" : "Bogota 1975",
        "code" : "EPSG:4218"
    }, {
        "description" : "Bukit Rimpah",
        "code" : "EPSG:4219"
    }, {
        "description" : "Camacupa",
        "code" : "EPSG:4220"
    }, {
        "description" : "Campo Inchauspe",
        "code" : "EPSG:4221"
    }, {
        "description" : "Cape",
        "code" : "EPSG:4222"
    }, {
        "description" : "Carthage",
        "code" : "EPSG:4223"
    }, {
        "description" : "Chua",
        "code" : "EPSG:4224"
    }, {
        "description" : "Corrego Alegre",
        "code" : "EPSG:4225"
    }, {
        "description" : "Cote d'Ivoire",
        "code" : "EPSG:4226"
    }, {
        "description" : "Deir ez Zor",
        "code" : "EPSG:4227"
    }, {
        "description" : "Douala",
        "code" : "EPSG:4228"
    }, {
        "description" : "Egypt 1907",
        "code" : "EPSG:4229"
    }, {
        "description" : "ED50",
        "code" : "EPSG:4230"
    }, {
        "description" : "ED87",
        "code" : "EPSG:4231"
    }, {
        "description" : "Fahud",
        "code" : "EPSG:4232"
    }, {
        "description" : "Gandajika 1970",
        "code" : "EPSG:4233"
    }, {
        "description" : "Garoua",
        "code" : "EPSG:4234"
    }, {
        "description" : "Guyane Francaise",
        "code" : "EPSG:4235"
    }, {
        "description" : "Hu Tzu Shan 1950",
        "code" : "EPSG:4236"
    }, {
        "description" : "HD72",
        "code" : "EPSG:4237"
    }, {
        "description" : "ID74",
        "code" : "EPSG:4238"
    }, {
        "description" : "Indian 1954",
        "code" : "EPSG:4239"
    }, {
        "description" : "Indian 1975",
        "code" : "EPSG:4240"
    }, {
        "description" : "Jamaica 1875",
        "code" : "EPSG:4241"
    }, {
        "description" : "JAD69",
        "code" : "EPSG:4242"
    }, {
        "description" : "Kalianpur 1880",
        "code" : "EPSG:4243"
    }, {
        "description" : "Kandawala",
        "code" : "EPSG:4244"
    }, {
        "description" : "Kertau 1968",
        "code" : "EPSG:4245"
    }, {
        "description" : "KOC",
        "code" : "EPSG:4246"
    }, {
        "description" : "La Canoa",
        "code" : "EPSG:4247"
    }, {
        "description" : "PSAD56",
        "code" : "EPSG:4248"
    }, {
        "description" : "Lake",
        "code" : "EPSG:4249"
    }, {
        "description" : "Leigon",
        "code" : "EPSG:4250"
    }, {
        "description" : "Liberia 1964",
        "code" : "EPSG:4251"
    }, {
        "description" : "Lome",
        "code" : "EPSG:4252"
    }, {
        "description" : "Luzon 1911",
        "code" : "EPSG:4253"
    }, {
        "description" : "Hito XVIII 1963",
        "code" : "EPSG:4254"
    }, {
        "description" : "Herat North",
        "code" : "EPSG:4255"
    }, {
        "description" : "Mahe 1971",
        "code" : "EPSG:4256"
    }, {
        "description" : "Makassar",
        "code" : "EPSG:4257"
    }, {
        "description" : "ETRS89",
        "code" : "EPSG:4258"
    }, {
        "description" : "Malongo 1987",
        "code" : "EPSG:4259"
    }, {
        "description" : "Manoca",
        "code" : "EPSG:4260"
    }, {
        "description" : "Merchich",
        "code" : "EPSG:4261"
    }, {
        "description" : "Massawa",
        "code" : "EPSG:4262"
    }, {
        "description" : "Minna",
        "code" : "EPSG:4263"
    }, {
        "description" : "Mhast",
        "code" : "EPSG:4264"
    }, {
        "description" : "Monte Mario",
        "code" : "EPSG:4265"
    }, {
        "description" : "M'poraloko",
        "code" : "EPSG:4266"
    }, {
        "description" : "NAD27",
        "code" : "EPSG:4267"
    }, {
        "description" : "NAD27 Michigan",
        "code" : "EPSG:4268"
    }, {
        "description" : "NAD83",
        "code" : "EPSG:4269"
    }, {
        "description" : "Nahrwan 1967",
        "code" : "EPSG:4270"
    }, {
        "description" : "Naparima 1972",
        "code" : "EPSG:4271"
    }, {
        "description" : "NZGD49",
        "code" : "EPSG:4272"
    }, {
        "description" : "NGO 1948",
        "code" : "EPSG:4273"
    }, {
        "description" : "Datum 73",
        "code" : "EPSG:4274"
    }, {
        "description" : "NTF",
        "code" : "EPSG:4275"
    }, {
        "description" : "NSWC 9Z-2",
        "code" : "EPSG:4276"
    }, {
        "description" : "OSGB 1936",
        "code" : "EPSG:4277"
    }, {
        "description" : "OSGB70",
        "code" : "EPSG:4278"
    }, {
        "description" : "OS(SN)80",
        "code" : "EPSG:4279"
    }, {
        "description" : "Padang",
        "code" : "EPSG:4280"
    }, {
        "description" : "Palestine 1923",
        "code" : "EPSG:4281"
    }, {
        "description" : "Pointe Noire",
        "code" : "EPSG:4282"
    }, {
        "description" : "GDA94",
        "code" : "EPSG:4283"
    }, {
        "description" : "Pulkovo 1942",
        "code" : "EPSG:4284"
    }, {
        "description" : "Qatar 1974",
        "code" : "EPSG:4285"
    }, {
        "description" : "Qatar 1948",
        "code" : "EPSG:4286"
    }, {
        "description" : "Qornoq",
        "code" : "EPSG:4287"
    }, {
        "description" : "Loma Quintana",
        "code" : "EPSG:4288"
    }, {
        "description" : "Amersfoort",
        "code" : "EPSG:4289"
    }, {
        "description" : "SAD69",
        "code" : "EPSG:4291"
    }, {
        "description" : "Sapper Hill 1943",
        "code" : "EPSG:4292"
    }, {
        "description" : "Schwarzeck",
        "code" : "EPSG:4293"
    }, {
        "description" : "Segora",
        "code" : "EPSG:4294"
    }, {
        "description" : "Serindung",
        "code" : "EPSG:4295"
    }, {
        "description" : "Sudan",
        "code" : "EPSG:4296"
    }, {
        "description" : "Tananarive",
        "code" : "EPSG:4297"
    }, {
        "description" : "Timbalai 1948",
        "code" : "EPSG:4298"
    }, {
        "description" : "TM65",
        "code" : "EPSG:4299"
    }, {
        "description" : "TM75",
        "code" : "EPSG:4300"
    }, {
        "description" : "Tokyo",
        "code" : "EPSG:4301"
    }, {
        "description" : "Trinidad 1903",
        "code" : "EPSG:4302"
    }, {
        "description" : "TC(1948)",
        "code" : "EPSG:4303"
    }, {
        "description" : "Voirol 1875",
        "code" : "EPSG:4304"
    }, {
        "description" : "Bern 1938",
        "code" : "EPSG:4306"
    }, {
        "description" : "Nord Sahara 1959",
        "code" : "EPSG:4307"
    }, {
        "description" : "RT38",
        "code" : "EPSG:4308"
    }, {
        "description" : "Yacare",
        "code" : "EPSG:4309"
    }, {
        "description" : "Yoff",
        "code" : "EPSG:4310"
    }, {
        "description" : "Zanderij",
        "code" : "EPSG:4311"
    }, {
        "description" : "MGI",
        "code" : "EPSG:4312"
    }, {
        "description" : "Belge 1972",
        "code" : "EPSG:4313"
    }, {
        "description" : "DHDN",
        "code" : "EPSG:4314"
    }, {
        "description" : "Conakry 1905",
        "code" : "EPSG:4315"
    }, {
        "description" : "Dealul Piscului 1930",
        "code" : "EPSG:4316"
    }, {
        "description" : "Dealul Piscului 1970",
        "code" : "EPSG:4317"
    }, {
        "description" : "NGN",
        "code" : "EPSG:4318"
    }, {
        "description" : "KUDAMS",
        "code" : "EPSG:4319"
    }, {
        "description" : "WGS 72",
        "code" : "EPSG:4322"
    }, {
        "description" : "WGS 72BE",
        "code" : "EPSG:4324"
    }, {
        "description" : "WGS 84",
        "code" : "EPSG:4326"
    }, {
        "description" : "Anguilla 1957",
        "code" : "EPSG:4600"
    }, {
        "description" : "Antigua 1943",
        "code" : "EPSG:4601"
    }, {
        "description" : "Dominica 1945",
        "code" : "EPSG:4602"
    }, {
        "description" : "Grenada 1953",
        "code" : "EPSG:4603"
    }, {
        "description" : "Montserrat 1958",
        "code" : "EPSG:4604"
    }, {
        "description" : "St. Kitts 1955",
        "code" : "EPSG:4605"
    }, {
        "description" : "St. Lucia 1955",
        "code" : "EPSG:4606"
    }, {
        "description" : "St. Vincent 1945",
        "code" : "EPSG:4607"
    }, {
        "description" : "NAD27(76)",
        "code" : "EPSG:4608"
    }, {
        "description" : "NAD27(CGQ77)",
        "code" : "EPSG:4609"
    }, {
        "description" : "Xian 1980",
        "code" : "EPSG:4610"
    }, {
        "description" : "Hong Kong 1980",
        "code" : "EPSG:4611"
    }, {
        "description" : "JGD2000",
        "code" : "EPSG:4612"
    }, {
        "description" : "Segara",
        "code" : "EPSG:4613"
    }, {
        "description" : "QND95",
        "code" : "EPSG:4614"
    }, {
        "description" : "Porto Santo",
        "code" : "EPSG:4615"
    }, {
        "description" : "Selvagem Grande",
        "code" : "EPSG:4616"
    }, {
        "description" : "NAD83(CSRS)",
        "code" : "EPSG:4617"
    }, {
        "description" : "SAD69",
        "code" : "EPSG:4618"
    }, {
        "description" : "SWEREF99",
        "code" : "EPSG:4619"
    }, {
        "description" : "Point 58",
        "code" : "EPSG:4620"
    }, {
        "description" : "Fort Marigot",
        "code" : "EPSG:4621"
    }, {
        "description" : "Guadeloupe 1948",
        "code" : "EPSG:4622"
    }, {
        "description" : "CSG67",
        "code" : "EPSG:4623"
    }, {
        "description" : "RGFG95",
        "code" : "EPSG:4624"
    }, {
        "description" : "Martinique 1938",
        "code" : "EPSG:4625"
    }, {
        "description" : "Reunion 1947",
        "code" : "EPSG:4626"
    }, {
        "description" : "RGR92",
        "code" : "EPSG:4627"
    }, {
        "description" : "Tahiti 52",
        "code" : "EPSG:4628"
    }, {
        "description" : "Tahaa 54",
        "code" : "EPSG:4629"
    }, {
        "description" : "IGN72 Nuku Hiva",
        "code" : "EPSG:4630"
    }, {
        "description" : "K0 1949",
        "code" : "EPSG:4631"
    }, {
        "description" : "Combani 1950",
        "code" : "EPSG:4632"
    }, {
        "description" : "IGN56 Lifou",
        "code" : "EPSG:4633"
    }, {
        "description" : "IGN72 Grand Terre",
        "code" : "EPSG:4634"
    }, {
        "description" : "ST87 Ouvea",
        "code" : "EPSG:4635"
    }, {
        "description" : "Petrels 1972",
        "code" : "EPSG:4636"
    }, {
        "description" : "Perroud 1950",
        "code" : "EPSG:4637"
    }, {
        "description" : "Saint Pierre et Miquelon 1950",
        "code" : "EPSG:4638"
    }, {
        "description" : "MOP78",
        "code" : "EPSG:4639"
    }, {
        "description" : "RRAF 1991",
        "code" : "EPSG:4640"
    }, {
        "description" : "IGN53 Mare",
        "code" : "EPSG:4641"
    }, {
        "description" : "ST84 Ile des Pins",
        "code" : "EPSG:4642"
    }, {
        "description" : "ST71 Belep",
        "code" : "EPSG:4643"
    }, {
        "description" : "NEA74 Noumea",
        "code" : "EPSG:4644"
    }, {
        "description" : "RGNC 1991",
        "code" : "EPSG:4645"
    }, {
        "description" : "Grand Comoros",
        "code" : "EPSG:4646"
    }, {
        "description" : "Reykjavik 1900",
        "code" : "EPSG:4657"
    }, {
        "description" : "Hjorsey 1955",
        "code" : "EPSG:4658"
    }, {
        "description" : "ISN93",
        "code" : "EPSG:4659"
    }, {
        "description" : "Helle 1954",
        "code" : "EPSG:4660"
    }, {
        "description" : "LKS92",
        "code" : "EPSG:4661"
    }, {
        "description" : "IGN72 Grande Terre",
        "code" : "EPSG:4662"
    }, {
        "description" : "Porto Santo 1995",
        "code" : "EPSG:4663"
    }, {
        "description" : "Azores Oriental 1995",
        "code" : "EPSG:4664"
    }, {
        "description" : "Azores Central 1995",
        "code" : "EPSG:4665"
    }, {
        "description" : "Lisbon 1890",
        "code" : "EPSG:4666"
    }, {
        "description" : "IKBD-92",
        "code" : "EPSG:4667"
    }, {
        "description" : "ED79",
        "code" : "EPSG:4668"
    }, {
        "description" : "LKS94",
        "code" : "EPSG:4669"
    }, {
        "description" : "IGM95",
        "code" : "EPSG:4670"
    }, {
        "description" : "Voirol 1879",
        "code" : "EPSG:4671"
    }, {
        "description" : "Chatham Islands 1971",
        "code" : "EPSG:4672"
    }, {
        "description" : "Chatham Islands 1979",
        "code" : "EPSG:4673"
    }, {
        "description" : "SIRGAS 2000",
        "code" : "EPSG:4674"
    }, {
        "description" : "Guam 1963",
        "code" : "EPSG:4675"
    }, {
        "description" : "Vientiane 1982",
        "code" : "EPSG:4676"
    }, {
        "description" : "Lao 1993",
        "code" : "EPSG:4677"
    }, {
        "description" : "Lao 1997",
        "code" : "EPSG:4678"
    }, {
        "description" : "Jouik 1961",
        "code" : "EPSG:4679"
    }, {
        "description" : "Nouakchott 1965",
        "code" : "EPSG:4680"
    }, {
        "description" : "Mauritania 1999",
        "code" : "EPSG:4681"
    }, {
        "description" : "Gulshan 303",
        "code" : "EPSG:4682"
    }, {
        "description" : "PRS92",
        "code" : "EPSG:4683"
    }, {
        "description" : "Gan 1970",
        "code" : "EPSG:4684"
    }, {
        "description" : "Gandajika",
        "code" : "EPSG:4685"
    }, {
        "description" : "MAGNA-SIRGAS",
        "code" : "EPSG:4686"
    }, {
        "description" : "RGPF",
        "code" : "EPSG:4687"
    }, {
        "description" : "Fatu Iva 72",
        "code" : "EPSG:4688"
    }, {
        "description" : "IGN63 Hiva Oa",
        "code" : "EPSG:4689"
    }, {
        "description" : "Tahiti 79",
        "code" : "EPSG:4690"
    }, {
        "description" : "Moorea 87",
        "code" : "EPSG:4691"
    }, {
        "description" : "Maupiti 83",
        "code" : "EPSG:4692"
    }, {
        "description" : "Nakhl-e Ghanem",
        "code" : "EPSG:4693"
    }, {
        "description" : "POSGAR 94",
        "code" : "EPSG:4694"
    }, {
        "description" : "Katanga 1955",
        "code" : "EPSG:4695"
    }, {
        "description" : "Kasai 1953",
        "code" : "EPSG:4696"
    }, {
        "description" : "IGC 1962 6th Parallel South",
        "code" : "EPSG:4697"
    }, {
        "description" : "IGN 1962 Kerguelen",
        "code" : "EPSG:4698"
    }, {
        "description" : "Le Pouce 1934",
        "code" : "EPSG:4699"
    }, {
        "description" : "IGN Astro 1960",
        "code" : "EPSG:4700"
    }, {
        "description" : "IGCB 1955",
        "code" : "EPSG:4701"
    }, {
        "description" : "Mauritania 1999",
        "code" : "EPSG:4702"
    }, {
        "description" : "Mhast 1951",
        "code" : "EPSG:4703"
    }, {
        "description" : "Mhast (onshore)",
        "code" : "EPSG:4704"
    }, {
        "description" : "Mhast (offshore)",
        "code" : "EPSG:4705"
    }, {
        "description" : "Egypt Gulf of Suez S-650 TL",
        "code" : "EPSG:4706"
    }, {
        "description" : "Tern Island 1961",
        "code" : "EPSG:4707"
    }, {
        "description" : "Cocos Islands 1965",
        "code" : "EPSG:4708"
    }, {
        "description" : "Iwo Jima 1945",
        "code" : "EPSG:4709"
    }, {
        "description" : "St. Helena 1971",
        "code" : "EPSG:4710"
    }, {
        "description" : "Marcus Island 1952",
        "code" : "EPSG:4711"
    }, {
        "description" : "Ascension Island 1958",
        "code" : "EPSG:4712"
    }, {
        "description" : "Ayabelle Lighthouse",
        "code" : "EPSG:4713"
    }, {
        "description" : "Bellevue",
        "code" : "EPSG:4714"
    }, {
        "description" : "Camp Area Astro",
        "code" : "EPSG:4715"
    }, {
        "description" : "Phoenix Islands 1966",
        "code" : "EPSG:4716"
    }, {
        "description" : "Cape Canaveral",
        "code" : "EPSG:4717"
    }, {
        "description" : "Solomon 1968",
        "code" : "EPSG:4718"
    }, {
        "description" : "Easter Island 1967",
        "code" : "EPSG:4719"
    }, {
        "description" : "Fiji 1986",
        "code" : "EPSG:4720"
    }, {
        "description" : "Fiji 1956",
        "code" : "EPSG:4721"
    }, {
        "description" : "South Georgia 1968",
        "code" : "EPSG:4722"
    }, {
        "description" : "Grand Cayman 1959",
        "code" : "EPSG:4723"
    }, {
        "description" : "Diego Garcia 1969",
        "code" : "EPSG:4724"
    }, {
        "description" : "Johnston Island 1961",
        "code" : "EPSG:4725"
    }, {
        "description" : "Little Cayman 1961",
        "code" : "EPSG:4726"
    }, {
        "description" : "Midway 1961",
        "code" : "EPSG:4727"
    }, {
        "description" : "Pico de las Nieves 1984",
        "code" : "EPSG:4728"
    }, {
        "description" : "Pitcairn 1967",
        "code" : "EPSG:4729"
    }, {
        "description" : "Santo 1965",
        "code" : "EPSG:4730"
    }, {
        "description" : "Viti Levu 1916",
        "code" : "EPSG:4731"
    }, {
        "description" : "Marshall Islands 1960",
        "code" : "EPSG:4732"
    }, {
        "description" : "Wake Island 1952",
        "code" : "EPSG:4733"
    }, {
        "description" : "Tristan 1968",
        "code" : "EPSG:4734"
    }, {
        "description" : "Kusaie 1951",
        "code" : "EPSG:4735"
    }, {
        "description" : "Deception Island",
        "code" : "EPSG:4736"
    }, {
        "description" : "Korea 2000",
        "code" : "EPSG:4737"
    }, {
        "description" : "Hong Kong 1963",
        "code" : "EPSG:4738"
    }, {
        "description" : "Hong Kong 1963(67)",
        "code" : "EPSG:4739"
    }, {
        "description" : "PZ-90",
        "code" : "EPSG:4740"
    }, {
        "description" : "FD54",
        "code" : "EPSG:4741"
    }, {
        "description" : "GDM2000",
        "code" : "EPSG:4742"
    }, {
        "description" : "Karbala 1979",
        "code" : "EPSG:4743"
    }, {
        "description" : "Nahrwan 1934",
        "code" : "EPSG:4744"
    }, {
        "description" : "RD/83",
        "code" : "EPSG:4745"
    }, {
        "description" : "PD/83",
        "code" : "EPSG:4746"
    }, {
        "description" : "GR96",
        "code" : "EPSG:4747"
    }, {
        "description" : "Vanua Levu 1915",
        "code" : "EPSG:4748"
    }, {
        "description" : "RGNC91-93",
        "code" : "EPSG:4749"
    }, {
        "description" : "ST87 Ouvea",
        "code" : "EPSG:4750"
    }, {
        "description" : "Kertau (RSO)",
        "code" : "EPSG:4751"
    }, {
        "description" : "Viti Levu 1912",
        "code" : "EPSG:4752"
    }, {
        "description" : "fk89",
        "code" : "EPSG:4753"
    }, {
        "description" : "LGD2006",
        "code" : "EPSG:4754"
    }, {
        "description" : "DGN95",
        "code" : "EPSG:4755"
    }, {
        "description" : "VN-2000",
        "code" : "EPSG:4756"
    }, {
        "description" : "SVY21",
        "code" : "EPSG:4757"
    }, {
        "description" : "JAD2001",
        "code" : "EPSG:4758"
    }, {
        "description" : "NAD83(NSRS2007)",
        "code" : "EPSG:4759"
    }, {
        "description" : "WGS 66",
        "code" : "EPSG:4760"
    }, {
        "description" : "HTRS96",
        "code" : "EPSG:4761"
    }, {
        "description" : "BDA2000",
        "code" : "EPSG:4762"
    }, {
        "description" : "Pitcairn 2006",
        "code" : "EPSG:4763"
    }, {
        "description" : "RSRGD2000",
        "code" : "EPSG:4764"
    }, {
        "description" : "Slovenia 1996",
        "code" : "EPSG:4765"
    }, {
        "description" : "Bern 1898 (Bern)",
        "code" : "EPSG:4801"
    }, {
        "description" : "Bogota 1975 (Bogota)",
        "code" : "EPSG:4802"
    }, {
        "description" : "Lisbon (Lisbon)",
        "code" : "EPSG:4803"
    }, {
        "description" : "Makassar (Jakarta)",
        "code" : "EPSG:4804"
    }, {
        "description" : "MGI (Ferro)",
        "code" : "EPSG:4805"
    }, {
        "description" : "Monte Mario (Rome)",
        "code" : "EPSG:4806"
    }, {
        "description" : "NTF (Paris)",
        "code" : "EPSG:4807"
    }, {
        "description" : "Padang (Jakarta)",
        "code" : "EPSG:4808"
    }, {
        "description" : "Belge 1950 (Brussels)",
        "code" : "EPSG:4809"
    }, {
        "description" : "Tananarive (Paris)",
        "code" : "EPSG:4810"
    }, {
        "description" : "Voirol 1875 (Paris)",
        "code" : "EPSG:4811"
    }, {
        "description" : "Batavia (Jakarta)",
        "code" : "EPSG:4813"
    }, {
        "description" : "RT38 (Stockholm)",
        "code" : "EPSG:4814"
    }, {
        "description" : "Greek (Athens)",
        "code" : "EPSG:4815"
    }, {
        "description" : "Carthage (Paris)",
        "code" : "EPSG:4816"
    }, {
        "description" : "NGO 1948 (Oslo)",
        "code" : "EPSG:4817"
    }, {
        "description" : "S-JTSK (Ferro)",
        "code" : "EPSG:4818"
    }, {
        "description" : "Nord Sahara 1959 (Paris)",
        "code" : "EPSG:4819"
    }, {
        "description" : "Segara (Jakarta)",
        "code" : "EPSG:4820"
    }, {
        "description" : "Voirol 1879 (Paris)",
        "code" : "EPSG:4821"
    }, {
        "description" : "ATF (Paris)",
        "code" : "EPSG:4901"
    }, {
        "description" : "NDG (Paris)",
        "code" : "EPSG:4902"
    }, {
        "description" : "Madrid 1870 (Madrid)",
        "code" : "EPSG:4903"
    }, {
        "description" : "Lisbon 1890 (Lisbon)",
        "code" : "EPSG:4904"
    }, {
        "description" : "Anguilla 1957 / British West Indies Grid",
        "code" : "EPSG:2000"
    }, {
        "description" : "Antigua 1943 / British West Indies Grid",
        "code" : "EPSG:2001"
    }, {
        "description" : "Dominica 1945 / British West Indies Grid",
        "code" : "EPSG:2002"
    }, {
        "description" : "Grenada 1953 / British West Indies Grid",
        "code" : "EPSG:2003"
    }, {
        "description" : "Montserrat 1958 / British West Indies Grid",
        "code" : "EPSG:2004"
    }, {
        "description" : "St. Kitts 1955 / British West Indies Grid",
        "code" : "EPSG:2005"
    }, {
        "description" : "St. Lucia 1955 / British West Indies Grid",
        "code" : "EPSG:2006"
    }, {
        "description" : "St. Vincent 45 / British West Indies Grid",
        "code" : "EPSG:2007"
    }, {
        "description" : "NAD27(CGQ77) / SCoPQ zone 2",
        "code" : "EPSG:2008"
    }, {
        "description" : "NAD27(CGQ77) / SCoPQ zone 3",
        "code" : "EPSG:2009"
    }, {
        "description" : "NAD27(CGQ77) / SCoPQ zone 4",
        "code" : "EPSG:2010"
    }, {
        "description" : "NAD27(CGQ77) / SCoPQ zone 5",
        "code" : "EPSG:2011"
    }, {
        "description" : "NAD27(CGQ77) / SCoPQ zone 6",
        "code" : "EPSG:2012"
    }, {
        "description" : "NAD27(CGQ77) / SCoPQ zone 7",
        "code" : "EPSG:2013"
    }, {
        "description" : "NAD27(CGQ77) / SCoPQ zone 8",
        "code" : "EPSG:2014"
    }, {
        "description" : "NAD27(CGQ77) / SCoPQ zone 9",
        "code" : "EPSG:2015"
    }, {
        "description" : "NAD27(CGQ77) / SCoPQ zone 10",
        "code" : "EPSG:2016"
    }, {
        "description" : "NAD27(76) / MTM zone 8",
        "code" : "EPSG:2017"
    }, {
        "description" : "NAD27(76) / MTM zone 9",
        "code" : "EPSG:2018"
    }, {
        "description" : "NAD27(76) / MTM zone 10",
        "code" : "EPSG:2019"
    }, {
        "description" : "NAD27(76) / MTM zone 11",
        "code" : "EPSG:2020"
    }, {
        "description" : "NAD27(76) / MTM zone 12",
        "code" : "EPSG:2021"
    }, {
        "description" : "NAD27(76) / MTM zone 13",
        "code" : "EPSG:2022"
    }, {
        "description" : "NAD27(76) / MTM zone 14",
        "code" : "EPSG:2023"
    }, {
        "description" : "NAD27(76) / MTM zone 15",
        "code" : "EPSG:2024"
    }, {
        "description" : "NAD27(76) / MTM zone 16",
        "code" : "EPSG:2025"
    }, {
        "description" : "NAD27(76) / MTM zone 17",
        "code" : "EPSG:2026"
    }, {
        "description" : "NAD27(76) / UTM zone 15N",
        "code" : "EPSG:2027"
    }, {
        "description" : "NAD27(76) / UTM zone 16N",
        "code" : "EPSG:2028"
    }, {
        "description" : "NAD27(76) / UTM zone 17N",
        "code" : "EPSG:2029"
    }, {
        "description" : "NAD27(76) / UTM zone 18N",
        "code" : "EPSG:2030"
    }, {
        "description" : "NAD27(CGQ77) / UTM zone 17N",
        "code" : "EPSG:2031"
    }, {
        "description" : "NAD27(CGQ77) / UTM zone 18N",
        "code" : "EPSG:2032"
    }, {
        "description" : "NAD27(CGQ77) / UTM zone 19N",
        "code" : "EPSG:2033"
    }, {
        "description" : "NAD27(CGQ77) / UTM zone 20N",
        "code" : "EPSG:2034"
    }, {
        "description" : "NAD27(CGQ77) / UTM zone 21N",
        "code" : "EPSG:2035"
    }, {
        "description" : "NAD83(CSRS98) / New Brunswick Stereo",
        "code" : "EPSG:2036"
    }, {
        "description" : "NAD83(CSRS98) / UTM zone 19N",
        "code" : "EPSG:2037"
    }, {
        "description" : "NAD83(CSRS98) / UTM zone 20N",
        "code" : "EPSG:2038"
    }, {
        "description" : "Israel / Israeli TM Grid",
        "code" : "EPSG:2039"
    }, {
        "description" : "Locodjo 1965 / UTM zone 30N",
        "code" : "EPSG:2040"
    }, {
        "description" : "Abidjan 1987 / UTM zone 30N",
        "code" : "EPSG:2041"
    }, {
        "description" : "Locodjo 1965 / UTM zone 29N",
        "code" : "EPSG:2042"
    }, {
        "description" : "Abidjan 1987 / UTM zone 29N",
        "code" : "EPSG:2043"
    }, {
        "description" : "Hanoi 1972 / Gauss-Kruger zone 18",
        "code" : "EPSG:2044"
    }, {
        "description" : "Hanoi 1972 / Gauss-Kruger zone 19",
        "code" : "EPSG:2045"
    }, {
        "description" : "Hartebeesthoek94 / Lo15",
        "code" : "EPSG:2046"
    }, {
        "description" : "Hartebeesthoek94 / Lo17",
        "code" : "EPSG:2047"
    }, {
        "description" : "Hartebeesthoek94 / Lo19",
        "code" : "EPSG:2048"
    }, {
        "description" : "Hartebeesthoek94 / Lo21",
        "code" : "EPSG:2049"
    }, {
        "description" : "Hartebeesthoek94 / Lo23",
        "code" : "EPSG:2050"
    }, {
        "description" : "Hartebeesthoek94 / Lo25",
        "code" : "EPSG:2051"
    }, {
        "description" : "Hartebeesthoek94 / Lo27",
        "code" : "EPSG:2052"
    }, {
        "description" : "Hartebeesthoek94 / Lo29",
        "code" : "EPSG:2053"
    }, {
        "description" : "Hartebeesthoek94 / Lo31",
        "code" : "EPSG:2054"
    }, {
        "description" : "Hartebeesthoek94 / Lo33",
        "code" : "EPSG:2055"
    }, {
        "description" : "CH1903+ / LV95",
        "code" : "EPSG:2056"
    }, {
        "description" : "Rassadiran / Nakhl e Taqi",
        "code" : "EPSG:2057"
    }, {
        "description" : "ED50(ED77) / UTM zone 38N",
        "code" : "EPSG:2058"
    }, {
        "description" : "ED50(ED77) / UTM zone 39N",
        "code" : "EPSG:2059"
    }, {
        "description" : "ED50(ED77) / UTM zone 40N",
        "code" : "EPSG:2060"
    }, {
        "description" : "ED50(ED77) / UTM zone 41N",
        "code" : "EPSG:2061"
    }, {
        "description" : "Madrid 1870 (Madrid) / Spain",
        "code" : "EPSG:2062"
    }, {
        "description" : "Dabola 1981 / UTM zone 28N",
        "code" : "EPSG:2063"
    }, {
        "description" : "Dabola 1981 / UTM zone 29N",
        "code" : "EPSG:2064"
    }, {
        "description" : "S-JTSK (Ferro) / Krovak",
        "code" : "EPSG:2065"
    }, {
        "description" : "Mount Dillon / Tobago Grid",
        "code" : "EPSG:2066"
    }, {
        "description" : "Naparima 1955 / UTM zone 20N",
        "code" : "EPSG:2067"
    }, {
        "description" : "ELD79 / Libya zone 5",
        "code" : "EPSG:2068"
    }, {
        "description" : "ELD79 / Libya zone 6",
        "code" : "EPSG:2069"
    }, {
        "description" : "ELD79 / Libya zone 7",
        "code" : "EPSG:2070"
    }, {
        "description" : "ELD79 / Libya zone 8",
        "code" : "EPSG:2071"
    }, {
        "description" : "ELD79 / Libya zone 9",
        "code" : "EPSG:2072"
    }, {
        "description" : "ELD79 / Libya zone 10",
        "code" : "EPSG:2073"
    }, {
        "description" : "ELD79 / Libya zone 11",
        "code" : "EPSG:2074"
    }, {
        "description" : "ELD79 / Libya zone 12",
        "code" : "EPSG:2075"
    }, {
        "description" : "ELD79 / Libya zone 13",
        "code" : "EPSG:2076"
    }, {
        "description" : "ELD79 / UTM zone 32N",
        "code" : "EPSG:2077"
    }, {
        "description" : "ELD79 / UTM zone 33N",
        "code" : "EPSG:2078"
    }, {
        "description" : "ELD79 / UTM zone 34N",
        "code" : "EPSG:2079"
    }, {
        "description" : "ELD79 / UTM zone 35N",
        "code" : "EPSG:2080"
    }, {
        "description" : "Chos Malal 1914 / Argentina zone 2",
        "code" : "EPSG:2081"
    }, {
        "description" : "Pampa del Castillo / Argentina zone 2",
        "code" : "EPSG:2082"
    }, {
        "description" : "Hito XVIII 1963 / Argentina zone 2",
        "code" : "EPSG:2083"
    }, {
        "description" : "Hito XVIII 1963 / UTM zone 19S",
        "code" : "EPSG:2084"
    }, {
        "description" : "NAD27 / Cuba Norte",
        "code" : "EPSG:2085"
    }, {
        "description" : "NAD27 / Cuba Sur",
        "code" : "EPSG:2086"
    }, {
        "description" : "ELD79 / TM 12 NE",
        "code" : "EPSG:2087"
    }, {
        "description" : "Carthage / TM 11 NE",
        "code" : "EPSG:2088"
    }, {
        "description" : "Yemen NGN96 / UTM zone 38N",
        "code" : "EPSG:2089"
    }, {
        "description" : "Yemen NGN96 / UTM zone 39N",
        "code" : "EPSG:2090"
    }, {
        "description" : "South Yemen / Gauss Kruger zone 8",
        "code" : "EPSG:2091"
    }, {
        "description" : "South Yemen / Gauss Kruger zone 9",
        "code" : "EPSG:2092"
    }, {
        "description" : "Hanoi 1972 / GK 106 NE",
        "code" : "EPSG:2093"
    }, {
        "description" : "WGS 72BE / TM 106 NE",
        "code" : "EPSG:2094"
    }, {
        "description" : "Bissau / UTM zone 28N",
        "code" : "EPSG:2095"
    }, {
        "description" : "Korean 1985 / Korea East Belt",
        "code" : "EPSG:2096"
    }, {
        "description" : "Korean 1985 / Korea Central Belt",
        "code" : "EPSG:2097"
    }, {
        "description" : "Korean 1985 / Korea West Belt",
        "code" : "EPSG:2098"
    }, {
        "description" : "Qatar 1948 / Qatar Grid",
        "code" : "EPSG:2099"
    }, {
        "description" : "GGRS87 / Greek Grid",
        "code" : "EPSG:2100"
    }, {
        "description" : "Lake / Maracaibo Grid M1",
        "code" : "EPSG:2101"
    }, {
        "description" : "Lake / Maracaibo Grid",
        "code" : "EPSG:2102"
    }, {
        "description" : "Lake / Maracaibo Grid M3",
        "code" : "EPSG:2103"
    }, {
        "description" : "Lake / Maracaibo La Rosa Grid",
        "code" : "EPSG:2104"
    }, {
        "description" : "NZGD2000 / Mount Eden 2000",
        "code" : "EPSG:2105"
    }, {
        "description" : "NZGD2000 / Bay of Plenty 2000",
        "code" : "EPSG:2106"
    }, {
        "description" : "NZGD2000 / Poverty Bay 2000",
        "code" : "EPSG:2107"
    }, {
        "description" : "NZGD2000 / Hawkes Bay 2000",
        "code" : "EPSG:2108"
    }, {
        "description" : "NZGD2000 / Taranaki 2000",
        "code" : "EPSG:2109"
    }, {
        "description" : "NZGD2000 / Tuhirangi 2000",
        "code" : "EPSG:2110"
    }, {
        "description" : "NZGD2000 / Wanganui 2000",
        "code" : "EPSG:2111"
    }, {
        "description" : "NZGD2000 / Wairarapa 2000",
        "code" : "EPSG:2112"
    }, {
        "description" : "NZGD2000 / Wellington 2000",
        "code" : "EPSG:2113"
    }, {
        "description" : "NZGD2000 / Collingwood 2000",
        "code" : "EPSG:2114"
    }, {
        "description" : "NZGD2000 / Nelson 2000",
        "code" : "EPSG:2115"
    }, {
        "description" : "NZGD2000 / Karamea 2000",
        "code" : "EPSG:2116"
    }, {
        "description" : "NZGD2000 / Buller 2000",
        "code" : "EPSG:2117"
    }, {
        "description" : "NZGD2000 / Grey 2000",
        "code" : "EPSG:2118"
    }, {
        "description" : "NZGD2000 / Amuri 2000",
        "code" : "EPSG:2119"
    }, {
        "description" : "NZGD2000 / Marlborough 2000",
        "code" : "EPSG:2120"
    }, {
        "description" : "NZGD2000 / Hokitika 2000",
        "code" : "EPSG:2121"
    }, {
        "description" : "NZGD2000 / Okarito 2000",
        "code" : "EPSG:2122"
    }, {
        "description" : "NZGD2000 / Jacksons Bay 2000",
        "code" : "EPSG:2123"
    }, {
        "description" : "NZGD2000 / Mount Pleasant 2000",
        "code" : "EPSG:2124"
    }, {
        "description" : "NZGD2000 / Gawler 2000",
        "code" : "EPSG:2125"
    }, {
        "description" : "NZGD2000 / Timaru 2000",
        "code" : "EPSG:2126"
    }, {
        "description" : "NZGD2000 / Lindis Peak 2000",
        "code" : "EPSG:2127"
    }, {
        "description" : "NZGD2000 / Mount Nicholas 2000",
        "code" : "EPSG:2128"
    }, {
        "description" : "NZGD2000 / Mount York 2000",
        "code" : "EPSG:2129"
    }, {
        "description" : "NZGD2000 / Observation Point 2000",
        "code" : "EPSG:2130"
    }, {
        "description" : "NZGD2000 / North Taieri 2000",
        "code" : "EPSG:2131"
    }, {
        "description" : "NZGD2000 / Bluff 2000",
        "code" : "EPSG:2132"
    }, {
        "description" : "NZGD2000 / UTM zone 58S",
        "code" : "EPSG:2133"
    }, {
        "description" : "NZGD2000 / UTM zone 59S",
        "code" : "EPSG:2134"
    }, {
        "description" : "NZGD2000 / UTM zone 60S",
        "code" : "EPSG:2135"
    }, {
        "description" : "Accra / Ghana National Grid",
        "code" : "EPSG:2136"
    }, {
        "description" : "Accra / TM 1 NW",
        "code" : "EPSG:2137"
    }, {
        "description" : "NAD27(CGQ77) / Quebec Lambert",
        "code" : "EPSG:2138"
    }, {
        "description" : "NAD83(CSRS98) / SCoPQ zone 2",
        "code" : "EPSG:2139"
    }, {
        "description" : "NAD83(CSRS98) / MTM zone 3",
        "code" : "EPSG:2140"
    }, {
        "description" : "NAD83(CSRS98) / MTM zone 4",
        "code" : "EPSG:2141"
    }, {
        "description" : "NAD83(CSRS98) / MTM zone 5",
        "code" : "EPSG:2142"
    }, {
        "description" : "NAD83(CSRS98) / MTM zone 6",
        "code" : "EPSG:2143"
    }, {
        "description" : "NAD83(CSRS98) / MTM zone 7",
        "code" : "EPSG:2144"
    }, {
        "description" : "NAD83(CSRS98) / MTM zone 8",
        "code" : "EPSG:2145"
    }, {
        "description" : "NAD83(CSRS98) / MTM zone 9",
        "code" : "EPSG:2146"
    }, {
        "description" : "NAD83(CSRS98) / MTM zone 10",
        "code" : "EPSG:2147"
    }, {
        "description" : "NAD83(CSRS98) / UTM zone 21N",
        "code" : "EPSG:2148"
    }, {
        "description" : "NAD83(CSRS98) / UTM zone 18N",
        "code" : "EPSG:2149"
    }, {
        "description" : "NAD83(CSRS98) / UTM zone 17N",
        "code" : "EPSG:2150"
    }, {
        "description" : "NAD83(CSRS98) / UTM zone 13N",
        "code" : "EPSG:2151"
    }, {
        "description" : "NAD83(CSRS98) / UTM zone 12N",
        "code" : "EPSG:2152"
    }, {
        "description" : "NAD83(CSRS98) / UTM zone 11N",
        "code" : "EPSG:2153"
    }, {
        "description" : "RGF93 / Lambert-93",
        "code" : "EPSG:2154"
    }, {
        "description" : "American Samoa 1962 / American Samoa Lambert",
        "code" : "EPSG:2155"
    }, {
        "description" : "NAD83(HARN) / UTM zone 59S",
        "code" : "EPSG:2156"
    }, {
        "description" : "IRENET95 / Irish Transverse Mercator",
        "code" : "EPSG:2157"
    }, {
        "description" : "IRENET95 / UTM zone 29N",
        "code" : "EPSG:2158"
    }, {
        "description" : "Sierra Leone 1924 / New Colony Grid",
        "code" : "EPSG:2159"
    }, {
        "description" : "Sierra Leone 1924 / New War Office Grid",
        "code" : "EPSG:2160"
    }, {
        "description" : "Sierra Leone 1968 / UTM zone 28N",
        "code" : "EPSG:2161"
    }, {
        "description" : "Sierra Leone 1968 / UTM zone 29N",
        "code" : "EPSG:2162"
    }, {
        "description" : "US National Atlas Equal Area",
        "code" : "EPSG:2163"
    }, {
        "description" : "Locodjo 1965 / TM 5 NW",
        "code" : "EPSG:2164"
    }, {
        "description" : "Abidjan 1987 / TM 5 NW",
        "code" : "EPSG:2165"
    }, {
        "description" : "Pulkovo 1942(83) / Gauss Kruger zone 3",
        "code" : "EPSG:2166"
    }, {
        "description" : "Pulkovo 1942(83) / Gauss Kruger zone 4",
        "code" : "EPSG:2167"
    }, {
        "description" : "Pulkovo 1942(83) / Gauss Kruger zone 5",
        "code" : "EPSG:2168"
    }, {
        "description" : "Luxembourg 1930 / Gauss",
        "code" : "EPSG:2169"
    }, {
        "description" : "MGI / Slovenia Grid",
        "code" : "EPSG:2170"
    }, {
        "description" : "Pulkovo 1942(58) / Poland zone I",
        "code" : "EPSG:2171"
    }, {
        "description" : "Pulkovo 1942(58) / Poland zone II",
        "code" : "EPSG:2172"
    }, {
        "description" : "Pulkovo 1942(58) / Poland zone III",
        "code" : "EPSG:2173"
    }, {
        "description" : "Pulkovo 1942(58) / Poland zone IV",
        "code" : "EPSG:2174"
    }, {
        "description" : "Pulkovo 1942(58) / Poland zone V",
        "code" : "EPSG:2175"
    }, {
        "description" : "ETRS89 / Poland CS2000 zone 5",
        "code" : "EPSG:2176"
    }, {
        "description" : "ETRS89 / Poland CS2000 zone 6",
        "code" : "EPSG:2177"
    }, {
        "description" : "ETRS89 / Poland CS2000 zone 7",
        "code" : "EPSG:2178"
    }, {
        "description" : "ETRS89 / Poland CS2000 zone 8",
        "code" : "EPSG:2179"
    }, {
        "description" : "ETRS89 / Poland CS92",
        "code" : "EPSG:2180"
    }, {
        "description" : "Azores Occidental 1939 / UTM zone 25N",
        "code" : "EPSG:2188"
    }, {
        "description" : "Azores Central 1948 / UTM zone 26N",
        "code" : "EPSG:2189"
    }, {
        "description" : "Azores Oriental 1940 / UTM zone 26N",
        "code" : "EPSG:2190"
    }, {
        "description" : "Madeira 1936 / UTM zone 28N",
        "code" : "EPSG:2191"
    }, {
        "description" : "ED50 / France EuroLambert",
        "code" : "EPSG:2192"
    }, {
        "description" : "NZGD2000 / New Zealand Transverse Mercator 2000",
        "code" : "EPSG:2193"
    }, {
        "description" : "American Samoa 1962 / American Samoa Lambert",
        "code" : "EPSG:2194"
    }, {
        "description" : "NAD83(HARN) / UTM zone 2S",
        "code" : "EPSG:2195"
    }, {
        "description" : "ETRS89 / Kp2000 Jutland",
        "code" : "EPSG:2196"
    }, {
        "description" : "ETRS89 / Kp2000 Zealand",
        "code" : "EPSG:2197"
    }, {
        "description" : "ETRS89 / Kp2000 Bornholm",
        "code" : "EPSG:2198"
    }, {
        "description" : "Albanian 1987 / Gauss Kruger zone 4",
        "code" : "EPSG:2199"
    }, {
        "description" : "ATS77 / New Brunswick Stereographic (ATS77)",
        "code" : "EPSG:2200"
    }, {
        "description" : "REGVEN / UTM zone 18N",
        "code" : "EPSG:2201"
    }, {
        "description" : "REGVEN / UTM zone 19N",
        "code" : "EPSG:2202"
    }, {
        "description" : "REGVEN / UTM zone 20N",
        "code" : "EPSG:2203"
    }, {
        "description" : "NAD27 / Tennessee",
        "code" : "EPSG:2204"
    }, {
        "description" : "NAD83 / Kentucky North",
        "code" : "EPSG:2205"
    }, {
        "description" : "ED50 / 3-degree Gauss-Kruger zone 9",
        "code" : "EPSG:2206"
    }, {
        "description" : "ED50 / 3-degree Gauss-Kruger zone 10",
        "code" : "EPSG:2207"
    }, {
        "description" : "ED50 / 3-degree Gauss-Kruger zone 11",
        "code" : "EPSG:2208"
    }, {
        "description" : "ED50 / 3-degree Gauss-Kruger zone 12",
        "code" : "EPSG:2209"
    }, {
        "description" : "ED50 / 3-degree Gauss-Kruger zone 13",
        "code" : "EPSG:2210"
    }, {
        "description" : "ED50 / 3-degree Gauss-Kruger zone 14",
        "code" : "EPSG:2211"
    }, {
        "description" : "ED50 / 3-degree Gauss-Kruger zone 15",
        "code" : "EPSG:2212"
    }, {
        "description" : "ETRS89 / TM 30 NE",
        "code" : "EPSG:2213"
    }, {
        "description" : "Douala 1948 / AOF west",
        "code" : "EPSG:2214"
    }, {
        "description" : "Manoca 1962 / UTM zone 32N",
        "code" : "EPSG:2215"
    }, {
        "description" : "Qornoq 1927 / UTM zone 22N",
        "code" : "EPSG:2216"
    }, {
        "description" : "Qornoq 1927 / UTM zone 23N",
        "code" : "EPSG:2217"
    }, {
        "description" : "Scoresbysund 1952 / Greenland zone 5 east",
        "code" : "EPSG:2218"
    }, {
        "description" : "ATS77 / UTM zone 19N",
        "code" : "EPSG:2219"
    }, {
        "description" : "ATS77 / UTM zone 20N",
        "code" : "EPSG:2220"
    }, {
        "description" : "Scoresbysund 1952 / Greenland zone 6 east",
        "code" : "EPSG:2221"
    }, {
        "description" : "NAD83 / Arizona East (ft)",
        "code" : "EPSG:2222"
    }, {
        "description" : "NAD83 / Arizona Central (ft)",
        "code" : "EPSG:2223"
    }, {
        "description" : "NAD83 / Arizona West (ft)",
        "code" : "EPSG:2224"
    }, {
        "description" : "NAD83 / California zone 1 (ftUS)",
        "code" : "EPSG:2225"
    }, {
        "description" : "NAD83 / California zone 2 (ftUS)",
        "code" : "EPSG:2226"
    }, {
        "description" : "NAD83 / California zone 3 (ftUS)",
        "code" : "EPSG:2227"
    }, {
        "description" : "NAD83 / California zone 4 (ftUS)",
        "code" : "EPSG:2228"
    }, {
        "description" : "NAD83 / California zone 5 (ftUS)",
        "code" : "EPSG:2229"
    }, {
        "description" : "NAD83 / California zone 6 (ftUS)",
        "code" : "EPSG:2230"
    }, {
        "description" : "NAD83 / Colorado North (ftUS)",
        "code" : "EPSG:2231"
    }, {
        "description" : "NAD83 / Colorado Central (ftUS)",
        "code" : "EPSG:2232"
    }, {
        "description" : "NAD83 / Colorado South (ftUS)",
        "code" : "EPSG:2233"
    }, {
        "description" : "NAD83 / Connecticut (ftUS)",
        "code" : "EPSG:2234"
    }, {
        "description" : "NAD83 / Delaware (ftUS)",
        "code" : "EPSG:2235"
    }, {
        "description" : "NAD83 / Florida East (ftUS)",
        "code" : "EPSG:2236"
    }, {
        "description" : "NAD83 / Florida West (ftUS)",
        "code" : "EPSG:2237"
    }, {
        "description" : "NAD83 / Florida North (ftUS)",
        "code" : "EPSG:2238"
    }, {
        "description" : "NAD83 / Georgia East (ftUS)",
        "code" : "EPSG:2239"
    }, {
        "description" : "NAD83 / Georgia West (ftUS)",
        "code" : "EPSG:2240"
    }, {
        "description" : "NAD83 / Idaho East (ftUS)",
        "code" : "EPSG:2241"
    }, {
        "description" : "NAD83 / Idaho Central (ftUS)",
        "code" : "EPSG:2242"
    }, {
        "description" : "NAD83 / Idaho West (ftUS)",
        "code" : "EPSG:2243"
    }, {
        "description" : "NAD83 / Indiana East (ftUS)",
        "code" : "EPSG:2244"
    }, {
        "description" : "NAD83 / Indiana West (ftUS)",
        "code" : "EPSG:2245"
    }, {
        "description" : "NAD83 / Kentucky North (ftUS)",
        "code" : "EPSG:2246"
    }, {
        "description" : "NAD83 / Kentucky South (ftUS)",
        "code" : "EPSG:2247"
    }, {
        "description" : "NAD83 / Maryland (ftUS)",
        "code" : "EPSG:2248"
    }, {
        "description" : "NAD83 / Massachusetts Mainland (ftUS)",
        "code" : "EPSG:2249"
    }, {
        "description" : "NAD83 / Massachusetts Island (ftUS)",
        "code" : "EPSG:2250"
    }, {
        "description" : "NAD83 / Michigan North (ft)",
        "code" : "EPSG:2251"
    }, {
        "description" : "NAD83 / Michigan Central (ft)",
        "code" : "EPSG:2252"
    }, {
        "description" : "NAD83 / Michigan South (ft)",
        "code" : "EPSG:2253"
    }, {
        "description" : "NAD83 / Mississippi East (ftUS)",
        "code" : "EPSG:2254"
    }, {
        "description" : "NAD83 / Mississippi West (ftUS)",
        "code" : "EPSG:2255"
    }, {
        "description" : "NAD83 / Montana (ft)",
        "code" : "EPSG:2256"
    }, {
        "description" : "NAD83 / New Mexico East (ftUS)",
        "code" : "EPSG:2257"
    }, {
        "description" : "NAD83 / New Mexico Central (ftUS)",
        "code" : "EPSG:2258"
    }, {
        "description" : "NAD83 / New Mexico West (ftUS)",
        "code" : "EPSG:2259"
    }, {
        "description" : "NAD83 / New York East (ftUS)",
        "code" : "EPSG:2260"
    }, {
        "description" : "NAD83 / New York Central (ftUS)",
        "code" : "EPSG:2261"
    }, {
        "description" : "NAD83 / New York West (ftUS)",
        "code" : "EPSG:2262"
    }, {
        "description" : "NAD83 / New York Long Island (ftUS)",
        "code" : "EPSG:2263"
    }, {
        "description" : "NAD83 / North Carolina (ftUS)",
        "code" : "EPSG:2264"
    }, {
        "description" : "NAD83 / North Dakota North (ft)",
        "code" : "EPSG:2265"
    }, {
        "description" : "NAD83 / North Dakota South (ft)",
        "code" : "EPSG:2266"
    }, {
        "description" : "NAD83 / Oklahoma North (ftUS)",
        "code" : "EPSG:2267"
    }, {
        "description" : "NAD83 / Oklahoma South (ftUS)",
        "code" : "EPSG:2268"
    }, {
        "description" : "NAD83 / Oregon North (ft)",
        "code" : "EPSG:2269"
    }, {
        "description" : "NAD83 / Oregon South (ft)",
        "code" : "EPSG:2270"
    }, {
        "description" : "NAD83 / Pennsylvania North (ftUS)",
        "code" : "EPSG:2271"
    }, {
        "description" : "NAD83 / Pennsylvania South (ftUS)",
        "code" : "EPSG:2272"
    }, {
        "description" : "NAD83 / South Carolina (ft)",
        "code" : "EPSG:2273"
    }, {
        "description" : "NAD83 / Tennessee (ftUS)",
        "code" : "EPSG:2274"
    }, {
        "description" : "NAD83 / Texas North (ftUS)",
        "code" : "EPSG:2275"
    }, {
        "description" : "NAD83 / Texas North Central (ftUS)",
        "code" : "EPSG:2276"
    }, {
        "description" : "NAD83 / Texas Central (ftUS)",
        "code" : "EPSG:2277"
    }, {
        "description" : "NAD83 / Texas South Central (ftUS)",
        "code" : "EPSG:2278"
    }, {
        "description" : "NAD83 / Texas South (ftUS)",
        "code" : "EPSG:2279"
    }, {
        "description" : "NAD83 / Utah North (ft)",
        "code" : "EPSG:2280"
    }, {
        "description" : "NAD83 / Utah Central (ft)",
        "code" : "EPSG:2281"
    }, {
        "description" : "NAD83 / Utah South (ft)",
        "code" : "EPSG:2282"
    }, {
        "description" : "NAD83 / Virginia North (ftUS)",
        "code" : "EPSG:2283"
    }, {
        "description" : "NAD83 / Virginia South (ftUS)",
        "code" : "EPSG:2284"
    }, {
        "description" : "NAD83 / Washington North (ftUS)",
        "code" : "EPSG:2285"
    }, {
        "description" : "NAD83 / Washington South (ftUS)",
        "code" : "EPSG:2286"
    }, {
        "description" : "NAD83 / Wisconsin North (ftUS)",
        "code" : "EPSG:2287"
    }, {
        "description" : "NAD83 / Wisconsin Central (ftUS)",
        "code" : "EPSG:2288"
    }, {
        "description" : "NAD83 / Wisconsin South (ftUS)",
        "code" : "EPSG:2289"
    }, {
        "description" : "ATS77 / Prince Edward Isl. Stereographic (ATS77)",
        "code" : "EPSG:2290"
    }, {
        "description" : "NAD83(CSRS98) / Prince Edward Isl. Stereographic (NAD83)",
        "code" : "EPSG:2291"
    }, {
        "description" : "NAD83(CSRS98) / Prince Edward Isl. Stereographic (NAD83)",
        "code" : "EPSG:2292"
    }, {
        "description" : "ATS77 / MTM Nova Scotia zone 4",
        "code" : "EPSG:2294"
    }, {
        "description" : "ATS77 / MTM Nova Scotia zone 5",
        "code" : "EPSG:2295"
    }, {
        "description" : "Ammassalik 1958 / Greenland zone 7 east",
        "code" : "EPSG:2296"
    }, {
        "description" : "Qornoq 1927 / Greenland zone 1 east",
        "code" : "EPSG:2297"
    }, {
        "description" : "Qornoq 1927 / Greenland zone 2 east",
        "code" : "EPSG:2298"
    }, {
        "description" : "Qornoq 1927 / Greenland zone 2 west",
        "code" : "EPSG:2299"
    }, {
        "description" : "Qornoq 1927 / Greenland zone 3 east",
        "code" : "EPSG:2300"
    }, {
        "description" : "Qornoq 1927 / Greenland zone 3 west",
        "code" : "EPSG:2301"
    }, {
        "description" : "Qornoq 1927 / Greenland zone 4 east",
        "code" : "EPSG:2302"
    }, {
        "description" : "Qornoq 1927 / Greenland zone 4 west",
        "code" : "EPSG:2303"
    }, {
        "description" : "Qornoq 1927 / Greenland zone 5 west",
        "code" : "EPSG:2304"
    }, {
        "description" : "Qornoq 1927 / Greenland zone 6 west",
        "code" : "EPSG:2305"
    }, {
        "description" : "Qornoq 1927 / Greenland zone 7 west",
        "code" : "EPSG:2306"
    }, {
        "description" : "Qornoq 1927 / Greenland zone 8 east",
        "code" : "EPSG:2307"
    }, {
        "description" : "Batavia / TM 109 SE",
        "code" : "EPSG:2308"
    }, {
        "description" : "WGS 84 / TM 116 SE",
        "code" : "EPSG:2309"
    }, {
        "description" : "WGS 84 / TM 132 SE",
        "code" : "EPSG:2310"
    }, {
        "description" : "WGS 84 / TM 6 NE",
        "code" : "EPSG:2311"
    }, {
        "description" : "Garoua / UTM zone 33N",
        "code" : "EPSG:2312"
    }, {
        "description" : "Kousseri / UTM zone 33N",
        "code" : "EPSG:2313"
    }, {
        "description" : "Trinidad 1903 / Trinidad Grid (ftCla)",
        "code" : "EPSG:2314"
    }, {
        "description" : "Campo Inchauspe / UTM zone 19S",
        "code" : "EPSG:2315"
    }, {
        "description" : "Campo Inchauspe / UTM zone 20S",
        "code" : "EPSG:2316"
    }, {
        "description" : "PSAD56 / ICN Regional",
        "code" : "EPSG:2317"
    }, {
        "description" : "Ain el Abd / Aramco Lambert",
        "code" : "EPSG:2318"
    }, {
        "description" : "ED50 / TM27",
        "code" : "EPSG:2319"
    }, {
        "description" : "ED50 / TM30",
        "code" : "EPSG:2320"
    }, {
        "description" : "ED50 / TM33",
        "code" : "EPSG:2321"
    }, {
        "description" : "ED50 / TM36",
        "code" : "EPSG:2322"
    }, {
        "description" : "ED50 / TM39",
        "code" : "EPSG:2323"
    }, {
        "description" : "ED50 / TM42",
        "code" : "EPSG:2324"
    }, {
        "description" : "ED50 / TM45",
        "code" : "EPSG:2325"
    }, {
        "description" : "Hong Kong 1980 Grid System",
        "code" : "EPSG:2326"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger zone 13",
        "code" : "EPSG:2327"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger zone 14",
        "code" : "EPSG:2328"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger zone 15",
        "code" : "EPSG:2329"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger zone 16",
        "code" : "EPSG:2330"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger zone 17",
        "code" : "EPSG:2331"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger zone 18",
        "code" : "EPSG:2332"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger zone 19",
        "code" : "EPSG:2333"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger zone 20",
        "code" : "EPSG:2334"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger zone 21",
        "code" : "EPSG:2335"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger zone 22",
        "code" : "EPSG:2336"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger zone 23",
        "code" : "EPSG:2337"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger CM 75E",
        "code" : "EPSG:2338"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger CM 81E",
        "code" : "EPSG:2339"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger CM 87E",
        "code" : "EPSG:2340"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger CM 93E",
        "code" : "EPSG:2341"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger CM 99E",
        "code" : "EPSG:2342"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger CM 105E",
        "code" : "EPSG:2343"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger CM 111E",
        "code" : "EPSG:2344"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger CM 117E",
        "code" : "EPSG:2345"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger CM 123E",
        "code" : "EPSG:2346"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger CM 129E",
        "code" : "EPSG:2347"
    }, {
        "description" : "Xian 1980 / Gauss-Kruger CM 135E",
        "code" : "EPSG:2348"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 25",
        "code" : "EPSG:2349"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 26",
        "code" : "EPSG:2350"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 27",
        "code" : "EPSG:2351"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 28",
        "code" : "EPSG:2352"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 29",
        "code" : "EPSG:2353"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 30",
        "code" : "EPSG:2354"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 31",
        "code" : "EPSG:2355"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 32",
        "code" : "EPSG:2356"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 33",
        "code" : "EPSG:2357"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 34",
        "code" : "EPSG:2358"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 35",
        "code" : "EPSG:2359"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 36",
        "code" : "EPSG:2360"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 37",
        "code" : "EPSG:2361"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 38",
        "code" : "EPSG:2362"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 39",
        "code" : "EPSG:2363"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 40",
        "code" : "EPSG:2364"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 41",
        "code" : "EPSG:2365"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 42",
        "code" : "EPSG:2366"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 43",
        "code" : "EPSG:2367"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 44",
        "code" : "EPSG:2368"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger zone 45",
        "code" : "EPSG:2369"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 75E",
        "code" : "EPSG:2370"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 78E",
        "code" : "EPSG:2371"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 81E",
        "code" : "EPSG:2372"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 84E",
        "code" : "EPSG:2373"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 87E",
        "code" : "EPSG:2374"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 90E",
        "code" : "EPSG:2375"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 93E",
        "code" : "EPSG:2376"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 96E",
        "code" : "EPSG:2377"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 99E",
        "code" : "EPSG:2378"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 102E",
        "code" : "EPSG:2379"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 105E",
        "code" : "EPSG:2380"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 108E",
        "code" : "EPSG:2381"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 111E",
        "code" : "EPSG:2382"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 114E",
        "code" : "EPSG:2383"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 117E",
        "code" : "EPSG:2384"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 120E",
        "code" : "EPSG:2385"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 123E",
        "code" : "EPSG:2386"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 126E",
        "code" : "EPSG:2387"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 129E",
        "code" : "EPSG:2388"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 132E",
        "code" : "EPSG:2389"
    }, {
        "description" : "Xian 1980 / 3-degree Gauss-Kruger CM 135E",
        "code" : "EPSG:2390"
    }, {
        "description" : "KKJ / Finland zone 1",
        "code" : "EPSG:2391"
    }, {
        "description" : "KKJ / Finland zone 2",
        "code" : "EPSG:2392"
    }, {
        "description" : "KKJ / Finland Uniform Coordinate System",
        "code" : "EPSG:2393"
    }, {
        "description" : "KKJ / Finland zone 4",
        "code" : "EPSG:2394"
    }, {
        "description" : "South Yemen / Gauss-Kruger zone 8",
        "code" : "EPSG:2395"
    }, {
        "description" : "South Yemen / Gauss-Kruger zone 9",
        "code" : "EPSG:2396"
    }, {
        "description" : "Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 3",
        "code" : "EPSG:2397"
    }, {
        "description" : "Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 4",
        "code" : "EPSG:2398"
    }, {
        "description" : "Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 5",
        "code" : "EPSG:2399"
    }, {
        "description" : "RT90 2.5 gon W",
        "code" : "EPSG:2400"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 25",
        "code" : "EPSG:2401"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 26",
        "code" : "EPSG:2402"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 27",
        "code" : "EPSG:2403"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 28",
        "code" : "EPSG:2404"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 29",
        "code" : "EPSG:2405"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 30",
        "code" : "EPSG:2406"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 31",
        "code" : "EPSG:2407"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 32",
        "code" : "EPSG:2408"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 33",
        "code" : "EPSG:2409"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 34",
        "code" : "EPSG:2410"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 35",
        "code" : "EPSG:2411"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 36",
        "code" : "EPSG:2412"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 37",
        "code" : "EPSG:2413"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 38",
        "code" : "EPSG:2414"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 39",
        "code" : "EPSG:2415"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 40",
        "code" : "EPSG:2416"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 41",
        "code" : "EPSG:2417"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 42",
        "code" : "EPSG:2418"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 43",
        "code" : "EPSG:2419"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 44",
        "code" : "EPSG:2420"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger zone 45",
        "code" : "EPSG:2421"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 75E",
        "code" : "EPSG:2422"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 78E",
        "code" : "EPSG:2423"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 81E",
        "code" : "EPSG:2424"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 84E",
        "code" : "EPSG:2425"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 87E",
        "code" : "EPSG:2426"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 90E",
        "code" : "EPSG:2427"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 93E",
        "code" : "EPSG:2428"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 96E",
        "code" : "EPSG:2429"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 99E",
        "code" : "EPSG:2430"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 102E",
        "code" : "EPSG:2431"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 105E",
        "code" : "EPSG:2432"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 108E",
        "code" : "EPSG:2433"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 111E",
        "code" : "EPSG:2434"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 114E",
        "code" : "EPSG:2435"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 117E",
        "code" : "EPSG:2436"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 120E",
        "code" : "EPSG:2437"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 123E",
        "code" : "EPSG:2438"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 126E",
        "code" : "EPSG:2439"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 129E",
        "code" : "EPSG:2440"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 132E",
        "code" : "EPSG:2441"
    }, {
        "description" : "Beijing 1954 / 3-degree Gauss-Kruger CM 135E",
        "code" : "EPSG:2442"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS I",
        "code" : "EPSG:2443"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS II",
        "code" : "EPSG:2444"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS III",
        "code" : "EPSG:2445"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS IV",
        "code" : "EPSG:2446"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS V",
        "code" : "EPSG:2447"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS VI",
        "code" : "EPSG:2448"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS VII",
        "code" : "EPSG:2449"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS VIII",
        "code" : "EPSG:2450"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS IX",
        "code" : "EPSG:2451"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS X",
        "code" : "EPSG:2452"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS XI",
        "code" : "EPSG:2453"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS XII",
        "code" : "EPSG:2454"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS XIII",
        "code" : "EPSG:2455"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS XIV",
        "code" : "EPSG:2456"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS XV",
        "code" : "EPSG:2457"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS XVI",
        "code" : "EPSG:2458"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS XVII",
        "code" : "EPSG:2459"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS XVIII",
        "code" : "EPSG:2460"
    }, {
        "description" : "JGD2000 / Japan Plane Rectangular CS XIX",
        "code" : "EPSG:2461"
    }, {
        "description" : "Albanian 1987 / Gauss-Kruger zone 4",
        "code" : "EPSG:2462"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 21E",
        "code" : "EPSG:2463"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 27E",
        "code" : "EPSG:2464"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 33E",
        "code" : "EPSG:2465"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 39E",
        "code" : "EPSG:2466"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 45E",
        "code" : "EPSG:2467"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 51E",
        "code" : "EPSG:2468"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 57E",
        "code" : "EPSG:2469"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 63E",
        "code" : "EPSG:2470"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 69E",
        "code" : "EPSG:2471"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 75E",
        "code" : "EPSG:2472"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 81E",
        "code" : "EPSG:2473"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 87E",
        "code" : "EPSG:2474"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 93E",
        "code" : "EPSG:2475"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 99E",
        "code" : "EPSG:2476"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 105E",
        "code" : "EPSG:2477"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 111E",
        "code" : "EPSG:2478"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 117E",
        "code" : "EPSG:2479"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 123E",
        "code" : "EPSG:2480"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 129E",
        "code" : "EPSG:2481"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 135E",
        "code" : "EPSG:2482"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 141E",
        "code" : "EPSG:2483"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 147E",
        "code" : "EPSG:2484"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 153E",
        "code" : "EPSG:2485"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 159E",
        "code" : "EPSG:2486"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 165E",
        "code" : "EPSG:2487"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 171E",
        "code" : "EPSG:2488"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 177E",
        "code" : "EPSG:2489"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 177W",
        "code" : "EPSG:2490"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger CM 171W",
        "code" : "EPSG:2491"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 9E",
        "code" : "EPSG:2492"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 15E",
        "code" : "EPSG:2493"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 21E",
        "code" : "EPSG:2494"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 27E",
        "code" : "EPSG:2495"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 33E",
        "code" : "EPSG:2496"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 39E",
        "code" : "EPSG:2497"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 45E",
        "code" : "EPSG:2498"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 51E",
        "code" : "EPSG:2499"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 57E",
        "code" : "EPSG:2500"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 63E",
        "code" : "EPSG:2501"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 69E",
        "code" : "EPSG:2502"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 75E",
        "code" : "EPSG:2503"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 81E",
        "code" : "EPSG:2504"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 87E",
        "code" : "EPSG:2505"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 93E",
        "code" : "EPSG:2506"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 99E",
        "code" : "EPSG:2507"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 105E",
        "code" : "EPSG:2508"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 111E",
        "code" : "EPSG:2509"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 117E",
        "code" : "EPSG:2510"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 123E",
        "code" : "EPSG:2511"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 129E",
        "code" : "EPSG:2512"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 135E",
        "code" : "EPSG:2513"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 141E",
        "code" : "EPSG:2514"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 147E",
        "code" : "EPSG:2515"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 153E",
        "code" : "EPSG:2516"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 159E",
        "code" : "EPSG:2517"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 165E",
        "code" : "EPSG:2518"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 171E",
        "code" : "EPSG:2519"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 177E",
        "code" : "EPSG:2520"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 177W",
        "code" : "EPSG:2521"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger CM 171W",
        "code" : "EPSG:2522"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 7",
        "code" : "EPSG:2523"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 8",
        "code" : "EPSG:2524"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 9",
        "code" : "EPSG:2525"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 10",
        "code" : "EPSG:2526"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 11",
        "code" : "EPSG:2527"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 12",
        "code" : "EPSG:2528"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 13",
        "code" : "EPSG:2529"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 14",
        "code" : "EPSG:2530"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 15",
        "code" : "EPSG:2531"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 16",
        "code" : "EPSG:2532"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 17",
        "code" : "EPSG:2533"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 18",
        "code" : "EPSG:2534"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 19",
        "code" : "EPSG:2535"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 20",
        "code" : "EPSG:2536"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 21",
        "code" : "EPSG:2537"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 22",
        "code" : "EPSG:2538"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 23",
        "code" : "EPSG:2539"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 24",
        "code" : "EPSG:2540"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 25",
        "code" : "EPSG:2541"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 26",
        "code" : "EPSG:2542"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 27",
        "code" : "EPSG:2543"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 28",
        "code" : "EPSG:2544"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 29",
        "code" : "EPSG:2545"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 30",
        "code" : "EPSG:2546"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 31",
        "code" : "EPSG:2547"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 32",
        "code" : "EPSG:2548"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 33",
        "code" : "EPSG:2549"
    }, {
        "description" : "Samboja / UTM zone 50S",
        "code" : "EPSG:2550"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 34",
        "code" : "EPSG:2551"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 35",
        "code" : "EPSG:2552"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 36",
        "code" : "EPSG:2553"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 37",
        "code" : "EPSG:2554"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 38",
        "code" : "EPSG:2555"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 39",
        "code" : "EPSG:2556"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 40",
        "code" : "EPSG:2557"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 41",
        "code" : "EPSG:2558"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 42",
        "code" : "EPSG:2559"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 43",
        "code" : "EPSG:2560"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 44",
        "code" : "EPSG:2561"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 45",
        "code" : "EPSG:2562"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 46",
        "code" : "EPSG:2563"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 47",
        "code" : "EPSG:2564"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 48",
        "code" : "EPSG:2565"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 49",
        "code" : "EPSG:2566"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 50",
        "code" : "EPSG:2567"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 51",
        "code" : "EPSG:2568"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 52",
        "code" : "EPSG:2569"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 53",
        "code" : "EPSG:2570"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 54",
        "code" : "EPSG:2571"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 55",
        "code" : "EPSG:2572"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 56",
        "code" : "EPSG:2573"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 57",
        "code" : "EPSG:2574"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 58",
        "code" : "EPSG:2575"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 59",
        "code" : "EPSG:2576"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 60",
        "code" : "EPSG:2577"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 61",
        "code" : "EPSG:2578"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 62",
        "code" : "EPSG:2579"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 63",
        "code" : "EPSG:2580"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 64",
        "code" : "EPSG:2581"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 21E",
        "code" : "EPSG:2582"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 24E",
        "code" : "EPSG:2583"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 27E",
        "code" : "EPSG:2584"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 30E",
        "code" : "EPSG:2585"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 33E",
        "code" : "EPSG:2586"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 36E",
        "code" : "EPSG:2587"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 39E",
        "code" : "EPSG:2588"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 42E",
        "code" : "EPSG:2589"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 45E",
        "code" : "EPSG:2590"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 48E",
        "code" : "EPSG:2591"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 51E",
        "code" : "EPSG:2592"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 54E",
        "code" : "EPSG:2593"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 57E",
        "code" : "EPSG:2594"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 60E",
        "code" : "EPSG:2595"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 63E",
        "code" : "EPSG:2596"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 66E",
        "code" : "EPSG:2597"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 69E",
        "code" : "EPSG:2598"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 72E",
        "code" : "EPSG:2599"
    }, {
        "description" : "Lietuvos Koordinoei Sistema 1994",
        "code" : "EPSG:2600"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 75E",
        "code" : "EPSG:2601"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 78E",
        "code" : "EPSG:2602"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 81E",
        "code" : "EPSG:2603"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 84E",
        "code" : "EPSG:2604"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 87E",
        "code" : "EPSG:2605"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 90E",
        "code" : "EPSG:2606"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 93E",
        "code" : "EPSG:2607"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 96E",
        "code" : "EPSG:2608"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 99E",
        "code" : "EPSG:2609"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 102E",
        "code" : "EPSG:2610"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 105E",
        "code" : "EPSG:2611"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 108E",
        "code" : "EPSG:2612"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 111E",
        "code" : "EPSG:2613"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 114E",
        "code" : "EPSG:2614"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 117E",
        "code" : "EPSG:2615"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 120E",
        "code" : "EPSG:2616"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 123E",
        "code" : "EPSG:2617"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 126E",
        "code" : "EPSG:2618"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 129E",
        "code" : "EPSG:2619"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 132E",
        "code" : "EPSG:2620"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 135E",
        "code" : "EPSG:2621"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 138E",
        "code" : "EPSG:2622"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 141E",
        "code" : "EPSG:2623"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 144E",
        "code" : "EPSG:2624"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 147E",
        "code" : "EPSG:2625"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 150E",
        "code" : "EPSG:2626"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 153E",
        "code" : "EPSG:2627"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 156E",
        "code" : "EPSG:2628"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 159E",
        "code" : "EPSG:2629"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 162E",
        "code" : "EPSG:2630"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 165E",
        "code" : "EPSG:2631"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 168E",
        "code" : "EPSG:2632"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 171E",
        "code" : "EPSG:2633"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 174E",
        "code" : "EPSG:2634"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 177E",
        "code" : "EPSG:2635"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 180E",
        "code" : "EPSG:2636"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 177W",
        "code" : "EPSG:2637"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 174W",
        "code" : "EPSG:2638"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 171W",
        "code" : "EPSG:2639"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 168W",
        "code" : "EPSG:2640"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 7",
        "code" : "EPSG:2641"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 8",
        "code" : "EPSG:2642"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 9",
        "code" : "EPSG:2643"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 10",
        "code" : "EPSG:2644"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 11",
        "code" : "EPSG:2645"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 12",
        "code" : "EPSG:2646"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 13",
        "code" : "EPSG:2647"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 14",
        "code" : "EPSG:2648"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 15",
        "code" : "EPSG:2649"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 16",
        "code" : "EPSG:2650"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 17",
        "code" : "EPSG:2651"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 18",
        "code" : "EPSG:2652"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 19",
        "code" : "EPSG:2653"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 20",
        "code" : "EPSG:2654"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 21",
        "code" : "EPSG:2655"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 22",
        "code" : "EPSG:2656"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 23",
        "code" : "EPSG:2657"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 24",
        "code" : "EPSG:2658"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 25",
        "code" : "EPSG:2659"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 26",
        "code" : "EPSG:2660"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 27",
        "code" : "EPSG:2661"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 28",
        "code" : "EPSG:2662"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 29",
        "code" : "EPSG:2663"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 30",
        "code" : "EPSG:2664"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 31",
        "code" : "EPSG:2665"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 32",
        "code" : "EPSG:2666"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 33",
        "code" : "EPSG:2667"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 34",
        "code" : "EPSG:2668"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 35",
        "code" : "EPSG:2669"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 36",
        "code" : "EPSG:2670"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 37",
        "code" : "EPSG:2671"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 38",
        "code" : "EPSG:2672"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 39",
        "code" : "EPSG:2673"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 40",
        "code" : "EPSG:2674"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 41",
        "code" : "EPSG:2675"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 42",
        "code" : "EPSG:2676"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 43",
        "code" : "EPSG:2677"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 44",
        "code" : "EPSG:2678"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 45",
        "code" : "EPSG:2679"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 46",
        "code" : "EPSG:2680"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 47",
        "code" : "EPSG:2681"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 48",
        "code" : "EPSG:2682"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 49",
        "code" : "EPSG:2683"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 50",
        "code" : "EPSG:2684"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 51",
        "code" : "EPSG:2685"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 52",
        "code" : "EPSG:2686"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 53",
        "code" : "EPSG:2687"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 54",
        "code" : "EPSG:2688"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 55",
        "code" : "EPSG:2689"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 56",
        "code" : "EPSG:2690"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 57",
        "code" : "EPSG:2691"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 58",
        "code" : "EPSG:2692"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 59",
        "code" : "EPSG:2693"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 60",
        "code" : "EPSG:2694"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 61",
        "code" : "EPSG:2695"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 62",
        "code" : "EPSG:2696"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 63",
        "code" : "EPSG:2697"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 64",
        "code" : "EPSG:2698"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 21E",
        "code" : "EPSG:2699"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 24E",
        "code" : "EPSG:2700"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 27E",
        "code" : "EPSG:2701"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 30E",
        "code" : "EPSG:2702"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 33E",
        "code" : "EPSG:2703"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 36E",
        "code" : "EPSG:2704"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 39E",
        "code" : "EPSG:2705"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 42E",
        "code" : "EPSG:2706"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 45E",
        "code" : "EPSG:2707"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 48E",
        "code" : "EPSG:2708"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 51E",
        "code" : "EPSG:2709"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 54E",
        "code" : "EPSG:2710"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 57E",
        "code" : "EPSG:2711"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 60E",
        "code" : "EPSG:2712"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 63E",
        "code" : "EPSG:2713"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 66E",
        "code" : "EPSG:2714"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 69E",
        "code" : "EPSG:2715"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 72E",
        "code" : "EPSG:2716"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 75E",
        "code" : "EPSG:2717"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 78E",
        "code" : "EPSG:2718"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 81E",
        "code" : "EPSG:2719"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 84E",
        "code" : "EPSG:2720"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 87E",
        "code" : "EPSG:2721"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 90E",
        "code" : "EPSG:2722"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 93E",
        "code" : "EPSG:2723"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 96E",
        "code" : "EPSG:2724"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 99E",
        "code" : "EPSG:2725"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 102E",
        "code" : "EPSG:2726"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 105E",
        "code" : "EPSG:2727"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 108E",
        "code" : "EPSG:2728"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 111E",
        "code" : "EPSG:2729"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 114E",
        "code" : "EPSG:2730"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 117E",
        "code" : "EPSG:2731"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 120E",
        "code" : "EPSG:2732"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 123E",
        "code" : "EPSG:2733"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 126E",
        "code" : "EPSG:2734"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 129E",
        "code" : "EPSG:2735"
    }, {
        "description" : "Tete / UTM zone 36S",
        "code" : "EPSG:2736"
    }, {
        "description" : "Tete / UTM zone 37S",
        "code" : "EPSG:2737"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 132E",
        "code" : "EPSG:2738"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 135E",
        "code" : "EPSG:2739"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 138E",
        "code" : "EPSG:2740"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 141E",
        "code" : "EPSG:2741"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 144E",
        "code" : "EPSG:2742"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 147E",
        "code" : "EPSG:2743"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 150E",
        "code" : "EPSG:2744"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 153E",
        "code" : "EPSG:2745"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 156E",
        "code" : "EPSG:2746"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 159E",
        "code" : "EPSG:2747"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 162E",
        "code" : "EPSG:2748"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 165E",
        "code" : "EPSG:2749"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 168E",
        "code" : "EPSG:2750"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 171E",
        "code" : "EPSG:2751"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 174E",
        "code" : "EPSG:2752"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 177E",
        "code" : "EPSG:2753"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 180E",
        "code" : "EPSG:2754"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 177W",
        "code" : "EPSG:2755"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 174W",
        "code" : "EPSG:2756"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 171W",
        "code" : "EPSG:2757"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 168W",
        "code" : "EPSG:2758"
    }, {
        "description" : "NAD83(HARN) / Alabama East",
        "code" : "EPSG:2759"
    }, {
        "description" : "NAD83(HARN) / Alabama West",
        "code" : "EPSG:2760"
    }, {
        "description" : "NAD83(HARN) / Arizona East",
        "code" : "EPSG:2761"
    }, {
        "description" : "NAD83(HARN) / Arizona Central",
        "code" : "EPSG:2762"
    }, {
        "description" : "NAD83(HARN) / Arizona West",
        "code" : "EPSG:2763"
    }, {
        "description" : "NAD83(HARN) / Arkansas North",
        "code" : "EPSG:2764"
    }, {
        "description" : "NAD83(HARN) / Arkansas South",
        "code" : "EPSG:2765"
    }, {
        "description" : "NAD83(HARN) / California zone 1",
        "code" : "EPSG:2766"
    }, {
        "description" : "NAD83(HARN) / California zone 2",
        "code" : "EPSG:2767"
    }, {
        "description" : "NAD83(HARN) / California zone 3",
        "code" : "EPSG:2768"
    }, {
        "description" : "NAD83(HARN) / California zone 4",
        "code" : "EPSG:2769"
    }, {
        "description" : "NAD83(HARN) / California zone 5",
        "code" : "EPSG:2770"
    }, {
        "description" : "NAD83(HARN) / California zone 6",
        "code" : "EPSG:2771"
    }, {
        "description" : "NAD83(HARN) / Colorado North",
        "code" : "EPSG:2772"
    }, {
        "description" : "NAD83(HARN) / Colorado Central",
        "code" : "EPSG:2773"
    }, {
        "description" : "NAD83(HARN) / Colorado South",
        "code" : "EPSG:2774"
    }, {
        "description" : "NAD83(HARN) / Connecticut",
        "code" : "EPSG:2775"
    }, {
        "description" : "NAD83(HARN) / Delaware",
        "code" : "EPSG:2776"
    }, {
        "description" : "NAD83(HARN) / Florida East",
        "code" : "EPSG:2777"
    }, {
        "description" : "NAD83(HARN) / Florida West",
        "code" : "EPSG:2778"
    }, {
        "description" : "NAD83(HARN) / Florida North",
        "code" : "EPSG:2779"
    }, {
        "description" : "NAD83(HARN) / Georgia East",
        "code" : "EPSG:2780"
    }, {
        "description" : "NAD83(HARN) / Georgia West",
        "code" : "EPSG:2781"
    }, {
        "description" : "NAD83(HARN) / Hawaii zone 1",
        "code" : "EPSG:2782"
    }, {
        "description" : "NAD83(HARN) / Hawaii zone 2",
        "code" : "EPSG:2783"
    }, {
        "description" : "NAD83(HARN) / Hawaii zone 3",
        "code" : "EPSG:2784"
    }, {
        "description" : "NAD83(HARN) / Hawaii zone 4",
        "code" : "EPSG:2785"
    }, {
        "description" : "NAD83(HARN) / Hawaii zone 5",
        "code" : "EPSG:2786"
    }, {
        "description" : "NAD83(HARN) / Idaho East",
        "code" : "EPSG:2787"
    }, {
        "description" : "NAD83(HARN) / Idaho Central",
        "code" : "EPSG:2788"
    }, {
        "description" : "NAD83(HARN) / Idaho West",
        "code" : "EPSG:2789"
    }, {
        "description" : "NAD83(HARN) / Illinois East",
        "code" : "EPSG:2790"
    }, {
        "description" : "NAD83(HARN) / Illinois West",
        "code" : "EPSG:2791"
    }, {
        "description" : "NAD83(HARN) / Indiana East",
        "code" : "EPSG:2792"
    }, {
        "description" : "NAD83(HARN) / Indiana West",
        "code" : "EPSG:2793"
    }, {
        "description" : "NAD83(HARN) / Iowa North",
        "code" : "EPSG:2794"
    }, {
        "description" : "NAD83(HARN) / Iowa South",
        "code" : "EPSG:2795"
    }, {
        "description" : "NAD83(HARN) / Kansas North",
        "code" : "EPSG:2796"
    }, {
        "description" : "NAD83(HARN) / Kansas South",
        "code" : "EPSG:2797"
    }, {
        "description" : "NAD83(HARN) / Kentucky North",
        "code" : "EPSG:2798"
    }, {
        "description" : "NAD83(HARN) / Kentucky South",
        "code" : "EPSG:2799"
    }, {
        "description" : "NAD83(HARN) / Louisiana North",
        "code" : "EPSG:2800"
    }, {
        "description" : "NAD83(HARN) / Louisiana South",
        "code" : "EPSG:2801"
    }, {
        "description" : "NAD83(HARN) / Maine East",
        "code" : "EPSG:2802"
    }, {
        "description" : "NAD83(HARN) / Maine West",
        "code" : "EPSG:2803"
    }, {
        "description" : "NAD83(HARN) / Maryland",
        "code" : "EPSG:2804"
    }, {
        "description" : "NAD83(HARN) / Massachusetts Mainland",
        "code" : "EPSG:2805"
    }, {
        "description" : "NAD83(HARN) / Massachusetts Island",
        "code" : "EPSG:2806"
    }, {
        "description" : "NAD83(HARN) / Michigan North",
        "code" : "EPSG:2807"
    }, {
        "description" : "NAD83(HARN) / Michigan Central",
        "code" : "EPSG:2808"
    }, {
        "description" : "NAD83(HARN) / Michigan South",
        "code" : "EPSG:2809"
    }, {
        "description" : "NAD83(HARN) / Minnesota North",
        "code" : "EPSG:2810"
    }, {
        "description" : "NAD83(HARN) / Minnesota Central",
        "code" : "EPSG:2811"
    }, {
        "description" : "NAD83(HARN) / Minnesota South",
        "code" : "EPSG:2812"
    }, {
        "description" : "NAD83(HARN) / Mississippi East",
        "code" : "EPSG:2813"
    }, {
        "description" : "NAD83(HARN) / Mississippi West",
        "code" : "EPSG:2814"
    }, {
        "description" : "NAD83(HARN) / Missouri East",
        "code" : "EPSG:2815"
    }, {
        "description" : "NAD83(HARN) / Missouri Central",
        "code" : "EPSG:2816"
    }, {
        "description" : "NAD83(HARN) / Missouri West",
        "code" : "EPSG:2817"
    }, {
        "description" : "NAD83(HARN) / Montana",
        "code" : "EPSG:2818"
    }, {
        "description" : "NAD83(HARN) / Nebraska",
        "code" : "EPSG:2819"
    }, {
        "description" : "NAD83(HARN) / Nevada East",
        "code" : "EPSG:2820"
    }, {
        "description" : "NAD83(HARN) / Nevada Central",
        "code" : "EPSG:2821"
    }, {
        "description" : "NAD83(HARN) / Nevada West",
        "code" : "EPSG:2822"
    }, {
        "description" : "NAD83(HARN) / New Hampshire",
        "code" : "EPSG:2823"
    }, {
        "description" : "NAD83(HARN) / New Jersey",
        "code" : "EPSG:2824"
    }, {
        "description" : "NAD83(HARN) / New Mexico East",
        "code" : "EPSG:2825"
    }, {
        "description" : "NAD83(HARN) / New Mexico Central",
        "code" : "EPSG:2826"
    }, {
        "description" : "NAD83(HARN) / New Mexico West",
        "code" : "EPSG:2827"
    }, {
        "description" : "NAD83(HARN) / New York East",
        "code" : "EPSG:2828"
    }, {
        "description" : "NAD83(HARN) / New York Central",
        "code" : "EPSG:2829"
    }, {
        "description" : "NAD83(HARN) / New York West",
        "code" : "EPSG:2830"
    }, {
        "description" : "NAD83(HARN) / New York Long Island",
        "code" : "EPSG:2831"
    }, {
        "description" : "NAD83(HARN) / North Dakota North",
        "code" : "EPSG:2832"
    }, {
        "description" : "NAD83(HARN) / North Dakota South",
        "code" : "EPSG:2833"
    }, {
        "description" : "NAD83(HARN) / Ohio North",
        "code" : "EPSG:2834"
    }, {
        "description" : "NAD83(HARN) / Ohio South",
        "code" : "EPSG:2835"
    }, {
        "description" : "NAD83(HARN) / Oklahoma North",
        "code" : "EPSG:2836"
    }, {
        "description" : "NAD83(HARN) / Oklahoma South",
        "code" : "EPSG:2837"
    }, {
        "description" : "NAD83(HARN) / Oregon North",
        "code" : "EPSG:2838"
    }, {
        "description" : "NAD83(HARN) / Oregon South",
        "code" : "EPSG:2839"
    }, {
        "description" : "NAD83(HARN) / Rhode Island",
        "code" : "EPSG:2840"
    }, {
        "description" : "NAD83(HARN) / South Dakota North",
        "code" : "EPSG:2841"
    }, {
        "description" : "NAD83(HARN) / South Dakota South",
        "code" : "EPSG:2842"
    }, {
        "description" : "NAD83(HARN) / Tennessee",
        "code" : "EPSG:2843"
    }, {
        "description" : "NAD83(HARN) / Texas North",
        "code" : "EPSG:2844"
    }, {
        "description" : "NAD83(HARN) / Texas North Central",
        "code" : "EPSG:2845"
    }, {
        "description" : "NAD83(HARN) / Texas Central",
        "code" : "EPSG:2846"
    }, {
        "description" : "NAD83(HARN) / Texas South Central",
        "code" : "EPSG:2847"
    }, {
        "description" : "NAD83(HARN) / Texas South",
        "code" : "EPSG:2848"
    }, {
        "description" : "NAD83(HARN) / Utah North",
        "code" : "EPSG:2849"
    }, {
        "description" : "NAD83(HARN) / Utah Central",
        "code" : "EPSG:2850"
    }, {
        "description" : "NAD83(HARN) / Utah South",
        "code" : "EPSG:2851"
    }, {
        "description" : "NAD83(HARN) / Vermont",
        "code" : "EPSG:2852"
    }, {
        "description" : "NAD83(HARN) / Virginia North",
        "code" : "EPSG:2853"
    }, {
        "description" : "NAD83(HARN) / Virginia South",
        "code" : "EPSG:2854"
    }, {
        "description" : "NAD83(HARN) / Washington North",
        "code" : "EPSG:2855"
    }, {
        "description" : "NAD83(HARN) / Washington South",
        "code" : "EPSG:2856"
    }, {
        "description" : "NAD83(HARN) / West Virginia North",
        "code" : "EPSG:2857"
    }, {
        "description" : "NAD83(HARN) / West Virginia South",
        "code" : "EPSG:2858"
    }, {
        "description" : "NAD83(HARN) / Wisconsin North",
        "code" : "EPSG:2859"
    }, {
        "description" : "NAD83(HARN) / Wisconsin Central",
        "code" : "EPSG:2860"
    }, {
        "description" : "NAD83(HARN) / Wisconsin South",
        "code" : "EPSG:2861"
    }, {
        "description" : "NAD83(HARN) / Wyoming East",
        "code" : "EPSG:2862"
    }, {
        "description" : "NAD83(HARN) / Wyoming East Central",
        "code" : "EPSG:2863"
    }, {
        "description" : "NAD83(HARN) / Wyoming West Central",
        "code" : "EPSG:2864"
    }, {
        "description" : "NAD83(HARN) / Wyoming West",
        "code" : "EPSG:2865"
    }, {
        "description" : "NAD83(HARN) / Puerto Rico and Virgin Is.",
        "code" : "EPSG:2866"
    }, {
        "description" : "NAD83(HARN) / Arizona East (ft)",
        "code" : "EPSG:2867"
    }, {
        "description" : "NAD83(HARN) / Arizona Central (ft)",
        "code" : "EPSG:2868"
    }, {
        "description" : "NAD83(HARN) / Arizona West (ft)",
        "code" : "EPSG:2869"
    }, {
        "description" : "NAD83(HARN) / California zone 1 (ftUS)",
        "code" : "EPSG:2870"
    }, {
        "description" : "NAD83(HARN) / California zone 2 (ftUS)",
        "code" : "EPSG:2871"
    }, {
        "description" : "NAD83(HARN) / California zone 3 (ftUS)",
        "code" : "EPSG:2872"
    }, {
        "description" : "NAD83(HARN) / California zone 4 (ftUS)",
        "code" : "EPSG:2873"
    }, {
        "description" : "NAD83(HARN) / California zone 5 (ftUS)",
        "code" : "EPSG:2874"
    }, {
        "description" : "NAD83(HARN) / California zone 6 (ftUS)",
        "code" : "EPSG:2875"
    }, {
        "description" : "NAD83(HARN) / Colorado North (ftUS)",
        "code" : "EPSG:2876"
    }, {
        "description" : "NAD83(HARN) / Colorado Central (ftUS)",
        "code" : "EPSG:2877"
    }, {
        "description" : "NAD83(HARN) / Colorado South (ftUS)",
        "code" : "EPSG:2878"
    }, {
        "description" : "NAD83(HARN) / Connecticut (ftUS)",
        "code" : "EPSG:2879"
    }, {
        "description" : "NAD83(HARN) / Delaware (ftUS)",
        "code" : "EPSG:2880"
    }, {
        "description" : "NAD83(HARN) / Florida East (ftUS)",
        "code" : "EPSG:2881"
    }, {
        "description" : "NAD83(HARN) / Florida West (ftUS)",
        "code" : "EPSG:2882"
    }, {
        "description" : "NAD83(HARN) / Florida North (ftUS)",
        "code" : "EPSG:2883"
    }, {
        "description" : "NAD83(HARN) / Georgia East (ftUS)",
        "code" : "EPSG:2884"
    }, {
        "description" : "NAD83(HARN) / Georgia West (ftUS)",
        "code" : "EPSG:2885"
    }, {
        "description" : "NAD83(HARN) / Idaho East (ftUS)",
        "code" : "EPSG:2886"
    }, {
        "description" : "NAD83(HARN) / Idaho Central (ftUS)",
        "code" : "EPSG:2887"
    }, {
        "description" : "NAD83(HARN) / Idaho West (ftUS)",
        "code" : "EPSG:2888"
    }, {
        "description" : "NAD83(HARN) / Indiana East (ftUS)",
        "code" : "EPSG:2889"
    }, {
        "description" : "NAD83(HARN) / Indiana West (ftUS)",
        "code" : "EPSG:2890"
    }, {
        "description" : "NAD83(HARN) / Kentucky North (ftUS)",
        "code" : "EPSG:2891"
    }, {
        "description" : "NAD83(HARN) / Kentucky South (ftUS)",
        "code" : "EPSG:2892"
    }, {
        "description" : "NAD83(HARN) / Maryland (ftUS)",
        "code" : "EPSG:2893"
    }, {
        "description" : "NAD83(HARN) / Massachusetts Mainland (ftUS)",
        "code" : "EPSG:2894"
    }, {
        "description" : "NAD83(HARN) / Massachusetts Island (ftUS)",
        "code" : "EPSG:2895"
    }, {
        "description" : "NAD83(HARN) / Michigan North (ft)",
        "code" : "EPSG:2896"
    }, {
        "description" : "NAD83(HARN) / Michigan Central (ft)",
        "code" : "EPSG:2897"
    }, {
        "description" : "NAD83(HARN) / Michigan South (ft)",
        "code" : "EPSG:2898"
    }, {
        "description" : "NAD83(HARN) / Mississippi East (ftUS)",
        "code" : "EPSG:2899"
    }, {
        "description" : "NAD83(HARN) / Mississippi West (ftUS)",
        "code" : "EPSG:2900"
    }, {
        "description" : "NAD83(HARN) / Montana (ft)",
        "code" : "EPSG:2901"
    }, {
        "description" : "NAD83(HARN) / New Mexico East (ftUS)",
        "code" : "EPSG:2902"
    }, {
        "description" : "NAD83(HARN) / New Mexico Central (ftUS)",
        "code" : "EPSG:2903"
    }, {
        "description" : "NAD83(HARN) / New Mexico West (ftUS)",
        "code" : "EPSG:2904"
    }, {
        "description" : "NAD83(HARN) / New York East (ftUS)",
        "code" : "EPSG:2905"
    }, {
        "description" : "NAD83(HARN) / New York Central (ftUS)",
        "code" : "EPSG:2906"
    }, {
        "description" : "NAD83(HARN) / New York West (ftUS)",
        "code" : "EPSG:2907"
    }, {
        "description" : "NAD83(HARN) / New York Long Island (ftUS)",
        "code" : "EPSG:2908"
    }, {
        "description" : "NAD83(HARN) / North Dakota North (ft)",
        "code" : "EPSG:2909"
    }, {
        "description" : "NAD83(HARN) / North Dakota South (ft)",
        "code" : "EPSG:2910"
    }, {
        "description" : "NAD83(HARN) / Oklahoma North (ftUS)",
        "code" : "EPSG:2911"
    }, {
        "description" : "NAD83(HARN) / Oklahoma South (ftUS)",
        "code" : "EPSG:2912"
    }, {
        "description" : "NAD83(HARN) / Oregon North (ft)",
        "code" : "EPSG:2913"
    }, {
        "description" : "NAD83(HARN) / Oregon South (ft)",
        "code" : "EPSG:2914"
    }, {
        "description" : "NAD83(HARN) / Tennessee (ftUS)",
        "code" : "EPSG:2915"
    }, {
        "description" : "NAD83(HARN) / Texas North (ftUS)",
        "code" : "EPSG:2916"
    }, {
        "description" : "NAD83(HARN) / Texas North Central (ftUS)",
        "code" : "EPSG:2917"
    }, {
        "description" : "NAD83(HARN) / Texas Central (ftUS)",
        "code" : "EPSG:2918"
    }, {
        "description" : "NAD83(HARN) / Texas South Central (ftUS)",
        "code" : "EPSG:2919"
    }, {
        "description" : "NAD83(HARN) / Texas South (ftUS)",
        "code" : "EPSG:2920"
    }, {
        "description" : "NAD83(HARN) / Utah North (ft)",
        "code" : "EPSG:2921"
    }, {
        "description" : "NAD83(HARN) / Utah Central (ft)",
        "code" : "EPSG:2922"
    }, {
        "description" : "NAD83(HARN) / Utah South (ft)",
        "code" : "EPSG:2923"
    }, {
        "description" : "NAD83(HARN) / Virginia North (ftUS)",
        "code" : "EPSG:2924"
    }, {
        "description" : "NAD83(HARN) / Virginia South (ftUS)",
        "code" : "EPSG:2925"
    }, {
        "description" : "NAD83(HARN) / Washington North (ftUS)",
        "code" : "EPSG:2926"
    }, {
        "description" : "NAD83(HARN) / Washington South (ftUS)",
        "code" : "EPSG:2927"
    }, {
        "description" : "NAD83(HARN) / Wisconsin North (ftUS)",
        "code" : "EPSG:2928"
    }, {
        "description" : "NAD83(HARN) / Wisconsin Central (ftUS)",
        "code" : "EPSG:2929"
    }, {
        "description" : "NAD83(HARN) / Wisconsin South (ftUS)",
        "code" : "EPSG:2930"
    }, {
        "description" : "Beduaram / TM 13 NE",
        "code" : "EPSG:2931"
    }, {
        "description" : "QND95 / Qatar National Grid",
        "code" : "EPSG:2932"
    }, {
        "description" : "Segara / UTM zone 50S",
        "code" : "EPSG:2933"
    }, {
        "description" : "Segara (Jakarta) / NEIEZ",
        "code" : "EPSG:2934"
    }, {
        "description" : "Pulkovo 1942 / CS63 zone A1",
        "code" : "EPSG:2935"
    }, {
        "description" : "Pulkovo 1942 / CS63 zone A2",
        "code" : "EPSG:2936"
    }, {
        "description" : "Pulkovo 1942 / CS63 zone A3",
        "code" : "EPSG:2937"
    }, {
        "description" : "Pulkovo 1942 / CS63 zone A4",
        "code" : "EPSG:2938"
    }, {
        "description" : "Pulkovo 1942 / CS63 zone K2",
        "code" : "EPSG:2939"
    }, {
        "description" : "Pulkovo 1942 / CS63 zone K3",
        "code" : "EPSG:2940"
    }, {
        "description" : "Pulkovo 1942 / CS63 zone K4",
        "code" : "EPSG:2941"
    }, {
        "description" : "Porto Santo / UTM zone 28N",
        "code" : "EPSG:2942"
    }, {
        "description" : "Selvagem Grande / UTM zone 28N",
        "code" : "EPSG:2943"
    }, {
        "description" : "NAD83(CSRS) / SCoPQ zone 2",
        "code" : "EPSG:2944"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 3",
        "code" : "EPSG:2945"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 4",
        "code" : "EPSG:2946"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 5",
        "code" : "EPSG:2947"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 6",
        "code" : "EPSG:2948"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 7",
        "code" : "EPSG:2949"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 8",
        "code" : "EPSG:2950"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 9",
        "code" : "EPSG:2951"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 10",
        "code" : "EPSG:2952"
    }, {
        "description" : "NAD83(CSRS) / New Brunswick Stereographic",
        "code" : "EPSG:2953"
    }, {
        "description" : "NAD83(CSRS) / Prince Edward Isl. Stereographic (NAD83)",
        "code" : "EPSG:2954"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 11N",
        "code" : "EPSG:2955"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 12N",
        "code" : "EPSG:2956"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 13N",
        "code" : "EPSG:2957"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 17N",
        "code" : "EPSG:2958"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 18N",
        "code" : "EPSG:2959"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 19N",
        "code" : "EPSG:2960"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 20N",
        "code" : "EPSG:2961"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 21N",
        "code" : "EPSG:2962"
    }, {
        "description" : "Lisbon 1890 (Lisbon) / Portugal Bonne",
        "code" : "EPSG:2963"
    }, {
        "description" : "NAD27 / Alaska Albers",
        "code" : "EPSG:2964"
    }, {
        "description" : "NAD83 / Indiana East (ftUS)",
        "code" : "EPSG:2965"
    }, {
        "description" : "NAD83 / Indiana West (ftUS)",
        "code" : "EPSG:2966"
    }, {
        "description" : "NAD83(HARN) / Indiana East (ftUS)",
        "code" : "EPSG:2967"
    }, {
        "description" : "NAD83(HARN) / Indiana West (ftUS)",
        "code" : "EPSG:2968"
    }, {
        "description" : "Fort Marigot / UTM zone 20N",
        "code" : "EPSG:2969"
    }, {
        "description" : "Guadeloupe 1948 / UTM zone 20N",
        "code" : "EPSG:2970"
    }, {
        "description" : "CSG67 / UTM zone 22N",
        "code" : "EPSG:2971"
    }, {
        "description" : "RGFG95 / UTM zone 22N",
        "code" : "EPSG:2972"
    }, {
        "description" : "Martinique 1938 / UTM zone 20N",
        "code" : "EPSG:2973"
    }, {
        "description" : "RGR92 / UTM zone 40S",
        "code" : "EPSG:2975"
    }, {
        "description" : "Tahiti 52 / UTM zone 6S",
        "code" : "EPSG:2976"
    }, {
        "description" : "Tahaa 54 / UTM zone 5S",
        "code" : "EPSG:2977"
    }, {
        "description" : "IGN72 Nuku Hiva / UTM zone 7S",
        "code" : "EPSG:2978"
    }, {
        "description" : "K0 1949 / UTM zone 42S",
        "code" : "EPSG:2979"
    }, {
        "description" : "Combani 1950 / UTM zone 38S",
        "code" : "EPSG:2980"
    }, {
        "description" : "IGN56 Lifou / UTM zone 58S",
        "code" : "EPSG:2981"
    }, {
        "description" : "IGN72 Grand Terre / UTM zone 58S",
        "code" : "EPSG:2982"
    }, {
        "description" : "ST87 Ouvea / UTM zone 58S",
        "code" : "EPSG:2983"
    }, {
        "description" : "RGNC 1991 / Lambert New Caledonia",
        "code" : "EPSG:2984"
    }, {
        "description" : "Petrels 1972 / Terre Adelie Polar Stereographic",
        "code" : "EPSG:2985"
    }, {
        "description" : "Perroud 1950 / Terre Adelie Polar Stereographic",
        "code" : "EPSG:2986"
    }, {
        "description" : "Saint Pierre et Miquelon 1950 / UTM zone 21N",
        "code" : "EPSG:2987"
    }, {
        "description" : "MOP78 / UTM zone 1S",
        "code" : "EPSG:2988"
    }, {
        "description" : "RRAF 1991 / UTM zone 20N",
        "code" : "EPSG:2989"
    }, {
        "description" : "Reunion 1947 / TM Reunion",
        "code" : "EPSG:2990"
    }, {
        "description" : "NAD83 / Oregon Lambert",
        "code" : "EPSG:2991"
    }, {
        "description" : "NAD83 / Oregon Lambert (ft)",
        "code" : "EPSG:2992"
    }, {
        "description" : "NAD83(HARN) / Oregon Lambert",
        "code" : "EPSG:2993"
    }, {
        "description" : "NAD83(HARN) / Oregon Lambert (ft)",
        "code" : "EPSG:2994"
    }, {
        "description" : "IGN53 Mare / UTM zone 58S",
        "code" : "EPSG:2995"
    }, {
        "description" : "ST84 Ile des Pins / UTM zone 58S",
        "code" : "EPSG:2996"
    }, {
        "description" : "ST71 Belep / UTM zone 58S",
        "code" : "EPSG:2997"
    }, {
        "description" : "NEA74 Noumea / UTM zone 58S",
        "code" : "EPSG:2998"
    }, {
        "description" : "Grand Comoros / UTM zone 38S",
        "code" : "EPSG:2999"
    }, {
        "description" : "Segara / NEIEZ",
        "code" : "EPSG:3000"
    }, {
        "description" : "Batavia / NEIEZ",
        "code" : "EPSG:3001"
    }, {
        "description" : "Makassar / NEIEZ",
        "code" : "EPSG:3002"
    }, {
        "description" : "Monte Mario / Italy zone 1",
        "code" : "EPSG:3003"
    }, {
        "description" : "Monte Mario / Italy zone 2",
        "code" : "EPSG:3004"
    }, {
        "description" : "NAD83 / BC Albers",
        "code" : "EPSG:3005"
    }, {
        "description" : "SWEREF99 TM",
        "code" : "EPSG:3006"
    }, {
        "description" : "SWEREF99 12 00",
        "code" : "EPSG:3007"
    }, {
        "description" : "SWEREF99 13 30",
        "code" : "EPSG:3008"
    }, {
        "description" : "SWEREF99 15 00",
        "code" : "EPSG:3009"
    }, {
        "description" : "SWEREF99 16 30",
        "code" : "EPSG:3010"
    }, {
        "description" : "SWEREF99 18 00",
        "code" : "EPSG:3011"
    }, {
        "description" : "SWEREF99 14 15",
        "code" : "EPSG:3012"
    }, {
        "description" : "SWEREF99 15 45",
        "code" : "EPSG:3013"
    }, {
        "description" : "SWEREF99 17 15",
        "code" : "EPSG:3014"
    }, {
        "description" : "SWEREF99 18 45",
        "code" : "EPSG:3015"
    }, {
        "description" : "SWEREF99 20 15",
        "code" : "EPSG:3016"
    }, {
        "description" : "SWEREF99 21 45",
        "code" : "EPSG:3017"
    }, {
        "description" : "SWEREF99 23 15",
        "code" : "EPSG:3018"
    }, {
        "description" : "RT90 7.5 gon V",
        "code" : "EPSG:3019"
    }, {
        "description" : "RT90 5 gon V",
        "code" : "EPSG:3020"
    }, {
        "description" : "RT90 2.5 gon V",
        "code" : "EPSG:3021"
    }, {
        "description" : "RT90 0 gon",
        "code" : "EPSG:3022"
    }, {
        "description" : "RT90 2.5 gon O",
        "code" : "EPSG:3023"
    }, {
        "description" : "RT90 5 gon O",
        "code" : "EPSG:3024"
    }, {
        "description" : "RT38 7.5 gon V",
        "code" : "EPSG:3025"
    }, {
        "description" : "RT38 5 gon V",
        "code" : "EPSG:3026"
    }, {
        "description" : "RT38 2.5 gon V",
        "code" : "EPSG:3027"
    }, {
        "description" : "RT38 0 gon",
        "code" : "EPSG:3028"
    }, {
        "description" : "RT38 2.5 gon O",
        "code" : "EPSG:3029"
    }, {
        "description" : "RT38 5 gon O",
        "code" : "EPSG:3030"
    }, {
        "description" : "WGS 84 / Antarctic Polar Stereographic",
        "code" : "EPSG:3031"
    }, {
        "description" : "WGS 84 / Australian Antarctic Polar Stereographic",
        "code" : "EPSG:3032"
    }, {
        "description" : "WGS 84 / Australian Antarctic Lambert",
        "code" : "EPSG:3033"
    }, {
        "description" : "ETRS89 / ETRS-LCC",
        "code" : "EPSG:3034"
    }, {
        "description" : "ETRS89 / ETRS-LAEA",
        "code" : "EPSG:3035"
    }, {
        "description" : "Moznet / UTM zone 36S",
        "code" : "EPSG:3036"
    }, {
        "description" : "Moznet / UTM zone 37S",
        "code" : "EPSG:3037"
    }, {
        "description" : "ETRS89 / ETRS-TM26",
        "code" : "EPSG:3038"
    }, {
        "description" : "ETRS89 / ETRS-TM27",
        "code" : "EPSG:3039"
    }, {
        "description" : "ETRS89 / ETRS-TM28",
        "code" : "EPSG:3040"
    }, {
        "description" : "ETRS89 / ETRS-TM29",
        "code" : "EPSG:3041"
    }, {
        "description" : "ETRS89 / ETRS-TM30",
        "code" : "EPSG:3042"
    }, {
        "description" : "ETRS89 / ETRS-TM31",
        "code" : "EPSG:3043"
    }, {
        "description" : "ETRS89 / ETRS-TM32",
        "code" : "EPSG:3044"
    }, {
        "description" : "ETRS89 / ETRS-TM33",
        "code" : "EPSG:3045"
    }, {
        "description" : "ETRS89 / ETRS-TM34",
        "code" : "EPSG:3046"
    }, {
        "description" : "ETRS89 / ETRS-TM35",
        "code" : "EPSG:3047"
    }, {
        "description" : "ETRS89 / ETRS-TM36",
        "code" : "EPSG:3048"
    }, {
        "description" : "ETRS89 / ETRS-TM37",
        "code" : "EPSG:3049"
    }, {
        "description" : "ETRS89 / ETRS-TM38",
        "code" : "EPSG:3050"
    }, {
        "description" : "ETRS89 / ETRS-TM39",
        "code" : "EPSG:3051"
    }, {
        "description" : "Reykjavik 1900 / Lambert 1900",
        "code" : "EPSG:3052"
    }, {
        "description" : "Hjorsey 1955 / Lambert 1955",
        "code" : "EPSG:3053"
    }, {
        "description" : "Hjorsey 1955 / UTM zone 26N",
        "code" : "EPSG:3054"
    }, {
        "description" : "Hjorsey 1955 / UTM zone 27N",
        "code" : "EPSG:3055"
    }, {
        "description" : "Hjorsey 1955 / UTM zone 28N",
        "code" : "EPSG:3056"
    }, {
        "description" : "ISN93 / Lambert 1993",
        "code" : "EPSG:3057"
    }, {
        "description" : "Helle 1954 / Jan Mayen Grid",
        "code" : "EPSG:3058"
    }, {
        "description" : "LKS92 / Latvia TM",
        "code" : "EPSG:3059"
    }, {
        "description" : "IGN72 Grande Terre / UTM zone 58S",
        "code" : "EPSG:3060"
    }, {
        "description" : "Porto Santo 1995 / UTM zone 28N",
        "code" : "EPSG:3061"
    }, {
        "description" : "Azores Oriental 1995 / UTM zone 26N",
        "code" : "EPSG:3062"
    }, {
        "description" : "Azores Central 1995 / UTM zone 26N",
        "code" : "EPSG:3063"
    }, {
        "description" : "IGM95 / UTM zone 32N",
        "code" : "EPSG:3064"
    }, {
        "description" : "IGM95 / UTM zone 33N",
        "code" : "EPSG:3065"
    }, {
        "description" : "ED50 / Jordan TM",
        "code" : "EPSG:3066"
    }, {
        "description" : "ETRS89 / ETRS-TM35FIN",
        "code" : "EPSG:3067"
    }, {
        "description" : "DHDN / Soldner Berlin",
        "code" : "EPSG:3068"
    }, {
        "description" : "NAD27 / Wisconsin Transverse Mercator",
        "code" : "EPSG:3069"
    }, {
        "description" : "NAD83 / Wisconsin Transverse Mercator",
        "code" : "EPSG:3070"
    }, {
        "description" : "NAD83(HARN) / Wisconsin Transverse Mercator",
        "code" : "EPSG:3071"
    }, {
        "description" : "NAD83 / Maine CS2000 East",
        "code" : "EPSG:3072"
    }, {
        "description" : "NAD83 / Maine CS2000 Central",
        "code" : "EPSG:3073"
    }, {
        "description" : "NAD83 / Maine CS2000 West",
        "code" : "EPSG:3074"
    }, {
        "description" : "NAD83(HARN) / Maine CS2000 East",
        "code" : "EPSG:3075"
    }, {
        "description" : "NAD83(HARN) / Maine CS2000 Central",
        "code" : "EPSG:3076"
    }, {
        "description" : "NAD83(HARN) / Maine CS2000 West",
        "code" : "EPSG:3077"
    }, {
        "description" : "NAD83 / Michigan Oblique Mercator",
        "code" : "EPSG:3078"
    }, {
        "description" : "NAD83(HARN) / Michigan Oblique Mercator",
        "code" : "EPSG:3079"
    }, {
        "description" : "NAD27 / Shackleford",
        "code" : "EPSG:3080"
    }, {
        "description" : "NAD83 / Texas State Mapping System",
        "code" : "EPSG:3081"
    }, {
        "description" : "NAD83 / Texas Centric Lambert Conformal",
        "code" : "EPSG:3082"
    }, {
        "description" : "NAD83 / Texas Centric Albers Equal Area",
        "code" : "EPSG:3083"
    }, {
        "description" : "NAD83(HARN) / Texas Centric Lambert Conformal",
        "code" : "EPSG:3084"
    }, {
        "description" : "NAD83(HARN) / Texas Centric Albers Equal Area",
        "code" : "EPSG:3085"
    }, {
        "description" : "NAD83 / Florida GDL Albers",
        "code" : "EPSG:3086"
    }, {
        "description" : "NAD83(HARN) / Florida GDL Albers",
        "code" : "EPSG:3087"
    }, {
        "description" : "NAD83 / Kentucky Single Zone",
        "code" : "EPSG:3088"
    }, {
        "description" : "NAD83 / Kentucky Single Zone (ftUS)",
        "code" : "EPSG:3089"
    }, {
        "description" : "NAD83(HARN) / Kentucky Single Zone",
        "code" : "EPSG:3090"
    }, {
        "description" : "NAD83(HARN) / Kentucky Single Zone (ftUS)",
        "code" : "EPSG:3091"
    }, {
        "description" : "Tokyo / UTM zone 51N",
        "code" : "EPSG:3092"
    }, {
        "description" : "Tokyo / UTM zone 52N",
        "code" : "EPSG:3093"
    }, {
        "description" : "Tokyo / UTM zone 53N",
        "code" : "EPSG:3094"
    }, {
        "description" : "Tokyo / UTM zone 54N",
        "code" : "EPSG:3095"
    }, {
        "description" : "Tokyo / UTM zone 55N",
        "code" : "EPSG:3096"
    }, {
        "description" : "JGD2000 / UTM zone 51N",
        "code" : "EPSG:3097"
    }, {
        "description" : "JGD2000 / UTM zone 52N",
        "code" : "EPSG:3098"
    }, {
        "description" : "JGD2000 / UTM zone 53N",
        "code" : "EPSG:3099"
    }, {
        "description" : "JGD2000 / UTM zone 54N",
        "code" : "EPSG:3100"
    }, {
        "description" : "JGD2000 / UTM zone 55N",
        "code" : "EPSG:3101"
    }, {
        "description" : "American Samoa 1962 / American Samoa Lambert",
        "code" : "EPSG:3102"
    }, {
        "description" : "Mauritania 1999 / UTM zone 28N",
        "code" : "EPSG:3103"
    }, {
        "description" : "Mauritania 1999 / UTM zone 29N",
        "code" : "EPSG:3104"
    }, {
        "description" : "Mauritania 1999 / UTM zone 30N",
        "code" : "EPSG:3105"
    }, {
        "description" : "Gulshan 303 / Bangladesh Transverse Mercator",
        "code" : "EPSG:3106"
    }, {
        "description" : "GDA94 / SA Lambert",
        "code" : "EPSG:3107"
    }, {
        "description" : "ETRS89 / Guernsey Grid",
        "code" : "EPSG:3108"
    }, {
        "description" : "ETRS89 / Jersey Transverse Mercator",
        "code" : "EPSG:3109"
    }, {
        "description" : "AGD66 / Vicgrid66",
        "code" : "EPSG:3110"
    }, {
        "description" : "GDA94 / Vicgrid94",
        "code" : "EPSG:3111"
    }, {
        "description" : "GDA94 / Geoscience Australia Lambert",
        "code" : "EPSG:3112"
    }, {
        "description" : "GDA94 / BCSG02",
        "code" : "EPSG:3113"
    }, {
        "description" : "MAGNA-SIRGAS / Colombia Far West zone",
        "code" : "EPSG:3114"
    }, {
        "description" : "MAGNA-SIRGAS / Colombia West zone",
        "code" : "EPSG:3115"
    }, {
        "description" : "MAGNA-SIRGAS / Colombia Bogota zone",
        "code" : "EPSG:3116"
    }, {
        "description" : "MAGNA-SIRGAS / Colombia East Central zone",
        "code" : "EPSG:3117"
    }, {
        "description" : "MAGNA-SIRGAS / Colombia East zone",
        "code" : "EPSG:3118"
    }, {
        "description" : "Douala 1948 / AEF west",
        "code" : "EPSG:3119"
    }, {
        "description" : "Pulkovo 1942(58) / Poland zone I",
        "code" : "EPSG:3120"
    }, {
        "description" : "PRS92 / Philippines zone 1",
        "code" : "EPSG:3121"
    }, {
        "description" : "PRS92 / Philippines zone 2",
        "code" : "EPSG:3122"
    }, {
        "description" : "PRS92 / Philippines zone 3",
        "code" : "EPSG:3123"
    }, {
        "description" : "PRS92 / Philippines zone 4",
        "code" : "EPSG:3124"
    }, {
        "description" : "PRS92 / Philippines zone 5",
        "code" : "EPSG:3125"
    }, {
        "description" : "ETRS89 / ETRS-GK19FIN",
        "code" : "EPSG:3126"
    }, {
        "description" : "ETRS89 / ETRS-GK20FIN",
        "code" : "EPSG:3127"
    }, {
        "description" : "ETRS89 / ETRS-GK21FIN",
        "code" : "EPSG:3128"
    }, {
        "description" : "ETRS89 / ETRS-GK22FIN",
        "code" : "EPSG:3129"
    }, {
        "description" : "ETRS89 / ETRS-GK23FIN",
        "code" : "EPSG:3130"
    }, {
        "description" : "ETRS89 / ETRS-GK24FIN",
        "code" : "EPSG:3131"
    }, {
        "description" : "ETRS89 / ETRS-GK25FIN",
        "code" : "EPSG:3132"
    }, {
        "description" : "ETRS89 / ETRS-GK26FIN",
        "code" : "EPSG:3133"
    }, {
        "description" : "ETRS89 / ETRS-GK27FIN",
        "code" : "EPSG:3134"
    }, {
        "description" : "ETRS89 / ETRS-GK28FIN",
        "code" : "EPSG:3135"
    }, {
        "description" : "ETRS89 / ETRS-GK29FIN",
        "code" : "EPSG:3136"
    }, {
        "description" : "ETRS89 / ETRS-GK30FIN",
        "code" : "EPSG:3137"
    }, {
        "description" : "ETRS89 / ETRS-GK31FIN",
        "code" : "EPSG:3138"
    }, {
        "description" : "Vanua Levu 1915 / Vanua Levu Grid",
        "code" : "EPSG:3139"
    }, {
        "description" : "Viti Levu 1912 / Viti Levu Grid",
        "code" : "EPSG:3140"
    }, {
        "description" : "Fiji 1956 / UTM zone 60S",
        "code" : "EPSG:3141"
    }, {
        "description" : "Fiji 1956 / UTM zone 1S",
        "code" : "EPSG:3142"
    }, {
        "description" : "Fiji 1986 / Fiji Map Grid",
        "code" : "EPSG:3143"
    }, {
        "description" : "FD54 / Faroe Lambert",
        "code" : "EPSG:3144"
    }, {
        "description" : "ETRS89 / Faroe Lambert",
        "code" : "EPSG:3145"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 6",
        "code" : "EPSG:3146"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger CM 18E",
        "code" : "EPSG:3147"
    }, {
        "description" : "Indian 1960 / UTM zone 48N",
        "code" : "EPSG:3148"
    }, {
        "description" : "Indian 1960 / UTM zone 49N",
        "code" : "EPSG:3149"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 6",
        "code" : "EPSG:3150"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger CM 18E",
        "code" : "EPSG:3151"
    }, {
        "description" : "ST74",
        "code" : "EPSG:3152"
    }, {
        "description" : "NAD83(CSRS) / BC Albers",
        "code" : "EPSG:3153"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 7N",
        "code" : "EPSG:3154"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 8N",
        "code" : "EPSG:3155"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 9N",
        "code" : "EPSG:3156"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 10N",
        "code" : "EPSG:3157"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 14N",
        "code" : "EPSG:3158"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 15N",
        "code" : "EPSG:3159"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 16N",
        "code" : "EPSG:3160"
    }, {
        "description" : "NAD83 / Ontario MNR Lambert",
        "code" : "EPSG:3161"
    }, {
        "description" : "NAD83(CSRS) / Ontario MNR Lambert",
        "code" : "EPSG:3162"
    }, {
        "description" : "RGNC91-93 / Lambert New Caledonia",
        "code" : "EPSG:3163"
    }, {
        "description" : "ST87 Ouvea / UTM zone 58S",
        "code" : "EPSG:3164"
    }, {
        "description" : "NEA74 Noumea / Noumea Lambert",
        "code" : "EPSG:3165"
    }, {
        "description" : "NEA74 Noumea / Noumea Lambert 2",
        "code" : "EPSG:3166"
    }, {
        "description" : "Kertau (RSO) / RSO Malaya (ch)",
        "code" : "EPSG:3167"
    }, {
        "description" : "Kertau (RSO) / RSO Malaya (m)",
        "code" : "EPSG:3168"
    }, {
        "description" : "RGNC91-93 / UTM zone 57S",
        "code" : "EPSG:3169"
    }, {
        "description" : "RGNC91-93 / UTM zone 58S",
        "code" : "EPSG:3170"
    }, {
        "description" : "RGNC91-93 / UTM zone 59S",
        "code" : "EPSG:3171"
    }, {
        "description" : "IGN53 Mare / UTM zone 59S",
        "code" : "EPSG:3172"
    }, {
        "description" : "fk89 / Faroe Lambert FK89",
        "code" : "EPSG:3173"
    }, {
        "description" : "NAD83 / Great Lakes Albers",
        "code" : "EPSG:3174"
    }, {
        "description" : "NAD83 / Great Lakes and St Lawrence Albers",
        "code" : "EPSG:3175"
    }, {
        "description" : "Indian 1960 / TM 106 NE",
        "code" : "EPSG:3176"
    }, {
        "description" : "LGD2006 / Libya TM",
        "code" : "EPSG:3177"
    }, {
        "description" : "GR96 / UTM zone 18N",
        "code" : "EPSG:3178"
    }, {
        "description" : "GR96 / UTM zone 19N",
        "code" : "EPSG:3179"
    }, {
        "description" : "GR96 / UTM zone 20N",
        "code" : "EPSG:3180"
    }, {
        "description" : "GR96 / UTM zone 21N",
        "code" : "EPSG:3181"
    }, {
        "description" : "GR96 / UTM zone 22N",
        "code" : "EPSG:3182"
    }, {
        "description" : "GR96 / UTM zone 23N",
        "code" : "EPSG:3183"
    }, {
        "description" : "GR96 / UTM zone 24N",
        "code" : "EPSG:3184"
    }, {
        "description" : "GR96 / UTM zone 25N",
        "code" : "EPSG:3185"
    }, {
        "description" : "GR96 / UTM zone 26N",
        "code" : "EPSG:3186"
    }, {
        "description" : "GR96 / UTM zone 27N",
        "code" : "EPSG:3187"
    }, {
        "description" : "GR96 / UTM zone 28N",
        "code" : "EPSG:3188"
    }, {
        "description" : "GR96 / UTM zone 29N",
        "code" : "EPSG:3189"
    }, {
        "description" : "LGD2006 / Libya TM zone 5",
        "code" : "EPSG:3190"
    }, {
        "description" : "LGD2006 / Libya TM zone 6",
        "code" : "EPSG:3191"
    }, {
        "description" : "LGD2006 / Libya TM zone 7",
        "code" : "EPSG:3192"
    }, {
        "description" : "LGD2006 / Libya TM zone 8",
        "code" : "EPSG:3193"
    }, {
        "description" : "LGD2006 / Libya TM zone 9",
        "code" : "EPSG:3194"
    }, {
        "description" : "LGD2006 / Libya TM zone 10",
        "code" : "EPSG:3195"
    }, {
        "description" : "LGD2006 / Libya TM zone 11",
        "code" : "EPSG:3196"
    }, {
        "description" : "LGD2006 / Libya TM zone 12",
        "code" : "EPSG:3197"
    }, {
        "description" : "LGD2006 / Libya TM zone 13",
        "code" : "EPSG:3198"
    }, {
        "description" : "LGD2006 / UTM zone 32N",
        "code" : "EPSG:3199"
    }, {
        "description" : "FD58 / Iraq zone",
        "code" : "EPSG:3200"
    }, {
        "description" : "LGD2006 / UTM zone 33N",
        "code" : "EPSG:3201"
    }, {
        "description" : "LGD2006 / UTM zone 34N",
        "code" : "EPSG:3202"
    }, {
        "description" : "LGD2006 / UTM zone 35N",
        "code" : "EPSG:3203"
    }, {
        "description" : "WGS 84 / SCAR IMW SP19-20",
        "code" : "EPSG:3204"
    }, {
        "description" : "WGS 84 / SCAR IMW SP21-22",
        "code" : "EPSG:3205"
    }, {
        "description" : "WGS 84 / SCAR IMW SP23-24",
        "code" : "EPSG:3206"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ01-02",
        "code" : "EPSG:3207"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ19-20",
        "code" : "EPSG:3208"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ21-22",
        "code" : "EPSG:3209"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ37-38",
        "code" : "EPSG:3210"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ39-40",
        "code" : "EPSG:3211"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ41-42",
        "code" : "EPSG:3212"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ43-44",
        "code" : "EPSG:3213"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ45-46",
        "code" : "EPSG:3214"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ47-48",
        "code" : "EPSG:3215"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ49-50",
        "code" : "EPSG:3216"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ51-52",
        "code" : "EPSG:3217"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ53-54",
        "code" : "EPSG:3218"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ55-56",
        "code" : "EPSG:3219"
    }, {
        "description" : "WGS 84 / SCAR IMW SQ57-58",
        "code" : "EPSG:3220"
    }, {
        "description" : "WGS 84 / SCAR IMW SR13-14",
        "code" : "EPSG:3221"
    }, {
        "description" : "WGS 84 / SCAR IMW SR15-16",
        "code" : "EPSG:3222"
    }, {
        "description" : "WGS 84 / SCAR IMW SR17-18",
        "code" : "EPSG:3223"
    }, {
        "description" : "WGS 84 / SCAR IMW SR19-20",
        "code" : "EPSG:3224"
    }, {
        "description" : "WGS 84 / SCAR IMW SR27-28",
        "code" : "EPSG:3225"
    }, {
        "description" : "WGS 84 / SCAR IMW SR29-30",
        "code" : "EPSG:3226"
    }, {
        "description" : "WGS 84 / SCAR IMW SR31-32",
        "code" : "EPSG:3227"
    }, {
        "description" : "WGS 84 / SCAR IMW SR33-34",
        "code" : "EPSG:3228"
    }, {
        "description" : "WGS 84 / SCAR IMW SR35-36",
        "code" : "EPSG:3229"
    }, {
        "description" : "WGS 84 / SCAR IMW SR37-38",
        "code" : "EPSG:3230"
    }, {
        "description" : "WGS 84 / SCAR IMW SR39-40",
        "code" : "EPSG:3231"
    }, {
        "description" : "WGS 84 / SCAR IMW SR41-42",
        "code" : "EPSG:3232"
    }, {
        "description" : "WGS 84 / SCAR IMW SR43-44",
        "code" : "EPSG:3233"
    }, {
        "description" : "WGS 84 / SCAR IMW SR45-46",
        "code" : "EPSG:3234"
    }, {
        "description" : "WGS 84 / SCAR IMW SR47-48",
        "code" : "EPSG:3235"
    }, {
        "description" : "WGS 84 / SCAR IMW SR49-50",
        "code" : "EPSG:3236"
    }, {
        "description" : "WGS 84 / SCAR IMW SR51-52",
        "code" : "EPSG:3237"
    }, {
        "description" : "WGS 84 / SCAR IMW SR53-54",
        "code" : "EPSG:3238"
    }, {
        "description" : "WGS 84 / SCAR IMW SR55-56",
        "code" : "EPSG:3239"
    }, {
        "description" : "WGS 84 / SCAR IMW SR57-58",
        "code" : "EPSG:3240"
    }, {
        "description" : "WGS 84 / SCAR IMW SR59-60",
        "code" : "EPSG:3241"
    }, {
        "description" : "WGS 84 / SCAR IMW SS04-06",
        "code" : "EPSG:3242"
    }, {
        "description" : "WGS 84 / SCAR IMW SS07-09",
        "code" : "EPSG:3243"
    }, {
        "description" : "WGS 84 / SCAR IMW SS10-12",
        "code" : "EPSG:3244"
    }, {
        "description" : "WGS 84 / SCAR IMW SS13-15",
        "code" : "EPSG:3245"
    }, {
        "description" : "WGS 84 / SCAR IMW SS16-18",
        "code" : "EPSG:3246"
    }, {
        "description" : "WGS 84 / SCAR IMW SS19-21",
        "code" : "EPSG:3247"
    }, {
        "description" : "WGS 84 / SCAR IMW SS25-27",
        "code" : "EPSG:3248"
    }, {
        "description" : "WGS 84 / SCAR IMW SS28-30",
        "code" : "EPSG:3249"
    }, {
        "description" : "WGS 84 / SCAR IMW SS31-33",
        "code" : "EPSG:3250"
    }, {
        "description" : "WGS 84 / SCAR IMW SS34-36",
        "code" : "EPSG:3251"
    }, {
        "description" : "WGS 84 / SCAR IMW SS37-39",
        "code" : "EPSG:3252"
    }, {
        "description" : "WGS 84 / SCAR IMW SS40-42",
        "code" : "EPSG:3253"
    }, {
        "description" : "WGS 84 / SCAR IMW SS43-45",
        "code" : "EPSG:3254"
    }, {
        "description" : "WGS 84 / SCAR IMW SS46-48",
        "code" : "EPSG:3255"
    }, {
        "description" : "WGS 84 / SCAR IMW SS49-51",
        "code" : "EPSG:3256"
    }, {
        "description" : "WGS 84 / SCAR IMW SS52-54",
        "code" : "EPSG:3257"
    }, {
        "description" : "WGS 84 / SCAR IMW SS55-57",
        "code" : "EPSG:3258"
    }, {
        "description" : "WGS 84 / SCAR IMW SS58-60",
        "code" : "EPSG:3259"
    }, {
        "description" : "WGS 84 / SCAR IMW ST01-04",
        "code" : "EPSG:3260"
    }, {
        "description" : "WGS 84 / SCAR IMW ST05-08",
        "code" : "EPSG:3261"
    }, {
        "description" : "WGS 84 / SCAR IMW ST09-12",
        "code" : "EPSG:3262"
    }, {
        "description" : "WGS 84 / SCAR IMW ST13-16",
        "code" : "EPSG:3263"
    }, {
        "description" : "WGS 84 / SCAR IMW ST17-20",
        "code" : "EPSG:3264"
    }, {
        "description" : "WGS 84 / SCAR IMW ST21-24",
        "code" : "EPSG:3265"
    }, {
        "description" : "WGS 84 / SCAR IMW ST25-28",
        "code" : "EPSG:3266"
    }, {
        "description" : "WGS 84 / SCAR IMW ST29-32",
        "code" : "EPSG:3267"
    }, {
        "description" : "WGS 84 / SCAR IMW ST33-36",
        "code" : "EPSG:3268"
    }, {
        "description" : "WGS 84 / SCAR IMW ST37-40",
        "code" : "EPSG:3269"
    }, {
        "description" : "WGS 84 / SCAR IMW ST41-44",
        "code" : "EPSG:3270"
    }, {
        "description" : "WGS 84 / SCAR IMW ST45-48",
        "code" : "EPSG:3271"
    }, {
        "description" : "WGS 84 / SCAR IMW ST49-52",
        "code" : "EPSG:3272"
    }, {
        "description" : "WGS 84 / SCAR IMW ST53-56",
        "code" : "EPSG:3273"
    }, {
        "description" : "WGS 84 / SCAR IMW ST57-60",
        "code" : "EPSG:3274"
    }, {
        "description" : "WGS 84 / SCAR IMW SU01-05",
        "code" : "EPSG:3275"
    }, {
        "description" : "WGS 84 / SCAR IMW SU06-10",
        "code" : "EPSG:3276"
    }, {
        "description" : "WGS 84 / SCAR IMW SU11-15",
        "code" : "EPSG:3277"
    }, {
        "description" : "WGS 84 / SCAR IMW SU16-20",
        "code" : "EPSG:3278"
    }, {
        "description" : "WGS 84 / SCAR IMW SU21-25",
        "code" : "EPSG:3279"
    }, {
        "description" : "WGS 84 / SCAR IMW SU26-30",
        "code" : "EPSG:3280"
    }, {
        "description" : "WGS 84 / SCAR IMW SU31-35",
        "code" : "EPSG:3281"
    }, {
        "description" : "WGS 84 / SCAR IMW SU36-40",
        "code" : "EPSG:3282"
    }, {
        "description" : "WGS 84 / SCAR IMW SU41-45",
        "code" : "EPSG:3283"
    }, {
        "description" : "WGS 84 / SCAR IMW SU46-50",
        "code" : "EPSG:3284"
    }, {
        "description" : "WGS 84 / SCAR IMW SU51-55",
        "code" : "EPSG:3285"
    }, {
        "description" : "WGS 84 / SCAR IMW SU56-60",
        "code" : "EPSG:3286"
    }, {
        "description" : "WGS 84 / SCAR IMW SV01-10",
        "code" : "EPSG:3287"
    }, {
        "description" : "WGS 84 / SCAR IMW SV11-20",
        "code" : "EPSG:3288"
    }, {
        "description" : "WGS 84 / SCAR IMW SV21-30",
        "code" : "EPSG:3289"
    }, {
        "description" : "WGS 84 / SCAR IMW SV31-40",
        "code" : "EPSG:3290"
    }, {
        "description" : "WGS 84 / SCAR IMW SV41-50",
        "code" : "EPSG:3291"
    }, {
        "description" : "WGS 84 / SCAR IMW SV51-60",
        "code" : "EPSG:3292"
    }, {
        "description" : "WGS 84 / SCAR IMW SW01-60",
        "code" : "EPSG:3293"
    }, {
        "description" : "WGS 84 / USGS Transantarctic Mountains",
        "code" : "EPSG:3294"
    }, {
        "description" : "Guam 1963 / Yap Islands",
        "code" : "EPSG:3295"
    }, {
        "description" : "RGPF / UTM zone 5S",
        "code" : "EPSG:3296"
    }, {
        "description" : "RGPF / UTM zone 6S",
        "code" : "EPSG:3297"
    }, {
        "description" : "RGPF / UTM zone 7S",
        "code" : "EPSG:3298"
    }, {
        "description" : "RGPF / UTM zone 8S",
        "code" : "EPSG:3299"
    }, {
        "description" : "Estonian Coordinate System of 1992",
        "code" : "EPSG:3300"
    }, {
        "description" : "Estonian Coordinate System of 1997",
        "code" : "EPSG:3301"
    }, {
        "description" : "IGN63 Hiva Oa / UTM zone 7S",
        "code" : "EPSG:3302"
    }, {
        "description" : "Fatu Iva 72 / UTM zone 7S",
        "code" : "EPSG:3303"
    }, {
        "description" : "Tahiti 79 / UTM zone 6S",
        "code" : "EPSG:3304"
    }, {
        "description" : "Moorea 87 / UTM zone 6S",
        "code" : "EPSG:3305"
    }, {
        "description" : "Maupiti 83 / UTM zone 5S",
        "code" : "EPSG:3306"
    }, {
        "description" : "Nakhl-e Ghanem / UTM zone 39N",
        "code" : "EPSG:3307"
    }, {
        "description" : "GDA94 / NSW Lambert",
        "code" : "EPSG:3308"
    }, {
        "description" : "NAD27 / California Albers",
        "code" : "EPSG:3309"
    }, {
        "description" : "NAD83 / California Albers",
        "code" : "EPSG:3310"
    }, {
        "description" : "NAD83(HARN) / California Albers",
        "code" : "EPSG:3311"
    }, {
        "description" : "CSG67 / UTM zone 21N",
        "code" : "EPSG:3312"
    }, {
        "description" : "RGFG95 / UTM zone 21N",
        "code" : "EPSG:3313"
    }, {
        "description" : "Katanga 1955 / Katanga Lambert",
        "code" : "EPSG:3314"
    }, {
        "description" : "Katanga 1955 / Katanga TM",
        "code" : "EPSG:3315"
    }, {
        "description" : "Kasai 1953 / Congo TM zone 22",
        "code" : "EPSG:3316"
    }, {
        "description" : "Kasai 1953 / Congo TM zone 24",
        "code" : "EPSG:3317"
    }, {
        "description" : "IGC 1962 / Congo TM zone 12",
        "code" : "EPSG:3318"
    }, {
        "description" : "IGC 1962 / Congo TM zone 14",
        "code" : "EPSG:3319"
    }, {
        "description" : "IGC 1962 / Congo TM zone 16",
        "code" : "EPSG:3320"
    }, {
        "description" : "IGC 1962 / Congo TM zone 18",
        "code" : "EPSG:3321"
    }, {
        "description" : "IGC 1962 / Congo TM zone 20",
        "code" : "EPSG:3322"
    }, {
        "description" : "IGC 1962 / Congo TM zone 22",
        "code" : "EPSG:3323"
    }, {
        "description" : "IGC 1962 / Congo TM zone 24",
        "code" : "EPSG:3324"
    }, {
        "description" : "IGC 1962 / Congo TM zone 26",
        "code" : "EPSG:3325"
    }, {
        "description" : "IGC 1962 / Congo TM zone 28",
        "code" : "EPSG:3326"
    }, {
        "description" : "IGC 1962 / Congo TM zone 30",
        "code" : "EPSG:3327"
    }, {
        "description" : "Pulkovo 1942(58) / GUGiK-80",
        "code" : "EPSG:3328"
    }, {
        "description" : "Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 5",
        "code" : "EPSG:3329"
    }, {
        "description" : "Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 6",
        "code" : "EPSG:3330"
    }, {
        "description" : "Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 7",
        "code" : "EPSG:3331"
    }, {
        "description" : "Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 8",
        "code" : "EPSG:3332"
    }, {
        "description" : "Pulkovo 1942(58) / Gauss-Kruger zone 3",
        "code" : "EPSG:3333"
    }, {
        "description" : "Pulkovo 1942(58) / Gauss-Kruger zone 4",
        "code" : "EPSG:3334"
    }, {
        "description" : "Pulkovo 1942(58) / Gauss-Kruger zone 5",
        "code" : "EPSG:3335"
    }, {
        "description" : "IGN 1962 Kerguelen / UTM zone 42S",
        "code" : "EPSG:3336"
    }, {
        "description" : "Le Pouce 1934 / Mauritius Grid",
        "code" : "EPSG:3337"
    }, {
        "description" : "NAD83 / Alaska Albers",
        "code" : "EPSG:3338"
    }, {
        "description" : "IGCB 1955 / Congo TM zone 12",
        "code" : "EPSG:3339"
    }, {
        "description" : "IGCB 1955 / Congo TM zone 14",
        "code" : "EPSG:3340"
    }, {
        "description" : "IGCB 1955 / Congo TM zone 16",
        "code" : "EPSG:3341"
    }, {
        "description" : "IGCB 1955 / UTM zone 33S",
        "code" : "EPSG:3342"
    }, {
        "description" : "Mauritania 1999 / UTM zone 28N",
        "code" : "EPSG:3343"
    }, {
        "description" : "Mauritania 1999 / UTM zone 29N",
        "code" : "EPSG:3344"
    }, {
        "description" : "Mauritania 1999 / UTM zone 30N",
        "code" : "EPSG:3345"
    }, {
        "description" : "LKS94 / Lithuania TM",
        "code" : "EPSG:3346"
    }, {
        "description" : "NAD83 / Statistics Canada Lambert",
        "code" : "EPSG:3347"
    }, {
        "description" : "NAD83(CSRS) / Statistics Canada Lambert",
        "code" : "EPSG:3348"
    }, {
        "description" : "WGS 84 / PDC Mercator",
        "code" : "EPSG:3349"
    }, {
        "description" : "Pulkovo 1942 / CS63 zone C0",
        "code" : "EPSG:3350"
    }, {
        "description" : "Pulkovo 1942 / CS63 zone C1",
        "code" : "EPSG:3351"
    }, {
        "description" : "Pulkovo 1942 / CS63 zone C2",
        "code" : "EPSG:3352"
    }, {
        "description" : "Mhast (onshore) / UTM zone 32S",
        "code" : "EPSG:3353"
    }, {
        "description" : "Mhast (offshore) / UTM zone 32S",
        "code" : "EPSG:3354"
    }, {
        "description" : "Egypt Gulf of Suez S-650 TL / Red Belt",
        "code" : "EPSG:3355"
    }, {
        "description" : "Grand Cayman 1959 / UTM zone 17N",
        "code" : "EPSG:3356"
    }, {
        "description" : "Little Cayman 1961 / UTM zone 17N",
        "code" : "EPSG:3357"
    }, {
        "description" : "NAD83(HARN) / North Carolina",
        "code" : "EPSG:3358"
    }, {
        "description" : "NAD83(HARN) / North Carolina (ftUS)",
        "code" : "EPSG:3359"
    }, {
        "description" : "NAD83(HARN) / South Carolina",
        "code" : "EPSG:3360"
    }, {
        "description" : "NAD83(HARN) / South Carolina (ft)",
        "code" : "EPSG:3361"
    }, {
        "description" : "NAD83(HARN) / Pennsylvania North",
        "code" : "EPSG:3362"
    }, {
        "description" : "NAD83(HARN) / Pennsylvania North (ftUS)",
        "code" : "EPSG:3363"
    }, {
        "description" : "NAD83(HARN) / Pennsylvania South",
        "code" : "EPSG:3364"
    }, {
        "description" : "NAD83(HARN) / Pennsylvania South (ftUS)",
        "code" : "EPSG:3365"
    }, {
        "description" : "Hong Kong 1963 Grid System",
        "code" : "EPSG:3366"
    }, {
        "description" : "IGN Astro 1960 / UTM zone 28N",
        "code" : "EPSG:3367"
    }, {
        "description" : "IGN Astro 1960 / UTM zone 29N",
        "code" : "EPSG:3368"
    }, {
        "description" : "IGN Astro 1960 / UTM zone 30N",
        "code" : "EPSG:3369"
    }, {
        "description" : "NAD27 / UTM zone 59N",
        "code" : "EPSG:3370"
    }, {
        "description" : "NAD27 / UTM zone 60N",
        "code" : "EPSG:3371"
    }, {
        "description" : "NAD83 / UTM zone 59N",
        "code" : "EPSG:3372"
    }, {
        "description" : "NAD83 / UTM zone 60N",
        "code" : "EPSG:3373"
    }, {
        "description" : "FD54 / UTM zone 29N",
        "code" : "EPSG:3374"
    }, {
        "description" : "GDM2000 / Peninsula RSO",
        "code" : "EPSG:3375"
    }, {
        "description" : "GDM2000 / East Malaysia BRSO",
        "code" : "EPSG:3376"
    }, {
        "description" : "GDM2000 / Johor Grid",
        "code" : "EPSG:3377"
    }, {
        "description" : "GDM2000 / Sembilan and Melaka Grid",
        "code" : "EPSG:3378"
    }, {
        "description" : "GDM2000 / PahangGrid",
        "code" : "EPSG:3379"
    }, {
        "description" : "GDM2000 / Selangor Grid",
        "code" : "EPSG:3380"
    }, {
        "description" : "GDM2000 / Terengganu Grid",
        "code" : "EPSG:3381"
    }, {
        "description" : "GDM2000 / Pinang Grid",
        "code" : "EPSG:3382"
    }, {
        "description" : "GDM2000 / Kedah and Perlis Grid",
        "code" : "EPSG:3383"
    }, {
        "description" : "GDM2000 / Perak Grid",
        "code" : "EPSG:3384"
    }, {
        "description" : "GDM2000 / Kelantan Grid",
        "code" : "EPSG:3385"
    }, {
        "description" : "KKJ / Finland zone 0",
        "code" : "EPSG:3386"
    }, {
        "description" : "KKJ / Finland zone 5",
        "code" : "EPSG:3387"
    }, {
        "description" : "Pulkovo 1942 / Caspian Sea Mercator",
        "code" : "EPSG:3388"
    }, {
        "description" : "Pulkovo 1942 / 3-degree Gauss-Kruger zone 60",
        "code" : "EPSG:3389"
    }, {
        "description" : "Pulkovo 1995 / 3-degree Gauss-Kruger zone 60",
        "code" : "EPSG:3390"
    }, {
        "description" : "Karbala 1979 / UTM zone 37N",
        "code" : "EPSG:3391"
    }, {
        "description" : "Karbala 1979 / UTM zone 38N",
        "code" : "EPSG:3392"
    }, {
        "description" : "Karbala 1979 / UTM zone 39N",
        "code" : "EPSG:3393"
    }, {
        "description" : "Nahrwan 1934 / Iraq zone",
        "code" : "EPSG:3394"
    }, {
        "description" : "WGS 84 / World Mercator",
        "code" : "EPSG:3395"
    }, {
        "description" : "PD/83 / 3-degree Gauss-Kruger zone 3",
        "code" : "EPSG:3396"
    }, {
        "description" : "PD/83 / 3-degree Gauss-Kruger zone 4",
        "code" : "EPSG:3397"
    }, {
        "description" : "RD/83 / 3-degree Gauss-Kruger zone 4",
        "code" : "EPSG:3398"
    }, {
        "description" : "RD/83 / 3-degree Gauss-Kruger zone 5",
        "code" : "EPSG:3399"
    }, {
        "description" : "NAD83 / Alberta 10-TM (Forest)",
        "code" : "EPSG:3400"
    }, {
        "description" : "NAD83 / Alberta 10-TM (Resource)",
        "code" : "EPSG:3401"
    }, {
        "description" : "NAD83(CSRS) / Alberta 10-TM (Forest)",
        "code" : "EPSG:3402"
    }, {
        "description" : "NAD83(CSRS) / Alberta 10-TM (Resource)",
        "code" : "EPSG:3403"
    }, {
        "description" : "NAD83(HARN) / North Carolina (ftUS)",
        "code" : "EPSG:3404"
    }, {
        "description" : "VN-2000 / UTM zone 48N",
        "code" : "EPSG:3405"
    }, {
        "description" : "VN-2000 / UTM zone 49N",
        "code" : "EPSG:3406"
    }, {
        "description" : "Hong Kong 1963 Grid System",
        "code" : "EPSG:3407"
    }, {
        "description" : "NSIDC EASE-Grid North",
        "code" : "EPSG:3408"
    }, {
        "description" : "NSIDC EASE-Grid South",
        "code" : "EPSG:3409"
    }, {
        "description" : "NSIDC EASE-Grid Global",
        "code" : "EPSG:3410"
    }, {
        "description" : "NSIDC Sea Ice Polar Stereographic North",
        "code" : "EPSG:3411"
    }, {
        "description" : "NSIDC Sea Ice Polar Stereographic South",
        "code" : "EPSG:3412"
    }, {
        "description" : "WGS 84 / NSIDC Sea Ice Polar Stereographic North",
        "code" : "EPSG:3413"
    }, {
        "description" : "SVY21 / Singapore TM",
        "code" : "EPSG:3414"
    }, {
        "description" : "WGS 72BE / South China Sea Lambert",
        "code" : "EPSG:3415"
    }, {
        "description" : "ETRS89 / Austria Lambert",
        "code" : "EPSG:3416"
    }, {
        "description" : "NAD83 / Iowa North (ft US)",
        "code" : "EPSG:3417"
    }, {
        "description" : "NAD83 / Iowa South (ft US)",
        "code" : "EPSG:3418"
    }, {
        "description" : "NAD83 / Kansas North (ft US)",
        "code" : "EPSG:3419"
    }, {
        "description" : "NAD83 / Kansas South (ft US)",
        "code" : "EPSG:3420"
    }, {
        "description" : "NAD83 / Nevada East (ft US)",
        "code" : "EPSG:3421"
    }, {
        "description" : "NAD83 / Nevada Central (ft US)",
        "code" : "EPSG:3422"
    }, {
        "description" : "NAD83 / Nevada West (ft US)",
        "code" : "EPSG:3423"
    }, {
        "description" : "NAD83 / New Jersey (ft US)",
        "code" : "EPSG:3424"
    }, {
        "description" : "NAD83(HARN) / Iowa North (ft US)",
        "code" : "EPSG:3425"
    }, {
        "description" : "NAD83(HARN) / Iowa South (ft US)",
        "code" : "EPSG:3426"
    }, {
        "description" : "NAD83(HARN) / Kansas North (ft US)",
        "code" : "EPSG:3427"
    }, {
        "description" : "NAD83(HARN) / Kansas South (ft US)",
        "code" : "EPSG:3428"
    }, {
        "description" : "NAD83(HARN) / Nevada East (ft US)",
        "code" : "EPSG:3429"
    }, {
        "description" : "NAD83(HARN) / Nevada Central (ft US)",
        "code" : "EPSG:3430"
    }, {
        "description" : "NAD83(HARN) / Nevada West (ft US)",
        "code" : "EPSG:3431"
    }, {
        "description" : "NAD83(HARN) / New Jersey (ft US)",
        "code" : "EPSG:3432"
    }, {
        "description" : "NAD83 / Arkansas North (ftUS)",
        "code" : "EPSG:3433"
    }, {
        "description" : "NAD83 / Arkansas South (ftUS)",
        "code" : "EPSG:3434"
    }, {
        "description" : "NAD83 / Illinois East (ftUS)",
        "code" : "EPSG:3435"
    }, {
        "description" : "NAD83 / Illinois West (ftUS)",
        "code" : "EPSG:3436"
    }, {
        "description" : "NAD83 / New Hampshire (ftUS)",
        "code" : "EPSG:3437"
    }, {
        "description" : "NAD83 / Rhode Island (ftUS)",
        "code" : "EPSG:3438"
    }, {
        "description" : "PSD93 / UTM zone 39N",
        "code" : "EPSG:3439"
    }, {
        "description" : "PSD93 / UTM zone 40N",
        "code" : "EPSG:3440"
    }, {
        "description" : "NAD83(HARN) / Arkansas North (ftUS)",
        "code" : "EPSG:3441"
    }, {
        "description" : "NAD83(HARN) / Arkansas South (ftUS)",
        "code" : "EPSG:3442"
    }, {
        "description" : "NAD83(HARN) / Illinois East (ftUS)",
        "code" : "EPSG:3443"
    }, {
        "description" : "NAD83(HARN) / Illinois West (ftUS)",
        "code" : "EPSG:3444"
    }, {
        "description" : "NAD83(HARN) / New Hampshire (ftUS)",
        "code" : "EPSG:3445"
    }, {
        "description" : "NAD83(HARN) / Rhode Island (ftUS)",
        "code" : "EPSG:3446"
    }, {
        "description" : "ETRS89 / Belgian Lambert 2005",
        "code" : "EPSG:3447"
    }, {
        "description" : "JAD2001 / Jamaica Metric Grid",
        "code" : "EPSG:3448"
    }, {
        "description" : "JAD2001 / UTM zone 17N",
        "code" : "EPSG:3449"
    }, {
        "description" : "JAD2001 / UTM zone 18N",
        "code" : "EPSG:3450"
    }, {
        "description" : "NAD83 / Louisiana North (ftUS)",
        "code" : "EPSG:3451"
    }, {
        "description" : "NAD83 / Louisiana South (ftUS)",
        "code" : "EPSG:3452"
    }, {
        "description" : "NAD83 / Louisiana Offshore (ftUS)",
        "code" : "EPSG:3453"
    }, {
        "description" : "NAD83 / South Dakota North (ftUS)",
        "code" : "EPSG:3454"
    }, {
        "description" : "NAD83 / South Dakota South (ftUS)",
        "code" : "EPSG:3455"
    }, {
        "description" : "NAD83(HARN) / Louisiana North (ftUS)",
        "code" : "EPSG:3456"
    }, {
        "description" : "NAD83(HARN) / Louisiana South (ftUS)",
        "code" : "EPSG:3457"
    }, {
        "description" : "NAD83(HARN) / South Dakota North (ftUS)",
        "code" : "EPSG:3458"
    }, {
        "description" : "NAD83(HARN) / South Dakota South (ftUS)",
        "code" : "EPSG:3459"
    }, {
        "description" : "Fiji 1986 / Fiji Map Grid",
        "code" : "EPSG:3460"
    }, {
        "description" : "Dabola 1981 / UTM zone 28N",
        "code" : "EPSG:3461"
    }, {
        "description" : "Dabola 1981 / UTM zone 29N",
        "code" : "EPSG:3462"
    }, {
        "description" : "NAD83 / Maine CS2000 Central",
        "code" : "EPSG:3463"
    }, {
        "description" : "NAD83(HARN) / Maine CS2000 Central",
        "code" : "EPSG:3464"
    }, {
        "description" : "NAD83(NSRS2007) / Alabama East",
        "code" : "EPSG:3465"
    }, {
        "description" : "NAD83(NSRS2007) / Alabama West",
        "code" : "EPSG:3466"
    }, {
        "description" : "NAD83(NSRS2007) / Alaska Albers",
        "code" : "EPSG:3467"
    }, {
        "description" : "NAD83(NSRS2007) / Alaska zone 1",
        "code" : "EPSG:3468"
    }, {
        "description" : "NAD83(NSRS2007) / Alaska zone 2",
        "code" : "EPSG:3469"
    }, {
        "description" : "NAD83(NSRS2007) / Alaska zone 3",
        "code" : "EPSG:3470"
    }, {
        "description" : "NAD83(NSRS2007) / Alaska zone 4",
        "code" : "EPSG:3471"
    }, {
        "description" : "NAD83(NSRS2007) / Alaska zone 5",
        "code" : "EPSG:3472"
    }, {
        "description" : "NAD83(NSRS2007) / Alaska zone 6",
        "code" : "EPSG:3473"
    }, {
        "description" : "NAD83(NSRS2007) / Alaska zone 7",
        "code" : "EPSG:3474"
    }, {
        "description" : "NAD83(NSRS2007) / Alaska zone 8",
        "code" : "EPSG:3475"
    }, {
        "description" : "NAD83(NSRS2007) / Alaska zone 9",
        "code" : "EPSG:3476"
    }, {
        "description" : "NAD83(NSRS2007) / Alaska zone 10",
        "code" : "EPSG:3477"
    }, {
        "description" : "NAD83(NSRS2007) / Arizona Central",
        "code" : "EPSG:3478"
    }, {
        "description" : "NAD83(NSRS2007) / Arizona Central (ft)",
        "code" : "EPSG:3479"
    }, {
        "description" : "NAD83(NSRS2007) / Arizona East",
        "code" : "EPSG:3480"
    }, {
        "description" : "NAD83(NSRS2007) / Arizona East (ft)",
        "code" : "EPSG:3481"
    }, {
        "description" : "NAD83(NSRS2007) / Arizona West",
        "code" : "EPSG:3482"
    }, {
        "description" : "NAD83(NSRS2007) / Arizona West (ft)",
        "code" : "EPSG:3483"
    }, {
        "description" : "NAD83(NSRS2007) / Arkansas North",
        "code" : "EPSG:3484"
    }, {
        "description" : "NAD83(NSRS2007) / Arkansas North (ftUS)",
        "code" : "EPSG:3485"
    }, {
        "description" : "NAD83(NSRS2007) / Arkansas South",
        "code" : "EPSG:3486"
    }, {
        "description" : "NAD83(NSRS2007) / Arkansas South (ftUS)",
        "code" : "EPSG:3487"
    }, {
        "description" : "NAD83(NSRS2007) / California Albers",
        "code" : "EPSG:3488"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 1",
        "code" : "EPSG:3489"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 1 (ftUS)",
        "code" : "EPSG:3490"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 2",
        "code" : "EPSG:3491"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 2 (ftUS)",
        "code" : "EPSG:3492"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 3",
        "code" : "EPSG:3493"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 3 (ftUS)",
        "code" : "EPSG:3494"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 4",
        "code" : "EPSG:3495"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 4 (ftUS)",
        "code" : "EPSG:3496"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 5",
        "code" : "EPSG:3497"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 5 (ftUS)",
        "code" : "EPSG:3498"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 6",
        "code" : "EPSG:3499"
    }, {
        "description" : "NAD83(NSRS2007) / California zone 6 (ftUS)",
        "code" : "EPSG:3500"
    }, {
        "description" : "NAD83(NSRS2007) / Colorado Central",
        "code" : "EPSG:3501"
    }, {
        "description" : "NAD83(NSRS2007) / Colorado Central (ftUS)",
        "code" : "EPSG:3502"
    }, {
        "description" : "NAD83(NSRS2007) / Colorado North",
        "code" : "EPSG:3503"
    }, {
        "description" : "NAD83(NSRS2007) / Colorado North (ftUS)",
        "code" : "EPSG:3504"
    }, {
        "description" : "NAD83(NSRS2007) / Colorado South",
        "code" : "EPSG:3505"
    }, {
        "description" : "NAD83(NSRS2007) / Colorado South (ftUS)",
        "code" : "EPSG:3506"
    }, {
        "description" : "NAD83(NSRS2007) / Connecticut",
        "code" : "EPSG:3507"
    }, {
        "description" : "NAD83(NSRS2007) / Connecticut (ftUS)",
        "code" : "EPSG:3508"
    }, {
        "description" : "NAD83(NSRS2007) / Delaware",
        "code" : "EPSG:3509"
    }, {
        "description" : "NAD83(NSRS2007) / Delaware (ftUS)",
        "code" : "EPSG:3510"
    }, {
        "description" : "NAD83(NSRS2007) / Florida East",
        "code" : "EPSG:3511"
    }, {
        "description" : "NAD83(NSRS2007) / Florida East (ftUS)",
        "code" : "EPSG:3512"
    }, {
        "description" : "NAD83(NSRS2007) / Florida GDL Albers",
        "code" : "EPSG:3513"
    }, {
        "description" : "NAD83(NSRS2007) / Florida North",
        "code" : "EPSG:3514"
    }, {
        "description" : "NAD83(NSRS2007) / Florida North (ftUS)",
        "code" : "EPSG:3515"
    }, {
        "description" : "NAD83(NSRS2007) / Florida West",
        "code" : "EPSG:3516"
    }, {
        "description" : "NAD83(NSRS2007) / Florida West (ftUS)",
        "code" : "EPSG:3517"
    }, {
        "description" : "NAD83(NSRS2007) / Georgia East",
        "code" : "EPSG:3518"
    }, {
        "description" : "NAD83(NSRS2007) / Georgia East (ftUS)",
        "code" : "EPSG:3519"
    }, {
        "description" : "NAD83(NSRS2007) / Georgia West",
        "code" : "EPSG:3520"
    }, {
        "description" : "NAD83(NSRS2007) / Georgia West (ftUS)",
        "code" : "EPSG:3521"
    }, {
        "description" : "NAD83(NSRS2007) / Idaho Central",
        "code" : "EPSG:3522"
    }, {
        "description" : "NAD83(NSRS2007) / Idaho Central (ftUS)",
        "code" : "EPSG:3523"
    }, {
        "description" : "NAD83(NSRS2007) / Idaho East",
        "code" : "EPSG:3524"
    }, {
        "description" : "NAD83(NSRS2007) / Idaho East (ftUS)",
        "code" : "EPSG:3525"
    }, {
        "description" : "NAD83(NSRS2007) / Idaho West",
        "code" : "EPSG:3526"
    }, {
        "description" : "NAD83(NSRS2007) / Idaho West (ftUS)",
        "code" : "EPSG:3527"
    }, {
        "description" : "NAD83(NSRS2007) / Illinois East",
        "code" : "EPSG:3528"
    }, {
        "description" : "NAD83(NSRS2007) / Illinois East (ftUS)",
        "code" : "EPSG:3529"
    }, {
        "description" : "NAD83(NSRS2007) / Illinois West",
        "code" : "EPSG:3530"
    }, {
        "description" : "NAD83(NSRS2007) / Illinois West (ftUS)",
        "code" : "EPSG:3531"
    }, {
        "description" : "NAD83(NSRS2007) / Indiana East",
        "code" : "EPSG:3532"
    }, {
        "description" : "NAD83(NSRS2007) / Indiana East (ftUS)",
        "code" : "EPSG:3533"
    }, {
        "description" : "NAD83(NSRS2007) / Indiana West",
        "code" : "EPSG:3534"
    }, {
        "description" : "NAD83(NSRS2007) / Indiana West (ftUS)",
        "code" : "EPSG:3535"
    }, {
        "description" : "NAD83(NSRS2007) / Iowa North",
        "code" : "EPSG:3536"
    }, {
        "description" : "NAD83(NSRS2007) / Iowa North (ft US)",
        "code" : "EPSG:3537"
    }, {
        "description" : "NAD83(NSRS2007) / Iowa South",
        "code" : "EPSG:3538"
    }, {
        "description" : "NAD83(NSRS2007) / Iowa South (ft US)",
        "code" : "EPSG:3539"
    }, {
        "description" : "NAD83(NSRS2007) / Kansas North",
        "code" : "EPSG:3540"
    }, {
        "description" : "NAD83(NSRS2007) / Kansas North (ft US)",
        "code" : "EPSG:3541"
    }, {
        "description" : "NAD83(NSRS2007) / Kansas South",
        "code" : "EPSG:3542"
    }, {
        "description" : "NAD83(NSRS2007) / Kansas South (ft US)",
        "code" : "EPSG:3543"
    }, {
        "description" : "NAD83(NSRS2007) / Kentucky North",
        "code" : "EPSG:3544"
    }, {
        "description" : "NAD83(NSRS2007) / Kentucky North (ftUS)",
        "code" : "EPSG:3545"
    }, {
        "description" : "NAD83(NSRS2007) / Kentucky Single Zone",
        "code" : "EPSG:3546"
    }, {
        "description" : "NAD83(NSRS2007) / Kentucky Single Zone (ftUS)",
        "code" : "EPSG:3547"
    }, {
        "description" : "NAD83(NSRS2007) / Kentucky South",
        "code" : "EPSG:3548"
    }, {
        "description" : "NAD83(NSRS2007) / Kentucky South (ftUS)",
        "code" : "EPSG:3549"
    }, {
        "description" : "NAD83(NSRS2007) / Louisiana North",
        "code" : "EPSG:3550"
    }, {
        "description" : "NAD83(NSRS2007) / Louisiana North (ftUS)",
        "code" : "EPSG:3551"
    }, {
        "description" : "NAD83(NSRS2007) / Louisiana South",
        "code" : "EPSG:3552"
    }, {
        "description" : "NAD83(NSRS2007) / Louisiana South (ftUS)",
        "code" : "EPSG:3553"
    }, {
        "description" : "NAD83(NSRS2007) / Maine CS2000 Central",
        "code" : "EPSG:3554"
    }, {
        "description" : "NAD83(NSRS2007) / Maine CS2000 East",
        "code" : "EPSG:3555"
    }, {
        "description" : "NAD83(NSRS2007) / Maine CS2000 West",
        "code" : "EPSG:3556"
    }, {
        "description" : "NAD83(NSRS2007) / Maine East",
        "code" : "EPSG:3557"
    }, {
        "description" : "NAD83(NSRS2007) / Maine West",
        "code" : "EPSG:3558"
    }, {
        "description" : "NAD83(NSRS2007) / Maryland",
        "code" : "EPSG:3559"
    }, {
        "description" : "NAD83 / Utah North (ftUS)",
        "code" : "EPSG:3560"
    }, {
        "description" : "Old Hawaiian / Hawaii zone 1",
        "code" : "EPSG:3561"
    }, {
        "description" : "Old Hawaiian / Hawaii zone 2",
        "code" : "EPSG:3562"
    }, {
        "description" : "Old Hawaiian / Hawaii zone 3",
        "code" : "EPSG:3563"
    }, {
        "description" : "Old Hawaiian / Hawaii zone 4",
        "code" : "EPSG:3564"
    }, {
        "description" : "Old Hawaiian / Hawaii zone 5",
        "code" : "EPSG:3565"
    }, {
        "description" : "NAD83 / Utah Central (ftUS)",
        "code" : "EPSG:3566"
    }, {
        "description" : "NAD83 / Utah South (ftUS)",
        "code" : "EPSG:3567"
    }, {
        "description" : "NAD83(HARN) / Utah North (ftUS)",
        "code" : "EPSG:3568"
    }, {
        "description" : "NAD83(HARN) / Utah Central (ftUS)",
        "code" : "EPSG:3569"
    }, {
        "description" : "NAD83(HARN) / Utah South (ftUS)",
        "code" : "EPSG:3570"
    }, {
        "description" : "WGS 84 / North Pole LAEA Bering Sea",
        "code" : "EPSG:3571"
    }, {
        "description" : "WGS 84 / North Pole LAEA Alaska",
        "code" : "EPSG:3572"
    }, {
        "description" : "WGS 84 / North Pole LAEA Canada",
        "code" : "EPSG:3573"
    }, {
        "description" : "WGS 84 / North Pole LAEA Atlantic",
        "code" : "EPSG:3574"
    }, {
        "description" : "WGS 84 / North Pole LAEA Europe",
        "code" : "EPSG:3575"
    }, {
        "description" : "WGS 84 / North Pole LAEA Russia",
        "code" : "EPSG:3576"
    }, {
        "description" : "GDA94 / Australian Albers",
        "code" : "EPSG:3577"
    }, {
        "description" : "NAD83 / Yukon Albers",
        "code" : "EPSG:3578"
    }, {
        "description" : "NAD83(CSRS) / Yukon Albers",
        "code" : "EPSG:3579"
    }, {
        "description" : "NAD83 / NWT Lambert",
        "code" : "EPSG:3580"
    }, {
        "description" : "NAD83(CSRS) / NWT Lambert",
        "code" : "EPSG:3581"
    }, {
        "description" : "NAD83(NSRS2007) / Maryland (ftUS)",
        "code" : "EPSG:3582"
    }, {
        "description" : "NAD83(NSRS2007) / Massachusetts Island",
        "code" : "EPSG:3583"
    }, {
        "description" : "NAD83(NSRS2007) / Massachusetts Island (ftUS)",
        "code" : "EPSG:3584"
    }, {
        "description" : "NAD83(NSRS2007) / Massachusetts Mainland",
        "code" : "EPSG:3585"
    }, {
        "description" : "NAD83(NSRS2007) / Massachusetts Mainland (ftUS)",
        "code" : "EPSG:3586"
    }, {
        "description" : "NAD83(NSRS2007) / Michigan Central",
        "code" : "EPSG:3587"
    }, {
        "description" : "NAD83(NSRS2007) / Michigan Central (ft)",
        "code" : "EPSG:3588"
    }, {
        "description" : "NAD83(NSRS2007) / Michigan North",
        "code" : "EPSG:3589"
    }, {
        "description" : "NAD83(NSRS2007) / Michigan North (ft)",
        "code" : "EPSG:3590"
    }, {
        "description" : "NAD83(NSRS2007) / Michigan Oblique Mercator",
        "code" : "EPSG:3591"
    }, {
        "description" : "NAD83(NSRS2007) / Michigan South",
        "code" : "EPSG:3592"
    }, {
        "description" : "NAD83(NSRS2007) / Michigan South (ft)",
        "code" : "EPSG:3593"
    }, {
        "description" : "NAD83(NSRS2007) / Minnesota Central",
        "code" : "EPSG:3594"
    }, {
        "description" : "NAD83(NSRS2007) / Minnesota North",
        "code" : "EPSG:3595"
    }, {
        "description" : "NAD83(NSRS2007) / Minnesota South",
        "code" : "EPSG:3596"
    }, {
        "description" : "NAD83(NSRS2007) / Mississippi East",
        "code" : "EPSG:3597"
    }, {
        "description" : "NAD83(NSRS2007) / Mississippi East (ftUS)",
        "code" : "EPSG:3598"
    }, {
        "description" : "NAD83(NSRS2007) / Mississippi West",
        "code" : "EPSG:3599"
    }, {
        "description" : "NAD83(NSRS2007) / Mississippi West (ftUS)",
        "code" : "EPSG:3600"
    }, {
        "description" : "NAD83(NSRS2007) / Missouri Central",
        "code" : "EPSG:3601"
    }, {
        "description" : "NAD83(NSRS2007) / Missouri East",
        "code" : "EPSG:3602"
    }, {
        "description" : "NAD83(NSRS2007) / Missouri West",
        "code" : "EPSG:3603"
    }, {
        "description" : "NAD83(NSRS2007) / Montana",
        "code" : "EPSG:3604"
    }, {
        "description" : "NAD83(NSRS2007) / Montana (ft)",
        "code" : "EPSG:3605"
    }, {
        "description" : "NAD83(NSRS2007) / Nebraska",
        "code" : "EPSG:3606"
    }, {
        "description" : "NAD83(NSRS2007) / Nevada Central",
        "code" : "EPSG:3607"
    }, {
        "description" : "NAD83(NSRS2007) / Nevada Central (ft US)",
        "code" : "EPSG:3608"
    }, {
        "description" : "NAD83(NSRS2007) / Nevada East",
        "code" : "EPSG:3609"
    }, {
        "description" : "NAD83(NSRS2007) / Nevada East (ft US)",
        "code" : "EPSG:3610"
    }, {
        "description" : "NAD83(NSRS2007) / Nevada West",
        "code" : "EPSG:3611"
    }, {
        "description" : "NAD83(NSRS2007) / Nevada West (ft US)",
        "code" : "EPSG:3612"
    }, {
        "description" : "NAD83(NSRS2007) / New Hampshire",
        "code" : "EPSG:3613"
    }, {
        "description" : "NAD83(NSRS2007) / New Hampshire (ftUS)",
        "code" : "EPSG:3614"
    }, {
        "description" : "NAD83(NSRS2007) / New Jersey",
        "code" : "EPSG:3615"
    }, {
        "description" : "NAD83(NSRS2007) / New Jersey (ft US)",
        "code" : "EPSG:3616"
    }, {
        "description" : "NAD83(NSRS2007) / New Mexico Central",
        "code" : "EPSG:3617"
    }, {
        "description" : "NAD83(NSRS2007) / New Mexico Central (ftUS)",
        "code" : "EPSG:3618"
    }, {
        "description" : "NAD83(NSRS2007) / New Mexico East",
        "code" : "EPSG:3619"
    }, {
        "description" : "NAD83(NSRS2007) / New Mexico East (ftUS)",
        "code" : "EPSG:3620"
    }, {
        "description" : "NAD83(NSRS2007) / New Mexico West",
        "code" : "EPSG:3621"
    }, {
        "description" : "NAD83(NSRS2007) / New Mexico West (ftUS)",
        "code" : "EPSG:3622"
    }, {
        "description" : "NAD83(NSRS2007) / New York Central",
        "code" : "EPSG:3623"
    }, {
        "description" : "NAD83(NSRS2007) / New York Central (ftUS)",
        "code" : "EPSG:3624"
    }, {
        "description" : "NAD83(NSRS2007) / New York East",
        "code" : "EPSG:3625"
    }, {
        "description" : "NAD83(NSRS2007) / New York East (ftUS)",
        "code" : "EPSG:3626"
    }, {
        "description" : "NAD83(NSRS2007) / New York Long Island",
        "code" : "EPSG:3627"
    }, {
        "description" : "NAD83(NSRS2007) / New York Long Island (ftUS)",
        "code" : "EPSG:3628"
    }, {
        "description" : "NAD83(NSRS2007) / New York West",
        "code" : "EPSG:3629"
    }, {
        "description" : "NAD83(NSRS2007) / New York West (ftUS)",
        "code" : "EPSG:3630"
    }, {
        "description" : "NAD83(NSRS2007) / North Carolina",
        "code" : "EPSG:3631"
    }, {
        "description" : "NAD83(NSRS2007) / North Carolina (ftUS)",
        "code" : "EPSG:3632"
    }, {
        "description" : "NAD83(NSRS2007) / North Dakota North",
        "code" : "EPSG:3633"
    }, {
        "description" : "NAD83(NSRS2007) / North Dakota North (ft)",
        "code" : "EPSG:3634"
    }, {
        "description" : "NAD83(NSRS2007) / North Dakota South",
        "code" : "EPSG:3635"
    }, {
        "description" : "NAD83(NSRS2007) / North Dakota South (ft)",
        "code" : "EPSG:3636"
    }, {
        "description" : "NAD83(NSRS2007) / Ohio North",
        "code" : "EPSG:3637"
    }, {
        "description" : "NAD83(NSRS2007) / Ohio South",
        "code" : "EPSG:3638"
    }, {
        "description" : "NAD83(NSRS2007) / Oklahoma North",
        "code" : "EPSG:3639"
    }, {
        "description" : "NAD83(NSRS2007) / Oklahoma North (ftUS)",
        "code" : "EPSG:3640"
    }, {
        "description" : "NAD83(NSRS2007) / Oklahoma South",
        "code" : "EPSG:3641"
    }, {
        "description" : "NAD83(NSRS2007) / Oklahoma South (ftUS)",
        "code" : "EPSG:3642"
    }, {
        "description" : "NAD83(NSRS2007) / Oregon Lambert",
        "code" : "EPSG:3643"
    }, {
        "description" : "NAD83(NSRS2007) / Oregon Lambert (ft)",
        "code" : "EPSG:3644"
    }, {
        "description" : "NAD83(NSRS2007) / Oregon North",
        "code" : "EPSG:3645"
    }, {
        "description" : "NAD83(NSRS2007) / Oregon North (ft)",
        "code" : "EPSG:3646"
    }, {
        "description" : "NAD83(NSRS2007) / Oregon South",
        "code" : "EPSG:3647"
    }, {
        "description" : "NAD83(NSRS2007) / Oregon South (ft)",
        "code" : "EPSG:3648"
    }, {
        "description" : "NAD83(NSRS2007) / Pennsylvania North",
        "code" : "EPSG:3649"
    }, {
        "description" : "NAD83(NSRS2007) / Pennsylvania North (ftUS)",
        "code" : "EPSG:3650"
    }, {
        "description" : "NAD83(NSRS2007) / Pennsylvania South",
        "code" : "EPSG:3651"
    }, {
        "description" : "NAD83(NSRS2007) / Pennsylvania South (ftUS)",
        "code" : "EPSG:3652"
    }, {
        "description" : "NAD83(NSRS2007) / Rhode Island",
        "code" : "EPSG:3653"
    }, {
        "description" : "NAD83(NSRS2007) / Rhode Island (ftUS)",
        "code" : "EPSG:3654"
    }, {
        "description" : "NAD83(NSRS2007) / South Carolina",
        "code" : "EPSG:3655"
    }, {
        "description" : "NAD83(NSRS2007) / South Carolina (ft)",
        "code" : "EPSG:3656"
    }, {
        "description" : "NAD83(NSRS2007) / South Dakota North",
        "code" : "EPSG:3657"
    }, {
        "description" : "NAD83(NSRS2007) / South Dakota North (ftUS)",
        "code" : "EPSG:3658"
    }, {
        "description" : "NAD83(NSRS2007) / South Dakota South",
        "code" : "EPSG:3659"
    }, {
        "description" : "NAD83(NSRS2007) / South Dakota South (ftUS)",
        "code" : "EPSG:3660"
    }, {
        "description" : "NAD83(NSRS2007) / Tennessee",
        "code" : "EPSG:3661"
    }, {
        "description" : "NAD83(NSRS2007) / Tennessee (ftUS)",
        "code" : "EPSG:3662"
    }, {
        "description" : "NAD83(NSRS2007) / Texas Central",
        "code" : "EPSG:3663"
    }, {
        "description" : "NAD83(NSRS2007) / Texas Central (ftUS)",
        "code" : "EPSG:3664"
    }, {
        "description" : "NAD83(NSRS2007) / Texas Centric Albers Equal Area",
        "code" : "EPSG:3665"
    }, {
        "description" : "NAD83(NSRS2007) / Texas Centric Lambert Conformal",
        "code" : "EPSG:3666"
    }, {
        "description" : "NAD83(NSRS2007) / Texas North",
        "code" : "EPSG:3667"
    }, {
        "description" : "NAD83(NSRS2007) / Texas North (ftUS)",
        "code" : "EPSG:3668"
    }, {
        "description" : "NAD83(NSRS2007) / Texas North Central",
        "code" : "EPSG:3669"
    }, {
        "description" : "NAD83(NSRS2007) / Texas North Central (ftUS)",
        "code" : "EPSG:3670"
    }, {
        "description" : "NAD83(NSRS2007) / Texas South",
        "code" : "EPSG:3671"
    }, {
        "description" : "NAD83(NSRS2007) / Texas South (ftUS)",
        "code" : "EPSG:3672"
    }, {
        "description" : "NAD83(NSRS2007) / Texas South Central",
        "code" : "EPSG:3673"
    }, {
        "description" : "NAD83(NSRS2007) / Texas South Central (ftUS)",
        "code" : "EPSG:3674"
    }, {
        "description" : "NAD83(NSRS2007) / Utah Central",
        "code" : "EPSG:3675"
    }, {
        "description" : "NAD83(NSRS2007) / Utah Central (ft)",
        "code" : "EPSG:3676"
    }, {
        "description" : "NAD83(NSRS2007) / Utah Central (ftUS)",
        "code" : "EPSG:3677"
    }, {
        "description" : "NAD83(NSRS2007) / Utah North",
        "code" : "EPSG:3678"
    }, {
        "description" : "NAD83(NSRS2007) / Utah North (ft)",
        "code" : "EPSG:3679"
    }, {
        "description" : "NAD83(NSRS2007) / Utah North (ftUS)",
        "code" : "EPSG:3680"
    }, {
        "description" : "NAD83(NSRS2007) / Utah South",
        "code" : "EPSG:3681"
    }, {
        "description" : "NAD83(NSRS2007) / Utah South (ft)",
        "code" : "EPSG:3682"
    }, {
        "description" : "NAD83(NSRS2007) / Utah South (ftUS)",
        "code" : "EPSG:3683"
    }, {
        "description" : "NAD83(NSRS2007) / Vermont",
        "code" : "EPSG:3684"
    }, {
        "description" : "NAD83(NSRS2007) / Virginia North",
        "code" : "EPSG:3685"
    }, {
        "description" : "NAD83(NSRS2007) / Virginia North (ftUS)",
        "code" : "EPSG:3686"
    }, {
        "description" : "NAD83(NSRS2007) / Virginia South",
        "code" : "EPSG:3687"
    }, {
        "description" : "NAD83(NSRS2007) / Virginia South (ftUS)",
        "code" : "EPSG:3688"
    }, {
        "description" : "NAD83(NSRS2007) / Washington North",
        "code" : "EPSG:3689"
    }, {
        "description" : "NAD83(NSRS2007) / Washington North (ftUS)",
        "code" : "EPSG:3690"
    }, {
        "description" : "NAD83(NSRS2007) / Washington South",
        "code" : "EPSG:3691"
    }, {
        "description" : "NAD83(NSRS2007) / Washington South (ftUS)",
        "code" : "EPSG:3692"
    }, {
        "description" : "NAD83(NSRS2007) / West Virginia North",
        "code" : "EPSG:3693"
    }, {
        "description" : "NAD83(NSRS2007) / West Virginia South",
        "code" : "EPSG:3694"
    }, {
        "description" : "NAD83(NSRS2007) / Wisconsin Central",
        "code" : "EPSG:3695"
    }, {
        "description" : "NAD83(NSRS2007) / Wisconsin Central (ftUS)",
        "code" : "EPSG:3696"
    }, {
        "description" : "NAD83(NSRS2007) / Wisconsin North",
        "code" : "EPSG:3697"
    }, {
        "description" : "NAD83(NSRS2007) / Wisconsin North (ftUS)",
        "code" : "EPSG:3698"
    }, {
        "description" : "NAD83(NSRS2007) / Wisconsin South",
        "code" : "EPSG:3699"
    }, {
        "description" : "NAD83(NSRS2007) / Wisconsin South (ftUS)",
        "code" : "EPSG:3700"
    }, {
        "description" : "NAD83(NSRS2007) / Wisconsin Transverse Mercator",
        "code" : "EPSG:3701"
    }, {
        "description" : "NAD83(NSRS2007) / Wyoming East",
        "code" : "EPSG:3702"
    }, {
        "description" : "NAD83(NSRS2007) / Wyoming East Central",
        "code" : "EPSG:3703"
    }, {
        "description" : "NAD83(NSRS2007) / Wyoming West Central",
        "code" : "EPSG:3704"
    }, {
        "description" : "NAD83(NSRS2007) / Wyoming West",
        "code" : "EPSG:3705"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 59N",
        "code" : "EPSG:3706"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 60N",
        "code" : "EPSG:3707"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 1N",
        "code" : "EPSG:3708"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 2N",
        "code" : "EPSG:3709"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 3N",
        "code" : "EPSG:3710"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 4N",
        "code" : "EPSG:3711"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 5N",
        "code" : "EPSG:3712"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 6N",
        "code" : "EPSG:3713"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 7N",
        "code" : "EPSG:3714"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 8N",
        "code" : "EPSG:3715"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 9N",
        "code" : "EPSG:3716"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 10N",
        "code" : "EPSG:3717"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 11N",
        "code" : "EPSG:3718"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 12N",
        "code" : "EPSG:3719"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 13N",
        "code" : "EPSG:3720"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 14N",
        "code" : "EPSG:3721"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 15N",
        "code" : "EPSG:3722"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 16N",
        "code" : "EPSG:3723"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 17N",
        "code" : "EPSG:3724"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 18N",
        "code" : "EPSG:3725"
    }, {
        "description" : "NAD83(NSRS2007) / UTM zone 19N",
        "code" : "EPSG:3726"
    }, {
        "description" : "Reunion 1947 / TM Reunion",
        "code" : "EPSG:3727"
    }, {
        "description" : "NAD83(NSRS2007) / Ohio North (ftUS)",
        "code" : "EPSG:3728"
    }, {
        "description" : "NAD83(NSRS2007) / Ohio South (ftUS)",
        "code" : "EPSG:3729"
    }, {
        "description" : "NAD83(NSRS2007) / Wyoming East (ftUS)",
        "code" : "EPSG:3730"
    }, {
        "description" : "NAD83(NSRS2007) / Wyoming East Central (ftUS)",
        "code" : "EPSG:3731"
    }, {
        "description" : "NAD83(NSRS2007) / Wyoming West Central (ftUS)",
        "code" : "EPSG:3732"
    }, {
        "description" : "NAD83(NSRS2007) / Wyoming West (ftUS)",
        "code" : "EPSG:3733"
    }, {
        "description" : "NAD83 / Ohio North (ftUS)",
        "code" : "EPSG:3734"
    }, {
        "description" : "NAD83 / Ohio South (ftUS)",
        "code" : "EPSG:3735"
    }, {
        "description" : "NAD83 / Wyoming East (ftUS)",
        "code" : "EPSG:3736"
    }, {
        "description" : "NAD83 / Wyoming East Central (ftUS)",
        "code" : "EPSG:3737"
    }, {
        "description" : "NAD83 / Wyoming West Central (ftUS)",
        "code" : "EPSG:3738"
    }, {
        "description" : "NAD83 / Wyoming West (ftUS)",
        "code" : "EPSG:3739"
    }, {
        "description" : "NAD83(HARN) / UTM zone 10N",
        "code" : "EPSG:3740"
    }, {
        "description" : "NAD83(HARN) / UTM zone 11N",
        "code" : "EPSG:3741"
    }, {
        "description" : "NAD83(HARN) / UTM zone 12N",
        "code" : "EPSG:3742"
    }, {
        "description" : "NAD83(HARN) / UTM zone 13N",
        "code" : "EPSG:3743"
    }, {
        "description" : "NAD83(HARN) / UTM zone 14N",
        "code" : "EPSG:3744"
    }, {
        "description" : "NAD83(HARN) / UTM zone 15N",
        "code" : "EPSG:3745"
    }, {
        "description" : "NAD83(HARN) / UTM zone 16N",
        "code" : "EPSG:3746"
    }, {
        "description" : "NAD83(HARN) / UTM zone 17N",
        "code" : "EPSG:3747"
    }, {
        "description" : "NAD83(HARN) / UTM zone 18N",
        "code" : "EPSG:3748"
    }, {
        "description" : "NAD83(HARN) / UTM zone 19N",
        "code" : "EPSG:3749"
    }, {
        "description" : "NAD83(HARN) / UTM zone 4N",
        "code" : "EPSG:3750"
    }, {
        "description" : "NAD83(HARN) / UTM zone 5N",
        "code" : "EPSG:3751"
    }, {
        "description" : "WGS 84 / Mercator 41",
        "code" : "EPSG:3752"
    }, {
        "description" : "NAD83(HARN) / Ohio North (ftUS)",
        "code" : "EPSG:3753"
    }, {
        "description" : "NAD83(HARN) / Ohio South (ftUS)",
        "code" : "EPSG:3754"
    }, {
        "description" : "NAD83(HARN) / Wyoming East (ftUS)",
        "code" : "EPSG:3755"
    }, {
        "description" : "NAD83(HARN) / Wyoming East Central (ftUS)",
        "code" : "EPSG:3756"
    }, {
        "description" : "NAD83(HARN) / Wyoming West Central (ftUS)",
        "code" : "EPSG:3757"
    }, {
        "description" : "NAD83(HARN) / Wyoming West (ftUS)",
        "code" : "EPSG:3758"
    }, {
        "description" : "NAD83 / Hawaii zone 3 (ftUS)",
        "code" : "EPSG:3759"
    }, {
        "description" : "NAD83(HARN) / Hawaii zone 3 (ftUS)",
        "code" : "EPSG:3760"
    }, {
        "description" : "NAD83(CSRS) / UTM zone 22N",
        "code" : "EPSG:3761"
    }, {
        "description" : "WGS 84 / South Georgia Lambert",
        "code" : "EPSG:3762"
    }, {
        "description" : "ETRS89 / Portugal TM06",
        "code" : "EPSG:3763"
    }, {
        "description" : "NZGD2000 / Chatham Island Circuit 2000",
        "code" : "EPSG:3764"
    }, {
        "description" : "HTRS96 / Croatia TM",
        "code" : "EPSG:3765"
    }, {
        "description" : "HTRS96 / Croatia LCC",
        "code" : "EPSG:3766"
    }, {
        "description" : "HTRS96 / UTM zone 33N",
        "code" : "EPSG:3767"
    }, {
        "description" : "HTRS96 / UTM zone 34N",
        "code" : "EPSG:3768"
    }, {
        "description" : "Bermuda 1957 / UTM zone 20N",
        "code" : "EPSG:3769"
    }, {
        "description" : "BDA2000 / Bermuda 2000 National Grid",
        "code" : "EPSG:3770"
    }, {
        "description" : "NAD27 / Alberta 3TM ref merid 111 W",
        "code" : "EPSG:3771"
    }, {
        "description" : "NAD27 / Alberta 3TM ref merid 114 W",
        "code" : "EPSG:3772"
    }, {
        "description" : "NAD27 / Alberta 3TM ref merid 117 W",
        "code" : "EPSG:3773"
    }, {
        "description" : "NAD27 / Alberta 3TM ref merid 120 W",
        "code" : "EPSG:3774"
    }, {
        "description" : "NAD83 / Alberta 3TM ref merid 111 W",
        "code" : "EPSG:3775"
    }, {
        "description" : "NAD83 / Alberta 3TM ref merid 114 W",
        "code" : "EPSG:3776"
    }, {
        "description" : "NAD83 / Alberta 3TM ref merid 117 W",
        "code" : "EPSG:3777"
    }, {
        "description" : "NAD83 / Alberta 3TM ref merid 120 W",
        "code" : "EPSG:3778"
    }, {
        "description" : "NAD83(CSRS) / Alberta 3TM ref merid 111 W",
        "code" : "EPSG:3779"
    }, {
        "description" : "NAD83(CSRS) / Alberta 3TM ref merid 114 W",
        "code" : "EPSG:3780"
    }, {
        "description" : "NAD83(CSRS) / Alberta 3TM ref merid 117 W",
        "code" : "EPSG:3781"
    }, {
        "description" : "NAD83(CSRS) / Alberta 3TM ref merid 120 W",
        "code" : "EPSG:3782"
    }, {
        "description" : "Pitcairn 2006 / Pitcairn TM 2006",
        "code" : "EPSG:3783"
    }, {
        "description" : "Pitcairn 1967 / UTM zone 9S",
        "code" : "EPSG:3784"
    }, {
        "description" : "Popular Visualisation CRS / Mercator",
        "code" : "EPSG:3785"
    }, {
        "description" : "World Equidistant Cylindrical (Sphere)",
        "code" : "EPSG:3786"
    }, {
        "description" : "MGI / Slovene National Grid",
        "code" : "EPSG:3787"
    }, {
        "description" : "NZGD2000 / Auckland Islands TM 2000",
        "code" : "EPSG:3788"
    }, {
        "description" : "NZGD2000 / Campbell Island TM 2000",
        "code" : "EPSG:3789"
    }, {
        "description" : "NZGD2000 / Antipodes Islands TM 2000",
        "code" : "EPSG:3790"
    }, {
        "description" : "NZGD2000 / Raoul Island TM 2000",
        "code" : "EPSG:3791"
    }, {
        "description" : "NZGD2000 / Chatham Islands TM 2000",
        "code" : "EPSG:3793"
    }, {
        "description" : "Slovenia 1996 / Slovene National Grid",
        "code" : "EPSG:3794"
    }, {
        "description" : "NAD27 / Cuba Norte",
        "code" : "EPSG:3795"
    }, {
        "description" : "NAD27 / Cuba Sur",
        "code" : "EPSG:3796"
    }, {
        "description" : "NAD27 / MTQ Lambert",
        "code" : "EPSG:3797"
    }, {
        "description" : "NAD83 / MTQ Lambert",
        "code" : "EPSG:3798"
    }, {
        "description" : "NAD83(CSRS) / MTQ Lambert",
        "code" : "EPSG:3799"
    }, {
        "description" : "NAD27 / Alberta 3TM ref merid 120 W",
        "code" : "EPSG:3800"
    }, {
        "description" : "NAD83 / Alberta 3TM ref merid 120 W",
        "code" : "EPSG:3801"
    }, {
        "description" : "NAD83(CSRS) / Alberta 3TM ref merid 120 W",
        "code" : "EPSG:3802"
    }, {
        "description" : "ETRS89 / Belgian Lambert 2008",
        "code" : "EPSG:3812"
    }, {
        "description" : "NAD83 / Mississippi TM",
        "code" : "EPSG:3814"
    }, {
        "description" : "NAD83(HARN) / Mississippi TM",
        "code" : "EPSG:3815"
    }, {
        "description" : "NAD83(NSRS2007) / Mississippi TM",
        "code" : "EPSG:3816"
    }, {
        "description" : "TWD97 / TM2 zone 119",
        "code" : "EPSG:3825"
    }, {
        "description" : "TWD97 / TM2 zone 121",
        "code" : "EPSG:3826"
    }, {
        "description" : "TWD67 / TM2 zone 119",
        "code" : "EPSG:3827"
    }, {
        "description" : "TWD67 / TM2 zone 121",
        "code" : "EPSG:3828"
    }, {
        "description" : "Hu Tzu Shan / UTM zone 51N",
        "code" : "EPSG:3829"
    }, {
        "description" : "WGS 84 / PDC Mercator",
        "code" : "EPSG:3832"
    }, {
        "description" : "Pulkovo 1942(58) / Gauss-Kruger zone 2",
        "code" : "EPSG:3833"
    }, {
        "description" : "Pulkovo 1942(83) / Gauss-Kruger zone 2",
        "code" : "EPSG:3834"
    }, {
        "description" : "Pulkovo 1942(83) / Gauss-Kruger zone 3",
        "code" : "EPSG:3835"
    }, {
        "description" : "Pulkovo 1942(83) / Gauss-Kruger zone 4",
        "code" : "EPSG:3836"
    }, {
        "description" : "Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 3",
        "code" : "EPSG:3837"
    }, {
        "description" : "Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 4",
        "code" : "EPSG:3838"
    }, {
        "description" : "Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 9",
        "code" : "EPSG:3839"
    }, {
        "description" : "Pulkovo 1942(58) / 3-degree Gauss-Kruger zone 10",
        "code" : "EPSG:3840"
    }, {
        "description" : "Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 6",
        "code" : "EPSG:3841"
    }, {
        "description" : "Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 7",
        "code" : "EPSG:3842"
    }, {
        "description" : "Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 8",
        "code" : "EPSG:3843"
    }, {
        "description" : "Pulkovo 1942(58) / Stereo70",
        "code" : "EPSG:3844"
    }, {
        "description" : "SWEREF99 / RT90 7.5 gon V emulation",
        "code" : "EPSG:3845"
    }, {
        "description" : "SWEREF99 / RT90 5 gon V emulation",
        "code" : "EPSG:3846"
    }, {
        "description" : "SWEREF99 / RT90 2.5 gon V emulation",
        "code" : "EPSG:3847"
    }, {
        "description" : "SWEREF99 / RT90 0 gon emulation",
        "code" : "EPSG:3848"
    }, {
        "description" : "SWEREF99 / RT90 2.5 gon O emulation",
        "code" : "EPSG:3849"
    }, {
        "description" : "SWEREF99 / RT90 5 gon O emulation",
        "code" : "EPSG:3850"
    }, {
        "description" : "NZGD2000 / NZCS2000",
        "code" : "EPSG:3851"
    }, {
        "description" : "RSRGD2000 / DGLC2000",
        "code" : "EPSG:3852"
    }, {
        "description" : "County ST74",
        "code" : "EPSG:3854"
    }, {
        "description" : "WGS 84 / Pseudo-Mercator",
        "code" : "EPSG:3857"
    }, {
        "description" : "WGS 84 / Pseudo-Mercator",
        "code" : "EPSG:900913"
    }, {
        "description" : "IGRS / UTM zone 37N",
        "code" : "EPSG:3890"
    }, {
        "description" : "IGRS / UTM zone 38N",
        "code" : "EPSG:3891"
    }, {
        "description" : "IGRS / UTM zone 39N",
        "code" : "EPSG:3892"
    }, {
        "description" : "ED50 / Iraq National Grid",
        "code" : "EPSG:3893"
    }, {
        "description" : "MGI 1901 / Balkans zone 5",
        "code" : "EPSG:3907"
    }, {
        "description" : "MGI 1901 / Balkans zone 6",
        "code" : "EPSG:3908"
    }, {
        "description" : "MGI 1901 / Balkans zone 7",
        "code" : "EPSG:3909"
    }, {
        "description" : "MGI 1901 / Balkans zone 8",
        "code" : "EPSG:3910"
    }, {
        "description" : "MGI 1901 / Slovenia Grid",
        "code" : "EPSG:3911"
    }, {
        "description" : "MGI 1901 / Slovene National Grid",
        "code" : "EPSG:3912"
    }, {
        "description" : "Puerto Rico / UTM zone 20N",
        "code" : "EPSG:3920"
    }, {
        "description" : "RGF93 / CC42",
        "code" : "EPSG:3942"
    }, {
        "description" : "RGF93 / CC43",
        "code" : "EPSG:3943"
    }, {
        "description" : "RGF93 / CC44",
        "code" : "EPSG:3944"
    }, {
        "description" : "RGF93 / CC45",
        "code" : "EPSG:3945"
    }, {
        "description" : "RGF93 / CC46",
        "code" : "EPSG:3946"
    }, {
        "description" : "RGF93 / CC47",
        "code" : "EPSG:3947"
    }, {
        "description" : "RGF93 / CC48",
        "code" : "EPSG:3948"
    }, {
        "description" : "RGF93 / CC49",
        "code" : "EPSG:3949"
    }, {
        "description" : "RGF93 / CC50",
        "code" : "EPSG:3950"
    }, {
        "description" : "NAD83 / Virginia Lambert",
        "code" : "EPSG:3968"
    }, {
        "description" : "NAD83(HARN) / Virginia Lambert",
        "code" : "EPSG:3969"
    }, {
        "description" : "NAD83(NSRS2007) / Virginia Lambert",
        "code" : "EPSG:3970"
    }, {
        "description" : "WGS 84 / NSIDC EASE-Grid North",
        "code" : "EPSG:3973"
    }, {
        "description" : "WGS 84 / NSIDC EASE-Grid South",
        "code" : "EPSG:3974"
    }, {
        "description" : "WGS 84 / NSIDC EASE-Grid Global",
        "code" : "EPSG:3975"
    }, {
        "description" : "WGS 84 / NSIDC Sea Ice Polar Stereographic South",
        "code" : "EPSG:3976"
    }, {
        "description" : "NAD83 / Canada Atlas Lambert",
        "code" : "EPSG:3978"
    }, {
        "description" : "NAD83(CSRS) / Canada Atlas Lambert",
        "code" : "EPSG:3979"
    }, {
        "description" : "Katanga 1955 / Katanga Lambert",
        "code" : "EPSG:3985"
    }, {
        "description" : "Katanga 1955 / Katanga Gauss zone A",
        "code" : "EPSG:3986"
    }, {
        "description" : "Katanga 1955 / Katanga Gauss zone B",
        "code" : "EPSG:3987"
    }, {
        "description" : "Katanga 1955 / Katanga Gauss zone C",
        "code" : "EPSG:3988"
    }, {
        "description" : "Katanga 1955 / Katanga Gauss zone D",
        "code" : "EPSG:3989"
    }, {
        "description" : "Puerto Rico State Plane CS of 1927",
        "code" : "EPSG:3991"
    }, {
        "description" : "Puerto Rico / St. Croix",
        "code" : "EPSG:3992"
    }, {
        "description" : "Guam 1963 / Guam SPCS",
        "code" : "EPSG:3993"
    }, {
        "description" : "WGS 84 / Mercator 41",
        "code" : "EPSG:3994"
    }, {
        "description" : "WGS 84 / Arctic Polar Stereographic",
        "code" : "EPSG:3995"
    }, {
        "description" : "WGS 84 / IBCAO Polar Stereographic",
        "code" : "EPSG:3996"
    }, {
        "description" : "WGS 84 / Dubai Local TM",
        "code" : "EPSG:3997"
    }, {
        "description" : "MOLDREF99 / Moldova TM",
        "code" : "EPSG:4026"
    }, {
        "description" : "WGS 84 / TMzn35N",
        "code" : "EPSG:4037"
    }, {
        "description" : "WGS 84 / TMzn36N",
        "code" : "EPSG:4038"
    }, {
        "description" : "RGRDC 2005 / Congo TM zone 12",
        "code" : "EPSG:4048"
    }, {
        "description" : "RGRDC 2005 / Congo TM zone 14",
        "code" : "EPSG:4049"
    }, {
        "description" : "RGRDC 2005 / Congo TM zone 16",
        "code" : "EPSG:4050"
    }, {
        "description" : "RGRDC 2005 / Congo TM zone 18",
        "code" : "EPSG:4051"
    }, {
        "description" : "RGRDC 2005 / Congo TM zone 20",
        "code" : "EPSG:4056"
    }, {
        "description" : "RGRDC 2005 / Congo TM zone 22",
        "code" : "EPSG:4057"
    }, {
        "description" : "RGRDC 2005 / Congo TM zone 24",
        "code" : "EPSG:4058"
    }, {
        "description" : "RGRDC 2005 / Congo TM zone 26",
        "code" : "EPSG:4059"
    }, {
        "description" : "RGRDC 2005 / Congo TM zone 28",
        "code" : "EPSG:4060"
    }, {
        "description" : "RGRDC 2005 / UTM zone 33S",
        "code" : "EPSG:4061"
    }, {
        "description" : "RGRDC 2005 / UTM zone 34S",
        "code" : "EPSG:4062"
    }, {
        "description" : "RGRDC 2005 / UTM zone 35S",
        "code" : "EPSG:4063"
    }, {
        "description" : "Chua / UTM zone 23S",
        "code" : "EPSG:4071"
    }, {
        "description" : "REGCAN95 / UTM zone 27N",
        "code" : "EPSG:4082"
    }, {
        "description" : "REGCAN95 / UTM zone 28N",
        "code" : "EPSG:4083"
    }, {
        "description" : "WGS 84 / World Equidistant Cylindrical",
        "code" : "EPSG:4087"
    }, {
        "description" : "World Equidistant Cylindrical (Sphere)",
        "code" : "EPSG:4088"
    }, {
        "description" : "EPSG topocentric example A",
        "code" : "EPSG:5819"
    }, {
        "description" : "EPSG topocentric example B",
        "code" : "EPSG:5820"
    }, {
        "description" : "EPSG vertical perspective example",
        "code" : "EPSG:5821"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 4",
        "code" : "EPSG:20004"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 5",
        "code" : "EPSG:20005"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 6",
        "code" : "EPSG:20006"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 7",
        "code" : "EPSG:20007"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 8",
        "code" : "EPSG:20008"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 9",
        "code" : "EPSG:20009"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 10",
        "code" : "EPSG:20010"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 11",
        "code" : "EPSG:20011"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 12",
        "code" : "EPSG:20012"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 13",
        "code" : "EPSG:20013"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 14",
        "code" : "EPSG:20014"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 15",
        "code" : "EPSG:20015"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 16",
        "code" : "EPSG:20016"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 17",
        "code" : "EPSG:20017"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 18",
        "code" : "EPSG:20018"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 19",
        "code" : "EPSG:20019"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 20",
        "code" : "EPSG:20020"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 21",
        "code" : "EPSG:20021"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 22",
        "code" : "EPSG:20022"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 23",
        "code" : "EPSG:20023"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 24",
        "code" : "EPSG:20024"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 25",
        "code" : "EPSG:20025"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 26",
        "code" : "EPSG:20026"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 27",
        "code" : "EPSG:20027"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 28",
        "code" : "EPSG:20028"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 29",
        "code" : "EPSG:20029"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 30",
        "code" : "EPSG:20030"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 31",
        "code" : "EPSG:20031"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger zone 32",
        "code" : "EPSG:20032"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 4N",
        "code" : "EPSG:20064"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 5N",
        "code" : "EPSG:20065"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 6N",
        "code" : "EPSG:20066"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 7N",
        "code" : "EPSG:20067"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 8N",
        "code" : "EPSG:20068"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 9N",
        "code" : "EPSG:20069"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 10N",
        "code" : "EPSG:20070"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 11N",
        "code" : "EPSG:20071"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 12N",
        "code" : "EPSG:20072"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 13N",
        "code" : "EPSG:20073"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 14N",
        "code" : "EPSG:20074"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 15N",
        "code" : "EPSG:20075"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 16N",
        "code" : "EPSG:20076"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 17N",
        "code" : "EPSG:20077"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 18N",
        "code" : "EPSG:20078"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 19N",
        "code" : "EPSG:20079"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 20N",
        "code" : "EPSG:20080"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 21N",
        "code" : "EPSG:20081"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 22N",
        "code" : "EPSG:20082"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 23N",
        "code" : "EPSG:20083"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 24N",
        "code" : "EPSG:20084"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 25N",
        "code" : "EPSG:20085"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 26N",
        "code" : "EPSG:20086"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 27N",
        "code" : "EPSG:20087"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 28N",
        "code" : "EPSG:20088"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 29N",
        "code" : "EPSG:20089"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 30N",
        "code" : "EPSG:20090"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 31N",
        "code" : "EPSG:20091"
    }, {
        "description" : "Pulkovo 1995 / Gauss-Kruger 32N",
        "code" : "EPSG:20092"
    }, {
        "description" : "Adindan / UTM zone 35N",
        "code" : "EPSG:20135"
    }, {
        "description" : "Adindan / UTM zone 36N",
        "code" : "EPSG:20136"
    }, {
        "description" : "Adindan / UTM zone 37N",
        "code" : "EPSG:20137"
    }, {
        "description" : "Adindan / UTM zone 38N",
        "code" : "EPSG:20138"
    }, {
        "description" : "AGD66 / AMG zone 48",
        "code" : "EPSG:20248"
    }, {
        "description" : "AGD66 / AMG zone 49",
        "code" : "EPSG:20249"
    }, {
        "description" : "AGD66 / AMG zone 50",
        "code" : "EPSG:20250"
    }, {
        "description" : "AGD66 / AMG zone 51",
        "code" : "EPSG:20251"
    }, {
        "description" : "AGD66 / AMG zone 52",
        "code" : "EPSG:20252"
    }, {
        "description" : "AGD66 / AMG zone 53",
        "code" : "EPSG:20253"
    }, {
        "description" : "AGD66 / AMG zone 54",
        "code" : "EPSG:20254"
    }, {
        "description" : "AGD66 / AMG zone 55",
        "code" : "EPSG:20255"
    }, {
        "description" : "AGD66 / AMG zone 56",
        "code" : "EPSG:20256"
    }, {
        "description" : "AGD66 / AMG zone 57",
        "code" : "EPSG:20257"
    }, {
        "description" : "AGD66 / AMG zone 58",
        "code" : "EPSG:20258"
    }, {
        "description" : "AGD84 / AMG zone 48",
        "code" : "EPSG:20348"
    }, {
        "description" : "AGD84 / AMG zone 49",
        "code" : "EPSG:20349"
    }, {
        "description" : "AGD84 / AMG zone 50",
        "code" : "EPSG:20350"
    }, {
        "description" : "AGD84 / AMG zone 51",
        "code" : "EPSG:20351"
    }, {
        "description" : "AGD84 / AMG zone 52",
        "code" : "EPSG:20352"
    }, {
        "description" : "AGD84 / AMG zone 53",
        "code" : "EPSG:20353"
    }, {
        "description" : "AGD84 / AMG zone 54",
        "code" : "EPSG:20354"
    }, {
        "description" : "AGD84 / AMG zone 55",
        "code" : "EPSG:20355"
    }, {
        "description" : "AGD84 / AMG zone 56",
        "code" : "EPSG:20356"
    }, {
        "description" : "AGD84 / AMG zone 57",
        "code" : "EPSG:20357"
    }, {
        "description" : "AGD84 / AMG zone 58",
        "code" : "EPSG:20358"
    }, {
        "description" : "Ain el Abd / UTM zone 36N",
        "code" : "EPSG:20436"
    }, {
        "description" : "Ain el Abd / UTM zone 37N",
        "code" : "EPSG:20437"
    }, {
        "description" : "Ain el Abd / UTM zone 38N",
        "code" : "EPSG:20438"
    }, {
        "description" : "Ain el Abd / UTM zone 39N",
        "code" : "EPSG:20439"
    }, {
        "description" : "Ain el Abd / UTM zone 40N",
        "code" : "EPSG:20440"
    }, {
        "description" : "Ain el Abd / Bahrain Grid",
        "code" : "EPSG:20499"
    }, {
        "description" : "Afgooye / UTM zone 38N",
        "code" : "EPSG:20538"
    }, {
        "description" : "Afgooye / UTM zone 39N",
        "code" : "EPSG:20539"
    }, {
        "description" : "Lisbon (Lisbon) / Portuguese National Grid",
        "code" : "EPSG:20790"
    }, {
        "description" : "Lisbon (Lisbon) / Portuguese Grid",
        "code" : "EPSG:20791"
    }, {
        "description" : "Aratu / UTM zone 22S",
        "code" : "EPSG:20822"
    }, {
        "description" : "Aratu / UTM zone 23S",
        "code" : "EPSG:20823"
    }, {
        "description" : "Aratu / UTM zone 24S",
        "code" : "EPSG:20824"
    }, {
        "description" : "Arc 1950 / UTM zone 34S",
        "code" : "EPSG:20934"
    }, {
        "description" : "Arc 1950 / UTM zone 35S",
        "code" : "EPSG:20935"
    }, {
        "description" : "Arc 1950 / UTM zone 36S",
        "code" : "EPSG:20936"
    }, {
        "description" : "Arc 1960 / UTM zone 35S",
        "code" : "EPSG:21035"
    }, {
        "description" : "Arc 1960 / UTM zone 36S",
        "code" : "EPSG:21036"
    }, {
        "description" : "Arc 1960 / UTM zone 37S",
        "code" : "EPSG:21037"
    }, {
        "description" : "Arc 1960 / UTM zone 35N",
        "code" : "EPSG:21095"
    }, {
        "description" : "Arc 1960 / UTM zone 36N",
        "code" : "EPSG:21096"
    }, {
        "description" : "Arc 1960 / UTM zone 37N",
        "code" : "EPSG:21097"
    }, {
        "description" : "Batavia (Jakarta) / NEIEZ",
        "code" : "EPSG:21100"
    }, {
        "description" : "Batavia / UTM zone 48S",
        "code" : "EPSG:21148"
    }, {
        "description" : "Batavia / UTM zone 49S",
        "code" : "EPSG:21149"
    }, {
        "description" : "Batavia / UTM zone 50S",
        "code" : "EPSG:21150"
    }, {
        "description" : "Barbados 1938 / British West Indies Grid",
        "code" : "EPSG:21291"
    }, {
        "description" : "Barbados 1938 / Barbados National Grid",
        "code" : "EPSG:21292"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger zone 13",
        "code" : "EPSG:21413"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger zone 14",
        "code" : "EPSG:21414"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger zone 15",
        "code" : "EPSG:21415"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger zone 16",
        "code" : "EPSG:21416"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger zone 17",
        "code" : "EPSG:21417"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger zone 18",
        "code" : "EPSG:21418"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger zone 19",
        "code" : "EPSG:21419"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger zone 20",
        "code" : "EPSG:21420"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger zone 21",
        "code" : "EPSG:21421"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger zone 22",
        "code" : "EPSG:21422"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger zone 23",
        "code" : "EPSG:21423"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger CM 75E",
        "code" : "EPSG:21453"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger CM 81E",
        "code" : "EPSG:21454"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger CM 87E",
        "code" : "EPSG:21455"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger CM 93E",
        "code" : "EPSG:21456"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger CM 99E",
        "code" : "EPSG:21457"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger CM 105E",
        "code" : "EPSG:21458"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger CM 111E",
        "code" : "EPSG:21459"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger CM 117E",
        "code" : "EPSG:21460"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger CM 123E",
        "code" : "EPSG:21461"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger CM 129E",
        "code" : "EPSG:21462"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger CM 135E",
        "code" : "EPSG:21463"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger 13N",
        "code" : "EPSG:21473"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger 14N",
        "code" : "EPSG:21474"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger 15N",
        "code" : "EPSG:21475"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger 16N",
        "code" : "EPSG:21476"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger 17N",
        "code" : "EPSG:21477"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger 18N",
        "code" : "EPSG:21478"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger 19N",
        "code" : "EPSG:21479"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger 20N",
        "code" : "EPSG:21480"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger 21N",
        "code" : "EPSG:21481"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger 22N",
        "code" : "EPSG:21482"
    }, {
        "description" : "Beijing 1954 / Gauss-Kruger 23N",
        "code" : "EPSG:21483"
    }, {
        "description" : "Belge 1950 (Brussels) / Belge Lambert 50",
        "code" : "EPSG:21500"
    }, {
        "description" : "Bern 1898 (Bern) / LV03C",
        "code" : "EPSG:21780"
    }, {
        "description" : "CH1903 / LV03",
        "code" : "EPSG:21781"
    }, {
        "description" : "CH1903 / LV03C-G",
        "code" : "EPSG:21782"
    }, {
        "description" : "Bogota 1975 / UTM zone 17N",
        "code" : "EPSG:21817"
    }, {
        "description" : "Bogota 1975 / UTM zone 18N",
        "code" : "EPSG:21818"
    }, {
        "description" : "Bogota 1975 / Colombia West zone",
        "code" : "EPSG:21891"
    }, {
        "description" : "Bogota 1975 / Colombia Bogota zone",
        "code" : "EPSG:21892"
    }, {
        "description" : "Bogota 1975 / Colombia East Central zone",
        "code" : "EPSG:21893"
    }, {
        "description" : "Bogota 1975 / Colombia East",
        "code" : "EPSG:21894"
    }, {
        "description" : "Bogota 1975 / Colombia West zone",
        "code" : "EPSG:21896"
    }, {
        "description" : "Bogota 1975 / Colombia Bogota zone",
        "code" : "EPSG:21897"
    }, {
        "description" : "Bogota 1975 / Colombia East Central zone",
        "code" : "EPSG:21898"
    }, {
        "description" : "Bogota 1975 / Colombia East",
        "code" : "EPSG:21899"
    }, {
        "description" : "Camacupa / UTM zone 32S",
        "code" : "EPSG:22032"
    }, {
        "description" : "Camacupa / UTM zone 33S",
        "code" : "EPSG:22033"
    }, {
        "description" : "Camacupa / TM 11.30 SE",
        "code" : "EPSG:22091"
    }, {
        "description" : "Camacupa / TM 12 SE",
        "code" : "EPSG:22092"
    }, {
        "description" : "POSGAR 98 / Argentina 1",
        "code" : "EPSG:22171"
    }, {
        "description" : "POSGAR 98 / Argentina 2",
        "code" : "EPSG:22172"
    }, {
        "description" : "POSGAR 98 / Argentina 3",
        "code" : "EPSG:22173"
    }, {
        "description" : "POSGAR 98 / Argentina 4",
        "code" : "EPSG:22174"
    }, {
        "description" : "POSGAR 98 / Argentina 5",
        "code" : "EPSG:22175"
    }, {
        "description" : "POSGAR 98 / Argentina 6",
        "code" : "EPSG:22176"
    }, {
        "description" : "POSGAR 98 / Argentina 7",
        "code" : "EPSG:22177"
    }, {
        "description" : "POSGAR 94 / Argentina 1",
        "code" : "EPSG:22181"
    }, {
        "description" : "POSGAR 94 / Argentina 2",
        "code" : "EPSG:22182"
    }, {
        "description" : "POSGAR 94 / Argentina 3",
        "code" : "EPSG:22183"
    }, {
        "description" : "POSGAR 94 / Argentina 4",
        "code" : "EPSG:22184"
    }, {
        "description" : "POSGAR 94 / Argentina 5",
        "code" : "EPSG:22185"
    }, {
        "description" : "POSGAR 94 / Argentina 6",
        "code" : "EPSG:22186"
    }, {
        "description" : "POSGAR 94 / Argentina 7",
        "code" : "EPSG:22187"
    }, {
        "description" : "Campo Inchauspe / Argentina 1",
        "code" : "EPSG:22191"
    }, {
        "description" : "Campo Inchauspe / Argentina 2",
        "code" : "EPSG:22192"
    }, {
        "description" : "Campo Inchauspe / Argentina 3",
        "code" : "EPSG:22193"
    }, {
        "description" : "Campo Inchauspe / Argentina 4",
        "code" : "EPSG:22194"
    }, {
        "description" : "Campo Inchauspe / Argentina 5",
        "code" : "EPSG:22195"
    }, {
        "description" : "Campo Inchauspe / Argentina 6",
        "code" : "EPSG:22196"
    }, {
        "description" : "Campo Inchauspe / Argentina 7",
        "code" : "EPSG:22197"
    }, {
        "description" : "Cape / UTM zone 34S",
        "code" : "EPSG:22234"
    }, {
        "description" : "Cape / UTM zone 35S",
        "code" : "EPSG:22235"
    }, {
        "description" : "Cape / UTM zone 36S",
        "code" : "EPSG:22236"
    }, {
        "description" : "Cape / Lo15",
        "code" : "EPSG:22275"
    }, {
        "description" : "Cape / Lo17",
        "code" : "EPSG:22277"
    }, {
        "description" : "Cape / Lo19",
        "code" : "EPSG:22279"
    }, {
        "description" : "Cape / Lo21",
        "code" : "EPSG:22281"
    }, {
        "description" : "Cape / Lo23",
        "code" : "EPSG:22283"
    }, {
        "description" : "Cape / Lo25",
        "code" : "EPSG:22285"
    }, {
        "description" : "Cape / Lo27",
        "code" : "EPSG:22287"
    }, {
        "description" : "Cape / Lo29",
        "code" : "EPSG:22289"
    }, {
        "description" : "Cape / Lo31",
        "code" : "EPSG:22291"
    }, {
        "description" : "Cape / Lo33",
        "code" : "EPSG:22293"
    }, {
        "description" : "Carthage (Paris) / Tunisia Mining Grid",
        "code" : "EPSG:22300"
    }, {
        "description" : "Carthage / UTM zone 32N",
        "code" : "EPSG:22332"
    }, {
        "description" : "Carthage / Nord Tunisie",
        "code" : "EPSG:22391"
    }, {
        "description" : "Carthage / Sud Tunisie",
        "code" : "EPSG:22392"
    }, {
        "description" : "Corrego Alegre / UTM zone 21S",
        "code" : "EPSG:22521"
    }, {
        "description" : "Corrego Alegre / UTM zone 22S",
        "code" : "EPSG:22522"
    }, {
        "description" : "Corrego Alegre / UTM zone 23S",
        "code" : "EPSG:22523"
    }, {
        "description" : "Corrego Alegre / UTM zone 24S",
        "code" : "EPSG:22524"
    }, {
        "description" : "Corrego Alegre / UTM zone 25S",
        "code" : "EPSG:22525"
    }, {
        "description" : "Deir ez Zor / Levant Zone",
        "code" : "EPSG:22700"
    }, {
        "description" : "Deir ez Zor / Syria Lambert",
        "code" : "EPSG:22770"
    }, {
        "description" : "Deir ez Zor / Levant Stereographic",
        "code" : "EPSG:22780"
    }, {
        "description" : "Douala / UTM zone 32N",
        "code" : "EPSG:22832"
    }, {
        "description" : "Egypt 1907 / Blue Belt",
        "code" : "EPSG:22991"
    }, {
        "description" : "Egypt 1907 / Red Belt",
        "code" : "EPSG:22992"
    }, {
        "description" : "Egypt 1907 / Purple Belt",
        "code" : "EPSG:22993"
    }, {
        "description" : "Egypt 1907 / Extended Purple Belt",
        "code" : "EPSG:22994"
    }, {
        "description" : "ED50 / UTM zone 28N",
        "code" : "EPSG:23028"
    }, {
        "description" : "ED50 / UTM zone 29N",
        "code" : "EPSG:23029"
    }, {
        "description" : "ED50 / UTM zone 30N",
        "code" : "EPSG:23030"
    }, {
        "description" : "ED50 / UTM zone 31N",
        "code" : "EPSG:23031"
    }, {
        "description" : "ED50 / UTM zone 32N",
        "code" : "EPSG:23032"
    }, {
        "description" : "ED50 / UTM zone 33N",
        "code" : "EPSG:23033"
    }, {
        "description" : "ED50 / UTM zone 34N",
        "code" : "EPSG:23034"
    }, {
        "description" : "ED50 / UTM zone 35N",
        "code" : "EPSG:23035"
    }, {
        "description" : "ED50 / UTM zone 36N",
        "code" : "EPSG:23036"
    }, {
        "description" : "ED50 / UTM zone 37N",
        "code" : "EPSG:23037"
    }, {
        "description" : "ED50 / UTM zone 38N",
        "code" : "EPSG:23038"
    }, {
        "description" : "ED50 / TM 0 N",
        "code" : "EPSG:23090"
    }, {
        "description" : "ED50 / TM 5 NE",
        "code" : "EPSG:23095"
    }, {
        "description" : "Fahud / UTM zone 39N",
        "code" : "EPSG:23239"
    }, {
        "description" : "Fahud / UTM zone 40N",
        "code" : "EPSG:23240"
    }, {
        "description" : "Garoua / UTM zone 33N",
        "code" : "EPSG:23433"
    }, {
        "description" : "HD72 / EOV",
        "code" : "EPSG:23700"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 46.2",
        "code" : "EPSG:23830"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 47.1",
        "code" : "EPSG:23831"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 47.2",
        "code" : "EPSG:23832"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 48.1",
        "code" : "EPSG:23833"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 48.2",
        "code" : "EPSG:23834"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 49.1",
        "code" : "EPSG:23835"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 49.2",
        "code" : "EPSG:23836"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 50.1",
        "code" : "EPSG:23837"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 50.2",
        "code" : "EPSG:23838"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 51.1",
        "code" : "EPSG:23839"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 51.2",
        "code" : "EPSG:23840"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 52.1",
        "code" : "EPSG:23841"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 52.2",
        "code" : "EPSG:23842"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 53.1",
        "code" : "EPSG:23843"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 53.2",
        "code" : "EPSG:23844"
    }, {
        "description" : "DGN95 / Indonesia TM-3 zone 54.1",
        "code" : "EPSG:23845"
    }, {
        "description" : "ID74 / UTM zone 46N",
        "code" : "EPSG:23846"
    }, {
        "description" : "ID74 / UTM zone 47N",
        "code" : "EPSG:23847"
    }, {
        "description" : "ID74 / UTM zone 48N",
        "code" : "EPSG:23848"
    }, {
        "description" : "ID74 / UTM zone 49N",
        "code" : "EPSG:23849"
    }, {
        "description" : "ID74 / UTM zone 50N",
        "code" : "EPSG:23850"
    }, {
        "description" : "ID74 / UTM zone 51N",
        "code" : "EPSG:23851"
    }, {
        "description" : "ID74 / UTM zone 52N",
        "code" : "EPSG:23852"
    }, {
        "description" : "ID74 / UTM zone 53N",
        "code" : "EPSG:23853"
    }, {
        "description" : "DGN95 / UTM zone 46N",
        "code" : "EPSG:23866"
    }, {
        "description" : "DGN95 / UTM zone 47N",
        "code" : "EPSG:23867"
    }, {
        "description" : "DGN95 / UTM zone 48N",
        "code" : "EPSG:23868"
    }, {
        "description" : "DGN95 / UTM zone 49N",
        "code" : "EPSG:23869"
    }, {
        "description" : "DGN95 / UTM zone 50N",
        "code" : "EPSG:23870"
    }, {
        "description" : "DGN95 / UTM zone 51N",
        "code" : "EPSG:23871"
    }, {
        "description" : "DGN95 / UTM zone 52N",
        "code" : "EPSG:23872"
    }, {
        "description" : "DGN95 / UTM zone 47S",
        "code" : "EPSG:23877"
    }, {
        "description" : "DGN95 / UTM zone 48S",
        "code" : "EPSG:23878"
    }, {
        "description" : "DGN95 / UTM zone 49S",
        "code" : "EPSG:23879"
    }, {
        "description" : "DGN95 / UTM zone 50S",
        "code" : "EPSG:23880"
    }, {
        "description" : "DGN95 / UTM zone 51S",
        "code" : "EPSG:23881"
    }, {
        "description" : "DGN95 / UTM zone 52S",
        "code" : "EPSG:23882"
    }, {
        "description" : "DGN95 / UTM zone 53S",
        "code" : "EPSG:23883"
    }, {
        "description" : "DGN95 / UTM zone 54S",
        "code" : "EPSG:23884"
    }, {
        "description" : "ID74 / UTM zone 46S",
        "code" : "EPSG:23886"
    }, {
        "description" : "ID74 / UTM zone 47S",
        "code" : "EPSG:23887"
    }, {
        "description" : "ID74 / UTM zone 48S",
        "code" : "EPSG:23888"
    }, {
        "description" : "ID74 / UTM zone 49S",
        "code" : "EPSG:23889"
    }, {
        "description" : "ID74 / UTM zone 50S",
        "code" : "EPSG:23890"
    }, {
        "description" : "ID74 / UTM zone 51S",
        "code" : "EPSG:23891"
    }, {
        "description" : "ID74 / UTM zone 52S",
        "code" : "EPSG:23892"
    }, {
        "description" : "ID74 / UTM zone 53S",
        "code" : "EPSG:23893"
    }, {
        "description" : "ID74 / UTM zone 54S",
        "code" : "EPSG:23894"
    }, {
        "description" : "Indian 1954 / UTM zone 46N",
        "code" : "EPSG:23946"
    }, {
        "description" : "Indian 1954 / UTM zone 47N",
        "code" : "EPSG:23947"
    }, {
        "description" : "Indian 1954 / UTM zone 48N",
        "code" : "EPSG:23948"
    }, {
        "description" : "Indian 1975 / UTM zone 47N",
        "code" : "EPSG:24047"
    }, {
        "description" : "Indian 1975 / UTM zone 48N",
        "code" : "EPSG:24048"
    }, {
        "description" : "Jamaica 1875 / Jamaica (Old Grid)",
        "code" : "EPSG:24100"
    }, {
        "description" : "JAD69 / Jamaica National Grid",
        "code" : "EPSG:24200"
    }, {
        "description" : "Kalianpur 1937 / UTM zone 45N",
        "code" : "EPSG:24305"
    }, {
        "description" : "Kalianpur 1937 / UTM zone 46N",
        "code" : "EPSG:24306"
    }, {
        "description" : "Kalianpur 1962 / UTM zone 41N",
        "code" : "EPSG:24311"
    }, {
        "description" : "Kalianpur 1962 / UTM zone 42N",
        "code" : "EPSG:24312"
    }, {
        "description" : "Kalianpur 1962 / UTM zone 43N",
        "code" : "EPSG:24313"
    }, {
        "description" : "Kalianpur 1975 / UTM zone 42N",
        "code" : "EPSG:24342"
    }, {
        "description" : "Kalianpur 1975 / UTM zone 43N",
        "code" : "EPSG:24343"
    }, {
        "description" : "Kalianpur 1975 / UTM zone 44N",
        "code" : "EPSG:24344"
    }, {
        "description" : "Kalianpur 1975 / UTM zone 45N",
        "code" : "EPSG:24345"
    }, {
        "description" : "Kalianpur 1975 / UTM zone 46N",
        "code" : "EPSG:24346"
    }, {
        "description" : "Kalianpur 1975 / UTM zone 47N",
        "code" : "EPSG:24347"
    }, {
        "description" : "Kalianpur 1880 / India zone 0",
        "code" : "EPSG:24370"
    }, {
        "description" : "Kalianpur 1880 / India zone I",
        "code" : "EPSG:24371"
    }, {
        "description" : "Kalianpur 1880 / India zone IIa",
        "code" : "EPSG:24372"
    }, {
        "description" : "Kalianpur 1880 / India zone III",
        "code" : "EPSG:24373"
    }, {
        "description" : "Kalianpur 1880 / India zone IV",
        "code" : "EPSG:24374"
    }, {
        "description" : "Kalianpur 1937 / India zone IIb",
        "code" : "EPSG:24375"
    }, {
        "description" : "Kalianpur 1962 / India zone I",
        "code" : "EPSG:24376"
    }, {
        "description" : "Kalianpur 1962 / India zone IIa",
        "code" : "EPSG:24377"
    }, {
        "description" : "Kalianpur 1975 / India zone I",
        "code" : "EPSG:24378"
    }, {
        "description" : "Kalianpur 1975 / India zone IIa",
        "code" : "EPSG:24379"
    }, {
        "description" : "Kalianpur 1975 / India zone IIb",
        "code" : "EPSG:24380"
    }, {
        "description" : "Kalianpur 1975 / India zone III",
        "code" : "EPSG:24381"
    }, {
        "description" : "Kalianpur 1880 / India zone IIb",
        "code" : "EPSG:24382"
    }, {
        "description" : "Kalianpur 1975 / India zone IV",
        "code" : "EPSG:24383"
    }, {
        "description" : "Kertau 1968 / Singapore Grid",
        "code" : "EPSG:24500"
    }, {
        "description" : "Kertau 1968 / UTM zone 47N",
        "code" : "EPSG:24547"
    }, {
        "description" : "Kertau 1968 / UTM zone 48N",
        "code" : "EPSG:24548"
    }, {
        "description" : "Kertau / R.S.O. Malaya (ch)",
        "code" : "EPSG:24571"
    }, {
        "description" : "KOC Lambert",
        "code" : "EPSG:24600"
    }, {
        "description" : "La Canoa / UTM zone 18N",
        "code" : "EPSG:24718"
    }, {
        "description" : "La Canoa / UTM zone 19N",
        "code" : "EPSG:24719"
    }, {
        "description" : "La Canoa / UTM zone 20N",
        "code" : "EPSG:24720"
    }, {
        "description" : "PSAD56 / UTM zone 17N",
        "code" : "EPSG:24817"
    }, {
        "description" : "PSAD56 / UTM zone 18N",
        "code" : "EPSG:24818"
    }, {
        "description" : "PSAD56 / UTM zone 19N",
        "code" : "EPSG:24819"
    }, {
        "description" : "PSAD56 / UTM zone 20N",
        "code" : "EPSG:24820"
    }, {
        "description" : "PSAD56 / UTM zone 21N",
        "code" : "EPSG:24821"
    }, {
        "description" : "PSAD56 / UTM zone 17S",
        "code" : "EPSG:24877"
    }, {
        "description" : "PSAD56 / UTM zone 18S",
        "code" : "EPSG:24878"
    }, {
        "description" : "PSAD56 / UTM zone 19S",
        "code" : "EPSG:24879"
    }, {
        "description" : "PSAD56 / UTM zone 20S",
        "code" : "EPSG:24880"
    }, {
        "description" : "PSAD56 / UTM zone 21S",
        "code" : "EPSG:24881"
    }, {
        "description" : "PSAD56 / UTM zone 22S",
        "code" : "EPSG:24882"
    }, {
        "description" : "PSAD56 / Peru west zone",
        "code" : "EPSG:24891"
    }, {
        "description" : "PSAD56 / Peru central zone",
        "code" : "EPSG:24892"
    }, {
        "description" : "PSAD56 / Peru east zone",
        "code" : "EPSG:24893"
    }, {
        "description" : "Leigon / Ghana Metre Grid",
        "code" : "EPSG:25000"
    }, {
        "description" : "Lome / UTM zone 31N",
        "code" : "EPSG:25231"
    }, {
        "description" : "Luzon 1911 / Philippines zone I",
        "code" : "EPSG:25391"
    }, {
        "description" : "Luzon 1911 / Philippines zone II",
        "code" : "EPSG:25392"
    }, {
        "description" : "Luzon 1911 / Philippines zone III",
        "code" : "EPSG:25393"
    }, {
        "description" : "Luzon 1911 / Philippines zone IV",
        "code" : "EPSG:25394"
    }, {
        "description" : "Luzon 1911 / Philippines zone V",
        "code" : "EPSG:25395"
    }, {
        "description" : "Makassar (Jakarta) / NEIEZ",
        "code" : "EPSG:25700"
    }, {
        "description" : "ETRS89 / UTM zone 28N",
        "code" : "EPSG:25828"
    }, {
        "description" : "ETRS89 / UTM zone 29N",
        "code" : "EPSG:25829"
    }, {
        "description" : "ETRS89 / UTM zone 30N",
        "code" : "EPSG:25830"
    }, {
        "description" : "ETRS89 / UTM zone 31N",
        "code" : "EPSG:25831"
    }, {
        "description" : "ETRS89 / UTM zone 32N",
        "code" : "EPSG:25832"
    }, {
        "description" : "ETRS89 / UTM zone 33N",
        "code" : "EPSG:25833"
    }, {
        "description" : "ETRS89 / UTM zone 34N",
        "code" : "EPSG:25834"
    }, {
        "description" : "ETRS89 / UTM zone 35N",
        "code" : "EPSG:25835"
    }, {
        "description" : "ETRS89 / UTM zone 36N",
        "code" : "EPSG:25836"
    }, {
        "description" : "ETRS89 / UTM zone 37N",
        "code" : "EPSG:25837"
    }, {
        "description" : "ETRS89 / UTM zone 38N",
        "code" : "EPSG:25838"
    }, {
        "description" : "ETRS89 / TM Baltic93",
        "code" : "EPSG:25884"
    }, {
        "description" : "Malongo 1987 / UTM zone 32S",
        "code" : "EPSG:25932"
    }, {
        "description" : "Merchich / Nord Maroc",
        "code" : "EPSG:26191"
    }, {
        "description" : "Merchich / Sud Maroc",
        "code" : "EPSG:26192"
    }, {
        "description" : "Merchich / Sahara",
        "code" : "EPSG:26193"
    }, {
        "description" : "Merchich / Sahara Nord",
        "code" : "EPSG:26194"
    }, {
        "description" : "Merchich / Sahara Sud",
        "code" : "EPSG:26195"
    }, {
        "description" : "Massawa / UTM zone 37N",
        "code" : "EPSG:26237"
    }, {
        "description" : "Minna / UTM zone 31N",
        "code" : "EPSG:26331"
    }, {
        "description" : "Minna / UTM zone 32N",
        "code" : "EPSG:26332"
    }, {
        "description" : "Minna / Nigeria West Belt",
        "code" : "EPSG:26391"
    }, {
        "description" : "Minna / Nigeria Mid Belt",
        "code" : "EPSG:26392"
    }, {
        "description" : "Minna / Nigeria East Belt",
        "code" : "EPSG:26393"
    }, {
        "description" : "Mhast / UTM zone 32S",
        "code" : "EPSG:26432"
    }, {
        "description" : "Monte Mario (Rome) / Italy zone 1",
        "code" : "EPSG:26591"
    }, {
        "description" : "Monte Mario (Rome) / Italy zone 2",
        "code" : "EPSG:26592"
    }, {
        "description" : "M'poraloko / UTM zone 32N",
        "code" : "EPSG:26632"
    }, {
        "description" : "M'poraloko / UTM zone 32S",
        "code" : "EPSG:26692"
    }, {
        "description" : "NAD27 / UTM zone 1N",
        "code" : "EPSG:26701"
    }, {
        "description" : "NAD27 / UTM zone 2N",
        "code" : "EPSG:26702"
    }, {
        "description" : "NAD27 / UTM zone 3N",
        "code" : "EPSG:26703"
    }, {
        "description" : "NAD27 / UTM zone 4N",
        "code" : "EPSG:26704"
    }, {
        "description" : "NAD27 / UTM zone 5N",
        "code" : "EPSG:26705"
    }, {
        "description" : "NAD27 / UTM zone 6N",
        "code" : "EPSG:26706"
    }, {
        "description" : "NAD27 / UTM zone 7N",
        "code" : "EPSG:26707"
    }, {
        "description" : "NAD27 / UTM zone 8N",
        "code" : "EPSG:26708"
    }, {
        "description" : "NAD27 / UTM zone 9N",
        "code" : "EPSG:26709"
    }, {
        "description" : "NAD27 / UTM zone 10N",
        "code" : "EPSG:26710"
    }, {
        "description" : "NAD27 / UTM zone 11N",
        "code" : "EPSG:26711"
    }, {
        "description" : "NAD27 / UTM zone 12N",
        "code" : "EPSG:26712"
    }, {
        "description" : "NAD27 / UTM zone 13N",
        "code" : "EPSG:26713"
    }, {
        "description" : "NAD27 / UTM zone 14N",
        "code" : "EPSG:26714"
    }, {
        "description" : "NAD27 / UTM zone 15N",
        "code" : "EPSG:26715"
    }, {
        "description" : "NAD27 / UTM zone 16N",
        "code" : "EPSG:26716"
    }, {
        "description" : "NAD27 / UTM zone 17N",
        "code" : "EPSG:26717"
    }, {
        "description" : "NAD27 / UTM zone 18N",
        "code" : "EPSG:26718"
    }, {
        "description" : "NAD27 / UTM zone 19N",
        "code" : "EPSG:26719"
    }, {
        "description" : "NAD27 / UTM zone 20N",
        "code" : "EPSG:26720"
    }, {
        "description" : "NAD27 / UTM zone 21N",
        "code" : "EPSG:26721"
    }, {
        "description" : "NAD27 / UTM zone 22N",
        "code" : "EPSG:26722"
    }, {
        "description" : "NAD27 / Alabama East",
        "code" : "EPSG:26729"
    }, {
        "description" : "NAD27 / Alabama West",
        "code" : "EPSG:26730"
    }, {
        "description" : "NAD27 / Alaska zone 1",
        "code" : "EPSG:26731"
    }, {
        "description" : "NAD27 / Alaska zone 2",
        "code" : "EPSG:26732"
    }, {
        "description" : "NAD27 / Alaska zone 3",
        "code" : "EPSG:26733"
    }, {
        "description" : "NAD27 / Alaska zone 4",
        "code" : "EPSG:26734"
    }, {
        "description" : "NAD27 / Alaska zone 5",
        "code" : "EPSG:26735"
    }, {
        "description" : "NAD27 / Alaska zone 6",
        "code" : "EPSG:26736"
    }, {
        "description" : "NAD27 / Alaska zone 7",
        "code" : "EPSG:26737"
    }, {
        "description" : "NAD27 / Alaska zone 8",
        "code" : "EPSG:26738"
    }, {
        "description" : "NAD27 / Alaska zone 9",
        "code" : "EPSG:26739"
    }, {
        "description" : "NAD27 / Alaska zone 10",
        "code" : "EPSG:26740"
    }, {
        "description" : "NAD27 / California zone I",
        "code" : "EPSG:26741"
    }, {
        "description" : "NAD27 / California zone II",
        "code" : "EPSG:26742"
    }, {
        "description" : "NAD27 / California zone III",
        "code" : "EPSG:26743"
    }, {
        "description" : "NAD27 / California zone IV",
        "code" : "EPSG:26744"
    }, {
        "description" : "NAD27 / California zone V",
        "code" : "EPSG:26745"
    }, {
        "description" : "NAD27 / California zone VI",
        "code" : "EPSG:26746"
    }, {
        "description" : "NAD27 / California zone VII",
        "code" : "EPSG:26747"
    }, {
        "description" : "NAD27 / Arizona East",
        "code" : "EPSG:26748"
    }, {
        "description" : "NAD27 / Arizona Central",
        "code" : "EPSG:26749"
    }, {
        "description" : "NAD27 / Arizona West",
        "code" : "EPSG:26750"
    }, {
        "description" : "NAD27 / Arkansas North",
        "code" : "EPSG:26751"
    }, {
        "description" : "NAD27 / Arkansas South",
        "code" : "EPSG:26752"
    }, {
        "description" : "NAD27 / Colorado North",
        "code" : "EPSG:26753"
    }, {
        "description" : "NAD27 / Colorado Central",
        "code" : "EPSG:26754"
    }, {
        "description" : "NAD27 / Colorado South",
        "code" : "EPSG:26755"
    }, {
        "description" : "NAD27 / Connecticut",
        "code" : "EPSG:26756"
    }, {
        "description" : "NAD27 / Delaware",
        "code" : "EPSG:26757"
    }, {
        "description" : "NAD27 / Florida East",
        "code" : "EPSG:26758"
    }, {
        "description" : "NAD27 / Florida West",
        "code" : "EPSG:26759"
    }, {
        "description" : "NAD27 / Florida North",
        "code" : "EPSG:26760"
    }, {
        "description" : "NAD27 / Georgia East",
        "code" : "EPSG:26766"
    }, {
        "description" : "NAD27 / Georgia West",
        "code" : "EPSG:26767"
    }, {
        "description" : "NAD27 / Idaho East",
        "code" : "EPSG:26768"
    }, {
        "description" : "NAD27 / Idaho Central",
        "code" : "EPSG:26769"
    }, {
        "description" : "NAD27 / Idaho West",
        "code" : "EPSG:26770"
    }, {
        "description" : "NAD27 / Illinois East",
        "code" : "EPSG:26771"
    }, {
        "description" : "NAD27 / Illinois West",
        "code" : "EPSG:26772"
    }, {
        "description" : "NAD27 / Indiana East",
        "code" : "EPSG:26773"
    }, {
        "description" : "NAD27 / Indiana West",
        "code" : "EPSG:26774"
    }, {
        "description" : "NAD27 / Iowa North",
        "code" : "EPSG:26775"
    }, {
        "description" : "NAD27 / Iowa South",
        "code" : "EPSG:26776"
    }, {
        "description" : "NAD27 / Kansas North",
        "code" : "EPSG:26777"
    }, {
        "description" : "NAD27 / Kansas South",
        "code" : "EPSG:26778"
    }, {
        "description" : "NAD27 / Kentucky North",
        "code" : "EPSG:26779"
    }, {
        "description" : "NAD27 / Kentucky South",
        "code" : "EPSG:26780"
    }, {
        "description" : "NAD27 / Louisiana North",
        "code" : "EPSG:26781"
    }, {
        "description" : "NAD27 / Louisiana South",
        "code" : "EPSG:26782"
    }, {
        "description" : "NAD27 / Maine East",
        "code" : "EPSG:26783"
    }, {
        "description" : "NAD27 / Maine West",
        "code" : "EPSG:26784"
    }, {
        "description" : "NAD27 / Maryland",
        "code" : "EPSG:26785"
    }, {
        "description" : "NAD27 / Massachusetts Mainland",
        "code" : "EPSG:26786"
    }, {
        "description" : "NAD27 / Massachusetts Island",
        "code" : "EPSG:26787"
    }, {
        "description" : "NAD27 / Minnesota North",
        "code" : "EPSG:26791"
    }, {
        "description" : "NAD27 / Minnesota Central",
        "code" : "EPSG:26792"
    }, {
        "description" : "NAD27 / Minnesota South",
        "code" : "EPSG:26793"
    }, {
        "description" : "NAD27 / Mississippi East",
        "code" : "EPSG:26794"
    }, {
        "description" : "NAD27 / Mississippi West",
        "code" : "EPSG:26795"
    }, {
        "description" : "NAD27 / Missouri East",
        "code" : "EPSG:26796"
    }, {
        "description" : "NAD27 / Missouri Central",
        "code" : "EPSG:26797"
    }, {
        "description" : "NAD27 / Missouri West",
        "code" : "EPSG:26798"
    }, {
        "description" : "NAD27 / California zone VII",
        "code" : "EPSG:26799"
    }, {
        "description" : "NAD Michigan / Michigan East",
        "code" : "EPSG:26801"
    }, {
        "description" : "NAD Michigan / Michigan Old Central",
        "code" : "EPSG:26802"
    }, {
        "description" : "NAD Michigan / Michigan West",
        "code" : "EPSG:26803"
    }, {
        "description" : "NAD Michigan / Michigan North",
        "code" : "EPSG:26811"
    }, {
        "description" : "NAD Michigan / Michigan Central",
        "code" : "EPSG:26812"
    }, {
        "description" : "NAD Michigan / Michigan South",
        "code" : "EPSG:26813"
    }, {
        "description" : "NAD83 / Maine East (ftUS)",
        "code" : "EPSG:26814"
    }, {
        "description" : "NAD83 / Maine West (ftUS)",
        "code" : "EPSG:26815"
    }, {
        "description" : "NAD83 / Minnesota North (ftUS)",
        "code" : "EPSG:26819"
    }, {
        "description" : "NAD83 / Minnesota Central (ftUS)",
        "code" : "EPSG:26820"
    }, {
        "description" : "NAD83 / Minnesota South (ftUS)",
        "code" : "EPSG:26821"
    }, {
        "description" : "NAD83 / Nebraska (ftUS)",
        "code" : "EPSG:26822"
    }, {
        "description" : "NAD83 / West Virginia North (ftUS)",
        "code" : "EPSG:26823"
    }, {
        "description" : "NAD83 / West Virginia South (ftUS)",
        "code" : "EPSG:26824"
    }, {
        "description" : "NAD83(HARN) / Maine East (ftUS)",
        "code" : "EPSG:26825"
    }, {
        "description" : "NAD83(HARN) / Maine West (ftUS)",
        "code" : "EPSG:26826"
    }, {
        "description" : "NAD83(HARN) / Minnesota North (ftUS)",
        "code" : "EPSG:26830"
    }, {
        "description" : "NAD83(HARN) / Minnesota Central (ftUS)",
        "code" : "EPSG:26831"
    }, {
        "description" : "NAD83(HARN) / Minnesota South (ftUS)",
        "code" : "EPSG:26832"
    }, {
        "description" : "NAD83(HARN) / Nebraska (ftUS)",
        "code" : "EPSG:26833"
    }, {
        "description" : "NAD83(HARN) / West Virginia North (ftUS)",
        "code" : "EPSG:26834"
    }, {
        "description" : "NAD83(HARN) / West Virginia South (ftUS)",
        "code" : "EPSG:26835"
    }, {
        "description" : "NAD83(NSRS2007) / Maine East (ftUS)",
        "code" : "EPSG:26836"
    }, {
        "description" : "NAD83(NSRS2007) / Maine West (ftUS)",
        "code" : "EPSG:26837"
    }, {
        "description" : "NAD83(NSRS2007) / Minnesota North (ftUS)",
        "code" : "EPSG:26841"
    }, {
        "description" : "NAD83(NSRS2007) / Minnesota Central (ftUS)",
        "code" : "EPSG:26842"
    }, {
        "description" : "NAD83(NSRS2007) / Minnesota South (ftUS)",
        "code" : "EPSG:26843"
    }, {
        "description" : "NAD83(NSRS2007) / Nebraska (ftUS)",
        "code" : "EPSG:26844"
    }, {
        "description" : "NAD83(NSRS2007) / West Virginia North (ftUS)",
        "code" : "EPSG:26845"
    }, {
        "description" : "NAD83(NSRS2007) / West Virginia South (ftUS)",
        "code" : "EPSG:26846"
    }, {
        "description" : "NAD83 / Maine East (ftUS)",
        "code" : "EPSG:26847"
    }, {
        "description" : "NAD83 / Maine West (ftUS)",
        "code" : "EPSG:26848"
    }, {
        "description" : "NAD83 / Minnesota North (ftUS)",
        "code" : "EPSG:26849"
    }, {
        "description" : "NAD83 / Minnesota Central (ftUS)",
        "code" : "EPSG:26850"
    }, {
        "description" : "NAD83 / Minnesota South (ftUS)",
        "code" : "EPSG:26851"
    }, {
        "description" : "NAD83 / Nebraska (ftUS)",
        "code" : "EPSG:26852"
    }, {
        "description" : "NAD83 / West Virginia North (ftUS)",
        "code" : "EPSG:26853"
    }, {
        "description" : "NAD83 / West Virginia South (ftUS)",
        "code" : "EPSG:26854"
    }, {
        "description" : "NAD83(HARN) / Maine East (ftUS)",
        "code" : "EPSG:26855"
    }, {
        "description" : "NAD83(HARN) / Maine West (ftUS)",
        "code" : "EPSG:26856"
    }, {
        "description" : "NAD83(HARN) / Minnesota North (ftUS)",
        "code" : "EPSG:26857"
    }, {
        "description" : "NAD83(HARN) / Minnesota Central (ftUS)",
        "code" : "EPSG:26858"
    }, {
        "description" : "NAD83(HARN) / Minnesota South (ftUS)",
        "code" : "EPSG:26859"
    }, {
        "description" : "NAD83(HARN) / Nebraska (ftUS)",
        "code" : "EPSG:26860"
    }, {
        "description" : "NAD83(HARN) / West Virginia North (ftUS)",
        "code" : "EPSG:26861"
    }, {
        "description" : "NAD83(HARN) / West Virginia South (ftUS)",
        "code" : "EPSG:26862"
    }, {
        "description" : "NAD83(NSRS2007) / Maine East (ftUS)",
        "code" : "EPSG:26863"
    }, {
        "description" : "NAD83(NSRS2007) / Maine West (ftUS)",
        "code" : "EPSG:26864"
    }, {
        "description" : "NAD83(NSRS2007) / Minnesota North (ftUS)",
        "code" : "EPSG:26865"
    }, {
        "description" : "NAD83(NSRS2007) / Minnesota Central (ftUS)",
        "code" : "EPSG:26866"
    }, {
        "description" : "NAD83(NSRS2007) / Minnesota South (ftUS)",
        "code" : "EPSG:26867"
    }, {
        "description" : "NAD83(NSRS2007) / Nebraska (ftUS)",
        "code" : "EPSG:26868"
    }, {
        "description" : "NAD83(NSRS2007) / West Virginia North (ftUS)",
        "code" : "EPSG:26869"
    }, {
        "description" : "NAD83(NSRS2007) / West Virginia South (ftUS)",
        "code" : "EPSG:26870"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 11",
        "code" : "EPSG:26891"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 12",
        "code" : "EPSG:26892"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 13",
        "code" : "EPSG:26893"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 14",
        "code" : "EPSG:26894"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 15",
        "code" : "EPSG:26895"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 16",
        "code" : "EPSG:26896"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 17",
        "code" : "EPSG:26897"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 1",
        "code" : "EPSG:26898"
    }, {
        "description" : "NAD83(CSRS) / MTM zone 2",
        "code" : "EPSG:26899"
    }, {
        "description" : "NAD83 / UTM zone 1N",
        "code" : "EPSG:26901"
    }, {
        "description" : "NAD83 / UTM zone 2N",
        "code" : "EPSG:26902"
    }, {
        "description" : "NAD83 / UTM zone 3N",
        "code" : "EPSG:26903"
    }, {
        "description" : "NAD83 / UTM zone 4N",
        "code" : "EPSG:26904"
    }, {
        "description" : "NAD83 / UTM zone 5N",
        "code" : "EPSG:26905"
    }, {
        "description" : "NAD83 / UTM zone 6N",
        "code" : "EPSG:26906"
    }, {
        "description" : "NAD83 / UTM zone 7N",
        "code" : "EPSG:26907"
    }, {
        "description" : "NAD83 / UTM zone 8N",
        "code" : "EPSG:26908"
    }, {
        "description" : "NAD83 / UTM zone 9N",
        "code" : "EPSG:26909"
    }, {
        "description" : "NAD83 / UTM zone 10N",
        "code" : "EPSG:26910"
    }, {
        "description" : "NAD83 / UTM zone 11N",
        "code" : "EPSG:26911"
    }, {
        "description" : "NAD83 / UTM zone 12N",
        "code" : "EPSG:26912"
    }, {
        "description" : "NAD83 / UTM zone 13N",
        "code" : "EPSG:26913"
    }, {
        "description" : "NAD83 / UTM zone 14N",
        "code" : "EPSG:26914"
    }, {
        "description" : "NAD83 / UTM zone 15N",
        "code" : "EPSG:26915"
    }, {
        "description" : "NAD83 / UTM zone 16N",
        "code" : "EPSG:26916"
    }, {
        "description" : "NAD83 / UTM zone 17N",
        "code" : "EPSG:26917"
    }, {
        "description" : "NAD83 / UTM zone 18N",
        "code" : "EPSG:26918"
    }, {
        "description" : "NAD83 / UTM zone 19N",
        "code" : "EPSG:26919"
    }, {
        "description" : "NAD83 / UTM zone 20N",
        "code" : "EPSG:26920"
    }, {
        "description" : "NAD83 / UTM zone 21N",
        "code" : "EPSG:26921"
    }, {
        "description" : "NAD83 / UTM zone 22N",
        "code" : "EPSG:26922"
    }, {
        "description" : "NAD83 / UTM zone 23N",
        "code" : "EPSG:26923"
    }, {
        "description" : "NAD83 / Alabama East",
        "code" : "EPSG:26929"
    }, {
        "description" : "NAD83 / Alabama West",
        "code" : "EPSG:26930"
    }, {
        "description" : "NAD83 / Alaska zone 1",
        "code" : "EPSG:26931"
    }, {
        "description" : "NAD83 / Alaska zone 2",
        "code" : "EPSG:26932"
    }, {
        "description" : "NAD83 / Alaska zone 3",
        "code" : "EPSG:26933"
    }, {
        "description" : "NAD83 / Alaska zone 4",
        "code" : "EPSG:26934"
    }, {
        "description" : "NAD83 / Alaska zone 5",
        "code" : "EPSG:26935"
    }, {
        "description" : "NAD83 / Alaska zone 6",
        "code" : "EPSG:26936"
    }, {
        "description" : "NAD83 / Alaska zone 7",
        "code" : "EPSG:26937"
    }, {
        "description" : "NAD83 / Alaska zone 8",
        "code" : "EPSG:26938"
    }, {
        "description" : "NAD83 / Alaska zone 9",
        "code" : "EPSG:26939"
    }, {
        "description" : "NAD83 / Alaska zone 10",
        "code" : "EPSG:26940"
    }, {
        "description" : "NAD83 / California zone 1",
        "code" : "EPSG:26941"
    }, {
        "description" : "NAD83 / California zone 2",
        "code" : "EPSG:26942"
    }, {
        "description" : "NAD83 / California zone 3",
        "code" : "EPSG:26943"
    }, {
        "description" : "NAD83 / California zone 4",
        "code" : "EPSG:26944"
    }, {
        "description" : "NAD83 / California zone 5",
        "code" : "EPSG:26945"
    }, {
        "description" : "NAD83 / California zone 6",
        "code" : "EPSG:26946"
    }, {
        "description" : "NAD83 / Arizona East",
        "code" : "EPSG:26948"
    }, {
        "description" : "NAD83 / Arizona Central",
        "code" : "EPSG:26949"
    }, {
        "description" : "NAD83 / Arizona West",
        "code" : "EPSG:26950"
    }, {
        "description" : "NAD83 / Arkansas North",
        "code" : "EPSG:26951"
    }, {
        "description" : "NAD83 / Arkansas South",
        "code" : "EPSG:26952"
    }, {
        "description" : "NAD83 / Colorado North",
        "code" : "EPSG:26953"
    }, {
        "description" : "NAD83 / Colorado Central",
        "code" : "EPSG:26954"
    }, {
        "description" : "NAD83 / Colorado South",
        "code" : "EPSG:26955"
    }, {
        "description" : "NAD83 / Connecticut",
        "code" : "EPSG:26956"
    }, {
        "description" : "NAD83 / Delaware",
        "code" : "EPSG:26957"
    }, {
        "description" : "NAD83 / Florida East",
        "code" : "EPSG:26958"
    }, {
        "description" : "NAD83 / Florida West",
        "code" : "EPSG:26959"
    }, {
        "description" : "NAD83 / Florida North",
        "code" : "EPSG:26960"
    }, {
        "description" : "NAD83 / Hawaii zone 1",
        "code" : "EPSG:26961"
    }, {
        "description" : "NAD83 / Hawaii zone 2",
        "code" : "EPSG:26962"
    }, {
        "description" : "NAD83 / Hawaii zone 3",
        "code" : "EPSG:26963"
    }, {
        "description" : "NAD83 / Hawaii zone 4",
        "code" : "EPSG:26964"
    }, {
        "description" : "NAD83 / Hawaii zone 5",
        "code" : "EPSG:26965"
    }, {
        "description" : "NAD83 / Georgia East",
        "code" : "EPSG:26966"
    }, {
        "description" : "NAD83 / Georgia West",
        "code" : "EPSG:26967"
    }, {
        "description" : "NAD83 / Idaho East",
        "code" : "EPSG:26968"
    }, {
        "description" : "NAD83 / Idaho Central",
        "code" : "EPSG:26969"
    }, {
        "description" : "NAD83 / Idaho West",
        "code" : "EPSG:26970"
    }, {
        "description" : "NAD83 / Illinois East",
        "code" : "EPSG:26971"
    }, {
        "description" : "NAD83 / Illinois West",
        "code" : "EPSG:26972"
    }, {
        "description" : "NAD83 / Indiana East",
        "code" : "EPSG:26973"
    }, {
        "description" : "NAD83 / Indiana West",
        "code" : "EPSG:26974"
    }, {
        "description" : "NAD83 / Iowa North",
        "code" : "EPSG:26975"
    }, {
        "description" : "NAD83 / Iowa South",
        "code" : "EPSG:26976"
    }, {
        "description" : "NAD83 / Kansas North",
        "code" : "EPSG:26977"
    }, {
        "description" : "NAD83 / Kansas South",
        "code" : "EPSG:26978"
    }, {
        "description" : "NAD83 / Kentucky North",
        "code" : "EPSG:26979"
    }, {
        "description" : "NAD83 / Kentucky South",
        "code" : "EPSG:26980"
    }, {
        "description" : "NAD83 / Louisiana North",
        "code" : "EPSG:26981"
    }, {
        "description" : "NAD83 / Louisiana South",
        "code" : "EPSG:26982"
    }, {
        "description" : "NAD83 / Maine East",
        "code" : "EPSG:26983"
    }, {
        "description" : "NAD83 / Maine West",
        "code" : "EPSG:26984"
    }, {
        "description" : "NAD83 / Maryland",
        "code" : "EPSG:26985"
    }, {
        "description" : "NAD83 / Massachusetts Mainland",
        "code" : "EPSG:26986"
    }, {
        "description" : "NAD83 / Massachusetts Island",
        "code" : "EPSG:26987"
    }, {
        "description" : "NAD83 / Michigan North",
        "code" : "EPSG:26988"
    }, {
        "description" : "NAD83 / Michigan Central",
        "code" : "EPSG:26989"
    }, {
        "description" : "NAD83 / Michigan South",
        "code" : "EPSG:26990"
    }, {
        "description" : "NAD83 / Minnesota North",
        "code" : "EPSG:26991"
    }, {
        "description" : "NAD83 / Minnesota Central",
        "code" : "EPSG:26992"
    }, {
        "description" : "NAD83 / Minnesota South",
        "code" : "EPSG:26993"
    }, {
        "description" : "NAD83 / Mississippi East",
        "code" : "EPSG:26994"
    }, {
        "description" : "NAD83 / Mississippi West",
        "code" : "EPSG:26995"
    }, {
        "description" : "NAD83 / Missouri East",
        "code" : "EPSG:26996"
    }, {
        "description" : "NAD83 / Missouri Central",
        "code" : "EPSG:26997"
    }, {
        "description" : "NAD83 / Missouri West",
        "code" : "EPSG:26998"
    }, {
        "description" : "Nahrwan 1967 / UTM zone 37N",
        "code" : "EPSG:27037"
    }, {
        "description" : "Nahrwan 1967 / UTM zone 38N",
        "code" : "EPSG:27038"
    }, {
        "description" : "Nahrwan 1967 / UTM zone 39N",
        "code" : "EPSG:27039"
    }, {
        "description" : "Nahrwan 1967 / UTM zone 40N",
        "code" : "EPSG:27040"
    }, {
        "description" : "Naparima 1972 / UTM zone 20N",
        "code" : "EPSG:27120"
    }, {
        "description" : "NZGD49 / New Zealand Map Grid",
        "code" : "EPSG:27200"
    }, {
        "description" : "NZGD49 / Mount Eden Circuit",
        "code" : "EPSG:27205"
    }, {
        "description" : "NZGD49 / Bay of Plenty Circuit",
        "code" : "EPSG:27206"
    }, {
        "description" : "NZGD49 / Poverty Bay Circuit",
        "code" : "EPSG:27207"
    }, {
        "description" : "NZGD49 / Hawkes Bay Circuit",
        "code" : "EPSG:27208"
    }, {
        "description" : "NZGD49 / Taranaki Circuit",
        "code" : "EPSG:27209"
    }, {
        "description" : "NZGD49 / Tuhirangi Circuit",
        "code" : "EPSG:27210"
    }, {
        "description" : "NZGD49 / Wanganui Circuit",
        "code" : "EPSG:27211"
    }, {
        "description" : "NZGD49 / Wairarapa Circuit",
        "code" : "EPSG:27212"
    }, {
        "description" : "NZGD49 / Wellington Circuit",
        "code" : "EPSG:27213"
    }, {
        "description" : "NZGD49 / Collingwood Circuit",
        "code" : "EPSG:27214"
    }, {
        "description" : "NZGD49 / Nelson Circuit",
        "code" : "EPSG:27215"
    }, {
        "description" : "NZGD49 / Karamea Circuit",
        "code" : "EPSG:27216"
    }, {
        "description" : "NZGD49 / Buller Circuit",
        "code" : "EPSG:27217"
    }, {
        "description" : "NZGD49 / Grey Circuit",
        "code" : "EPSG:27218"
    }, {
        "description" : "NZGD49 / Amuri Circuit",
        "code" : "EPSG:27219"
    }, {
        "description" : "NZGD49 / Marlborough Circuit",
        "code" : "EPSG:27220"
    }, {
        "description" : "NZGD49 / Hokitika Circuit",
        "code" : "EPSG:27221"
    }, {
        "description" : "NZGD49 / Okarito Circuit",
        "code" : "EPSG:27222"
    }, {
        "description" : "NZGD49 / Jacksons Bay Circuit",
        "code" : "EPSG:27223"
    }, {
        "description" : "NZGD49 / Mount Pleasant Circuit",
        "code" : "EPSG:27224"
    }, {
        "description" : "NZGD49 / Gawler Circuit",
        "code" : "EPSG:27225"
    }, {
        "description" : "NZGD49 / Timaru Circuit",
        "code" : "EPSG:27226"
    }, {
        "description" : "NZGD49 / Lindis Peak Circuit",
        "code" : "EPSG:27227"
    }, {
        "description" : "NZGD49 / Mount Nicholas Circuit",
        "code" : "EPSG:27228"
    }, {
        "description" : "NZGD49 / Mount York Circuit",
        "code" : "EPSG:27229"
    }, {
        "description" : "NZGD49 / Observation Point Circuit",
        "code" : "EPSG:27230"
    }, {
        "description" : "NZGD49 / North Taieri Circuit",
        "code" : "EPSG:27231"
    }, {
        "description" : "NZGD49 / Bluff Circuit",
        "code" : "EPSG:27232"
    }, {
        "description" : "NZGD49 / UTM zone 58S",
        "code" : "EPSG:27258"
    }, {
        "description" : "NZGD49 / UTM zone 59S",
        "code" : "EPSG:27259"
    }, {
        "description" : "NZGD49 / UTM zone 60S",
        "code" : "EPSG:27260"
    }, {
        "description" : "NZGD49 / North Island Grid",
        "code" : "EPSG:27291"
    }, {
        "description" : "NZGD49 / South Island Grid",
        "code" : "EPSG:27292"
    }, {
        "description" : "NGO 1948 (Oslo) / NGO zone I",
        "code" : "EPSG:27391"
    }, {
        "description" : "NGO 1948 (Oslo) / NGO zone II",
        "code" : "EPSG:27392"
    }, {
        "description" : "NGO 1948 (Oslo) / NGO zone III",
        "code" : "EPSG:27393"
    }, {
        "description" : "NGO 1948 (Oslo) / NGO zone IV",
        "code" : "EPSG:27394"
    }, {
        "description" : "NGO 1948 (Oslo) / NGO zone V",
        "code" : "EPSG:27395"
    }, {
        "description" : "NGO 1948 (Oslo) / NGO zone VI",
        "code" : "EPSG:27396"
    }, {
        "description" : "NGO 1948 (Oslo) / NGO zone VII",
        "code" : "EPSG:27397"
    }, {
        "description" : "NGO 1948 (Oslo) / NGO zone VIII",
        "code" : "EPSG:27398"
    }, {
        "description" : "Datum 73 / UTM zone 29N",
        "code" : "EPSG:27429"
    }, {
        "description" : "Datum 73 / Modified Portuguese Grid",
        "code" : "EPSG:27492"
    }, {
        "description" : "Datum 73 / Modified Portuguese Grid",
        "code" : "EPSG:27493"
    }, {
        "description" : "ATF (Paris) / Nord de Guerre",
        "code" : "EPSG:27500"
    }, {
        "description" : "NTF (Paris) / Lambert Nord France",
        "code" : "EPSG:27561"
    }, {
        "description" : "NTF (Paris) / Lambert Centre France",
        "code" : "EPSG:27562"
    }, {
        "description" : "NTF (Paris) / Lambert Sud France",
        "code" : "EPSG:27563"
    }, {
        "description" : "NTF (Paris) / Lambert Corse",
        "code" : "EPSG:27564"
    }, {
        "description" : "NTF (Paris) / Lambert zone I",
        "code" : "EPSG:27571"
    }, {
        "description" : "NTF (Paris) / Lambert zone II",
        "code" : "EPSG:27572"
    }, {
        "description" : "NTF (Paris) / Lambert zone III",
        "code" : "EPSG:27573"
    }, {
        "description" : "NTF (Paris) / Lambert zone IV",
        "code" : "EPSG:27574"
    }, {
        "description" : "NTF (Paris) / France I",
        "code" : "EPSG:27581"
    }, {
        "description" : "NTF (Paris) / France II",
        "code" : "EPSG:27582"
    }, {
        "description" : "NTF (Paris) / France III",
        "code" : "EPSG:27583"
    }, {
        "description" : "NTF (Paris) / France IV",
        "code" : "EPSG:27584"
    }, {
        "description" : "NTF (Paris) / Nord France",
        "code" : "EPSG:27591"
    }, {
        "description" : "NTF (Paris) / Centre France",
        "code" : "EPSG:27592"
    }, {
        "description" : "NTF (Paris) / Sud France",
        "code" : "EPSG:27593"
    }, {
        "description" : "NTF (Paris) / Corse",
        "code" : "EPSG:27594"
    }, {
        "description" : "OSGB 1936 / British National Grid",
        "code" : "EPSG:27700"
    }, {
        "description" : "Palestine 1923 / Palestine Grid",
        "code" : "EPSG:28191"
    }, {
        "description" : "Palestine 1923 / Palestine Belt",
        "code" : "EPSG:28192"
    }, {
        "description" : "Palestine 1923 / Israeli CS Grid",
        "code" : "EPSG:28193"
    }, {
        "description" : "Pointe Noire / UTM zone 32S",
        "code" : "EPSG:28232"
    }, {
        "description" : "GDA94 / MGA zone 48",
        "code" : "EPSG:28348"
    }, {
        "description" : "GDA94 / MGA zone 49",
        "code" : "EPSG:28349"
    }, {
        "description" : "GDA94 / MGA zone 50",
        "code" : "EPSG:28350"
    }, {
        "description" : "GDA94 / MGA zone 51",
        "code" : "EPSG:28351"
    }, {
        "description" : "GDA94 / MGA zone 52",
        "code" : "EPSG:28352"
    }, {
        "description" : "GDA94 / MGA zone 53",
        "code" : "EPSG:28353"
    }, {
        "description" : "GDA94 / MGA zone 54",
        "code" : "EPSG:28354"
    }, {
        "description" : "GDA94 / MGA zone 55",
        "code" : "EPSG:28355"
    }, {
        "description" : "GDA94 / MGA zone 56",
        "code" : "EPSG:28356"
    }, {
        "description" : "GDA94 / MGA zone 57",
        "code" : "EPSG:28357"
    }, {
        "description" : "GDA94 / MGA zone 58",
        "code" : "EPSG:28358"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 2",
        "code" : "EPSG:28402"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 3",
        "code" : "EPSG:28403"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 4",
        "code" : "EPSG:28404"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 5",
        "code" : "EPSG:28405"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 6",
        "code" : "EPSG:28406"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 7",
        "code" : "EPSG:28407"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 8",
        "code" : "EPSG:28408"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 9",
        "code" : "EPSG:28409"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 10",
        "code" : "EPSG:28410"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 11",
        "code" : "EPSG:28411"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 12",
        "code" : "EPSG:28412"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 13",
        "code" : "EPSG:28413"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 14",
        "code" : "EPSG:28414"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 15",
        "code" : "EPSG:28415"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 16",
        "code" : "EPSG:28416"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 17",
        "code" : "EPSG:28417"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 18",
        "code" : "EPSG:28418"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 19",
        "code" : "EPSG:28419"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 20",
        "code" : "EPSG:28420"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 21",
        "code" : "EPSG:28421"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 22",
        "code" : "EPSG:28422"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 23",
        "code" : "EPSG:28423"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 24",
        "code" : "EPSG:28424"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 25",
        "code" : "EPSG:28425"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 26",
        "code" : "EPSG:28426"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 27",
        "code" : "EPSG:28427"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 28",
        "code" : "EPSG:28428"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 29",
        "code" : "EPSG:28429"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 30",
        "code" : "EPSG:28430"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 31",
        "code" : "EPSG:28431"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger zone 32",
        "code" : "EPSG:28432"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 2N",
        "code" : "EPSG:28462"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 3N",
        "code" : "EPSG:28463"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 4N",
        "code" : "EPSG:28464"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 5N",
        "code" : "EPSG:28465"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 6N",
        "code" : "EPSG:28466"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 7N",
        "code" : "EPSG:28467"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 8N",
        "code" : "EPSG:28468"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 9N",
        "code" : "EPSG:28469"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 10N",
        "code" : "EPSG:28470"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 11N",
        "code" : "EPSG:28471"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 12N",
        "code" : "EPSG:28472"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 13N",
        "code" : "EPSG:28473"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 14N",
        "code" : "EPSG:28474"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 15N",
        "code" : "EPSG:28475"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 16N",
        "code" : "EPSG:28476"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 17N",
        "code" : "EPSG:28477"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 18N",
        "code" : "EPSG:28478"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 19N",
        "code" : "EPSG:28479"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 20N",
        "code" : "EPSG:28480"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 21N",
        "code" : "EPSG:28481"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 22N",
        "code" : "EPSG:28482"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 23N",
        "code" : "EPSG:28483"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 24N",
        "code" : "EPSG:28484"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 25N",
        "code" : "EPSG:28485"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 26N",
        "code" : "EPSG:28486"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 27N",
        "code" : "EPSG:28487"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 28N",
        "code" : "EPSG:28488"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 29N",
        "code" : "EPSG:28489"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 30N",
        "code" : "EPSG:28490"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 31N",
        "code" : "EPSG:28491"
    }, {
        "description" : "Pulkovo 1942 / Gauss-Kruger 32N",
        "code" : "EPSG:28492"
    }, {
        "description" : "Qatar 1974 / Qatar National Grid",
        "code" : "EPSG:28600"
    }, {
        "description" : "Amersfoort / RD Old",
        "code" : "EPSG:28991"
    }, {
        "description" : "Amersfoort / RD New",
        "code" : "EPSG:28992"
    }, {
        "description" : "SAD69 / Brazil Polyconic",
        "code" : "EPSG:29100"
    }, {
        "description" : "SAD69 / Brazil Polyconic",
        "code" : "EPSG:29101"
    }, {
        "description" : "SAD69 / UTM zone 18N",
        "code" : "EPSG:29118"
    }, {
        "description" : "SAD69 / UTM zone 19N",
        "code" : "EPSG:29119"
    }, {
        "description" : "SAD69 / UTM zone 20N",
        "code" : "EPSG:29120"
    }, {
        "description" : "SAD69 / UTM zone 21N",
        "code" : "EPSG:29121"
    }, {
        "description" : "SAD69 / UTM zone 22N",
        "code" : "EPSG:29122"
    }, {
        "description" : "SAD69 / UTM zone 18N",
        "code" : "EPSG:29168"
    }, {
        "description" : "SAD69 / UTM zone 19N",
        "code" : "EPSG:29169"
    }, {
        "description" : "SAD69 / UTM zone 20N",
        "code" : "EPSG:29170"
    }, {
        "description" : "SAD69 / UTM zone 21N",
        "code" : "EPSG:29171"
    }, {
        "description" : "SAD69 / UTM zone 22N",
        "code" : "EPSG:29172"
    }, {
        "description" : "SAD69 / UTM zone 17S",
        "code" : "EPSG:29177"
    }, {
        "description" : "SAD69 / UTM zone 18S",
        "code" : "EPSG:29178"
    }, {
        "description" : "SAD69 / UTM zone 19S",
        "code" : "EPSG:29179"
    }, {
        "description" : "SAD69 / UTM zone 20S",
        "code" : "EPSG:29180"
    }, {
        "description" : "SAD69 / UTM zone 21S",
        "code" : "EPSG:29181"
    }, {
        "description" : "SAD69 / UTM zone 22S",
        "code" : "EPSG:29182"
    }, {
        "description" : "SAD69 / UTM zone 23S",
        "code" : "EPSG:29183"
    }, {
        "description" : "SAD69 / UTM zone 24S",
        "code" : "EPSG:29184"
    }, {
        "description" : "SAD69 / UTM zone 25S",
        "code" : "EPSG:29185"
    }, {
        "description" : "SAD69 / UTM zone 17S",
        "code" : "EPSG:29187"
    }, {
        "description" : "SAD69 / UTM zone 18S",
        "code" : "EPSG:29188"
    }, {
        "description" : "SAD69 / UTM zone 19S",
        "code" : "EPSG:29189"
    }, {
        "description" : "SAD69 / UTM zone 20S",
        "code" : "EPSG:29190"
    }, {
        "description" : "SAD69 / UTM zone 21S",
        "code" : "EPSG:29191"
    }, {
        "description" : "SAD69 / UTM zone 22S",
        "code" : "EPSG:29192"
    }, {
        "description" : "SAD69 / UTM zone 23S",
        "code" : "EPSG:29193"
    }, {
        "description" : "SAD69 / UTM zone 24S",
        "code" : "EPSG:29194"
    }, {
        "description" : "SAD69 / UTM zone 25S",
        "code" : "EPSG:29195"
    }, {
        "description" : "Sapper Hill 1943 / UTM zone 20S",
        "code" : "EPSG:29220"
    }, {
        "description" : "Sapper Hill 1943 / UTM zone 21S",
        "code" : "EPSG:29221"
    }, {
        "description" : "Schwarzeck / UTM zone 33S",
        "code" : "EPSG:29333"
    }, {
        "description" : "Schwarzeck / Lo22/11",
        "code" : "EPSG:29371"
    }, {
        "description" : "Schwarzeck / Lo22/13",
        "code" : "EPSG:29373"
    }, {
        "description" : "Schwarzeck / Lo22/15",
        "code" : "EPSG:29375"
    }, {
        "description" : "Schwarzeck / Lo22/17",
        "code" : "EPSG:29377"
    }, {
        "description" : "Schwarzeck / Lo22/19",
        "code" : "EPSG:29379"
    }, {
        "description" : "Schwarzeck / Lo22/21",
        "code" : "EPSG:29381"
    }, {
        "description" : "Schwarzeck / Lo22/23",
        "code" : "EPSG:29383"
    }, {
        "description" : "Schwarzeck / Lo22/25",
        "code" : "EPSG:29385"
    }, {
        "description" : "Sudan / UTM zone 35N",
        "code" : "EPSG:29635"
    }, {
        "description" : "Sudan / UTM zone 36N",
        "code" : "EPSG:29636"
    }, {
        "description" : "Tananarive (Paris) / Laborde Grid",
        "code" : "EPSG:29700"
    }, {
        "description" : "Tananarive (Paris) / Laborde Grid",
        "code" : "EPSG:29701"
    }, {
        "description" : "Tananarive (Paris) / Laborde Grid approximation",
        "code" : "EPSG:29702"
    }, {
        "description" : "Tananarive / UTM zone 38S",
        "code" : "EPSG:29738"
    }, {
        "description" : "Tananarive / UTM zone 39S",
        "code" : "EPSG:29739"
    }, {
        "description" : "Timbalai 1948 / UTM zone 49N",
        "code" : "EPSG:29849"
    }, {
        "description" : "Timbalai 1948 / UTM zone 50N",
        "code" : "EPSG:29850"
    }, {
        "description" : "Timbalai 1948 / RSO Borneo (ch)",
        "code" : "EPSG:29871"
    }, {
        "description" : "Timbalai 1948 / RSO Borneo (ft)",
        "code" : "EPSG:29872"
    }, {
        "description" : "Timbalai 1948 / RSO Borneo (m)",
        "code" : "EPSG:29873"
    }, {
        "description" : "TM65 / Irish National Grid",
        "code" : "EPSG:29900"
    }, {
        "description" : "OSNI 1952 / Irish National Grid",
        "code" : "EPSG:29901"
    }, {
        "description" : "TM65 / Irish Grid",
        "code" : "EPSG:29902"
    }, {
        "description" : "TM75 / Irish Grid",
        "code" : "EPSG:29903"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS I",
        "code" : "EPSG:30161"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS II",
        "code" : "EPSG:30162"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS III",
        "code" : "EPSG:30163"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS IV",
        "code" : "EPSG:30164"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS V",
        "code" : "EPSG:30165"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS VI",
        "code" : "EPSG:30166"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS VII",
        "code" : "EPSG:30167"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS VIII",
        "code" : "EPSG:30168"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS IX",
        "code" : "EPSG:30169"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS X",
        "code" : "EPSG:30170"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS XI",
        "code" : "EPSG:30171"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS XII",
        "code" : "EPSG:30172"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS XIII",
        "code" : "EPSG:30173"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS XIV",
        "code" : "EPSG:30174"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS XV",
        "code" : "EPSG:30175"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS XVI",
        "code" : "EPSG:30176"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS XVII",
        "code" : "EPSG:30177"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS XVIII",
        "code" : "EPSG:30178"
    }, {
        "description" : "Tokyo / Japan Plane Rectangular CS XIX",
        "code" : "EPSG:30179"
    }, {
        "description" : "Trinidad 1903 / Trinidad Grid",
        "code" : "EPSG:30200"
    }, {
        "description" : "TC(1948) / UTM zone 39N",
        "code" : "EPSG:30339"
    }, {
        "description" : "TC(1948) / UTM zone 40N",
        "code" : "EPSG:30340"
    }, {
        "description" : "Voirol 1875 / Nord Algerie (ancienne)",
        "code" : "EPSG:30491"
    }, {
        "description" : "Voirol 1875 / Sud Algerie (ancienne)",
        "code" : "EPSG:30492"
    }, {
        "description" : "Voirol 1879 / Nord Algerie (ancienne)",
        "code" : "EPSG:30493"
    }, {
        "description" : "Voirol 1879 / Sud Algerie (ancienne)",
        "code" : "EPSG:30494"
    }, {
        "description" : "Nord Sahara 1959 / UTM zone 29N",
        "code" : "EPSG:30729"
    }, {
        "description" : "Nord Sahara 1959 / UTM zone 30N",
        "code" : "EPSG:30730"
    }, {
        "description" : "Nord Sahara 1959 / UTM zone 31N",
        "code" : "EPSG:30731"
    }, {
        "description" : "Nord Sahara 1959 / UTM zone 32N",
        "code" : "EPSG:30732"
    }, {
        "description" : "Nord Sahara 1959 / Voirol Unifie Nord",
        "code" : "EPSG:30791"
    }, {
        "description" : "Nord Sahara 1959 / Voirol Unifie Sud",
        "code" : "EPSG:30792"
    }, {
        "description" : "RT38 2.5 gon W",
        "code" : "EPSG:30800"
    }, {
        "description" : "Yoff / UTM zone 28N",
        "code" : "EPSG:31028"
    }, {
        "description" : "Zanderij / UTM zone 21N",
        "code" : "EPSG:31121"
    }, {
        "description" : "Zanderij / TM 54 NW",
        "code" : "EPSG:31154"
    }, {
        "description" : "Zanderij / Suriname Old TM",
        "code" : "EPSG:31170"
    }, {
        "description" : "Zanderij / Suriname TM",
        "code" : "EPSG:31171"
    }, {
        "description" : "MGI (Ferro) / Austria GK West Zone",
        "code" : "EPSG:31251"
    }, {
        "description" : "MGI (Ferro) / Austria GK Central Zone",
        "code" : "EPSG:31252"
    }, {
        "description" : "MGI (Ferro) / Austria GK East Zone",
        "code" : "EPSG:31253"
    }, {
        "description" : "MGI / Austria GK West",
        "code" : "EPSG:31254"
    }, {
        "description" : "MGI / Austria GK Central",
        "code" : "EPSG:31255"
    }, {
        "description" : "MGI / Austria GK East",
        "code" : "EPSG:31256"
    }, {
        "description" : "MGI / Austria GK M28",
        "code" : "EPSG:31257"
    }, {
        "description" : "MGI / Austria GK M31",
        "code" : "EPSG:31258"
    }, {
        "description" : "MGI / Austria GK M34",
        "code" : "EPSG:31259"
    }, {
        "description" : "MGI / 3-degree Gauss zone 5",
        "code" : "EPSG:31265"
    }, {
        "description" : "MGI / 3-degree Gauss zone 6",
        "code" : "EPSG:31266"
    }, {
        "description" : "MGI / 3-degree Gauss zone 7",
        "code" : "EPSG:31267"
    }, {
        "description" : "MGI / 3-degree Gauss zone 8",
        "code" : "EPSG:31268"
    }, {
        "description" : "MGI / Balkans zone 5",
        "code" : "EPSG:31275"
    }, {
        "description" : "MGI / Balkans zone 6",
        "code" : "EPSG:31276"
    }, {
        "description" : "MGI / Balkans zone 7",
        "code" : "EPSG:31277"
    }, {
        "description" : "MGI / Balkans zone 8",
        "code" : "EPSG:31278"
    }, {
        "description" : "MGI / Balkans zone 8",
        "code" : "EPSG:31279"
    }, {
        "description" : "MGI (Ferro) / Austria West Zone",
        "code" : "EPSG:31281"
    }, {
        "description" : "MGI (Ferro) / Austria Central Zone",
        "code" : "EPSG:31282"
    }, {
        "description" : "MGI (Ferro) / Austria East Zone",
        "code" : "EPSG:31283"
    }, {
        "description" : "MGI / Austria M28",
        "code" : "EPSG:31284"
    }, {
        "description" : "MGI / Austria M31",
        "code" : "EPSG:31285"
    }, {
        "description" : "MGI / Austria M34",
        "code" : "EPSG:31286"
    }, {
        "description" : "MGI / Austria Lambert",
        "code" : "EPSG:31287"
    }, {
        "description" : "MGI (Ferro) / M28",
        "code" : "EPSG:31288"
    }, {
        "description" : "MGI (Ferro) / M31",
        "code" : "EPSG:31289"
    }, {
        "description" : "MGI (Ferro) / M34",
        "code" : "EPSG:31290"
    }, {
        "description" : "MGI (Ferro) / Austria West Zone",
        "code" : "EPSG:31291"
    }, {
        "description" : "MGI (Ferro) / Austria Central Zone",
        "code" : "EPSG:31292"
    }, {
        "description" : "MGI (Ferro) / Austria East Zone",
        "code" : "EPSG:31293"
    }, {
        "description" : "MGI / M28",
        "code" : "EPSG:31294"
    }, {
        "description" : "MGI / M31",
        "code" : "EPSG:31295"
    }, {
        "description" : "MGI / M34",
        "code" : "EPSG:31296"
    }, {
        "description" : "MGI / Austria Lambert",
        "code" : "EPSG:31297"
    }, {
        "description" : "Belge 1972 / Belge Lambert 72",
        "code" : "EPSG:31300"
    }, {
        "description" : "Belge 1972 / Belgian Lambert 72",
        "code" : "EPSG:31370"
    }, {
        "description" : "DHDN / 3-degree Gauss zone 1",
        "code" : "EPSG:31461"
    }, {
        "description" : "DHDN / 3-degree Gauss zone 2",
        "code" : "EPSG:31462"
    }, {
        "description" : "DHDN / 3-degree Gauss zone 3",
        "code" : "EPSG:31463"
    }, {
        "description" : "DHDN / 3-degree Gauss zone 4",
        "code" : "EPSG:31464"
    }, {
        "description" : "DHDN / 3-degree Gauss zone 5",
        "code" : "EPSG:31465"
    }, {
        "description" : "DHDN / 3-degree Gauss-Kruger zone 2",
        "code" : "EPSG:31466"
    }, {
        "description" : "DHDN / 3-degree Gauss-Kruger zone 3",
        "code" : "EPSG:31467"
    }, {
        "description" : "DHDN / 3-degree Gauss-Kruger zone 4",
        "code" : "EPSG:31468"
    }, {
        "description" : "DHDN / 3-degree Gauss-Kruger zone 5",
        "code" : "EPSG:31469"
    }, {
        "description" : "Conakry 1905 / UTM zone 28N",
        "code" : "EPSG:31528"
    }, {
        "description" : "Conakry 1905 / UTM zone 29N",
        "code" : "EPSG:31529"
    }, {
        "description" : "Dealul Piscului 1930 / Stereo 33",
        "code" : "EPSG:31600"
    }, {
        "description" : "Dealul Piscului 1970/ Stereo 70",
        "code" : "EPSG:31700"
    }, {
        "description" : "NGN / UTM zone 38N",
        "code" : "EPSG:31838"
    }, {
        "description" : "NGN / UTM zone 39N",
        "code" : "EPSG:31839"
    }, {
        "description" : "KUDAMS / KTM",
        "code" : "EPSG:31900"
    }, {
        "description" : "KUDAMS / KTM",
        "code" : "EPSG:31901"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 11N",
        "code" : "EPSG:31965"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 12N",
        "code" : "EPSG:31966"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 13N",
        "code" : "EPSG:31967"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 14N",
        "code" : "EPSG:31968"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 15N",
        "code" : "EPSG:31969"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 16N",
        "code" : "EPSG:31970"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 17N",
        "code" : "EPSG:31971"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 18N",
        "code" : "EPSG:31972"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 19N",
        "code" : "EPSG:31973"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 20N",
        "code" : "EPSG:31974"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 21N",
        "code" : "EPSG:31975"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 22N",
        "code" : "EPSG:31976"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 17S",
        "code" : "EPSG:31977"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 18S",
        "code" : "EPSG:31978"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 19S",
        "code" : "EPSG:31979"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 20S",
        "code" : "EPSG:31980"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 21S",
        "code" : "EPSG:31981"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 22S",
        "code" : "EPSG:31982"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 23S",
        "code" : "EPSG:31983"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 24S",
        "code" : "EPSG:31984"
    }, {
        "description" : "SIRGAS 2000 / UTM zone 25S",
        "code" : "EPSG:31985"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 17N",
        "code" : "EPSG:31986"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 18N",
        "code" : "EPSG:31987"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 19N",
        "code" : "EPSG:31988"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 20N",
        "code" : "EPSG:31989"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 21N",
        "code" : "EPSG:31990"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 22N",
        "code" : "EPSG:31991"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 17S",
        "code" : "EPSG:31992"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 18S",
        "code" : "EPSG:31993"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 19S",
        "code" : "EPSG:31994"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 20S",
        "code" : "EPSG:31995"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 21S",
        "code" : "EPSG:31996"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 22S",
        "code" : "EPSG:31997"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 23S",
        "code" : "EPSG:31998"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 24S",
        "code" : "EPSG:31999"
    }, {
        "description" : "SIRGAS 1995 / UTM zone 25S",
        "code" : "EPSG:32000"
    }, {
        "description" : "NAD27 / Montana North",
        "code" : "EPSG:32001"
    }, {
        "description" : "NAD27 / Montana Central",
        "code" : "EPSG:32002"
    }, {
        "description" : "NAD27 / Montana South",
        "code" : "EPSG:32003"
    }, {
        "description" : "NAD27 / Nebraska North",
        "code" : "EPSG:32005"
    }, {
        "description" : "NAD27 / Nebraska South",
        "code" : "EPSG:32006"
    }, {
        "description" : "NAD27 / Nevada East",
        "code" : "EPSG:32007"
    }, {
        "description" : "NAD27 / Nevada Central",
        "code" : "EPSG:32008"
    }, {
        "description" : "NAD27 / Nevada West",
        "code" : "EPSG:32009"
    }, {
        "description" : "NAD27 / New Hampshire",
        "code" : "EPSG:32010"
    }, {
        "description" : "NAD27 / New Jersey",
        "code" : "EPSG:32011"
    }, {
        "description" : "NAD27 / New Mexico East",
        "code" : "EPSG:32012"
    }, {
        "description" : "NAD27 / New Mexico Central",
        "code" : "EPSG:32013"
    }, {
        "description" : "NAD27 / New Mexico West",
        "code" : "EPSG:32014"
    }, {
        "description" : "NAD27 / New York East",
        "code" : "EPSG:32015"
    }, {
        "description" : "NAD27 / New York Central",
        "code" : "EPSG:32016"
    }, {
        "description" : "NAD27 / New York West",
        "code" : "EPSG:32017"
    }, {
        "description" : "NAD27 / New York Long Island",
        "code" : "EPSG:32018"
    }, {
        "description" : "NAD27 / North Carolina",
        "code" : "EPSG:32019"
    }, {
        "description" : "NAD27 / North Dakota North",
        "code" : "EPSG:32020"
    }, {
        "description" : "NAD27 / North Dakota South",
        "code" : "EPSG:32021"
    }, {
        "description" : "NAD27 / Ohio North",
        "code" : "EPSG:32022"
    }, {
        "description" : "NAD27 / Ohio South",
        "code" : "EPSG:32023"
    }, {
        "description" : "NAD27 / Oklahoma North",
        "code" : "EPSG:32024"
    }, {
        "description" : "NAD27 / Oklahoma South",
        "code" : "EPSG:32025"
    }, {
        "description" : "NAD27 / Oregon North",
        "code" : "EPSG:32026"
    }, {
        "description" : "NAD27 / Oregon South",
        "code" : "EPSG:32027"
    }, {
        "description" : "NAD27 / Pennsylvania North",
        "code" : "EPSG:32028"
    }, {
        "description" : "NAD27 / Pennsylvania South",
        "code" : "EPSG:32029"
    }, {
        "description" : "NAD27 / Rhode Island",
        "code" : "EPSG:32030"
    }, {
        "description" : "NAD27 / South Carolina North",
        "code" : "EPSG:32031"
    }, {
        "description" : "NAD27 / South Carolina South",
        "code" : "EPSG:32033"
    }, {
        "description" : "NAD27 / South Dakota North",
        "code" : "EPSG:32034"
    }, {
        "description" : "NAD27 / South Dakota South",
        "code" : "EPSG:32035"
    }, {
        "description" : "NAD27 / Tennessee",
        "code" : "EPSG:32036"
    }, {
        "description" : "NAD27 / Texas North",
        "code" : "EPSG:32037"
    }, {
        "description" : "NAD27 / Texas North Central",
        "code" : "EPSG:32038"
    }, {
        "description" : "NAD27 / Texas Central",
        "code" : "EPSG:32039"
    }, {
        "description" : "NAD27 / Texas South Central",
        "code" : "EPSG:32040"
    }, {
        "description" : "NAD27 / Texas South",
        "code" : "EPSG:32041"
    }, {
        "description" : "NAD27 / Utah North",
        "code" : "EPSG:32042"
    }, {
        "description" : "NAD27 / Utah Central",
        "code" : "EPSG:32043"
    }, {
        "description" : "NAD27 / Utah South",
        "code" : "EPSG:32044"
    }, {
        "description" : "NAD27 / Vermont",
        "code" : "EPSG:32045"
    }, {
        "description" : "NAD27 / Virginia North",
        "code" : "EPSG:32046"
    }, {
        "description" : "NAD27 / Virginia South",
        "code" : "EPSG:32047"
    }, {
        "description" : "NAD27 / Washington North",
        "code" : "EPSG:32048"
    }, {
        "description" : "NAD27 / Washington South",
        "code" : "EPSG:32049"
    }, {
        "description" : "NAD27 / West Virginia North",
        "code" : "EPSG:32050"
    }, {
        "description" : "NAD27 / West Virginia South",
        "code" : "EPSG:32051"
    }, {
        "description" : "NAD27 / Wisconsin North",
        "code" : "EPSG:32052"
    }, {
        "description" : "NAD27 / Wisconsin Central",
        "code" : "EPSG:32053"
    }, {
        "description" : "NAD27 / Wisconsin South",
        "code" : "EPSG:32054"
    }, {
        "description" : "NAD27 / Wyoming East",
        "code" : "EPSG:32055"
    }, {
        "description" : "NAD27 / Wyoming East Central",
        "code" : "EPSG:32056"
    }, {
        "description" : "NAD27 / Wyoming West Central",
        "code" : "EPSG:32057"
    }, {
        "description" : "NAD27 / Wyoming West",
        "code" : "EPSG:32058"
    }, {
        "description" : "NAD27 / Guatemala Norte",
        "code" : "EPSG:32061"
    }, {
        "description" : "NAD27 / Guatemala Sur",
        "code" : "EPSG:32062"
    }, {
        "description" : "NAD27 / BLM 14N (ftUS)",
        "code" : "EPSG:32064"
    }, {
        "description" : "NAD27 / BLM 15N (ftUS)",
        "code" : "EPSG:32065"
    }, {
        "description" : "NAD27 / BLM 16N (ftUS)",
        "code" : "EPSG:32066"
    }, {
        "description" : "NAD27 / BLM 17N (ftUS)",
        "code" : "EPSG:32067"
    }, {
        "description" : "NAD27 / BLM 14N (feet)",
        "code" : "EPSG:32074"
    }, {
        "description" : "NAD27 / BLM 15N (feet)",
        "code" : "EPSG:32075"
    }, {
        "description" : "NAD27 / BLM 16N (feet)",
        "code" : "EPSG:32076"
    }, {
        "description" : "NAD27 / BLM 17N (feet)",
        "code" : "EPSG:32077"
    }, {
        "description" : "NAD27 / MTM zone 1",
        "code" : "EPSG:32081"
    }, {
        "description" : "NAD27 / MTM zone 2",
        "code" : "EPSG:32082"
    }, {
        "description" : "NAD27 / MTM zone 3",
        "code" : "EPSG:32083"
    }, {
        "description" : "NAD27 / MTM zone 4",
        "code" : "EPSG:32084"
    }, {
        "description" : "NAD27 / MTM zone 5",
        "code" : "EPSG:32085"
    }, {
        "description" : "NAD27 / MTM zone 6",
        "code" : "EPSG:32086"
    }, {
        "description" : "NAD27 / Quebec Lambert",
        "code" : "EPSG:32098"
    }, {
        "description" : "NAD27 / Louisiana Offshore",
        "code" : "EPSG:32099"
    }, {
        "description" : "NAD83 / Montana",
        "code" : "EPSG:32100"
    }, {
        "description" : "NAD83 / Nebraska",
        "code" : "EPSG:32104"
    }, {
        "description" : "NAD83 / Nevada East",
        "code" : "EPSG:32107"
    }, {
        "description" : "NAD83 / Nevada Central",
        "code" : "EPSG:32108"
    }, {
        "description" : "NAD83 / Nevada West",
        "code" : "EPSG:32109"
    }, {
        "description" : "NAD83 / New Hampshire",
        "code" : "EPSG:32110"
    }, {
        "description" : "NAD83 / New Jersey",
        "code" : "EPSG:32111"
    }, {
        "description" : "NAD83 / New Mexico East",
        "code" : "EPSG:32112"
    }, {
        "description" : "NAD83 / New Mexico Central",
        "code" : "EPSG:32113"
    }, {
        "description" : "NAD83 / New Mexico West",
        "code" : "EPSG:32114"
    }, {
        "description" : "NAD83 / New York East",
        "code" : "EPSG:32115"
    }, {
        "description" : "NAD83 / New York Central",
        "code" : "EPSG:32116"
    }, {
        "description" : "NAD83 / New York West",
        "code" : "EPSG:32117"
    }, {
        "description" : "NAD83 / New York Long Island",
        "code" : "EPSG:32118"
    }, {
        "description" : "NAD83 / North Carolina",
        "code" : "EPSG:32119"
    }, {
        "description" : "NAD83 / North Dakota North",
        "code" : "EPSG:32120"
    }, {
        "description" : "NAD83 / North Dakota South",
        "code" : "EPSG:32121"
    }, {
        "description" : "NAD83 / Ohio North",
        "code" : "EPSG:32122"
    }, {
        "description" : "NAD83 / Ohio South",
        "code" : "EPSG:32123"
    }, {
        "description" : "NAD83 / Oklahoma North",
        "code" : "EPSG:32124"
    }, {
        "description" : "NAD83 / Oklahoma South",
        "code" : "EPSG:32125"
    }, {
        "description" : "NAD83 / Oregon North",
        "code" : "EPSG:32126"
    }, {
        "description" : "NAD83 / Oregon South",
        "code" : "EPSG:32127"
    }, {
        "description" : "NAD83 / Pennsylvania North",
        "code" : "EPSG:32128"
    }, {
        "description" : "NAD83 / Pennsylvania South",
        "code" : "EPSG:32129"
    }, {
        "description" : "NAD83 / Rhode Island",
        "code" : "EPSG:32130"
    }, {
        "description" : "NAD83 / South Carolina",
        "code" : "EPSG:32133"
    }, {
        "description" : "NAD83 / South Dakota North",
        "code" : "EPSG:32134"
    }, {
        "description" : "NAD83 / South Dakota South",
        "code" : "EPSG:32135"
    }, {
        "description" : "NAD83 / Tennessee",
        "code" : "EPSG:32136"
    }, {
        "description" : "NAD83 / Texas North",
        "code" : "EPSG:32137"
    }, {
        "description" : "NAD83 / Texas North Central",
        "code" : "EPSG:32138"
    }, {
        "description" : "NAD83 / Texas Central",
        "code" : "EPSG:32139"
    }, {
        "description" : "NAD83 / Texas South Central",
        "code" : "EPSG:32140"
    }, {
        "description" : "NAD83 / Texas South",
        "code" : "EPSG:32141"
    }, {
        "description" : "NAD83 / Utah North",
        "code" : "EPSG:32142"
    }, {
        "description" : "NAD83 / Utah Central",
        "code" : "EPSG:32143"
    }, {
        "description" : "NAD83 / Utah South",
        "code" : "EPSG:32144"
    }, {
        "description" : "NAD83 / Vermont",
        "code" : "EPSG:32145"
    }, {
        "description" : "NAD83 / Virginia North",
        "code" : "EPSG:32146"
    }, {
        "description" : "NAD83 / Virginia South",
        "code" : "EPSG:32147"
    }, {
        "description" : "NAD83 / Washington North",
        "code" : "EPSG:32148"
    }, {
        "description" : "NAD83 / Washington South",
        "code" : "EPSG:32149"
    }, {
        "description" : "NAD83 / West Virginia North",
        "code" : "EPSG:32150"
    }, {
        "description" : "NAD83 / West Virginia South",
        "code" : "EPSG:32151"
    }, {
        "description" : "NAD83 / Wisconsin North",
        "code" : "EPSG:32152"
    }, {
        "description" : "NAD83 / Wisconsin Central",
        "code" : "EPSG:32153"
    }, {
        "description" : "NAD83 / Wisconsin South",
        "code" : "EPSG:32154"
    }, {
        "description" : "NAD83 / Wyoming East",
        "code" : "EPSG:32155"
    }, {
        "description" : "NAD83 / Wyoming East Central",
        "code" : "EPSG:32156"
    }, {
        "description" : "NAD83 / Wyoming West Central",
        "code" : "EPSG:32157"
    }, {
        "description" : "NAD83 / Wyoming West",
        "code" : "EPSG:32158"
    }, {
        "description" : "NAD83 / Puerto Rico \u0026 Virgin Is.",
        "code" : "EPSG:32161"
    }, {
        "description" : "NAD83 / BLM 14N (ftUS)",
        "code" : "EPSG:32164"
    }, {
        "description" : "NAD83 / BLM 15N (ftUS)",
        "code" : "EPSG:32165"
    }, {
        "description" : "NAD83 / BLM 16N (ftUS)",
        "code" : "EPSG:32166"
    }, {
        "description" : "NAD83 / BLM 17N (ftUS)",
        "code" : "EPSG:32167"
    }, {
        "description" : "NAD83 / SCoPQ zone 2",
        "code" : "EPSG:32180"
    }, {
        "description" : "NAD83 / MTM zone 1",
        "code" : "EPSG:32181"
    }, {
        "description" : "NAD83 / MTM zone 2",
        "code" : "EPSG:32182"
    }, {
        "description" : "NAD83 / MTM zone 3",
        "code" : "EPSG:32183"
    }, {
        "description" : "NAD83 / MTM zone 4",
        "code" : "EPSG:32184"
    }, {
        "description" : "NAD83 / MTM zone 5",
        "code" : "EPSG:32185"
    }, {
        "description" : "NAD83 / MTM zone 6",
        "code" : "EPSG:32186"
    }, {
        "description" : "NAD83 / MTM zone 7",
        "code" : "EPSG:32187"
    }, {
        "description" : "NAD83 / MTM zone 8",
        "code" : "EPSG:32188"
    }, {
        "description" : "NAD83 / MTM zone 9",
        "code" : "EPSG:32189"
    }, {
        "description" : "NAD83 / MTM zone 10",
        "code" : "EPSG:32190"
    }, {
        "description" : "NAD83 / MTM zone 11",
        "code" : "EPSG:32191"
    }, {
        "description" : "NAD83 / MTM zone 12",
        "code" : "EPSG:32192"
    }, {
        "description" : "NAD83 / MTM zone 13",
        "code" : "EPSG:32193"
    }, {
        "description" : "NAD83 / MTM zone 14",
        "code" : "EPSG:32194"
    }, {
        "description" : "NAD83 / MTM zone 15",
        "code" : "EPSG:32195"
    }, {
        "description" : "NAD83 / MTM zone 16",
        "code" : "EPSG:32196"
    }, {
        "description" : "NAD83 / MTM zone 17",
        "code" : "EPSG:32197"
    }, {
        "description" : "NAD83 / Quebec Lambert",
        "code" : "EPSG:32198"
    }, {
        "description" : "NAD83 / Louisiana Offshore",
        "code" : "EPSG:32199"
    }, {
        "description" : "WGS 72 / UTM zone 1N",
        "code" : "EPSG:32201"
    }, {
        "description" : "WGS 72 / UTM zone 2N",
        "code" : "EPSG:32202"
    }, {
        "description" : "WGS 72 / UTM zone 3N",
        "code" : "EPSG:32203"
    }, {
        "description" : "WGS 72 / UTM zone 4N",
        "code" : "EPSG:32204"
    }, {
        "description" : "WGS 72 / UTM zone 5N",
        "code" : "EPSG:32205"
    }, {
        "description" : "WGS 72 / UTM zone 6N",
        "code" : "EPSG:32206"
    }, {
        "description" : "WGS 72 / UTM zone 7N",
        "code" : "EPSG:32207"
    }, {
        "description" : "WGS 72 / UTM zone 8N",
        "code" : "EPSG:32208"
    }, {
        "description" : "WGS 72 / UTM zone 9N",
        "code" : "EPSG:32209"
    }, {
        "description" : "WGS 72 / UTM zone 10N",
        "code" : "EPSG:32210"
    }, {
        "description" : "WGS 72 / UTM zone 11N",
        "code" : "EPSG:32211"
    }, {
        "description" : "WGS 72 / UTM zone 12N",
        "code" : "EPSG:32212"
    }, {
        "description" : "WGS 72 / UTM zone 13N",
        "code" : "EPSG:32213"
    }, {
        "description" : "WGS 72 / UTM zone 14N",
        "code" : "EPSG:32214"
    }, {
        "description" : "WGS 72 / UTM zone 15N",
        "code" : "EPSG:32215"
    }, {
        "description" : "WGS 72 / UTM zone 16N",
        "code" : "EPSG:32216"
    }, {
        "description" : "WGS 72 / UTM zone 17N",
        "code" : "EPSG:32217"
    }, {
        "description" : "WGS 72 / UTM zone 18N",
        "code" : "EPSG:32218"
    }, {
        "description" : "WGS 72 / UTM zone 19N",
        "code" : "EPSG:32219"
    }, {
        "description" : "WGS 72 / UTM zone 20N",
        "code" : "EPSG:32220"
    }, {
        "description" : "WGS 72 / UTM zone 21N",
        "code" : "EPSG:32221"
    }, {
        "description" : "WGS 72 / UTM zone 22N",
        "code" : "EPSG:32222"
    }, {
        "description" : "WGS 72 / UTM zone 23N",
        "code" : "EPSG:32223"
    }, {
        "description" : "WGS 72 / UTM zone 24N",
        "code" : "EPSG:32224"
    }, {
        "description" : "WGS 72 / UTM zone 25N",
        "code" : "EPSG:32225"
    }, {
        "description" : "WGS 72 / UTM zone 26N",
        "code" : "EPSG:32226"
    }, {
        "description" : "WGS 72 / UTM zone 27N",
        "code" : "EPSG:32227"
    }, {
        "description" : "WGS 72 / UTM zone 28N",
        "code" : "EPSG:32228"
    }, {
        "description" : "WGS 72 / UTM zone 29N",
        "code" : "EPSG:32229"
    }, {
        "description" : "WGS 72 / UTM zone 30N",
        "code" : "EPSG:32230"
    }, {
        "description" : "WGS 72 / UTM zone 31N",
        "code" : "EPSG:32231"
    }, {
        "description" : "WGS 72 / UTM zone 32N",
        "code" : "EPSG:32232"
    }, {
        "description" : "WGS 72 / UTM zone 33N",
        "code" : "EPSG:32233"
    }, {
        "description" : "WGS 72 / UTM zone 34N",
        "code" : "EPSG:32234"
    }, {
        "description" : "WGS 72 / UTM zone 35N",
        "code" : "EPSG:32235"
    }, {
        "description" : "WGS 72 / UTM zone 36N",
        "code" : "EPSG:32236"
    }, {
        "description" : "WGS 72 / UTM zone 37N",
        "code" : "EPSG:32237"
    }, {
        "description" : "WGS 72 / UTM zone 38N",
        "code" : "EPSG:32238"
    }, {
        "description" : "WGS 72 / UTM zone 39N",
        "code" : "EPSG:32239"
    }, {
        "description" : "WGS 72 / UTM zone 40N",
        "code" : "EPSG:32240"
    }, {
        "description" : "WGS 72 / UTM zone 41N",
        "code" : "EPSG:32241"
    }, {
        "description" : "WGS 72 / UTM zone 42N",
        "code" : "EPSG:32242"
    }, {
        "description" : "WGS 72 / UTM zone 43N",
        "code" : "EPSG:32243"
    }, {
        "description" : "WGS 72 / UTM zone 44N",
        "code" : "EPSG:32244"
    }, {
        "description" : "WGS 72 / UTM zone 45N",
        "code" : "EPSG:32245"
    }, {
        "description" : "WGS 72 / UTM zone 46N",
        "code" : "EPSG:32246"
    }, {
        "description" : "WGS 72 / UTM zone 47N",
        "code" : "EPSG:32247"
    }, {
        "description" : "WGS 72 / UTM zone 48N",
        "code" : "EPSG:32248"
    }, {
        "description" : "WGS 72 / UTM zone 49N",
        "code" : "EPSG:32249"
    }, {
        "description" : "WGS 72 / UTM zone 50N",
        "code" : "EPSG:32250"
    }, {
        "description" : "WGS 72 / UTM zone 51N",
        "code" : "EPSG:32251"
    }, {
        "description" : "WGS 72 / UTM zone 52N",
        "code" : "EPSG:32252"
    }, {
        "description" : "WGS 72 / UTM zone 53N",
        "code" : "EPSG:32253"
    }, {
        "description" : "WGS 72 / UTM zone 54N",
        "code" : "EPSG:32254"
    }, {
        "description" : "WGS 72 / UTM zone 55N",
        "code" : "EPSG:32255"
    }, {
        "description" : "WGS 72 / UTM zone 56N",
        "code" : "EPSG:32256"
    }, {
        "description" : "WGS 72 / UTM zone 57N",
        "code" : "EPSG:32257"
    }, {
        "description" : "WGS 72 / UTM zone 58N",
        "code" : "EPSG:32258"
    }, {
        "description" : "WGS 72 / UTM zone 59N",
        "code" : "EPSG:32259"
    }, {
        "description" : "WGS 72 / UTM zone 60N",
        "code" : "EPSG:32260"
    }, {
        "description" : "WGS 72 / UTM zone 1S",
        "code" : "EPSG:32301"
    }, {
        "description" : "WGS 72 / UTM zone 2S",
        "code" : "EPSG:32302"
    }, {
        "description" : "WGS 72 / UTM zone 3S",
        "code" : "EPSG:32303"
    }, {
        "description" : "WGS 72 / UTM zone 4S",
        "code" : "EPSG:32304"
    }, {
        "description" : "WGS 72 / UTM zone 5S",
        "code" : "EPSG:32305"
    }, {
        "description" : "WGS 72 / UTM zone 6S",
        "code" : "EPSG:32306"
    }, {
        "description" : "WGS 72 / UTM zone 7S",
        "code" : "EPSG:32307"
    }, {
        "description" : "WGS 72 / UTM zone 8S",
        "code" : "EPSG:32308"
    }, {
        "description" : "WGS 72 / UTM zone 9S",
        "code" : "EPSG:32309"
    }, {
        "description" : "WGS 72 / UTM zone 10S",
        "code" : "EPSG:32310"
    }, {
        "description" : "WGS 72 / UTM zone 11S",
        "code" : "EPSG:32311"
    }, {
        "description" : "WGS 72 / UTM zone 12S",
        "code" : "EPSG:32312"
    }, {
        "description" : "WGS 72 / UTM zone 13S",
        "code" : "EPSG:32313"
    }, {
        "description" : "WGS 72 / UTM zone 14S",
        "code" : "EPSG:32314"
    }, {
        "description" : "WGS 72 / UTM zone 15S",
        "code" : "EPSG:32315"
    }, {
        "description" : "WGS 72 / UTM zone 16S",
        "code" : "EPSG:32316"
    }, {
        "description" : "WGS 72 / UTM zone 17S",
        "code" : "EPSG:32317"
    }, {
        "description" : "WGS 72 / UTM zone 18S",
        "code" : "EPSG:32318"
    }, {
        "description" : "WGS 72 / UTM zone 19S",
        "code" : "EPSG:32319"
    }, {
        "description" : "WGS 72 / UTM zone 20S",
        "code" : "EPSG:32320"
    }, {
        "description" : "WGS 72 / UTM zone 21S",
        "code" : "EPSG:32321"
    }, {
        "description" : "WGS 72 / UTM zone 22S",
        "code" : "EPSG:32322"
    }, {
        "description" : "WGS 72 / UTM zone 23S",
        "code" : "EPSG:32323"
    }, {
        "description" : "WGS 72 / UTM zone 24S",
        "code" : "EPSG:32324"
    }, {
        "description" : "WGS 72 / UTM zone 25S",
        "code" : "EPSG:32325"
    }, {
        "description" : "WGS 72 / UTM zone 26S",
        "code" : "EPSG:32326"
    }, {
        "description" : "WGS 72 / UTM zone 27S",
        "code" : "EPSG:32327"
    }, {
        "description" : "WGS 72 / UTM zone 28S",
        "code" : "EPSG:32328"
    }, {
        "description" : "WGS 72 / UTM zone 29S",
        "code" : "EPSG:32329"
    }, {
        "description" : "WGS 72 / UTM zone 30S",
        "code" : "EPSG:32330"
    }, {
        "description" : "WGS 72 / UTM zone 31S",
        "code" : "EPSG:32331"
    }, {
        "description" : "WGS 72 / UTM zone 32S",
        "code" : "EPSG:32332"
    }, {
        "description" : "WGS 72 / UTM zone 33S",
        "code" : "EPSG:32333"
    }, {
        "description" : "WGS 72 / UTM zone 34S",
        "code" : "EPSG:32334"
    }, {
        "description" : "WGS 72 / UTM zone 35S",
        "code" : "EPSG:32335"
    }, {
        "description" : "WGS 72 / UTM zone 36S",
        "code" : "EPSG:32336"
    }, {
        "description" : "WGS 72 / UTM zone 37S",
        "code" : "EPSG:32337"
    }, {
        "description" : "WGS 72 / UTM zone 38S",
        "code" : "EPSG:32338"
    }, {
        "description" : "WGS 72 / UTM zone 39S",
        "code" : "EPSG:32339"
    }, {
        "description" : "WGS 72 / UTM zone 40S",
        "code" : "EPSG:32340"
    }, {
        "description" : "WGS 72 / UTM zone 41S",
        "code" : "EPSG:32341"
    }, {
        "description" : "WGS 72 / UTM zone 42S",
        "code" : "EPSG:32342"
    }, {
        "description" : "WGS 72 / UTM zone 43S",
        "code" : "EPSG:32343"
    }, {
        "description" : "WGS 72 / UTM zone 44S",
        "code" : "EPSG:32344"
    }, {
        "description" : "WGS 72 / UTM zone 45S",
        "code" : "EPSG:32345"
    }, {
        "description" : "WGS 72 / UTM zone 46S",
        "code" : "EPSG:32346"
    }, {
        "description" : "WGS 72 / UTM zone 47S",
        "code" : "EPSG:32347"
    }, {
        "description" : "WGS 72 / UTM zone 48S",
        "code" : "EPSG:32348"
    }, {
        "description" : "WGS 72 / UTM zone 49S",
        "code" : "EPSG:32349"
    }, {
        "description" : "WGS 72 / UTM zone 50S",
        "code" : "EPSG:32350"
    }, {
        "description" : "WGS 72 / UTM zone 51S",
        "code" : "EPSG:32351"
    }, {
        "description" : "WGS 72 / UTM zone 52S",
        "code" : "EPSG:32352"
    }, {
        "description" : "WGS 72 / UTM zone 53S",
        "code" : "EPSG:32353"
    }, {
        "description" : "WGS 72 / UTM zone 54S",
        "code" : "EPSG:32354"
    }, {
        "description" : "WGS 72 / UTM zone 55S",
        "code" : "EPSG:32355"
    }, {
        "description" : "WGS 72 / UTM zone 56S",
        "code" : "EPSG:32356"
    }, {
        "description" : "WGS 72 / UTM zone 57S",
        "code" : "EPSG:32357"
    }, {
        "description" : "WGS 72 / UTM zone 58S",
        "code" : "EPSG:32358"
    }, {
        "description" : "WGS 72 / UTM zone 59S",
        "code" : "EPSG:32359"
    }, {
        "description" : "WGS 72 / UTM zone 60S",
        "code" : "EPSG:32360"
    }, {
        "description" : "WGS 72BE / UTM zone 1N",
        "code" : "EPSG:32401"
    }, {
        "description" : "WGS 72BE / UTM zone 2N",
        "code" : "EPSG:32402"
    }, {
        "description" : "WGS 72BE / UTM zone 3N",
        "code" : "EPSG:32403"
    }, {
        "description" : "WGS 72BE / UTM zone 4N",
        "code" : "EPSG:32404"
    }, {
        "description" : "WGS 72BE / UTM zone 5N",
        "code" : "EPSG:32405"
    }, {
        "description" : "WGS 72BE / UTM zone 6N",
        "code" : "EPSG:32406"
    }, {
        "description" : "WGS 72BE / UTM zone 7N",
        "code" : "EPSG:32407"
    }, {
        "description" : "WGS 72BE / UTM zone 8N",
        "code" : "EPSG:32408"
    }, {
        "description" : "WGS 72BE / UTM zone 9N",
        "code" : "EPSG:32409"
    }, {
        "description" : "WGS 72BE / UTM zone 10N",
        "code" : "EPSG:32410"
    }, {
        "description" : "WGS 72BE / UTM zone 11N",
        "code" : "EPSG:32411"
    }, {
        "description" : "WGS 72BE / UTM zone 12N",
        "code" : "EPSG:32412"
    }, {
        "description" : "WGS 72BE / UTM zone 13N",
        "code" : "EPSG:32413"
    }, {
        "description" : "WGS 72BE / UTM zone 14N",
        "code" : "EPSG:32414"
    }, {
        "description" : "WGS 72BE / UTM zone 15N",
        "code" : "EPSG:32415"
    }, {
        "description" : "WGS 72BE / UTM zone 16N",
        "code" : "EPSG:32416"
    }, {
        "description" : "WGS 72BE / UTM zone 17N",
        "code" : "EPSG:32417"
    }, {
        "description" : "WGS 72BE / UTM zone 18N",
        "code" : "EPSG:32418"
    }, {
        "description" : "WGS 72BE / UTM zone 19N",
        "code" : "EPSG:32419"
    }, {
        "description" : "WGS 72BE / UTM zone 20N",
        "code" : "EPSG:32420"
    }, {
        "description" : "WGS 72BE / UTM zone 21N",
        "code" : "EPSG:32421"
    }, {
        "description" : "WGS 72BE / UTM zone 22N",
        "code" : "EPSG:32422"
    }, {
        "description" : "WGS 72BE / UTM zone 23N",
        "code" : "EPSG:32423"
    }, {
        "description" : "WGS 72BE / UTM zone 24N",
        "code" : "EPSG:32424"
    }, {
        "description" : "WGS 72BE / UTM zone 25N",
        "code" : "EPSG:32425"
    }, {
        "description" : "WGS 72BE / UTM zone 26N",
        "code" : "EPSG:32426"
    }, {
        "description" : "WGS 72BE / UTM zone 27N",
        "code" : "EPSG:32427"
    }, {
        "description" : "WGS 72BE / UTM zone 28N",
        "code" : "EPSG:32428"
    }, {
        "description" : "WGS 72BE / UTM zone 29N",
        "code" : "EPSG:32429"
    }, {
        "description" : "WGS 72BE / UTM zone 30N",
        "code" : "EPSG:32430"
    }, {
        "description" : "WGS 72BE / UTM zone 31N",
        "code" : "EPSG:32431"
    }, {
        "description" : "WGS 72BE / UTM zone 32N",
        "code" : "EPSG:32432"
    }, {
        "description" : "WGS 72BE / UTM zone 33N",
        "code" : "EPSG:32433"
    }, {
        "description" : "WGS 72BE / UTM zone 34N",
        "code" : "EPSG:32434"
    }, {
        "description" : "WGS 72BE / UTM zone 35N",
        "code" : "EPSG:32435"
    }, {
        "description" : "WGS 72BE / UTM zone 36N",
        "code" : "EPSG:32436"
    }, {
        "description" : "WGS 72BE / UTM zone 37N",
        "code" : "EPSG:32437"
    }, {
        "description" : "WGS 72BE / UTM zone 38N",
        "code" : "EPSG:32438"
    }, {
        "description" : "WGS 72BE / UTM zone 39N",
        "code" : "EPSG:32439"
    }, {
        "description" : "WGS 72BE / UTM zone 40N",
        "code" : "EPSG:32440"
    }, {
        "description" : "WGS 72BE / UTM zone 41N",
        "code" : "EPSG:32441"
    }, {
        "description" : "WGS 72BE / UTM zone 42N",
        "code" : "EPSG:32442"
    }, {
        "description" : "WGS 72BE / UTM zone 43N",
        "code" : "EPSG:32443"
    }, {
        "description" : "WGS 72BE / UTM zone 44N",
        "code" : "EPSG:32444"
    }, {
        "description" : "WGS 72BE / UTM zone 45N",
        "code" : "EPSG:32445"
    }, {
        "description" : "WGS 72BE / UTM zone 46N",
        "code" : "EPSG:32446"
    }, {
        "description" : "WGS 72BE / UTM zone 47N",
        "code" : "EPSG:32447"
    }, {
        "description" : "WGS 72BE / UTM zone 48N",
        "code" : "EPSG:32448"
    }, {
        "description" : "WGS 72BE / UTM zone 49N",
        "code" : "EPSG:32449"
    }, {
        "description" : "WGS 72BE / UTM zone 50N",
        "code" : "EPSG:32450"
    }, {
        "description" : "WGS 72BE / UTM zone 51N",
        "code" : "EPSG:32451"
    }, {
        "description" : "WGS 72BE / UTM zone 52N",
        "code" : "EPSG:32452"
    }, {
        "description" : "WGS 72BE / UTM zone 53N",
        "code" : "EPSG:32453"
    }, {
        "description" : "WGS 72BE / UTM zone 54N",
        "code" : "EPSG:32454"
    }, {
        "description" : "WGS 72BE / UTM zone 55N",
        "code" : "EPSG:32455"
    }, {
        "description" : "WGS 72BE / UTM zone 56N",
        "code" : "EPSG:32456"
    }, {
        "description" : "WGS 72BE / UTM zone 57N",
        "code" : "EPSG:32457"
    }, {
        "description" : "WGS 72BE / UTM zone 58N",
        "code" : "EPSG:32458"
    }, {
        "description" : "WGS 72BE / UTM zone 59N",
        "code" : "EPSG:32459"
    }, {
        "description" : "WGS 72BE / UTM zone 60N",
        "code" : "EPSG:32460"
    }, {
        "description" : "WGS 72BE / UTM zone 1S",
        "code" : "EPSG:32501"
    }, {
        "description" : "WGS 72BE / UTM zone 2S",
        "code" : "EPSG:32502"
    }, {
        "description" : "WGS 72BE / UTM zone 3S",
        "code" : "EPSG:32503"
    }, {
        "description" : "WGS 72BE / UTM zone 4S",
        "code" : "EPSG:32504"
    }, {
        "description" : "WGS 72BE / UTM zone 5S",
        "code" : "EPSG:32505"
    }, {
        "description" : "WGS 72BE / UTM zone 6S",
        "code" : "EPSG:32506"
    }, {
        "description" : "WGS 72BE / UTM zone 7S",
        "code" : "EPSG:32507"
    }, {
        "description" : "WGS 72BE / UTM zone 8S",
        "code" : "EPSG:32508"
    }, {
        "description" : "WGS 72BE / UTM zone 9S",
        "code" : "EPSG:32509"
    }, {
        "description" : "WGS 72BE / UTM zone 10S",
        "code" : "EPSG:32510"
    }, {
        "description" : "WGS 72BE / UTM zone 11S",
        "code" : "EPSG:32511"
    }, {
        "description" : "WGS 72BE / UTM zone 12S",
        "code" : "EPSG:32512"
    }, {
        "description" : "WGS 72BE / UTM zone 13S",
        "code" : "EPSG:32513"
    }, {
        "description" : "WGS 72BE / UTM zone 14S",
        "code" : "EPSG:32514"
    }, {
        "description" : "WGS 72BE / UTM zone 15S",
        "code" : "EPSG:32515"
    }, {
        "description" : "WGS 72BE / UTM zone 16S",
        "code" : "EPSG:32516"
    }, {
        "description" : "WGS 72BE / UTM zone 17S",
        "code" : "EPSG:32517"
    }, {
        "description" : "WGS 72BE / UTM zone 18S",
        "code" : "EPSG:32518"
    }, {
        "description" : "WGS 72BE / UTM zone 19S",
        "code" : "EPSG:32519"
    }, {
        "description" : "WGS 72BE / UTM zone 20S",
        "code" : "EPSG:32520"
    }, {
        "description" : "WGS 72BE / UTM zone 21S",
        "code" : "EPSG:32521"
    }, {
        "description" : "WGS 72BE / UTM zone 22S",
        "code" : "EPSG:32522"
    }, {
        "description" : "WGS 72BE / UTM zone 23S",
        "code" : "EPSG:32523"
    }, {
        "description" : "WGS 72BE / UTM zone 24S",
        "code" : "EPSG:32524"
    }, {
        "description" : "WGS 72BE / UTM zone 25S",
        "code" : "EPSG:32525"
    }, {
        "description" : "WGS 72BE / UTM zone 26S",
        "code" : "EPSG:32526"
    }, {
        "description" : "WGS 72BE / UTM zone 27S",
        "code" : "EPSG:32527"
    }, {
        "description" : "WGS 72BE / UTM zone 28S",
        "code" : "EPSG:32528"
    }, {
        "description" : "WGS 72BE / UTM zone 29S",
        "code" : "EPSG:32529"
    }, {
        "description" : "WGS 72BE / UTM zone 30S",
        "code" : "EPSG:32530"
    }, {
        "description" : "WGS 72BE / UTM zone 31S",
        "code" : "EPSG:32531"
    }, {
        "description" : "WGS 72BE / UTM zone 32S",
        "code" : "EPSG:32532"
    }, {
        "description" : "WGS 72BE / UTM zone 33S",
        "code" : "EPSG:32533"
    }, {
        "description" : "WGS 72BE / UTM zone 34S",
        "code" : "EPSG:32534"
    }, {
        "description" : "WGS 72BE / UTM zone 35S",
        "code" : "EPSG:32535"
    }, {
        "description" : "WGS 72BE / UTM zone 36S",
        "code" : "EPSG:32536"
    }, {
        "description" : "WGS 72BE / UTM zone 37S",
        "code" : "EPSG:32537"
    }, {
        "description" : "WGS 72BE / UTM zone 38S",
        "code" : "EPSG:32538"
    }, {
        "description" : "WGS 72BE / UTM zone 39S",
        "code" : "EPSG:32539"
    }, {
        "description" : "WGS 72BE / UTM zone 40S",
        "code" : "EPSG:32540"
    }, {
        "description" : "WGS 72BE / UTM zone 41S",
        "code" : "EPSG:32541"
    }, {
        "description" : "WGS 72BE / UTM zone 42S",
        "code" : "EPSG:32542"
    }, {
        "description" : "WGS 72BE / UTM zone 43S",
        "code" : "EPSG:32543"
    }, {
        "description" : "WGS 72BE / UTM zone 44S",
        "code" : "EPSG:32544"
    }, {
        "description" : "WGS 72BE / UTM zone 45S",
        "code" : "EPSG:32545"
    }, {
        "description" : "WGS 72BE / UTM zone 46S",
        "code" : "EPSG:32546"
    }, {
        "description" : "WGS 72BE / UTM zone 47S",
        "code" : "EPSG:32547"
    }, {
        "description" : "WGS 72BE / UTM zone 48S",
        "code" : "EPSG:32548"
    }, {
        "description" : "WGS 72BE / UTM zone 49S",
        "code" : "EPSG:32549"
    }, {
        "description" : "WGS 72BE / UTM zone 50S",
        "code" : "EPSG:32550"
    }, {
        "description" : "WGS 72BE / UTM zone 51S",
        "code" : "EPSG:32551"
    }, {
        "description" : "WGS 72BE / UTM zone 52S",
        "code" : "EPSG:32552"
    }, {
        "description" : "WGS 72BE / UTM zone 53S",
        "code" : "EPSG:32553"
    }, {
        "description" : "WGS 72BE / UTM zone 54S",
        "code" : "EPSG:32554"
    }, {
        "description" : "WGS 72BE / UTM zone 55S",
        "code" : "EPSG:32555"
    }, {
        "description" : "WGS 72BE / UTM zone 56S",
        "code" : "EPSG:32556"
    }, {
        "description" : "WGS 72BE / UTM zone 57S",
        "code" : "EPSG:32557"
    }, {
        "description" : "WGS 72BE / UTM zone 58S",
        "code" : "EPSG:32558"
    }, {
        "description" : "WGS 72BE / UTM zone 59S",
        "code" : "EPSG:32559"
    }, {
        "description" : "WGS 72BE / UTM zone 60S",
        "code" : "EPSG:32560"
    }, {
        "description" : "WGS 84 / UTM grid system (northern hemisphere)",
        "code" : "EPSG:32600"
    }, {
        "description" : "WGS 84 / UTM zone 1N",
        "code" : "EPSG:32601"
    }, {
        "description" : "WGS 84 / UTM zone 2N",
        "code" : "EPSG:32602"
    }, {
        "description" : "WGS 84 / UTM zone 3N",
        "code" : "EPSG:32603"
    }, {
        "description" : "WGS 84 / UTM zone 4N",
        "code" : "EPSG:32604"
    }, {
        "description" : "WGS 84 / UTM zone 5N",
        "code" : "EPSG:32605"
    }, {
        "description" : "WGS 84 / UTM zone 6N",
        "code" : "EPSG:32606"
    }, {
        "description" : "WGS 84 / UTM zone 7N",
        "code" : "EPSG:32607"
    }, {
        "description" : "WGS 84 / UTM zone 8N",
        "code" : "EPSG:32608"
    }, {
        "description" : "WGS 84 / UTM zone 9N",
        "code" : "EPSG:32609"
    }, {
        "description" : "WGS 84 / UTM zone 10N",
        "code" : "EPSG:32610"
    }, {
        "description" : "WGS 84 / UTM zone 11N",
        "code" : "EPSG:32611"
    }, {
        "description" : "WGS 84 / UTM zone 12N",
        "code" : "EPSG:32612"
    }, {
        "description" : "WGS 84 / UTM zone 13N",
        "code" : "EPSG:32613"
    }, {
        "description" : "WGS 84 / UTM zone 14N",
        "code" : "EPSG:32614"
    }, {
        "description" : "WGS 84 / UTM zone 15N",
        "code" : "EPSG:32615"
    }, {
        "description" : "WGS 84 / UTM zone 16N",
        "code" : "EPSG:32616"
    }, {
        "description" : "WGS 84 / UTM zone 17N",
        "code" : "EPSG:32617"
    }, {
        "description" : "WGS 84 / UTM zone 18N",
        "code" : "EPSG:32618"
    }, {
        "description" : "WGS 84 / UTM zone 19N",
        "code" : "EPSG:32619"
    }, {
        "description" : "WGS 84 / UTM zone 20N",
        "code" : "EPSG:32620"
    }, {
        "description" : "WGS 84 / UTM zone 21N",
        "code" : "EPSG:32621"
    }, {
        "description" : "WGS 84 / UTM zone 22N",
        "code" : "EPSG:32622"
    }, {
        "description" : "WGS 84 / UTM zone 23N",
        "code" : "EPSG:32623"
    }, {
        "description" : "WGS 84 / UTM zone 24N",
        "code" : "EPSG:32624"
    }, {
        "description" : "WGS 84 / UTM zone 25N",
        "code" : "EPSG:32625"
    }, {
        "description" : "WGS 84 / UTM zone 26N",
        "code" : "EPSG:32626"
    }, {
        "description" : "WGS 84 / UTM zone 27N",
        "code" : "EPSG:32627"
    }, {
        "description" : "WGS 84 / UTM zone 28N",
        "code" : "EPSG:32628"
    }, {
        "description" : "WGS 84 / UTM zone 29N",
        "code" : "EPSG:32629"
    }, {
        "description" : "WGS 84 / UTM zone 30N",
        "code" : "EPSG:32630"
    }, {
        "description" : "WGS 84 / UTM zone 31N",
        "code" : "EPSG:32631"
    }, {
        "description" : "WGS 84 / UTM zone 32N",
        "code" : "EPSG:32632"
    }, {
        "description" : "WGS 84 / UTM zone 33N",
        "code" : "EPSG:32633"
    }, {
        "description" : "WGS 84 / UTM zone 34N",
        "code" : "EPSG:32634"
    }, {
        "description" : "WGS 84 / UTM zone 35N",
        "code" : "EPSG:32635"
    }, {
        "description" : "WGS 84 / UTM zone 36N",
        "code" : "EPSG:32636"
    }, {
        "description" : "WGS 84 / UTM zone 37N",
        "code" : "EPSG:32637"
    }, {
        "description" : "WGS 84 / UTM zone 38N",
        "code" : "EPSG:32638"
    }, {
        "description" : "WGS 84 / UTM zone 39N",
        "code" : "EPSG:32639"
    }, {
        "description" : "WGS 84 / UTM zone 40N",
        "code" : "EPSG:32640"
    }, {
        "description" : "WGS 84 / UTM zone 41N",
        "code" : "EPSG:32641"
    }, {
        "description" : "WGS 84 / UTM zone 42N",
        "code" : "EPSG:32642"
    }, {
        "description" : "WGS 84 / UTM zone 43N",
        "code" : "EPSG:32643"
    }, {
        "description" : "WGS 84 / UTM zone 44N",
        "code" : "EPSG:32644"
    }, {
        "description" : "WGS 84 / UTM zone 45N",
        "code" : "EPSG:32645"
    }, {
        "description" : "WGS 84 / UTM zone 46N",
        "code" : "EPSG:32646"
    }, {
        "description" : "WGS 84 / UTM zone 47N",
        "code" : "EPSG:32647"
    }, {
        "description" : "WGS 84 / UTM zone 48N",
        "code" : "EPSG:32648"
    }, {
        "description" : "WGS 84 / UTM zone 49N",
        "code" : "EPSG:32649"
    }, {
        "description" : "WGS 84 / UTM zone 50N",
        "code" : "EPSG:32650"
    }, {
        "description" : "WGS 84 / UTM zone 51N",
        "code" : "EPSG:32651"
    }, {
        "description" : "WGS 84 / UTM zone 52N",
        "code" : "EPSG:32652"
    }, {
        "description" : "WGS 84 / UTM zone 53N",
        "code" : "EPSG:32653"
    }, {
        "description" : "WGS 84 / UTM zone 54N",
        "code" : "EPSG:32654"
    }, {
        "description" : "WGS 84 / UTM zone 55N",
        "code" : "EPSG:32655"
    }, {
        "description" : "WGS 84 / UTM zone 56N",
        "code" : "EPSG:32656"
    }, {
        "description" : "WGS 84 / UTM zone 57N",
        "code" : "EPSG:32657"
    }, {
        "description" : "WGS 84 / UTM zone 58N",
        "code" : "EPSG:32658"
    }, {
        "description" : "WGS 84 / UTM zone 59N",
        "code" : "EPSG:32659"
    }, {
        "description" : "WGS 84 / UTM zone 60N",
        "code" : "EPSG:32660"
    }, {
        "description" : "WGS 84 / UPS North",
        "code" : "EPSG:32661"
    }, {
        "description" : "WGS 84 / Plate Carree",
        "code" : "EPSG:32662"
    }, {
        "description" : "WGS 84 / World Equidistant Cylindrical",
        "code" : "EPSG:32663"
    }, {
        "description" : "WGS 84 / BLM 14N (ftUS)",
        "code" : "EPSG:32664"
    }, {
        "description" : "WGS 84 / BLM 15N (ftUS)",
        "code" : "EPSG:32665"
    }, {
        "description" : "WGS 84 / BLM 16N (ftUS)",
        "code" : "EPSG:32666"
    }, {
        "description" : "WGS 84 / BLM 17N (ftUS)",
        "code" : "EPSG:32667"
    }, {
        "description" : "WGS 84 / UTM grid system (southern hemisphere)",
        "code" : "EPSG:32700"
    }, {
        "description" : "WGS 84 / UTM zone 1S",
        "code" : "EPSG:32701"
    }, {
        "description" : "WGS 84 / UTM zone 2S",
        "code" : "EPSG:32702"
    }, {
        "description" : "WGS 84 / UTM zone 3S",
        "code" : "EPSG:32703"
    }, {
        "description" : "WGS 84 / UTM zone 4S",
        "code" : "EPSG:32704"
    }, {
        "description" : "WGS 84 / UTM zone 5S",
        "code" : "EPSG:32705"
    }, {
        "description" : "WGS 84 / UTM zone 6S",
        "code" : "EPSG:32706"
    }, {
        "description" : "WGS 84 / UTM zone 7S",
        "code" : "EPSG:32707"
    }, {
        "description" : "WGS 84 / UTM zone 8S",
        "code" : "EPSG:32708"
    }, {
        "description" : "WGS 84 / UTM zone 9S",
        "code" : "EPSG:32709"
    }, {
        "description" : "WGS 84 / UTM zone 10S",
        "code" : "EPSG:32710"
    }, {
        "description" : "WGS 84 / UTM zone 11S",
        "code" : "EPSG:32711"
    }, {
        "description" : "WGS 84 / UTM zone 12S",
        "code" : "EPSG:32712"
    }, {
        "description" : "WGS 84 / UTM zone 13S",
        "code" : "EPSG:32713"
    }, {
        "description" : "WGS 84 / UTM zone 14S",
        "code" : "EPSG:32714"
    }, {
        "description" : "WGS 84 / UTM zone 15S",
        "code" : "EPSG:32715"
    }, {
        "description" : "WGS 84 / UTM zone 16S",
        "code" : "EPSG:32716"
    }, {
        "description" : "WGS 84 / UTM zone 17S",
        "code" : "EPSG:32717"
    }, {
        "description" : "WGS 84 / UTM zone 18S",
        "code" : "EPSG:32718"
    }, {
        "description" : "WGS 84 / UTM zone 19S",
        "code" : "EPSG:32719"
    }, {
        "description" : "WGS 84 / UTM zone 20S",
        "code" : "EPSG:32720"
    }, {
        "description" : "WGS 84 / UTM zone 21S",
        "code" : "EPSG:32721"
    }, {
        "description" : "WGS 84 / UTM zone 22S",
        "code" : "EPSG:32722"
    }, {
        "description" : "WGS 84 / UTM zone 23S",
        "code" : "EPSG:32723"
    }, {
        "description" : "WGS 84 / UTM zone 24S",
        "code" : "EPSG:32724"
    }, {
        "description" : "WGS 84 / UTM zone 25S",
        "code" : "EPSG:32725"
    }, {
        "description" : "WGS 84 / UTM zone 26S",
        "code" : "EPSG:32726"
    }, {
        "description" : "WGS 84 / UTM zone 27S",
        "code" : "EPSG:32727"
    }, {
        "description" : "WGS 84 / UTM zone 28S",
        "code" : "EPSG:32728"
    }, {
        "description" : "WGS 84 / UTM zone 29S",
        "code" : "EPSG:32729"
    }, {
        "description" : "WGS 84 / UTM zone 30S",
        "code" : "EPSG:32730"
    }, {
        "description" : "WGS 84 / UTM zone 31S",
        "code" : "EPSG:32731"
    }, {
        "description" : "WGS 84 / UTM zone 32S",
        "code" : "EPSG:32732"
    }, {
        "description" : "WGS 84 / UTM zone 33S",
        "code" : "EPSG:32733"
    }, {
        "description" : "WGS 84 / UTM zone 34S",
        "code" : "EPSG:32734"
    }, {
        "description" : "WGS 84 / UTM zone 35S",
        "code" : "EPSG:32735"
    }, {
        "description" : "WGS 84 / UTM zone 36S",
        "code" : "EPSG:32736"
    }, {
        "description" : "WGS 84 / UTM zone 37S",
        "code" : "EPSG:32737"
    }, {
        "description" : "WGS 84 / UTM zone 38S",
        "code" : "EPSG:32738"
    }, {
        "description" : "WGS 84 / UTM zone 39S",
        "code" : "EPSG:32739"
    }, {
        "description" : "WGS 84 / UTM zone 40S",
        "code" : "EPSG:32740"
    }, {
        "description" : "WGS 84 / UTM zone 41S",
        "code" : "EPSG:32741"
    }, {
        "description" : "WGS 84 / UTM zone 42S",
        "code" : "EPSG:32742"
    }, {
        "description" : "WGS 84 / UTM zone 43S",
        "code" : "EPSG:32743"
    }, {
        "description" : "WGS 84 / UTM zone 44S",
        "code" : "EPSG:32744"
    }, {
        "description" : "WGS 84 / UTM zone 45S",
        "code" : "EPSG:32745"
    }, {
        "description" : "WGS 84 / UTM zone 46S",
        "code" : "EPSG:32746"
    }, {
        "description" : "WGS 84 / UTM zone 47S",
        "code" : "EPSG:32747"
    }, {
        "description" : "WGS 84 / UTM zone 48S",
        "code" : "EPSG:32748"
    }, {
        "description" : "WGS 84 / UTM zone 49S",
        "code" : "EPSG:32749"
    }, {
        "description" : "WGS 84 / UTM zone 50S",
        "code" : "EPSG:32750"
    }, {
        "description" : "WGS 84 / UTM zone 51S",
        "code" : "EPSG:32751"
    }, {
        "description" : "WGS 84 / UTM zone 52S",
        "code" : "EPSG:32752"
    }, {
        "description" : "WGS 84 / UTM zone 53S",
        "code" : "EPSG:32753"
    }, {
        "description" : "WGS 84 / UTM zone 54S",
        "code" : "EPSG:32754"
    }, {
        "description" : "WGS 84 / UTM zone 55S",
        "code" : "EPSG:32755"
    }, {
        "description" : "WGS 84 / UTM zone 56S",
        "code" : "EPSG:32756"
    }, {
        "description" : "WGS 84 / UTM zone 57S",
        "code" : "EPSG:32757"
    }, {
        "description" : "WGS 84 / UTM zone 58S",
        "code" : "EPSG:32758"
    }, {
        "description" : "WGS 84 / UTM zone 59S",
        "code" : "EPSG:32759"
    }, {
        "description" : "WGS 84 / UTM zone 60S",
        "code" : "EPSG:32760"
    }, {
        "description" : "WGS 84 / UPS South",
        "code" : "EPSG:32761"
    }, {
        "description" : "WGS 84 / TM 36 SE",
        "code" : "EPSG:32766"
    } ],

    /**
     * APIMethod: getReference
     * 
     * Takes either an epsg or urn value and returns an object containing the
     * description of the coordinage system, the epsg value and the urn value.
     * Parameters:
     * 
     * {String} epsg - or a urn value to find a matching description.
     * 
     * Returns:
     * 
     * {Object} containing description, epsg, urn properties.
     */
    getReference : function(epsg) {
        if (this.isUrn(epsg)) {
            epsg = oscar.Util.EpsgConversion.urnToEpsg(epsg);
        }
        for ( var ref in this.coords) {
            var coordRef = this.coords[ref];
            if (coordRef.code == epsg) {
                coordRef.urn = oscar.Util.EpsgConversion.epsgToUrn(epsg);
                return coordRef;
            }
        }
        return {
            code : epsg,
            urn : oscar.Util.EpsgConversion.epsgToUrn(epsg),
            description : epsg
        };

    },
    /**
     * APIMethod: isUrn
     * 
     * Parameters:
     * 
     * {String} epsg - Checks to see if the value passed in is an epsg or urn
     * value.
     * 
     * Returns:
     * 
     * boolean
     */
    isUrn : function(epsg) {
        if (epsg.indexOf("::") == -1) {
            return false;
        } else {
            return true
        }
    }

}
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Util.Transform
 * 
 * Provides xslt transformation for xml documents.
 * 
 * Usage: var results = oscar.Util.Transform.transform(xml,xsl);
 * 
 * 
 */

oscar.Util.Transform = {
    /**
     * APIMethod: transform Transforms an xml node based on the rules passed in
     * by the xsl stylesheet.
     * 
     * Parameters:
     * 
     * xml - The xml node. xsl - The xsl stylesheet.
     * 
     * Returns: {String} (if IE) or {Element}
     */
    transform : function(xml, xsl) {
        try {
            var processor = new XSLTProcessor();
            processor.importStylesheet(xsl);
            var ex = processor.transformToFragment(xml, document);
            return ex;
        } catch (err) {
            var ex = xml.transformNode(xsl);
            return ex;
        }
    }
};
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Util.Help
 * 
 * Class used to display help documentation provided by an xml file.
 * 
 * Usage: new oscar.Util.Help('DataExtractor');
 */

oscar.Util.Help = oscar.BaseClass({
    /**
     * APIProperty: baseUrl {String} - Base path to the help files.
     */
    baseUrl : oscar._getScriptLocation(),

    /**
     * APIProperty: helpDir {String} - Path to the directory containing the help
     * files relative to the baseUrl.
     * 
     * Default: help/
     */
    helpDir : "help/",

    /**
     * Property: helpFile Name of the help file with local variable.
     */
    helpFile : "help_{local}.xml",

    /**
     * Property: helpStyle {String} - Name of the xsl file containing
     * transformation definitions.
     */
    helpStyle : "help.xsl",
    /**
     * Propery: dialog Reference to the help dialog.
     */
    dialog : null,
    /**
     * Constructor Parameters: *helpReference* - {String} Must match and
     * 
     * @id attribute of a component element in the help xml file. *options* -
     *     {Object} (Optional) Used to override default properties.
     */
    initialize : function(helpReference, options) {
        if (options) {
            OpenLayers.Util.extend(this, options);
        }
        this.helpReference = helpReference;
        this.help();
    },
    /**
     * Method: help Makes the requests for the xml and xsl files.
     */
    help : function() {

        var helpDoc = this.baseUrl + this.helpDir + this.helpFile;
        helpDoc = helpDoc.replaceAll("{local}", OpenLayers.Lang.getCode());
        OpenLayers.Request.GET({
            url : helpDoc,
            async : false,
            success : function(resp) {
                this.xmlDocument = resp.responseXML;
            },
            failure : function(resp) {
                new oscar.Gui.AlertDialog(oscar.i18n("Error"), oscar.i18n("FileNotFound"), {
                    width : 300,
                    height : 100,
                    draggable : true
                });
            },
            scope : this
        });
        if (!this.xmlDocument)
            return;
        OpenLayers.Request.GET({
            url : this.baseUrl + this.helpDir + this.helpStyle,
            async : false,
            success : function(resp) {

                this.xslDocument = resp.responseXML;
            },
            failure : function(resp) {
                alert('Unable to retrive help document');
            },
            scope : this
        });
        if (!this.xslDocument)
            return;
        this.createDialog();

    },
    /**
     * Method: render Initiates the xml transformation and displays the contents
     * in the dialog.
     */
    render : function() {
        if (this.xmlDocument && this.xslDocument) {
            oscar.jQuery(this.div).hide();
            if ((helpNode = this.getHelpNode())) {
                var helpContent = oscar.Util["Transform"].transform(helpNode, this.xslDocument);
                if (typeof helpContent == "string") {
                    this.div.innerHTML = helpContent;
                } else {
                    this.div.appendChild(helpContent);
                }
            }
            oscar.jQuery(this.div).fadeIn();
        }
    },
    /**
     * Method: getHelpNode Traverses through the xml file to find the correct
     * help component node.
     */
    getHelpNode : function() {
        var children = this.xmlDocument.documentElement.childNodes;
        var childNode, processor;
        for (var i = 0; i < children.length; ++i) {
            childNode = children[i];
            if (childNode.nodeType == 1) {
                var props = {};
                for (var j = 0; j < childNode.attributes.length; j++) {
                    var attribute = childNode.attributes[j];
                    if (attribute.nodeName == "id" && attribute.nodeValue == this.helpReference) {
                        return childNode;
                    }
                }
            }
        }
        return false;

    },
    /**
     * Method: createDialog Creates the dialog.
     */
    createDialog : function() {
        this.dialog = new oscar.Gui.Dialog("Help", {
            draggable : true,
            modal : false,
            width : 350,
            height : 350,
            zIndex : 2000
        });
        var header = oscar.i18n("Help");
        this.dialog.setHeader(header);
        this.div = document.createElement("div");
        this.div.setAttribute("class", "help");
        this.dialog.setContent(this.div);
        var closeFn = function(obj) {
            return function() {
                obj.dialog.destroy();
                obj.dialog = null;
            }
        };
        this.dialog.createButton(oscar.i18n("Close"), closeFn(this));
        this.dialog.show();
        this.render();
    },
    /**
     * Constant: CLASS_NAME
     */
    CLASS_NAME : "oscar.Util.Help"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Util.PluginManager
 * 
 * This class is used to manage plugins
 */
oscar.Util.PluginManager = new oscar.BaseClass({
    /**
     * APIProperty: registeredPlugins
     * 
     * This is a mapping of registered plugins
     */
    registeredPlugins : {},
    initialize : function() {
    },
    /**
     * APIMethod: register
     * 
     * This is the method to register a plugin
     * 
     * Parameter: type - The type of plugin. pluginClass: The class of the
     * plugin. Ex:
     * .register(oscar.Util.Plugin.Download.WCSService.prototype.pluginType,
     * oscar.Util.Plugin.Download.WCSService);
     */
    register : function(type, pluginClass) {
        this.registeredPlugins[type] = pluginClass;
    },
    /**
     * APIMethod: getPluginFor
     * 
     * Returns a plugin for the defined protocol string
     * 
     * Parameter:
     * 
     * getPluginType - The type of plugin
     */
    getPluginFor : function(pluginType) {
        var plugin = this.registeredPlugins[pluginType];
        if (!plugin) {
            return null;
        }
        return new plugin();
    },
    CLASS_NAME : "oscar.Util.PluginManager"
});

/**
 * This sets creates a plugin manager in oscar by default and creates a
 * getPluginManager method.
 */
oscar._pluginManager = new oscar.Util.PluginManager();
oscar.getPluginManager = function() {
    return oscar._pluginManager;
}
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Util.Plugin
 * 
 * This is a base class for all plugins.
 */
oscar.Util.Plugin = new oscar.BaseClass({
    EVENT_TYPES : [],
    events : null,

    catalogueService : null,
    /**
     * APIProperty: pluginType
     * 
     * A string to define what the plugin is for. This will be used for the
     * plugin registration
     * 
     */
    pluginType : null,
    /**
     * APIPropery: icon
     * 
     * A css class that defines the icon for this plugin. The class should
     * follow the jQueryUI icon styles
     */
    icon : null,
    /**
     * APIMethod: getIcon Returns the icon
     */
    getIcon : function() {
        return this.icon;
    },
    getLabel : function() {
        return oscar.i18n(this.getPluginType());
    },
    initialize : function(options) {
        this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false, {
            includeXY : true
        });
        this.setOptions(options);
    },
    /**
     * APIMethod: getPluginType
     * 
     * Returns the plugin type
     */
    getPluginType : function() {
        return this.pluginType || "&nbsp;";
    },
    /**
     * APIMethod: setOptions
     * 
     * This method will take a JSON class definition and apply the properties to
     * this object. Parameter: - options A JSON class
     * 
     */
    setOptions : function(options) {
        OpenLayers.Util.extend(this, options);
    },

    destroy : function() {
    },

    /**
     * APIMethod: play
     * 
     * This is the method that is called to run the plugin.
     */
    play : function() {
    },

    drawTo : function($jqDiv) {
        var $button = $$("<button></button>").html(this.getPluginType());
        var scope = this;
        $button.button({
            icons : {
                primary : this.getIcon() || "ui-icon-disk"
            },
            text : false,
            label : this.getLabel()
        }).click($$.proxy(function() {
            this.play();
        }, this));
        $jqDiv.append($button);

    },
    CLASS_NAME : "oscar.Util.Plugin"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Util.Plugin.Download
 * 
 * This is the base class for all plugins related to downloading.
 * 
 */
oscar.Util.Plugin.Download = new oscar.BaseClass(oscar.Util.Plugin, {
    /**
     * APIMethod: initialize
     * 
     * The initialize method
     */
    initialize : function(options) {
        oscar.Util.Plugin.prototype.initialize.apply(this, [ options ]);
    },
    CLASS_NAME : "oscar.Util.Plugin.Download"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Util.Plugin.Download.Options = new oscar.BaseClass(oscar.Util.Plugin.Download, {
    EVENT_TYPES : [ 'enterMode', 'exitMode' ],
    parent : null,
    initialize : function(options) {
        oscar.Util.Plugin.Download.prototype.initialize.apply(this, [ options ]);
        this.events.on({
            "enterMode" : function() {
                this.mode_container.show('slide', {
                    direction : "left"
                }, 500);
            },
            "exitMode" : function() {
                var scope = this;
                this.mode_container.hide('slide', {
                    direction : "left"
                }, 500, $$.proxy(function() {
                    this.mode_container.empty();
                }, this));
            },
            scope : this
        });
    },
    destroy : function() {
        this.events.triggerEvent("exitMode");

    },
    /**
     * @Override
     * @see oscar.Util.Plugin
     */
    play : function() {
        this.events.triggerEvent("enterMode", this);
        var scope = this;
        this.container = $$("<div></div>");
        this.container.addClass("options-mode-container");
        this.container.css("width", "100%");
        this.container.css("height", "100%");

        this.mode_container.append(this.container);

        var scope = this;

        var north = $$("<div></div>");
        var title_panel = $$("<div></div>");
        this.$title = $$("<h2></h2>").html("&nbsp;");
        var btn_panel = $$("<div></div>");
        btn_panel.css({
            "float" : "right"
        });
        var close_btn = $$("<button></button>").html("close");
        btn_panel.append(close_btn);
        title_panel.append(btn_panel);
        title_panel.append(this.$title);
        north.append(title_panel);
        close_btn.button({
            icons : {
                primary : "ui-icon-oux-close"
            },
            text : false,
            label : "Close"
        }).click(function() {
            scope.destroy();
        });
        var center = $$("<div></div>");
        north.addClass("ui-layout-north");
        north.css("padding", "5px");
        center.addClass("ui-layout-center");
        center.css("padding", "5px");
        this.container.append(north);
        this.container.append(center);

        this.content_pane = $$("<div></div>");
        this.content_pane.addClass("coverage-mode-content");

        center.append(this.content_pane);
        this.content_pane.css({
            "margin-left" : "5px",
            "margin-right" : "10px",
            "height" : "100% !important"
        });
        this.layout = this.container.layout();
        $$(window).resize($$.proxy(function() {
            this.layout.resizeAll();
        }, this));
        var scope = this;

        this.content_pane.slimScroll({
            "height" : "100%"
        });
    },
    setTitle : function(str) {
        this.$title.html(str);
    },
    addOption : function(jDiv) {
        this.content_pane.append(jDiv);
    },
    CLASS_NAME : "oscar.Util.Plugin.Download.Options"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 * 
 * Copyright 2014 CARIS <http://www.caris.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Util.Plugin.Download.WCSService
 * 
 * This plugin is used for downloading from the WCS service.
 * 
 */

oscar.Util.Plugin.Download.WCSService = new oscar.BaseClass(
		oscar.Util.Plugin.Download.Options,
		{
			pluginType : "OGC:WCS-1.1.0-http-get-capabilities",
			icon : "ui-icon-wcs-download",
			cropControl : null,
			downloadOptions : {},
			initialize : function(options) {
				oscar.Util.Plugin.Download.Options.prototype.initialize.apply(
						this, [ options ]);
				this.events.on({
					"enterMode" : function() {
						try {
							this.map.getLayersByName("results")[0]
									.setVisibility(false);
						} catch (err) {
						}
					},
					"exitMode" : function() {
						try {
							this.map.getLayersByName("results")[0]
									.setVisibility(true);
						} catch (err) {
						}
					},
					scope : this
				});
			},

			/**
			 * Method: sendRequest
			 * 
			 * This method sends the GetCapabilities request to the WCS service.
			 */
			sendRequest : function() {
				var params = {
					request : "GetCapabilities",
					service : "WCS",
					version : "1.1.0"
				}

				var baseUrl = $$.trim(this.link.url).split("?")[0];

				this.wcsRequest = OpenLayers.Request.GET({
					url : baseUrl,
					params : params,
					async : false,
					success : this.success,
					timeout : 1000,
					scope : this
				});
			},

			/**
			 * Method: success
			 * 
			 * Called when the response was successful.
			 */
			success : function(response) {
				var reader = new oscar.Format.WCSCapabilities();
				this.capabilities = reader.read(response.responseXML);
				var coverage = null;

				var url = oscar.Util.Metadata.getOperationHref(
						this.capabilities, "DescribeCoverage");

				OpenLayers.Request.GET({
					url : $$.trim(url),
					async : false,
					params : {
						request : "DescribeCoverage",
						service : "WCS",
						version : "1.1.0",
						identifiers : this.record.dataIdentifier
					},
					success : function(resp) {
						var reader = new oscar.Format.WCSDescribeCoverage();
						coverage = reader.read(resp.responseXML);
					},
					timeout : 1000,
					scope : this
				});
				this.coverageDescription = coverage.coverageDescription;
			},

			/**
			 * @Override
			 * @see oscar.Util.Plugin
			 */
			play : function() {
				oscar.Util.Plugin.Download.Options.prototype.play.apply(this);
				this.sendRequest();
				this.showPreviewLayer();
				this.buildInformationPanel();
				this.buildDownloadOptionsPanel();
			},
			buildInformationPanel : function() {
				var $panel = $$("<div></div>");
				var title = "";
				try {
					title = this.record.title[0].value
							|| this.record.identifier[0].value;
				} catch (err) {
					title = "";
				}
				this.setTitle(title);
				var abs = "";
				try {
					abs = this.record["abstract"][0];
				} catch (err) {
					abs = "";
				}
				$panel = $$("<div></div>");
				var $abstract = $$("<p></p>").html(abs);
				$panel.append($abstract);
				this.addOption($panel);

			},
			buildDownloadOptionsPanel : function() {
				this.$panel = $$("<div></div>");
				var $header = $$("<h2></h2>").html("Download Options");
				$header.css("border-bottom", "1px solid black");

				this.$panel.append($header);
				this._createFormatList();
				this._createCRSList();
				this._createFields();
				this._createResolution();
				this._createToolButtons();

				this.addOption(this.$panel);
			},
			_createToolButtons : function() {
				var styleMap = new OpenLayers.StyleMap({
					"default" : new OpenLayers.Style({
						strokeDashstyle : "dash",
						fillColor : "#0f0f0f",
						fillOpacity : 0.5,
						strokeWidth : 0,
						strokeColor : "#0f0f0f",
						strokeOpacity : 0
					})
				});
				var crop_layer = new OpenLayers.Layer.Vector(
						"wcs-polygon-preview", {
							styleMap : styleMap,
							renderers : [ 'Canvas', 'VML' ],
							wrapDateLine : true,
							projection : this.map.getProjectionObject()
						});
				this.map.addLayer(crop_layer);
				var exitMode_function = function(evt) {
					this.map.removeLayer(crop_layer);
					this.events.un({
						"exitMode" : exitMode_function
					});
				};
				this.events.on({
					"exitMode" : exitMode_function,
					scope : this
				});

				var $btnPanel = $$("<div></div>");
				$btnPanel.addClass("buttonsPanel");

				var $crop_button = $$("<button></button?").html("Crop");
				var $download_button = $$("<button></button?").html("Download");

				var crop_btn_container = $$("<div></div>").addClass(
						"oscar-inline-container");
				var download_btn_container = $$("<div></div>").addClass(
						"oscar-inline-container");
				crop_btn_container.append($crop_button);
				download_btn_container.append($download_button);

				$btnPanel.append(crop_btn_container);
				$btnPanel.append(download_btn_container);

				$crop_button.button({
					icons : {
						primary : "ui-icon-crop"
					},
					text : false
				}).click($$.proxy(function() {
					this.cropControl.activate();
				}, this));

				$download_button.button({
					icons : {
						primary : "ui-icon-prepare"
					},
					text : false
				}).click($$.proxy(this.prepareDownload, this));

				this.cropControl = new oscar.Control.Box();
				this.map.addControl(this.cropControl);
				this.cropControl.events.on({
					'done' : function(geom) {
						if (this.cropFeature) {
							crop_layer.removeFeatures(this.cropFeature);

						}
						this.cropFeature = new OpenLayers.Feature.Vector(geom);
						crop_layer.addFeatures(this.cropFeature);
						// crop_layer.drawFeature(this.cropFeature);
						this.downloadOptions.bbox = geom.getBounds().clone();
						this.cropControl.deactivate();
					},
					scope : this
				});
				this.$panel.append($btnPanel);
				var $header = $$("<h2></h2>").html("Download Results");
				$header.css("border-bottom", "1px solid black");
				this.$panel.append($header);
				this.results_panel = $$("<div></div>");

				this.$panel.append(this.results_panel);
			},
			prepareDownload : function() {
				var GetCoverageOp = oscar.Util.Metadata.getOperation(
						this.capabilities, "GetCoverage");

				var isServiceStorageAllowed = function(op) {
					for (var i = 0; i < op.parameters.length; i++) {
						var param = op.parameters[i];
						if (param.name.toLowerCase("store")) {
							for (var j = 0; j < param.values.length; j++) {
								if (param.values[j].toLowerCase() == "true") {
									return true;
								}
							}

						}
					}
					return false;
				};

				var selectedBounds = this.downloadOptions.bbox;
				var resolution = parseFloat(this.downloadOptions.resolution);
				var longitude = Math.abs(resolution);
				var latitude = longitude;
				if (latitude > 0) {
					latitude *= -1;
				}
				var format = this.selectedFormat;

				// convert the crs to urn;
				var projection = new OpenLayers.Projection(
						this.downloadOptions.crs.code);
				var isGeographicCRS = oscar.Util.isGeographicCRS(projection);
				var urn = oscar.Util.EpsgConversion
						.epsgToUrn(projection.projCode);

				// Does the bounds need to be transformed
				if (projection.projCode != this.map.getProjection()) {
					selectedBounds = selectedBounds.clone().transform(
							this.map.getProjectionObject(), projection);
				}

				longitude /= oscar.Util.getMetersConversionFactor(projection);
				latitude /= oscar.Util.getMetersConversionFactor(projection);

				var bbox = selectedBounds.toArray(isGeographicCRS);
				bbox.push(urn);

				var offsets = [];
				if (isGeographicCRS) {
					offsets.push(latitude);
					offsets.push(longitude);
				} else {
					offsets.push(longitude);
					offsets.push(latitude);
				}

				if (this.gridType == "urn:ogc:def:method:WCS:1.1:2dGridIn2dCrs") {
					offsets.splice(1, 0, 0, 0);
				}
				var fields = [];
				for (var i = 0; i < this.downloadOptions.fields.length; i++) {
					var field = this.downloadOptions.fields[i];
					fields.push(field.name + ":" + field.interpolation);
				}
				var rangeSubset = "";
				if (fields.length > 1) {
					rangeSubset = fields.join(";");
				} else {
					rangeSubset = fields.join(" ");
				}
				var url = GetCoverageOp.dcp.http.get;
				var qStringParams = {
					"version" : "1.1.0",
					"service" : "WCS",
					"request" : "GetCoverage",
					"store" : isServiceStorageAllowed(GetCoverageOp),
					"identifier" : this.record.dataIdentifier,
					"format" : this.downloadOptions.format,
					"BoundingBox" : bbox.toString(),
					"GridOrigin" : this.downloadOptions.gridOrigin.toString(),
					"GridOffsets" : offsets.toString(),
					"RangeSubset" : rangeSubset,
					"GridBaseCRS" : urn,
					"GridType" : this.downloadOptions.gridType
				};

				var $div = $$("<div></div>")
				$div.css({
					"border-bottom" : "1px solid black",
					"padding-bottom" : "3px",
					"width" : "100%"
				});
				$div.addClass("md_loadingActive");
				this.results_panel.prepend($div);
				OpenLayers.Request.GET({
					url : url,
					params : qStringParams,
					scope : this,
					success : function(resp) {
						this.transformResult($div, qStringParams, resp);
					},
					failure : function(a) {

					},
				});
			},
			transformResult : function($div, params, resp) {

				var xml = resp.responseXML;
				var xsl = null;
				OpenLayers.Request.GET({
					url : oscar._getScriptLocation()
							+ "resources/CoverageDownload.xsl",
					async : false,
					success : function(resp) {
						xsl = resp.responseXML;
					},
					scope : this
				});

				$div.removeClass("md_loadingActive");

				var $timestamp = $$("<div></div>").html(new Date());
				var $format = $$("<div></div>").html("Format:" + params.format);
				var $crs = $$("<div></div>").html("CRS:" + params.GridBaseCRS);
				$div.append($format);
				$div.append($crs);
				$div.append($timestamp);

				var transformation = oscar.Util.Transform.transform(xml, xsl);
				var $btns = $$("<div></div>");
				$btns.css({
					"text-align" : "left"
				});

				$btns.append(transformation);
				$div.append($btns);
				$btns.find("a").each(function() {
					var $this = $$(this);
					var icon = $this.attr("data-class");
					$this.button({
						icons : {
							primary : icon
						},
						text : false,
						label : "&nbsp;"
					}).click(function() {
						var url = $this.attr("href");
						oscar.Util.downloadFromService(url, "123-456");
						return false;
					});
				});

			},
			_createResolution : function() {
				var projection = new OpenLayers.Projection(
						this.coverageDescription.domain.spatialDomain.gridCRS.gridBaseCRS);
				this.downloadOptions.gridType = "urn:ogc:def:method:WCS:1.1:2dSimpleGrid"
				this.downloadOptions.gridOrigin = "0 0";
				var gridOffsets = "0 0";

				if (this.coverageDescription.domain.spatialDomain.gridCRS.gridOrigin) {
					this.downloadOptions.gridOrigin = this.coverageDescription.domain.spatialDomain.gridCRS.gridOrigin;
				}

				this.downloadOptions.gridOrigin = this.downloadOptions.gridOrigin
						.split(" ");

				if (this.coverageDescription.domain.spatialDomain.gridCRS.gridType) {
					this.downloadOptions.gridType = this.coverageDescription.domain.spatialDomain.gridCRS.gridType;
				}

				if (this.coverageDescription.domain.spatialDomain.gridCRS.gridOffsets) {
					gridOffsets = this.coverageDescription.domain.spatialDomain.gridCRS.gridOffsets;
				}

				var offsets = oscar.Util.getGridOffsets(gridOffsets);
				this.downloadOptions.gridOffsets = {
					longitude : 0,
					latitude : 0
				};

				// Are the offsets in XY or YX order?
				if (oscar.Util.isGeographicCRS(projection)) {
					this.downloadOptions.resolution = parseFloat(offsets[1])
							* oscar.Util.getMetersConversionFactor(projection);
				} else {
					this.downloadOptions.resolution = parseFloat(offsets[0])
							* oscar.Util.getMetersConversionFactor(projection);
				}

				var baseResolution = this.downloadOptions.resolution;

				var $resolution = $$("<h3></h3>").html("Resolution");
				var $slider = $$("<div></div>");
				var $spinnerDiv = $$("<div></div>");
				$spinnerDiv.css({
					"margin-top" : "5px",
					"width" : "200px",
					"margin-left" : "8px",
					"outline" : "none",
					"font-size" : "99%"
				});
				var $spinner = $$("<input type='text' id='resolutionSpinner'/>");
				$spinner.css({
					"outline" : "none",
					"width" : "50px"
				});
				var $sliderLblDiv = $$("<div></div>")
				$sliderLblDiv.css("position", "relative");
				var coarseLbl = $$("<label></label>")
						.html(oscar.i18n("Coarse"));
				var fineLblText = oscar.i18n("Fine") + " ({0} m)";
				var fineLbl = $$("<label></label>").html(
						fineLblText.format(baseResolution));
				fineLbl.css({
					"position" : "absolute",
					"right" : "0px"
				});

				$sliderLblDiv.css({
					"width" : "200px",
					"margin-left" : "8px",
					"outline" : "none"
				});

				$slider.css({
					"width" : "200px",
					"margin-left" : "8px",
					"outline" : "none"
				});
				$sliderLblDiv.append(fineLbl);
				$sliderLblDiv.append(coarseLbl);

				$spinnerDiv.append($spinner);
				this.$panel.append($resolution);
				this.$panel.append($sliderLblDiv);
				this.$panel.append($slider);
				this.$panel.append($spinnerDiv);

				var scope = this;
				$slider.slider({
					value : 100,
					step : 10,
					slide : function(e, u) {
						var multiplier = 1 + (1 - u.value / 100);
						var product = baseResolution * multiplier;
						$spinner.spinner("value", product);
					}
				});
				$spinner.val(baseResolution);
				$spinner
						.spinner(
								{
									min : baseResolution,
									max : baseResolution * 2,
									step : 0.010,
									numberFormat : "N3",
									change : function(e, ui) {
										var val = $spinner.val();
										var pct = ((1 - (val / baseResolution - 1)) * 100)
												.toFixed(2);
										if (pct >= 0 && pct <= 100) {
											$slider.slider("value", pct);
										}
										scope.downloadOptions.resolution = val;
									},
									spin : function(e, ui) {
										var val = $spinner.val();
										var pct = ((1 - (val / baseResolution - 1)) * 100)
												.toFixed(2);
										if (pct >= 0 && pct <= 100) {
											$slider.slider("value", pct);
										}
										scope.downloadOptions.resolution = val;
									}
								}).blur(
								function(e) {
									var $this = $$(this);
									if (isNaN($this.val())
											|| ($this.val() < baseResolution)) {
										$this.val(baseResolution);
										e.stopImmediatePropagation();
									}

								});
			},
			_createFormatList : function() {
				var scope = this;
				var outputFormats = this.coverageDescription.supportedFormats;
				var simpleFormats = [];
				for (var i = 0; i < outputFormats.length; i++) {
					simpleFormats.push({
						label : oscar.Util.convertFormat(outputFormats[i]),
						value : outputFormats[i]
					});
				}
				var $format = $$("<h3></h3>").html("Format");

				var $format_input = $$("<input/>");
				$format_input.attr("readonly", "readonly");
				$format_input.val(simpleFormats[0]);
				$format_input.css("cursor", "pointer");
				this.$panel.append($format);

				this.$panel.append($format_input);
				$format_input.autocomplete({
					minLength : 0,
					source : simpleFormats,
					select : $$.proxy(function(event, ui) {
						var $this = $$(event.target);
						$this.val(ui.item.label);
						this.downloadOptions.format = ui.item.value
						return false;
					}, this)
				}).focus(function() {
					$$(this).autocomplete("search", "");
				}).click(function() {
					$$(this).focus();
				});

				$format_input.val(simpleFormats[0].label);
				this.downloadOptions.format = simpleFormats[0].value;
			},
			_createCRSList : function() {
				var gridBaseReference = oscar.Util.CoordinateReferences
						.getReference(this.coverageDescription.domain.spatialDomain.gridCRS.gridBaseCRS);
				var gridBaseProjection = new OpenLayers.Projection(
						gridBaseReference.code);
				var scope = this;
				supportedCRSList = this.coverageDescription.supportedCRS;
				var projections = [];
				for (var i = 0; i < supportedCRSList.length; i++) {
					var crs = oscar.Util.CoordinateReferences
							.getReference(supportedCRSList[i]);
					var tmpProj = new OpenLayers.Projection(crs.code);
					projections.push(crs);
				}

				var $crs_input = $$("<input/>");
				$crs_input.css({
					"cursor" : "pointer",
					"width" : "90%"
				});

				var $crs = $$("<h3></h3>").html("Coordinate Reference System");
				this.$panel.append($crs);
				this.$panel.append($crs_input);
				var scope = this;
				$crs_input
						.autocomplete(
								{
									minLength : 0,
									source : function(request, response) {
										var matches = [];
										for ( var i in projections) {
											var projection = projections[i];
											if (request.term.length > 0) {
												var q = request.term
														.toLowerCase();
												if (projection.description
														.toLowerCase().indexOf(
																q) != -1
														|| projection.code
																.toLowerCase()
																.indexOf(q) != -1) {
													matches.push(projection);
												} else {
													$crs_input
															.val(scope.downloadOptions.crs.description);
												}
											} else {
												matches.push(projection);
											}
										}
										response(matches);
									},
									select : $$.proxy(function(event, ui) {
										var $this = $$(event.target);
										$this.val(ui.item.description);
										this.downloadOptions.crs = ui.item;
										return false;
									}, this)
								})
						.focus(function() {
							$$(this).autocomplete("search", "");
						})
						.click(function() {
							$$(this).focus();
						})
						.blur(
								function(event) {
									if ($$(event.target).val().length == 0) {
										$crs_input
												.val(this.downloadOptions.crs.description);
									}
								}).data('ui-autocomplete')._renderItem = function(
						ui, item) {
					return $$("<li></li>").data("ui-autocomplete-item", item)
							.append(
									"<a>" + item.code + "<br/>"
											+ item.description + "</a>")
							.appendTo(ui);
				}
				$crs_input.val(gridBaseReference.description);
				this.downloadOptions.crs = gridBaseReference;

				oscar.isProjectionReady = window
						.setInterval(
								$$
										.proxy(
												function() {
													if (!gridBaseProjection.proj.readyToUse) {
														return;
													}
													window
															.clearInterval(oscar.isProjectionReady);
													var spatialDomain = this.coverageDescription.domain.spatialDomain;
													for (var i = 0; i < this.coverageDescription.domain.spatialDomain.boundingBoxes.length; i++) {
														var bbox = this.coverageDescription.domain.spatialDomain.boundingBoxes[i];
														if (bbox.crs == this.coverageDescription.domain.spatialDomain.gridCRS.gridBaseCRS) {
															var bbox = new OpenLayers.Bounds(
																	bbox.west,
																	bbox.south,
																	bbox.east,
																	bbox.north);
															this.downloadOptions.bbox = bbox
																	.transform(
																			this.downloadOptions.crs.code,
																			this.map
																					.getProjectionObject());
														}
													}
												}, this), 500);
			},
			_createFields : function() {
				// create a table row to return.
				this.downloadOptions.fields = [];
				var createFieldRow = function(field, isDefaultField, scope) {
					var numeric_id = Math.round(Math.random() * 1000);

					var $row = $$("<tr></tr>");
					var $checkbox_cell = $$("<td></td>");
					var $field_label_cell = $$("<td></td>");
					var $field_interpolation_cell = $$("<td></td>");

					var identifier = field.identifier;
					var defaultMethod = field.interpolationMethods.defaultMethod;
					var methods = field.interpolationMethods.methods;

					// Create the field checkbox
					var $field_checkbox = $$("<input/>");
					$field_checkbox.click($$.proxy(function(event) {
						this.updateFields($$(event.target));
					}, scope));

					var checkbox_id = field.identifier + "_" + numeric_id + "_"
							+ "checkbox";
					var select_id = field.identifier + "_" + numeric_id + "_"
							+ "select"
					$field_checkbox.attr("type", "checkbox");
					$field_checkbox.attr("id", checkbox_id);
					$field_checkbox.attr("data-field", field.identifier);
					$field_checkbox.attr("data-interpolation", select_id);
					$checkbox_cell.append($field_checkbox);

					// Create the field label
					var $field_label = $$("<label></label>").html(identifier);
					$field_label_cell.append($field_label);

					// Create the field interpolation methods dropdown.
					var $select_interpolation = $$("<select></select>");
					$select_interpolation.attr("id", select_id);
					$select_interpolation.attr("data-checkbox", checkbox_id);
					for (var i = 0; i < methods.length; i++) {
						var method = methods[i];
						var $option = $$("<option></option>").html(method);
						$option.val(method);
						if (method == defaultMethod) {
							$option.attr("selected", "selected");
						}
						$select_interpolation.append($option);
						$field_interpolation_cell.append($select_interpolation);
					}

					$select_interpolation.change($$.proxy(function(event) {
						var $this = $$(event.target);
						var $checkbox = $$("#" + $this.attr("data-checkbox"));
						this.updateFields($checkbox);
					}, scope));

					$row.append($checkbox_cell);
					$row.append($field_label_cell);
					$row.append($field_interpolation_cell);
					if (isDefaultField) {
						setTimeout(function() {
							$field_checkbox.click();
						}, 500);
					}
					return $row;
				};

				var fields = this.coverageDescription.range.fields
				var $fields = $$("<h3></h3>").html("Fields");
				var $table = $$("<table></table");
				this.$panel.append($fields);
				this.$panel.append($table);
				for (var i = 0; i < fields.length; i++) {
					var field = fields[i];
					isDefaultField = (i == 0) ? true : false;
					$table.append(createFieldRow(field, isDefaultField, this));
				}

			},
			updateFields : function($checkbox) {
				var $select = $$("#" + $checkbox.attr("data-interpolation"));
				var field = {
					name : $checkbox.attr("data-field"),
					interpolation : $select.val()
				}
				if ($checkbox.is(":checked")) {
					var select_id = $checkbox.attr("data-field") + "_"
							+ $checkbox.attr("data-numeric-id") + "_select";
					var $interpolation = $$("#" + select_id);

					var found = false;
					for (var i = 0; i < this.downloadOptions.fields.length; i++) {
						var existingField = this.downloadOptions.fields[i];
						if (existingField.name == field.name) {
							existingField.interpolation = field.interpolation;
							found = true;
							break;
						}
					}
					if (!found) {
						this.downloadOptions.fields.push(field);
					}
				} else {
					for (var i = 0; i < this.downloadOptions.fields.length; i++) {
						var existingField = this.downloadOptions.fields[i];
						if (existingField.name == field.name) {
							this.downloadOptions.fields.splice(i, 1);
						}
					}
				}
			},
			showPreviewLayer : function() {

				var isPreviewSupported = function(formats) {
					for (var i = 0; i < formats.length; i++) {
						if (formats[i] == "image/png") {
							return true;
						}
					}
					return false;
				}
				if (isPreviewSupported(this.coverageDescription.supportedFormats)) {
					var GetCoverageOp = oscar.Util.Metadata.getOperation(
							this.capabilities, "GetCoverage");

					//Get the first available field with it's default interpolation method.
                                        var rangeSubset = function(coverage) {
                                            subsetting = coverage.range;
                                            var field = subsetting.fields[0];
                                            var id = field.identifier;
                                            var interpolation = field.interpolationMethods.defaultMethod;
                                            return id + ":" + interpolation;
                                        };

                                        var previewLayer = new oscar.Layer.GetCoveragePreview(
							this.coverageDescription.identifier,
							oscar.PreviewCoverageProxy,
							{
								identifier : this.coverageDescription.identifier,
								serviceEndpoint : GetCoverageOp.dcp.http.get,
								rangeSubset : rangeSubset(this.coverageDescription),
								version : "1.1.0"
							}, {
								isBaseLayer : false,
								singleTile : true
							});
					previewLayer.events
							.on({
								"loadstart" : function() {
									try {
										this.map
												.getControlsByClass("oscar.Control.LoadingBar")[0]
												.show();
									} catch (err) {
									}
								},
								"loadend" : function() {
									try {
										this.map
												.getControlsByClass("oscar.Control.LoadingBar")[0]
												.hide();
									} catch (err) {
									}
								},
								scope : this

							});
					this.map.addLayer(previewLayer);
					var exitMode_function = function(evt) {
						this.map.removeLayer(previewLayer);
						this.events.un({
							"exitMode" : exitMode_function
						});
					};
					this.events.on({
						"exitMode" : exitMode_function,
						scope : this
					});
				} else {
					this.showPolyPreviewLayer();
				}
			},
			showPolyPreviewLayer : function() {
				var styleMap = new OpenLayers.StyleMap({
					"default" : new OpenLayers.Style({
						fillColor : "#0f0f0f",
						fillOpacity : 0.01,
						strokeWidth : 0,
						strokeColor : "#0f0f0f",
						strokeOpacity : 0.3
					})
				});
				var poly_layer = new OpenLayers.Layer.Vector(
						"wcs-polygon-preview", {
							styleMap : styleMap,
							renderers : [ 'Canvas', 'VML' ],
							wrapDateLine : true,
							projection : map.getProjectionObject()
						});
				this.map.addLayer(poly_layer);

				var t = this.record.feature.clone();
				poly_layer.addFeatures(t);

				var exitMode_function = function(evt) {
					this.map.removeLayer(poly_layer);
					this.events.un({
						"exitMode" : exitMode_function
					});
				};

				this.events.on({
					"exitMode" : exitMode_function,
					scope : this
				});
			},
			CLASS_NAME : "oscar.Util.Plugin.Download.WCSService"
		});

oscar.getPluginManager().register(
		oscar.Util.Plugin.Download.WCSService.prototype.pluginType,
		oscar.Util.Plugin.Download.WCSService);
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Util.Plugin.Download.Zipfile = new oscar.BaseClass(oscar.Util.Plugin.Download, {
    pluginType : "application/zip",
    icon : "ui-icon-oux-direct-download",
    initialize : function(options) {
        oscar.Util.Plugin.Download.prototype.initialize.apply(this, [ options ]);

    },

    /**
     * @Override
     * @see oscar.Util.Plugin
     */
    play : function() {
        var elemIF = document.createElement("iframe");
        elemIF.src = this.link.url;
        elemIF.style.display = "none";
        document.body.appendChild(elemIF);
    },
    CLASS_NAME : "oscar.Util.Plugin.Download.Zipfile"
});

oscar.getPluginManager().register(oscar.Util.Plugin.Download.Zipfile.prototype.pluginType, oscar.Util.Plugin.Download.Zipfile);
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Lang
 * 
 * This class will provide additional functionality to the OpenLayers.Lang
 * class. It will be used as an entry point for products to include their own
 * translation definitions.
 */

oscar.Lang = {
    /**
     * Method: addTranslation
     * 
     * This method will append a key/value pair to a Language object and return
     * the object. If the Language object does not exist it will create one.
     * 
     * Parameters: code - {String} key - {String} value - {String}
     * 
     * Returns: library - {Object} The dictionary object.
     */
    addTranslation : function(code, key, value) {
        var library = oscar.Lang.getDictionary(code);
        if (key && value) {
            library[key] = value;
        }
        return library;
    },

    /**
     * Method: getDictionary Get the dictionary.
     * 
     * Parameters: code - {String}
     * 
     */
    getDictionary : function(code) {
        var dictionary = OpenLayers.Lang[code];
        if (!dictionary) {
            OpenLayers.Lang[code] = {};
        }
        return OpenLayers.Lang[code];
    },
    /**
     * Constant: CLASS_NAME - oscar.Lang
     */
    CLASS_NAME : "oscar.Lang"
};
/**
 * API Method: oscar.addi18n Alias for <oscar.Lang.addTranslation> Parameters:
 * 
 * code - {String} The dictionary to add the key / value pair to. key - {String}
 * The key for an i18n string value in the dictionary. value - {String} The
 * value for the key.
 * 
 * Returns: {Object} The dictionary object.
 */
oscar.addi18n = oscar.Lang.addTranslation;
oscar.i18n = OpenLayers.i18n;

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 * 
 * Copyright 2014 CARIS <http://www.caris.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Namespace: oscar.Lang.en Dictionary for English.
 */
OpenLayers.Util.applyDefaults(oscar.Lang.getDictionary("en"), {
    "alertBoxHeader" : "Warning!",
    
    "confirmBoxHeader" : "Warning!",
    
    "yesButtonLabel" : "Yes",
    
    "cancelButtonLabel" : "Cancel",
    
    "okayButtonLabel" : "Okay",
    
    "confirmDeleteMessage" : "Are you sure you want to delete this item?",
    
    "confirmRemoveMessage" : "Are you sure you want to remove this item?",
    
    "oneChecked" : "At least one item must be checked.",
    
    "nameColumnLabel" : "Name",
    
    "statusColumnLabel" : "Status",
    
    "typeColumnLabel" : "Type",
    
    "valueColumnLabel" : "Value",
    
    "resetColumnLabel" : "Reset",
    
    "modified" : "Modified",
    
    "default" : "Default",
    
    "confirmReset" : "Reset value to default?",
    
    "saveButtonLabel" : "Save",
    
    "Themes" : "&nbsp;",
    
    "Basic" : "Basic",
    
    "Advanced" : "Advanced",
    
    "ExtractDataDlgHeader" : "Extract Data",
    
    "Coverages" : "Coverages",
    
    "Custom" : "Custom",
    
    "Current View" : "Current View",
    
    "Features" : "Features",
    
    "Fields" : "Fields",
    
    "Format Type" : "Format Type",
    
    "Layers" : "Layers",
    
    "Data" : "Data",
    
    "Download" : "Download",
    
    "Format" : "Format",
    
    "Help" : "Help",
    
    "Close" : "Close",
    
    "Loading" : "Loading",
    
    "Capabilities" : "Capabilities",
    
    "Output" : "Output",
    
    "Please Wait" : "Please wait ...",
    
    "Retrieving Features" : "Retrieving Features",
    
    "Service" : "Service",
    
    "Title" : "Title",
    
    "Units" : "Units",
    
    "english" : "Imperial",
    
    "metric" : "Metric",
    
    "geographic" : "Geographic",
    
    "Area" : "Area",
    
    "Distance" : "Distance",
    
    "Range" : "Range",
    
    "Point" : "Point",
    
    "MeasurementToole" : "Measurement Tool",
    
    "srsCodeColumnLabel" : "Code",
    
    "srsDescriptionColumnLabel" : "Description",
    
    "first" : "&lt;&lt;",
    
    "last" : "&gt;&gt;",
    
    "previous" : "&lt;",
    
    "next" : "&gt;",
    
    "MapInformationDlgHeader" : "Map Information",
    
    "servicepanel_info_header" : "Service Information",
    
    "servicepanel_content_header" : "Service Contents",
    
    "md_abstract_label" : "Abstract",
    
    "md_keywords_label" : "Keywords",
    
    "md_resources_label" : "Resources",
    
    "md_service_info" : "Service Information",
    
    "md_service_Info" : "Service Information",
    
    "md_service_contactInfo" : "Contact Information",
    
    "md_service_ContactInfo" : "Contact Information",
    
    "md_contact_caption" : "Contact Information",
    
    "md_contact_Caption" : "Contact Information",
    
    "md_contact_providerName" : "Provider Name",
    
    "md_contact_ProviderName" : "Provider Name",
    
    "md_contact_providerSite" : "Provider Site",
    
    "md_contact_ProviderSite" : "Provider Site",
    
    "md_contact_individualName" : "Individual Name",
    
    "md_contact_IndividualName" : "Individual Name",
    
    "md_contact_person" : "Individual Name",
    
    "md_contact_Person" : "Individual Name",
    
    "md_contact_organization" : "Organization",
    
    "md_contact_Organization" : "Organization",
    
    "md_contact_positionName" : "Position Name",
    
    "md_contact_PositionName" : "Position Name",
    
    "md_contact_position" : "Position Name",
    
    "md_contact_Position" : "Position Name",
    
    "md_contact_type" : "Type",
    
    "md_contact_Type" : "Type",
    
    "md_contact_address" : "Address",
    
    "md_contact_Address" : "Address",
    
    "md_contact_deliveryPoint" : "Address",
    
    "md_contact_DeliveryPoint" : "Address",
    
    "md_contact_city" : "City",
    
    "md_contact_City" : "City",
    
    "md_contact_stateOrProvince" : "Province/State",
    
    "md_contact_StateOrProvince" : "Province/State",
    
    "md_contact_administrativeArea" : "Province/State",
    
    "md_contact_AdministrativeArea" : "Province/State",
    
    "md_contact_postcode" : "Zip / Postal Code",
    
    "md_contact_Postcode" : "Zip / Postal Code",
    
    "md_contact_postalCode" : "Zip / Postal Code",
    
    "md_contact_PostalCode" : "Zip / Postal Code",
    
    "md_contact_country" : "Country",
    
    "md_contact_Country" : "Country",
    
    "md_contact_phone" : "Phone",
    
    "md_contact_Phone" : "Phone",
    
    "md_contact_voice" : "Phone",
    
    "md_contact_Voice" : "Phone",
    
    "md_contact_facsimile" : "Fax",
    
    "md_contact_Facsimile" : "Fax",
    
    "md_contact_fax" : "Fax",
    
    "md_contact_Fax" : "Fax",
    
    "md_contact_email" : "Email",
    
    "md_contact_Email" : "Email",
    
    "md_contact_electronicMailAddress" : "Email",
    
    "md_contact_ElectronicMailAddress" : "Email",
    
    "md_contact_info_unavailable" : "Contact information unavailable",
    
    "md_contact_Info_Unavailable" : "Contact information unavailable",
    
    "md_contact_onlineResource" : "Online Resource",
    
    "md_legend_graphic" : "Legend",
    
    "md_request_failed" : "Service Unavailable",
    
    "map.information.url.column.type" : "Type",
    
    "map.information.url.column.format" : "Format",
    
    "map.information.url.column.onlineresource" : "Online Resource",
    
    "map.information.no.records.found" : "No records found.",
    
    "mict_clearFilter_icon" : "Clear Filter",
    
    "mict_reset_icon" : "Reset",
    
    "mict_autocomplete_label" : "Search",
    
    "mict_checkAll_label" : "Select All",
    
    "mdec_coverage_type" : "Coverages",
    
    "mdec_feature_type" : "Feature Types",
    
    "mdec_data_connections" : "Data Connections",
    
    "mdec_data_connection_none" : "None",
    
    "oscar_control_select" : "Select areas of interest.",
    
    "openlayers_control_navigation" : "Use the SHIFT key for zoom activation.",
    
    "oscar_control_extractor" : "Download areas of interest.",
    
    "NoDataAvailable" : "No Data Available",
    
    "NotAvailable" : "Not Available",
    
    "tooltip_oscar.control.previousview" : "Previous View",
    
    "tooltip_oscar.control.measure" : "Measurement Tools",
    
    "tooltip_oscar.control.select" : "Selection Tools",
    
    "tooltip_oscar.control.dataextractor" : "Data Download Tool",
    
    "MICT_MSG_EMPTY" : "No records found.",
    
    "MICT_MSG_ERROR" : "Data error.",
    
    "MICT_MSG_LOADING" : "Loading...",
    
    "MICT_MSG_SORTASC" : "Click to sort ascending",
    
    "MICT_MSG_SORTDESC" : "Click to sort descending",
    
    "LayerToggleToolTip" : "Turn layers on or off",
    
    "LayerToggleTitle" : "Layers",
    
    "ThemeSwitcherToolTip" : "Switch between themes",
    
    "ThemeSwitcherTitle" : "Themes",
    
    "SelectionError" : "Selection Error",
    
    "ProcessingInfo" : "Processing ...",
    
    "metadataKeywordLabel" : "Keyword",
    
    "metadataVocabularyLabel" : "Vocabulary",
    
    "DataDiscoverySearchToolTip" : "Search for data in the map",
    
    "DataDiscoverySearchTitle" : "Download",
    
    "Search" : "Search",
    
    "DownloadOptions" : "Options",
    
    "DownloadQueue" : "Queue",
    
    "Crop" : "Crop",
    
    "AddToQueue" : "Queue",
    
    "units:meters" : "metres",
    
    "resolution-x" : "Resolution x",
    
    "resolution-y" : "Resolution y",
    
    "resolutionTooHigh" : "The Resolution value is too high, please adjust the value.",
    
    "resolutionTooLow" : "The Resolution value is too low, please adjust the value.",
    
    "MoveAllToTheLeft" : "Move all items to the left.",
    
    "MoveAllToTheRight" : "Move all items to the right.",
    
    "showThemeSwitcher" : "Show Theme Switcher",
    
    "dCoverage" : "Download Coverage",
    
    "dMetadata" : "Download Metadata (ISO)",
    
    "dGML" : "Download Metadata (GML)",
    
    "save-metadata" : "Download Metadata",
    
    "view-metadata" : "View Metadata",
    
    "OGC:WCS-1.1.0-http-get-capabilities" : "Coverage",
    
    "simple-search-placeholder" : "Enter Search. Use !e for enhanced mode.",
    
    "Coarse" : "Coarse",
    
    "Fine" : "Fine",
    
    "Syntax Help":"Syntax Help"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 * 
 * Copyright 2014 CARIS <http://www.caris.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Namespace: oscar.Lang.es Dictionary for Spanish.
 */
OpenLayers.Util.applyDefaults(oscar.Lang.getDictionary("es"), {
    "alertBoxHeader" : "Advertencia!",
    
    "confirmBoxHeader" : "Advertencia!",
    
    "yesButtonLabel" : "S\u00ed",
    
    "cancelButtonLabel" : "Cancelar",
    
    "okayButtonLabel" : "Aceptar",
    
    "confirmDeleteMessage" : "\u017cEst\u00e1 seguro que quiere suprimir este elemento?",
    
    "confirmRemoveMessage" : "\u017cEst\u00e1 seguro que quiere eliminar este elemento?",
    
    "oneChecked" : "Al menos un elemento debe ser comprobado.",
    
    "nameColumnLabel" : "Nombre",
    
    "statusColumnLabel" : "Estado",
    
    "typeColumnLabel" : "Tipo",
    
    "valueColumnLabel" : "Valor",
    
    "resetColumnLabel" : "Restituir",
    
    "modified" : "Modificado",
    
    "default" : "Por defecto",
    
    "confirmReset" : "\u00bfRestablecer el valor por defecto?",
    
    "saveButtonLabel" : "Guardar",
    
    "Themes" : "&nbsp;",
    
    "Basic" : "B\u00e1sico",
    
    "Advanced" : "Avanzado",
    
    "Coverages" : "Coberturas",
    
    "ExtractDataDlgHeader" : "Extracci\u00f3n de datos",
    
    "Data" : "datos",
    
    "Custom" : "Personalizar",
    
    "Current View" : "Vista actual",
    
    "Features" : "Objetos",
    
    "Fields" : "Campos",
    
    "Format Type" : "Tipo de formato",
    
    "Layers" : "Capas",
    
    "Download" : "Descargar",
    
    "Format" : "Formato",
    
    "Help" : "Ayuda",
    
    "Close" : "Cerrar",
    
    "Loading" : "Carga",
    
    "Capabilities" : "Capacidades",
    
    "Output" : "Salida",
    
    "Please Wait" : "Espere por favor...",
    
    "Retrieving Features" : "Recuperaci\u00f3n de objetos ",
    
    "Service" : "Servicio",
    
    "Units" : "Unidades",
    
    "english" : "Imperial",
    
    "metric" : "M\u00e9trico",
    
    "geographic" : "Geogr\u00e1fico",
    
    "Area" : "\u00c1rea",
    
    "Title" : "T\u00edtulo",
    
    "Distance" : "Distancia",
    
    "Range" : "Rango",
    
    "Point" : "Punto",
    
    "MeasurementToole" : "Herramienta de medici\u00f3n",
    
    "srsCodeColumnLabel" : "C\u00f3digo",
    
    "srsDescriptionColumnLabel" : "Descripci\u00f3n",
    
    "first" : "&lt;&lt;",
    
    "last" : "&gt;&gt;",
    
    "previous" : "&lt;",
    
    "next" : "&gt;",
    
    "MapInformationDlgHeader" : "Informati\u00f3n de la carta",
    
    "servicepanel_info_header" : "Informaci\u00f3n del Servicio",
    
    "servicepanel_content_header" : "Contenido del Servicio",
    
    "md_abstract_label" : "Resumen",
    
    "md_keywords_label" : "Palabras claves",
    
    "md_resources_label" : "Recursos",
    
    "md_service_info" : "Informaci\u00f3n del Servicio",
    
    "md_service_Info" : "Informaci\u00f3n del Servicio",
    
    "md_service_contactInfo" : "Informaci\u00f3n de Contacto",
    
    "md_service_ContactInfo" : "Informaci\u00f3n de Contacto",
    
    "md_contact_caption" : "Informaci\u00f3n de Contacto",
    
    "md_contact_Caption" : "Informaci\u00f3n de Contacto",
    
    "md_contact_providerName" : "Nombre del proveedor",
    
    "md_contact_ProviderName" : "Nombre del proveedor",
    
    "md_contact_providerSite" : "Proveedor del Sitio",
    
    "md_contact_ProviderSite" : "Proveedor del Sitio",
    
    "md_contact_individualName" : "Nombre de la persona",
    
    "md_contact_IndividualName" : "Nombre de la persona",
    
    "md_contact_person" : "Nombre de la persona",
    
    "md_contact_Person" : "Nombre de la persona",
    
    "md_contact_organization" : "Organizaci\u00f3n",
    
    "md_contact_Organization" : "Organizaci\u00f3n",
    
    "md_contact_positionName" : "Nombre de Posici\u00f3n",
    
    "md_contact_PositionName" : "Nombre de Posici\u00f3n",
    
    "md_contact_position" : "Nombre de Posici\u00f3n",
    
    "md_contact_Position" : "Nombre de Posici\u00f3n",
    
    "md_contact_type" : "Tipo",
    
    "md_contact_Type" : "Tipo",
    
    "md_contact_address" : "Domicilio",
    
    "md_contact_Address" : "Domicilio",
    
    "md_contact_deliveryPoint" : "Domicilio",
    
    "md_contact_DeliveryPoint" : "Domicilio",
    
    "md_contact_city" : "Ciudad",
    
    "md_contact_City" : "Ciudad",
    
    "md_contact_stateOrProvince" : "Provincia o Estado",
    
    "md_contact_StateOrProvince" : "Provincia o Estado",
    
    "md_contact_administrativeArea" : "Provincia o Estado",
    
    "md_contact_AdministrativeArea" : "Provincia o Estado",
    
    "md_contact_postcode" : "C\u00f3digo postal o zip",
    
    "md_contact_Postcode" : "C\u00f3digo postal o zip",
    
    "md_contact_postalCode" : "C\u00f3digo postal o zip",
    
    "md_contact_PostalCode" : "C\u00f3digo postal o zip",
    
    "md_contact_country" : "Pa\u00eds",
    
    "md_contact_Country" : "Pa\u00eds",
    
    "md_contact_phone" : "Tel\u00e9fono",
    
    "md_contact_Phone" : "Tel\u00e9fono",
    
    "md_contact_voice" : "Tel\u00e9fono",
    
    "md_contact_Voice" : "Tel\u00e9fono",
    
    "md_contact_fax" : "Fax",
    
    "md_contact_Fax" : "Fax",
    
    "md_contact_facsimile" : "Fax",
    
    "md_contact_Facsimile" : "Fax",
    
    "md_contact_email" : "Correo electr\u00f3nico",
    
    "md_contact_Email" : "Correo electr\u00f3nico",
    
    "md_contact_electronicMailAddress" : "Correo electr\u00f3nico",
    
    "md_contact_ElectronicMailAddress" : "Correo electr\u00f3nico",
    
    "md_contact_info_unavailable" : "Informaci\u00f3n de contacto no disponible",
    
    "md_contact_Info_Unavailable" : "Informaci\u00f3n de contacto no disponible",
    
    "md_contact_onlineResource" : "Recurso en L\u00ednea",
    
    "md_legend_graphic" : "Leyenda",
    
    "md_request_failed" : "Service Unavailable",
    
    "map.information.url.column.type" : "Tipo",
    
    "map.information.url.column.format" : "Formato",
    
    "map.information.url.column.onlineresource" : "Recursos en l\u00ednea",
    
    "map.information.no.records.found" : "No se encontraron registros.",
    
    "mict_clearFilter_icon" : "Borrar filtro",
    
    "mict_reset_icon" : "Restituir",
    
    "mict_autocomplete_label" : "Buscar",
    
    "mict_checkAll_label" : "Seleccionar todo",
    
    "mdec_coverage_type" : "Coberturas",
    
    "mdec_feature_type" : "Tipos de Objetos",
    
    "mdec_data_connections" : "Conexi\u00f3n de datos",
    
    "mdec_data_connection_none" : "Ninguno",
    
    "oscar_control_select" : "Selecionar \u00e1reas de inter\u00e9s",
    
    "openlayers_control_navigation" : "{translate} - Use the SHIFT key for zoom activation.",
    
    "oscar_control_extractor" : "Descargar \u00e1reas de inter\u00e9s",
    
    "NoDataAvailable" : "No hay datos disponibles",
    
    "NotAvailable" : "No disponible",
    
    "tooltip_oscar.control.previousview" : "Vista Anterior",
    
    "tooltip_oscar.control.measure" : "Herramientas de Medici\u00f3n",
    
    "tooltip_oscar.control.select" : "Herramientas de Selecci\u00f3n",
    
    "tooltip_oscar.control.dataextractor" : "Herramienta de Descarga de Datos",
    
    "MICT_MSG_EMPTY" : "No se encontraron registros.",
    
    "MICT_MSG_ERROR" : "Error de datos.",
    
    "MICT_MSG_LOADING" : "Cargando ...",
    
    "MICT_MSG_SORTASC" : "Haga clic para para organizar orden ascendente",
    
    "MICT_MSG_SORTDESC" : "Haga clic para organizar en orden descendente",
    
    "LayerToggleToolTip" : "Capas Activar o desactivar",
    
    "LayerToggleTitle" : "Capas",
    
    "ThemeSwitcherToolTip" : "Cambiar de un tema a otro",
    
    "ThemeSwitcherTitle" : "Temas",
    
    "SelectionError" : "Error en Selecci\u00f3n",
    
    "ProcessingInfo" : "Procesando ...",
    
    "DownloadManagerTitle" : "Administrador de descargas",
    
    "DownloadManagerToolTip" : "Realiza un seguimiento de las descargas de wcs.",
    
    "metadataKeywordLabel" : "Palabras clave",
    
    "metadataVocabularyLabel" : "Vocabulario",
    
    "DataDiscoverySearchToolTip" : "Buscar datos en el mapa",
    
    "DataDiscoverySearchTitle" : "Descarga de datos",
    
    "Search" : "Buscar",
    
    "DownloadOptions" : "Opciones de descarga",
    
    "DownloadQueue" : "Cola de descarga",
    
    "Crop" : "Recortar",
    
    "AddToQueue" : "Cola de descarga",
    
    "units:meters" : "metros",
    
    "resolution-x" : "Resoluci\u00f3n x",
    
    "resolution-y" : "Resoluci\u00f3n y",
    
    "resolutionTooHigh" : "El valor de la resoluci\u00f3n es muy alto, por favor ajuste el valor",
    
    "resolutionTooLow" : "El valor de la resoluci\u00f3n es muy bajo, por favor ajuste el valor.",
    
    "MoveAllToTheLeft" : "Mover todo hacia la izquierda.",
    
    "MoveAllToTheRight" : "Mover todo hacia la derecha.",
    
    "showThemeSwitcher" : "Mostrar selector de Temas",
    
    "dCoverage" : "Descargar Cobertura",
    
    "dMetadata" : "Descargar Metadatos (ISO)",
    
    "dGML" : "Descargar Metadatos (GML)",
    
    "save-metadata" : "Descargar metadatos",
    
    "view-metadata" : "Ver metadatos",
    
    "OGC:WCS-1.1.0-http-get-capabilities" : "Cobertura",
    
    "simple-search-placeholder" : "Entrar B\u00fasqueda. Utilice !e para el modo mejorado.",
    
    "Coarse" : "Gruesa",
    
    "Fine" : "Fina",
    
    "Syntax Help":"Ayuda para la sintaxis"
});

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 * 
 * Copyright 2014 CARIS <http://www.caris.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Namespace: oscar.Lang.fr Dictionary for French.
 */
OpenLayers.Util.applyDefaults(oscar.Lang.getDictionary("fr"), {
    "alertBoxHeader" : "Attention!",
    
    "confirmBoxHeader" : "Attention!",
    
    "yesButtonLabel" : "Oui",
    
    "cancelButtonLabel" : "Annuler",
    
    "okayButtonLabel" : "Okay",
    
    "confirmDeleteMessage" : "\u0118tes-vous s\u0171r de vouloir supprimer cet \u00e9l\u00e9ment ?",
    
    "confirmRemoveMessage" : "\u0118tes-vous s\u0171r de vouloir \u00e9liminer cet \u00e9l\u00e9ment ?",
    
    "oneChecked" : "Au moins un \u00e9l\u00e9ment doit \u0119tre v\u00e9rifi\u00e9.",
    
    "nameColumnLabel" : "Nom",
    
    "statusColumnLabel" : "\u00c9tat",
    
    "typeColumnLabel" : "Type",
    
    "valueColumnLabel" : "Valeur",
    
    "resetColumnLabel" : "R\u00e9tablir",
    
    "modified" : "Modifi\u00e9",
    
    "default" : "Par d\u00e9faut",
    
    "confirmReset" : "R\u00e9tablir la valeur par d\u00e9faut ?",
    
    "saveButtonLabel" : "Sauvegarder",
    
    "Themes" : "&nbsp;",
    
    "Basic" : "Base",
    
    "Advanced" : "Avanc\u00e9",
    
    "Coverages" : "Couvertures",
    
    "ExtractDataDlgHeader" : "Extraction de donn\u00e9es",
    
    "Custom" : "Personnaliser",
    
    "Current View" : "Vue en cours",
    
    "Features" : "Objets",
    
    "Fields" : "Champs",
    
    "Format Type" : "Type de format",
    
    "Layers" : "Couches",
    
    "Data" : "Donn\u00e9es",
    
    "Download" : "T\u00e9l\u00e9charger",
    
    "Format" : "Format",
    
    "Help" : "Aide",
    
    "Close" : "Fermer",
    
    "Loading" : "Chargement",
    
    "Capabilities" : "Capacit\u00e9s",
    
    "Output" : "Sortie",
    
    "Please Wait" : "Attendez s'il vous pla\u00eet...",
    
    "Retrieving Features" : "R\u00e9cup\u00e9ration des objets",
    
    "Service" : "Service",
    
    "Units" : "Unit\u00e9s",
    
    "english" : "Imp\u00e9riale",
    
    "metric" : "M\u00e9trique",
    
    "geographic" : "G\u00e9ographique",
    
    "Area" : "Aire",
    
    "Distance" : "Distance",
    
    "Range" : "P\u00e9rim\u00e8tre",
    
    "Point" : "Point",
    
    "Title" : "Titre",
    
    "MeasurementToole" : "Outil de mesure",
    
    "srsCodeColumnLabel" : "Code",
    
    "srsDescriptionColumnLabel" : "Description",
    
    "first" : "&lt;&lt;",
    
    "last" : "&gt;&gt;",
    
    "previous" : "&lt;",
    
    "next" : "&gt;",
    
    "MapInformationDlgHeader" : "Informations de carte",
    
    "servicepanel_info_header" : "Information de Service",
    
    "servicepanel_content_header" : "Contenu du Service",
    
    "md_abstract_label" : "R\u00e9sum\u00e9",
    
    "md_keywords_label" : "Mots-cl\u00e9s",
    
    "md_resources_label" : "Ressources",
    
    "md_service_info" : "Information de Service",
    
    "md_service_Info" : "Information de Service",
    
    "md_service_contactInfo" : "Coordonn\u00e9es",
    
    "md_service_ContactInfo" : "Coordonn\u00e9es",
    
    "md_contact_caption" : "Coordonn\u00e9es",
    
    "md_contact_Caption" : "Coordonn\u00e9es",
    
    "md_contact_providerName" : "Nom du fournisseur",
    
    "md_contact_ProviderName" : "Nom du fournisseur",
    
    "md_contact_providerSite" : "Fournisseur de site",
    
    "md_contact_ProviderSite" : "Fournisseur de site",
    
    "md_contact_individualName" : "Nom de la personne",
    
    "md_contact_IndividualName" : "Nom de la personne",
    
    "md_contact_person" : "Nom de la personne",
    
    "md_contact_Person" : "Nom de la personne",
    
    "md_contact_organization" : "Organisation",
    
    "md_contact_Organization" : "Organisation",
    
    "md_contact_positionName" : "Nom de Poste",
    
    "md_contact_PositionName" : "Nom de Poste",
    
    "md_contact_position" : "Poste",
    
    "md_contact_Position" : "Poste",
    
    "md_contact_type" : "Type",
    
    "md_contact_Type" : "Type",
    
    "md_contact_address" : "Adresse",
    
    "md_contact_Address" : "Adresse",
    
    "md_contact_deliveryPoint" : "Adresse",
    
    "md_contact_DeliveryPoint" : "Adresse",
    
    "md_contact_city" : "Ville",
    
    "md_contact_City" : "Ville",
    
    "md_contact_stateOrProvince" : "Province ou \u00c9tat",
    
    "md_contact_StateOrProvince" : "Province ou \u00c9tat",
    
    "md_contact_administrativeArea" : "Province ou \u00c9tat",
    
    "md_contact_AdministrativeArea" : "Province ou \u00c9tat",
    
    "md_contact_postcode" : "Code postal ou zip",
    
    "md_contact_Postcode" : "Code postal ou zip",
    
    "md_contact_postalCode" : "Code postal ou zip",
    
    "md_contact_PostalCode" : "Code postal ou zip",
    
    "md_contact_country" : "Pays",
    
    "md_contact_Country" : "Pays",
    
    "md_contact_phone" : "Num\u00e9ro de t\u00e9l\u00e9phone",
    
    "md_contact_Phone" : "Num\u00e9ro de t\u00e9l\u00e9phone",
    
    "md_contact_voice" : "Num\u00e9ro de t\u00e9l\u00e9phone",
    
    "md_contact_Voice" : "Num\u00e9ro de t\u00e9l\u00e9phone",
    
    "md_contact_fax" : "Fax",
    
    "md_contact_Fax" : "Fax",
    
    "md_contact_facsimile" : "Fax",
    
    "md_contact_Facsimile" : "Fax",
    
    "md_contact_email" : "Courrier \u00e9lectronique",
    
    "md_contact_Email" : "Courrier \u00e9lectronique",
    
    "md_contact_electronicMailAddress" : "Courrier \u00e9lectronique",
    
    "md_contact_ElectronicMailAddress" : "Courrier \u00e9lectronique",
    
    "md_contact_info_unavailable" : "Information n'est pas disponible",
    
    "md_contact_Info_Unavailable" : "Information n'est pas disponible",
    
    "md_contact_onlineResource" : "Ressource en Ligne",
    
    "md_legend_graphic" : "L\u00e9gende",
    
    "md_request_failed" : "Service Unavailable",
    
    "map.information.url.column.type" : "Type",
    
    "map.information.url.column.format" : "Format",
    
    "map.information.url.column.onlineresource" : "Ressources en ligne",
    
    "map.information.no.records.found" : "Aucun document n'a \u00e9t\u00e9 trouv\u00e9.",
    
    "mict_clearFilter_icon" : "Effacer le filtre",
    
    "mict_reset_icon" : "R\u00e9tablir",
    
    "mict_autocomplete_label" : "Chercher",
    
    "mict_checkAll_label" : "S\u00e9lectionner Tout",
    
    "mdec_coverage_type" : "Couvertures",
    
    "mdec_feature_type" : "Types d\u2019Objets",
    
    "mdec_data_connections" : "Connexions de donn\u00e9e",
    
    "mdec_data_connection_none" : "Aucun",
    
    "oscar_control_select" : "S\u00e9lectionner les zones d'int\u00e9r\u00eat",
    
    "openlayers_control_navigation" : "{translate} - Use the SHIFT key for zoom activation.",
    
    "oscar_control_extractor" : "T\u00e9l\u00e9charger les zones d'int\u00e9r\u00eat",
    
    "NoDataAvailable" : "Aucune donn\u00e9e n'est disponible",
    
    "NotAvailable" : "Non disponible",
    
    "tooltip_oscar.control.previousview" : "Affichage Pr\u00e9c\u00e9dent",
    
    "tooltip_oscar.control.measure" : "Outils de Mesure",
    
    "tooltip_oscar.control.select" : "Outils de S\u00e9lection",
    
    "tooltip_oscar.control.dataextractor" : "Outil de T\u00e9l\u00e9chargement de Donn\u00e9es",
    
    "MICT_MSG_EMPTY" : "Aucun r\u00e9sultat trouv\u00e9.",
    
    "MICT_MSG_ERROR" : "Erreur de donn\u00e9es.",
    
    "MICT_MSG_LOADING" : "Chargement en cours ...",
    
    "MICT_MSG_SORTASC" : "Cliquez ici pour organiser en ordre croissant",
    
    "MICT_MSG_SORTDESC" : "Cliquez ici pour organiser en ordre d\u00e9croissant",
    
    "LayerToggleToolTip" : "Couches Activer ou d\u00e9sactiver",
    
    "LayerToggleTitle" : "Couches",
    
    "ThemeSwitcherToolTip" : "Basculer entre les th\u00e8mes",
    
    "ThemeSwitcherTitle" : "Th\u00e8mes",
    
    "SelectionError" : "Erreur dans la S\u00e9lection",
    
    "ProcessingInfo" : "Traitement en cours ...",
    
    "DownloadManagerTitle" : "Gestionnaire de t\u00e9l\u00e9chargement",
    
    "DownloadManagerToolTip" : "Recherche de donn\u00c3\u00a9es dans la carte",
    
    "metadataKeywordLabel" : "Mot-cl\u00e9",
    
    "metadataVocabularyLabel" : "Vocabulaire",
    
    "DataDiscoverySearchToolTip" : "Recherche de donn\u00e9es dans la carte",
    
    "DataDiscoverySearchTitle" : "T\u00e9l\u00e9chargement des donn\u00e9es",
    
    "Search" : "Chercher",
    
    "DownloadOptions" : "Options de t\u00e9l\u00e9chargement",
    
    "DownloadQueue" : "File d'attente de t\u00e9l\u00e9chargement",
    
    "Crop" : "Rognage",
    
    "AddToQueue" : "File d'attente de t\u00e9l\u00e9chargement",
    
    "units:meters" : "m\u00e8tres",
    
    "resolution-x" : "R\u00e9solution x",
    
    "resolution-y" : "R\u00e9solution y",
    
    "resolutionTooHigh" : "La valeur de la r\u00e9solution est trop \u00e9lev\u00e9e, s'il vous pla\u00eet ajustez la valeur.",
    
    "resolutionTooLow" : "La valeur de la r\u00e9solution est trop faible, s'il vous pla\u00eet ajustez la valeur.",
    
    "MoveAllToTheLeft" : "D\u00e9placer tout vers la gauche.",
    
    "MoveAllToTheRight" : "D\u00e9placer tout vers la droite.",
    
    "showThemeSwitcher" : "Afficher s\u00e9lecteur de Th\u00e8mes",
    
    "dCoverage" : "T\u00e9l\u00e9charger Couverture",
    
    "dMetadata" : "T\u00e9l\u00e9charger M\u00e9tadonn\u00e9es (ISO)",
    
    "dGML" : "T\u00e9l\u00e9charger M\u00e9tadonn\u00e9es (GML)",
    
    "save-metadata" : "T\u00e9l\u00e9charger M\u00e9tadonn\u00e9es",
    
    "view-metadata" : "Voir M\u00e9tadonn\u00e9es",
    
    "OGC:WCS-1.1.0-http-get-capabilities" : "Couverture",
    
    "simple-search-placeholder" : "Entrez Recherche. Utilisez !e pour le mode am\u00e9lior\u00e9.",
    
    "Coarse" : "Grossi\u00e8re",
    
    "Fine" : "Fine",
    
    "Syntax Help":"Aide pour la sintaxe"
});

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Namespace definition
 */

oscar.Layer = {}
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Layer.GetCoveragePreview
 * 
 * Inherits from: - <OpenLayers.Layer.WMS>
 */

oscar.Layer.GetCoveragePreview = new oscar.BaseClass(OpenLayers.Layer.WMS, {

    DEFAULT_PARAMS : {
        "service" : "WCS",
        "format" : "image/png",
        "request" : "GetCoverage"
    },

    initialize : function(name, url, params, options) {
        params.format = "image/png";
        OpenLayers.Layer.WMS.prototype.initialize.apply(this, [ name, url, params, options ]);
    },
    getFullRequestString : function(newparams, altUrl) {
        
        var projection = this.map.getProjection();
        
        if (projection == "EPSG:900913") {
            projection = "EPSG:3857";
        }
        
        var resolution = this.map.getResolution();
        
        newparams.GRIDOFFSETS = resolution + "," + (resolution * -1);
        newparams.IDENTIFIER = this.params.IDENTIFIER;
        newparams.SERVICEENDPOINT = this.params.SERVICEENDPOINT;
        newparams.RANGESUBSET = this.params.RANGESUBSET;
        newparams.GRIDBASECRS = oscar.Util.EpsgConversion.epsgToUrn(projection);
        var qString = OpenLayers.Layer.WMS.prototype.getFullRequestString.apply(this, [ newparams, altUrl ]);
        
        return qString;
    },
    CLASS_NAME : "oscar.Layer.GetCoveragePreview"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.ogc = {};
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.ogc.BaseService = new oscar.BaseClass({
    serviceEntry : null,
    autoload : true,
    defaultVersion : "2.0.2",
    capabilities : null,
    initialize : function(serviceEntry, options) {
        this.serviceEntry = serviceEntry;
        OpenLayers.Util.extend(this, options);
        if (this.autoload) {
            this.loadCapabilities();
        }
    },
    /**
     * Method: loadCapabilities
     * 
     * Loads the capabilities of the service.
     */
    loadCapabilities : function() {
        var params = {
            request : "GetCapabilities",
            service : "CSW",
            version : (this.version) ? this.version : this.defaultVersion
        }
        OpenLayers.Request.GET({
            url : this.serviceEntry.url,
            async : false,
            params : params,
            success : function(r) {
                var cswFormat = new oscar.Format.CSW();
                this.capabilities = cswFormat.read(r.responseXML);
            },
            failure : function() {
                console.log("Unable to retreive capabilities from " + this.serviceEntry.url);
            },
            scope : this
        });
    },
    /**
     * APIMethod: getUrl
     * 
     * Parameters:
     * 
     * operation The operation. method The method to get the url for. GET, POST
     * currently supported.
     * 
     * 
     * 
     */
    getUrl : function(operation, method) {
        method = method.toLowerCase();
        var url = "";
        var op = this.capabilities.operationsMetadata[operation];
        switch (method) {
        case "get":
            url = op.dcp.http.get;
            break;
        case "post":
            url = op.dcp.http.post;
            break;
        default:
        }
        return url;
    },
    /**
     * CLASS_NAME
     */
    CLASS_NAME : "oscar.ogc.BaseService"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.ogc.CatalogueService
 * 
 * Used to represent a catalogue service.
 */
oscar.ogc.CatalogueService = new oscar.BaseClass(oscar.ogc.BaseService, {
    defaultSearchFields : [ "csw:AnyText" ],
    GETRECORDBYID : "GetRecordById",
    GETRECORDS : "GetRecords",
    initialize : function(serviceEntry, options) {
        oscar.ogc.BaseService.prototype.initialize.apply(this, [ serviceEntry, options ]);
    },
    /**
     * APIMethod: isSetDefaultOutputSchema
     * 
     * Returns: true or false if there is a default output schema value set.
     */
    isSetDefaultOutputSchema : function() {
        if (this.serviceEntry.attributes && this.serviceEntry.attributes.outputSchema) {
            return true;
        }
    },
    /**
     * APIMethod: getDefaultOutputSchema
     * 
     * Returns: String The default output schema.
     */
    getDefaultOutputSchema : function() {
        if (this.serviceEntry.attributes && this.serviceEntry.attributes.outputSchema) {
            return this.serviceEntry.attributes.outputSchema;
        }
    },
    CLASS_NAME : "oscar.ogc.CatalogueService"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.FramedCloud
 * 
 * Inherits from: - <OpenLayers.Popup.FramedCloud>
 */

oscar.FramedCloud = oscar.BaseClass(OpenLayers.Popup.FramedCloud, {
    /**
     * Property: positionBlocks {Object} Hash of different position blocks,
     * keyed by relativePosition two-character code string (ie "tl", "tr", "bl",
     * "br")
     */
    positionBlocks : {
        "tl" : {
            'offset' : new OpenLayers.Pixel(51, 4),
            'padding' : new OpenLayers.Bounds(8, 40, 8, 9),
            'blocks' : [ { // top-left
                size : new OpenLayers.Size('auto', 'auto'),
                anchor : new OpenLayers.Bounds(0, 51, 22, 0),
                position : new OpenLayers.Pixel(0, 0)
            }, { // top-right
                size : new OpenLayers.Size(22, 'auto'),
                anchor : new OpenLayers.Bounds(null, 50, 0, 0),
                position : new OpenLayers.Pixel(-638, 0)
            }, { // bottom-left
                size : new OpenLayers.Size('auto', 19),
                anchor : new OpenLayers.Bounds(0, 32, 22, null),
                position : new OpenLayers.Pixel(0, -631)
            }, { // bottom-right
                size : new OpenLayers.Size(22, 18),
                anchor : new OpenLayers.Bounds(null, 32, 0, null),
                position : new OpenLayers.Pixel(-638, -632)
            }, { // stem
                size : new OpenLayers.Size(80, 40),
                anchor : new OpenLayers.Bounds(null, 0, 0, null),
                position : new OpenLayers.Pixel(0, -681)
            } ]
        },
        "tr" : {
            'offset' : new OpenLayers.Pixel(-54, 4),
            'padding' : new OpenLayers.Bounds(8, 40, 8, 9),
            'blocks' : [ { // top-left
                size : new OpenLayers.Size('auto', 'auto'),
                anchor : new OpenLayers.Bounds(0, 51, 22, 0),
                position : new OpenLayers.Pixel(0, 0)
            }, { // top-right
                size : new OpenLayers.Size(22, 'auto'),
                anchor : new OpenLayers.Bounds(null, 50, 0, 0),
                position : new OpenLayers.Pixel(-638, 0)
            }, { // bottom-left
                size : new OpenLayers.Size('auto', 19),
                anchor : new OpenLayers.Bounds(0, 32, 22, null),
                position : new OpenLayers.Pixel(0, -631)
            }, { // bottom-right
                size : new OpenLayers.Size(22, 19),
                anchor : new OpenLayers.Bounds(null, 32, 0, null),
                position : new OpenLayers.Pixel(-638, -631)
            }, { // stem
                size : new OpenLayers.Size(80, 40),
                anchor : new OpenLayers.Bounds(0, 0, null, null),
                position : new OpenLayers.Pixel(-215, -681)
            } ]
        },
        "bl" : {
            'offset' : new OpenLayers.Pixel(41, -24),
            'padding' : new OpenLayers.Bounds(8, 9, 8, 40),
            'blocks' : [ { // top-left
                size : new OpenLayers.Size('auto', 'auto'),
                anchor : new OpenLayers.Bounds(0, 21, 22, 32),
                position : new OpenLayers.Pixel(0, 0)
            }, { // top-right
                size : new OpenLayers.Size(22, 'auto'),
                anchor : new OpenLayers.Bounds(null, 21, 0, 32),
                position : new OpenLayers.Pixel(-638, 0)
            }, { // bottom-left
                size : new OpenLayers.Size('auto', 21),
                anchor : new OpenLayers.Bounds(0, 0, 22, null),
                position : new OpenLayers.Pixel(0, -629)
            }, { // bottom-right
                size : new OpenLayers.Size(22, 21),
                anchor : new OpenLayers.Bounds(null, 0, 0, null),
                position : new OpenLayers.Pixel(-638, -629)
            }, { // stem,
                size : new OpenLayers.Size(80, 40),
                anchor : new OpenLayers.Bounds(null, null, 0, 0),
                position : new OpenLayers.Pixel(-100, -674)
            } ]
        },
        "br" : {
            'offset' : new OpenLayers.Pixel(-61, -24),
            'padding' : new OpenLayers.Bounds(8, 9, 8, 40),
            'blocks' : [ { // top-left
                size : new OpenLayers.Size('auto', 'auto'),
                anchor : new OpenLayers.Bounds(0, 21, 22, 32),
                position : new OpenLayers.Pixel(0, 0)
            }, { // top-right
                size : new OpenLayers.Size(22, 'auto'),
                anchor : new OpenLayers.Bounds(null, 21, 0, 32),
                position : new OpenLayers.Pixel(-638, 0)
            }, { // bottom-left
                size : new OpenLayers.Size('auto', 21),
                anchor : new OpenLayers.Bounds(0, 0, 22, null),
                position : new OpenLayers.Pixel(0, -629)
            }, { // bottom-right
                size : new OpenLayers.Size(22, 21),
                anchor : new OpenLayers.Bounds(null, 0, 0, null),
                position : new OpenLayers.Pixel(-638, -629)
            }, { // stem
                size : new OpenLayers.Size(80, 40),
                anchor : new OpenLayers.Bounds(0, null, null, 0),
                position : new OpenLayers.Pixel(-310, -674)
            } ]
        }
    },
    /**
     * Property: imageSize
     */
    imageSize : new OpenLayers.Size(676, 736),
    /**
     * Constructor: oscar.FramedCloud
     * 
     * Parameters: id - {String} a unique id of the popup. If null is passed an
     * identifier will be automatically generated. lonlat - {OpenLayers.LonLat}
     * a longitude and latitude pair. The position on the map the popup will be
     * shown. contentSize - {OpenLayers.Size} a width/height pair size of the
     * content. contentHTML - {String} the HTML content to be displayed inside
     * the popup. anchor - {Object} object to anchor the popup too. closeBox -
     * {Boolean} whether to display a close box inside the popup.
     * closeBoxCallback - {Function} Function to be called on closeBox click.
     */
    initialize : function(id, lonlat, contentSize, contentHTML, anchor, closeBox, closeBoxCallback) {
        OpenLayers.Popup.FramedCloud.prototype.initialize.apply(this, arguments);
        this.imageSrc = oscar.getImagePath() + "cloud_popup_relative_outlined_a.png";
    },
    /**
     * APIMethod: destroy
     */
    destroy : function() {
        OpenLayers.Popup.FramedCloud.prototype.destroy.apply(this, arguments);
    },
    /**
     * Constant: CLASS_NAME - oscar.Popup.FramedCloud
     */
    CLASS_NAME : 'oscar.Popup.FramedCloud'
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * Class: oscar.Map
 * 
 * Inherits from : <OpenLayers.Map>
 * 
 */

oscar.Map = oscar.BaseClass(OpenLayers.Map, {
    /**
     * Property: defaultControls
     * 
     * Contains a list of default controls to be placed on the map.
     */
    defaultControls : {
        Navigation : OpenLayers.Control.Navigation,
        SelectFeature : oscar.Control.SelectFeature
    },
    /**
     * Constructor: oscar.Map
     * 
     * Parameters: div - id of DIV element to contain the map options - {Object}
     * Optional object with properties to tag onto the map.
     */
    initialize : function(div, options) {
        var newArgs = [];
        if (options == null) {
            options = {};
        }
        options.theme = oscar._getScriptLocation() + 'theme/default/style.css';
        newArgs.push(div, options);
        OpenLayers.Map.prototype.initialize.apply(this, newArgs);
        this.addControl(new this.defaultControls.Navigation({
            mouseWheelOptions : {
                interval : 500
            }
        }));
        this.addControl(new this.defaultControls.SelectFeature());

    },
    /**
     * Constant: CLASS_NAME - <oscar.Map>
     */
    CLASS_NAME : "oscar.Map"
});
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * This file contains patches for OpenLayers objects
 */

/**
 * Method: outerBoundaryIs
 * 
 * Reads the outerBoundaryIs node from a GML3 response
 */
OpenLayers.Format.GML.v3.prototype.readers["gml"].outerBoundaryIs = function(node, obj) {
    this.readChildNodes(node, obj);
    obj.outer = obj.components[0];
};

/**
 * Method: LatLongBoundingBox
 * 
 * This method reads the LatLongBoundingBox from a WFS 1.0.0 capabilities file.
 */
OpenLayers.Format.WFSCapabilities.v1_0_0.prototype.readers["wfs"]["LatLongBoundingBox"] = function(node, obj) {
    var boundingBox = {};
    var maxx = node.getAttribute("maxx");
    var maxy = node.getAttribute("maxy");
    var minx = node.getAttribute("minx");
    var miny = node.getAttribute("miny");
    var bounds = new OpenLayers.Bounds(minx, miny, maxx, maxy);
    obj.bounds = bounds;
};

OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.namespaces["gmd"] = "http://www.isotc211.org/2005/gmd";
OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.namespaces["gco"] = "http://www.isotc211.org/2005/gco";
OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.namespaces["gmi"] = "http://www.isotc211.org/2005/gmi";
OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.namespaces["srv"] = "http://www.isotc211.org/2005/srv";
OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.namespaces["dsc"] = "http://www.caris.com/dsc/1.0";
OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.readNode = function(node, obj) {
    if (!obj) {
        obj = {};
    }
    var group = this.readers[node.namespaceURI ? this.namespaceAlias[node.namespaceURI] : this.defaultPrefix];
    if (group) {
        var local = node.localName || node.nodeName.split(":").pop();
        var reader = group[local] || group["*"];
        if (reader) {
            reader.apply(this, [ node, obj ]);
        }
    }
    return obj;
}

OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.nonMappedNodes = function(node) {
    if (!this.errornodes) {
        this.errornodes = [];
    }
    if (node) {
        if ($$.inArray(node.nodeName, this.errornodes) == -1) {
            this.errornodes.push(node.nodeName);
            console.warn(node.parentNode.nodeName + " > " + node.nodeName + " is not mapped.");
        }
    }
}

OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.readers["gmd"] = {
    "MI_Metadata" : function(node, obj) {
        var metadata = {
            type : "MI_Metadata",
            title : [ {
                value : null
            } ]
        };
        metadata["abstract"] = [ "" ];
        this.readChildNodes(node, metadata);
        obj.records.push(metadata);
    },
    "MD_Metadata" : function(node, obj) {
        var metadata = {
            type : "MD_Metadata"
        };
        metadata["abstract"] = [ "" ];
        this.readChildNodes(node, metadata);
        obj.records.push(metadata);
    },
    
    "*" : function(node, obj) {
        var name = node.localName || node.nodeName.split(":").pop();
        obj[name] = {};
        if (node.attributes.length > 0) {
            for (var i = 0; i < node.attributes.length; i++) {
                var attribute = node.attributes[i];
                var attrName = attribute.name;
                obj[name][attrName] = this.getAttributeNS(node, "", attrName);
            }
        }
        this.readChildNodes(node, obj[name]);
        
    },
    
    "fileIdentifier" : function(node, obj) { // contains gco:CharacterString
        // element
        this.readChildNodes(node, obj);
    },
    "identifier" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "language" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    /**
     * LanaguageCode has to attributes: codeList and codeListValue
     */
    "LanguageCode" : function(node, obj) {
        var codeList = this.getAttributeNS(node, "", "codeList");
        var codeListValue = this.getAttributeNS(node, "", "codeListValue");
        obj.language = {};
        obj.language.codeList = codeList;
        obj.language.codeListValue = codeListValue;
    },
    
    "characterSet" : function(node, obj) {
        var CharacterSet = {}
        this.readChildNodes(node, CharacterSet);
        obj.CharacterSet = CharacterSet;
    },
    /**
     * MD_CharacerSetCode has to attributes: codeList and codeListValue
     */
    "MD_CharacterSetCode" : function(node, obj) {
        var codeList = this.getAttributeNS(node, "", "codeList");
        var codeListValue = this.getAttributeNS(node, "", "codeListValue");
        obj.codeList = codeList;
        obj.codeListValue = codeListValue;
    },
    /**
     * MD_ScopeCode has to attributes: codeList and codeListValue
     */
    "hierarchyLevel" : function(node, obj) {
        var ScopeCode = {}
        this.readChildNodes(node, ScopeCode);
        obj.ScopeCode = ScopeCode;
    },
    
    /**
     * MD_ScopeCode has to attributes: codeList and codeListValue
     */
    "MD_ScopeCode" : function(node, obj) {
        var codeList = this.getAttributeNS(node, "", "codeList");
        var codeListValue = this.getAttributeNS(node, "", "codeListValue");
        obj.codeList = codeList;
        obj.codeListValue = codeListValue;
    },
    
    "identificationInfo" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "distributionInfo" : function(node, obj) {
        obj.distributionInfo = {};
        this.readChildNodes(node, obj.distributionInfo);
    },
    
    "MD_Distribution" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    
    "transferOptions" : function(node, obj) {
        if (!obj.transferOptions) {
            obj.transferOptions = [];
        }
        var option = this.readChildNodes(node, {});
        obj.transferOptions.push(option);
    },
    "MD_DigitalTransferOptions" : function(node, obj) {
        obj.digitalTransferOptions
        this.readChildNodes(node, obj);
    },
    "onLine" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "CI_OnlineResource" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "linkage" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    
    "contact" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "dateStamp" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "metadataStandardName" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "metadataStandardVersion" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "referenceSystemInfo" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "date" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "abstract" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "resourceConstraints" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "spatialReferenceType" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    
    "MD_DataIdentification" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "citation" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "CI_Citation" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "title" : function(node, obj) { // contains gco:CharacterString element
        this.readChildNodes(node, obj);
    },
    "extent" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "EX_Extent" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "geographicElement" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "EX_GeographicBoundingBox" : function(node, obj) {
        var bounds = {
            left : 0,
            right : 0,
            bottom : 0,
            top : 0
        };
        this.readChildNodes(node, bounds);
        obj.bounds = new OpenLayers.Bounds(bounds.left, bounds.bottom, bounds.right, bounds.top);
    },
    "westBoundLongitude" : function(node, obj) {
        var num = {};
        this.readChildNodes(node, num);
        obj.left = num.value;
    },
    "eastBoundLongitude" : function(node, obj) {
        var num = {};
        this.readChildNodes(node, num);
        obj.right = num.value;
    },
    "southBoundLatitude" : function(node, obj) {
        var num = {};
        this.readChildNodes(node, num);
        obj.bottom = num.value;
    },
    "northBoundLatitude" : function(node, obj) {
        var num = {};
        this.readChildNodes(node, num);
        obj.top = num.value;
    },
    "URL" : function(node, obj) {
        obj.URL = this.getChildValue(node);
    }
};

OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.readers["gmi"] = OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.readers["gmd"];
OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.readers["srv"] = {
    "coupledResource" : function(node, obj) {
        if (!obj.coupledResources) {
            obj.coupledResources = [];
        }
        var coupledResource = {};
        this.readChildNodes(node, coupledResource);
        obj.coupledResources.push(coupledResource);
    },
    "SV_CoupledResource" : function(node, obj) {
        this.readChildNodes(node, obj);
    },
    "operationName" : function(node, obj) {// gco:CharacterString
        this.readChildNodes(node, obj);
    },
    "operatesOn" : function(node, obj) {
        if (!obj.operatesOn) {
            obj.operatesOn = [];
        }
        var operatesOn = {
            uuidref : this.getAttributeNS(node, "", "uuidref"),
            href : this.getAttributeNS(node, "http://www.w3.org/1999/xlink", "href")
        }

        obj.operatesOn.push(operatesOn);
        
    },
    "identifier" : function(node, obj) {
        this.readChildNodes(node, obj);
    }
};
OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.readers["srv"]["*"] = OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.readers["gmd"]["*"];

OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.readers["dsc"] = {
    "DiscoveryBriefRecord" : function(node, obj) {
        var record = {
            type : "DiscoveryBriefRecord"
        };
        this.readChildNodes(node, record);
        obj.records.push(record);
    },
    "DiscoverySummaryRecord" : function(node, obj) {
        var record = {
            type : "DiscoverySummaryRecord"
        };
        this.readChildNodes(node, record);
        obj.records.push(record);
    },
    "DiscoveryRecord" : function(node, obj) {
        var record = {
            type : "DiscoveryRecord"
        };
        this.readChildNodes(node, record);
        obj.records.push(record);
    },
    "Link" : function(node, obj) {
        if (!obj.links) {
            obj.links = [];
        }
        var link = {
            protocol : "",
            url : ""
        };
        link.protocol = this.getAttributeNS(node, "", "protocol");
        link.url = this.getChildValue(node);
        obj.links.push(link);
    },
    "DataIdentifier" : function(node, obj) {
        obj.dataIdentifier = this.getChildValue(node);
    }
}

OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.readers["gco"] = {
    "CharacterString" : function(node, obj) {
        var localName = node.parentNode.localName;
        obj[localName] = this.getChildValue(node);
    },
    "Decimal" : function(node, num) {
        var dec = this.getChildValue(node);
        num.value = dec;
    },
    "ScopedName" : function(node, obj) {
        obj.scopedName = this.getChildValue(node);
    },
    "*" : function(node, obj) {
        var localName = node.parentNode.localName;
        obj[localName] = this.getChildValue(node);
    }
};

OpenLayers.Format.CSWGetRecordById = function(options) {
    options = OpenLayers.Util.applyDefaults(options, OpenLayers.Format.CSWGetRecordById.DEFAULTS);
    var cls = OpenLayers.Format.CSWGetRecordById["v" + options.version.replace(/\./g, "_")];
    if (!cls) {
        throw "Unsupported CSWGetRecordById version: " + options.version;
    }
    return new cls(options);
}
OpenLayers.Format.CSWGetRecordById.DEFAULTS = {
    "version" : "2.0.2"
}

OpenLayers.Format.CSWGetRecordById.v2_0_2 = OpenLayers.Class(OpenLayers.Format.CSWGetRecords.v2_0_2, {
    CLASS_NAME : "OpenLayers.Format.CSWGetRecordById"
});
OpenLayers.Format.CSWGetRecordById.v2_0_2.prototype.readers["csw"]["GetRecordByIdResponse"] = function(node, obj) {
    obj.records = [];
    this.readChildNodes(node, obj);
    var version = this.getAttributeNS(node, "", "version");
    if (version != "") {
        obj.version = version;
    }
    obj.record = obj.records[0];
    delete obj.records;
};

// XMLHttpRequest.js Copyright (C) 2010 Sergey Ilinsky (http://www.ilinsky.com)
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @requires OpenLayers/Request.js
 */

(function() {
    
    // Save reference to earlier defined object implementation (if any)
    var oXMLHttpRequest = window.XMLHttpRequest;
    
    // Define on browser type
    var bGecko = !!window.controllers, bIE = window.document.all && !window.opera, bIE7 = bIE && window.navigator.userAgent.match(/MSIE 7.0/);
    
    // Enables "XMLHttpRequest()" call next to "new XMLHttpReques()"
    function fXMLHttpRequest() {
        this._object = oXMLHttpRequest && !bIE7 ? new oXMLHttpRequest : new window.ActiveXObject("Microsoft.XMLHTTP");
        this._listeners = [];
    }
    ;
    
    // Constructor
    function cXMLHttpRequest() {
        return new fXMLHttpRequest;
    }
    ;
    cXMLHttpRequest.prototype = fXMLHttpRequest.prototype;
    
    // BUGFIX: Firefox with Firebug installed would break pages if not executed
    if (bGecko && oXMLHttpRequest.wrapped)
        cXMLHttpRequest.wrapped = oXMLHttpRequest.wrapped;
    
    // Constants
    cXMLHttpRequest.UNSENT = 0;
    cXMLHttpRequest.OPENED = 1;
    cXMLHttpRequest.HEADERS_RECEIVED = 2;
    cXMLHttpRequest.LOADING = 3;
    cXMLHttpRequest.DONE = 4;
    
    // Public Properties
    cXMLHttpRequest.prototype.readyState = cXMLHttpRequest.UNSENT;
    cXMLHttpRequest.prototype.responseText = '';
    cXMLHttpRequest.prototype.responseXML = null;
    cXMLHttpRequest.prototype.status = 0;
    cXMLHttpRequest.prototype.statusText = '';
    
    // Priority proposal
    cXMLHttpRequest.prototype.priority = "NORMAL";
    
    // Instance-level Events Handlers
    cXMLHttpRequest.prototype.onreadystatechange = null;
    
    // Class-level Events Handlers
    cXMLHttpRequest.onreadystatechange = null;
    cXMLHttpRequest.onopen = null;
    cXMLHttpRequest.onsend = null;
    cXMLHttpRequest.onabort = null;
    
    // Public Methods
    cXMLHttpRequest.prototype.open = function(sMethod, sUrl, bAsync, sUser, sPassword) {
        // Delete headers, required when object is reused
        delete this._headers;
        
        // When bAsync parameter value is omitted, use true as default
        if (arguments.length < 3)
            bAsync = true;
        
        // Save async parameter for fixing Gecko bug with missing
        // readystatechange in synchronous requests
        this._async = bAsync;
        
        // Set the onreadystatechange handler
        var oRequest = this, nState = this.readyState, fOnUnload;
        
        // BUGFIX: IE - memory leak on page unload (inter-page leak)
        if (bIE && bAsync) {
            fOnUnload = function() {
                if (nState != cXMLHttpRequest.DONE) {
                    fCleanTransport(oRequest);
                    // Safe to abort here since onreadystatechange handler
                    // removed
                    oRequest.abort();
                }
            };
            window.attachEvent("onunload", fOnUnload);
        }
        
        // Add method sniffer
        if (cXMLHttpRequest.onopen)
            cXMLHttpRequest.onopen.apply(this, arguments);
        
        if (arguments.length > 4)
            this._object.open(sMethod, sUrl, bAsync, sUser, sPassword);
        else if (arguments.length > 3)
            this._object.open(sMethod, sUrl, bAsync, sUser);
        else
            this._object.open(sMethod, sUrl, bAsync);
        
        try {
            this._object.responseType = "msxml-document";
        } catch (err) {
            // don't do anything because IE might break
        }
        
        this.readyState = cXMLHttpRequest.OPENED;
        fReadyStateChange(this);
        
        this._object.onreadystatechange = function() {
            if (bGecko && !bAsync)
                return;
            
            // Synchronize state
            oRequest.readyState = oRequest._object.readyState;
            
            //
            fSynchronizeValues(oRequest);
            
            // BUGFIX: Firefox fires unnecessary DONE when aborting
            if (oRequest._aborted) {
                // Reset readyState to UNSENT
                oRequest.readyState = cXMLHttpRequest.UNSENT;
                
                // Return now
                return;
            }
            
            if (oRequest.readyState == cXMLHttpRequest.DONE) {
                // Free up queue
                delete oRequest._data;
                /*
                 * if (bAsync) fQueue_remove(oRequest);
                 */
                //
                fCleanTransport(oRequest);
                // Uncomment this block if you need a fix for IE cache
                /*
                 * // BUGFIX: IE - cache issue if
                 * (!oRequest._object.getResponseHeader("Date")) { // Save
                 * object to cache oRequest._cached = oRequest._object; //
                 * Instantiate a new transport object
                 * cXMLHttpRequest.call(oRequest); // Re-send request if (sUser) {
                 * if (sPassword) oRequest._object.open(sMethod, sUrl, bAsync,
                 * sUser, sPassword); else oRequest._object.open(sMethod, sUrl,
                 * bAsync, sUser); } else oRequest._object.open(sMethod, sUrl,
                 * bAsync);
                 * oRequest._object.setRequestHeader("If-Modified-Since",
                 * oRequest._cached.getResponseHeader("Last-Modified") || new
                 * window.Date(0)); // Copy headers set if (oRequest._headers)
                 * for (var sHeader in oRequest._headers) if (typeof
                 * oRequest._headers[sHeader] == "string") // Some frameworks
                 * prototype objects with functions
                 * oRequest._object.setRequestHeader(sHeader,
                 * oRequest._headers[sHeader]);
                 * 
                 * oRequest._object.onreadystatechange = function() { //
                 * Synchronize state oRequest.readyState =
                 * oRequest._object.readyState;
                 * 
                 * if (oRequest._aborted) { // oRequest.readyState =
                 * cXMLHttpRequest.UNSENT; // Return return; }
                 * 
                 * if (oRequest.readyState == cXMLHttpRequest.DONE) { // Clean
                 * Object fCleanTransport(oRequest); // get cached request if
                 * (oRequest.status == 304) oRequest._object = oRequest._cached; //
                 * delete oRequest._cached; // fSynchronizeValues(oRequest); //
                 * fReadyStateChange(oRequest); // BUGFIX: IE - memory leak in
                 * interrupted if (bIE && bAsync) window.detachEvent("onunload",
                 * fOnUnload); } }; oRequest._object.send(null); // Return now -
                 * wait until re-sent request is finished return; };
                 */
                // BUGFIX: IE - memory leak in interrupted
                if (bIE && bAsync)
                    window.detachEvent("onunload", fOnUnload);
            }
            
            // BUGFIX: Some browsers (Internet Explorer, Gecko) fire OPEN
            // readystate twice
            if (nState != oRequest.readyState)
                fReadyStateChange(oRequest);
            
            nState = oRequest.readyState;
        }
    };
    function fXMLHttpRequest_send(oRequest) {
        oRequest._object.send(oRequest._data);
        
        // BUGFIX: Gecko - missing readystatechange calls in synchronous
        // requests
        if (bGecko && !oRequest._async) {
            oRequest.readyState = cXMLHttpRequest.OPENED;
            
            // Synchronize state
            fSynchronizeValues(oRequest);
            
            // Simulate missing states
            while (oRequest.readyState < cXMLHttpRequest.DONE) {
                oRequest.readyState++;
                fReadyStateChange(oRequest);
                // Check if we are aborted
                if (oRequest._aborted)
                    return;
            }
        }
    }
    ;
    cXMLHttpRequest.prototype.send = function(vData) {
        // Add method sniffer
        if (cXMLHttpRequest.onsend)
            cXMLHttpRequest.onsend.apply(this, arguments);
        
        if (!arguments.length)
            vData = null;
        
        // BUGFIX: Safari - fails sending documents created/modified
        // dynamically, so an explicit serialization required
        // BUGFIX: IE - rewrites any custom mime-type to "text/xml" in case an
        // XMLNode is sent
        // BUGFIX: Gecko - fails sending Element (this is up to the
        // implementation either to standard)
        if (vData && vData.nodeType) {
            vData = window.XMLSerializer ? new window.XMLSerializer().serializeToString(vData) : vData.xml;
            if (!this._headers["Content-Type"])
                this._object.setRequestHeader("Content-Type", "application/xml");
        }
        
        this._data = vData;
        /*
         * // Add to queue if (this._async) fQueue_add(this); else
         */
        fXMLHttpRequest_send(this);
    };
    cXMLHttpRequest.prototype.abort = function() {
        // Add method sniffer
        if (cXMLHttpRequest.onabort)
            cXMLHttpRequest.onabort.apply(this, arguments);
        
        // BUGFIX: Gecko - unnecessary DONE when aborting
        if (this.readyState > cXMLHttpRequest.UNSENT)
            this._aborted = true;
        
        this._object.abort();
        
        // BUGFIX: IE - memory leak
        fCleanTransport(this);
        
        this.readyState = cXMLHttpRequest.UNSENT;
        
        delete this._data;
        /*
         * if (this._async) fQueue_remove(this);
         */
    };
    cXMLHttpRequest.prototype.getAllResponseHeaders = function() {
        return this._object.getAllResponseHeaders();
    };
    cXMLHttpRequest.prototype.getResponseHeader = function(sName) {
        return this._object.getResponseHeader(sName);
    };
    cXMLHttpRequest.prototype.setRequestHeader = function(sName, sValue) {
        // BUGFIX: IE - cache issue
        if (!this._headers)
            this._headers = {};
        this._headers[sName] = sValue;
        
        return this._object.setRequestHeader(sName, sValue);
    };
    
    // EventTarget interface implementation
    cXMLHttpRequest.prototype.addEventListener = function(sName, fHandler, bUseCapture) {
        for (var nIndex = 0, oListener; oListener = this._listeners[nIndex]; nIndex++)
            if (oListener[0] == sName && oListener[1] == fHandler && oListener[2] == bUseCapture)
                return;
        // Add listener
        this._listeners.push([ sName, fHandler, bUseCapture ]);
    };
    
    cXMLHttpRequest.prototype.removeEventListener = function(sName, fHandler, bUseCapture) {
        for (var nIndex = 0, oListener; oListener = this._listeners[nIndex]; nIndex++)
            if (oListener[0] == sName && oListener[1] == fHandler && oListener[2] == bUseCapture)
                break;
        // Remove listener
        if (oListener)
            this._listeners.splice(nIndex, 1);
    };
    
    cXMLHttpRequest.prototype.dispatchEvent = function(oEvent) {
        var oEventPseudo = {
            'type' : oEvent.type,
            'target' : this,
            'currentTarget' : this,
            'eventPhase' : 2,
            'bubbles' : oEvent.bubbles,
            'cancelable' : oEvent.cancelable,
            'timeStamp' : oEvent.timeStamp,
            'stopPropagation' : function() {
            }, // There is no flow
            'preventDefault' : function() {
            }, // There is no default action
            'initEvent' : function() {
            } // Original event object should be initialized
        };
        
        // Execute onreadystatechange
        if (oEventPseudo.type == "readystatechange" && this.onreadystatechange)
            (this.onreadystatechange.handleEvent || this.onreadystatechange).apply(this, [ oEventPseudo ]);
        
        // Execute listeners
        for (var nIndex = 0, oListener; oListener = this._listeners[nIndex]; nIndex++)
            if (oListener[0] == oEventPseudo.type && !oListener[2])
                (oListener[1].handleEvent || oListener[1]).apply(this, [ oEventPseudo ]);
    };
    
    //
    cXMLHttpRequest.prototype.toString = function() {
        return '[' + "object" + ' ' + "XMLHttpRequest" + ']';
    };
    
    cXMLHttpRequest.toString = function() {
        return '[' + "XMLHttpRequest" + ']';
    };
    
    // Helper function
    function fReadyStateChange(oRequest) {
        // Sniffing code
        if (cXMLHttpRequest.onreadystatechange)
            cXMLHttpRequest.onreadystatechange.apply(oRequest);
        
        // Fake event
        oRequest.dispatchEvent({
            'type' : "readystatechange",
            'bubbles' : false,
            'cancelable' : false,
            'timeStamp' : new Date + 0
        });
    }
    ;
    
    function fGetDocument(oRequest) {
        var oDocument = oRequest.responseXML, sResponse = oRequest.responseText;
        // Try parsing responseText
        if (bIE && sResponse && oDocument && !oDocument.documentElement && oRequest.getResponseHeader("Content-Type").match(/[^\/]+\/[^\+]+\+xml/)) {
            oDocument = new window.ActiveXObject("Microsoft.XMLDOM");
            oDocument.async = false;
            oDocument.validateOnParse = false;
            oDocument.loadXML(sResponse);
        }
        // Check if there is no error in document
        if (oDocument)
            if ((bIE && oDocument.parseError != 0) || !oDocument.documentElement || (oDocument.documentElement && oDocument.documentElement.tagName == "parsererror"))
                return null;
        return oDocument;
    }
    ;
    
    function fSynchronizeValues(oRequest) {
        try {
            oRequest.responseText = oRequest._object.responseText;
        } catch (e) {
        }
        try {
            oRequest.responseXML = fGetDocument(oRequest._object);
        } catch (e) {
        }
        try {
            oRequest.status = oRequest._object.status;
        } catch (e) {
        }
        try {
            oRequest.statusText = oRequest._object.statusText;
        } catch (e) {
        }
    }
    ;
    
    function fCleanTransport(oRequest) {
        // BUGFIX: IE - memory leak (on-page leak)
        oRequest._object.onreadystatechange = new window.Function;
    }
    ;
    
    // Internet Explorer 5.0 (missing apply)
    if (!window.Function.prototype.apply) {
        window.Function.prototype.apply = function(oRequest, oArguments) {
            if (!oArguments)
                oArguments = [];
            oRequest.__func = this;
            oRequest.__func(oArguments[0], oArguments[1], oArguments[2], oArguments[3], oArguments[4]);
            delete oRequest.__func;
        };
    }
    ;
    
    // Register new object with window
    /**
     * Class: OpenLayers.Request.XMLHttpRequest Standard-compliant (W3C)
     * cross-browser implementation of the XMLHttpRequest object. From
     * http://code.google.com/p/xmlhttprequest/.
     */
    if (!OpenLayers.Request) {
        /**
         * This allows for OpenLayers/Request.js to be included before or after
         * this script.
         */
        OpenLayers.Request = {};
    }
    OpenLayers.Request.XMLHttpRequest = cXMLHttpRequest;
})();

/**
 * Checking for function availability for IE8
 */
if (!Function.prototype.bind) {
    Function.prototype.bind = function(oThis) {
        if (typeof this !== "function") {
            // closest thing possible to the ECMAScript 5 internal IsCallable
            // function
            throw new TypeError("Function.prototype.bind - what is trying to be bound is not callable");
        }
        
        var aArgs = Array.prototype.slice.call(arguments, 1), fToBind = this, fNOP = function() {
        }, fBound = function() {
            return fToBind.apply(this instanceof fNOP && oThis ? this : oThis, aArgs.concat(Array.prototype.slice.call(arguments)));
        };
        
        fNOP.prototype = this.prototype;
        fBound.prototype = new fNOP();
        
        return fBound;
    };
}
// Override the default value for these OL objects.
OpenLayers.Control.Measure.prototype.geodesic = true;
OpenLayers.Control.ScaleLine.prototype.geodesic = true;

//Adding missing namespace
OpenLayers.Format.CSWGetRecords.v2_0_2.prototype.namespaces.xmlns = "http://www.w3.org/2000/xmlns/";

OpenLayers.Format.CQL = (function() {
    
    var tokens = [
        "PROPERTY", "COMPARISON", "VALUE", "LOGICAL"
    ],

    patterns = {
        PROPERTY: /^[_a-zA-Z:-]*\w*/,
        COMPARISON: /^(=|<>|<=|<|>=|>|LIKE)/i,
        IS_NULL: /^IS NULL/i,
        COMMA: /^,/,
        LOGICAL: /^(AND|OR)/i,
        VALUE: /^(["']).*?\1(?=\s+AND|\s+OR|\s*\)|\s*$)/i,
        LPAREN: /^\(/,
        RPAREN: /^\)/,
        SPATIAL: /^(BBOX|INTERSECTS|DWITHIN|WITHIN|CONTAINS)/i,
        NOT: /^NOT/i,
        BETWEEN: /^BETWEEN/i,
        GEOMETRY: function(text) {
            var type = /^(POINT|LINESTRING|POLYGON|MULTIPOINT|MULTILINESTRING|MULTIPOLYGON|GEOMETRYCOLLECTION)/.exec(text);
            if (type) {
                var len = text.length;
                var idx = text.indexOf("(", type[0].length);
                if (idx > -1) {
                    var depth = 1;
                    while (idx < len && depth > 0) {
                        idx++;
                        switch(text.charAt(idx)) {
                            case '(':
                                depth++;
                                break;
                            case ')':
                                depth--;
                                break;
                            default:
                                // in default case, do nothing
                        }
                    }
                }
                return [text.substr(0, idx+1)];
            }
        },
        END: /^$/
    },

    follows = {
        LPAREN: ['GEOMETRY', 'SPATIAL', 'PROPERTY', 'VALUE', 'LPAREN'],
        RPAREN: ['NOT', 'LOGICAL', 'END', 'RPAREN'],
        PROPERTY: ['COMPARISON', 'BETWEEN', 'COMMA', 'IS_NULL'],
        BETWEEN: ['VALUE'],
        IS_NULL: ['END'],
        COMPARISON: ['VALUE'],
        COMMA: ['GEOMETRY', 'VALUE', 'PROPERTY'],
        VALUE: ['LOGICAL', 'COMMA', 'RPAREN', 'END'],
        SPATIAL: ['LPAREN'],
        LOGICAL: ['NOT', 'VALUE', 'SPATIAL', 'LPAREN', 'PROPERTY'],
        NOT: ['LPAREN', 'PROPERTY'],
        GEOMETRY: ['COMMA', 'RPAREN']
    },

    operators = {
        '=': OpenLayers.Filter.Comparison.EQUAL_TO,
        '<>': OpenLayers.Filter.Comparison.NOT_EQUAL_TO,
        '<': OpenLayers.Filter.Comparison.LESS_THAN,
        '<=': OpenLayers.Filter.Comparison.LESS_THAN_OR_EQUAL_TO,
        '>': OpenLayers.Filter.Comparison.GREATER_THAN,
        '>=': OpenLayers.Filter.Comparison.GREATER_THAN_OR_EQUAL_TO,
        'LIKE': OpenLayers.Filter.Comparison.LIKE,
        'BETWEEN': OpenLayers.Filter.Comparison.BETWEEN,
        'IS NULL': OpenLayers.Filter.Comparison.IS_NULL
    },

    operatorReverse = {},

    logicals = {
        'AND': OpenLayers.Filter.Logical.AND,
        'OR': OpenLayers.Filter.Logical.OR
    },

    logicalReverse = {},

    precedence = {
        'RPAREN': 3,
        'LOGICAL': 2,
        'COMPARISON': 1
    };

    var i;
    for (i in operators) {
        if (operators.hasOwnProperty(i)) {
            operatorReverse[operators[i]] = i;
        }
    }

    for (i in logicals) {
        if (logicals.hasOwnProperty(i)) {
            logicalReverse[logicals[i]] = i;
        }
    }

    function tryToken(text, pattern) {
        if (pattern instanceof RegExp) {
            return pattern.exec(text);
        } else {
            return pattern(text);
        }
    }

    function nextToken(text, tokens) {
        var i, token, len = tokens.length;
        for (i=0; i<len; i++) {
            token = tokens[i];
            var pat = patterns[token];
            var matches = tryToken(text, pat);
            if (matches) {
                var match = matches[0];
                var remainder = text.substr(match.length).replace(/^\s*/, "");
                return {
                    type: token,
                    text: match,
                    remainder: remainder
                };
            }
        }

        var msg = "ERROR: In parsing: [" + text + "], expected one of: ";
        for (i=0; i<len; i++) {
            token = tokens[i];
            msg += "\n    " + token + ": " + patterns[token];
        }

        throw new Error(msg);
    }

    function tokenize(text) {
        var results = [];
        var token, expect = ["NOT", "GEOMETRY", "SPATIAL", "LPAREN", "PROPERTY"];

        do {
            token = nextToken(text, expect);
            text = token.remainder;
            expect = follows[token.type];
            if (token.type != "END" && !expect) {
                throw new Error("No follows list for " + token.type);
            }
            results.push(token);
        } while (token.type != "END");

        return results;
    }

    function buildAst(tokens) {
        var operatorStack = [],
            postfix = [];

        while (tokens.length) {
            var tok = tokens.shift();
            switch (tok.type) {
                case "PROPERTY":
                case "GEOMETRY":
                case "VALUE":
                    postfix.push(tok);
                    break;
                case "COMPARISON":
                case "BETWEEN":
                case "IS_NULL":
                case "LOGICAL":
                    var p = precedence[tok.type];

                    while (operatorStack.length > 0 &&
                        (precedence[operatorStack[operatorStack.length - 1].type] <= p)
                    ) {
                        postfix.push(operatorStack.pop());
                    }

                    operatorStack.push(tok);
                    break;
                case "SPATIAL":
                case "NOT":
                case "LPAREN":
                    operatorStack.push(tok);
                    break;
                case "RPAREN":
                    while (operatorStack.length > 0 &&
                        (operatorStack[operatorStack.length - 1].type != "LPAREN")
                    ) {
                        postfix.push(operatorStack.pop());
                    }
                    operatorStack.pop(); // toss out the LPAREN

                    if (operatorStack.length > 0 &&
                        operatorStack[operatorStack.length-1].type == "SPATIAL") {
                        postfix.push(operatorStack.pop());
                    }
                case "COMMA":
                case "END":
                    break;
                default:
                    throw new Error("Unknown token type " + tok.type);
            }
        }

        while (operatorStack.length > 0) {
            postfix.push(operatorStack.pop());
        }

        function buildTree() {
            var tok = postfix.pop();
            switch (tok.type) {
                case "LOGICAL":
                    var rhs = buildTree(),
                        lhs = buildTree();
                    return new OpenLayers.Filter.Logical({
                        filters: [lhs, rhs],
                        type: logicals[tok.text.toUpperCase()]
                    });
                case "NOT":
                    var operand = buildTree();
                    return new OpenLayers.Filter.Logical({
                        filters: [operand],
                        type: OpenLayers.Filter.Logical.NOT
                    });
                case "BETWEEN":
                    var min, max, property;
                    postfix.pop(); // unneeded AND token here
                    max = buildTree();
                    min = buildTree();
                    property = buildTree();
                    return new OpenLayers.Filter.Comparison({
                        property: property,
                        lowerBoundary: min,
                        upperBoundary: max,
                        type: OpenLayers.Filter.Comparison.BETWEEN
                    });
                case "COMPARISON":
                    var value = buildTree(),
                        property = buildTree();
                    return new OpenLayers.Filter.Comparison({
                        property: property,
                        value: value,
                        type: operators[tok.text.toUpperCase()]
                    });
                case "IS_NULL":
                    var property = buildTree();
                    return new OpenLayers.Filter.Comparison({
                        property: property,
                        type: operators[tok.text.toUpperCase()]
                    });
                case "VALUE":
                    var match = tok.text.match(/^'(.*)'$/);
                    if (match) {
                        return match[1].replace(/''/g, "'");
                    } else {
                        return Number(tok.text);
                    }
                case "SPATIAL":
                    switch(tok.text.toUpperCase()) {
                        case "BBOX":
                            var maxy = buildTree(),
                                maxx = buildTree(),
                                miny = buildTree(),
                                minx = buildTree(),
                                prop = buildTree();

                            return new OpenLayers.Filter.Spatial({
                                type: OpenLayers.Filter.Spatial.BBOX,
                                property: prop,
                                value: OpenLayers.Bounds.fromArray(
                                    [minx, miny, maxx, maxy]
                                )
                            });
                        case "INTERSECTS":
                            var value = buildTree(),
                                property = buildTree();
                            return new OpenLayers.Filter.Spatial({
                                type: OpenLayers.Filter.Spatial.INTERSECTS,
                                property: property,
                                value: value
                            });
                        case "WITHIN":
                            var value = buildTree(),
                                property = buildTree();
                            return new OpenLayers.Filter.Spatial({
                                type: OpenLayers.Filter.Spatial.WITHIN,
                                property: property,
                                value: value
                            });
                        case "CONTAINS":
                            var value = buildTree(),
                                property = buildTree();
                            return new OpenLayers.Filter.Spatial({
                                type: OpenLayers.Filter.Spatial.CONTAINS,
                                property: property,
                                value: value
                            });
                        case "DWITHIN":
                            var distance = buildTree(),
                                value = buildTree(),
                                property = buildTree();
                            return new OpenLayers.Filter.Spatial({
                                type: OpenLayers.Filter.Spatial.DWITHIN,
                                value: value,
                                property: property,
                                distance: Number(distance)
                            });
                    }
                case "GEOMETRY":
                    return OpenLayers.Geometry.fromWKT(tok.text);
                default:
                    return tok.text;
            }
        }

        var result = buildTree();
        if (postfix.length > 0) {
            var msg = "Remaining tokens after building AST: \n";
            for (var i = postfix.length - 1; i >= 0; i--) {
                msg += postfix[i].type + ": " + postfix[i].text + "\n";
            }
            throw new Error(msg);
        }

        return result;
    }

    return OpenLayers.Class(OpenLayers.Format, {
        /**
         * APIMethod: read
         * Generate a filter from a CQL string.
         * Parameters:
         * text - {String} The CQL text.
         *
         * Returns:
         * {<OpenLayers.Filter>} A filter based on the CQL text.
         */
        read: function(text) { 
            var result = buildAst(tokenize(text));
            if (this.keepData) {
                this.data = result;
            }
            return result;
        },

        /**
         * APIMethod: write
         * Convert a filter into a CQL string.
         * Parameters:
         * filter - {<OpenLayers.Filter>} The filter.
         *
         * Returns:
         * {String} A CQL string based on the filter.
         */
        write: function(filter) {
            if (filter instanceof OpenLayers.Geometry) {
                return filter.toString();
            }
            switch (filter.CLASS_NAME) {
                case "OpenLayers.Filter.Spatial":
                    switch(filter.type) {
                        case OpenLayers.Filter.Spatial.BBOX:
                            return "BBOX(" +
                                filter.property + "," +
                                filter.value.toBBOX() +
                                ")";
                        case OpenLayers.Filter.Spatial.DWITHIN:
                            return "DWITHIN(" +
                                filter.property + ", " +
                                this.write(filter.value) + ", " +
                                filter.distance + ")";
                        case OpenLayers.Filter.Spatial.WITHIN:
                            return "WITHIN(" +
                                filter.property + ", " +
                                this.write(filter.value) + ")";
                        case OpenLayers.Filter.Spatial.INTERSECTS:
                            return "INTERSECTS(" +
                                filter.property + ", " +
                                this.write(filter.value) + ")";
                        case OpenLayers.Filter.Spatial.CONTAINS:
                            return "CONTAINS(" +
                                filter.property + ", " +
                                this.write(filter.value) + ")";
                        default:
                            throw new Error("Unknown spatial filter type: " + filter.type);
                    }
                case "OpenLayers.Filter.Logical":
                    if (filter.type == OpenLayers.Filter.Logical.NOT) {
                        // TODO: deal with precedence of logical operators to 
                        // avoid extra parentheses (not urgent)
                        return "NOT (" + this.write(filter.filters[0]) + ")";
                    } else {
                        var res = "(";
                        var first = true;
                        for (var i = 0; i < filter.filters.length; i++) {
                            if (first) {
                                first = false;
                            } else {
                                res += ") " + logicalReverse[filter.type] + " (";
                            }
                            res += this.write(filter.filters[i]);
                        }
                        return res + ")";
                    }
                case "OpenLayers.Filter.Comparison":
                    if (filter.type == OpenLayers.Filter.Comparison.BETWEEN) {
                        return filter.property + " BETWEEN " + 
                            this.write(filter.lowerBoundary) + " AND " + 
                            this.write(filter.upperBoundary);
                    } else {
                        return (filter.value !== null) ? filter.property +
                            " " + operatorReverse[filter.type] + " " + 
                            this.write(filter.value) : filter.property +
                            " " + operatorReverse[filter.type];
                    }
                case undefined:
                    if (typeof filter === "string") {
                        return "'" + filter.replace(/'/g, "''") + "'";
                    } else if (typeof filter === "number") {
                        return String(filter);
                    }
                default:
                    throw new Error("Can't encode: " + filter.CLASS_NAME + " " + filter);
            }
        },

        CLASS_NAME: "OpenLayers.Format.CQL"

    });
})();
/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
oscar.Util.Parser = new oscar.BaseClass({
	EVENT_TYPES:['parseerror','parsesuccess'],
	events:null,
	initialize:function(parser) {
		this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES, false);
		this.parser = parser;
		this.parser.yy.parseError = $$.proxy(this.parseError,this);
	},
	parse:function(str) {
		this.parser.parse(str);
		this.events.triggerEvent('parsesuccess',str);
	},	
	parseError:function(msg,hash) {
		this.events.triggerEvent('parseerror',hash);
		throw new oscar.Util.ParserError('parser error', hash);
	},
	CLASS_NAME:"oscar.Util.Parser"
});

oscar.Util.ParserError = function(msg,hash) {
	this.msg = msg;
	this.hash = hash;

},
oscar.Util.ParserError.prototype = new Error;

/*
 * CARIS oscar - Open Spatial Component ARchitecture
 *
 * Copyright 2014 CARIS <http://www.caris.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/* parser generated by jison 0.4.15 */
/*
  Returns a Parser object of the following structure:

  Parser: {
    yy: {}
  }

  Parser.prototype: {
    yy: {},
    trace: function(),
    symbols_: {associative list: name ==> number},
    terminals_: {associative list: number ==> name},
    productions_: [...],
    performAction: function anonymous(yytext, yyleng, yylineno, yy, yystate, $$, _$),
    table: [...],
    defaultActions: {...},
    parseError: function(str, hash),
    parse: function(input),

    lexer: {
        EOF: 1,
        parseError: function(str, hash),
        setInput: function(input),
        input: function(),
        unput: function(str),
        more: function(),
        less: function(n),
        pastInput: function(),
        upcomingInput: function(),
        showPosition: function(),
        test_match: function(regex_match_array, rule_index),
        next: function(),
        lex: function(),
        begin: function(condition),
        popState: function(),
        _currentRules: function(),
        topState: function(),
        pushState: function(condition),

        options: {
            ranges: boolean           (optional: true ==> token location info will include a .range[] member)
            flex: boolean             (optional: true ==> flex-like lexing behaviour where the rules are tested exhaustively to find the longest match)
            backtrack_lexer: boolean  (optional: true ==> lexer regexes are tested in order and for each matching regex the action code is invoked; the lexer terminates the scan when a token is returned by the action code)
        },

        performAction: function(yy, yy_, $avoiding_name_collisions, YY_START),
        rules: [...],
        conditions: {associative list: name ==> set},
    }
  }


  token location info (@$, _$, etc.): {
    first_line: n,
    last_line: n,
    first_column: n,
    last_column: n,
    range: [start_number, end_number]       (where the numbers are indexes into the input string, regular zero-based)
  }


  the parseError function receives a 'hash' object with these members for lexer and parser errors: {
    text:        (matched text)
    token:       (the produced terminal token, if any)
    line:        (yylineno)
  }
  while parser (grammar) errors will also provide these members, i.e. parser errors deliver a superset of attributes: {
    loc:         (yylloc)
    expected:    (string describing the set of expected tokens)
    recoverable: (boolean: TRUE when the parser has a error recovery rule available for this particular error)
  }
*/
var CQL_Parser = (function(){
var o=function(k,v,o,l){for(o=o||{},l=k.length;l--;o[k[l]]=v);return o},$V0=[1,3],$V1=[1,4],$V2=[1,5],$V3=[1,7],$V4=[1,8],$V5=[5,6,7,18],$V6=[5,18];
var parser = {trace: function trace() { },
yy: {},
symbols_: {"error":2,"START":3,"EXP":4,"EOF":5,"AND":6,"OR":7,"FIELD":8,"LIKE":9,"WORD":10,"EQUALS":11,"LT":12,"LTE":13,"GT":14,"GTE":15,"NEQ":16,"OPEN":17,"CLOSE":18,"ARGS":19,"$accept":0,"$end":1},
terminals_: {2:"error",5:"EOF",6:"AND",7:"OR",8:"FIELD",9:"LIKE",10:"WORD",11:"EQUALS",12:"LT",13:"LTE",14:"GT",15:"GTE",16:"NEQ",17:"OPEN",18:"CLOSE",19:"ARGS"},
productions_: [0,[3,2],[4,3],[4,3],[4,3],[4,3],[4,3],[4,3],[4,3],[4,3],[4,3],[4,3],[4,1]],
performAction: function anonymous(yytext, yyleng, yylineno, yy, yystate /* action[1] */, $$ /* vstack */, _$ /* lstack */) {
/* this == yyval */

var $0 = $$.length - 1;
switch (yystate) {
case 1:
 return $$[$0-1]
break;
case 2:
this.$ = function(obj){ return ($$[$0-2](obj) && $$[$0](obj));};
break;
case 3:
this.$ = function(obj){return ($$[$0-2](obj) || $$[$0](obj));};
break;
case 4:
 this.$ = function(obj){return $$[$0-2](obj) + $$[$0-1](obj) + $$[$0](obj)};
break;
case 5: case 6: case 7: case 8: case 9: case 10:
 this.$ = function(obj){return $$[$0-2](obj)};
break;
case 11:
 this.$ = $$[$0-1]; 
break;
case 12:
 this.$ = function(obj) { return parser.processArgs(obj, $$[$0])(obj); }; 
break;
}
},
table: [{3:1,4:2,8:$V0,17:$V1,19:$V2},{1:[3]},{5:[1,6],6:$V3,7:$V4},{9:[1,9],11:[1,10],12:[1,11],13:[1,12],14:[1,13],15:[1,14],16:[1,15]},{4:16,8:$V0,17:$V1,19:$V2},o($V5,[2,12]),{1:[2,1]},{4:17,8:$V0,17:$V1,19:$V2},{4:18,8:$V0,17:$V1,19:$V2},{10:[1,19]},{10:[1,20]},{10:[1,21]},{10:[1,22]},{10:[1,23]},{10:[1,24]},{10:[1,25]},{6:$V3,7:$V4,18:[1,26]},o($V6,[2,2],{6:$V3,7:$V4}),o($V6,[2,3],{6:$V3,7:$V4}),o($V5,[2,4]),o($V5,[2,5]),o($V5,[2,6]),o($V5,[2,7]),o($V5,[2,8]),o($V5,[2,9]),o($V5,[2,10]),o($V5,[2,11])],
defaultActions: {6:[2,1]},
parseError: function parseError(str, hash) {
    if (hash.recoverable) {
        this.trace(str);
    } else {
        throw new Error(str);
    }
},
parse: function parse(input) {
    var self = this, stack = [0], tstack = [], vstack = [null], lstack = [], table = this.table, yytext = '', yylineno = 0, yyleng = 0, recovering = 0, TERROR = 2, EOF = 1;
    var args = lstack.slice.call(arguments, 1);
    var lexer = Object.create(this.lexer);
    var sharedState = { yy: {} };
    for (var k in this.yy) {
        if (Object.prototype.hasOwnProperty.call(this.yy, k)) {
            sharedState.yy[k] = this.yy[k];
        }
    }
    lexer.setInput(input, sharedState.yy);
    sharedState.yy.lexer = lexer;
    sharedState.yy.parser = this;
    if (typeof lexer.yylloc == 'undefined') {
        lexer.yylloc = {};
    }
    var yyloc = lexer.yylloc;
    lstack.push(yyloc);
    var ranges = lexer.options && lexer.options.ranges;
    if (typeof sharedState.yy.parseError === 'function') {
        this.parseError = sharedState.yy.parseError;
    } else {
        this.parseError = Object.getPrototypeOf(this).parseError;
    }
    function popStack(n) {
        stack.length = stack.length - 2 * n;
        vstack.length = vstack.length - n;
        lstack.length = lstack.length - n;
    }
    _token_stack:
        function lex() {
            var token;
            token = lexer.lex() || EOF;
            if (typeof token !== 'number') {
                token = self.symbols_[token] || token;
            }
            return token;
        }
    var symbol, preErrorSymbol, state, action, a, r, yyval = {}, p, len, newState, expected;
    while (true) {
        state = stack[stack.length - 1];
        if (this.defaultActions[state]) {
            action = this.defaultActions[state];
        } else {
            if (symbol === null || typeof symbol == 'undefined') {
                symbol = lex();
            }
            action = table[state] && table[state][symbol];
        }
                    if (typeof action === 'undefined' || !action.length || !action[0]) {
                var errStr = '';
                expected = [];
                for (p in table[state]) {
                    if (this.terminals_[p] && p > TERROR) {
                        expected.push('\'' + this.terminals_[p] + '\'');
                    }
                }
                if (lexer.showPosition) {
                    errStr = 'Parse error on line ' + (yylineno + 1) + ':\n' + lexer.showPosition() + '\nExpecting ' + expected.join(', ') + ', got \'' + (this.terminals_[symbol] || symbol) + '\'';
                } else {
                    errStr = 'Parse error on line ' + (yylineno + 1) + ': Unexpected ' + (symbol == EOF ? 'end of input' : '\'' + (this.terminals_[symbol] || symbol) + '\'');
                }
                this.parseError(errStr, {
                    text: lexer.match,
                    token: this.terminals_[symbol] || symbol,
                    line: lexer.yylineno,
                    loc: yyloc,
                    expected: expected
                });
            }
        if (action[0] instanceof Array && action.length > 1) {
            throw new Error('Parse Error: multiple actions possible at state: ' + state + ', token: ' + symbol);
        }
        switch (action[0]) {
        case 1:
            stack.push(symbol);
            vstack.push(lexer.yytext);
            lstack.push(lexer.yylloc);
            stack.push(action[1]);
            symbol = null;
            if (!preErrorSymbol) {
                yyleng = lexer.yyleng;
                yytext = lexer.yytext;
                yylineno = lexer.yylineno;
                yyloc = lexer.yylloc;
                if (recovering > 0) {
                    recovering--;
                }
            } else {
                symbol = preErrorSymbol;
                preErrorSymbol = null;
            }
            break;
        case 2:
            len = this.productions_[action[1]][1];
            yyval.$ = vstack[vstack.length - len];
            yyval._$ = {
                first_line: lstack[lstack.length - (len || 1)].first_line,
                last_line: lstack[lstack.length - 1].last_line,
                first_column: lstack[lstack.length - (len || 1)].first_column,
                last_column: lstack[lstack.length - 1].last_column
            };
            if (ranges) {
                yyval._$.range = [
                    lstack[lstack.length - (len || 1)].range[0],
                    lstack[lstack.length - 1].range[1]
                ];
            }
            r = this.performAction.apply(yyval, [
                yytext,
                yyleng,
                yylineno,
                sharedState.yy,
                action[1],
                vstack,
                lstack
            ].concat(args));
            if (typeof r !== 'undefined') {
                return r;
            }
            if (len) {
                stack = stack.slice(0, -1 * len * 2);
                vstack = vstack.slice(0, -1 * len);
                lstack = lstack.slice(0, -1 * len);
            }
            stack.push(this.productions_[action[1]][0]);
            vstack.push(yyval.$);
            lstack.push(yyval._$);
            newState = table[stack[stack.length - 2]][stack[stack.length - 1]];
            stack.push(newState);
            break;
        case 3:
            return true;
        }
    }
    return true;
}};
/* generated by jison-lex 0.3.4 */
var lexer = (function(){
var lexer = ({

EOF:1,

parseError:function parseError(str, hash) {
        if (this.yy.parser) {
            this.yy.parser.parseError(str, hash);
        } else {
            throw new Error(str);
        }
    },

// resets the lexer, sets new input
setInput:function (input, yy) {
        this.yy = yy || this.yy || {};
        this._input = input;
        this._more = this._backtrack = this.done = false;
        this.yylineno = this.yyleng = 0;
        this.yytext = this.matched = this.match = '';
        this.conditionStack = ['INITIAL'];
        this.yylloc = {
            first_line: 1,
            first_column: 0,
            last_line: 1,
            last_column: 0
        };
        if (this.options.ranges) {
            this.yylloc.range = [0,0];
        }
        this.offset = 0;
        return this;
    },

// consumes and returns one char from the input
input:function () {
        var ch = this._input[0];
        this.yytext += ch;
        this.yyleng++;
        this.offset++;
        this.match += ch;
        this.matched += ch;
        var lines = ch.match(/(?:\r\n?|\n).*/g);
        if (lines) {
            this.yylineno++;
            this.yylloc.last_line++;
        } else {
            this.yylloc.last_column++;
        }
        if (this.options.ranges) {
            this.yylloc.range[1]++;
        }

        this._input = this._input.slice(1);
        return ch;
    },

// unshifts one char (or a string) into the input
unput:function (ch) {
        var len = ch.length;
        var lines = ch.split(/(?:\r\n?|\n)/g);

        this._input = ch + this._input;
        this.yytext = this.yytext.substr(0, this.yytext.length - len);
        //this.yyleng -= len;
        this.offset -= len;
        var oldLines = this.match.split(/(?:\r\n?|\n)/g);
        this.match = this.match.substr(0, this.match.length - 1);
        this.matched = this.matched.substr(0, this.matched.length - 1);

        if (lines.length - 1) {
            this.yylineno -= lines.length - 1;
        }
        var r = this.yylloc.range;

        this.yylloc = {
            first_line: this.yylloc.first_line,
            last_line: this.yylineno + 1,
            first_column: this.yylloc.first_column,
            last_column: lines ?
                (lines.length === oldLines.length ? this.yylloc.first_column : 0)
                 + oldLines[oldLines.length - lines.length].length - lines[0].length :
              this.yylloc.first_column - len
        };

        if (this.options.ranges) {
            this.yylloc.range = [r[0], r[0] + this.yyleng - len];
        }
        this.yyleng = this.yytext.length;
        return this;
    },

// When called from action, caches matched text and appends it on next action
more:function () {
        this._more = true;
        return this;
    },

// When called from action, signals the lexer that this rule fails to match the input, so the next matching rule (regex) should be tested instead.
reject:function () {
        if (this.options.backtrack_lexer) {
            this._backtrack = true;
        } else {
            return this.parseError('Lexical error on line ' + (this.yylineno + 1) + '. You can only invoke reject() in the lexer when the lexer is of the backtracking persuasion (options.backtrack_lexer = true).\n' + this.showPosition(), {
                text: "",
                token: null,
                line: this.yylineno
            });

        }
        return this;
    },

// retain first n characters of the match
less:function (n) {
        this.unput(this.match.slice(n));
    },

// displays already matched input, i.e. for error messages
pastInput:function () {
        var past = this.matched.substr(0, this.matched.length - this.match.length);
        return (past.length > 20 ? '...':'') + past.substr(-20).replace(/\n/g, "");
    },

// displays upcoming input, i.e. for error messages
upcomingInput:function () {
        var next = this.match;
        if (next.length < 20) {
            next += this._input.substr(0, 20-next.length);
        }
        return (next.substr(0,20) + (next.length > 20 ? '...' : '')).replace(/\n/g, "");
    },

// displays the character position where the lexing error occurred, i.e. for error messages
showPosition:function () {
        var pre = this.pastInput();
        var c = new Array(pre.length + 1).join("-");
        return pre + this.upcomingInput() + "\n" + c + "^";
    },

// test the lexed token: return FALSE when not a match, otherwise return token
test_match:function (match, indexed_rule) {
        var token,
            lines,
            backup;

        if (this.options.backtrack_lexer) {
            // save context
            backup = {
                yylineno: this.yylineno,
                yylloc: {
                    first_line: this.yylloc.first_line,
                    last_line: this.last_line,
                    first_column: this.yylloc.first_column,
                    last_column: this.yylloc.last_column
                },
                yytext: this.yytext,
                match: this.match,
                matches: this.matches,
                matched: this.matched,
                yyleng: this.yyleng,
                offset: this.offset,
                _more: this._more,
                _input: this._input,
                yy: this.yy,
                conditionStack: this.conditionStack.slice(0),
                done: this.done
            };
            if (this.options.ranges) {
                backup.yylloc.range = this.yylloc.range.slice(0);
            }
        }

        lines = match[0].match(/(?:\r\n?|\n).*/g);
        if (lines) {
            this.yylineno += lines.length;
        }
        this.yylloc = {
            first_line: this.yylloc.last_line,
            last_line: this.yylineno + 1,
            first_column: this.yylloc.last_column,
            last_column: lines ?
                         lines[lines.length - 1].length - lines[lines.length - 1].match(/\r?\n?/)[0].length :
                         this.yylloc.last_column + match[0].length
        };
        this.yytext += match[0];
        this.match += match[0];
        this.matches = match;
        this.yyleng = this.yytext.length;
        if (this.options.ranges) {
            this.yylloc.range = [this.offset, this.offset += this.yyleng];
        }
        this._more = false;
        this._backtrack = false;
        this._input = this._input.slice(match[0].length);
        this.matched += match[0];
        token = this.performAction.call(this, this.yy, this, indexed_rule, this.conditionStack[this.conditionStack.length - 1]);
        if (this.done && this._input) {
            this.done = false;
        }
        if (token) {
            return token;
        } else if (this._backtrack) {
            // recover context
            for (var k in backup) {
                this[k] = backup[k];
            }
            return false; // rule action called reject() implying the next rule should be tested instead.
        }
        return false;
    },

// return next match in input
next:function () {
        if (this.done) {
            return this.EOF;
        }
        if (!this._input) {
            this.done = true;
        }

        var token,
            match,
            tempMatch,
            index;
        if (!this._more) {
            this.yytext = '';
            this.match = '';
        }
        var rules = this._currentRules();
        for (var i = 0; i < rules.length; i++) {
            tempMatch = this._input.match(this.rules[rules[i]]);
            if (tempMatch && (!match || tempMatch[0].length > match[0].length)) {
                match = tempMatch;
                index = i;
                if (this.options.backtrack_lexer) {
                    token = this.test_match(tempMatch, rules[i]);
                    if (token !== false) {
                        return token;
                    } else if (this._backtrack) {
                        match = false;
                        continue; // rule action called reject() implying a rule MISmatch.
                    } else {
                        // else: this is a lexer rule which consumes input without producing a token (e.g. whitespace)
                        return false;
                    }
                } else if (!this.options.flex) {
                    break;
                }
            }
        }
        if (match) {
            token = this.test_match(match, rules[index]);
            if (token !== false) {
                return token;
            }
            // else: this is a lexer rule which consumes input without producing a token (e.g. whitespace)
            return false;
        }
        if (this._input === "") {
            return this.EOF;
        } else {
            return this.parseError('Lexical error on line ' + (this.yylineno + 1) + '. Unrecognized text.\n' + this.showPosition(), {
                text: "",
                token: null,
                line: this.yylineno
            });
        }
    },

// return next match that has a token
lex:function lex() {
        var r = this.next();
        if (r) {
            return r;
        } else {
            return this.lex();
        }
    },

// activates a new lexer condition state (pushes the new lexer condition state onto the condition stack)
begin:function begin(condition) {
        this.conditionStack.push(condition);
    },

// pop the previously active lexer condition state off the condition stack
popState:function popState() {
        var n = this.conditionStack.length - 1;
        if (n > 0) {
            return this.conditionStack.pop();
        } else {
            return this.conditionStack[0];
        }
    },

// produce the lexer rule set which is active for the currently active lexer condition state
_currentRules:function _currentRules() {
        if (this.conditionStack.length && this.conditionStack[this.conditionStack.length - 1]) {
            return this.conditions[this.conditionStack[this.conditionStack.length - 1]].rules;
        } else {
            return this.conditions["INITIAL"].rules;
        }
    },

// return the currently active lexer condition state; when an index argument is provided it produces the N-th previous condition state, if available
topState:function topState(n) {
        n = this.conditionStack.length - 1 - Math.abs(n || 0);
        if (n >= 0) {
            return this.conditionStack[n];
        } else {
            return "INITIAL";
        }
    },

// alias for begin(condition)
pushState:function pushState(condition) {
        this.begin(condition);
    },

// return the number of states currently on the stack
stateStackSize:function stateStackSize() {
        return this.conditionStack.length;
    },
options: {},
performAction: function anonymous(yy,yy_,$avoiding_name_collisions,YY_START) {
var YYSTATE=YY_START;
switch($avoiding_name_collisions) {
case 0:
break;
case 1:return "AND"
break;
case 2:return "OR"
break;
case 3:return "NEQ"
break;
case 4:return "LTE"
break;
case 5:return "GTE"
break;
case 6:return "LT"
break;
case 7:return "GT"
break;
case 8:return "LIKE"
break;
case 9:return "EQUALS"
break;
case 10:return "WORD"
break;
case 11: return "OPEN" 
break;
case 12: return "CLOSE" 
break;
case 13:return 8
break;
case 14:return 8
break;
case 15:return "EOF"
break;
}
},
rules: [/^(?:\s+)/,/^(?:AND\b)/,/^(?:OR\b)/,/^(?:NEQ|<>)/,/^(?:LTE|<=)/,/^(?:GTE|>=)/,/^(?:LT|<)/,/^(?:GT|>)/,/^(?:LIKE|~)/,/^(?:EQUALS|=)/,/^(?:(["']).*?\1(?=\s+AND|\s+OR|\s*\)|\s*$))/,/^(?:\()/,/^(?:\))/,/^(?:^[_a-zA-Z:-]*\w\b)/,/^(?:^[_a-zA-Z]*\w\b)/,/^(?:$)/],
conditions: {"INITIAL":{"rules":[0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15],"inclusive":true}}
});
return lexer;
})();
parser.lexer = lexer;
function Parser () {
  this.yy = {};
}
Parser.prototype = parser;parser.Parser = Parser;
return new Parser;
})();


if (typeof require !== 'undefined' && typeof exports !== 'undefined') {
exports.parser = CQL_Parser;
exports.Parser = CQL_Parser.Parser;
exports.parse = function () { return CQL_Parser.parse.apply(CQL_Parser, arguments); };
exports.main = function commonjsMain(args) {
    if (!args[1]) {
        console.log('Usage: '+args[0]+' FILE');
        process.exit(1);
    }
    var source = require('fs').readFileSync(require('path').normalize(args[1]), "utf8");
    return exports.parser.parse(source);
};
if (typeof module !== 'undefined' && require.main === module) {
  exports.main(process.argv.slice(1));
}
}
