/*
 * Decompiled with CFR 0.152.
 */
package org.wikidata.query.rdf.blazegraph.throttling;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableList;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.Locale;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.LongAdder;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.isomorphism.util.TokenBuckets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wikidata.query.rdf.blazegraph.throttling.BanThrottler;
import org.wikidata.query.rdf.blazegraph.throttling.Bucketing;
import org.wikidata.query.rdf.blazegraph.throttling.RegexpBucketing;
import org.wikidata.query.rdf.blazegraph.throttling.ThrottlingFilterConfig;
import org.wikidata.query.rdf.blazegraph.throttling.ThrottlingMXBean;
import org.wikidata.query.rdf.blazegraph.throttling.ThrottlingState;
import org.wikidata.query.rdf.blazegraph.throttling.TimeAndErrorsThrottler;
import org.wikidata.query.rdf.blazegraph.throttling.UserAgentIpAddressBucketing;

public class ThrottlingFilter
implements Filter,
ThrottlingMXBean {
    private static final Logger log = LoggerFactory.getLogger(ThrottlingFilter.class);
    private boolean enabled;
    private Bucketing userAgentIpBucketing;
    private Bucketing regexBucketing;
    private Bucketing agentBucketing;
    private TimeAndErrorsThrottler<ThrottlingState> timeAndErrorsThrottler;
    private BanThrottler<ThrottlingState> banThrottler;
    private final LongAdder nbThrottledRequests = new LongAdder();
    private final LongAdder nbBannedRequests = new LongAdder();
    @Nullable
    private ObjectName objectName;
    private Cache<Object, ThrottlingState> stateStore;

    public void init(FilterConfig filterConfig) {
        ThrottlingFilterConfig config = new ThrottlingFilterConfig(filterConfig);
        this.enabled = config.isFilterEnabled();
        this.userAgentIpBucketing = new UserAgentIpAddressBucketing();
        this.regexBucketing = new RegexpBucketing(this.loadRegexPatterns(config.getRegexPatternsFile()), r -> r.getParameter("query"));
        this.agentBucketing = new RegexpBucketing(this.loadRegexPatterns(config.getAgentPatternsFile()), r -> r.getHeader("User-Agent"));
        this.stateStore = CacheBuilder.newBuilder().maximumSize((long)config.getMaxStateSize()).expireAfterAccess(config.getStateExpiration().toMillis(), TimeUnit.MILLISECONDS).build();
        Callable<ThrottlingState> stateInitializer = ThrottlingFilter.createThrottlingState(config.getTimeBucketCapacity(), config.getTimeBucketRefillAmount(), config.getTimeBucketRefillPeriod(), config.getErrorBucketCapacity(), config.getErrorBucketRefillAmount(), config.getErrorBucketRefillPeriod(), config.getThrottleBucketCapacity(), config.getThrottleBucketRefillAmount(), config.getThrottleBucketRefillPeriod(), config.getBanDuration());
        this.timeAndErrorsThrottler = new TimeAndErrorsThrottler<ThrottlingState>(config.getRequestDurationThreshold(), stateInitializer, this.stateStore, config.getEnableThrottlingIfHeader(), config.getAlwaysThrottleParam(), Clock.systemUTC());
        this.banThrottler = new BanThrottler<ThrottlingState>(stateInitializer, this.stateStore, config.getEnableBanIfHeader(), config.getAlwaysBanParam(), Clock.systemUTC());
        this.objectName = this.registerMBean(filterConfig.getFilterName());
    }

    private ObjectName registerMBean(String filterName) {
        ObjectName name = null;
        try {
            name = new ObjectName(ThrottlingFilter.class.getName(), "filterName", filterName);
            MBeanServer platformMBeanServer = ManagementFactory.getPlatformMBeanServer();
            platformMBeanServer.registerMBean(this, name);
            log.info("ThrottlingFilter MBean registered as {}.", (Object)name);
        }
        catch (MalformedObjectNameException e) {
            log.error("filter name {} is invalid as an MBean property.", (Object)filterName, (Object)e);
        }
        catch (InstanceAlreadyExistsException e) {
            log.error("MBean for ThrottlingFilter has already been registered.", (Throwable)e);
        }
        catch (MBeanRegistrationException | NotCompliantMBeanException e) {
            log.error("Could not register MBean for ThrottlingFilter.", (Throwable)e);
        }
        return name;
    }

    private static Callable<ThrottlingState> createThrottlingState(Duration timeBucketCapacity, Duration timeBucketRefillAmount, Duration timeBucketRefillPeriod, int errorBucketCapacity, int errorBucketRefillAmount, Duration errorBucketRefillPeriod, int throttleBucketCapacity, int throttleBucketRefillAmount, Duration throttleBucketRefillPeriod, Duration banDuration) {
        return () -> new ThrottlingState(TokenBuckets.builder().withCapacity(timeBucketCapacity.toMillis()).withFixedIntervalRefillStrategy(timeBucketRefillAmount.toMillis(), timeBucketRefillPeriod.toMillis(), TimeUnit.MILLISECONDS).build(), TokenBuckets.builder().withCapacity((long)errorBucketCapacity).withFixedIntervalRefillStrategy((long)errorBucketRefillAmount, errorBucketRefillPeriod.toMillis(), TimeUnit.MILLISECONDS).build(), TokenBuckets.builder().withCapacity((long)throttleBucketCapacity).withFixedIntervalRefillStrategy((long)throttleBucketRefillAmount, throttleBucketRefillPeriod.toMillis(), TimeUnit.MILLISECONDS).build(), banDuration);
    }

    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        Duration throttledDuration;
        Instant bannedUntil;
        HttpServletRequest httpRequest = (HttpServletRequest)request;
        HttpServletResponse httpResponse = (HttpServletResponse)response;
        Object bucket = this.regexBucketing.bucket(httpRequest);
        if (bucket == null) {
            bucket = this.agentBucketing.bucket(httpRequest);
        }
        if (bucket == null) {
            bucket = this.userAgentIpBucketing.bucket(httpRequest);
        }
        if ((bannedUntil = this.banThrottler.throttledUntil(bucket, httpRequest)).isAfter(Instant.now())) {
            log.info("A request is being banned.");
            if (this.enabled) {
                this.nbBannedRequests.increment();
                this.notifyUserBanned(httpResponse, bannedUntil);
                return;
            }
        }
        if (!(throttledDuration = this.timeAndErrorsThrottler.throttledDuration(bucket, httpRequest)).isNegative()) {
            log.info("A request is being throttled.");
            if (this.enabled) {
                this.nbThrottledRequests.increment();
                this.notifyUserThrottled(httpResponse, throttledDuration);
                this.banThrottler.throttled(bucket, httpRequest);
                return;
            }
        }
        Stopwatch stopwatch = Stopwatch.createStarted();
        try {
            chain.doFilter(request, response);
            if (httpResponse.getStatus() < 400) {
                this.timeAndErrorsThrottler.success(bucket, httpRequest, stopwatch.elapsed());
            } else {
                this.timeAndErrorsThrottler.failure(bucket, httpRequest, stopwatch.elapsed());
            }
        }
        catch (IOException | ServletException e) {
            this.timeAndErrorsThrottler.failure(bucket, httpRequest, stopwatch.elapsed());
            throw e;
        }
    }

    private void notifyUserBanned(HttpServletResponse response, Instant bannedUntil) throws IOException {
        response.sendError(403, ThrottlingFilter.formattedBanMessage(bannedUntil));
    }

    @VisibleForTesting
    static String formattedBanMessage(Instant bannedUntil) {
        String banEndTime = DateTimeFormatter.ISO_INSTANT.format(bannedUntil);
        return String.format(Locale.ENGLISH, "You have been banned until %s, please respect throttling and retry-after headers.", banEndTime);
    }

    private void notifyUserThrottled(HttpServletResponse response, Duration duration) throws IOException {
        String retryAfter = Long.toString(duration.getSeconds());
        response.setHeader("Retry-After", retryAfter);
        response.sendError(429, String.format(Locale.ENGLISH, "Too Many Requests - Please retry in %s seconds.", retryAfter));
    }

    public void destroy() {
        if (this.objectName == null) {
            return;
        }
        MBeanServer platformMBeanServer = ManagementFactory.getPlatformMBeanServer();
        try {
            platformMBeanServer.unregisterMBean(this.objectName);
            log.info("ThrottlingFilter MBean {} unregistered.", (Object)this.objectName);
            this.objectName = null;
        }
        catch (InstanceNotFoundException e) {
            log.warn("MBean already unregistered.", (Throwable)e);
        }
        catch (MBeanRegistrationException e) {
            log.error("Could not unregister MBean.", (Throwable)e);
        }
    }

    @Override
    public long getStateSize() {
        return this.stateStore.size();
    }

    @Override
    public long getNumberOfThrottledRequests() {
        return this.nbThrottledRequests.longValue();
    }

    @Override
    public long getNumberOfBannedRequests() {
        return this.nbBannedRequests.longValue();
    }

    private Pattern safeCompile(String line) {
        try {
            return Pattern.compile(line, 32);
        }
        catch (PatternSyntaxException e) {
            log.warn("Invalid pattern: {}", (Object)line);
            return null;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Collection<Pattern> loadRegexPatterns(String patternFilename) {
        try {
            Path patternFile = Paths.get(patternFilename, new String[0]);
            if (!patternFile.toFile().exists()) {
                log.info("Patterns file {} not found, ignoring.", (Object)patternFilename);
                return ImmutableList.of();
            }
            try (Stream<String> lines = Files.lines(patternFile, StandardCharsets.UTF_8);){
                ImmutableList patterns = (ImmutableList)lines.map(this::safeCompile).filter(Objects::nonNull).collect(ImmutableList.toImmutableList());
                log.info("Loaded {} patterns from {}", (Object)patterns.size(), (Object)patternFilename);
                ImmutableList immutableList = patterns;
                return immutableList;
            }
        }
        catch (IOException e) {
            log.warn("Failed reading from patterns file {}.", (Object)patternFilename);
            return ImmutableList.of();
        }
    }
}

