/* exported CONFIG */
var CONFIG = ( function ( window, $ ) {
	'use strict';

	function getUserLanguage() {
		var lang = ( navigator.languages && navigator.languages[0] ) ||
			navigator.language ||
			navigator.userLanguage;

		if ( lang && typeof lang === 'string' ) {
			return lang.split( '-' ).shift();
		}

		return null;
	}

	var root = 'https://query.wikidata.org/';

	var configDeploy = {
		language: getUserLanguage() || 'en',
		api: {
			sparql: {
				uri: '/sparql'
			},
			wikibase: {
				uri: 'https://www.wikidata.org/w/api.php'
			}
		},
		i18nLoad: function( lang ) {
			var loadFallbackLang = null;
			if ( lang !== this.language ) {
				//load default language as fallback language
				loadFallbackLang = $.i18n().load( 'i18n/' + this.language + '.json', this.language );
			}
			return $.when(
					loadFallbackLang,
					$.i18n().load( 'i18n/' + lang + '.json', lang )
				);
		},
		brand: {
			logo: 'logo.svg',
			title: 'Wikidata Query'
		},
		location: {
			root: root,
			index: root
		},
		showBirthdayPresents: new Date().getTime() >= Date.UTC( 2017, 10 + 1, 29 )
	};

	var hostname = window.location.hostname.toLowerCase();

	if ( hostname === '' || hostname === 'localhost' || hostname === '127.0.0.1' ) {

		// Override for local debugging
		return $.extend( true, {}, configDeploy, {
			api: {
				sparql: {
					uri: 'https://query.wikidata.org/sparql'

				}
			},
			i18nLoad: function( lang ) {
				return $.when(
						$.i18n().load( 'i18n/' + lang + '.json', lang ),
						$.i18n().load( 'node_modules/jquery.uls/i18n/' + lang + '.json', lang )
					);
			},
			brand: {
				title: 'Localhost'
			},
			location: {
				root: './',
				index: './index.html'
			},
			showBirthdayPresents: true
		} );
	}

	return configDeploy;

} )( window, jQuery );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};

wikibase.queryService.ui.App = ( function( $, download, window, _, Cookies, moment ) {
	'use strict';

	var SHORTURL_API = '//tinyurl.com/api-create.php?url=',
		RAWGRAPHS_BASE_URL = 'http://wikidata.rawgraphs.io/?url=',
		TRACKING_NAMESPACE = 'wikibase.queryService.ui.app.',
		DEFAULT_QUERY = 'SELECT * WHERE {  SERVICE wikibase:label { bd:serviceParam wikibase:language "[AUTO_LANGUAGE],en". } } LIMIT 100';

	/**
	 * A ui application for the Wikibase query service
	 *
	 * @class wikibase.queryService.ui.App
	 * @license GNU GPL v2+
	 *
	 * @author Stanislav Malyshev
	 * @author Jonas Kress
	 * @constructor
	 *
	 * @param {jQuery} $element
	 * @param {wikibase.queryService.ui.editor.Editor} editor
	 * @param {wikibase.queryService.api.Sparql} queryHelper
	 */
	function SELF( $element, editor, queryHelper, sparqlApi, querySamplesApi ) {
		this._$element = $element;
		this._editor = editor;
		this._queryHelper = queryHelper;
		this._sparqlApi = sparqlApi;
		this._querySamplesApi = querySamplesApi;

		this._init();
	}

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._$element = null;

	/**
	 * @property {wikibase.queryService.ui.ResultView}
	 * @private
	 */
	SELF.prototype._resultView = null;

	/**
	 * @property {wikibase.queryService.api.Sparql}
	 * @private
	 */
	SELF.prototype._sparqlApi = null;

	/**
	 * @property {wikibase.queryService.api.QuerySamples}
	 * @private
	 */
	SELF.prototype._querySamplesApi = null;

	/**
	 * @property {wikibase.queryService.ui.editor.Editor}
	 * @private
	 */
	SELF.prototype._editor = null;

	/**
	 * @property {wikibase.queryService.ui.queryHelper.QueryHelper}
	 * @private
	 */
	SELF.prototype._queryHelper = null;

	/**
	 * @property {boolean}
	 * @private
	 */
	SELF.prototype._isHistoryDisabled = false;

	/**
	 * @property {wikibase.queryService.api.Tracking}
	 * @private
	 */
	SELF.prototype._trackingApi = null;

	/**
	 * @property {boolean}
	 * @private
	 */
	SELF.prototype._hasRunFirstQuery = false;

	/**
	 * Initialize private members and call delegate to specific init methods
	 *
	 * @private
	 */
	SELF.prototype._init = function() {
		if ( !this._sparqlApi ) {
			this._sparqlApi = new wikibase.queryService.api.Sparql();
		}

		if ( !this._resultView ) {
			this._resultView = new wikibase.queryService.ui.ResultView( this._sparqlApi );
		}

		if ( !this._querySamplesApi ) {
			this._querySamplesApi = new wikibase.queryService.api.QuerySamples();
		}

		if ( !this._trackingApi ) {
			this._trackingApi = new wikibase.queryService.api.Tracking();
		}

		if ( !this._editor ) {
			this._editor = new wikibase.queryService.ui.editor.Editor();
		}

		this._track( 'init' );

		this._initApp();
		this._initEditor();
		this._initQueryHelper();
		this._initExamples();
		this._initDataUpdated();
		this._initQuery();
		this._initRdfNamespaces();
		this._initHandlers();
		/*
		// Temporarily disabled as fix for T172728
		// TODO: re-enable once the problems are resolved
		$( window ).scroll( function () {
			var minScroll = $( '#query-box' ).offset().top + $( '#query-box' ).height() - $( window ).height();

			if ( minScroll + 100 < $( this ).scrollTop() && $( this ).scrollTop() > 1 ) {
				if (
					$( '#query-result' ).is( ':visible' ) &&
					$( document ).height() - $( '#query-box' ).height() > $( window ).height()
				) {
					$( '#query-box' ).hide( {
						duration: 500
					} );
				}
			} else {
				$( '#query-box' ).show( {
					duration: 100
				} );
			}
		} ); */
	};

	/**
	 * @private
	 */
	SELF.prototype._initApp = function() {
		// ctr + enter executes query
		$( window ).keydown( function( e ) {
			if ( e.ctrlKey && e.keyCode === 13 ) {
				$( '#execute-button' ).click();
			}
		} );

		// add tooltip to dropdown
		$( '#display-button' ).tooltip();
		$( '#download-button' ).tooltip();
		$( '#link-button' ).tooltip();

		this._actionBar = new wikibase.queryService.ui.toolbar.Actionbar( $( '.action-bar' ) );
	};

	/**
	 * @private
	 */
	SELF.prototype._initEditor = function() {
		var self = this;

		this._editor.fromTextArea( this._$element.find( '.queryEditor' )[0] );

		this._editor.registerCallback( 'change', function( editor, changeObj ) {
			if ( changeObj.text[0] === ':' ) {
				var $help = $( '<a target="_blank" rel="noopener" href="https://www.wikidata.org/wiki/Wikidata:SPARQL_query_service/Wikidata_Query_Help/SPARQL_Editor#Code_Completion">' )
					.append( $.i18n( 'wdqs-app-footer-help' ) );
				self._toast( $help, 'wdqs-app-footer-help' );
			}
		} );

		// if(window.history.pushState) {//this works only in modern browser
		// this._editor.registerCallback( 'change', $.proxy( this._updateQueryUrl, this) );
		// }
	};

	/**
	 * @private
	 */
	SELF.prototype._initQueryHelper = function() {
		var self = this,
			cookieHide = 'query-helper-hide';

		if ( !this._queryHelper ) {
			this._queryHelper = new wikibase.queryService.ui.queryHelper.QueryHelper();
		}
		this._queryHelper.setChangeListener( function( ve ) {
			self._editor.setValue( ve.getQuery() );

			_.debounce( function () {
				self._resultView.drawPreview( ve.getQuery() );
			}, 1000 )();
		} );

		$( '.query-helper' ).resizable( {
			handleSelector: '.splitter',
			resizeHeight: false,
			onDrag: this._updateQueryHelperMinWidth.bind( this ),
			onDragEnd: this._updateQueryEditorSize.bind( this )
		} );

		if ( Cookies.get( cookieHide ) !== 'true' ) {
			$( '.query-helper' ).removeClass( 'query-helper-hidden' );
		}

		if ( this._editor ) {
			this._editor.registerCallback( 'change', _.debounce( function() {
				if ( self._editor.getValue() === self._queryHelper.getQuery() ) {
					return;
				}

				self._drawQueryHelper();
			}, 1500 ) );
		}

		$( '.query-helper' ).bind( 'DOMSubtreeModified', _.debounce( function () {
			self._updateQueryHelperMinWidth();
			self._updateQueryEditorSize();
		}, 100 ) );

		$( '.query-helper .panel-heading .close' ).click( function() {
			Cookies.set( cookieHide, true );
			$( '.query-helper' ).addClass( 'query-helper-hidden' );
			self._updateQueryEditorSize();

			self._track( 'buttonClick.queryHelperTrigger.close' );
			return false;
		} );

		$( '.query-helper-trigger' ).click( function () {
			var visible = $( '.query-helper' ).is( ':visible' );

			$( '.query-helper' ).toggleClass( 'query-helper-hidden', visible );
			Cookies.set( cookieHide, visible );
			self._updateQueryEditorSize();

			self._track( 'buttonClick.queryHelperTrigger.' + ( visible ? 'close' : 'open' ) );

			return false;
		} );

		window.setTimeout( $.proxy( this._drawQueryHelper, this ), 500 );
	};

	/**
	 * @private
	 */
	SELF.prototype._drawQueryHelper = function() {
		try {
			this._queryHelper.setQuery( this._editor.getValue() || DEFAULT_QUERY );
			this._queryHelper.draw( $( '.query-helper .panel-body' ) );
			$( '.query-helper' ).css( 'min-width', '' );
		} catch ( e ) {
			// Temporarily disabled due to T171935
			// TODO: Re-enable when handling of WITH is fixed
			// this._editor.highlightError( e.message );
			window.console.error( e );
		}
	};

	/**
	 * @private
	 */
	SELF.prototype._updateQueryHelperMinWidth = function() {
		var $queryHelper = $( '.query-helper' ),
			$tables = $queryHelper.find( 'table' ),
			tableWidth = _.max( _.map(
				$tables,
				function( e ) {
					return $( e ).width();
				}
			) );
		if ( tableWidth > $queryHelper.width() ) {
			$queryHelper.css(
				'min-width',
				Math.min(
					tableWidth + $tables.offset().left - $queryHelper.offset().left,
					$( window ).width() * 0.5
				)
			);
		}
	};

	/**
	 * @private
	 */
	SELF.prototype._updateQueryEditorSize = function() {
		if ( this._editor ) {
			// set CodeMirror width to container width determined by Flex
			this._editor._editor.setSize( 0, null ); // unset width so container width is unaffected by CodeMirror
			this._editor._editor.setSize( $( '.query-editor-container' ).width(), null ); // set width to container width
		}
	};

	/**
	 * @private
	 */
	SELF.prototype._initExamples = function() {
		var self = this;
		new wikibase.queryService.ui.QueryExampleDialog( $( '#QueryExamples' ),
				this._querySamplesApi, function( query, title ) {
					if ( !query || !query.trim() ) {
						return;
					}

					self._editor.setValue( '#' + title + '\n' + query );

					$( '#QueryExamples' ).one( 'hidden.bs.modal', function() {
						setTimeout( function() { self._editor.focus(); }, 0 );
					} );
				} );
	};

	/**
	 * @private
	 */
	SELF.prototype._initRdfNamespaces = function() {
		var category,
			select,
			ns,
			container = $( '.namespace-shortcuts' ),
			namespaces = wikibase.queryService.RdfNamespaces.NAMESPACE_SHORTCUTS;

		container.click( function( e ) {
			e.stopPropagation();
		} );

		// add namespaces to dropdowns
		for ( category in namespaces ) {
			select = $( '<select>' ).attr( 'class', 'form-control' ).append(
					$( '<option>' ).text( category ) ).appendTo( container );
			for ( ns in namespaces[category] ) {
				select.append(
					$( '<option>' ).text( ns ).attr( 'value', namespaces[category][ns] )
				);
			}
		}
	};

	/**
	 * @private
	 */
	SELF.prototype._initQuery = function() {
		if ( window.location.hash !== '' ) {
			if ( location.hash.indexOf( '#result#' ) === 0 ) {
				location.hash = location.hash.replace( '#result#', '#' );
			}

			this._isHistoryDisabled = true;
			this._editor.setValue( decodeURIComponent( window.location.hash.substr( 1 ) ) );
			this._editor.refresh();
			this._isHistoryDisabled = false;
		}
	};

	/**
	 * @private
	 */
	SELF.prototype._initDataUpdated = function() {
		var self = this,
			$label = $( '.dataUpdated' );

		var updateDataStatus = function() {
			self._sparqlApi.queryDataUpdatedTime().done( function( time, difference ) {
				var labelClass = 'list-group-item-danger';
				if ( difference <= 60 * 2 ) {
					labelClass = 'list-group-item-success';
				} else if ( difference <= 60 * 15 ) {
					labelClass =  'list-group-item-warning';
				}

				$label.html( $( '<a>' ).addClass( 'fa fa-refresh badge ' + labelClass ).html( ' ' ) );
			} );
		};

		updateDataStatus();

		window.setInterval( updateDataStatus, 10 * 60 * 1000 );

		$label.hover( function() {
			updateDataStatus();

			var e = $( this );
			self._sparqlApi.queryDataUpdatedTime().done( function( time, difference ) {
				var text = moment.duration( difference, 'seconds' ).humanize(),
					title = time,
					badge = '<span class="badge">' + text + '</span>';

				$label.attr( 'title', title );
				e.popover( {
					html: true,
					trigger: 'hover',
					placement: 'top',
					content: $.i18n( 'wdqs-app-footer-updated', badge )
				} );
			} ).fail( function() {
				e.popover( {
					content: '[unable to connect]'
				} );
			} );
		}, function() {
			var e = $( this );
			e.popover( 'destroy' );
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._initHandlers = function() {
		var self = this;

		$( '#query-form' ).submit( $.proxy( this._handleQuerySubmit, this ) );
		$( '.namespace-shortcuts' ).on( 'change', 'select',
				$.proxy( this._handleNamespaceSelected, this ) );

		$( '.addPrefixes' ).click( function() {
			var standardPrefixes = wikibase.queryService.RdfNamespaces.STANDARD_PREFIXES,
				prefixes = Object.keys( standardPrefixes ).map( function( x ) {
					return standardPrefixes[x];
				} ).join( '\n' );

			self._editor.prepandValue( prefixes + '\n\n' );
			self._track( 'buttonClick.addPrefixes' );
		} );

		$( '[data-target="#QueryExamples"]' ).click( function() {
			self._track( 'buttonClick.examples' );
		} );

		$( '#clear-button' ).click( function() {
			self._editor.setValue( '' );
			self._track( 'buttonClick.clear' );
		} );

		$( '.explorer-close' ).click( function( e ) {
			e.preventDefault();
			$( '.explorer-panel' ).hide();
		} );

		$( '.restore' ).click( function( e ) {
			self._track( 'buttonClick.restore' );
			e.preventDefault();
			self._editor.restoreValue();
		} );

		$( '.fullscreen' ).click( function( e ) {
			self._track( 'buttonClick.fullscreen' );
			e.preventDefault();
			self._toggleFullscreen();
		} );

		$( window ).on( 'popstate', $.proxy( this._initQuery, this ) );

		this._initPopovers();
		this._initHandlersDownloads();
	};

	/**
	 * @private
	 */
	SELF.prototype._toggleFullscreen = function () {
		if ( document.fullscreenElement || document.mozFullScreenElement ||
				document.webkitIsFullScreen || document.msFullscreenElement ) {
			if ( document.exitFullscreen ) {
				document.exitFullscreen();
			} else if ( document.msExitFullscreen ) {
				document.msExitFullscreen();
			} else if ( document.mozCancelFullScreen ) {
				document.mozCancelFullScreen();
			} else if ( document.webkitExitFullscreen ) {
				document.webkitExitFullscreen();
			}
		} else {
			var el = document.documentElement;

			if ( el.requestFullscreen ) {
				el.requestFullscreen();
			} else if ( el.msRequestFullscreen ) {
				el = document.body; // overwrite the element (for IE)
				el.msRequestFullscreen();
			} else if ( el.mozRequestFullScreen ) {
				el.mozRequestFullScreen();
			} else if ( el.webkitRequestFullscreen ) {
				el.webkitRequestFullScreen();
			}
		}
	};

	/**
	 * @private
	 */
	SELF.prototype._initPopovers = function() {
		var self = this;

		$( '.shortUrlTrigger.query' ).clickover( {
			placement: 'right',
			'global_close': true,
			'html': true,
			'content': function() {
				self._updateQueryUrl();
				return '<iframe ' +
					'class="shortUrl" ' +
					'src="' + SHORTURL_API + encodeURIComponent( window.location ) + '" ' +
					'referrerpolicy="origin" ' +
					'sandbox="" ' +
					'></iframe>';
			}
		} ).click( function() {
			self._track( 'buttonClick.shortUrlQuery' );
		} );

		$( '.shortUrlTrigger.result' ).clickover( {
			placement: 'left',
			'global_close': true,
			'html': true,
			'content': function() {
				self._updateQueryUrl();
				var $link = $( '<a>' ).attr( 'href', 'embed.html' + window.location.hash );
				return '<iframe ' +
					'class="shortUrl" ' +
					'src="' + SHORTURL_API + encodeURIComponent( $link[0].href ) + '" ' +
					'referrerpolicy="origin" ' +
					'sandbox="" ' +
					'></iframe>';
			}
		} ).click( function() {
			self._track( 'buttonClick.shortUrlResult' );
		} );

		$( '.embed.result' ).clickover( {
			placement: 'left',
			'global_close': true,
			'html': true,
			'content': function() {
				self._updateQueryUrl();

				var b = '';
				if ( self._selectedResultBrowser ) {
					b = '#defaultView:' + self._selectedResultBrowser + '\n';
					b = encodeURIComponent( b );
				}
				var $link = $( '<a>' )
					.attr( 'href', 'embed.html#' + b + window.location.hash.substring( 1 ) );
				var $html = $( '<textarea>' ).text(
					'<iframe ' +
						'style="width: 80vw; height: 50vh; border: none;" ' +
						'src="' + $link[0].href + '" ' +
						'referrerpolicy="origin" ' +
						'sandbox="allow-scripts allow-same-origin allow-popups" ' +
						'></iframe>'
				).click( function() {
					$html.select();
				} );

				return $html;
			}
		} ).click( function() {
			self._track( 'buttonClick.embedResult' );
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._initHandlersDownloads = function() {
		var api = this._sparqlApi;
		var DOWNLOAD_FORMATS = {
			'CSV': {
				handler: $.proxy( api.getResultAsCsv, api ),
				mimetype: 'text/csv;charset=utf-8'
			},
			'JSON': {
				handler: $.proxy( api.getResultAsJson, api ),
				mimetype: 'application/json;charset=utf-8'
			},
			'TSV': {
				handler: $.proxy( api.getSparqlTsv, api ),
				mimetype: 'text/tab-separated-values;charset=utf-8'
			},
			'Simple TSV': {
				handler: $.proxy( api.getSimpleTsv, api ),
				mimetype: 'text/tab-separated-values;charset=utf-8',
				ext: 'tsv'
			},
			'Full JSON': {
				handler: $.proxy( api.getResultAsAllJson, api ),
				mimetype: 'application/json;charset=utf-8',
				ext: 'json'
			},
			'SVG': {
				handler: function() {
					var $svg = $( '#query-result svg' );

					if ( !$svg.length ) {
						return null;
					}

					$svg.attr( {
						version: '1.1',
						'xmlns': 'http://www.w3.org/2000/svg',
						'xmlns:svg': 'http://www.w3.org/2000/svg',
						'xmlns:xlink': 'http://www.w3.org/1999/xlink'
					} );

					try {
						return '<?xml version="1.0" encoding="utf-8"?>\n'
							+ window.unescape( encodeURIComponent( $svg[0].outerHTML ) );
					} catch ( ex ) {
						return null;
					}
				},
				mimetype: 'data:image/svg+xml;charset=utf-8',
				ext: 'svg'
			}
		};

		var self = this;
		var downloadHandler = function( filename, handler, mimetype ) {
			return function( e ) {
				e.preventDefault();

				// see: http://danml.com/download.html
				self._track( 'buttonClick.download.' + filename );

				var data = handler();

				if ( data ) {
					download( data, filename, mimetype );
				}
			};
		};

		for ( var format in DOWNLOAD_FORMATS ) {
			var extension = DOWNLOAD_FORMATS[format].ext || format.toLowerCase();
			var formatName = format.replace( /\s/g, '-' );
			$( '#download' + formatName ).click( downloadHandler(
				'query.' + extension,
				DOWNLOAD_FORMATS[format].handler,
				DOWNLOAD_FORMATS[format].mimetype
			) );
		}
	};

	/**
	 * @private
	 */
	SELF.prototype._handleQuerySubmit = function( e ) {
		var self = this;

		this._track( 'buttonClick.execute' );
		if ( !this._hasRunFirstQuery ) {
			this._track( 'firstQuery' );
			this._hasRunFirstQuery = true;
		}

		e.preventDefault();
		this._editor.save();
		this._updateQueryUrl();

		$( '#execute-button' ).prop( 'disabled', true );
		this._resultView.draw( this._editor.getValue() ).fail( function ( error ) {
			self._editor.highlightError( error );
		} ).always( function () {
			$( '#execute-button' ).prop( 'disabled', false );
		} );

		$( '.queryUri' ).attr( 'href', self._sparqlApi.getQueryUri() );
		$( '.rawGraphsUri' ).attr( 'href', RAWGRAPHS_BASE_URL + $( '.queryUri' )[0].href );
	};

	/**
	 * @private
	 */
	SELF.prototype._handleNamespaceSelected = function( e ) {
		var ns,
			uri = e.target.value,
			current = this._editor.getValue();

		if ( current.indexOf( '<' + uri + '>' ) === -1 ) {
			ns = $( e.target ).find( ':selected' ).text();
			this._editor.setValue( 'PREFIX ' + ns + ': <' + uri + '>\n' + current );
		}

		// reselect group label
		e.target.selectedIndex = 0;
	};

	/**
	 * @private
	 */
	SELF.prototype._updateQueryUrl = function() {
		if ( this._isHistoryDisabled ) {
			return;
		}

		var hash = encodeURIComponent( this._editor.getValue() );
		hash = hash.replace( /[!'()*]/g, function( c ) {
			return '%' + c.charCodeAt( 0 ).toString( 16 );
		} );

		if ( window.location.hash !== hash ) {
			if ( window.history.pushState ) {
				window.history.pushState( null, null, '#' + hash );
			} else {
				window.location.hash = hash;
			}
		}
	};

	/**
	 * @private
	 */
	SELF.prototype._toast = function( $el, id ) {
		var cookie = 'hide-toast-' + id;
		if ( Cookies.get( cookie ) ) {
			return;
		}

		$.toast( {
			loader: false,
			stack: 1,
			text: $el[0].outerHTML,
			afterShown: function () {
				$( '.close-jq-toast-single' ).click( function () {
					Cookies.set( cookie, true );
				} );
			}
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._track = function( metricName, value, valueType ) {
		this._trackingApi.track( TRACKING_NAMESPACE + metricName, value, valueType );
	};

	return SELF;

}( jQuery, download, window, _, Cookies, moment ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.toolbar = wikibase.queryService.toolbar || {};

wikibase.queryService.ui.toolbar.Actionbar = ( function( $ ) {
	'use strict';

	/**
	 * An action bar showing actions like loading, rendering and also errors
	 *
	 * @class wikibase.queryService.ui.toolbar.Actionbar
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 *
	 * @param {jQuery} $element
	 */
	function SELF( $element ) {
		this._$element = $element;
	}

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._$element = null;

	/**
	 * Show action bar
	 *
	 * @param {string} messageKey primary message key
	 * @param {string} extraText to be appended to the primary message
	 * @param {string} type of message: primary, success, info, warning, danger
	 * @param {number|boolean} [progress] false if no progress, or actual progress 0-100
	 */
	SELF.prototype.show = function( messageKey, extraText, type, progress ) {
		var text = messageKey;

		if ( $.i18n ) {
			text = $.i18n( messageKey );
		}
		if ( extraText !== '' ) {
			text = text + ': ' + extraText;
		}

		this._$element.find( '.message' ).html( '' );

		if ( !type ) {
			type = 'info';
		}

		if ( progress === undefined || progress === false ) {
			this._$element.find( '.message' ).append( this._getBar( text, type ) );
		} else {
			this._$element.find( '.message' ).append( this._getProgressbar( text, type, progress ) );
		}

		this._$element.show();
	};

	/**
	 * @param {string} text
	 * @param {string} type
	 * @param {number} progress in percent (0 to 100)
	 * @private
	 */
	SELF.prototype._getProgressbar = function( text, type, progress ) {
		return $( '<div class="progress"><div class="progress-bar progress-bar-' + type +
				' progress-bar-striped active" role="progressbar" style="width: ' + progress +
				'%">' + text + '</div></div>' );
	};

	/**
	 * @param {string} text
	 * @param {string} type
	 * @private
	 */
	SELF.prototype._getBar = function( text, type ) {
		return $( '<div class="label label-' + type + '"/>' ).text( text );
	};

	/**
	 * Hide the action bar
	 */
	SELF.prototype.hide = function() {
		this._$element.hide();
	};

	return SELF;

}( jQuery ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.i18n = wikibase.queryService.ui.i18n || {};

wikibase.queryService.ui.i18n.LanguageSelector = ( function( $, wikibase ) {
	'use strict';

	/**
	 * A language selector to selectt the UI language
	 *
	 * @class wikibase.queryService.ui.i18n.LanguageSelector
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 * @param {jQuery} $element
	 * @param {wikibase.queryService.api.Wikibase} api
	 * @param {string} lang default language
	 */
	function SELF( $element, api, lang ) {
		this._$element = $element;

		if ( api ) {
			this._api = api;
		} else {
			this._api = new wikibase.queryService.api.Wikibase();
		}

		this._defaultLanguage = lang;

		this._create();
	}

	/**
	 * @property {wikibase.queryService.api.Wikibase}
	 * @private
	 */
	SELF.prototype._api = null;

	/**
	 * @property {Function}
	 * @private
	 */
	SELF.prototype._$element = null;

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._defaultLanguage = null;

	/**
	 * @property {Function}
	 * @private
	 */
	SELF.prototype._changeListener = null;

	/**
	 * Set the change listener
	 *
	 * @param {Function} listener a function called when value selected
	 */
	SELF.prototype.setChangeListener = function( listener ) {
		this._changeListener = listener;
	};

	/**
	 * @private
	 */
	SELF.prototype._create = function( listener ) {
		var self = this;

		this._$element.text( $.uls.data.getAutonym( this._defaultLanguage ) );
		this._getLanguages().done( function( langs ) {
			self._$element.uls( {
				onSelect: function( lang ) {
					self._$element.text( $.uls.data.getAutonym( lang ) );

					if ( self._changeListener ) {
						self._changeListener( lang );
					}
				},
				languages: langs
			// quickList: langs
			} ).click( function() {
				$( '.uls-menu' ).addClass( 'uls-mobile' ).css( 'left', '' ).css( 'right', '2.5%' );
			} ).css( 'display', 'block' );
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._getLanguages = function() {
		var deferred = $.Deferred();

		this._api.getLanguages().done( function( data ) {

			var langs = {};
			$.each( data.query.languages, function( k, v ) {
				langs[v.code] = v['*'];
			} );

			deferred.resolve( langs );
		} );

		return deferred.promise();
	};

	return SELF;
}( jQuery, wikibase ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.editor = wikibase.queryService.ui.editor || {};
wikibase.queryService.ui.editor.hint = wikibase.queryService.ui.editor.hint || {};

( function( $, wb ) {
	'use strict';

	var MODULE = wb.queryService.ui.editor.hint;

	var SPARQL_KEYWORDS = [
			'SELECT', 'SELECT * WHERE {\n\n}', 'OPTIONAL', 'OPTIONAL {\n\n}', 'WHERE',
			'WHERE {\n\n}', 'ORDER', 'ORDER BY', 'DISTINCT', 'SERVICE',
			'SERVICE wikibase:label { bd:serviceParam wikibase:language "[AUTO_LANGUAGE],en". }', 'BASE',
			'PREFIX', 'REDUCED', 'FROM', 'LIMIT', 'OFFSET', 'HAVING', 'UNION', 'SAMPLE',
			'(SAMPLE() AS )', 'COUNT', '(COUNT() AS )', 'DESC', 'DESC()', 'ASC', 'ASC()',
			'FILTER ()', 'FILTER NOT EXISTS', 'FILTER NOT EXISTS {\n\n}', 'UNION', 'UNION {\n\n}',
			'BIND', 'BIND ()', 'GROUP_CONCAT', '(GROUP_CONCAT() as )', 'ORDER BY',
			'#defaultView:Map', '#defaultView:ImageGrid', '#defaultView:Map', '#defaultView:BubbleChart',
			'#defaultView:TreeMap', '#defaultView:Tree', '#defaultView:Timeline', '#defaultView:Dimensions', '#defaultView:Graph', '#defaultView:LineChart', '#defaultView:BarChart', '#defaultView:ScatterChart', '#defaultView:AreaChart',
			'SERVICE wikibase:around {\n    ?place wdt:P625 ?location.\n    bd:serviceParam wikibase:center ? .\n    bd:serviceParam wikibase:radius ? .\n    bd:serviceParam wikibase:distance ?dist.\n  }',
			'SERVICE wikibase:box {\n    ?place wdt:P625 ?location.\n    bd:serviceParam wikibase:cornerWest ? .\n    bd:serviceParam wikibase:cornerEast ? .\n  }',
			'hint:Query hint:optimizer "None".',
			'#TEMPLATE={ "template": "Textual description of template, referencing ?var", "variables": { "?var": { "query": "SELECT ?id WHERE { ?id wdt:P31 wd:Q146. }" } } }'
	];

	var SPARQL_PREDICATES = [
			// wikibase:
			// property predicates
			'wikibase:rank', 'wikibase:badge', 'wikibase:propertyType', 'wikibase:directClaim',
			'wikibase:claim', 'wikibase:statementProperty', 'wikibase:statementValue',
			'wikibase:qualifier', 'wikibase:qualifierValue', 'wikibase:reference', 'wikibase:referenceValue',
			'wikibase:statementValueNormalized', 'wikibase:qualifierValueNormalized',
			'wikibase:referenceValueNormalized', 'wikibase:novalue',
			// entity types
			'wikibase:Property', // 'wikibase:Item' disabled on WDQS for performance reasons
			// data types
			'wikibase:Reference', 'wikibase:Dump', // 'wikibase:Statement' disabled on WDQS for performance reasons
			// ranks
			'wikibase:PreferredRank', 'wikibase:NormalRank', 'wikibase:DeprecatedRank', 'wikibase:BestRank',
			// value types
			'wikibase:TimeValue', 'wikibase:QuantityValue', 'wikibase:GlobecoordinateValue',
			// property types
			'wikibase:WikibaseItem', 'wikibase:CommonsMedia', 'wikibase:GlobeCoordinate',
			'wikibase:Monolingualtext', 'wikibase:Quantity', 'wikibase:String', 'wikibase:Time',
			'wikibase:Url', 'wikibase:ExternalId', 'wikibase:WikibaseProperty', 'wikibase:Math',
			// pageprops
			'wikibase:statements', 'wikibase:sitelinks',
			// time
			'wikibase:timeValue', 'wikibase:timePrecision', 'wikibase:timeTimezone', 'wikibase:timeCalendarModel',
			// quantity
			'wikibase:quantityAmount', 'wikibase:quantityUpperBound', 'wikibase:quantityLowerBound',
			'wikibase:quantityUnit', 'wikibase:quantityNormalized',
			// coordinate
			'wikibase:geoLatitude', 'wikibase:geoLongitude', 'wikibase:geoPrecision', 'wikibase:geoGlobe',
			// other
			'wikibase:wikiGroup',
			// schema: things
			'schema:about', 'schema:name', 'schema:description', 'schema:dateModified',
			'schema:Article', 'schema:inLanguage', 'schema:isPartOf',
			// rdfs: things
			'rdfs:label', 'rdf:type',
			// skos: things
			'skos:altLabel',
			// xsd:
			'xsd:dateTime', 'xsd:integer', 'xsd:decimal',
			// geo:
			'geo:wktLiteral',
			// owl:
			'owl:sameAs',
			// prov:
			'prov:wasDerivedFrom'

	];

	/**
	 * Code completion for Wikibase entities RDF prefixes in SPARQL completes SPARQL keywords and ?variables
	 *
	 * @class wikibase.queryService.ui.editor.hint.Sparql licence GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 */
	var SELF = MODULE.Sparql = function Sparql() {
	};

	/**
	 * Get list of hints
	 *
	 * @return {jQuery.Promise} Returns the completion as promise ({list:[], from:, to:})
	 */
	SELF.prototype.getHint = function( editorContent, lineContent, lineNum, cursorPos ) {
		var currentWord = this._getCurrentWord( lineContent, cursorPos ),
			hintList = [],
			deferred = new $.Deferred();

		if ( currentWord.word.indexOf( '?' ) === 0 ) {
			hintList = hintList.concat( this._getVariableHints( currentWord.word, this
					._getDefinedVariables( editorContent ) ) );
		}

		hintList = hintList.concat( this._getSPARQLHints( currentWord.word ) );

		if ( hintList.length > 0 ) {
			var hint = this._getHintCompletion( currentWord, hintList, lineNum );
			return deferred.resolve( hint ).promise();
		}

		return deferred.reject().promise();
	};

	SELF.prototype._getSPARQLHints = function( term ) {
		var list = [];

		$.each( SPARQL_KEYWORDS, function( key, keyword ) {
			if ( keyword.toLowerCase().indexOf( term.toLowerCase() ) >= 0 ) {
				list.push( keyword );
			}
		} );

		$.each( SPARQL_PREDICATES, function( key, keyword ) {
			if ( keyword.toLowerCase().indexOf( term.toLowerCase() ) === 0 ) {
				list.push( keyword );
			}
		} );

		return list;
	};

	SELF.prototype._getDefinedVariables = function( text ) {
		var variables = {};

		$.each( text.match( /\?\w+/g ), function( key, word ) {
			variables[ word ] = true;
		} );

		return Object.keys( variables );
	};

	SELF.prototype._getVariableHints = function( term, variables ) {
		var list = [];

		if ( !term || term === '?' ) {
			return variables;
		}

		$.each( variables, function( key, variable ) {
			if ( variable.toLowerCase().indexOf( term.toLowerCase() ) === 0 ) {
				list.push( variable );
			}
		} );

		return list;
	};

	SELF.prototype._getHintCompletion = function( currentWord, list, lineNumber ) {
		var completion = {
			list: []
		};
		completion.from = {
			line: lineNumber,
			char: currentWord.start
		};
		completion.to = {
			line: lineNumber,
			char: currentWord.end
		};
		completion.list = list;

		return completion;
	};

	SELF.prototype._getCurrentWord = function( line, position ) {
		var pos = position - 1;

		if ( pos < 0 ) {
			pos = 0;
		}

		while ( /[\w?#:]/.test( line.charAt( pos ) ) ) {
			pos--;
			if ( pos < 0 ) {
				break;
			}
		}
		var left = pos + 1;

		pos = position;
		while ( /[\w:]/.test( line.charAt( pos ) ) ) {
			pos++;
			if ( pos >= line.length ) {
				break;
			}
		}
		var right = pos;
		var word = line.substring( left, right );
		return {
			word: word,
			start: left,
			end: right
		};
	};

}( jQuery, wikibase ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.editor = wikibase.queryService.ui.editor || {};
wikibase.queryService.ui.editor.hint = wikibase.queryService.ui.editor.hint || {};

( function( $, wikibase ) {
	'use strict';

	var MODULE = wikibase.queryService.ui.editor.hint;

	/**
	 * Code completion for Wikibase entities RDF prefixes in SPARQL completes SPARQL keywords and ?variables
	 *
	 * @class wikibase.queryService.ui.editor.hint.Rdf licence GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @param {wikibase.queryService.api.Wikibase} api
	 * @param {wikibase.queryService.RdfNamespaces} rdfNamespaces
	 * @constructor
	 */
	var SELF = MODULE.Rdf = function( api, rdfNamespaces ) {
		this._api = api;
		this._rdfNamespaces = rdfNamespaces;

		if ( !this._api ) {
			this._api = new wikibase.queryService.api.Wikibase();
		}

		if ( !this._rdfNamespaces ) {
			this._rdfNamespaces = wikibase.queryService.RdfNamespaces;
		}
	};

	/**
	 * @property {wikibase.queryService.RdfNamespaces}
	 * @private
	 */
	SELF.prototype._rdfNamespaces = null;

	/**
	 * @property {wikibase.queryService.api.Wikibase}
	 * @private
	 */
	SELF.prototype._api = null;

	/**
	 * Get list of hints
	 *
	 * @return {jQuery.Promise} Returns the completion as promise ({list:[], from:, to:})
	 */
	SELF.prototype.getHint = function( editorContent, lineContent, lineNum, cursorPos ) {
		var deferred = new $.Deferred(),
			currentWord = this._getCurrentWord( lineContent, cursorPos ),
			list,
			prefix,
			term,
			entityPrefixes,
			self = this;

		if ( !currentWord.word.match( /\S+:\S*/ ) ) {
			return deferred.reject().promise();
		}

		prefix = this._getPrefixFromWord( currentWord.word.trim() );
		term = this._getTermFromWord( currentWord.word.trim() );
		entityPrefixes = this._extractPrefixes( editorContent );

		if ( !entityPrefixes[prefix] ) { // unknown prefix
			if ( this._rdfNamespaces.ALL_PREFIXES && this._rdfNamespaces.ALL_PREFIXES[prefix] ) {
				// Sparql.js may deal with those prefixes
				return deferred.reject().promise();
			}
			list = [ {
				text: term,
				displayText: 'Unknown prefix \'' + prefix + ':\''
			} ];
			return deferred.resolve( this._getHintCompletion( lineNum, currentWord, prefix, list ) )
					.promise();
		}

		if ( term.length === 0 ) { // empty search term
			list = [ {
				text: term,
				displayText: 'Type to search for an entity'
			} ];
			return deferred.resolve( this._getHintCompletion( lineNum, currentWord, prefix, list ) )
					.promise();
		}

		if ( entityPrefixes[prefix] ) { // search entity
			this._searchEntities( term, entityPrefixes[prefix] ).done(
					function( list ) {
						return deferred.resolve( self._getHintCompletion( lineNum, currentWord,
								prefix, list ) );
					} );
		}

		return deferred.promise();
	};

	SELF.prototype._getPrefixFromWord = function( word ) {
		return word.split( ':', 1 )[0];
	};

	SELF.prototype._getTermFromWord = function( word ) {
		return word.split( ':' ).pop();
	};

	SELF.prototype._getHintCompletion = function( lineNum, currentWord, prefix, list ) {
		var completion = {
			list: []
		};
		completion.from = {
			line: lineNum,
			char: currentWord.start + prefix.length + 1
		};
		completion.to = {
			line: lineNum,
			char: currentWord.end
		};
		completion.list = list;

		return completion;
	};

	SELF.prototype._searchEntities = function( term, type ) {
		var entityList = [],
			deferred = $.Deferred();

		this._api.searchEntities( term, type ).done( function( data ) {
			$.each( data.search, function( key, value ) {
				entityList.push( {
					className: 'wikibase-rdf-hint',
					text: value.id,
					displayText: value.label + ' (' + value.id + ') ' + value.description + '\n'
				} );
			} );

			deferred.resolve( entityList );
		} );

		return deferred.promise();
	};

	SELF.prototype._getCurrentWord = function( line, position ) {
		var pos = position - 1,
			colon = false,
			wordSeparator = [ '/', '*', '+', ' ', ';', '.', '\n', '\t', '(', ')', '{', '}', '[', ']' ];

		if ( pos < 0 ) {
			pos = 0;
		}

		while ( wordSeparator.indexOf( line.charAt( pos ) ) === -1
				|| ( line.charAt( pos ) === ' ' && colon === false )
				|| ( line.charAt( pos ) === ':' && colon === false ) ) {

			if ( line.charAt( pos ) === ':' ) {
				colon = true;
			}
			pos--;
			if ( pos < 0 ) {
				break;
			}
		}
		var left = pos + 1;

		pos = position;
		while ( wordSeparator.indexOf( line.charAt( pos ) ) === -1 ) {
			pos++;
			if ( pos >= line.length ) {
				break;
			}
		}
		var right = pos;

		var word = line.substring( left, right );
		return {
			word: word,
			start: left,
			end: right
		};
	};

	SELF.prototype._extractPrefixes = function( text ) {
		var prefixes = this._rdfNamespaces.getPrefixMap( this._rdfNamespaces.ENTITY_TYPES ),
			types = this._rdfNamespaces.ENTITY_TYPES,
			lines = text.split( '\n' ),
			matches;

		$.each( lines, function( index, line ) {
			// PREFIX wd: <http://www.wikidata.org/entity/>
			if ( ( matches = line.match( /(PREFIX) (\S+): <([^>]+)>/ ) ) ) {
				if ( types[matches[3]] ) {
					prefixes[matches[2]] = types[matches[3]];
				}
			}
		} );

		return prefixes;
	};

}( jQuery, wikibase ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.editor = wikibase.queryService.ui.editor || {};
wikibase.queryService.ui.editor.tooltip = wikibase.queryService.ui.editor.tooltip || {};

wikibase.queryService.ui.editor.tooltip.Rdf = ( function( CodeMirror, $, _ ) {
	'use strict';

	/**
	 * Wikibase RDF tooltip for codemirror editor
	 *
	 * @license GNU GPL v2+
	 * @class wikibase.queryService.ui.editor.tooltip.Rdf
	 *
	 * @author Jonas Kress
	 * @constructor
	 * @param {wikibase.queryService.api.Wikibase} api
	 */
	function SELF( api, rdfNamespaces ) {
		this._api = api;
		this._rdfNamespaces = rdfNamespaces;

		if ( !this._api ) {
			this._api = new wikibase.queryService.api.Wikibase();
		}

		if ( !this._rdfNamespaces ) {
			this._rdfNamespaces = wikibase.queryService.RdfNamespaces;
		}
	}

	/**
	 * @property {wikibase.queryService.RdfNamespaces}
	 * @private
	 */
	SELF.prototype._rdfNamespaces = null;

	/**
	 * @property {CodeMirror}
	 * @private
	 */
	SELF.prototype._editor = null;

	/**
	 * Set the editor the onmouseover callback is registered to
	 *
	 * @param {CodeMirror} editor
	 */
	SELF.prototype.setEditor = function( editor ) {
		this._editor = editor;
		this._registerHandler();
	};

	SELF.prototype._registerHandler = function() {
		var self = this;

		CodeMirror.on( this._editor.getWrapperElement(), 'mouseover', _.debounce( function( e ) {
			self._triggerTooltip( e );
		}, 300 ) );
	};// TODO: Remove CodeMirror dependency

	SELF.prototype._triggerTooltip = function( e ) {
		this._removeToolTip();
		this._createTooltip( e );
	};

	SELF.prototype._createTooltip = function( e ) {
		var posX = e.clientX,
			posY = e.clientY + $( window ).scrollTop(),
			token = this._editor.getTokenAt( this._editor.coordsChar( {
				left: posX,
				top: posY
			} ) ).string;

		if ( !token.match( /.+\:(Q|P)[0-9]*/ ) ) {
			return;
		}

		var prefixes = this._extractPrefixes( this._editor.doc.getValue() );
		var prefix = token.split( ':', 1 )[0];
		var entityId = token.split( ':' ).pop();

		if ( !prefixes[prefix] ) {
			return;
		}

		var self = this;
		this._searchEntities( entityId, prefixes[prefix] ).done( function( list ) {
			self._showToolTip( list.shift(), {
				x: posX,
				y: posY
			} );
		} );
	};

	SELF.prototype._removeToolTip = function() {
		$( '.wikibaseRDFtoolTip' ).remove();
	};

	SELF.prototype._showToolTip = function( text, pos ) {
		if ( !text || !pos ) {
			return;
		}

		$( '<div class="panel panel-info">' ).css( 'position', 'absolute' ).css( 'z-index', '100' )
				.css( 'max-width', '200px' ).css( {
					top: pos.y + 2,
					left: pos.x + 2
				} ).addClass( 'wikibaseRDFtoolTip' ).append(
						$( '<div class="panel-body">' ).html( text ).css( 'padding', '10px' ) )
				.appendTo( 'body' ).fadeIn( 'slow' );
	};

	SELF.prototype._extractPrefixes = function( text ) {
		var prefixes = this._rdfNamespaces.getPrefixMap( this._rdfNamespaces.ENTITY_TYPES ),
			lines = text.split( '\n' ),
			matches;

		var self = this;
		$.each( lines, function( index, line ) {
			// PREFIX wd: <http://www.wikidata.org/entity/>
			if ( ( matches = line.match( /(PREFIX) (\S+): <([^>]+)>/ ) ) ) {
				if ( self._rdfNamespaces.ENTITY_TYPES[matches[3]] ) {
					prefixes[matches[2]] = self._rdfNamespaces.ENTITY_TYPES[matches[3]];
				}
			}
		} );

		return prefixes;
	};

	SELF.prototype._searchEntities = function( term, type ) {
		var entityList = [],
			deferred = $.Deferred();

		this._api.searchEntities( term, type ).done(
				function( data ) {
					$.each( data.search, function( key, value ) {
						entityList.push( value.label + ' (' + value.id + ')<br/><small>' +
							( value.description || '' ) + '</small>' );
					} );

					deferred.resolve( entityList );
				} );

		return deferred.promise();
	};

	return SELF;

}( CodeMirror, jQuery, _ ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.editor = wikibase.queryService.ui.editor || {};

wikibase.queryService.ui.editor.Editor = ( function( $, wikibase, CodeMirror, WikibaseRDFTooltip ) {
	'use strict';

	var CODEMIRROR_DEFAULTS = {
			'lineNumbers': true,
			'matchBrackets': true,
			'mode': 'sparql',
			'extraKeys': {
				'Ctrl-Space': 'autocomplete',
				'Tab': function( cm ) {
					var spaces = Array( cm.getOption( 'indentUnit' ) + 1 ).join( ' ' );
					cm.replaceSelection( spaces );
				},
				'F11': function( cm ) {
					cm.setOption( 'fullScreen', !cm.getOption( 'fullScreen' ) );
				},
				'Esc': function( cm ) {
					if ( cm.getOption( 'fullScreen' ) ) {
						cm.setOption( 'fullScreen', false );
					}
				}
			},
			'viewportMargin': Infinity,
			'hintOptions': {
				closeCharacters: /[]/,
				completeSingle: false
			}
		},
		ERROR_LINE_MARKER = null,
		ERROR_CHARACTER_MARKER = null;

	var LOCAL_STORAGE_KEY = 'wikibase.queryService.ui.Editor';

	/**
	 * An ui editor for the Wikibase query service
	 *
	 * @class wikibase.queryService.ui.editor.Editor
	 * @license GNU GPL v2+
	 *
	 * @author Stanislav Malyshev
	 * @author Jonas Kress
	 * @constructor
	 * @param {wikibase.queryService.ui.editor.hint.Rdf} rdfHint
	 * @param {wikibase.queryService.ui.editor.hint.Sparql} sparqlHint
	 * @param {wikibase.queryService.ui.editor.tooltip.Rdf} rdfTooltip
	 */
	function SELF( rdfHint, sparqlHint, rdfTooltip ) {
		this._rdfHint = rdfHint;
		this._sparqlHint = sparqlHint;
		this._rdfTooltip = rdfTooltip;

		if ( !this._sparqlHint ) {
			this._sparqlHint = new wikibase.queryService.ui.editor.hint.Sparql();
		}
		if ( !this._rdfHint ) {
			this._rdfHint = new wikibase.queryService.ui.editor.hint.Rdf();
		}
		if ( !this._rdfTooltip ) {
			this._rdfTooltip = new wikibase.queryService.ui.editor.tooltip.Rdf();
		}
	}

	/**
	 * @property {CodeMirror}
	 * @type CodeMirror
	 * @private
	 */
	SELF.prototype._editor = null;

	/**
	 * @property {wikibase.queryService.ui.editor.hint.Sparql}
	 * @private
	 */
	SELF.prototype._sparqlHint = null;

	/**
	 * @property {wikibase.queryService.ui.editor.hint.Rdf}
	 * @private
	 */
	SELF.prototype._rdfHint = null;

	/**
	 * @property {wikibase.queryService.ui.editor.tooltip.Rdf}
	 * @private
	 */
	SELF.prototype._rdfTooltip = null;

	/**
	 * Construct an this._editor on the given textarea DOM element
	 *
	 * @param {HTMLElement} element
	 */
	SELF.prototype.fromTextArea = function( element ) {
		var self = this;

		this._editor = CodeMirror.fromTextArea( element, CODEMIRROR_DEFAULTS );
		this._editor.on( 'change', function( editor, changeObj ) {
			if ( self.getValue() !== '' ) {
				self.storeValue( self.getValue() );
			}
			self.clearError();
			if ( changeObj.text[0] === '?' || changeObj.text[0] === '#' ) {
				editor.showHint( {
					closeCharacters: /[\s]/
				} );
			}
		} );
		this._editor.focus();

		this._rdfTooltip.setEditor( this._editor );

		this._registerHints();
	};

	SELF.prototype._registerHints = function() {
		var self = this;

		CodeMirror
				.registerHelper(
						'hint',
						'sparql',
						function( editor, callback, options ) {
							if ( editor !== self._editor ) {
								return;
							}
							var lineContent = editor.getLine( editor.getCursor().line ),
								editorContent = editor.doc.getValue(),
								cursorPos = editor.getCursor().ch,
								lineNum = editor.getCursor().line;

							self._getHints( editorContent, lineContent, lineNum, cursorPos ).done(
									function( hint ) {
										callback( hint );
									} );
						} );

		CodeMirror.hint.sparql.async = true;
	};

	SELF.prototype._getHints = function( editorContent, lineContent, lineNum, cursorPos ) {
		var self = this;

		return this._rdfHint.getHint(
			editorContent, lineContent, lineNum, cursorPos
		).catch( function() {
			// if rdf hint is rejected try sparql hint
			return self._sparqlHint.getHint( editorContent, lineContent, lineNum, cursorPos );
		} ).then( function( hint ) {
			hint.from = CodeMirror.Pos( hint.from.line, hint.from.char );
			hint.to = CodeMirror.Pos( hint.to.line, hint.to.char );
			return hint;
		} );
	};

	/**
	 * Construct an this._editor on the given textarea DOM element
	 *
	 * @param {string} keyMap
	 * @throws {Function} callback
	 */
	SELF.prototype.addKeyMap = function( keyMap, callback ) {
		this._editor.addKeyMap( {
			keyMap: callback
		} );
	};

	/**
	 * @param {string} value
	 */
	SELF.prototype.setValue = function( value ) {
		this._editor.setValue( value );
	};

	/**
	 * @return {string}
	 */
	SELF.prototype.getValue = function() {
		return this._editor.getValue();
	};

	SELF.prototype.save = function() {
		this._editor.save();
	};

	/**
	 * @param {string} value
	 */
	SELF.prototype.prepandValue = function( value ) {
		this._editor.setValue( value + this._editor.getValue() );
	};

	SELF.prototype.refresh = function() {
		this._editor.refresh();
	};

	/**
	 * Highlight SPARQL error in editor window.
	 *
	 * @param {string} description
	 */
	SELF.prototype.highlightError = function( description ) {
		var line,
			character,
			match = null;

		match = description.match( /line ([0-9]*)\:/i );
		if ( match ) {
			line = match[1];
		}

		match = description.match( /line (\d+), column (\d+)/ );
		if ( match ) {
			// highlight character at error position
			line = match[1] - 1;
			character = match[2] - 1;
			if ( character >= this._editor.doc.getLine( line ).length - 1 ) {
				character = this._editor.doc.getLine( line ).length - 1;
			}

			ERROR_CHARACTER_MARKER = this._editor.doc.markText( {
				'line': line,
				'ch': character
			}, {
				'line': line,
				'ch': character + 1
			}, {
				'className': 'error-character'
			} );
		}

		if ( line ) {
			ERROR_LINE_MARKER = this._editor.doc.markText( {
				'line': line,
				'ch': 0
			}, {
				'line': line
			}, {
				'className': 'error-line'
			} );
		}
	};

	/**
	 * Clear SPARQL error in editor window.
	 */
	SELF.prototype.clearError = function() {
		if ( ERROR_LINE_MARKER ) {
			ERROR_LINE_MARKER.clear();
		}
		if ( ERROR_CHARACTER_MARKER ) {
			ERROR_CHARACTER_MARKER.clear();
		}
	};

	/**
	 * Stores the given value in the local storage
	 *
	 * @param {string} value
	 */
	SELF.prototype.storeValue = function( value ) {
		try {
			if ( localStorage ) {
				localStorage.setItem( LOCAL_STORAGE_KEY, value );
			}
		} catch ( e ) {
		}
	};

	/**
	 * Restores the value from the local storage
	 */
	SELF.prototype.restoreValue = function() {
		try {
			if ( localStorage ) {
				var value = localStorage.getItem( LOCAL_STORAGE_KEY );
				if ( value ) {
					this.setValue( value );
					this.refresh();
				}
			}
		} catch ( e ) {
		}
	};

	/**
	 * Register callback handler
	 *
	 * @param {string} type
	 * @param {Function} callback
	 */
	SELF.prototype.registerCallback = function( type, callback ) {
		this._editor.on( type, callback );
	};

	/**
	 * Toggle editor fullscreen
	 */
	SELF.prototype.toggleFullscreen = function( fullscreen ) {
		this._editor.setOption( 'fullScreen', !this._editor.getOption( 'fullScreen' ) );
	};

	/**
	 * Set focus on the editor
	 */
	SELF.prototype.focus = function() {
		this._editor.focus();
	};

	return SELF;

}( jQuery, wikibase, CodeMirror ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.queryHelper = wikibase.queryService.ui.queryHelper || {};

wikibase.queryService.ui.queryHelper.QueryHelper = ( function( $, wikibase, _ ) {
	'use strict';

	var FILTER_PREDICATES = {
			'http://www.w3.org/2000/01/rdf-schema#label': true,
			'http://schema.org/description': true,
			'http://www.bigdata.com/queryHints#optimizer': true,
			'http://www.w3.org/1999/02/22-rdf-syntax-ns#type': true
		},
		I18N_PREFIX = 'wdqs-ve-',
		TABLE_OPTIONS = {
			formatNoMatches: function () {
			}
		};

	/**
	 * A visual SPARQL editor for the Wikibase query service
	 *
	 * @class wikibase.queryService.ui.queryHelper.QueryHelper
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 * @param {wikibase.queryService.api.Wikibase} [api]
	 * @param {wikibase.queryService.api.Sparql} [sparqlApi]
	 * @param {wikibase.queryService.ui.queryHelper.SelectorBox} [selectorBox]
	 */
	function SELF( api, sparqlApi, selectorBox ) {
		this._api = api || new wikibase.queryService.api.Wikibase();
		this._selectorBox = selectorBox
			|| new wikibase.queryService.ui.queryHelper.SelectorBox( this._api, sparqlApi );
		this._query = new wikibase.queryService.ui.queryHelper.SparqlQuery();
	}

	/**
	 * @property {wikibase.queryService.api.Wikibase}
	 * @private
	 */
	SELF.prototype._api = null;

	/**
	 * @property {wikibase.queryService.ui.queryHelper.SelectorBox}
	 * @private
	 */
	SELF.prototype._selectorBox = null;

	/**
	 * @property {Function}
	 * @private
	 */
	SELF.prototype._changeListener = null;

	/**
	 * @property {wikibase.queryService.ui.queryHelper.SparqlQuery}
	 * @private
	 */
	SELF.prototype._query = null;

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._triples = [];

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._isSimpleMode = false;

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._labels = {
		filter: 'Filter',
		show: 'Show',
		anything: 'anything',
		'with': 'with',
		any: 'any',
		or: 'or',
		subtype: 'subtype'
	};

	/**
	 * Set the SPARQL query string
	 *
	 * @param {string} query SPARQL query string
	 */
	SELF.prototype.setQuery = function( query ) {
		var prefixes = wikibase.queryService.RdfNamespaces.ALL_PREFIXES;
		this._query.parse( query, prefixes );

		this._selectorBox.setQuery( this._query );
	};

	/**
	 * Get the SPARQL query string
	 *
	 * @return {string|null}
	 */
	SELF.prototype.getQuery = function() {
		try {
			var q = this._query.getQueryString();
			q = this._cleanQueryPrefixes( q ).trim();
			return q.trim();
		} catch ( e ) {
			return null;
		}
	};

	/**
	 * Workaround for https://phabricator.wikimedia.org/T133316
	 *
	 * @private
	 */
	SELF.prototype._cleanQueryPrefixes = function( query ) {
		var prefixRegex = /PREFIX ([a-z]+): <(.*)>/gi,
			m,
			prefixes = {},
			cleanQuery = query.replace( prefixRegex, '' ).replace( /\n+/g, '\n' );

		while ( ( m = prefixRegex.exec( query ) ) ) {
			var prefix = m[1];
			var uri = m[2].replace( /[\-\[\]\/\{\}\(\)\*\+\?\.\\\^\$\|]/g, '\\$&' );

			var newQuery = cleanQuery.replace( new RegExp( '<' + uri + '([^/>#]+?)>', 'gi' ),
					prefix + ':$1' );

			if ( cleanQuery !== newQuery ) {
				cleanQuery = newQuery;
				if ( !wikibase.queryService.RdfNamespaces.STANDARD_PREFIXES[prefix] ) {
					prefixes[m[0]] = true;
				}
			}
		}

		cleanQuery = Object.keys( prefixes ).join( '\n' ) + '\n\n' + cleanQuery.trim();
		return cleanQuery;
	};

	/**
	 * Draw visual editor to given element
	 *
	 * @param {jQuery} $element
	 */
	SELF.prototype.draw = function( $element ) {
		var template = wikibase.queryService.ui.queryHelper.QueryTemplate.parse( this._query ),
			self = this,
			bindings = this._query.getBindings();

		if ( template !== null ) {
			try {
				return $element.html( template.getHtml(
					function( variable ) {
						return self._getLabel( bindings[ variable ].expression );
					},
					this._selectorBox,
					function( variable, oldId, newId ) {
						bindings[ variable ].expression = bindings[ variable ].expression
							.replace( new RegExp( oldId + '$' ), '' )
							+ newId;
						if ( self._changeListener ) {
							self._changeListener( self );
						}
					}
				) );
			} catch ( e ) {
				window.console.error( e );
			}
		}

		this._triples = this._query.getTriples();

		var subqueries = this._query.getSubQueries();
		while ( subqueries.length > 0 ) {
			var q = subqueries.pop();
			this._triples = this._triples.concat( q.getTriples() );
			subqueries.concat( q.getSubQueries() );
		}

		this._isSimpleMode = this._isSimpleQuery();
		$element.html( this._getHtml() );
	};

	/**
	 * Set the change listener
	 *
	 * @param {Function} listener a function called when query changed
	 */
	SELF.prototype.setChangeListener = function( listener ) {
		this._changeListener = listener;
	};

	/**
	 * @private
	 */
	SELF.prototype._i18n = function( key ) {
		if ( !$.i18n ) {
			return this._labels[key];
		}

		return $.i18n( I18N_PREFIX + key );
	};

	/**
	 * @private
	 */
	SELF.prototype._getHtml = function() {
		var self = this,
			$html = $( '<div>' ),
			$findTable = $( '<table>' ).bootstrapTable( TABLE_OPTIONS ),
			$showTable = $( '<table>' ).bootstrapTable( TABLE_OPTIONS );

		$.each( this._triples, function( k, triple ) {
			if ( self._isNotRelevant( triple.triple ) ) {
				return;
			}

			if ( self._isInShowSection( triple.triple ) ) {
				$showTable.append( self._getTripleHtml( triple ) );
				return;
			}

			$findTable.append( self._getTripleHtml( triple ) );
		} );

		return $html.append(
				this._createSection( $findTable, this._createFindButton( $findTable ) ),
				this._createSection( $showTable, this._createShowButton( $showTable ) ),
				this._getLimitSection()
			);
	};

	/**
	 * @private
	 */
	SELF.prototype._getLimitSection = function() {
		var $limitSection = $( '<div>' ),
			$limit = $( '<a data-type="number">' )
				.attr( 'href', '#' )
				.text( 'Limit' )
				.data( 'value', this._query.getLimit() ),
			$value = $( '<span>' )
				.text( this._query.getLimit() || '' );

		var self = this;
		this._selectorBox.add( $limit, null, function( value ) {
			if ( value === '0' ) {
				value = null;
			}

			$value.text( value || '' );
			self._query.setLimit( value );

			if ( self._changeListener ) {
				self._changeListener( self );
			}
		}, {
			trash: function() {
				self._query.setLimit( null );
				$limit.data( 'value', '' );
				$value.text( '' );
				if ( self._changeListener ) {
					self._changeListener( self );
				}
				return true;//close popover
			}
		} );

		return $limitSection.append( $limit.append( ' ', $value ) );
	};

	/**
	 * @private
	 */
	SELF.prototype._createSection = function( $table, $button ) {
		return $( '<table>' ).append( $( '<tr>' ).append(
				$( '<td>' ).append( $button ),
				$( '<td>' ).append( $table )
			) );
	};

	/**
	 * @private
	 */
	SELF.prototype._createFindButton = function( $table ) {
		// Show link
		var $button = $( '<a class="btn">' )
			.text( this._i18n( 'filter' ) )
			.attr( 'href', '#' ).prepend(
				'<span class="fa fa-plus" aria-hidden="true"></span>', ' ' )
				.tooltip( {
					title: 'Click to add new item'
				} ).attr( 'data-type', 'item' ).attr( 'data-auto_open', true );

		// SelectorBox
		var self = this;
		this._selectorBox.add( $button, null, function( id, name, propertyId ) {
			var entity = 'http://www.wikidata.org/entity/' + id;// FIXME technical debt

			var variable = self._query.getBoundVariables().shift();
			if ( !variable ) {
				variable = '?' + name.replace( /( |[^a-z0-9])/gi, '_' );
			}

			var prop = 'http://www.wikidata.org/prop/direct/' + ( propertyId || 'P31' );// FIXME technical debt
			var triple = self._query.addTriple( variable, prop, entity, false );
			if ( !self._query.hasVariable( variable ) ) {
				self._query.addVariable( variable );
			}

			self._addLabelVariableAfterItemColumn( prop.split( '/' ).pop(), triple );

			$table.append( self._getTripleHtml( triple ) );

			if ( self._changeListener ) {
				self._changeListener( self );
			}
		} );

		return $button;
	};

	/**
	 * @private
	 */
	SELF.prototype._createShowButton = function( $table ) {
		// Show link
		var $button = $( '<a class="btn">' )
			.text( this._i18n( 'show' ) )
			.attr( 'href', '#' ).prepend(
				'<span class="fa fa-plus" aria-hidden="true"></span>', ' ' )
				.tooltip( {
					title: 'Click to add new property'
				} ).attr( 'data-type', 'property' ).attr( 'data-auto_open', true );

		// SelectorBox
		var self = this;
		this._selectorBox.add( $button, null, function( id, name ) {
			var prop = 'http://www.wikidata.org/prop/direct/' + id;// FIXME technical debt

			var subject = self._query.getBoundVariables().shift() || '?item';
			var variable2 = '?' + name.replace( /( |[^a-z0-9])/gi, '_' );// FIXME technical debt

			var triple = self._query.addTriple( subject, prop, variable2, true );
			self._query.addVariable( variable2 );

			self._addLabelVariableAfterItemColumn( prop.split( '/' ).pop(), triple );

			$table.append( self._getTripleHtml( triple ) );

			if ( self._changeListener ) {
				self._changeListener( self );
			}
		} );

		return $button;
	};

	/**
	 * @private
	 */
	SELF.prototype._addLabelVariableAfterItemColumn = function( propertyId, triple ) {
		var self = this;

		this._api.getDataType( propertyId ).done( function ( type ) {
			if ( type === 'wikibase-item' ) {
				self._addLabelVariable( triple );
			}
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._isNotRelevant = function( triple ) {
		if ( FILTER_PREDICATES[triple.predicate] ) {
			return true;
		}

		if ( this._isSimpleMode && this._isInShowSection( triple ) &&
				( this._query.hasVariable( triple.object ) === false &&
						this._query.hasVariable( triple.object + 'Label' ) === false &&
						this._query.isWildcardQuery() === false ) ) {
			return true;
		}

		return false;
	};

	/**
	 * @private
	 */
	SELF.prototype._isInShowSection = function( triple ) {
		var bindings = this._query.getBindings();

		// Must match ?value wdt:Pxx ?item
		if ( this._isVariable( triple.subject ) && this._isVariable( triple.object ) &&
				!bindings[triple.subject] && !bindings[triple.object] ) {
			return true;
		}

		return false;
	};

	/**
	 * @private
	 */
	SELF.prototype._getTripleHtml = function( triple ) {
		var self = this,
			$triple = $( '<tr>' ),
			bindings = this._query.getBindings();

		$.each( triple.triple, function( k, entity ) {
			if ( self._isVariable( entity ) && bindings[entity] &&
					typeof bindings[entity].expression === 'string' ) {
				entity = bindings[entity].expression;
			}

			if (  self._isVariable( entity ) ) {
				if ( self._isSimpleMode ) {
					return;
				}
				entity = entity.replace( '?', '' ).replace( /_/g, '_<wbr>' );
				$triple.append(  $( '<td>' ).append( entity ) );
				return;
			}

			if ( entity.type && entity.type === 'path' ) {
				$triple.append( $( '<td>' ).append( self._getTripleEntityPathHtml( entity, triple, k ) ), ' ' );
			} else {
				$triple.append(  $( '<td>' ).append( self._getTripleEntityHtml( entity, triple, k ) ), ' ' );
			}
		} );

		return $triple.append( this._getTripleHtmlToolbar( $triple, triple ) );
	};

	/**
	 * @private
	 */
	SELF.prototype._getTripleHtmlToolbar = function( $triple, triple ) {
		var self = this;

		var $delete = $( '<a href="#">' ).addClass( 'fa fa-trash-o' ).click( function () {
			var variables = _.compact( triple.triple ).filter( function ( v ) {
					return typeof v === 'string' && v.startsWith( '?' );
				} ),
				variablesLabels = variables.map( function ( v ) {
					return v + 'Label';
				} );

			triple.remove();
			$triple.remove();
			self._query.cleanupVariables( variables.concat( variablesLabels ) );

			if ( self._changeListener ) {
				self._changeListener( self );
			}

			return false;
		} ).tooltip( {
			title: self._i18n( 'remove-row-title' )
		} );

		var $label = $( '<a href="#">' ).addClass( 'fa fa-tag' ).click( function () {
			self._addLabelVariable( triple );
			if ( self._changeListener ) {
				self._changeListener( self );
			}
			return false;
		} ).tooltip( {
			title: self._i18n( 'add-label-title' )
		} );

		return $( '<td class="toolbar">' ).append( $label, $delete );
	};

	/**
	 * @private
	 */
	SELF.prototype._addLabelVariable = function( triple ) {

		this._query.convertWildcardQueryToUseVariables();

		if ( triple.triple.object.startsWith( '?' ) ) {
			this._query.addVariable( triple.triple.object + 'Label' );
		} else {
			this._query.addVariable( triple.triple.subject + 'Label' );
		}
		if ( this._changeListener ) {
			this._changeListener( this );
		}
	};

	/**
	 * @private
	 */
	SELF.prototype._isVariable = function( entity ) {
		return typeof entity === 'string' && entity.startsWith( '?' );
	};

	/**
	 * @private
	 */
	SELF.prototype._isSimpleQuery = function() {
		var boundVariables = {};

		var self = this;
		$.each( this._triples, function( k, t ) {
			// Must match ?value wdt:Pxx ?item
			if ( self._isVariable( t.triple.subject ) &&
					self._isVariable( t.triple.object ) === false ) {
				boundVariables[t.triple.subject] = true;
			}
		} );

		return Object.keys( boundVariables ).length <= 1;
	};

	/**
	 * @private
	 */
	SELF.prototype._getTripleEntityPathHtml = function( path, triple ) {
		var self = this,
			$path = $( '<span>' );
		$.each( path.items, function( k, v ) {
			if ( v.type && v.type === 'path' ) {
				$path.append( self._getTripleEntityPathHtml( v, triple ) );
				return;
			}

			if ( k > 0 && path.pathType === '/' ) {
				$path.append( ' ' + self._i18n( 'or' ) + ' ' + self._i18n( 'subtype' ) + ' ' );
			}
			if ( path.pathType === '*' ) {
				$path.append( ' ' + self._i18n( 'any' ) + ' ' );
			}

			// FIXME: Do not fake triple here
			var newTriple = path.items.reduce( function( o, v, i ) {
				o[i] = v;
				return o;
			}, {} );
			newTriple = $.extend( newTriple, triple.triple );
			triple = $.extend( {}, triple );
			triple.triple = newTriple;

			$path.append( self._getTripleEntityHtml( v, triple, k ) );
		} );

		return $path;
	};

	/**
	 * @private
	 */
	SELF.prototype._getTripleEntityHtml = function( entity, triple, key ) {
		var self = this,
			$label = $( '<span>' );

		this._getLabel( entity ).done( function( label, id, description, type ) {
			var $link = $( '<a>' )
				.text( label )
				.attr( { 'data-id': id, 'data-type': type, href: '#' } )
				.appendTo( $label );

			$label.tooltip( {
				'title': '(' + id + ') ' + description
			} );
			$( $label ).on( 'show.bs.tooltip', function() {
				if ( $( '.tooltip' ).is( ':visible' ) ) {
					$( '.tooltip' ).not( this ).hide();
				}
			} );

			//TODO: refactor method
			self._selectorBox.add( $link, triple.triple, function( selectedId ) {
				var newEntity = entity.replace( new RegExp( id + '$' ), '' ) + selectedId;// TODO: technical debt

				$label.replaceWith( self._getTripleEntityHtml( newEntity, triple, key ) );

				if ( !self._isVariable( triple.triple[key] ) ) {
					triple.triple[key] = newEntity;
				} else {
					self._query.getBindings()[triple.triple[key]].expression = newEntity;
				}

				if ( self._changeListener ) {
					self._changeListener( self );
				}
			} );

		} ).fail( function() {
			$label.text( entity );
		} );

		return $label;
	};

	/**
	 * @private
	 */
	SELF.prototype._getLabel = function( url ) {
		var deferred = $.Deferred();

		var entity = url.match( /(Q|P)([0-9]+)/ );// TODO: make use of Rdf namespaces
		if ( !entity ) {
			return deferred.reject().promise();
		}

		var type = {
			P: 'property',
			Q: 'item'
		};
		type = type[entity[1]];
		var term = entity[0];

		this._api.searchEntities( term, type ).done( function( data ) {
			$.each( data.search, function( key, value ) {
				deferred.resolve( value.label, value.id, value.description, type );
				return false;
			} );
		} );

		return deferred.promise();
	};

	return SELF;
}( jQuery, wikibase, _ ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.queryHelper = wikibase.queryService.ui.queryHelper || {};

wikibase.queryService.ui.queryHelper.SparqlQuery = ( function( $, wikibase, sparqljs ) {
	'use strict';

	/**
	 * A SPARQL query representation
	 *
	 * @class wikibase.queryService.ui.queryHelper.SparqlQuery
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 * @param {Object} query
	 */
	function SELF( query ) {
		this._query = query;
	}

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._query = null;

	/**
	 * @property {Array}
	 * @private
	 */
	SELF.prototype._queryComments = null;

	/**
	 * Set the SPARQL query string
	 *
	 * @param {string} query SPARQL query string
	 * @param {Object} prefixes
	 */
	SELF.prototype.parse = function( query, prefixes ) {
		var parser = new sparqljs.Parser( prefixes ),
			queryComments = [];
		this._query = parser.parse( query );
		$.each( query.split( '\n' ), function( index, line ) {
			if ( line.indexOf( '#' ) === 0 ) {
				queryComments.push( line );
			}
		} );
		this._queryComments = queryComments;
	};

	/**
	 * Get the SPARQL query string
	 *
	 * @return {string|null}
	 */
	SELF.prototype.getQueryString = function() {
		try {
			var sparql = new sparqljs.Generator().stringify( this._query ),
				comments = this._queryComments.join( '\n' ).trim();

			if ( comments !== '' ) {
				return comments + '\n' + sparql;
			} else {
				return sparql;
			}
		} catch ( e ) {
			return null;
		}
	};

	/**
	 * @return {number|null}
	 */
	SELF.prototype.getLimit = function() {
		return this._query.limit || null;
	};

	/**
	 * @param {number|null} limit
	 * @return {wikibase.queryService.ui.queryHelper.SparqlQuery}
	 */
	SELF.prototype.setLimit = function( limit ) {
		if ( !limit ) {
			delete this._query.limit;
		} else {
			this._query.limit = limit;
		}

		return this;
	};

	/**
	 * Check whether a variable is used in the SELECT clause of query
	 *
	 * @param {string} name of variable e.g. ?foo
	 */
	SELF.prototype.hasVariable = function( name ) {
		if ( this._query.variables.length === 0 ) {
			return false;
		}

		return this._query.variables.indexOf( name ) >= 0;
	};

	/**
	 * Check whether query uses wildcard SELECT *
	 */
	SELF.prototype.isWildcardQuery = function() {
		if ( this._query.variables && this._query.variables[0] === '*' ) {
			return true;
		}

		return false;
	};

	/**
	 * Convert wildcard query to use variables
	 *
	 * @deprecated workaround for T171194
	 */
	SELF.prototype.convertWildcardQueryToUseVariables = function() {

		if ( !this.isWildcardQuery() ) {
			return;
		}

		var variables = this.getQueryString().match( /(\?\w+)/g );
		this._query.variables = $.unique( variables ) || '*';
	};

	/**
	 * Add a variable to the query SELECT
	 *
	 * @param {string} name
	 * @return {wikibase.queryService.ui.queryHelper.SparqlQuery}
	 */
	SELF.prototype.addVariable = function( name ) {
		if ( !name || !name.startsWith( '?' ) ) {
			return this;
		}

		if ( this._query.variables.indexOf( name ) >= 0 ) {
			return this;
		}

		if ( this.isWildcardQuery() ) {
			return this;
		}

		this._query.variables.push( name );

		return this;
	};

	/**
	 * Remove a variable from the query SELECT
	 *
	 * @param {string} name
	 */
	SELF.prototype.removeVariable = function( name ) {
		if ( !name.startsWith( '?' ) ) {
			return;
		}

		if ( this.isWildcardQuery() ) {
			return;
		}

		var index = this._query.variables.indexOf( name );
		if ( index >= 0 ) {
			this._query.variables.splice( index, 1 );
		}

		if ( this._query.variables.length === 0 ) {
			this._query.variables.push( '*' );
		}
	};

	/**
	 * Removes unused variables from the query SELECT
	 * Disclaimer: may remove too much
	 *
	 * @param {string[]} [variables] cleanup only variables in this list
	 */
	SELF.prototype.cleanupVariables = function( variables ) {
		var self = this,
			usedVariables = this.getTripleVariables();

		// TODO check sub queries
		var toRemove = this._query.variables.filter( function ( variable ) {
			if ( variables && variables.indexOf( variable ) === -1 ) {
				return false;
			}

			if ( usedVariables.indexOf( variable ) === -1 ) {
				return true;
			}

			return false;
		} );

		toRemove.map( function ( v ) {
			self.removeVariable( v );
		} );
	};

	/**
	 * Get triples defined in this query
	 *
	 * @return {Object}
	 */
	SELF.prototype.getTriples = function( node, isOptional ) {
		var triples = [];
		if ( !node ) {
			node = this._query.where;
		}
		if ( !isOptional ) {
			isOptional = false;
		}

		var self = this;
		$.each( node, function( k, v ) {
			if ( v.type && v.type === 'bgp' ) {
				triples = triples.concat( self._createTriples( v.triples, isOptional ) );
			}
			if ( v.type && v.type === 'optional' ) {
				triples = triples.concat( self.getTriples( v.patterns, true ) );
			}
			if ( v.type && v.type === 'union' ) {
				triples = triples.concat( self.getTriples( v.patterns, false ) );
			}
		} );

		return triples;
	};

	/**
	 * Get bindings defined in this query
	 *
	 * @return {Object}
	 */
	SELF.prototype.getBindings = function() {
		var bindings = {};

		$.each( this._query.where, function( k, v ) {
			if ( v.type && v.type === 'bind' ) {
				bindings[ v.variable ] = v;
			}
		} );

		return bindings;
	};

	/**
	 * @private
	 */
	SELF.prototype._createTriples = function( triplesData, isOptional ) {
		var self = this,
			triples = [];

		$.each( triplesData, function( i, triple ) {
			triples.push( {
				optional: isOptional,
				query: self,
				triple: triple,
				remove: function() {
					triplesData.splice( i, 1 );
				}
			} );
		} );

		return triples;
	};

	/**
	 * Get triples defined in this query
	 *
	 * @return {wikibase.queryService.ui.queryHelper.SparqlQuery[]}
	 */
	SELF.prototype.getSubQueries = function() {
		var queries = [];

		$.each( this._query.where, function( k, v ) {
			if ( v.queryType ) {
				queries.push( new SELF( v ) );
			}
		} );

		return queries;
	};

	/**
	 * Add a triple to the query
	 *
	 * @param {string} subject
	 * @param {string} predicate
	 * @param {string} object
	 * @param {boolean} isOptional
	 */
	SELF.prototype.addTriple = function( subject, predicate, object, isOptional ) {
		var triple = {
			type: 'bgp',
			triples: [
				{
					subject: subject,
					predicate: predicate,
					object: object
				}
			]
		};

		if ( isOptional ) {
			var optionalTriple = {
				type: 'optional',
				patterns: [
					triple
				]
			};
			this._query.where.push( optionalTriple );
		} else {
			this._query.where.push( triple );
		}

		return this._createTriples( triple.triples, isOptional )[0];
	};

	/**
	 * Get all variables used in triples
	 *
	 * @return {string[]}
	 */
	SELF.prototype.getTripleVariables = function() {
		var variables = {};

		$.each( this.getTriples(), function( i, t ) {
			if ( typeof t.triple.subject === 'string' && t.triple.subject.startsWith( '?' ) ) {
				variables[t.triple.subject] = true;
			}
			if ( typeof t.triple.predicate === 'string'  && t.triple.predicate.startsWith( '?' ) ) {
				variables[t.triple.predicate] = true;
			}
			if ( typeof t.triple.object === 'string' && t.triple.object.startsWith( '?' ) ) {
				variables[t.triple.object] = true;
			}

		} );

		return Object.keys( variables );
	};

	/**
	 * Get variables that are bound to a certain value
	 *
	 * @return {string[]}
	 */
	SELF.prototype.getBoundVariables = function() {
		var variables = {};

		$.each( this.getTriples(), function( i, t ) {
			if ( t.triple.subject.startsWith( '?' ) && !t.triple.object.startsWith( '?' ) ) {
				variables[t.triple.subject] = true;
			}

			if ( !t.triple.subject.startsWith( '?' ) && t.triple.object.startsWith( '?' ) ) {
				variables[t.triple.object] = true;
			}
		} );

		return Object.keys( variables );
	};

	/**
	 * Get services defined in query
	 *
	 * @return {object[]}
	 */
	SELF.prototype.getServices = function() {
		var services = [];

		$.each( this._query.where, function( i, node ) {
			if ( node && node.type === 'service' ) {
				services.push( node );
			}

		} );

		return services;
	};

	/**
	 * Remove a certain service from the query
	 *
	 * @param {string} serviceId of the service to be removed
	 * @return {wikibase.queryService.ui.queryHelper.SparqlQuery}
	 */
	SELF.prototype.removeService = function( serviceId ) {
		var self = this;

		$.each( this._query.where, function ( i, node ) {
			if ( node.type === 'service' && node.name === serviceId ) {
				delete self._query.where[i];
			}

		} );

		return this;
	};

	/**
	 * Get the content of a content beginning with start.
	 *
	 * For example, on a query with '#foo=bar',
	 * getCommentContent( 'foo=' ) will return 'bar'.
	 *
	 * @param {string} start The beginning of the comment, *without* the comment mark ('#').
	 *
	 * @return {?string}
	 */
	SELF.prototype.getCommentContent = function( start ) {
		var i, comment;
		for ( i = 0; i < this._queryComments.length; i++ ) {
			comment = this._queryComments[ i ];
			if ( comment.startsWith( '#' + start ) ) {
				return comment.substring( 1 + start.length );
			}
		}
		return null;
	};

	/**
	 * Clone query
	 *
	 * @return {wikibase.queryService.ui.queryHelper.SparqlQuery}
	 */
	SELF.prototype.clone = function() {
		var query = new SELF();
		query.parse( this.getQueryString() );
		return query;
	};

	return SELF;
}( jQuery, wikibase, sparqljs ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.queryHelper = wikibase.queryService.ui.queryHelper || {};

wikibase.queryService.ui.queryHelper.SelectorBox = ( function( $, wikibase ) {
	'use strict';

	var I18N_PREFIX = 'wdqs-ve-sb',
		SPARQL_TIMEOUT = 4 * 1000;

/*jshint multistr: true */
	var SPARQL_QUERY = {
			item: {
				suggest:// Find items that are used with a specifc property
					'SELECT ?id ?label ?description WHERE {\
					hint:Query hint:optimizer "None".\
							{\
								SELECT DISTINCT ?id WHERE { ?i <{PROPERTY_URI}> ?id. }\
								LIMIT 100\
							}\
						?id rdfs:label ?label.\
						?id schema:description ?description.\
						FILTER((LANG(?label)) = "{LANGUAGE}")\
						FILTER((LANG(?description)) = "{LANGUAGE}")\
					}\
					LIMIT 100',
				genericSuggest: function() { // Find items that are most often used with the first selected item of the current query
					var instanceOfTemplate =// Find items that are used with property 'instance of'
						'SELECT ?id ?label ?description (wd:P31 AS ?property) WHERE {\
						hint:Query hint:optimizer "None".\
							{\
								SELECT DISTINCT ?id WHERE { ?i wdt:P31 ?id. }\
								LIMIT 100\
							}\
							?id rdfs:label ?label.\
							?id schema:description ?description.\
							FILTER((LANG(?label)) = "{LANGUAGE}")\
							FILTER((LANG(?description)) = "{LANGUAGE}")\
						}\
						LIMIT 100';
					if ( this._query.getTriples().length === 0 ) {
						return instanceOfTemplate;
					}

					var template = '{PREFIXES}\n\
						SELECT ?id ?label ?description ?property WITH {\n\
							{QUERY}\n\
						} AS %query WITH {\n\
							SELECT ({VARIABLE} AS ?item) WHERE {\n\
								INCLUDE %query.\n\
							}\n\
						} AS %item WITH {\n\
							SELECT ?value ?property (COUNT(?statement) AS ?count) WHERE {\n\
								INCLUDE %item.\n\
								?item ?p ?statement.\n\
								?statement ?ps ?value.\n\
								?property a wikibase:Property; wikibase:claim ?p; wikibase:statementProperty ?ps.\n\
							}\n\
							GROUP BY ?value ?property\n\
							HAVING(isIRI(?value) && STRSTARTS(STR(?value), STR(wd:Q)))\n\
							ORDER BY DESC(?count)\n\
							LIMIT 100\n\
						} AS %values WHERE {\n\
							INCLUDE %values.\n\
							BIND(?value AS ?id).\n\
							SERVICE wikibase:label {\n\
								bd:serviceParam wikibase:language "[AUTO_LANGUAGE],en".\n\
								?id rdfs:label ?label; schema:description ?description.\n\
							}\n\
						}\n\
						ORDER BY DESC(?count)',
						variable = this._query.getBoundVariables().shift(),
						query = this._query.clone().setLimit( 1000 )
							.removeService( 'http://wikiba.se/ontology#label' )
							.addVariable( variable )
							.getQueryString(),
						prefixes = query.match( /.*\bPREFIX\b(.*)/gi ).join( '\n' );

					query = query.replace( /.*\bPREFIX\b.*/gi, '' );
					return template.replace( '{QUERY}', query )
						.replace( '{VARIABLE}', variable )
						.replace( '{PREFIXES}', prefixes );
				},
				search: null,
// Disable for now as requested by Smalyshev
//					'SELECT ?id ?label ?description WHERE {\
//						hint:Query hint:optimizer "None".\
//							{\
//								SELECT DISTINCT ?id WHERE { ?i <{PROPERTY_URI}> ?id. }\
//							}\
//						?id rdfs:label ?label.\
//						?id schema:description ?description.\
//						FILTER((LANG(?label)) = "{LANGUAGE}")\
//						FILTER((LANG(?description)) = "{LANGUAGE}")\
//						FILTER(STRSTARTS(LCASE(?label), LCASE("{TERM}")))\
//					}\
//					LIMIT 20',
				instanceOf:// Find items that are used with property 'instance of'
					'SELECT ?id ?label ?description WHERE {\
					hint:Query hint:optimizer "None".\
						{\
							SELECT DISTINCT ?id WHERE { ?i wdt:P31 ?id. }\
							LIMIT 100\
						}\
						?id rdfs:label ?label.\
						?id schema:description ?description.\
						FILTER((LANG(?label)) = "en")\
						FILTER((LANG(?description)) = "en")\
					}\
					LIMIT 100'
		},
		property: {
			suggest:// Find properties that are used with a specific item
				'SELECT ?id ?label ?description WHERE {\
				hint:Query hint:optimizer "None".\
					{\
						SELECT DISTINCT ?id WHERE {\
						?i ?prop <{ITEM_URI}>.\
						?id ?x ?prop.\
						?id rdf:type wikibase:Property.\
						}\
						LIMIT 100\
					}\
				?id rdfs:label ?label.\
				?id schema:description ?description.\
				FILTER((LANG(?label)) = "{LANGUAGE}")\
				FILTER((LANG(?description)) = "{LANGUAGE}")\
				}\
				LIMIT 100',
			genericSuggest: function() { // Find properties that are most often used with the first selected item of the current query

				var genericTemplate = // Find properties that are most often used with all items
				'SELECT ?id ?label ?description WITH {\
					SELECT ?pred (COUNT(?value) AS ?count) WHERE\
					{\
					?subj ?pred ?value .\
					} GROUP BY ?pred ORDER BY DESC(?count) LIMIT 1000\
					} AS %inner\
				WHERE {\
					INCLUDE %inner\
					?id wikibase:claim ?pred.\
					?id rdfs:label ?label.\
					?id schema:description ?description.\
					FILTER((LANG(?label)) = "en")\
					FILTER((LANG(?description)) = "en")\
				} ORDER BY DESC(?count)\
				LIMIT 100';

				if ( this._query.getTriples().length === 0 ) {
					return genericTemplate;
				}

				var template = '{PREFIXES}\n\
					SELECT ?id ?label ?description WITH {\n\
						{QUERY}\n\
					} AS %query WITH {\n\
						SELECT ({VARIABLE} AS ?item) WHERE {\n\
							INCLUDE %query.\n\
						}\n\
					} AS %item WITH {\n\
						SELECT ?property (COUNT(?statement) AS ?count) WHERE {\n\
							INCLUDE %item.\n\
							?item ?p ?statement.\n\
							?property a wikibase:Property; wikibase:claim ?p.\n\
						}\n\
						GROUP BY ?property\n\
						ORDER BY DESC(?count)\n\
						LIMIT 100\n\
					} AS %properties WHERE {\n\
						INCLUDE %properties.\n\
						BIND(?property AS ?id).\n\
						SERVICE wikibase:label {\n\
							bd:serviceParam wikibase:language "[AUTO_LANGUAGE],en".\n\
							?id rdfs:label ?label; schema:description ?description.\n\
						}\n\
					}\n\
					ORDER BY DESC(?count)',
					variable = this._query.getBoundVariables().shift(),
					query = this._query.clone()
						.setLimit( 500 )
						.removeService( 'http://wikiba.se/ontology#label' )
						.addVariable( variable )
						.getQueryString(),
					prefixes = query.match( /.*\bPREFIX\b(.*)/gi ).join( '\n' );

				query = query.replace( /.*\bPREFIX\b.*/gi, '' );
				return template.replace( '{QUERY}', query )
					.replace( '{VARIABLE}', variable )
					.replace( '{PREFIXES}', prefixes );
			},
			search: null,
//			search:// Find properties that are most often used with a specific item and filter with term prefix
//				'SELECT ?id ?label ?description WHERE {\
//					{\
//					SELECT ?id (COUNT(?id) AS ?count) WHERE {\
//						?i ?prop <{ITEM_URI}>.\
//						?id ?x ?prop.\
//						?id rdf:type wikibase:Property.\
//						}\
//						GROUP BY ?id\
//					}\
//				?id rdfs:label ?label.\
//				?id schema:description ?description.\
//				FILTER((LANG(?label)) = "{LANGUAGE}")\
//				FILTER((LANG(?description)) = "{LANGUAGE}")\
//				FILTER(STRSTARTS(LCASE(?label), LCASE("{TERM}")))\
//				}\
//				ORDER BY DESC(?count)\
//				LIMIT 20',
			seeAlso:// Read see also property from a specific property
				'SELECT ?id ?label ?description WHERE {\
					BIND( <{PROPERTY_URI}> as ?prop).\
					?props ?x  ?prop.\
					?props rdf:type wikibase:Property.\
					?props wdt:P1659 ?id.\
					?id rdfs:label ?label.\
					?id schema:description ?description.\
					FILTER((LANG(?label)) = "{LANGUAGE}")\
					FILTER((LANG(?description)) = "{LANGUAGE}")\
				}'
		},

		sparql: {
			search:
				'SELECT ?id ?label ?description WITH {\
					{SPARQL}\
					} AS %inner\
				WHERE {\
					INCLUDE %inner\
					?id rdfs:label ?label.\
					?id schema:description ?description.\
					FILTER((LANG(?label)) = "{LANGUAGE}")\
					FILTER((LANG(?description)) = "{LANGUAGE}")\
					FILTER(STRSTARTS(LCASE(?label), LCASE("{TERM}")))\
				} ORDER BY DESC(?count)\
				LIMIT 100',
			suggest:
				'SELECT ?id ?label ?description WITH {\
				{SPARQL}\
				} AS %inner\
			WHERE {\
				INCLUDE %inner\
				?id rdfs:label ?label.\
				?id schema:description ?description.\
				FILTER((LANG(?label)) = "{LANGUAGE}")\
				FILTER((LANG(?description)) = "{LANGUAGE}")\
			} ORDER BY DESC(?count)\
			LIMIT 100'
		}
	};

	/**
	 * A selector box for selecting and changing properties and items
	 *
	 * @class wikibase.queryService.ui.queryHelper.SelectorBox
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 * @param {wikibase.queryService.api.Wikibase} [api]
	 */
	function SELF( api, sparqlApi ) {
		this._api = api || new wikibase.queryService.api.Wikibase();
		this._sparqlApi = sparqlApi || new wikibase.queryService.api.Sparql();
	}

	/**
	 * @property {wikibase.queryService.api.Wikibase}
	 * @private
	 */
	SELF.prototype._api = null;

	/**
	 * @property {wikibase.queryService.api.Sparql}
	 * @private
	 */
	SELF.prototype._sparqlApi = null;

	/**
	 * @property {wikibase.queryService.ui.queryHelper.SparqlQuery}
	 * @private
	 */
	SELF.prototype._query = null;

	/**
	 * @param {wikibase.queryService.ui.queryHelper.SparqlQuery} query
	 */
	SELF.prototype.setQuery = function( query ) {
		this._query = query;
	};

	/**
	 * Add selector box to element
	 *
	 * @param {jQuery} $element
	 * @param {Object} triple
	 * @param {Function} listener a function called when value selected
	 * @param {Object} toolbar {icon:callback}
	 */
	SELF.prototype.add = function( $element, triple, listener, toolbar ) {
		switch ( $element.data( 'type' ).toLowerCase() ) {
		case 'number':
			this._createInput( $element, listener, toolbar );
			break;

		default:
			this._createSelect( $element, triple, listener, toolbar );
		}
	};

	/**
	 * @private
	 */
	SELF.prototype._createInput = function( $element, listener, toolbar ) {
		var $input = $( '<input>' ).attr( 'type', $element.data( 'type' ) ),
			$close = this._getCloseButton(),
			$toolbar = this._getToolbar( toolbar, $element ),
			$content = $( '<div>' ).append( $close, ' ', $input, ' ', $toolbar );

		$element.clickover( {
			placement: 'bottom',
			'global_close': false,
			'html': true,
			'content': function() {
				return $content;
			}
		} ).click( function( e ) {
			$input.val( $element.data( 'value' ) || '' );
		} );

		$input.on( 'keyup mouseup', function() {
			if ( listener ) {
				listener( $input.val() );
			}
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._createSelect = function( $element, triple, listener, toolbar ) {
		var self = this,
			$select = this._getSelectBox( $element ),
			$close = this._getCloseButton(),
			$toolbar = this._getToolbar( toolbar, $element ),
			$content = $( '<div>' ).append( $close, ' ', $select, ' ', $toolbar );

		if ( $element.children().length === 0 ) {
			this._createSelectInline( $element, triple, listener );
			return;
		}

		$element.clickover( {
			placement: 'bottom',
			'global_close': false,
			'html': true,
			'content': function() {
				return $content;
			}
		} ).click( function( e ) {
			$select.toggleClass( 'open' );

			if ( !$select.data( 'select2' ) ) {
				$.proxy( self._renderSelect2( $select, $element, triple ), self );
			}

			if ( $select.hasClass( 'open' ) ) {
				if ( $element.data( 'auto_open' ) ) {
					$select.data( 'select2' ).open();
				}
			}
			return false;
		} );

		$select.change( function( e ) {
			if ( listener ) {
				var id = $select.val(),
					text = $select.find( 'option:selected' ).text(),
					data = $element.data( 'items' ),
					propertyId = data && data[id] && data[id].propertyId || null;

				if ( propertyId ) {
					listener( id, text, propertyId );
				} else {
					self._suggestPropertyId( id ).always( function ( propertyId ) {
						listener( id, text, propertyId );
					} );
				}
			}
			$element.click();// hide clickover
			$select.html( '' );
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._suggestPropertyId = function( id ) {
		var deferred = $.Deferred(),
				query = 'SELECT ?id (COUNT(?id) AS ?count) WHERE { { SELECT ?id WHERE {\
							?i ?prop wd:{ITEM_ID}.\
							?id wikibase:directClaim ?prop.\
							?id rdf:type wikibase:Property.\
							}\
							LIMIT 10000 } } GROUP BY ?id ORDER BY DESC(?count) LIMIT 1'
					.replace( '{ITEM_ID}', id );

		this._sparqlApi.query( query, ( SPARQL_TIMEOUT / 2 ) ).done( function ( data ) {

			try {
				deferred.resolve( data.results.bindings[0].id.value.split( '/' ).pop() );
			} catch ( e ) {
				deferred.reject();
			}

		} ).fail( function() {
			deferred.reject();
		} );

		return deferred.promise();
	};

	/**
	 * @private
	 */
	SELF.prototype._createSelectInline = function( $element, triple, listener ) {
		var $select = this._getSelectBox( $element );

		$element.replaceWith( $select );
		this._renderSelect2( $select, $element, triple );

		$select.change( function( e ) {
			if ( listener ) {
				var id = $select.val(),
					data = $element.data( 'items' ),
					propertyId = data && data[id] && data[id].propertyId || null ;

				listener( id, $select.find( 'option:selected' ).text(), propertyId );
			}
		} );

	};

	/**
	 * @private
	 */
	SELF.prototype._getSelectBox = function( $element ) {
		var id = $element.data( 'id' );
		var label = $element.text();

		var $select = $( '<select>' );
		if ( id ) {
			$select.append( $( '<option>' ).attr( 'value', id ).text( label ) );
		}

		return $select;
	};

	/**
	 * @private
	 */
	SELF.prototype._getCloseButton = function() {
		return $( '<a href="#" data-dismiss="clickover">' ).append(
				'<span class="glyphicon glyphicon-remove" aria-hidden="true"></span>' );
	};

	/**
	 * @private
	 */
	SELF.prototype._getToolbar = function( toolbar, $element ) {
		var $toolbar = $( '<span>' );

		$.each( toolbar, function( icon, callback ) {
			var $link = $( '<a>' ).attr( 'href', '#' );
			$link.prepend( '<span class="glyphicon glyphicon-' + icon +
					'" aria-hidden="true"></span>', ' ' );

			$link.click( function() {
				if ( callback() ) {
					$element.click();// close popover
				}

				return false;
			} );
			$toolbar.append( $link, ' ' );
		} );

		return $toolbar;
	};

	/**
	 * @private
	 */
	SELF.prototype._createLookupService = function( $element, triple ) {
		var self = this,
			type = $element.data( 'type' ),
			sparql = $element.data( 'sparql' );

		return function( params, success, failure ) {
			$.when(
					self._searchEntitiesSparql( params.data.term, type, triple, sparql ),
					self._searchEntities( params.data.term, type )
					).done( function ( r1, r2 ) {

					self._addItemMetaData( $element, r1.concat( r2 ) );

					if ( r1.length > 0 ) {
						r1 = [ {
								text: self._i18n( 'suggestions', 'Suggestions' ),
								children: r1
						} ];
					}

					if ( r2.length > 0 &&  r1.length > 0 ) {
						r2 = [ {
							text: self._i18n( 'other', 'Other' ),
							children: r2
						} ];
					}

					success( {
						results: r1.concat( r2 )
					} );
			} );
		};
	};

	/**
	 * @private
	 */
	SELF.prototype._addItemMetaData = function( $element, items ) {
		var data = {};

		items.forEach( function ( item ) {

			if ( !data[item.data.id] ) {
				data[item.data.id] = {};
			}

			data[item.data.id].text = item.text;

			if ( item.data.propertyId ) {
				data[item.data.id].propertyId = item.data.propertyId;
			}

		} );

		$element.data( 'items', data );
	};

	/**
	 * @private
	 * @param {string} term
	 * @param {string} type
	 * @param {Object} triple
	 * @param {string} [sparql]
	 * @return {string}
	 */
	SELF.prototype._getSparqlTemplate = function( term, type, triple, sparql ) {
		var definition = this._getSparqlTemplateDefinition( term, type, triple, sparql ),
			template =  typeof definition === 'function' ? definition.apply( this ) : definition;

		if ( sparql ) {
			template = template.replace( '{SPARQL}', sparql );
		}

		return template;
	};

	/**
	 * @private
	 * @return {string|Function|null}
	 */
	SELF.prototype._getSparqlTemplateDefinition = function( term, type, triple, sparql ) {
		if ( sparql ) {
			if ( term ) {
				return SPARQL_QUERY.sparql.search;
			}

			return SPARQL_QUERY.sparql.suggest;
		}

		var query = SPARQL_QUERY[ type ];
		if ( term && term.trim() !== '' ) {
			if ( !triple ) {
				return null;
			}

			return query.search;

		} else {
			if ( type === 'property' ) {
				if ( !triple  ) {
					return query.genericSuggest;
				}

				if ( triple.object. indexOf( '?' ) === 0  ) {
					return query.seeAlso;
				}
			} else {
				if ( !triple ) {
					return query.genericSuggest;
				}
			}
			return query.suggest;
		}

	};

	/**
	 * @private
	 */
	SELF.prototype._searchEntitiesSparqlCreateQuery = function( term, type, triple, sparql ) {

		var query = this._getSparqlTemplate( term, type, triple, sparql );

		function findFirstStringProperty( predicate ) {
			if ( typeof predicate === 'string' ) {
				return predicate;
			} else {
				return findFirstStringProperty( predicate.items[0] );
			}
		}

		if ( query ) {
			if ( triple ) {
				query = query
						.replace( '{PROPERTY_URI}', findFirstStringProperty( triple.predicate ) )
						.replace( '{ITEM_URI}', triple.object );
			}
			if ( term ) {
				query = query.replace( '{TERM}', term );
			}

			query = query.replace( /\{LANGUAGE\}/g, $.i18n && $.i18n().locale || 'en' );
		}

		return query;
	};

	/**
	 * @private
	 */
	SELF.prototype._searchEntitiesSparql = function( term, type, triple, sparql ) {
		var deferred = $.Deferred();

		var query = this._searchEntitiesSparqlCreateQuery( term, type, triple, sparql );
		if ( !query ) {
			return deferred.resolve( [] ).promise();
		}

		this._sparqlApi.query( query, SPARQL_TIMEOUT ).done( function( data ) {
			var r = data.results.bindings.map( function( d ) {
				var id = d.id.value.split( '/' ).pop();
				var propertyId = d.property && d.property.value.split( '/' ).pop() || null;
				return {
					id: id,
					text: d.label.value,
					data: {
						id: id,
						propertyId: propertyId,
						description: d.description && d.description.value || ''
					}
				};
			} );

			deferred.resolve( r );
		} ).always( function () {
			deferred.resolve( [] );
		} );

		return deferred.promise();
	};

	/**
	 * @private
	 */
	SELF.prototype._searchEntities = function( term, type ) {
		var deferred = $.Deferred();

		if ( !term || term.trim() === '' ) {
			return deferred.resolve( [] ).promise();
		}

		this._api.searchEntities( term, type ).done( function( data ) {
			var r = data.search.map( function( d ) {
				return {
					id: d.id,
					text: d.label,
					data: d
				};
			} );
			deferred.resolve( r );
		} );

		return deferred.promise();
	};

	/**
	 * @private
	 */
	SELF.prototype._renderSelect2 = function( $select, $element, triple ) {
		var formatter = function( item, li ) {
				if ( !item.data ) {
					return item.text;
				}

				return $( '<span><b>' + item.text + ' (' + item.data.id + ')' + '</b></span><br/><small>' +
						item.data.description + '</small>' );

			},
			transport = this._createLookupService( $element, triple );

		$select.select2( {
			width: 'auto',
			templateResult: formatter,
			ajax: {
				delay: 250,
				transport: transport
			},
			cache: true
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._i18n = function( key, defaultMessage ) {
		if ( !$.i18n ) {
			return defaultMessage;
		}

		return $.i18n( I18N_PREFIX + '-' + key );
	};

	return SELF;
}( jQuery, wikibase ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.queryHelper = wikibase.queryService.ui.queryHelper || {};

wikibase.queryService.ui.queryHelper.QueryTemplate = ( function( $, wikibase ) {
	'use strict';

	/**
	 * A template for a SPARQL query
	 *
	 * @class wikibase.queryService.ui.queryHelper.QueryTemplate
	 * @license GNU GPL v2+
	 *
	 * @author Lucas Werkmeister
	 * @constructor
	 */
	function SELF() {
	}

	/**
	 * @property {Object} The parsed template definition.
	 * @private
	 */
	SELF.prototype._definition = {};

	/**
	 * @property {jQuery} A span with the rendered template.
	 * @private
	 */
	SELF.prototype._template = null;

	/**
	 * @property {Object.<string, Array.<jQuery>>} A map from variable names to lists of spans corresponding to that variable.
	 * @private
	 */
	SELF.prototype._variables = [];

	/**
	 * @param {SparqlQuery} query
	 * @return {?QueryTemplate}
	 */
	SELF.parse = function( query ) {
		var templateComment = query.getCommentContent( 'TEMPLATE=' ),
			templateJson,
			template;

		if ( !templateComment ) {
			return null;
		}
		try {
			templateJson = JSON.parse( templateComment );
		} catch ( e ) {
			return null;
		}

		template = new SELF;
		template._definition = templateJson;
		template._fragments = SELF._getQueryTemplateFragments( templateJson );

		return template;
	};

	/**
	 * Splits the template 'a ?b c ?d e' into
	 * [ 'a ', '?b', ' c ', '?d', ' e' ].
	 * Text and variable fragments always alternate,
	 * and the first and last fragment are always text fragments
	 * ('' if the template begins or ends in a variable).
	 *
	 * @param {{template: string, variables: string[]}} definition
	 * @return {string[]}
	 */
	SELF._getQueryTemplateFragments = function( definition ) {
		if ( definition.template.match( /\0/ ) ) {
			throw new Error( 'query template must not contain null bytes' );
		}
		var fragments = [ definition.template ],
			variable,
			newFragments;

		function splitFragment( fragment ) {
			var textFragments = fragment
				.replace( new RegExp( '\\' + variable, 'g' ), '\0' )
				.split( '\0' );
			newFragments.push( textFragments[0] );
			for ( var i = 1; i < textFragments.length; i++ ) {
				newFragments.push( variable );
				newFragments.push( textFragments[ i ] );
			}
		}

		for ( variable in definition.variables ) {
			if ( !variable.match( /\?[a-z][a-z0-9]*/i ) ) {
				// TODO this is more restrictive than SPARQL;
				// see https://www.w3.org/TR/sparql11-query/#rVARNAME
				throw new Error( 'invalid variable name in query template' );
			}
			newFragments = [];
			fragments.forEach( splitFragment );
			fragments = newFragments;
		}

		return fragments;
	};

	/**
	 * Assemble the template span out of the fragments.
	 *
	 * @param {string[]} fragments The template fragments (see {@link _getQueryTemplateFragments}).
	 * @param {Object.<string, jQuery>} variables The individual variables are stored in this object, indexed by variable name.
	 * @return {jQuery}
	 */
	SELF._buildTemplate = function( fragments, variables ) {
		var template = $( '<span>' );

		template.append( document.createTextNode( fragments[ 0 ] ) );
		for ( var i = 1; i < fragments.length; i += 2 ) {
			var variable = fragments[ i ],
				$variable = $( '<span>' ).text( variable );
			if ( !( variable in variables ) ) {
				variables[variable] = [];
			}
			variables[variable].push( $variable );
			template.append( $variable );
			template.append( document.createTextNode( fragments[ i + 1 ] ) );
		}

		return template;
	};

	/**
	 * @param {Function} getLabel Called with {string} variable name, should return {Promise} for label, id, description, type.
	 * @param {wikibase.queryService.ui.queryHelper.SelectorBox} selectorBox
	 * @param {Function} changeListener Called with {string} variable name, {string} old value, {string} new value.
	 * @return {jQuery}
	 */
	SELF.prototype.getHtml = function( getLabel, selectorBox, changeListener ) {
		if ( this._template !== null ) {
			return this._template;
		}

		this._template = SELF._buildTemplate( this._fragments, this._variables );

		var self = this;

		$.each( this._definition.variables, function( variable, variableDefinition ) {
			getLabel( variable ).done( function( label, id, description, type ) {
				$.each( self._variables[ variable ], function( index, $variable ) {
					$variable.text( '' );
					var $link = $( '<a>' ).text( label ).attr( {
						'data-id': id,
						'data-type': type,
						href: '#'
					} ).appendTo( $variable );

					if ( variableDefinition.query ) {
						$link.attr( 'data-sparql', variableDefinition.query );
					}

					selectorBox.add( $link, null, function( selectedId, name ) {
						for ( var j in self._variables[ variable ] ) {
							var $variable = self._variables[ variable ][ j ];
							$variable.find( 'a[data-id="' + id + '"]' )
								.attr( 'data-id', selectedId )
								.text( name );
							changeListener( variable, id, selectedId );
							id = selectedId;
						}
					} );
				} );
			} );
		} );

		return this._template;
	};

	return SELF;
}( jQuery, wikibase ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};

wikibase.queryService.ui.QueryExampleDialog = ( function( $ ) {
	'use strict';

	var TRACKING_NAMESPACE = 'wikibase.queryService.ui.examples.';

	/**
	 * A ui dialog for selecting a query example
	 *
	 * @class wikibase.queryService.ui.QueryExampleDialog
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @author Florian Rämisch, <raemisch@ub.uni-leipzig.de>
	 * @constructor
	 *
	 * @param {jQuery} $element
	 * @param {wikibase.queryService.api.QuerySamples} querySamplesApi
	 * @param {Function} callback that is called when selecting an example
	 */
	function SELF( $element, querySamplesApi, callback, previewUrl ) {

		this._$element = $element;
		this._querySamplesApi = querySamplesApi;
		this._callback = callback;
		this._previewUrl = previewUrl;

		this._init();
	}

	/**
	 * @property {wikibase.queryService.api.QuerySamples}
	 * @private
	 */
	SELF.prototype._querySamplesApi = null;

	/**
	 * @property {wikibase.queryService.api.Wikibase}
	 * @private
	 */
	SELF.prototype._wikibaseApi = null;

	/**
	 * @property {Function}
	 * @private
	 */
	SELF.prototype._callback = null;

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._previewUrl = null;

	/**
	 * @property {Function}
	 * @private
	 */
	SELF.prototype._examples = null;

	/**
	 * @property {wikibase.queryService.api.Tracking}
	 * @private
	 */
	SELF.prototype._trackingApi = null;

	/**
	 * Initialize private members and call delegate to specific init methods
	 *
	 * @private
	 */
	SELF.prototype._init = function() {
		if ( !this._querySamplesApi ) {
			this._querySamplesApi = new wikibase.queryService.api.QuerySamples();
		}

		if ( !this._trackingApi ) {
			this._trackingApi = new wikibase.queryService.api.Tracking();
		}

		this._wikibaseApi = new wikibase.queryService.api.Wikibase();
		this._wikibaseApi.setLanguage( this._querySamplesApi.getLanguage() );

		this._initFilter();
		this._initExamples();

		var self = this;
		this._$element.focus( function() {
			self._$element.find( '.tableFilter' ).focus();
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._initFilter = function() {
		var self = this;

		this._$element.find( '.tableFilter' ).keyup( $.proxy( this._filterTable, this ) );

		// tags
		this._$element.find( '.tagFilter' ).tags( {
			afterAddingTag: $.proxy( this._filterTable, this ),
			afterDeletingTag: function() {
				self._filterTable();
				self._drawTagCloud( true );
			}
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._initExamples = function() {
		var self = this,
			category = null;

		this._querySamplesApi.getExamples().then( function( examples ) {
			self._examples = examples;
			self._initTagCloud();
			self._updateExamplesCount( examples.length );

			$.each( examples, function( key, example ) {
				if ( example.category !== category ) {
					category = example.category;
					self._$element.find( '.searchable' ).append( $( '<tr>' ).addClass( 'active' )
							.append( $( '<td colspan="4">' ).text( category ) ) );
				}
				self._addExample( example.title, example.query, example.href, example.tags, category );
			} );
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._initTagCloud = function() {
		var self = this;

		var interval = window.setInterval( function() {
			if ( self._$element.is( ':visible' ) ) {
				self._drawTagCloud();
				clearInterval( interval );
			}
		}, 300 );
	};

	/**
	 * @private
	 */
	SELF.prototype._drawTagCloud = function( redraw ) {
		var self = this,
			jQCloudTags = [];

		this._getCloudTags().then( function ( tags ) {
			var tagCloud = $( '.tagCloud' );
			$.each( tags, function ( i, tag ) {
				var label = tag.label + ' (' + tag.id + ')';

				jQCloudTags.push( {
					text: label,
					weight: tag.weight,
					link: '#',
					html: {
						title: '(' + tag.weight + ')',
						'data-id': tag.id
					},
					handlers: {
						click: function ( e ) {
							self._$element.find( '.tagFilter' ).tags().addTag( $( this ).text() );
							self._drawTagCloud( true );
							return false;
						}
					}
				} );
			} );

			if ( redraw ) {
				tagCloud.jQCloud( 'update', jQCloudTags );
				return;
			}

			tagCloud.jQCloud( jQCloudTags, {
				delayedMode: true,
				autoResize: true
			} );

		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._getCloudTags = function() {
		var self = this,
			filterTags = self._$element.find( '.tagFilter' ).tags().getTags();

		// filter tags that don't effect the filter for examples
		var tagsFilter = function ( tags ) {
			return filterTags.every( function ( selectedTag ) {
				return tags.indexOf( selectedTag.match( /\((.*)\)/ )[1] ) !== -1;
			} );
		};

		// filter selected tags from tag cloud
		var tagFilter = function ( tag ) {
			var selectedTags = filterTags.map(
					function ( v ) {
						return v.match( /\((.*)\)/ )[1];
					} );

			return selectedTags.indexOf( tag ) !== -1;
		};

		var tagCloud = {};
		$.each( self._examples, function( key, example ) {
			if ( !tagsFilter( example.tags ) ) {
				return;
			}

			$.each( example.tags, function( key, tag ) {
				if ( tagFilter( tag ) ) {
					return;
				}

				if ( !tagCloud[tag] ) {
					tagCloud[tag] = { id: tag, weight: 1 };
				} else {
					tagCloud[tag].weight++;
				}
			} );
		} );

		tagCloud = _.compact( tagCloud ).sort( function ( a, b ) {
			return b.weight - a.weight;
		} ).slice( 0, 50 );

		return this._wikibaseApi.getLabels( tagCloud.map( function ( v ) {
			return v.id;
		} ) ).then( function ( data ) {
			tagCloud.forEach( function ( tag ) {
				tag.label = _.compact( data.entities[tag.id].labels )[0].value;
			} );
			return tagCloud;
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._addExample = function( title, query, href, tags, category ) {
		var self = this,
			$link = $( '<a title="Select" data-dismiss="modal">' ).text( title ).attr( 'href', '#' )
					.click( function() {
						self._callback( query, title );
						self._track( 'select' );
						self._track( 'select.category.' + category.replace( /[^a-zA-Z0-9]/g, '_' ) );
					} ),
			$edit = $( '<a>' )
				.attr( { title: 'Edit', href: href, target: '_blank' } )
				.append( '<span>' ).addClass( 'glyphicon glyphicon-pencil' )
				.click( function() {
					self._track( 'edit' );
				} ),

			$source = $( '<span>' ).addClass( 'glyphicon glyphicon-eye-open' ).popover(
				{
					placement: 'bottom',
					trigger: 'hover',
					container: 'body',
					title: self._i18n( 'wdqs-dialog-examples-preview-query', 'Preview query' ),
					content: $( '<pre style="white-space:pre-line; word-break:normal;"/>' ).text( query ),
					html: true
				} ),
			$preview = $( '<a href="#">' ).addClass( 'glyphicon glyphicon-camera' ).clickover(
				{
					placement: 'left',
					'global_close': true,
                    'esc_close': true,
					trigger: 'click',
					container: 'body',
					title: self._i18n( 'wdqs-dialog-examples-preview-result', 'Preview result' ),
					content: $( '<iframe width="400" height="350" frameBorder="0" src="' +
							( self._previewUrl || 'embed.html#' ) +	encodeURIComponent( query ) + '">' ),
					html: true
				} )
				.click( function() {
					self._track( 'preview' );
				} );

		$( '.exampleTable' ).scroll( function() {
			if ( $preview.clickover ) {
				$preview.clickover( 'hide' ).removeAttr( 'data-clickover-open' );
			}
		} );

		var example = $( '<tr>' );
		example.append( $( '<td>' ).append( $link ).append( ' ', $edit ) );
		example.append( $( '<td>' ).addClass( 'exampleIcons' ).append( $preview ) );
		example.append( $( '<td>' ).addClass( 'exampleIcons' ).append( $source ) );
		example.append( $( '<td>' ).text( tags.join( '|' ) ).hide() );
		example.append( $( '<td>' ).text( query ).hide() );

		this._$element.find( '.searchable' ).append( example );
	};

	/**
	 * @private
	 */
	SELF.prototype._filterTable = function() {
		var filter = this._$element.find( '.tableFilter' ),
			// FIXME: This crashs when the user enters an invalid regex. e.g. ".**".
			filterRegex = new RegExp( filter.val(), 'i' );

		var tags = this._$element.find( '.tagFilter' ).tags().getTags();

		var tagFilter = function( text ) {
			var matches = true;
			text = text.toLowerCase();

			$.each( tags, function( key, tag ) {
				if ( text.indexOf( tag.toLowerCase().match( /\((.*)\)/ )[1] ) === -1 ) {
					matches = false;
				}
			} );

			return matches;
		};

		this._$element.find( '.searchable tr' ).hide();
		var $matchingElements = this._$element.find( '.searchable tr' ).filter( function() {
			return filterRegex.test( $( this ).text() ) && tagFilter( $( this ).text() );
		} );

		$matchingElements.show();
		$matchingElements.each( function( i, el ) {
			$( el ).prevAll( 'tr.active' ).first().show();
		} );
		this._updateExamplesCount( $matchingElements.length );
	};

	/**
	 * @private
	 */
	SELF.prototype._updateExamplesCount = function( count ) {
		this._$element.find( '.count' ).text( count );
	};

	/**
	 * @private
	 */
	SELF.prototype._track = function( metricName, value, valueType ) {
		this._trackingApi.track( TRACKING_NAMESPACE + metricName, value, valueType );
	};

	/**
	 * @private
	 */
	SELF.prototype._i18n = function( key, message ) {
		if ( !$.i18n ) {
			return message;
		}
		var i18nMessage = $.i18n( key );
		return i18nMessage === key ? message : i18nMessage;
	};

	return SELF;
}( jQuery ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.dialog = wikibase.queryService.ui.dialog || {};

wikibase.queryService.ui.dialog.CodeExample = ( function( $, CodeMirror ) {
	'use strict';

	var CODEMIRROR_DEFAULTS = { readOnly: true, autofocus: true, autoRefresh: true, lineNumbers: true, lineWrapping: true };

	/**
	 * A ui dialog for code examples with current query
	 *
	 * @class wikibase.queryService.ui.dialog.CodeExample
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 *
	 * @param {jQuery} $element
	 * @param {Function} getCodeExamples
	 */
	function SELF( $element, getCodeExamples ) {

		this._$element = $element;
		this._getCodeExamples = getCodeExamples;

		this._init();
	}

	/**
	 * @property {jQuery}
	 * @private
	 */
	SELF.prototype._$element = null;

	/**
	 * @property {Function}
	 * @private
	 */
	SELF.prototype._getCodeExamples = null;

	/**
	 * @private
	 */
	SELF.prototype._init = function() {
		var self = this;

		this._$element.focus( function () {
			self._getCodeExamples().then( function ( d ) {
				self._update( d );
			} );
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._update = function( examples ) {
		var $tabs = this._$element.find( '.nav-tabs' ).empty(),
			$panes = this._$element.find( '.tab-content' ).empty(),
			$tab = $( '<li role="presentation"></li>' ),
			$pane = $( '<div role="tabpanel" class="tab-pane">' ),
			$button = $( '<a role="tab" data-toggle="tab">' );

		$.each( examples, function( lang, data ) {
			var code = data.code,
				mode = data.mimetype;
			var $text = $( '<textarea>' ).text( code );
			$tabs.append(
				$tab.clone()
					.addClass( $tabs.is( ':empty' ) ? 'active' : '' )
					.append(
						$button.clone()
							.attr( 'href', '#' + $.escapeSelector( lang ) )
							.text( lang )
					)
			);
			$panes.append(
				$pane.clone()
					.addClass( $panes.is( ':empty' ) ? 'active' : '' )
					.attr( 'id', lang )
					.append( $text )
			);
			CODEMIRROR_DEFAULTS.mode = mode;
			CodeMirror.fromTextArea( $text[0], CODEMIRROR_DEFAULTS );
		} );
	};

  return SELF;
}( jQuery, CodeMirror ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};

wikibase.queryService.ui.ResultView = ( function( $, window ) {
	'use strict';

	var TRACKING_NAMESPACE = 'wikibase.queryService.ui.app.';

	var PREVIEW_TIMEOUT = 1000,
		PREVIEW_LIMIT = 20;

	/**
	 * A result view for sparql queries
	 *
	 * @class wikibase.queryService.ui.ResultView
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 *
	 * @param {wikibase.queryService.api.Sparql} sparqlApi
	 */
	function SELF( sparqlApi ) {
		this._sparqlApi = sparqlApi;

		this._init();
	}

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._query = null;

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._sparqlQuery = null;

	/**
	 * @property {wikibase.queryService.api.Sparql}
	 * @private
	 */
	SELF.prototype._sparqlApi = null;

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._selectedResultBrowser = null;

	/**
	 * @property {wikibase.queryService.ui.toolbar.Actionbar}
	 * @private
	 */
	SELF.prototype._actionBar = null;

	/**
	 * @property {wikibase.queryService.api.Tracking}
	 * @private
	 */
	SELF.prototype._trackingApi = null;

	/**
	 * @property {boolean}
	 * @private
	 */
	SELF.prototype._hasRunFirstQuery = false;

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._resultBrowsers = {
		Table: {
			icon: 'glyphicon-th-list',
			label: 'Table',
			class: 'TableResultBrowser',
			object: null,
			$element: null
		},
		ImageGrid: {
			icon: 'glyphicon-picture',
			label: 'Image Grid',
			class: 'ImageResultBrowser',
			object: null,
			$element: null
		},
		Polestar: {
			icon: 'fa-certificate',
			label: 'Graph builder',
			class: 'PolestarResultBrowser',
			object: null,
			$element: null
		},
		Map: {
			icon: 'glyphicon-map-marker',
			label: 'Map',
			class: 'CoordinateResultBrowser',
			object: null,
			$element: null
		},
		LineChart: {
			icon: 'fa-line-chart',
			label: 'Line Chart',
			class: 'LineChartResultBrowser',
			object: null,
			$element: null
		},
		BarChart: {
			icon: 'fa-bar-chart',
			label: 'Bar Chart',
			class: 'BarChartResultBrowser',
			object: null,
			$element: null
		},
		ScatterChart: {
			icon: 'fa-braille',
			label: 'Scatter Chart',
			class: 'ScatterChartResultBrowser',
			object: null,
			$element: null
		},
		AreaChart: {
			icon: 'fa-area-chart',
			label: 'Area Chart',
			class: 'AreaChartResultBrowser',
			object: null,
			$element: null
		},
		BubbleChart: {
			icon: 'glyphicon-tint',
			label: 'Bubble Chart',
			class: 'BubbleChartResultBrowser',
			object: null,
			$element: null
		},
		TreeMap: {
			icon: 'glyphicon-th',
			label: 'Tree Map',
			class: 'TreeMapResultBrowser',
			object: null,
			$element: null
		},
		Tree: {
			icon: 'fa-tree',
			label: 'Tree',
			class: 'TreeResultBrowser',
			object: null,
			$element: null
		},
		Timeline: {
			icon: 'glyphicon-calendar',
			label: 'Timeline',
			class: 'TimelineResultBrowser',
			object: null,
			$element: null
		},
		Dimensions: {
			icon: 'glyphicon-random',
			label: 'Dimensions',
			class: 'MultiDimensionResultBrowser',
			object: null,
			$element: null
		},
		Graph: {
			icon: 'glyphicon-retweet',
			label: 'Graph',
			class: 'GraphResultBrowser',
			object: null,
			$element: null
		}
	};

	/**
	 * Initialize private members and call delegate to specific init methods
	 *
	 * @private
	 */
	SELF.prototype._init = function() {
		if ( !this._sparqlApi ) {
			this._sparqlApi = new wikibase.queryService.api.Sparql();
		}

		if ( !this._trackingApi ) {
			this._trackingApi = new wikibase.queryService.api.Tracking();
		}

		this._actionBar = new wikibase.queryService.ui.toolbar.Actionbar( $( '.action-bar' ) );

		this._sparqlQuery = this._query = new wikibase.queryService.ui.queryHelper.SparqlQuery();

		this._initResultBrowserMenu();
	};

	/**
	 * @private
	 */
	SELF.prototype._initResultBrowserMenu = function() {
		$.each( this._resultBrowsers, function( key, b ) {
			var $element = $( '<li><a class="result-browser" href="#">' +
					'<span class="' + b.icon.split( '-', 1 )[0] + ' ' + b.icon + '"></span>' + b.label +
					'</a></li>' );
			$element.appendTo( $( '#result-browser-menu' ) );
			b.$element = $element;
		} );
	};

	/**
	 * Render a given SPARQL query
	 *
	 * @param {String} query
	 * @return {JQuery.Promise}
	 */
	SELF.prototype.draw = function( query ) {
		var self = this,
			deferred = $.Deferred();

		this._query = query;

		this._actionBar.show( 'wdqs-action-query', '', 'info', 100 );

		$( '#query-result' ).empty().hide();
		$( '.result' ).hide();
		$( '#query-error' ).hide();

		this._sparqlApi.query( query )
			.done( function () {
				self._handleQueryResult();
				deferred.resolve();
			} )
			.fail( function() {
				var error = self._handleQueryError();
				deferred.reject( error );
			} );

		return deferred.promise();
	};

	/**
	 * Render a preview of the given SPARQL query
	 *
	 * @param {String} query
	 * @return {JQuery.Promise}
	 */
	SELF.prototype.drawPreview = function( query ) {
		var self = this,
			deferred = $.Deferred(),
			prefixes = wikibase.queryService.RdfNamespaces.ALL_PREFIXES,
			previousQueryString = this._sparqlQuery.getQueryString();

		this._query = query;
		this._sparqlQuery.parse( query, prefixes );
		this._sparqlQuery.setLimit( PREVIEW_LIMIT );

		if ( previousQueryString === this._sparqlQuery.getQueryString() ) {
			return deferred.reject().promise();
		}

		$( '#query-result' ).empty().hide();
		$( '.result' ).hide();
		$( '#query-error' ).hide();
		this._actionBar.hide();

		this._sparqlApi.query( this._sparqlQuery.getQueryString(), PREVIEW_TIMEOUT )
			.done( function () {
				self._handleQueryResult();
				deferred.resolve();
				window.setTimeout( function() {
					self._actionBar.show( 'wdqs-action-preview', '', 'default' );
				}, 200 );
			} )
			.fail( function() {
				deferred.reject();
			} );

		return deferred.promise();
	};

	/**
	 * @private
	 */
	SELF.prototype._handleQueryError = function() {
		$( '#execute-button' ).prop( 'disabled', false );

		var error = this._sparqlApi.getError(),
			errorMessageKey = null,
			codes = this._sparqlApi.ERROR_CODES;

		switch ( error.code ) {
		case codes.TIMEOUT:
			errorMessageKey = 'wdqs-action-timeout';
			break;
		case codes.MALFORMED:
			errorMessageKey = 'wdqs-action-malformed-query';
			break;
		case codes.SERVER:
			errorMessageKey = 'wdqs-action-server-error';
			break;
		default:
			errorMessageKey = 'wdqs-action-unknow-error';
			break;
		}

		if ( error.debug ) {
			$( '#query-error' ).html( $( '<pre>' ).text( error.debug ) ).show();
		}

		this._actionBar.show( errorMessageKey || '', error.message || '', 'danger' );
		this._track( 'result.error.' + ( errorMessageKey || 'unknown' ) );

		return error.debug === undefined ? '' : error.debug;
	};

	/**
	 * @private
	 */
	SELF.prototype._handleQueryResult = function() {
		var api = this._sparqlApi;

		$( '#total-results' ).text( api.getResultLength() );
		$( '#query-time' ).text( api.getExecutionTime() );
		$( '.result' ).show();

		$( '#execute-button' ).prop( 'disabled', false );

		var defaultBrowser = this._createResultBrowsers( api.getResultRawData() );
		this._drawResult( defaultBrowser );
		this._selectedResultBrowser = null;

		this._track( 'result.resultLength', api.getResultLength() );
		this._track( 'result.executionTime', api.getExecutionTime(), 'ms' );
		this._track( 'result.received.success' );

		return false;
	};

	/**
	 * @private
	 * @return {Object} default result browser
	 */
	SELF.prototype._createResultBrowsers = function( resultData ) {
		var self = this;

		var defaultBrowser = this._getDefaultResultBrowser();

		this._track( 'result.browser.' + ( defaultBrowser || 'default' ) );

		// instantiate
		$.each( this._resultBrowsers, function( key, b ) {
			var instance = new wikibase.queryService.ui.resultBrowser[b.class]();
			instance.setSparqlApi( self._sparqlApi );

			if ( defaultBrowser === null || defaultBrowser === key ) {
				self._setSelectedDisplayType( b );

				defaultBrowser = instance;
			}
			b.object = instance;
		} );

		defaultBrowser.resetVisitors();

		// wire up
		$.each( this._resultBrowsers, function( key, b ) {
			defaultBrowser.addVisitor( b.object );
			b.object.setResult( resultData );
		} );

		return defaultBrowser;
	};

	/**
	 * @private
	 */
	SELF.prototype._getDefaultResultBrowser = function() {
		var match = this._query.match( /#defaultView:(\w+)/ );

		if ( match && this._resultBrowsers[match[1]] ) {
			return match[1];
		}

		return null;
	};

	/**
	 * @private
	 */
	SELF.prototype._handleQueryResultBrowsers = function() {
		var self = this;

		$.each( this._resultBrowsers, function( key, b ) {
			b.$element.off( 'click' );

			if ( b.object.isDrawable() ) {
				b.$element.css( 'opacity', 1 ).attr( 'href', '#' );
				b.$element.click( function() {
					$( this ).closest( '.open' ).removeClass( 'open' );

					self._setSelectedDisplayType( b );

					$( '#query-result' ).html( '' );
					self._drawResult( b.object );
					self._selectedResultBrowser = key;
					self._track( 'buttonClick.display.' + key );
					return false;
				} );
			} else {
				b.$element.css( 'opacity', 0.5 ).removeAttr( 'href' );
			}
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._drawResult = function( resultBrowser ) {
		var self = this;

		this._actionBar.show( 'wdqs-action-render', '',  'success', 100 );
		window.setTimeout( function() {
			try {
				$( '#query-result' ).show();
				resultBrowser.draw( $( '#query-result' ) );
				self._actionBar.hide();
			} catch ( e ) {
				self._actionBar.show( 'wdqs-action-error-display', '', 'warning' );
				window.console.error( e );
			}

			self._handleQueryResultBrowsers();
		}, 20 );
	};

	/**
	 * @private
	 */
	SELF.prototype._setSelectedDisplayType = function ( browser ) {
		$( '#display-button-icon' ).attr( 'class', browser.icon.split( '-', 1 )[0] + ' ' + browser.icon );
		$( '#display-button-label' ).text( browser.label );
	};

	/**
	 * @private
	 */
	SELF.prototype._track = function( metricName, value, valueType ) {
		this._trackingApi.track( TRACKING_NAMESPACE + metricName, value, valueType );
	};

	return SELF;

}( jQuery, window ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};
wikibase.queryService.ui.resultBrowser.helper = wikibase.queryService.ui.resultBrowser.helper || {};

wikibase.queryService.ui.resultBrowser.helper.FormatterHelper = ( function( $, moment ) {
	'use strict';

	var COMMONS_FILE_PATH = 'http://commons.wikimedia.org/wiki/special:filepath/',
		COMMONS_FILE_PATH_MEDIAVIEWER = 'https://commons.wikimedia.org/wiki/File:{FILENAME}',
		DATATYPE_DATETIME = 'http://www.w3.org/2001/XMLSchema#dateTime',
		TYPE_URI = 'uri',
		DATATYPE_MATHML = 'http://www.w3.org/1998/Math/MathML';

	var NUMBER_TYPES = [
			'http://www.w3.org/2001/XMLSchema#double', 'http://www.w3.org/2001/XMLSchema#float',
			'http://www.w3.org/2001/XMLSchema#decimal', 'http://www.w3.org/2001/XMLSchema#integer',
			'http://www.w3.org/2001/XMLSchema#long', 'http://www.w3.org/2001/XMLSchema#int',
			'http://www.w3.org/2001/XMLSchema#short',
			'http://www.w3.org/2001/XMLSchema#nonNegativeInteger',
			'http://www.w3.org/2001/XMLSchema#positiveInteger',
			'http://www.w3.org/2001/XMLSchema#unsignedLong',
			'http://www.w3.org/2001/XMLSchema#unsignedInt',
			'http://www.w3.org/2001/XMLSchema#unsignedShort',
			'http://www.w3.org/2001/XMLSchema#nonPositiveInteger',
			'http://www.w3.org/2001/XMLSchema#negativeInteger'
	];

	/**
	 * Formatting helper provides methods useful for formatting results
	 *
	 * @class wikibase.queryService.ui.resultBrowser.helper.FormatterHelper
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 */
	function SELF() {
	}

	/**
	 * Format a data row
	 *
	 * @param {Object} row
	 * @param {boolean} embed media files
	 * @return {jQuery} element
	 */
	SELF.prototype.formatRow = function( row, embed ) {
		var self = this,
			$result = $( '<div/>' );

		$.each( row, function( key, value ) {
			if ( self._isLabelField( key, row ) ) {
				return;
			}

			value = $.extend( {
				label: self._getLabel( row, key )
			}, value );
			$result.prepend( $( '<div>' ).append( self.formatValue( value, key, embed ) ) );
		} );

		return $result;
	};

	/**
	 * @param {string} key
	 * @param {object} row
	 * @return {boolean}
	 * @private
	 */
	SELF.prototype._isLabelField = function( key, row ) {
		return key.endsWith( 'Label' ) && typeof row[key.slice( 0, -5 )] !== 'undefined';
	};

	/**
	 * @param {object} row
	 * @param {string} key
	 * @return {string|null}
	 * @private
	 */
	SELF.prototype._getLabel = function( row, key ) {
		var field = row[key + 'Label'];
		return field && field.value || null;
	};

	/**
	 * Format a data value
	 *
	 * @param {Object} data
	 * @param {string} [title]
	 * @param {boolean} [embed] media files
	 * @return {jQuery} element
	 */
	SELF.prototype.formatValue = function( data, title, embed ) {
		var value = data.value,
			$html = $( '<span>' );

		if ( !title ) {
			title = data.dataType || '';
		}

		if ( !data.type ) {
			return $( '<span>' ).text( value ).attr( 'title', title );
		}

		switch ( data.datatype || data.type ) {
		case TYPE_URI:
			var $link = $( '<a>' ).attr( { title: title, href: value, target: '_blank', rel: 'noopener' } );
			$html.append( $link );

			if ( this.isCommonsResource( value ) ) {
				if ( embed ) {
					$link.click( this.handleCommonResourceItem );
					$link.append(
							$( '<img>' ).attr( 'src',
									this.getCommonsResourceFileNameThumbnail( value, '120' ) ) )
							.width( '120' );
				} else {
					$link.attr( { href: COMMONS_FILE_PATH_MEDIAVIEWER.replace( /{FILENAME}/g,
							this.getCommonsResourceFileName( value ) ) } );
					$link.text( 'commons:' +
							decodeURIComponent( this.getCommonsResourceFileName( value ) ) );
					$html.prepend( this.createGalleryButton( value, title ), ' ' );
				}
			} else {
				$link.text( data.label || this.abbreviateUri( value ) );

				if ( this.isEntityUri( value ) ) {
					$html.prepend( this.createExploreButton( value ), ' ' );
				}
			}
			break;
		case DATATYPE_DATETIME:
			if ( !title ) {
				title = this._i18n( 'wdqs-app-result-formatter-title-datetime', 'Raw ISO timestamp' );
			}
			var $dateLabel = $( '<span>' ).text( this._formatDate( value ) );
			$dateLabel.attr( 'title', title + ': ' + value );
			$html.append( $dateLabel );
			break;

		case DATATYPE_MATHML:
			$html.append( $( data.value ) );
			break;

		default:
			var $label = $( '<span>' ).text( value );
			if ( data['xml:lang'] ) {
				$label.attr( 'title', title + ': ' + value + '@' + data['xml:lang'] );
			} else {
				$label.attr( 'title', title );
			}
			$html.append( $label );
		}

		return $html;
	};

	/**
	 * @param {string} dateTime
	 * @return {string}
	 * @private
	 */
	SELF.prototype._formatDate = function( dateTime ) {
		var isBce = false,
			positiveDate = dateTime.replace( /^(?:-\d+|\+?0+\b)/, function( year ) {
				isBce = true;
				return Math.abs( year ) + 1;
			} ),
			moment = this.parseDate( positiveDate ),
			formatted;

		if ( moment.isValid() ) {
			formatted = moment.format( 'll' );
		} else {
			var year = positiveDate.replace( /^\+?(\d+).*/, '$1' );
			formatted = '0000'.slice( year.length ) + year;
		}

		if ( isBce ) {
			// TODO: Translate.
			formatted += ' BCE';
		}

		return formatted;
	};

	/**
	 * Parse dateTime string to Date object
	 * Allows negative years without leading zeros http://www.ecma-international.org/ecma-262/5.1/#sec-15.9.1.15.1
	 *
	 * @param {string} dateTime
	 * @return {Object}
	 */
	SELF.prototype.parseDate = function( dateTime ) {
		// Add leading plus sign if it's missing
		dateTime = dateTime.replace( /^(?![+-])/, '+' );
		// Pad years to 6 digits
		dateTime = dateTime.replace( /^([+-]?)(\d{1,5}\b)/, function( $0, $1, $2 ) {
			return $1 + ( '00000' + $2 ).slice( -6 );
		} );
		// Remove timezone
		dateTime = dateTime.replace( /Z$/, '' );

		return moment( dateTime, moment.ISO_8601 );
	};

	/**
	 * Checks if a given URI appears to be a canonical Wikidata entity URI.
	 *
	 * @param {string} uri
	 * @return {boolean}
	 */
	SELF.prototype.isEntityUri = function( uri ) {
		return typeof uri === 'string'
			&& /^https?:\/\/www\.wikidata\.org\/entity\/./.test( uri );
	};

	/**
	 * Creates an explore button
	 *
	 * @return {jQuery}
	 */
	SELF.prototype.createExploreButton = function( url ) {
		var $button = $( '<a href="' + url +
				'" title="Explore item" class="explore glyphicon glyphicon-search" aria-hidden="true">' );
		$button.click( $.proxy( this.handleExploreItem, this ) );

		return $button;
	};

	/**
	 * Checks whether given url is commons resource URL
	 *
	 * @param {string} url
	 * @return {boolean}
	 */
	SELF.prototype.isCommonsResource = function( url ) {
		return url.toLowerCase().startsWith( COMMONS_FILE_PATH.toLowerCase() );
	};

	/**
	 * Returns the file name of a commons resource URL
	 *
	 * @param {string} url
	 * @return {string}
	 */
	SELF.prototype.getCommonsResourceFileName = function( url ) {
		// FIXME: Dots in the constant must be escaped before using it as a regex!
		var regExp = new RegExp( COMMONS_FILE_PATH, 'ig' );

		return decodeURIComponent( url.replace( regExp, '' ) );
	};

	/**
	 * Creates a thumbnail URL from given commons resource URL
	 *
	 * @param {string} url
	 * @param {number} [width]
	 * @return {string}
	 */
	SELF.prototype.getCommonsResourceFileNameThumbnail = function( url, width ) {
		if ( !this.isCommonsResource( url ) ) {
			return url;
		}

		return url.replace( /^http(?=:\/\/)/, 'https' ) + '?width=' + ( width || 400 );
	};

	/**
	 * Creates a gallery button
	 *
	 * @param {string} url
	 * @param {string} galleryId
	 * @return {jQuery}
	 */
	SELF.prototype.createGalleryButton = function( url, galleryId ) {
		var fileName = this.getCommonsResourceFileName( url ),
			thumbnail = this.getCommonsResourceFileNameThumbnail( url, 900 );

		var $button = $( '<a>' ).attr( {
			title: 'Show Gallery',
			href: thumbnail,
			'aria-hidden': 'true',
			'class': 'gallery glyphicon glyphicon-picture',
			'data-gallery': 'G_' + galleryId,
			'data-title': decodeURIComponent( fileName )
		} );

		$button.click( this.handleCommonResourceItem );

		return $button;
	};

	/**
	 * Produce abbreviation of the URI.
	 *
	 * @param {string} uri
	 * @return {string}
	 */
	SELF.prototype.abbreviateUri = function( uri ) {
		var NAMESPACE_SHORTCUTS = wikibase.queryService.RdfNamespaces.NAMESPACE_SHORTCUTS,
			nsGroup,
			ns,
			length = 0,
			longestNs;

		for ( nsGroup in NAMESPACE_SHORTCUTS ) {
			for ( ns in NAMESPACE_SHORTCUTS[nsGroup] ) {
				if ( uri.indexOf( NAMESPACE_SHORTCUTS[nsGroup][ns] ) === 0 ) {
					if ( NAMESPACE_SHORTCUTS[nsGroup][ns].length > length ) {
						length = NAMESPACE_SHORTCUTS[nsGroup][ns].length;
						longestNs = ns;
					}
				}
			}
		}
		if ( longestNs ) {
			return longestNs + ':' + uri.substr( length );
		} else {
			return '<' + uri + '>';
		}
	};

	/**
	 * Handler for explore links
	 */
	SELF.prototype.handleExploreItem = function( e ) {
		var url = $( e.target ).attr( 'href' );
		e.preventDefault();

		var lang = $.i18n && $.i18n().locale || 'en',
			query = 'SELECT ?item ?itemLabel WHERE { BIND( <' + url + '> as ?item ).	SERVICE wikibase:label { bd:serviceParam wikibase:language "' + lang + '" } }',
			embedUrl = 'embed.html#' + encodeURIComponent( '#defaultView:Graph\n' + query );

		$( '.explorer-panel .panel-body' ).html( $( '<iframe frameBorder="0" scrolling="no"></iframe>' ).attr( 'src', embedUrl ) );
		$( '.explorer-panel' ).show();

		return false;
	};

	/**
	 * Handler for commons resource links
	 */
	SELF.prototype.handleCommonResourceItem = function( e ) {
		e.preventDefault();

		$( this ).ekkoLightbox( {
			'scale_height': true
		} );
	};

	/**
	 * Checks whether the current cell contains a label:
	 * Has either a language property, or has the "literal" type without a datatype.
	 *
	 * @param {Object} cell
	 * @return {boolean}
	 */
	SELF.prototype.isLabel = function( cell ) {
		if ( !cell ) {
			return false;
		}

		return 'xml:lang' in cell
			|| ( cell.type === 'literal' && !cell.datatype );
	};

	/**
	 * Checks whether the current cell contains a number
	 *
	 * @param {Object} cell
	 * @return {boolean}
	 */
	SELF.prototype.isNumber = function( cell ) {
		return cell
			&& cell.datatype
			&& NUMBER_TYPES.indexOf( cell.datatype ) !== -1;
	};

	/**
	 * Checks whether the current cell is date time
	 *
	 * @param {Object} cell
	 * @return {boolean}
	 */
	SELF.prototype.isDateTime = function( cell ) {
		return cell
			&& cell.datatype === DATATYPE_DATETIME;
	};

	/**
	 * Checks whether the current cell is a WD entity URI
	 *
	 * @param {Object} cell
	 * @return {boolean}
	 */
	SELF.prototype.isEntity = function( cell ) {
		return cell
			&& cell.value
			&& this.isEntityUri( cell.value );
	};

	/**
	 * Checks whether the current cell is a color string according to the P465 format
	 *
	 * @param {Object} cell
	 * @return {boolean}
	 */
	SELF.prototype.isColor = function ( cell ) {
		return cell
			&& cell.type === 'literal'
			&& cell.value
			&& /^([\dA-F]{1,2}){3}$/i.test( cell.value );
	};

	/**
	 * Returns an HTML color string for the current cell
	 *
	 * @param {Object} cell
	 * @return {string}
	 */
	SELF.prototype.getColorForHtml = function ( cell ) {
		return '#' + cell.value;
	};

	/**
	 * Calculate the luminance of the given sRGB color.
	 *
	 * This uses the reverse sRGB transformation,
	 * as documented on https://en.wikipedia.org/wiki/SRGB#The_reverse_transformation,
	 * to calculate the luminance (Y coordinate in the CIE XYZ model).
	 *
	 * @param {string} color as six hex digits (no #)
	 * @return {Number} luminance of the color, or NaN if the color string is invalid
	 */
	SELF.prototype.calculateLuminance = function( color ) {
		var r = parseInt( color.substr( 1, 2 ), 16 ) / 255,
			g = parseInt( color.substr( 3, 2 ), 16 ) / 255,
			b = parseInt( color.substr( 5, 2 ), 16 ) / 255;

		if ( isFinite( r ) && isFinite( g ) && isFinite( b ) ) {
			// linearize gamma-corrected sRGB values
			r = r <= 0.04045 ? r / 12.92 : Math.pow( ( r + 0.055 ) / 1.055, 2.4 );
			g = g <= 0.04045 ? g / 12.92 : Math.pow( ( g + 0.055 ) / 1.055, 2.4 );
			b = b <= 0.04045 ? b / 12.92 : Math.pow( ( b + 0.055 ) / 1.055, 2.4 );
			// calculate luminance using Rec. 709 coefficients
			var luminance = 0.2126 * r + 0.7152 * g + 0.0722 * b;
			return luminance;
		} else {
			return NaN;
		}
	};

	/**
	 * Get an i18n string
	 *
	 * @param {string} key for the i18n message
	 * @param {string} message default text
	 * @return {string}
	 * @private
	 */
	SELF.prototype._i18n = function( key, message ) {
		var i18nMessage;

		if ( !$.i18n || ( i18nMessage = $.i18n( key ) ) === key ) {
			return message;
		}

		return i18nMessage;
	};

	return SELF;
}( jQuery, moment ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.AbstractResultBrowser = ( function( $, wikibase ) {
	'use strict';

	/**
	 * Abstract result browser
	 *
	 * @class wikibase.queryService.ui.resultBrowser.AbstractResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 */
	function SELF() {
		this._visitors = [];
	}

	/**
	 * @property {wikibase.queryService.ui.resultBrowser.helper.FormatterHelper}
	 * @private
	 */
	SELF.prototype._formatter = null;

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._result = null;

	/**
	 * @property {Function}
	 * List of visitor callbacks
	 */
	SELF.prototype._visitors = null;

	/**
	 * @protected
	 * @property {boolean}
	 * Is the browser drawable?
	 * Not drawable by default.
	 */
	SELF.prototype._drawable = false;

	/**
	 * @property {wikibase.queryService.api.Sparql}
	 * @private
	 */
	SELF.prototype._sparqlApi = null;

	/**
	 * Sets the result to be browsed
	 *
	 * @param {Object} result set
	 */
	SELF.prototype.setResult = function( result ) {
		this._result = result;
	};

	/**
	 * Sets the SPARQL API
	 *
	 * @param {wikibase.queryService.api.Sparql} sparqlApi
	 */
	SELF.prototype.setSparqlApi = function( sparqlApi ) {
		this._sparqlApi = sparqlApi;
	};

	/**
	 * Gets the SPARQL API
	 *
	 * @return {wikibase.queryService.api.Sparql}
	 */
	SELF.prototype.getSparqlApi = function() {
		return this._sparqlApi;
	};

	/**
	 * Iterate the result set and calls the visitors
	 *
	 * @protected
	 * @param {AbstractResultBrowser~resultCallback} cb - called for every column of the resultset
	 */
	SELF.prototype._iterateResult = function( cb ) {
		var self = this;

		$.each( this._result.results.bindings, function( rowNum, row ) {
			$.each( self._result.head.vars, function( rowNum1, key ) {
				var field = row[key] || null;
				self.processVisitors( field, key );
				cb( field, key, row, rowNum );
			} );
		} );
	};

	/**
	 * Callback used by _iterateResult
	 * @param {Object} field
	 * @param {string} key of the field
	 * @param {Object} row
	 */

	/**
	 * Checks whether the result browser can draw the given result
	 *
	 * @return {boolean}
	 */
	SELF.prototype.isDrawable = function() {
		return this._drawable;
	};

	/**
	 * Draws the result browser to the given element
	 *
	 * @param {jQuery} $element to draw at
	 */
	SELF.prototype.draw = function( $element ) {
		jQuery.error( 'Method draw() needs to be implemented!' );
	};

	/**
	 * Add visitor function.
	 *
	 * @param {Function} callback
	 */
	SELF.prototype.addVisitor = function( callback ) {
		this._visitors.push( callback );
	};

	/**
	 * Reset visitors array.
	 */
	SELF.prototype.resetVisitors = function() {
		this._visitors = [];
	};

	/**
	 * Call all visitors for the piece of data
	 *
	 * @protected
	 * @param {Object} data
	 * @param {string} columnKey
	 */
	SELF.prototype.processVisitors = function( data, columnKey ) {
		var self = this,
			removeVisitors = {};

		if ( this._visitors.length === 0 ) {
			return;
		}

		$.each( this._visitors, function( key, v ) {
			if ( v.visit && typeof v.visit === 'function' ) {
				if ( v.visit( data, columnKey ) === false ) {
					removeVisitors[key] = true;
				}
			}
		} );

		// need to use filter since removal changes keys
		self._visitors = self._visitors.filter( function( value, visitorIndex ) {
			return !removeVisitors[visitorIndex];
		} );
	};

	/**
	 * Receiving data from the a visit
	 *
	 * @param {Object} data
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( data ) {
		return false;
	};

	/**
	 * Set a formatter in order to replace the default formatter
	 *
	 * @param {wikibase.queryService.ui.resultBrowser.helper.FormatterHelper} formatter
	 */
	SELF.prototype.setFormatter = function( formatter ) {
		this._formatter = formatter;
	};

	/**
	 * Get the formatter
	 *
	 * @protected
	 * @return {wikibase.queryService.ui.resultBrowser.helper.FormatterHelper}
	 */
	SELF.prototype._getFormatter = function() {
		if ( this._formatter === null ) {
			this._formatter = new wikibase.queryService.ui.resultBrowser.helper.FormatterHelper();
		}

		return this._formatter;
	};

	/**
	 * Get an i18n string
	 *
	 * @protected
	 * @param {string} key for the i18n message
	 * @param {string} message default text
	 *
	 * @return {string}
	 */
	SELF.prototype._i18n = function( key, message ) {
		var i18nMessage = null;

		if ( !$.i18n || ( i18nMessage = $.i18n( key ) ) === key ) {
			return message;
		}

		return i18nMessage;
	};

	return SELF;
}( jQuery, wikibase ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.AbstractChartResultBrowser = ( function( $, window ) {
	'use strict';

	/**
	 * An abstract result browser for charts
	 *
	 * @class wikibase.queryService.ui.resultBrowser.TreeResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 *
	 * @constructor
	 */
	function SELF() {
	}

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractResultBrowser();

	/**
	 * Returns all columns that contain labels
	 *
	 * @protected
	 * @return {string[]}
	 */
	SELF.prototype._getLabelColumns = function() {
		var self = this,
			row = self._getRows()[0];

		return self._getColumns().filter( function( column ) {
			return self._getFormatter().isLabel( row[column] );
		} );
	};

	/**
	 * Returns all columns that contain numbers
	 *
	 * @protected
	 * @return {number[]}
	 */
	SELF.prototype._getNumberColumns = function() {
		var self = this,
			rows = self._getRows();

		return self._getColumns().filter( function( column ) {
			return rows.some( function( row ) {
				return row[column] && self._getFormatter().isNumber( row[column] );
			} );
		} );
	};

	/**
	 * @protected
	 * @return {string[]}
	 */
	SELF.prototype._getColumns = function() {
		return this._result.head.vars;
	};

	/**
	 * @protected
	 * @return {Object[]}
	 */
	SELF.prototype._getRows = function() {
		return this._result.results.bindings;
	};

	return SELF;
}( jQuery, window ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};
window.mediaWiki = window.mediaWiki || {};

wikibase.queryService.ui.resultBrowser.AbstractDimpleChartResultBrowser =
	( function( $, _, d3, window, dimple ) {
	'use strict';

	/**
	 * A line dimple chart result browser
	 *
	 * @class wikibase.queryService.ui.resultBrowser.AbstractDimpleChartResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 *
	 * @constructor
	 */
	function SELF() {
	}

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractChartResultBrowser();

	/**
	 * @property {object}
	 * @private
	 */
	SELF.prototype._dataColumns = null;

	/**
	 * @property {string[]}
	 * @private
	 */
	SELF.prototype._data = null;

	/**
	 * @property {jQuery}
	 * @private
	 */
	SELF.prototype._$element = null;

	/**
	 * @property {HTMLElement}
	 * @private
	 */
	SELF.prototype._svg = null;

	/**
	 * @property {dimple.chart}
	 * @private
	 */
	SELF.prototype._chart = null;

	/**
	 * @property {dimple.legend}
	 * @private
	 */
	SELF.prototype._chartLegend = null;

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._chartSeriesKey = null;

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._chartStoryKey = null;

	/**
	 * @property {boolean}
	 * @private
	 */
	SELF.prototype._isStoryPaused = false;

	/**
	 * Draw browser to the given element
	 *
	 * @param {jQuery} $element to draw at
	 */
	SELF.prototype.draw = function( $element ) {
		var self = this;

		this._$element = $( '<div>' ).css( { width: '100%', height: '98vh' } );
		$element.html( this._$element );

		this._createData();
		this._drawSvg();
		this._createChart();
		this._drawChart();
		this._createLegendFilter();

		window.onresize = function() {
			self._drawChart( 250, true );
		};
	};

	SELF.prototype._createData = function() {
		var data = [],
			rowData = {},
			prevRow = null;

		this._iterateResult( function( field, key, row ) {
			if ( prevRow === null ) {
				prevRow = row;
			}

			if ( row !== prevRow ) {
				prevRow = row;
				data.push( rowData );
				rowData = {};
			}

			if ( field && field.value ) {
				rowData[key] = field.value;
			}
		} );

		data.push( rowData );
		this._data = data;
	};

	SELF.prototype._drawSvg = function() {
		this._svg = dimple.newSvg( this._$element[0], '100%', '100%' );
	};

	SELF.prototype._createChart = function() {

		this._chart = new dimple.chart( this._svg, this._data );
		this._chart.setBounds( 0, 0, '100%', '100%' );
		this._chart.setMargins( '5%', '5%', '2%', '25%' );

		this._createChartAxis();

		var series = this._chart.addSeries( this._chartSeriesKey, this._getPlotType() );
		series.addOrderRule( this._chartSeriesKey );
		series.lineMarkers = true;

		if ( this._chartStoryKey ) {
			this._createChartStory();
		}

		this._chartLegend = this._chart.addLegend( '1%', '85%', '100%', '15%' );
	};

	SELF.prototype._getPlotType = function() {
		jQuery.error( 'Method _getPlotType() needs to be implemented!' );
	};

	SELF.prototype._createChartAxis = function() {
		var self = this,
			row = this._getRows()[0],
			formatter = this._getFormatter(),
			chart = this._chart,
			axis = [ 'y', 'x' ],
			hasSeriesAxis = false;

		$.each( this._getColumns(), function( i, key ) {
			if ( axis.length === 0 ) {
				if ( formatter.isLabel( row[key] ) ) {

					if ( !self._chartSeriesKey ) {
						self._chartSeriesKey = key;
						return;
					}
					if ( !self._chartStoryKey ) {
						self._chartStoryKey = key;
						return false;
					}
				}
				return;
			}
			if ( formatter.isLabel( row[key] ) ) {
				chart.addCategoryAxis( axis.pop(), key );
				hasSeriesAxis = true;
			}
			if ( formatter.isNumber( row[key] ) ) {
				chart.addMeasureAxis( axis.pop(), key );
			}
			if ( formatter.isDateTime( row[key] ) ) {
				chart.addTimeAxis( axis.pop(), key, '%Y-%m-%dT%H:%M:%SZ', '%m-%d-%Y' );
				hasSeriesAxis = true;
			}
		} );

		if ( !hasSeriesAxis && !this._chartSeriesKey && chart.axes[0] ) {
			this._chartSeriesKey = chart.axes[0].measure;
		}
	};

	SELF.prototype._createChartStory = function() {
		var self = this,
			story = this._chart.setStoryboard( this._chartStoryKey );

		story.frameDuration = 5 * 1000;
		this._$element.click( function() {
			if ( self._isStoryPaused ) {
				story.startAnimation();
				self._isStoryPaused = false;
				return;
			}

			story.pauseAnimation();
			self._isStoryPaused = true;
		} );
	};

	SELF.prototype._createLegendFilter = function() {
		var self = this,
			filterValues = dimple.getUniqueValues( this._data, this._chartSeriesKey );
		this._chart.legends = [];

		this._chartLegend.shapes.selectAll( 'rect' ).on( 'click', function( e ) {
			var hide = false,
				newFilters = [];

			filterValues.forEach( function( field ) {
				if ( field === e.aggField.slice( -1 )[0] ) {
					hide = true;
				} else {
					newFilters.push( field );
				}
			} );

			d3.select( this ).style( 'opacity', hide ? 0.2 : 0.8 );
			if ( !hide ) {
				newFilters.push( e.aggField.slice( -1 )[0] );
			}
			filterValues = newFilters;

			self._chart.data = dimple.filterData( self._data, self._chartSeriesKey, filterValues );
			self._drawChart( 800 );
		} );
	};

	SELF.prototype._drawChart = function( duration, noDataChange ) {
		var self = this;

		this._chart.draw( duration, noDataChange );

		_.delay( function() {
			self._svg.selectAll( '.dimple-marker,.dimple-marker-back' ).attr( 'r', 2 );
		}, duration * 1.2 );
	};

	/**
	 * Checks whether the browser can draw the given result
	 *
	 * @return {boolean}
	 */
	SELF.prototype.isDrawable = function() {
		return ( Object.keys( this._dataColumns ).length >= 2 );
	};

	/**
	 * Receiving data from the a visit
	 *
	 * @param {Object} value
	 * @param {string} key
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( value, key ) {
		return this._checkColumn( value, key );
	};

	/**
	 * Check if this value contains an coordinate value.
	 *
	 * @param {Object} value
	 * @param {string} key
	 * @return {boolean}
	 * @private
	 */
	SELF.prototype._checkColumn = function( value, key ) {
		if ( this._getFormatter().isLabel( value ) ) {
			this._dataColumns[key] = true;
		}

		if ( this._getFormatter().isNumber( value ) ) {
			this._dataColumns[key] = true;
		}

		if ( this._getFormatter().isDateTime( value ) ) {
			this._dataColumns[key] = true;
		}

		return !this.isDrawable();
	};

	return SELF;
}( jQuery, _, d3, window, dimple ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.ImageResultBrowser = ( function( $ ) {
	'use strict';

	/**
	 * A result browser for images
	 *
	 * @class wikibase.queryService.ui.resultBrowser.ImageResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 */
	function SELF() {
	}

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractResultBrowser();

	/**
	 * @property {jQuery}
	 * @private
	 */
	SELF.prototype._grid = null;

	/**
	 * Draw browser to the given element
	 *
	 * @param {jQuery} $element to draw at
	 */
	SELF.prototype.draw = function( $element ) {
		var self = this;
		this._grid = $( '<div class="masonry">' );

		this._iterateResult( function( field, key, row ) {
			if ( field && self._isCommonsResource( field.value ) ) {
				var url = field.value,
					fileName = self._getFormatter().getCommonsResourceFileName( url );

				self._grid.append( self._getItem( self._getThumbnail( url ), self._getThumbnail(
						url, 1000 ), fileName, row ) );
			}
		} );

		$element.html( this._grid );
	};

	/**
	 * @private
	 */
	SELF.prototype._getItem = function( thumbnailUrl, url, title, row ) {
		var $image = $( '<a>' )
				.click( this._getFormatter().handleCommonResourceItem )
				.attr( { href: url, 'data-gallery': 'g', 'data-title': title } )
				.append( $( '<img>' ).attr( 'src', thumbnailUrl ) ),
			$summary = this._getFormatter().formatRow( row );

		return $( '<div class="item">' ).append( $image, $summary );
	};

	/**
	 * @private
	 */
	SELF.prototype._isCommonsResource = function( url ) {
		return this._getFormatter().isCommonsResource( url );
	};

	/**
	 * @private
	 */
	SELF.prototype._getThumbnail = function( url, width ) {
		return this._getFormatter().getCommonsResourceFileNameThumbnail( url, width );
	};

	/**
	 * Receiving data from the a visit
	 *
	 * @param {Object} data
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( data ) {
		return this._checkImage( data );
	};

	/**
	 * Check if this value contains an image.
	 */
	SELF.prototype._checkImage = function( data ) {
		if ( data && data.value && this._isCommonsResource( data.value ) ) {
			this._drawable = true;
			return false;
		}

		return true;
	};

	return SELF;
}( jQuery ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.TableResultBrowser = ( function( $, window ) {
	'use strict';

	/**
	 * A result browser for tables
	 *
	 * @class wikibase.queryService.ui.resultBrowser.TableResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @author Jonas Keinholz
	 *
	 * @constructor
	 */
	function SELF() {
	}

	var TABLE_PAGE_SIZE = 200;
	var TABLE_PAGE_SIZE_LIST = [ 10, 50, 100, 200, 500, 1000 ];

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractResultBrowser();

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._columns = null;

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._rows = null;

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._sorter = {
		string: function( val1, val2 ) {
			return val1.localeCompare( val2 );
		},

		number: function( val1, val2 ) {
			if ( val1 >= val2 ) {
				return -1;
			}

			return 1;
		},

		generic: function( data1, data2 ) {
			if ( !data2 ) {
				return 1;
			}
			if ( !data1 ) {
				return -1;
			}

			var f = this._getFormatter();
			if ( f.isNumber( data1 ) && f.isNumber( data2 ) ) {
				return this._sorter.number( Number( data1.value ), Number( data2.value ) );
			}

			if ( f.isEntityUri( data1.value ) && f.isEntityUri( data2.value ) ) {
				return this._sorter.number( Number( data1.value.replace( /[^0-9]/gi, '' ) ),
						Number( data2.value.replace( /[^0-9]/gi, '' ) ) );
			}

			// default is string sorter
			return this._sorter.string( data1.value, data2.value );
		}
	};

	/**
	 * Draw browser to the given element
	 *
	 * @param {jQuery} $element to draw at
	 */
	SELF.prototype.draw = function( $element ) {
		var data = this._result;

		if ( typeof data.boolean !== 'undefined' ) {
			// ASK query
			var $table = $( '<table>' ).attr( 'class', 'table' );
			$table.append( '<tr><td>' + data.boolean + '</td></tr>' ).addClass( 'boolean' );
			$element.html( $table );
			return;
		}

		this.columns = data.head.vars;
		this.rows = data.results.bindings;

		var $wrapper = $( '<table/>' );
		$element.html( $wrapper );
		this.drawBootstrapTable( $wrapper );

		if ( $wrapper.children().width() > $( window ).width() ) {
			$wrapper.bootstrapTable( 'toggleView' );
		}
	};

	/**
	 * Draw browser to the given element
	 *
	 * @param {jQuery} $element to draw at
	 */
	SELF.prototype.drawBootstrapTable = function( $element ) {
		var self = this,
			showPagination = ( this.rows.length > TABLE_PAGE_SIZE );

		jQuery.fn.bootstrapTable.columnDefaults.formatter = function( data, row, index ) {
			if ( !data ) {
				return '';
			}
			self.processVisitors( data, this.field );
			return self._getFormatter().formatValue( data ).html();
		};

		var events = {
			'click .explore': $.proxy( this._getFormatter().handleExploreItem, this ),
			'click .gallery': this._getFormatter().handleCommonResourceItem
		};

		$element.bootstrapTable( {
			columns: this.columns.map( function( column ) {
				return {
					title: column,
					field: column,
					events: events,
					sortable: true,
					sorter: $.proxy( self._sorter.generic, self )
				};
			} ),
			data: this.rows,
			mobileResponsive: true,
			search: showPagination,
			pagination: showPagination,
			showPaginationSwitch: showPagination,
			pageSize: TABLE_PAGE_SIZE,
			pageList: TABLE_PAGE_SIZE_LIST,
			keyEvents: true,
			cookie: true,
			cookieIdTable: '1',
			cookieExpire: '1y',
			cookiesEnabled: [ 'bs.table.pageList' ]
		} );
	};

	/**
	 * Checks whether the browser can draw the given result
	 *
	 * @return {boolean}
	 */
	SELF.prototype.isDrawable = function() {
		return true;
	};

	/**
	 * Receiving data from the a visit
	 *
	 * @param {Object} data
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( data ) {
		return false;
	};

	return SELF;
}( jQuery, window ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.CoordinateResultBrowser = ( function( $, L, d3, _, wellknown, window ) {
	'use strict';

	/**
	 * A list of datatypes that contain geo:wktLiteral values conforming with GeoSPARQL.
	 * @private
	 */
	var MAP_DATATYPES = [
		'http://www.opengis.net/ont/geosparql#wktLiteral', // used by Wikidata
		'http://www.openlinksw.com/schemas/virtrdf#Geometry' // used by LinkedGeoData.org
	];
	var GLOBE_EARTH = 'http://www.wikidata.org/entity/Q2';
	var CRS84 = 'http://www.opengis.net/def/crs/OGC/1.3/CRS84';
	/**
	 * A list of coordinate reference systems / spatial reference systems
	 * that refer to Earth and use longitude-latitude axis order.
	 * @private
	 */
	var EARTH_LONGLAT_SYSTEMS = [
		GLOBE_EARTH,
		CRS84
	];

	var LAYER_COLUMNS = [ 'layerLabel', 'layer' ];
	var LAYER_DEFAULT_GROUP = '_LAYER_DEFAULT_GROUP';

	var TILE_LAYER = {
		wikimedia: {
			url: 'https://maps.wikimedia.org/osm-intl/{z}/{x}/{y}.png',
			options: {
				id: 'wikipedia-map-01',
				attribution: ' <a href="http://maps.wikimedia.org/">Wikimedia</a> | &copy; <a href="http://openstreetmap.org/copyright">Open Street Map</a> contributors'
			}
		},
		osm: {
			url: 'http://{s}.tile.osm.org/{z}/{x}/{y}.png',
			options: {
				attribution: '&copy; <a href="http://osm.org/copyright">OpenStreetMap</a> contributors'
			}
		}
	};

	var ScrollToTopButton = null;

	/**
	 * A result browser for long lat coordinates
	 *
	 * @class wikibase.queryService.ui.resultBrowser.CoordinateResultBrowser
	 * @licence GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @author Katie Filbert
	 * @constructor
	 *
	 */
	function SELF() {
		this._markerGroupColors = {};
		var _getDefaultMarkerGroupColor = d3.scale.category10();
		this._getMarkerGroupColor = function( group ) {
			if ( group in this._markerGroupColors ) {
				return this._markerGroupColors[ group ];
			}
			return _getDefaultMarkerGroupColor( group );
		};

		ScrollToTopButton = L.Control.extend( {
			options: {
				position: 'topright'
			},

			onAdd: function( map ) {
				var container = L.DomUtil.create( 'button' );
				$( container ).addClass( 'btn btn-default' );
				$( container ).append( $( ' <span class="glyphicon glyphicon-chevron-up"/> ' ) );

				container.onclick = function() {
					if ( map.isFullscreen() ) {
						map.toggleFullscreen();
					}
					$( window ).scrollTop( 0, 0 );
				};

				return container;
			}
		} );
	}

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractResultBrowser();

	/**
	 * @property {L.Map}
	 * @private
	 */
	SELF.prototype._map = null;

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._markerGroups = null;

	/**
	 * Sparse map from group to last-seen RGB color (?rgb column) for that group.
	 * _getMarkerGroupColor uses this map to look up colors,
	 * falling back to a static color map if no RGB color was recorded for a group.
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._markerGroupColors = null;

	/**
	 * Maps group name to a certain color
	 * @private
	 */
	SELF.prototype._getMarkerGroupColor = null;

	/**
	 * Draw a map to the given element
	 *
	 * @param {jQuery} $element target element
	 */
	SELF.prototype.draw = function( $element ) {
		var container = $( '<div>' ).attr( 'id', 'map' ).height( '100vh' );

		$element.html( container );

		this._createMarkerGroups();
		this._map = L.map( 'map', {
			center: [ 0, 0 ],
			maxZoom: 18,
			minZoom: 2,
			fullscreenControl: true,
			preferCanvas: true,
			layers: _.compact( this._markerGroups ) // convert object to array
		} ).fitBounds( this._markerGroups[ LAYER_DEFAULT_GROUP ].getBounds() );

		this._setTileLayer();
		this._createControls();
		this._createMarkerZoomResize();

		$element.html( container );
	};

	/**
	 * Create map controls
	 *
	 * @private
	 */
	SELF.prototype._createControls = function() {
		var self = this;

		//zoom control
		this._map.addControl( L.control.zoomBox( {
			modal: false,
			className: 'glyphicon glyphicon-zoom-in'
		} ) );
		this._map.addControl( new ScrollToTopButton() );

		//layers control
		var numberOfLayers = Object.keys( this._markerGroups ).length;
		if ( numberOfLayers > 1 ) {
			var control = this._getLayerControl( this._markerGroups ).addTo( this._map );

			// update layer control
			this._map.on( 'overlayadd overlayremove', function ( event ) {
				if ( event.layer !== self._markerGroups[ LAYER_DEFAULT_GROUP ] ) {
					return;
				}
				$.each( self._markerGroups, function( i, layer ) {
					if ( event.type === 'overlayadd' ) {
						self._map.addLayer( layer );
					} else {
						self._map.removeLayer( layer );
					}
				} );
				control._update();
			} );
		}
	};

	/**
	 * @private
	 */
	SELF.prototype._createMarkerZoomResize = function() {
		var self = this;

		if ( this._markerGroups[LAYER_DEFAULT_GROUP].getLayers().length > 1000 ) {
			return; // disable when to many markers (bad performance)
		}

		var resize = function() {
			self._markerGroups[LAYER_DEFAULT_GROUP].setStyle( {
				radius: self._getMarkerRadius()
			} );
		};

		this._map.on( 'zoomend', resize );
	};

	/**
	 * @private
	 */
	SELF.prototype._getMarkerRadius = function() {
		if ( !this._map ) {
			return 3;
		}

		var currentZoom = this._map.getZoom();
		return ( currentZoom * ( 1 / 2 ) );
	};

	/**
	 * @private
	 */
	SELF.prototype._getLayerControl = function() {
		var self = this,
			layerControls = {},
			control = '';

		$.each( this._markerGroups, function( name, markers ) {
			if ( name === LAYER_DEFAULT_GROUP ) {
				control = self._i18n( 'wdqs-result-map-layers-all', 'All layers' );
			} else {
				var color = self._getMarkerGroupColor( name );
				control = '<span style="color:' + color + '">&#x2b24;</span> ' + name;
			}

			layerControls[ control ] = markers;
		} );

		return L.control.layers( null, layerControls );
	};

	/**
	 * @private
	 */
	SELF.prototype._createMarkerGroups = function() {
		var self = this,
			markers = {};
		markers[ LAYER_DEFAULT_GROUP ] = [];

		this._iterateResult( function( field, key, row ) {
			if ( field && MAP_DATATYPES.indexOf( field.datatype ) !== -1 ) {
				var geoJson = self._extractGeoJson( field.value );
				if ( !geoJson ) {
					return;
				}

				var layer = self._getMarkerGroupsLayer( row );
				if ( !markers[ layer ] ) {
					markers[ layer ] = [];
				}
				var marker = L.geoJson( geoJson, {
					style: self._getMarkerStyle( layer, row ),
					pointToLayer: function( geoJsonPoint, latLon ) {
						return L.circleMarker( latLon, self._getMarkerStyle( layer, row ) );
					},
					onEachFeature: function( feature, layer ) {
						var popup = L.popup();
						layer.bindPopup( popup );
						layer.on( 'click', function() {
							var info = self._getItemDescription( row );
							popup.setContent( info[0] );
						} );
					}
				} );
				markers[ layer ].push( marker );
				markers[ LAYER_DEFAULT_GROUP ].push( marker );
			}
		} );

		if ( Object.keys( markers ).length === 0 ) {
			var marker = L.marker( [ 0, 0 ] ).bindPopup( 'Nothing found!' ).openPopup();
			return { null: L.featureGroup( [marker] ) };
		}

		$.each( markers, function( key ) {
			markers[ key ] = L.featureGroup( markers[ key ] );
		} );

		this._markerGroups = markers;
	};

	/**
	 * @private
	 */
	SELF.prototype._getMarkerGroupsLayer = function( row ) {
		var column = LAYER_COLUMNS.find( function( column ) {
			return row[column];
		} );

		return column ? row[column].value : LAYER_DEFAULT_GROUP;
	};

	/**
	 * @private
	 * @param {string} group
	 * @param {Object} row
	 */
	SELF.prototype._getMarkerStyle = function( group, row ) {
		var color,
			formatter = this._getFormatter();

		if ( 'rgb' in row && formatter.isColor( row.rgb ) ) {
			color = formatter.getColorForHtml( row.rgb );
			this._markerGroupColors[ group ] = color;
		} else if ( group !== LAYER_DEFAULT_GROUP ) {
			color = this._getMarkerGroupColor( group );
		} else {
			color = '#e04545';
		}

		return {
			color: color,
			opacity: 0.8,
			fillColor: color,
			fillOpacity: 0.9,
			radius: this._getMarkerRadius()
		};
	};

	/**
	 * Split a geo:wktLiteral or compatible value
	 * into coordinate reference system URI
	 * and Simple Features Well Known Text (WKT) string,
	 * according to GeoSPARQL, Req 10.
	 *
	 * If the coordinate reference system is not specified,
	 * CRS84 is used as default value, according to GeoSPARQL, Req 11.
	 *
	 * @private
	 * @param {string} literal
	 * @return {?{ crs: string, wkt: string }}
	 */
	SELF.prototype._splitWktLiteral = function( literal ) {
		var match = literal.match( /(<([^>]*)> +)?(.*)/ ); // only U+0020 spaces as separator, not other whitespace, according to GeoSPARQL, Req 10

		if ( match ) {
			return { crs: match[2] || CRS84, wkt: match[3] };
		} else {
			return null;
		}
	};

	/**
	 * Extract a GeoJSON object from the given geo:wktLiteral.
	 *
	 * @private
	 * @param {string} literal
	 * @return {?Object} GeoJSON
	 */
	SELF.prototype._extractGeoJson = function( literal ) {
		var split = this._splitWktLiteral( literal );
		if ( !split ) {
			return null;
		}

		if ( EARTH_LONGLAT_SYSTEMS.indexOf( split.crs ) === -1 ) {
			return null;
		}

		return wellknown.parse( split.wkt );
	};

	/**
	 * @private
	 */
	SELF.prototype._getItemDescription = function( row ) {
		var $result = $( '<div/>' ).append( this._getFormatter().formatRow( row, true ) );

		return $result;
	};

	/**
	 * @private
	 */
	SELF.prototype._setTileLayer = function() {
		var layer = TILE_LAYER.osm;

		if ( window.location.host === 'query.wikidata.org' ||
				window.location.host === 'localhost' ||
				window.location.host.endsWith( '.wmflabs.org' ) ) {
			layer = TILE_LAYER.wikimedia;
		}

		L.tileLayer( layer.url, layer.options ).addTo( this._map );
	};

	/**
	 * Receiving data from the a visit
	 *
	 * @param {Object} data
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( data ) {
		return this._checkCoordinate( data );
	};

	/**
	 * Check if this value contains an coordinate value.
	 */
	SELF.prototype._checkCoordinate = function( value ) {
		if ( value && MAP_DATATYPES.indexOf( value.datatype ) !== -1 ) {
			var longLat = this._extractGeoJson( value.value );
			if ( longLat !== null ) {
				this._drawable = true;
				return false;
			}
		}
		return true;
	};

	return SELF;
}( jQuery, L, d3, _, wellknown, window ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.TreeMapResultBrowser = ( function( $, d3, window ) {
	'use strict';

	/**
	 * A treemap result browser
	 *
	 * @class wikibase.queryService.ui.resultBrowser.TreeResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 *
	 * @constructor
	 */
	function SELF() {
		this._labelColumns = {};
	}

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractChartResultBrowser();

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._labelColumns = null;

	/**
	 * Draw browser to the given element
	 *
	 * @param {jQuery} $element to draw at
	 */
	SELF.prototype.draw = function( $element ) {
		var self = this,
			data = {},
			layer = data,
			size = null,
			url = null,
			prevRow = null;

		this._iterateResult( function( field, key, row ) {
			if ( row !== prevRow ) {
				if ( prevRow !== null ) {
					layer.data = {
						size: size,
						url: url
					};
					size = null;
					url = null;
					layer = data;
				}

				prevRow = row;
			}

			if ( self._getFormatter().isLabel( field ) ) {
				if ( !layer[field.value] ) {
					layer[field.value] = {};
				}
				layer = layer[field.value];
			}
			if ( self._getFormatter().isNumber( field ) ) {
				size = field.value;
			}
			if ( field && field.value && self._getFormatter().isEntityUri( field.value ) ) {
				url = field.value;
			}
		} );

		layer.data = {
			size: size,
			url: url
		};

		var children = this._createTreeData( data );
		var treeData = {
			name: 'treeMap',
			children: children
		};

		this._draw( $element, treeData );
	};

	SELF.prototype._createTreeData = function( data ) {
		var self = this,
			nodes = [],
			node;

		if ( data.data ) {
			return nodes;
		}

		$.each( data, function( key, value ) {
			var children = self._createTreeData( value );

			if ( children.length !== 0 ) {
				node = {
					id: key,
					name: key,
					children: children
				};
			} else {
				var size = 1;
				if ( value.data.size && value.data.size > 0 ) {
					size = value.data.size;
				}
				var id = key;
				var url = value.data.url;
				if ( url ) {
					id = url;
				}

				node = {
					id: id + Math.random(),
					name: key,
					size: size,
					url: url
				};
			}
			nodes.push( node );
		} );

		return nodes;
	};

	/* jshint ignore:start */
	SELF.prototype._draw = function( $element, data ) {
		// copied from http://www.billdwhite.com/wordpress/wp-content/js/treemap_headers_03.html
		// with little modification

		var supportsForeignObject = false;// FIXME:Modernizr.svgforeignobject;
		var chartWidth = $( window ).width(),
			chartHeight = $( window ).height(),
			xscale = d3.scale.linear().range( [ 0, chartWidth ] ),
			yscale = d3.scale.linear().range( [ 0, chartHeight ] ),
			color = d3.scale.category10(),
			headerHeight = 20,
			headerColor = '#555555',
			transitionDuration = 500,
			root,
			node;

		var treemap = d3.layout.treemap().round( false ).size( [ chartWidth, chartHeight ] )
				.sticky( true ).value( function( d ) {
					return d.size;
				} );

		var chart = d3.select( $element[0] ).append( 'svg:svg' ).attr( 'width', chartWidth ).attr(
				'height', chartHeight ).append( 'svg:g' );

		node = root = data;
		var nodes = treemap.nodes( root );

		var children = nodes.filter( function( d ) {
			return !d.children;
		} );
		var parents = nodes.filter( function( d ) {
			return d.children;
		} );

		// create parent cells
		var parentCells = chart.selectAll( 'g.cell.parent' ).data( parents, function( d ) {
			return 'p-' + d.id;
		} );
		var parentEnterTransition = parentCells.enter().append( 'g' ).attr( 'class', 'cell parent' )
				.on( 'click', function( d ) {
					zoom( d );
				} );
		parentEnterTransition.append( 'rect' ).attr( 'width', function( d ) {
			return Math.max( 0.01, d.dx );
		} ).attr( 'height', headerHeight ).style( 'fill', headerColor );
		parentEnterTransition.append( 'foreignObject' ).attr( 'class', 'foreignObject' ).append(
				'xhtml:body' ).attr( 'class', 'labelbody' ).append( 'div' ).attr( 'class', 'label' );
		// update transition
		var parentUpdateTransition = parentCells.transition().duration( transitionDuration );
		parentUpdateTransition.select( '.cell' ).attr( 'transform', function( d ) {
			return 'translate(' + d.dx + ',' + d.y + ')';
		} );
		parentUpdateTransition.select( 'rect' ).attr( 'width', function( d ) {
			return Math.max( 0.01, d.dx );
		} ).attr( 'height', headerHeight ).style( 'fill', headerColor );
		parentUpdateTransition.select( '.foreignObject' ).attr( 'width', function( d ) {
			return Math.max( 0.01, d.dx );
		} ).attr( 'height', headerHeight ).select( '.labelbody .label' ).text( function( d ) {
			return d.name;
		} );
		// remove transition
		parentCells.exit().remove();

		// create children cells
		var childrenCells = chart.selectAll( 'g.cell.child' ).data( children, function( d ) {
			return 'c-' + d.id;
		} );
		// enter transition
		var childEnterTransition = childrenCells.enter().append( 'g' ).attr( 'class', 'cell child' )
				.on( 'click', function( d ) {
					zoom( node === d.parent ? root : d.parent );
				} );
		childEnterTransition.append( 'rect' ).classed( 'background', true ).style( 'fill',
				function( d ) {
					return color( d.parent.name );
				} );
		childEnterTransition.append( 'foreignObject' ).attr( {
			'class': 'foreignObject',
			width: function( d ) {
				return Math.max( 0.01, d.dx );
			},
			height: function( d ) {
				return Math.max( 0.01, d.dy );
			}
		} ).append( 'xhtml:body' ).attr( 'class', 'labelbody' ).append( 'div' ).attr( 'class',
				'label' ).text( function( d ) {
			return d.name;
		} ).on( 'click', function( d ) {
			if ( d.url ) {
				window.open( d.url, '_blank' );
				d3.event.stopPropagation();
			}
		} );

		//		        if (supportsForeignObject) {
		//		            childEnterTransition.selectAll(".foreignObject")
		//		                    .style("display", "none");
		//		        } else {
		//		            childEnterTransition.selectAll(".foreignObject .labelbody .label")
		//		                    .style("display", "none");
		//		        }

		// update transition
		var childUpdateTransition = childrenCells.transition().duration( transitionDuration );
		childUpdateTransition.select( '.cell' ).attr( 'transform', function( d ) {
			return 'translate(' + d.x + ',' + d.y + ')';
		} );
		childUpdateTransition.select( 'rect' ).attr( 'width', function( d ) {
			return Math.max( 0.01, d.dx );
		} ).attr( 'height', function( d ) {
			return d.dy;
		} ).style( 'fill', function( d ) {
			return color( d.parent.name );
		} );
		childUpdateTransition.select( '.foreignObject' ).attr( 'width', function( d ) {
			return Math.max( 0.01, d.dx );
		} ).attr( 'height', function( d ) {
			return Math.max( 0.01, d.dy );
		} ).select( '.labelbody .label' ).text( function( d ) {
			return d.name;
		} );
		// exit transition
		childrenCells.exit().remove();

		d3.select( 'select' ).on( 'change', function() {
			console.log( 'select zoom(node)' );
			treemap.value( value == 'size' ? size : count ).nodes( root );
			zoom( node );
		} );

		zoom( node );

		function size( d ) {
			return d.size;
		}

		function count( d ) {
			return 1;
		}

		//and another one
		function textHeight( d ) {
			var ky = chartHeight / d.dy;
			yscale.domain( [ d.y, d.y + d.dy ] );
			return ( ky * d.dy ) / headerHeight;
		}

		function getRGBComponents( color ) {
			var r = color.substring( 1, 3 ),
				g = color.substring( 3, 5 ),
				b = color.substring( 5, 7 );

			return {
				R: parseInt( r, 16 ),
				G: parseInt( g, 16 ),
				B: parseInt( b, 16 )
			};
		}

		function idealTextColor( bgColor ) {
			var nThreshold = 105;
			var components = getRGBComponents( bgColor );
			var bgDelta = ( components.R * 0.299 ) + ( components.G * 0.587 )
					+ ( components.B * 0.114 );
			return ( ( 255 - bgDelta ) < nThreshold ) ? '#000000' : '#ffffff';
		}

		function zoom( d ) {
			treemap.padding( [ headerHeight / ( chartHeight / d.dy ), 0, 0, 0 ] ).nodes( d );

			// moving the next two lines above treemap layout messes up padding of zoom result
			var kx = chartWidth / d.dx;
			var ky = chartHeight / d.dy;
			var level = d;

			xscale.domain( [ d.x, d.x + d.dx ] );
			yscale.domain( [ d.y, d.y + d.dy ] );

			chart.selectAll( '.labelbody .label' ).style( 'display', 'block' );

			//		        if (node != level) {
			//		            if (supportsForeignObject) {
			//		                chart.selectAll(".cell.child .foreignObject")
			//		                        .style("display", "none");
			//		            } else {
			//		                chart.selectAll(".cell.child .foreignObject .labelbody .label")
			//		                        .style("display", "none");
			//		            }
			//		        }

			var zoomTransition = chart.selectAll( 'g.cell' ).transition().duration(
					transitionDuration ).attr( 'transform', function( d ) {
				return 'translate(' + xscale( d.x ) + ',' + yscale( d.y ) + ')';
			} ).each( 'end', function( d, i ) {
				if ( !i && ( level !== self.root ) ) {
					chart.selectAll( '.cell.child' ).filter( function( d ) {
						return d.parent === self.node; // only get the children for selected group
					} ).select( '.foreignObject .labelbody .label' ).style( 'color', function( d ) {
						return idealTextColor( color( d.parent.name ) );
					} );

					if ( supportsForeignObject ) {
						chart.selectAll( '.cell.child' ).filter( function( d ) {
							return d.parent === self.node; // only get the children for selected group
						} ).select( '.foreignObject' ).style( 'display', '' );
					} else {
						chart.selectAll( '.cell.child' ).filter( function( d ) {
							return d.parent === self.node; // only get the children for selected group
						} ).select( '.foreignObject .labelbody .label' ).style( 'display', '' );
					}
				}
			} );

			zoomTransition.select( '.foreignObject' ).attr( 'width', function( d ) {
				return Math.max( 0.01, kx * d.dx );
			} ).attr( 'height', function( d ) {
				return d.children ? headerHeight : Math.max( 0.01, ky * d.dy );
			} ).select( '.labelbody .label' ).text( function( d ) {
				return d.name;
			} );

			// update the width/height of the rects
			zoomTransition.select( 'rect' ).attr( 'width', function( d ) {
				return Math.max( 0.01, kx * d.dx );
			} ).attr( 'height', function( d ) {
				return d.children ? headerHeight : Math.max( 0.01, ky * d.dy );
			} ).style( 'fill', function( d ) {
				return d.children ? headerColor : color( d.parent.name );
			} );

			node = d;

			if ( d3.event ) {
				d3.event.stopPropagation();
			}
		}
	};
	/* jshint ignore:end */

	/**
	 * Checks whether the browser can draw the given result
	 *
	 * @return {boolean}
	 */
	SELF.prototype.isDrawable = function() {
		return Object.keys( this._labelColumns ).length > 1;
	};

	/**
	 * Receiving data from the a visit
	 *
	 * @param {Object} data
	 * @param {string} key
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( data, key ) {
		return this._checkColumn( data, key );
	};

	/**
	 * @param {Object} value
	 * @param {string} key
	 * @return {boolean}
	 * @private
	 */
	SELF.prototype._checkColumn = function( value, key ) {
		if ( this._getFormatter().isLabel( value, key ) ) {
			this._labelColumns[key] = true;
			return !this.isDrawable();
		}

		return true;
	};

	return SELF;
}( jQuery, d3, window ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.TreeResultBrowser = ( function( $, _, window ) {
	'use strict';

	// jscs:disable
	var SPARQL_ITEM_PROPERTIES =  'SELECT ?property ?propertyLabel ?value ?valueItemLabel ?valueImage WHERE {  '  +
	 '     {  '  +
	 '       SELECT ?property ?value ?valueImage ?valueItem WHERE {  '  +
	 '         BIND(<{ENTITY_URI}> AS ?item)  '  +
	 '         ?item ?prop ?value.  '  +
	 '         ?property wikibase:directClaim ?prop.  '  +
	 '         ?property rdf:type wikibase:Property.  '  +
	 '         OPTIONAL { ?value wdt:P18 ?valueImage. }  '  +
	 '         OPTIONAL { BIND(?value AS ?valueItem).  FILTER(STRSTARTS(STR(?value), STR(wd:))) }  '  +
	 '       }  '  +
	 '     }  '  +
	 '     SERVICE wikibase:label { bd:serviceParam wikibase:language "[AUTO_LANGUAGE],en". }  '  +
	 '  }  ' ;
	// jscs:enable

	/**
	 * A tree result browser
	 *
	 * @class wikibase.queryService.ui.resultBrowser.TreeResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 *
	 * @constructor
	 */
	function SELF() {
		this._labelColumns = {};
	}

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractChartResultBrowser();

	/**
	 * @property {boolean}
	 * @private
	 */
	SELF.prototype._isDrawable = false;

	/**
	 * @property {object}
	 * @private
	 */
	SELF.prototype._jsTree = null;

	/**
	 * @property {object}
	 * @private
	 */
	SELF.prototype._nodes = null;

	/**
	 * Draw browser to the given element
	 *
	 * @param {jQuery} $element to draw at
	 */
	SELF.prototype.draw = function( $element ) {

		var self = this;
		this._nodes = this._extractNodes( $.proxy( this._iterateResult, this )  );

		var $container = $( '<div>' ).jstree( {
			core: {
				data: _.compact( self._nodes ),
				'check_callback': true
			},
			plugins: [
				'types', 'wholerow', 'contextmenu', 'unique'
			],
			types: {
				default: {
					icon: 'glyphicon glyphicon-barcode'
				}
			},
			contextmenu: {
				items: function( $node ) {
					return {
						Open: {
							label: self._nodes[$node.id].url,
							icon: 'glyphicon glyphicon-link',
							action: function() {
								window.open( self._nodes[ $node.id ].url, '_blank' );
							}
						},
						explore: {
							label: self._nodes[$node.id].url,
							icon: 'glyphicon glyphicon-download-alt',
							action: function() {
								self._expandTree( self._nodes[ $node.id ] );
							}
						}

					};
				}
			}
		} );

		this._jsTree = $container.jstree();
		$element.empty().append( $container );
	};

	/**
	 * @private
	 */
	SELF.prototype._extractNodes = function( iterator, parentNode ) {
		var prevRow = null,
			node = {},
			currentNode = null,
			nodes = {},
			format = this._getFormatter();

		iterator( function ( field, key, row ) {
			if ( row !== prevRow ) {
				currentNode = null;
				prevRow = row;
			}

			if ( field && field.value ) {

				if ( currentNode && ( format.isLabel( field ) || format.isNumber( field ) ) ) {
					nodes[currentNode].text = ( nodes[currentNode].text || '' ) + ' ' + field.value;
				}

				if ( currentNode && format.isCommonsResource( field.value ) ) {
					nodes[currentNode].icon = format.getCommonsResourceFileNameThumbnail( field.value, 24 );
				}

				if ( format.isEntity( field ) ) {
					node = {};
					node.parent = currentNode ||  parentNode || '#';
					node.id = node.parent + field.value;
					node.url = field.value;
					node.state = { opened: false };

					nodes[node.id] = node;
					currentNode = node.id;
				}
			}

		} );

		return nodes;
	};

	SELF.prototype._expandTree = function( node ) {
		var self = this;

		this.getSparqlApi().query( SPARQL_ITEM_PROPERTIES.replace( '{ENTITY_URI}', node.url ) ).done(
			function() {
				var data = self.getSparqlApi().getResultRawData(),
					iterator = function( cb ) {
						$.each( data.results.bindings, function( rowNum, row ) {
							$.each( data.head.vars, function( rowNum1, key ) {
								var field = row[key] || null;
								cb( field, key, row, rowNum );
							} );
						} );
					},
					nodes = self._extractNodes( iterator, node.id );

				$.extend( self._nodes, nodes );
				$.each( nodes, function ( key, node ) {
					// jscs:disable
					self._jsTree.create_node( node.parent, node );// jshint ignore:line
					// jscs:enable
				} );
			}
		);
	};

	/**
	 * Checks whether the browser can draw the given result
	 *
	 * @return {boolean}
	 */
	SELF.prototype.isDrawable = function() {
		return this._isDrawable;
	};

	/**
	 * Receiving data from the visit
	 *
	 * @param {Object} data
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( data ) {
		if ( this._getFormatter().isEntity( data ) ) {
			this._isDrawable = true;
			return false;
		}
		return true;
	};

	return SELF;
}( jQuery, _, window ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.BubbleChartResultBrowser = ( function( $, d3, window ) {
	'use strict';

	/**
	 * A bubble chart result browser
	 *
	 * @class wikibase.queryService.ui.resultBrowser.BubbleChartResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 *
	 * @constructor
	 */
	function SELF() {
	}

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractChartResultBrowser();

	/**
	 * @property {boolean}
	 * @private
	 */
	SELF.prototype._hasLabel = false;

	/**
	 * @property {boolean}
	 * @private
	 */
	SELF.prototype._hasNumber = false;

	/**
	 * Draw browser to the given element
	 *
	 * @param {jQuery} $element to draw at
	 */
	SELF.prototype.draw = function( $element ) {
		var self = this,
			data = { 'name': 'bubblechart', 'children': [] },
			labelKey = this._getLabelColumns()[0],
			numberKey = this._getNumberColumns()[0],
			prevRow = null;

		this._iterateResult( function( field, key, row ) {

			if ( row !== prevRow ) {
				var item = {
					url: null
				};
				prevRow = row;

				if ( row.rgb && self._getFormatter().isColor( row.rgb ) ) {
					item.color = self._getFormatter().getColorForHtml( row.rgb );
				}

				if ( row[labelKey] && row[numberKey] ) {
					item.name = row[labelKey].value;
					item.size = row[numberKey].value;
					data.children.push( item );
				}
			}

			if ( field && field.value && self._getFormatter().isEntityUri( field.value ) ) {
				var createdItem = data.children[data.children.length - 1];
				if ( createdItem && createdItem.url === null ) {
					createdItem.url = field.value;
				}
			}
		} );

		var $wrapper = $( '<center>' );
		$element.html( $wrapper );

		this._drawBubbleChart( $wrapper, data );
	};

	SELF.prototype._drawBubbleChart = function( $element, root ) {
		var self = this;

		function classes( root ) {
			var classes = [];
			function recurse( name, node ) {
				if ( node.children ) {
					node.children.forEach( function( child ) {
						recurse( node.name, child );
					} );
				} else {
					classes.push( {
						packageName: name,
						className: node.name,
						value: node.size,
						url: node.url,
						color: node.color
					} );
				}
			}
			recurse( null, root );
			return {
				children: classes
			};
		}

		var diameter = Math.min( $( window ).height() * 0.98, $( window ).width() ),
			color = d3.scale.category20c(),
			bubble = d3.layout.pack().sort( null ).size( [ diameter, diameter ] ).padding( 1.5 );

		var svg = d3.select( $element[0] ).append( 'svg' ).attr( {
			'class': 'bubble',
			width: diameter,
			height: diameter
		} );

		var node = svg.selectAll( '.node' ).data(
				bubble.nodes( classes( root ) ).filter( function( d ) {
					return !d.children;
				} ) ).enter().append( 'g' ).attr( 'class', 'node' ).attr( 'transform',
				function( d ) {
					return 'translate(' + d.x + ',' + d.y + ')';
				} );

		node.append( 'title' ).text( function( d ) {
			return d.className + ': ' + d.value;
		} );

		node.append( 'circle' ).attr( 'r', function( d ) {
			return d.r;
		} ).style( 'fill', function( d ) {
			return d.color || color( d.className );
		} );

		node.append( 'text' ).attr( 'dy', '.3em' ).style( 'text-anchor', 'middle' ).text(
				function( d ) {
					return d.className.substring( 0, d.r / 4 );
				} ).on( 'click', function( d ) {
			if ( d.url ) {
				window.open( d.url, '_blank' );
			}
		} ).style( 'fill', function( d ) {
			if ( d.color ) {
				return self._getFormatter().calculateLuminance( d.color ) <= 0.5 ? '#FFF' : '#000';
			}
		} ).style( 'cursor', 'pointer' );
	};

	/**
	 * Checks whether the browser can draw the given result
	 *
	 * @return {boolean}
	 */
	SELF.prototype.isDrawable = function() {
		return this._hasLabel && this._hasNumber;
	};

	/**
	 * Receiving data from the a visit
	 *
	 * @param {Object} data
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( data ) {
		return this._checkColumn( data );
	};

	/**
	 * Check if this value contains an numeric value.
	 *
	 * @param {Object} value
	 * @return {boolean}
	 * @private
	 */
	SELF.prototype._checkColumn = function( value ) {
		if ( this._getFormatter().isNumber( value ) ) {
			this._hasNumber = true;
		}

		if ( this._getFormatter().isLabel( value ) ) {
			this._hasLabel = true;
		}

		return !this.isDrawable();
	};

	return SELF;
}( jQuery, d3, window ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};
window.mediaWiki = window.mediaWiki || {};

wikibase.queryService.ui.resultBrowser.LineChartResultBrowser = ( function( dimple ) {
	'use strict';

	var PARENT = wikibase.queryService.ui.resultBrowser.AbstractDimpleChartResultBrowser;

	/**
	 * A line chart result browser
	 *
	 * @class wikibase.queryService.ui.resultBrowser.LineChartResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 *
	 * @constructor
	 */
	function SELF() {
		this._dataColumns = {};
	}

	SELF.prototype = new PARENT();

	SELF.prototype._getPlotType = function() {
		return dimple.plot.line;
	};

	return SELF;
}( dimple ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};
window.mediaWiki = window.mediaWiki || {};

wikibase.queryService.ui.resultBrowser.BarChartResultBrowser = ( function( dimple ) {
	'use strict';

	var PARENT = wikibase.queryService.ui.resultBrowser.AbstractDimpleChartResultBrowser;

	/**
	 * A bar chart result browser
	 *
	 * @class wikibase.queryService.ui.resultBrowser.BarChartResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 *
	 * @constructor
	 */
	function SELF() {
		this._dataColumns = {};
	}

	SELF.prototype = new PARENT();

	SELF.prototype._getPlotType = function() {
		return dimple.plot.bar;
	};

	return SELF;
}( dimple ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};
window.mediaWiki = window.mediaWiki || {};

wikibase.queryService.ui.resultBrowser.ScatterChartResultBrowser = ( function( dimple ) {
	'use strict';

	var PARENT = wikibase.queryService.ui.resultBrowser.AbstractDimpleChartResultBrowser;

	/**
	 * A scatter chart result browser
	 *
	 * @class wikibase.queryService.ui.resultBrowser.ScatterChartResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 *
	 * @constructor
	 */
	function SELF() {
		this._dataColumns = {};
	}

	SELF.prototype = new PARENT();

	SELF.prototype._getPlotType = function() {
		return dimple.plot.scatter;
	};

	return SELF;
}( dimple ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};
window.mediaWiki = window.mediaWiki || {};

wikibase.queryService.ui.resultBrowser.AreaChartResultBrowser = ( function( dimple ) {
	'use strict';

	var PARENT = wikibase.queryService.ui.resultBrowser.AbstractDimpleChartResultBrowser;

	/**
	 * A area chart result browser
	 *
	 * @class wikibase.queryService.ui.resultBrowser.AreaChartResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 *
	 * @constructor
	 */
	function SELF() {
		this._dataColumns = {};
	}

	SELF.prototype = new PARENT();

	SELF.prototype._getPlotType = function() {
		return dimple.plot.area;
	};

	return SELF;
}( dimple ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.TimelineResultBrowser = ( function( $, vis, window, _ ) {
	'use strict';

	var TIMELINE_OPTIONS = {
		minHeight: '400px',
		orientation: {
			axis: 'both',
			item: 'top'
		},
		align: 'auto',
		zoomKey: 'ctrlKey'
	};

	/**
	 * A result browser for dateTime
	 *
	 * @class wikibase.queryService.ui.resultBrowser.TimelineResultBrowser
	 * @licence GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 *
	 */
	function SELF() {
	}

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractResultBrowser();

	/**
	 * Draw to the given element
	 *
	 * @param {jQuery} $element target element
	 */
	SELF.prototype.draw = function( $element ) {
		var $container = $( '<div>' );

		var timeline = new vis.Timeline( $container[0], this._getItems(), TIMELINE_OPTIONS );
		// copy width to min-width for T163984 hack
		timeline.on(
			'changed',
			function() {
				$( timeline.dom.root )
					.find( '.vis-item.vis-range' )
					.each(
						function() {
							var $this = $( this );
							/*
							 * First unset the min-width, so that $this.css( 'width' ) below returns the original width calculated by the timeline.
							 * Otherwise, repeated calls of this function would only ever increase the min-width, but never decrease it:
							 * .css() returns the *computed* style, i.e. min(width, min-width) if we don't unset min-width.
							 */
							$this.css( 'min-width', 'unset' );
							/*
							 * Now set the min-width to the width calculated by the timeline,
							 * so that wide time ranges are not shortened when the CSS hack for T163984 unsets the width.
							 * (Unsetting the width lets the browser expand the time range to fit its text content,
							 * but if the range is already wide enough for that, we don’t want the browser to shorten it.)
							 */
							$this.css( 'min-width', $this.css( 'width' ) );
						}
					);
			}
		);
		$element.append( $container.prepend( this._createToolbar( timeline ) ) );
	};

	/**
	 * @private
	 */
	SELF.prototype._getItems = function() {
		var self = this,
			items = [];

		this._iterateResult( function( field, key, row, rowIndex ) {
			if ( self._getFormatter().isDateTime( field ) ) {
				if ( !items[rowIndex] ) {// create new
					items[rowIndex] = {
						id: rowIndex,
						content: self._getHtml( row ).html(),
						start: self._getFormatter().parseDate( field.value )
					};
				} else { // create time span with start and end date
					var dates = [];
					dates.push( self._getFormatter().parseDate( field.value ) );
					if ( items[rowIndex].start ) {
						dates.push( items[rowIndex].start );
					}
					if ( items[rowIndex].end ) {
						dates.push( items[rowIndex].end );
					}

					items[rowIndex].start = dates.reduce( function( a, b ) {
						return a < b ? a : b;
					} );
					items[rowIndex].end = dates.reduce( function( a, b ) {
						return a > b ? a : b;
					} );
				}
			}
		} );

		return new vis.DataSet( _.compact( items ) );
	};

	/**
	 * @private
	 */
	SELF.prototype._getHtml = function( row ) {
		var $result = $( '<div/>' ).append( this._getFormatter().formatRow( row, true ) );

		return $result;
	};

	/**
	 * @private
	 */
	SELF.prototype._createToolbar = function( timeline ) {
		var $toolbar = $( '<div style="margin-top: -35px; text-align: center;">' );

		$( '<a class="btn btn-default">' ).click( $.proxy( timeline.redraw, timeline ) ).append(
				'<span class="glyphicon glyphicon-resize-vertical" aria-hidden="true"></span>' )
				.appendTo( $toolbar );
		$( '<a class="btn btn-default">' ).click( $.proxy( timeline.fit, timeline ) ).append(
				'<span class="glyphicon glyphicon-resize-horizontal" aria-hidden="true"></span>' )
				.appendTo( $toolbar );

		function zoom( percentage ) {
			var range = timeline.getWindow(),
				interval = range.end - range.start;

			timeline.setWindow( {
				start: range.start.valueOf() - interval * percentage,
				end: range.end.valueOf() + interval * percentage
			} );
		}

		$( '<a class="btn btn-default">' ).click( function() {
			zoom( 0.2 );
		} ).append( '<span class="glyphicon glyphicon-zoom-out" aria-hidden="true"></span>' )
				.appendTo( $toolbar );

		$( '<a class="btn btn-default">' ).click( function() {
			zoom( -0.2 );
		} ).append( '<span class="glyphicon glyphicon-zoom-in" aria-hidden="true"></span>' )
				.appendTo( $toolbar );

		return $toolbar;
	};

	/**
	 * Receiving data from the a visit
	 *
	 * @param {Object} data
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( data ) {
		if ( this._getFormatter().isDateTime( data ) ) {
			this._drawable = true;
			return false;
		}
		return true;
	};

	return SELF;
}( jQuery, vis, window, _ ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.MultiDimensionResultBrowser = ( function( $, d3, window ) {
	'use strict';

	var TIME_DATATYPE = 'http://www.w3.org/2001/XMLSchema#dateTime';
	/**
	 * A result browser for multi dimensional data
	 *
	 * @class wikibase.queryService.ui.resultBrowser.MultiDimensionResultBrowser
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 *
	 * @constructor
	 */
	function SELF() {
	}

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractChartResultBrowser();

	/**
	 * Draw browser to the given element
	 *
	 * @param {jQuery} $element to draw at
	 */
	SELF.prototype.draw = function( $element ) {
		var dimensions = {},
			data = [],
			f = this._getFormatter();

		this._iterateResult( function( field, key, row, rowNum ) {
			if ( !data[rowNum] ) {
				data[rowNum] = {};
			}

			data[rowNum][key] = field && field.value || 'undefined';
			if ( field && field.type === TIME_DATATYPE ) {
				data[rowNum][key] = new Date( field.value ).toLocaleDateString();
			}

			if ( !dimensions[key] ) {
				var d = {
					key: key,
					description: key,
					type: 'STRING'
				};
				if ( f.isNumber( field ) ) {
					d.type = 'NUMBER';
				}
				if ( field && field.type === TIME_DATATYPE ) {
					d.type = 'DATE';
				}
				dimensions[key] = d;
			}
		} );

		this._draw( $element, data, _.values( dimensions ) );
	};

	SELF.prototype._draw = function( $graph, data, dimensions ) {
		// source: http://bl.ocks.org/syntagmatic/94be812f8b410ae29ee2

		var margin = {
				top: 50,
				right: 80,
				bottom: 10,
				left: 100
			},
			width = $( window ).width() - ( margin.left + margin.right ),
			height = $( window ).height() - ( margin.top + margin.bottom );

		var types = {
			'NUMBER': {
				key: 'Number',
				coerce: function( d ) {
					return +d;
				},
				extent: d3.extent,
				within: function( d, extent ) {
					return extent[0] <= d && d <= extent[1];
				},
				defaultScale: d3.scale.linear().range( [ height, 0 ] )
			},
			'STRING': {
				key: 'String',
				coerce: String,
				extent: function( data ) {
					return data.sort();
				},
				within: function( d, extent, dim ) {
					return extent[0] <= dim.scale( d ) && dim.scale( d ) <= extent[1];
				},
				defaultScale: d3.scale.ordinal().rangePoints( [ 0, height ] )
			},
			'DATE': {
				key: 'Date',
				coerce: function( d ) {
					return new Date( d );
				},
				extent: d3.extent,
				within: function( d, extent ) {
					return extent[0] <= d && d <= extent[1];
				},
				defaultScale: d3.time.scale().range( [ 0, height ] )
			}
		};

		$.each( dimensions, function() {
			this.type = types[this.type];
		} );

		var x = d3.scale.ordinal().domain( dimensions.map( function( dim ) {
			return dim.key;
		} ) ).rangePoints( [ 0, width ] );

		var line = d3.svg.line().defined( function( d ) {
			return !isNaN( d[1] );
		} );

		var yAxis = d3.svg.axis().orient( 'left' );

		var svg = d3.select( $graph[0] )
			.append( 'svg' ).attr( {
				width: width + margin.left + margin.right,
				height: height + margin.top + margin.bottom
			} )
			.append( 'g' ).attr( 'transform', 'translate(' + margin.left + ',' + margin.top + ')' );

		var foreground = svg.append( 'g' ).attr( 'class', 'foreground' );

		var axes = svg.selectAll( '.axis' ).data( dimensions ).enter().append( 'g' ).attr( 'class',
				'axis' ).attr( 'transform', function( d ) {
			return 'translate(' + x( d.key ) + ')';
		} );

		data.forEach( function( d ) {
			dimensions.forEach( function( p ) {
				d[p.key] = p.type.coerce( d[p.key] );
			} );
		} );

		function draw( d ) {
			return line( dimensions.map( function( dim ) {
				return [
						x( dim.key ), dim.scale( d[dim.key] )
				];
			} ) );
		}

		function brushstart() {
			d3.event.sourceEvent.stopPropagation();
		}

		// Handles a brush event, toggling the display of foreground lines.
		function brush() {
			var actives = dimensions.filter( function( p ) {
					return !p.brush.empty();
				} ),
				extents = actives.map( function( p ) {
					return p.brush.extent();
				} );

			d3.selectAll( '.foreground path' ).style( 'display', function( d ) {
				if ( actives.every( function( dim, i ) {
					// test if point is within extents for each active brush
					return dim.type.within( d[dim.key], extents[i], dim );
				} ) ) {
					return null;
				}
				return 'none';
			} );
		}

		// type/dimension default setting happens here
		dimensions.forEach( function( dim ) {
			if ( !( 'domain' in dim ) ) {
				// detect domain using dimension type's extent function
				dim.domain = d3.functor( dim.type.extent )( data.map( function( d ) {
					return d[dim.key];
				} ) );

				// TODO - this line only works because the data encodes data with integers
				// Sorting/comparing should be defined at the type/dimension level
				dim.domain.sort( function( a, b ) {
					return a - b;
				} );
			}
			if ( !( 'scale' in dim ) ) {
				// use type's default scale for dimension
				dim.scale = dim.type.defaultScale.copy();
			}
			dim.scale.domain( dim.domain );
		} );

		foreground.selectAll( 'path' ).data( data ).enter().append( 'path' ).attr( 'd', draw )
				.style( 'stroke', function( d ) {
					return '#6ac';
				} );

		axes.append( 'g' ).attr( 'class', 'axis' ).each( function( d ) {
			var renderAxis = ( d.axis && d.axis.scale( d.scale ) ) // custom axis
				|| yAxis.scale( d.scale ); // default axis
			d3.select( this ).call( renderAxis );
		} ).append( 'text' ).attr( 'class', 'title' ).attr( 'text-anchor', 'start' ).text(
				function( d ) {
					return d.description || d.key;
				} );

		// Add and store a brush for each axis.
		axes.append( 'g' ).attr( 'class', 'brush' ).each(
				function( d ) {
					d3.select( this ).call(
							d.brush = d3.svg.brush().y( d.scale ).on( 'brushstart', brushstart )
									.on( 'brush', brush ) );
				} ).selectAll( 'rect' ).attr( 'x', -8 ).attr( 'width', 16 );
	};

	/**
	 * Checks whether the browser can draw the given result
	 *
	 * @return {boolean}
	 */
	SELF.prototype.isDrawable = function() {
		return this._drawable;
	};

	/**
	 * Receiving data from the visit
	 *
	 * @param {Object} data
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( data ) {
		this._drawable = true;
		return false;
	};

	return SELF;
}( jQuery, d3, window ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.GraphResultBrowser = ( function( $, vis, window, _ ) {
	'use strict';

	var GRAPH_OPTIONS = {
		autoResize: true,
		physics: {
			stabilization: {
				enabled: true,
				iterations: 10,
				fit: true
			},
			barnesHut: {
				springLength: 150,
				centralGravity: 0.5,
				damping: 0.2,
				avoidOverlap: 0.1
			}
		},
		nodes: {
			shadow: true,
			color: '#fff'
		},
		edges: {
			arrows: {
				to: true
			}
		}
	};

	/**
	 * A graph result browser
	 *
	 * @class wikibase.queryService.ui.resultBrowser.GraphResultBrowser
	 * @licence GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 *
	 */
	function SELF() {
	}

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractResultBrowser();

	/**
	 * Draw to the given element
	 *
	 * @param {jQuery} $element target element
	 */
	SELF.prototype.draw = function( $element ) {
		var $container = $( '<div>' ).height( '100vh' );

		var data = this._getData();
		var network = new vis.Network( $container[0], data, GRAPH_OPTIONS );

		network.on( 'doubleClick', function( properties ) {
			if ( properties.nodes.length === 1 ) {
				window.open( properties.nodes[0], '_blank' );
			}
		} );

		var nodeBrowser = new wikibase.queryService.ui.resultBrowser.GraphResultBrowserNodeBrowser( data.nodes, data.edges, this.getSparqlApi() );
		network.on( 'click', function( properties ) {
			var nodeId = properties.nodes[0] || null;
			nodeBrowser.browse( nodeId );
		} );

		$container.prepend( this._createToolbar( network ) );
		$element.append( $container );
	};

	/**
	 * @private
	 */
	SELF.prototype._createToolbar = function( network ) {
		var $toolbar = $( '<div style="margin-top: -35px; text-align: center;">' );

		function setLayout( type ) {
			network.setOptions( {
				layout: {
					hierarchical: {
						direction: type,
						sortMethod: 'directed'
					}
				}
			} );
		}

		$( '<a class="btn btn-default">' ).click( function() {
			network.stabilize( 100 );
		} ).append(
			'<span class="glyphicon glyphicon-fullscreen" aria-hidden="true" title="' +
			this._i18n( 'wdqs-app-resultbrowser-stabilize' ) +
			'"></span>'
		).appendTo( $toolbar );

		$( '<a class="btn btn-default">' ).click( function() {
			setLayout( 'LR' );
		} ).append( '<span class="glyphicon glyphicon-indent-left" aria-hidden="true" title="' +
			this._i18n( 'wdqs-app-resultbrowser-hierarchical-lr' ) +
			'"></span>'
		).appendTo( $toolbar );

		$( '<a class="btn btn-default">' ).click( function() {
			setLayout( 'UD' );
		} ).append( '<span class="glyphicon glyphicon-align-center" aria-hidden="true" title="' +
			this._i18n( 'wdqs-app-resultbrowser-hierarchical-ud' ) +
			'"></span>'
		).appendTo( $toolbar );

		$( '<a class="btn btn-default">' ).click( function() {
			setLayout( 'RL' );
		} ).append( '<span class="glyphicon glyphicon-indent-right" aria-hidden="true" title="' +
			this._i18n( 'wdqs-app-resultbrowser-hierarchical-rl' ) +
			'"></span>'
		).appendTo( $toolbar );

		return $toolbar;
	};

	/**
	 * @private
	 */
	SELF.prototype._getData = function() {
		var nodes = {},
			edges = {},
			rows = [],
			format = this._getFormatter(),
			node = {},
			edge = {};

		this._iterateResult( function( field, key, row, rowIndex ) {
			if ( !field || !field.value ) {
				return;
			}
			if ( format.isEntity( field ) ) {
				// create node
				var label = row[key + 'Label'] && row[key + 'Label'].value || field.value,
					nodeId = field.value;
				node = {
					id: nodeId,
					label: label,
					title: label
				};
				if ( rows[rowIndex] ) {// create new edge
					edge = {
							from: rows[rowIndex],
							to: nodeId
						};
					edges[ edge.from + edge.to ] = edge;
					if ( !nodes[nodeId] ) {// create new node if not exist
						nodes[nodeId] = node;
					}
				} else {
					nodes[nodeId] = node;
					rows[rowIndex] = node.id;
				}
			}
			if ( format.isCommonsResource( field.value ) ) {
				node.image = format.getCommonsResourceFileNameThumbnail( field.value, 150 );
				node.shape = 'image';
				node.font = { color: 'black' };
			}

			if ( format.isNumber( field ) ) {
				node.value = field.value;
				node.title += ' value:' + field.value;
				node.shape = 'dot';
				node.font = { color: 'black' };
			}

			if ( key === 'rgb' && format.isColor( field ) ) {
				node.color = format.getColorForHtml( field );
				if ( node.shape !== 'dot' && node.shape !== 'image' ) {
					var foreground = format.calculateLuminance( field.value ) <= 0.5 ? '#FFF' : '#000';
					node.font = { color: foreground };
				}
			}

			if ( key === 'edgeLabel' ) {
				edge.label = field.value;
			}
		} );

		return {
			nodes: new vis.DataSet( _.compact( nodes ) ),
			edges: new vis.DataSet( _.compact( edges ) )
		};
	};

	/**
	 * Receiving data from the visit
	 *
	 * @param {Object} data
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( data ) {
		if ( this._getFormatter().isEntity( data ) ) {
			this._drawable = true;
			return false;
		}
		return true;
	};

	return SELF;
}( jQuery, vis, window, _ ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.GraphResultBrowserNodeBrowser = ( function( $, vis, window, _ ) {
	'use strict';

	var SPARQL_PROPERTIES = 'SELECT ?p (SAMPLE(?pl) AS ?pl) (COUNT(?o) AS ?count ) (group_concat(?ol;separator=", ") AS ?ol)  WHERE {'
			+ '<{entityUri}> ?p ?o .'
			+ '   ?o <http://www.w3.org/2000/01/rdf-schema#label> ?ol .'
			+ '    FILTER ( LANG(?ol) = "[AUTO_LANGUAGE]" )'
			+ '    ?s <http://wikiba.se/ontology#directClaim> ?p .' + '    ?s rdfs:label ?pl .'
			+ '    FILTER ( LANG(?pl) = "[AUTO_LANGUAGE]" )' + '} group by ?p';

	var SPARQL_ENTITES = 'SELECT ?o ?ol WHERE {' + '<{entityUri}> <{propertyUri}> ?o .'
			+ '?o <http://www.w3.org/2000/01/rdf-schema#label> ?ol .'
			+ 'FILTER ( LANG(?ol) = "[AUTO_LANGUAGE]" )' + '} LIMIT 50';

	/**
	 * A browser for network nodes
	 *
	 * @constructor
	 * @param {DataSet} nodes
	 * @param {DataSet} edges
	 * @param {wikibase.queryService.api.Sparql} sparqlApi
	 */
	function SELF( nodes, edges, sparqlApi ) {
		this._nodes = nodes;
		this._edges = edges;

		this._sparql = sparqlApi;
	}

	/**
	 * @property {DataSet}
	 * @private
	 */
	SELF.prototype._nodes = null;

	/**
	 * @property {DataSet}
	 * @private
	 */
	SELF.prototype._nodes = null;

	/**
	 * @property {DataSet}
	 * @private
	 */
	SELF.prototype._sparql = null;

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._selectedNodeId = null;

	/**
	 * @property {object}
	 * @private
	 */
	SELF.prototype._temporaryNodes = {};

	/**
	 * @property {object}
	 * @private
	 */
	SELF.prototype._temporaryEdges = {};

	/**
	 * @private
	 */
	SELF.prototype._getEntites = function( entityUri, propertyUri ) {
		var self = this,
			deferred = $.Deferred();

		this._sparql.query(
				SPARQL_ENTITES.replace( '{entityUri}', entityUri ).replace( '{propertyUri}',
						propertyUri ) ).done( function() {
			var data = self._sparql.getResultRawData();
			var result = [];

			$.each( data.results.bindings, function( i, row ) {
				result.push( {
					id: row.o.value,
					label: row.ol.value
				} );
			} );

			deferred.resolve( result );
		} );

		return deferred;
	};

	/**
	 * @private
	 */
	SELF.prototype._getProperties = function( entityUri ) {
		var self = this,
			deferred = $.Deferred();

		this._sparql.query( SPARQL_PROPERTIES.replace( '{entityUri}', entityUri ) ).done(
				function() {
					var data = self._sparql.getResultRawData();
					var result = [];

					$.each( data.results.bindings, function( i, row ) {
						result.push( {
							id: row.p.value,
							label: row.pl.value,
							count: row.count.value,
							items: row.ol.value
						} );
					} );

					deferred.resolve( result );
				} );

		return deferred;
	};

	/**
	 * @private
	 */
	SELF.prototype._removeTemporaryNodes = function( entityUri ) {
		var self = this;

		$.each( this._temporaryNodes, function( i, n ) {
			self._nodes.remove( n.id );
		} );
		$.each( this._temporaryEdges, function( i, e ) {
			self._edges.remove( e.id );
		} );

		this._temporaryNodes = {};
		this._temporaryEdges = {};
	};

	/**
	 * @private
	 */
	SELF.prototype._expandPropertyNode = function( nodeId ) {
		var self = this,
			node = this._temporaryNodes[nodeId];

		this._getEntites( node.entityId, node.id ).done( function( entites ) {
			$.each( entites, function( i, e ) {
				if ( self._nodes.get( e.id ) === null ) {
					self._nodes.add( {
						id: e.id,
						label: e.label
					} );
				}
				self._edges.add( {
					dashes: true,
					from: node.entityId,
					to: e.id,
					label: node.propertyLabel,
					linkType: node.id
				} );
			} );
		} );
	};

	/**
	 * @private
	 */
	SELF.prototype._expandEntityNode = function( nodeId ) {
		var self = this;

		this._getProperties( nodeId ).done( function( properties ) {
			$.each( properties, function( i, p ) {
				// if already expanded skip
				if ( self._edges.get( {
					filter: function( e ) {
						return e.linkType === p.id && e.from === nodeId;
					}
				} ).length > 0 ) {
					return;
				}

				var node = {
					id: p.id,
					label: p.count === '1' ? p.items : p.count,
					title: p.items,
					entityId: nodeId,
					propertyLabel: p.label,
					color: '#abc9f2'
				};
				var edge = {
					id: p.id,
					dashes: true,
					label: p.label,
					from: nodeId,
					to: p.id
				};
				self._temporaryNodes[node.id] = node;
				self._nodes.add( node );

				self._temporaryEdges[edge.id] = edge;
				self._edges.add( edge );
			} );
		} );
	};

	/**
	 * Browse a node
	 *
	 * @param {string} nodeId
	 */
	SELF.prototype.browse = function( nodeId ) {
		if ( nodeId === null ) {
			this._removeTemporaryNodes();
			return;
		}

		if ( this._temporaryNodes[nodeId] ) {
			this._expandPropertyNode( nodeId );
			this._removeTemporaryNodes();
			return;
		}

		if ( this._selectedNodeId !== null && nodeId !== this._selectedNodeId ) {
			this._removeTemporaryNodes();
		}
		this._expandEntityNode( nodeId );
		this._selectedNodeId = nodeId;
	};

	return SELF;
}( jQuery, vis, window, _ ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.ui = wikibase.queryService.ui || {};
wikibase.queryService.ui.resultBrowser = wikibase.queryService.ui.resultBrowser || {};

wikibase.queryService.ui.resultBrowser.PolestarResultBrowser = ( function( $, window, _ ) {
	'use strict';

	var POLESTAR = 'polestar/embed.html';
	var GRAPH_QUERY_PREFIX = 'wikidatasparql://query.wikidata.org/?query=';

	/**
	 * A polestar graph maker.
	 *
	 * @class wikibase.queryService.ui.resultBrowser.PolestarResultBrowser
	 * @licence GNU GPL v2+
	 *
	 * @constructor
	 *
	 */
	function SELF() {
	}

	SELF.prototype = new wikibase.queryService.ui.resultBrowser.AbstractResultBrowser();

	/**
	 * Draw to the given element
	 *
	 * @param {jQuery} $element target element
	 */
	SELF.prototype.draw = function( $element ) {
		var polestarData = {
			url: GRAPH_QUERY_PREFIX + location.hash.substr( 1 ),
			name: 'Imported from Wikidata Query Service',
			_directEmbed: true
		};
		var $container = $( '<iframe>' ).attr( {
			'src': POLESTAR + '#' + encodeURIComponent( JSON.stringify( polestarData ) ),
			'class': 'graph-iframe'
		} ).height( '98vh' );

		$element.append( $container );
		window.setTimeout( function() { $container.scrollIntoView(); }, 50 );
	};

	/**
	 * Receiving data from the visit
	 *
	 * @param {Object} data
	 * @return {boolean} false if there is no revisit needed
	 */
	SELF.prototype.visit = function( data ) {
		this._drawable = true;
		return false;
	};

	return SELF;
}( jQuery, window, _ ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.api = wikibase.queryService.api || {};

wikibase.queryService.api.Sparql = ( function( $ ) {
	'use strict';

	var SPARQL_SERVICE_URI = 'https://query.wikidata.org/bigdata/namespace/wdq/sparql',
		ERROR_CODES = {
			TIMEOUT: 10,
			MALFORMED: 20,
			SERVER: 30,
			UNKNOWN: 100
		},
		ERROR_MAP = {
			'QueryTimeoutException: Query deadline is expired': ERROR_CODES.TIMEOUT,
			'java.util.concurrent.TimeoutException': ERROR_CODES.TIMEOUT,
			'MalformedQueryException: ': ERROR_CODES.MALFORMED
		},
		DEFAULT_LANGUAGE = 'en';

	/**
	 * SPARQL API for the Wikibase query service
	 *
	 * @class wikibase.queryService.api.Sparql
	 * @license GNU GPL v2+
	 *
	 * @author Stanislav Malyshev
	 * @author Jonas Kress
	 * @constructor
	 *
	 * @param {string} [serviceUri] Optional URI to the SPARQL service endpoint
	 * @param {string} [language]
	 */
	function SELF( serviceUri, language ) {
		this._serviceUri = serviceUri || SPARQL_SERVICE_URI;
		this._language = language || DEFAULT_LANGUAGE;
	}

	/**
	 * @property {Object}
	 */
	SELF.prototype.ERROR_CODES = ERROR_CODES;

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._serviceUri = null;

	/**
	 * @property {number}
	 * @private
	 */
	SELF.prototype._executionTime = null;

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._error = null;

	/**
	 * @property {number}
	 * @private
	 */
	SELF.prototype._resultLength = null;

	/**
	 * @property {Object}
	 * @private
	 */
	SELF.prototype._rawData = null;

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._queryUri = null;

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._language = null;

	/**
	 * Submit a query to the API
	 *
	 * @return {jQuery.Promise}
	 */
	SELF.prototype.queryDataUpdatedTime = function() {
		// Cache the update time only for a minute
		var deferred = $.Deferred(),
			query = encodeURI( 'prefix schema: <http://schema.org/> '
				+ 'SELECT * WHERE {<http://www.wikidata.org> schema:dateModified ?y}' ),
			url = this._serviceUri + '?query=' + query + '&nocache='
				+ Math.floor( Date.now() / 60000 ),
			settings = {
				headers: {
					Accept: 'application/sparql-results+json'
				}
			};

		$.ajax( url, settings ).done( function( data, textStatus, jqXHR ) {
			if ( !data.results.bindings[0] ) {
				deferred.reject();
				return;
			}

			var updateDate = new Date( data.results.bindings[0][data.head.vars[0]].value ),
				dateText = updateDate.toLocaleTimeString( navigator.language, {
						timeZoneName: 'short'
					} ) + ', ' + updateDate.toLocaleDateString( navigator.language, {
						month: 'short',
						day: 'numeric',
						year: 'numeric'
					} ),
				differenceInSeconds = Math.round( ( new Date() - updateDate ) / 1000 );

			deferred.resolve( dateText, differenceInSeconds );
		} ).fail( function() {
			deferred.reject();
		} );

		return deferred;
	};

	/**
	 * Submit a query to the API
	 *
	 * @param {string[]} query
	 * @param {number} [timeout] in millis
	 * @return {jQuery.Promise} query
	 */
	SELF.prototype.query = function( query, timeout ) {
		query = this._replaceAutoLanguage( query );

		var data = 'query=' + encodeURIComponent( query );
		if ( timeout ) {
			data += '&maxQueryTimeMillis=' + timeout;
		}

		var self = this,
			deferred = $.Deferred(),
			settings = {
				headers: { Accept: 'application/sparql-results+json' },
				data: data
			};
		function done( data, textStatus, request ) {
			self._executionTime = Date.now() - self._executionTime;

			if ( typeof data.boolean === 'boolean' ) {
				self._resultLength = 1;
			} else {
				self._resultLength = data.results.bindings.length || 0;
			}
			self._rawData = data;

			deferred.resolve( data );
		}
		function fail( request, options, exception ) {
			self._executionTime = null;
			self._rawData = null;
			self._resultLength = null;
			self._generateErrorMessage( request, options, exception );

			deferred.reject();
		}

		this._queryUri = this._serviceUri + '?' + settings.data;

		this._executionTime = Date.now();
		$.ajax( this._serviceUri, settings ).done( done ).fail( function( request, options, exception ) {
			if ( request.getAllResponseHeaders() === '' ) {
				// query might have been too long for GET, retry with POST
				settings.method = 'POST';
				$.ajax( self._serviceUri, settings ).done( done ).fail( fail );
			} else {
				fail.apply( this, arguments );
			}
		} );

		return deferred;
	};

	/**
	 * Get execution time in ms of the submitted query
	 */
	SELF.prototype._generateErrorMessage = function( request, options, exception ) {
		var error = {
			code: ERROR_CODES.UNKNOWN,
			message: null,
			debug: request.responseText
		};

		if ( request.status === 0 || exception ) {
			error.code = ERROR_CODES.SERVER;
			error.message = exception.message;
		}

		try {//extract error from server response
			var errorToMatch = error.debug.substring(
				error.debug.indexOf( 'java.util.concurrent.ExecutionException:' )
			);

			for ( var errorKey in ERROR_MAP ) {
				if ( errorToMatch.indexOf( errorKey ) !== -1 ) {
					error.code = ERROR_MAP[ errorKey ];
					error.message = null;
				}
			}

			if ( error.code === ERROR_CODES.UNKNOWN || error.code === ERROR_CODES.MALFORMED ) {
				error.message = error.debug
						.match(
								/(java\.util\.concurrent\.ExecutionException\:)+(.*)(Exception\:)+(.*)/ )
						.pop().trim();
			}
		} catch ( e ) {
		}

		this._error = error;
	};

	/**
	 * Get execution time in seconds of the submitted query
	 *
	 * @return {number}
	 */
	SELF.prototype.getExecutionTime = function() {
		return this._executionTime;
	};

	/**
	 * Get error of the submitted query if it has failed
	 *
	 * @return {object}
	 */
	SELF.prototype.getError = function() {
		return this._error;
	};

	/**
	 * Get result length of the submitted query if it has failed
	 *
	 * @return {number}
	 */
	SELF.prototype.getResultLength = function() {
		return this._resultLength;
	};

	/**
	 * Get query URI
	 *
	 * @return {string}
	 */
	SELF.prototype.getQueryUri = function() {
		return this._queryUri;
	};

	/**
	 * Process SPARQL query result.
	 *
	 * @param {Object} data
	 * @param {Function} rowHandler
	 * @param {*} context
	 * @private
	 * @return {*} The provided context, modified by the rowHandler.
	 */
	SELF.prototype._processData = function( data, rowHandler, context ) {
		var results = data.results.bindings.length;
		for ( var i = 0; i < results; i++ ) {
			var rowBindings = {};
			for ( var j = 0; j < data.head.vars.length; j++ ) {
				if ( data.head.vars[j] in data.results.bindings[i] ) {
					rowBindings[data.head.vars[j]] = data.results.bindings[i][data.head.vars[j]];
				} else {
					rowBindings[data.head.vars[j]] = undefined;
				}
			}
			context = rowHandler( rowBindings, context );
		}
		return context;
	};

	/**
	 * Encode string as CSV.
	 *
	 * @param {string} string
	 * @return {string}
	 */
	SELF.prototype._encodeCsv = function( string ) {
		var result = string.replace( /"/g, '""' );
		if ( /[",\n]/.test( result ) ) {
			result = '"' + result + '"';
		}
		return result;
	};

	/**
	 * Get the raw result
	 *
	 * @return {Object} result
	 */
	SELF.prototype.getResultRawData = function() {
		return this._rawData;
	};

	/**
	 * Get the result of the submitted query as CSV
	 *
	 * @return {string} csv
	 */
	SELF.prototype.getResultAsCsv = function() {
		var self = this,
			data = self._rawData,
			output = data.head.vars.map( this._encodeCsv ).join( ',' ) + '\n';

		output = this._processData( data, function( row, out ) {
			var rowOut = '';
			var first = true;
			var rowCSV;
			for ( var rowVar in row ) {
				if ( row[rowVar] === undefined ) {
					rowCSV = '';
				} else {
					rowCSV = self._encodeCsv( row[rowVar].value );
				}
				if ( !first ) {
					rowOut += ',';
				} else {
					first = false;
				}
				rowOut += rowCSV;
			}
			rowOut += '\n';
			return out + rowOut;
		}, output );
		return output;
	};

	/**
	 * Get the result of the submitted query as JSON
	 *
	 * @return {string}
	 */
	SELF.prototype.getResultAsJson = function() {
		var output = [],
			data = this._rawData;

		output = this._processData( data, function( row, out ) {
			var extractRow = {};
			for ( var rowVar in row ) {
				extractRow[rowVar] = ( row[rowVar] || {} ).value;
			}
			out.push( extractRow );
			return out;
		}, output );
		return JSON.stringify( output );
	};

	/**
	 * Get the result of the submitted query as raw JSON
	 *
	 * @return {string}
	 */
	SELF.prototype.getResultAsAllJson = function() {
		return JSON.stringify( this._rawData );
	};

	/**
	 * Render value as per http://www.w3.org/TR/sparql11-results-csv-tsv/#tsv
	 *
	 * @param {Object} binding
	 * @return {string}
	 */
	SELF.prototype._renderValueTSV = function( binding ) {
		var value = binding.value.replace( /\t/g, '' );
		switch ( binding.type ) {
		case 'uri':
			return '<' + value + '>';
		case 'bnode':
			return '_:' + value;
		case 'literal':
			var lvalue = JSON.stringify( value );
			if ( binding['xml:lang'] ) {
				return lvalue + '@' + binding['xml:lang'];
			}
			if ( binding.datatype ) {
				if ( binding.datatype === 'http://www.w3.org/2001/XMLSchema#integer' ||
						binding.datatype === 'http://www.w3.org/2001/XMLSchema#decimal' ||
						binding.datatype === 'http://www.w3.org/2001/XMLSchema#double' ) {
					return value;
				}
				return lvalue + '^^<' + binding.datatype + '>';
			}
			return lvalue;
		}
		return value;
	};

	/**
	 * Get the result of the submitted query as SPARQL TSV
	 *
	 * @return {string}
	 */
	SELF.prototype.getSparqlTsv = function() {
		var self = this,
			data = this._rawData,
			output = data.head.vars.map( function( vname ) {
			return '?' + vname;
		} ).join( '\t' ) + '\n';

		output = this._processData( data, function( row, out ) {
			var rowOut = '';
			var first = true;
			var rowTSV;
			for ( var rowVar in row ) {
				if ( row[rowVar] === undefined ) {
					rowTSV = '';
				} else {
					rowTSV = self._renderValueTSV( row[rowVar] );
				}
				if ( !first ) {
					rowOut += '\t';
				} else {
					first = false;
				}
				rowOut += rowTSV;
			}
			rowOut += '\n';
			return out + rowOut;
		}, output );
		return output;
	};

	/**
	 * Get the result of the submitted query as simplified TSV
	 *
	 * @return {string}
	 */
	SELF.prototype.getSimpleTsv = function() {
		var data = this._rawData,
			output = data.head.vars.join( '\t' ) + '\n';

		output = this._processData( data, function( row, out ) {
			var rowOut = '';
			var first = true;
			var rowTSV;
			for ( var rowVar in row ) {
				if ( row[rowVar] === undefined ) {
					rowTSV = '';
				} else {
					rowTSV = row[rowVar].value.replace( /\t/g, '' );
				}
				if ( !first ) {
					rowOut += '\t';
				} else {
					first = false;
				}
				rowOut += rowTSV;
			}
			rowOut += '\n';
			return out + rowOut;
		}, output );
		return output;
	};

	/**
	 * @private
	 */
	SELF.prototype._replaceAutoLanguage = function( query ) {
		return query.replace( /\[AUTO_LANGUAGE\]/g, this._language );
	};

	/**
	 * Set the default language
	 *
	 * @param {string} language of search string default:en
	 */
	SELF.prototype.setLanguage = function( language ) {
		this._language = language;
	};

	return SELF;

}( jQuery ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.api = wikibase.queryService.api || {};

wikibase.queryService.api.CodeSamples = ( function ( $ ) {
	'use strict';

	/**
	 * CodeSamples API for the Wikibase query service
	 *
	 * @class wikibase.queryService.api.CodeSamples
	 * @license GNU GPL v2+
	 *
	 * @author Lucas Werkmeister
	 * @author Jonas Kress
	 * @constructor
	 */
	function SELF( endpoint, root, index ) {
		this._endpoint = endpoint;
		this._languages = {
			URL: {
				code: function( query ) {
					return endpoint + '?query=' + encodeURIComponent( query );
				}
			},
			HTML: {
				code: function( query ) {
					return '<iframe style="width: 80vw; height: 50vh; border: none;" ' +
						'src="' + root + '/embed.html#' +
						encodeURIComponent( query ) + '" ' +
						'referrerpolicy="origin" ' +
						'sandbox="allow-scripts allow-same-origin allow-popups"></iframe>';
				},
				mimetype: 'text/html'
			},
			Wikilink: {
				code: function( query ) {
					return '[' + index + '#' + encodeURIComponent( query ) + ' Query]';
				}
			},
			PHP: {
				escape: function( query ) {
					var escapedQuery = query
						.replace( /\\/g, '\\\\' )
						.replace( /"/g, '\\"' )
						.replace( /\$/g, '\\$' )
						.replace( /\n/g, '\\n' );
					return '"' + escapedQuery + '"';
				}
			},
			'JavaScript (jQuery)': {
				escape: function( query ) {
					var escapedQuery = query
						.replace( /\\/g, '\\\\' )
						.replace( /"/g, '\\"' )
						.replace( /\n/g, '\\n' );
					return '"' + escapedQuery + '"';
				},
				mimetype: 'application/javascript'
			},
			'JavaScript (modern)': {
				escape: function( query ) {
					var escapedQuery = query
						.replace( /\\/g, '\\\\' )
						.replace( /`/g, '\\`' )
						.replace( /\$/g, '\\$' );
					return '`' + escapedQuery + '`';
				},
				mimetype: 'application/javascript'
			},
			Java: {
				escape: function( query ) {
					var escapedQuery = query
						.replace( /\\/g, '\\\\' )
						.replace( /"/g, '\\"' )
						.replace( /\n/g, '\\n' );
					return '"' + escapedQuery + '"';
				}
			},
			Python: {
				escape: function( query ) {
					var escapedQuery = query
						.replace( /\\/g, '\\\\' )
						.replace( /"/g, '\\"' )
						.replace( /\n/g, '\\n' );
					return '"' + escapedQuery + '"';
				}
			},
			Ruby: {
				escape: function( query ) {
					var escapedQuery = query
						.replace( /\\/g, '\\\\' )
						.replace( /"/g, '\\"' )
						.replace( /#/g, '\\#' )
						.replace( /\n/g, '\\n' );
					return '"' + escapedQuery + '"';
				}
			},
			R: {
				escape: function( query ) {
					var escapedQuery = query
						.replace( /\\/g, '\\\\' )
						.replace( /'/g, '\\\'' )
						.replace( /\n/g, '\\n' );
					return '\'' + escapedQuery + '\'';
				},
				mimetype: 'text/x-rsrc'
			},
			Matlab: {
				escape: function( query ) {
					var escapedQuery = query
						.replace( /\\/g, '\\\\' )
						.replace( /'/g, '\'\'' )
						.replace( /\n/g, '\\n' );
					return '\'' + escapedQuery + '\'';
				},
				mimetype: 'text/x-octave'
			},
			listeria: {
				escape: function( query ) {
					var escapedQuery = query
						.replace( /\|/g, '{{!}}' )
						.replace( /}}/g, '} }' ); // TODO try to exactly preserve query
					return escapedQuery;
				}
			}
		};
	}

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._endpoint = null;

	/**
	 * @return {jQuery.Promise} Object taking list of example queries { title:, query: }
	 */
	SELF.prototype.getExamples = function ( currentQuery ) {
		var self = this,
			deferred = new $.Deferred(),
			data = {},
			loadFiles = [];

		$.each( this._languages, function ( name, language ) {
			data[name] = {
				mimetype: 'mimetype' in language ?
					language.mimetype :
					'text/x-' + name.toLowerCase()
			};
			var query = currentQuery;
			if ( 'escape' in language ) {
				query = language.escape( query );
			}
			if ( 'code' in language ) {
				data[name].code = language.code( query );
			} else {
				loadFiles.push(
					$.get(
						'examples/code/' + name + '.txt',
						function ( code ) {
							data[name].code = code
								.replace( '{ENDPOINT_URL}', self._endpoint )
								.replace( '{SPARQL_QUERY}', query );
						},
						'text'
					)
				);
			}
		} );

		$.when.apply( $, loadFiles ).then( function () {
			deferred.resolve( data );
		} );

		return deferred.promise();
	};

	return SELF;

}( jQuery ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.api = wikibase.queryService.api || {};

wikibase.queryService.api.QuerySamples = ( function ( $ ) {
	'use strict';

	/**
	 * QuerySamples API for the Wikibase query service
	 *
	 * @class wikibase.queryService.api.QuerySamples
	 * @license GNU GPL v2+
	 *
	 * @author Stanislav Malyshev
	 * @author Jonas Kress
	 * @constructor
	 */
	function SELF( language ) {
		this._language = language;
		this._apiServer = 'https://www.wikidata.org/';
		this._apiEndpoint = this._apiServer + 'api/rest_v1/page/html/';
		this._pageTitle = 'Wikidata:SPARQL_query_service/queries/examples';
		this._pageUrl = this._apiServer + 'wiki/' + this._pageTitle;
	}

	/**
	 * @type {string}
	 * @private
	 */
	SELF.prototype._language = null;

	/**
	 * @return {jQuery.Promise} Object taking list of example queries { title:, query: }
	 */
	SELF.prototype.getExamples = function () {
		var self = this;

		return $.ajax( {
			url: self._apiEndpoint + encodeURIComponent( self._pageTitle + '/' + self._language ) + '?redirect=false',
			dataType: 'html'
		} ).catch( function() {
			// retry without language
			return $.ajax( {
				url: self._apiEndpoint + encodeURIComponent( self._pageTitle ) + '?redirect=false',
				dataType: 'html'
			} );
		} ).then( function ( data ) {
			return self._parseHTML( data );
		} );
	};

	/**
	 * Find closest header element one higher than this one
	 *
	 * @param {Element} element Header element
	 * @return {null|Element} Header element
     * @private
     */
	SELF.prototype._findPrevHeader = function ( element ) {
		var tag = element.prop( 'tagName' );
		if ( tag[0] !== 'H' && tag[0] !== 'h' ) {
			return null;
		}
		return this._findPrev( element, 'h' + ( tag.substr( 1 ) - 1 ) );
	};

	/**
	 * Find previous element matching the selector
	 *
	 * @param {Element} element
	 * @param {string} selector
	 * @return {Element}
     * @private
     */
	SELF.prototype._findPrev = function ( element, selector ) {
		var prev = element.prev().filter( selector );
		if ( prev.length > 0 ) {
			return prev;
		}
		prev = element.prevUntil( selector ).last().prev();
		if ( prev.length > 0 ) {
			return prev;
		}
		prev = element.parent().prev().filter( selector );
		return prev;
	};

	SELF.prototype._extractTagsFromSPARQL = function ( sparql ) {
		var tags = sparql.match( /\b[QP]\d+\b/g );

		if ( !tags ) {
			return [];
		}

		return tags;
	};

	SELF.prototype._parseHTML = function ( html ) {
		var div = document.createElement( 'div' ),
			self = this;
		div.innerHTML = html;
		// Find all SPARQL Templates
		var examples = $( div ).find( '[data-mw]' ).map( function() {
			var $this = $( this ),
				dataMW = $this.attr( 'data-mw' );
			if ( !dataMW ) {
				return;
			}

			var data = JSON.parse( dataMW ),
				templateHref,
				query;

			if ( data.parts && data.parts[0].template ) {
				templateHref = data.parts[0].template.target.href;
				if ( templateHref === './Template:SPARQL' || templateHref === './Template:SPARQL2' ) {
					// SPARQL/SPARQL2 template
					query = data.parts[0].template.params.query.wt;
				} else {
					return null;
				}
			} else {
				return null;
			}
			// Fix {{!}} hack
			query = query.replace( /\{\{!}}/g, '|' );

			// Find preceding title element
			var titleEl = self._findPrev( $this, 'h2,h3,h4,h5,h6,h7' );
			if ( !titleEl || !titleEl.length ) {
				return null;
			}
			var title = titleEl.text().trim();

			return {
				title:    title,
				query:    query,
				href:     self._pageUrl + '#' + encodeURIComponent( title.replace( / /g, '_' ) ).replace( /%/g, '.' ),
				tags:     self._extractTagsFromSPARQL( query ),
				category: self._findPrevHeader( titleEl ).text().trim()
			};
		} ).get();
		// group by category
		return _.flatten( _.toArray( _.groupBy( examples, 'category' ) ) );
	};

	/**
	 * Set the language for the query samples.
	 *
	 * @param {string} language
	 */
	SELF.prototype.setLanguage = function( language ) {
		this._language = language;
	};

	/**
	 * Get the language for the query samples.
	 *
	 * @return {string} language
	 */
	SELF.prototype.getLanguage = function() {
		return this._language;
	};
	return SELF;

}( jQuery ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.api = wikibase.queryService.api || {};

wikibase.queryService.api.Wikibase = ( function( $ ) {
	'use strict';

	var API_ENDPOINT = 'https://www.wikidata.org/w/api.php';
	var LANGUAGE = 'en';

	var SEARCH_ENTITES = {
		action: 'wbsearchentities',
		format: 'json',
		limit: 50,
		continue: 0,
		language: LANGUAGE,
		uselang: LANGUAGE,
		useCirrus: 1
	},
	QUERY_LANGUGES = {
		action: 'query',
		meta: 'siteinfo',
		format: 'json',
		siprop: 'languages'
	},
	QUERY_LABELS = {
		action: 'wbgetentities',
		props: 'labels',
		format: 'json',
		languages: LANGUAGE,
		languagefallback: '1'
	},
	QUERY_DATATYPE = {
		action: 'wbgetentities',
		props: 'datatype',
		format: 'json'
	};

	/**
	 * API for the Wikibase API
	 *
	 * @class wikibase.queryService.api.Wikibase
	 * @license GNU GPL v2+
	 *
	 * @author Jonas Kress
	 * @constructor
	 * @param {string} endpoint default: 'https://www.wikidata.org/w/api.php'
	 */
	function SELF( endpoint, defaultLanguage ) {
		this._endpoint = API_ENDPOINT;

		if ( endpoint ) {
			this._endpoint = endpoint;
		}

		if ( defaultLanguage ) {
			this._language = defaultLanguage;
		}
	}

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._endpoint = null;

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._language = null;

	/**
	 * Search an entity with using wbsearchentities
	 *
	 * @param {string} term search string
	 * @param {string} type entity type to search for
	 * @param {string} language of search string default:en
	 *
	 * @return {jQuery.Promise}
	 */
	SELF.prototype.searchEntities = function( term, type, language ) {
		var query = SEARCH_ENTITES;
		query.search = term;

		if ( type ) {
			query.type = type;
		}
		if ( this._language || language ) {
			query.language = language || this._language;
			query.uselang = language || this._language;
		} else {
			query.language = LANGUAGE;
			query.uselang = LANGUAGE;
		}

		return this._query( query );
	};

	/**
	 * List of supported languages
	 *
	 * @return {jQuery.Promise}
	 */
	SELF.prototype.getLanguages = function() {
		return this._query( QUERY_LANGUGES );
	};

	/**
	 * Get labels for given entities
	 *
	 * @param {string|string[]} ids entity IDs
	 * @return {jQuery.Promise}
	 */
	SELF.prototype.getLabels = function( ids ) {

		if ( typeof ids === 'string' ) {
			ids = [ ids ];
		}

		var query = QUERY_LABELS;
		query.ids = ids.join( '|' );

		if ( this._language  ) {
			query.languages = this._language;
		}

		return this._query( query );
	};

	/**
	 * Get datatype of property
	 *
	 * @param {string} id property ID
	 * @return {jQuery.Promise}
	 */
	SELF.prototype.getDataType = function( id ) {
		var query = QUERY_DATATYPE,
			deferred = $.Deferred();

		query.ids = id;

		this._query( query ).done( function( data ) {
			if ( data.entities && data.entities[id] && data.entities[id].datatype ) {
				deferred.resolve( data.entities[id].datatype );
			}
			deferred.reject();

		} ).fail( deferred.reject );

		return deferred.promise();
	};

	/**
	 * @private
	 */
	SELF.prototype._query = function( query ) {
		return $.ajax( {
			url: this._endpoint + '?' + jQuery.param( query ),
			dataType: 'jsonp'
		} );
	};

	/**
	 * Set the default language
	 *
	 * @param {string} language of search string default:en
	 */
	SELF.prototype.setLanguage = function( language ) {
		this._language = language;
	};

	return SELF;

}( jQuery ) );

var tracking = tracking || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.api = wikibase.queryService.api || {};

wikibase.queryService.api.Tracking = ( function( $ ) {
	'use strict';

	var API_ENDPOINT = 'https://www.wikidata.org/beacon/statsv';

	/**
	 * API for the Tracking API
	 *
	 * @class wikibase.queryService.api.Tracking
	 * @license GNU GPL v2+
	 *
	 * @author Addshore
	 * @constructor
	 * @param {string} endpoint default: 'https://www.wikidata.org/beacon/statsv'
	 */
	function SELF( endpoint ) {
		this._endpoint = API_ENDPOINT;

		if ( endpoint ) {
			this._endpoint = endpoint;
		}
	}

	/**
	 * @property {string}
	 * @private
	 */
	SELF.prototype._endpoint = null;

	/**
	 * @param {string} metricName
	 * @param {number} value
	 * @param {string} valueType
	 *
	 * @return {jQuery.Promise}
	 */
	SELF.prototype.track = function( metricName, value, valueType ) {
		if ( !value ) {
			value = 1;
		}
		if ( !valueType ) {
			valueType = 'c';
		}

		if ( location.hostname !== 'query.wikidata.org' ) {
			// FIXME: expected to return a promise
			return;// only track on wikidata.org
		}

		// https://www.wikidata.org/beacon/statsv?test.statsv.foo2=5c
		return this._track( metricName + '=' + value + valueType );
	};

	/**
	 * @private
	 */
	SELF.prototype._track = function( query ) {
		return $.ajax( {
			url: this._endpoint + '?' + query,
			dataType: 'jsonp'
		} );
	};

	return SELF;

}( jQuery ) );

var wikibase = wikibase || {};
wikibase.queryService = wikibase.queryService || {};
wikibase.queryService.RdfNamespaces = {};

( function ( $, RdfNamespaces ) {
	'use strict';

	RdfNamespaces.NAMESPACE_SHORTCUTS = {
		Wikidata: {
			wikibase: 'http://wikiba.se/ontology#',
			wd: 'http://www.wikidata.org/entity/',
			wdt: 'http://www.wikidata.org/prop/direct/',
			wds: 'http://www.wikidata.org/entity/statement/',
			p: 'http://www.wikidata.org/prop/',
			wdref: 'http://www.wikidata.org/reference/',
			wdv: 'http://www.wikidata.org/value/',
			ps: 'http://www.wikidata.org/prop/statement/',
			psv: 'http://www.wikidata.org/prop/statement/value/',
			psn: 'http://www.wikidata.org/prop/statement/value-normalized/',
			pq: 'http://www.wikidata.org/prop/qualifier/',
			pqv: 'http://www.wikidata.org/prop/qualifier/value/',
			pqn: 'http://www.wikidata.org/prop/qualifier/value-normalized/',
			pr: 'http://www.wikidata.org/prop/reference/',
			prv: 'http://www.wikidata.org/prop/reference/value/',
			prn: 'http://www.wikidata.org/prop/reference/value-normalized/',
			wdno: 'http://www.wikidata.org/prop/novalue/',
			wdata: 'http://www.wikidata.org/wiki/Special:EntityData/'
		},
		W3C: {
			rdfs: 'http://www.w3.org/2000/01/rdf-schema#',
			rdf: 'http://www.w3.org/1999/02/22-rdf-syntax-ns#',
			owl: 'http://www.w3.org/2002/07/owl#',
			skos: 'http://www.w3.org/2004/02/skos/core#',
			xsd: 'http://www.w3.org/2001/XMLSchema#',
			prov: 'http://www.w3.org/ns/prov#'
		},
		'Social/Other': {
			schema: 'http://schema.org/',
			geo: 'http://www.opengis.net/ont/geosparql#',
			geof: 'http://www.opengis.net/def/geosparql/function/'
		},
		Blazegraph: {
			bd: 'http://www.bigdata.com/rdf#',
			bds: 'http://www.bigdata.com/rdf/search#',
			gas: 'http://www.bigdata.com/rdf/gas#',
			hint: 'http://www.bigdata.com/queryHints#'
		}
	};

	RdfNamespaces.ENTITY_TYPES = {
		'http://www.wikidata.org/prop/direct/': 'property',
		'http://www.wikidata.org/prop/': 'property',
		'http://www.wikidata.org/prop/novalue/': 'property',
		'http://www.wikidata.org/prop/statement/': 'property',
		'http://www.wikidata.org/prop/statement/value/': 'property',
		'http://www.wikidata.org/prop/statement/value-normalized/': 'property',
		'http://www.wikidata.org/prop/qualifier/': 'property',
		'http://www.wikidata.org/prop/qualifier/value/': 'property',
		'http://www.wikidata.org/prop/qualifier/value-normalized/': 'property',
		'http://www.wikidata.org/prop/reference/': 'property',
		'http://www.wikidata.org/prop/reference/value/': 'property',
		'http://www.wikidata.org/prop/reference/value-normalized/': 'property',
		'http://www.wikidata.org/wiki/Special:EntityData/': 'item',
		'http://www.wikidata.org/entity/': 'item'
	};

	RdfNamespaces.ALL_PREFIXES = $.map( RdfNamespaces.NAMESPACE_SHORTCUTS, function ( n ) {
		return n;
	} ).reduce( function ( p, v, i ) {
		return $.extend( p, v );
	}, {} );

	RdfNamespaces.STANDARD_PREFIXES = {
		wd: 'PREFIX wd: <http://www.wikidata.org/entity/>',
		wdt: 'PREFIX wdt: <http://www.wikidata.org/prop/direct/>',
		wikibase: 'PREFIX wikibase: <http://wikiba.se/ontology#>',
		p: 'PREFIX p: <http://www.wikidata.org/prop/>',
		ps: 'PREFIX ps: <http://www.wikidata.org/prop/statement/>',
		pq: 'PREFIX pq: <http://www.wikidata.org/prop/qualifier/>',
		rdfs: 'PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>',
		bd: 'PREFIX bd: <http://www.bigdata.com/rdf#>'
	};

	RdfNamespaces.getPrefixMap = function ( entityTypes ) {
		var prefixes = {};
		$.each( RdfNamespaces.ALL_PREFIXES, function ( prefix, url ) {
			if ( entityTypes[url] ) {
				prefixes[prefix] = entityTypes[url];
			}
		} );
		return prefixes;
	};

} )( jQuery, wikibase.queryService.RdfNamespaces );

( function( $, config, moment ) {
	'use strict';

	var wb = wikibase.queryService;

	function setBrand() {
		$( '.navbar-brand img' ).attr( 'src', config.brand.logo );
		$( '.navbar-brand a > span' ).text( config.brand.title );
	}

	function setLanguage( lang, save ) {
		if ( save ) {
			Cookies.set( 'lang', lang );
		}

		$.i18n.debug = true;
		$.i18n().locale = lang;
		moment.locale( lang );

		$.when(
			config.i18nLoad( lang )
		).done( function() {
			$( '.wikibase-queryservice' ).i18n();
			$( 'html' ).attr( { lang: lang, dir: $.uls.data.getDir( lang ) } );
		} );
	}

	$( document ).ready(
		function() {
			setBrand();

			$( '#query-form' ).attr( 'action', config.api.sparql.uri );
			var lang = Cookies.get( 'lang' ) ? Cookies.get( 'lang' ) : config.language;
			setLanguage( lang, false );

			var api = new wb.api.Wikibase( config.api.wikibase.uri, lang ),
				sparqlApi = new wb.api.Sparql( config.api.sparql.uri, lang ),
				querySamplesApi = new wb.api.QuerySamples( lang ),
				codeSamplesApi = new wb.api.CodeSamples(
					config.api.sparql.uri,
					config.location.root,
					config.location.index
				),
				languageSelector = new wb.ui.i18n.LanguageSelector( $( '.uls-trigger' ), api, lang );

			languageSelector.setChangeListener( function( lang ) {
				api.setLanguage( lang );
				sparqlApi.setLanguage( lang );
				querySamplesApi.setLanguage( lang );
				setLanguage( lang, true );
			} );

			var rdfHint = new wb.ui.editor.hint.Rdf( api ),
					rdfTooltip = new wb.ui.editor.tooltip.Rdf( api ),
					editor = new wb.ui.editor.Editor( rdfHint, null, rdfTooltip );

			new wb.ui.App(
				$( '.wikibase-queryservice ' ),
				editor,
				new wb.ui.queryHelper.QueryHelper( api, sparqlApi ),
				sparqlApi,
				querySamplesApi
			);

			if ( !config.showBirthdayPresents ) {
				$( '[data-target="#CodeExamples"]' ).hide(); // TODO: remove after birthday
			}
			new wikibase.queryService.ui.dialog.CodeExample(
				$( '#CodeExamples' ),
				function () {
					return codeSamplesApi.getExamples( editor.getValue() );
				}
			);

		} );

} )( jQuery, CONFIG, moment );
