/*
 * Decompiled with CFR 0.152.
 */
package org.wikidata.query.rdf.tool.rdf;

import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.Sets;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import javax.xml.datatype.XMLGregorianCalendar;
import org.openrdf.model.Literal;
import org.openrdf.model.Statement;
import org.openrdf.query.Binding;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryEvaluationException;
import org.openrdf.query.TupleQueryResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wikidata.query.rdf.common.uri.Ontology;
import org.wikidata.query.rdf.common.uri.SchemaDotOrg;
import org.wikidata.query.rdf.common.uri.WikibaseUris;
import org.wikidata.query.rdf.tool.Utils;
import org.wikidata.query.rdf.tool.change.Change;
import org.wikidata.query.rdf.tool.exception.FatalException;
import org.wikidata.query.rdf.tool.rdf.ClassifiedStatements;
import org.wikidata.query.rdf.tool.rdf.UpdateBuilder;
import org.wikidata.query.rdf.tool.rdf.client.RdfClient;

public class RdfRepository {
    private static final Logger log = LoggerFactory.getLogger(RdfRepository.class);
    private final long maxStatementsPerBatch;
    private final long maxPostDataSize;
    private final WikibaseUris uris;
    private final String syncBody;
    private final String msyncBody;
    private final String getValues;
    private final String getRefs;
    private final String cleanUnused;
    private final String updateLeftOffTimeBody;
    private final String getRevisions;
    private final String verify;
    private final String getLexemes;
    protected final RdfClient rdfClient;

    public RdfRepository(WikibaseUris uris, RdfClient rdfClient, long maxPostSize) {
        this.uris = uris;
        this.rdfClient = rdfClient;
        this.msyncBody = RdfRepository.loadBody("multiSync");
        this.syncBody = RdfRepository.loadBody("sync");
        this.updateLeftOffTimeBody = RdfRepository.loadBody("updateLeftOffTime");
        this.getValues = RdfRepository.loadBody("GetValues");
        this.getRefs = RdfRepository.loadBody("GetRefs");
        this.cleanUnused = RdfRepository.loadBody("CleanUnused");
        this.getRevisions = RdfRepository.loadBody("GetRevisions");
        this.verify = RdfRepository.loadBody("verify");
        this.getLexemes = RdfRepository.loadBody("GetLexemes");
        this.maxStatementsPerBatch = maxPostSize / 400L;
        this.maxPostDataSize = (maxPostSize - 0x100000L) / 2L;
    }

    private static String loadBody(String name) {
        return Utils.loadBody(name, RdfRepository.class);
    }

    private Set<String> resultToSet(TupleQueryResult result, String binding) {
        HashSet<String> values = new HashSet<String>();
        try {
            while (result.hasNext()) {
                Binding value = ((BindingSet)result.next()).getBinding(binding);
                if (value == null) continue;
                values.add(value.getValue().stringValue());
            }
        }
        catch (QueryEvaluationException e) {
            throw new FatalException("Can't load results: " + e, e);
        }
        return values;
    }

    public ImmutableSetMultimap<String, String> getValues(Collection<String> entityIds) {
        UpdateBuilder b = new UpdateBuilder(this.getValues);
        b.bindUris("entityList", entityIds);
        b.bind("uris.value", this.uris.value());
        b.bind("uris.statement", this.uris.statement());
        b.bindUri("prov:wasDerivedFrom", "http://www.w3.org/ns/prov#wasDerivedFrom");
        return this.rdfClient.selectToMap(b.toString(), "entity", "s");
    }

    public ImmutableSetMultimap<String, String> getRefs(Collection<String> entityIds) {
        UpdateBuilder b = new UpdateBuilder(this.getRefs);
        b.bindUris("entityList", entityIds);
        b.bind("uris.statement", this.uris.statement());
        b.bindUri("prov:wasDerivedFrom", "http://www.w3.org/ns/prov#wasDerivedFrom");
        return this.rdfClient.selectToMap(b.toString(), "entity", "s");
    }

    private String getSyncQuery(String entityId, Collection<Statement> statements, Collection<String> valueList) {
        log.debug("Generating update for {}", (Object)entityId);
        UpdateBuilder b = new UpdateBuilder(this.syncBody);
        b.bindUri("entity:id", this.uris.entity() + entityId);
        b.bindUri("schema:about", "http://schema.org/about");
        b.bindUri("prov:wasDerivedFrom", "http://www.w3.org/ns/prov#wasDerivedFrom");
        b.bind("uris.value", this.uris.value());
        b.bind("uris.statement", this.uris.statement());
        b.bindStatements("insertStatements", statements);
        if (entityId.startsWith("L")) {
            b.bindUris("lexemeIds", this.fetchLexemeSubIds(Collections.singleton(entityId)), this.uris.entity());
        } else {
            b.bind("lexemeIds", "");
        }
        ClassifiedStatements classifiedStatements = new ClassifiedStatements(this.uris);
        classifiedStatements.classify(statements, entityId);
        b.bindValues("entityStatements", classifiedStatements.entityStatements);
        b.bindValues("statementStatements", classifiedStatements.statementStatements);
        b.bindValues("aboutStatements", classifiedStatements.aboutStatements);
        if (valueList != null && !valueList.isEmpty()) {
            UpdateBuilder cleanup = new UpdateBuilder(this.cleanUnused);
            cleanup.bindUris("values", valueList);
            cleanup.bindUri("wikibase:quantityNormalized", "http://wikiba.se/ontology#quantityNormalized");
            b.bind("cleanupQuery", cleanup.toString());
        } else {
            b.bind("cleanupQuery", "");
        }
        return b.toString();
    }

    public int syncFromChanges(Collection<Change> changes, boolean verifyResult) {
        if (changes.isEmpty()) {
            return 0;
        }
        UpdateBuilder b = new UpdateBuilder(this.msyncBody);
        b.bindUri("schema:about", "http://schema.org/about");
        b.bindUri("prov:wasDerivedFrom", "http://www.w3.org/ns/prov#wasDerivedFrom");
        b.bind("uris.value", this.uris.value());
        b.bind("uris.statement", this.uris.statement());
        b.bindValue("ts", Instant.now());
        HashSet<String> entityIds = Sets.newHashSetWithExpectedSize(changes.size());
        ArrayList<Statement> insertStatements = new ArrayList<Statement>();
        ClassifiedStatements classifiedStatements = new ClassifiedStatements(this.uris);
        HashSet<String> valueSet = new HashSet<String>();
        HashSet<String> refSet = new HashSet<String>();
        String queryTemplate = b.toString();
        int modified = 0;
        for (Change change : changes) {
            if (change.getStatements() == null) continue;
            entityIds.add(change.entityId());
            insertStatements.addAll(change.getStatements());
            classifiedStatements.classify(change.getStatements(), change.entityId());
            valueSet.addAll(change.getValueCleanupList());
            refSet.addAll(change.getRefCleanupList());
            if ((long)insertStatements.size() <= this.maxStatementsPerBatch && classifiedStatements.getDataSize() <= this.maxPostDataSize) continue;
            log.info("Too much data with {} bytes - sending batch out, last ID {}", (Object)classifiedStatements.getDataSize(), (Object)change.entityId());
            modified += this.sendUpdateBatch(queryTemplate, entityIds, insertStatements, classifiedStatements, refSet, valueSet, verifyResult);
            entityIds.clear();
            insertStatements.clear();
            classifiedStatements.clear();
            valueSet.clear();
            refSet.clear();
        }
        if (!entityIds.isEmpty()) {
            modified += this.sendUpdateBatch(queryTemplate, entityIds, insertStatements, classifiedStatements, refSet, valueSet, verifyResult);
        }
        return modified;
    }

    private int sendUpdateBatch(String queryTemplate, Set<String> entityIds, List<Statement> insertStatements, ClassifiedStatements classifiedStatements, Set<String> valueSet, Set<String> refSet, boolean verifyResult) {
        UpdateBuilder cleanup;
        log.debug("Processing {} IDs and {} statements", (Object)entityIds.size(), (Object)insertStatements.size());
        UpdateBuilder b = new UpdateBuilder(queryTemplate);
        b.bindUris("entityListTop", entityIds, this.uris.entity());
        entityIds.addAll(this.fetchLexemeSubIds(entityIds));
        b.bindUris("entityList", entityIds, this.uris.entity());
        b.bindStatements("insertStatements", insertStatements);
        b.bindValues("entityStatements", classifiedStatements.entityStatements);
        b.bindValues("statementStatements", classifiedStatements.statementStatements);
        b.bindValues("aboutStatements", classifiedStatements.aboutStatements);
        b.bindValue("ts", Instant.now());
        if (!refSet.isEmpty()) {
            cleanup = new UpdateBuilder(this.cleanUnused);
            cleanup.bindUris("values", refSet);
            cleanup.bindUri("wikibase:quantityNormalized", "http://wikiba.se/ontology#quantityNormalized");
            b.bind("refCleanupQuery", cleanup.toString());
        } else {
            b.bind("refCleanupQuery", "");
        }
        if (!valueSet.isEmpty()) {
            cleanup = new UpdateBuilder(this.cleanUnused);
            cleanup.bindUris("values", valueSet);
            cleanup.bindUri("wikibase:quantityNormalized", "http://wikiba.se/ontology#quantityNormalized");
            b.bind("valueCleanupQuery", cleanup.toString());
        } else {
            b.bind("valueCleanupQuery", "");
        }
        long start = System.currentTimeMillis();
        log.debug("Sending query {} bytes", (Object)b.toString().length());
        Integer modified = this.rdfClient.update(b.toString());
        log.debug("Update query took {} millis and modified {} statements", (Object)(System.currentTimeMillis() - start), (Object)modified);
        if (verifyResult) {
            try {
                this.verifyStatements(entityIds, insertStatements);
            }
            catch (QueryEvaluationException e) {
                throw new FatalException("Can't load verify results: " + e, e);
            }
        }
        return modified;
    }

    private List<String> fetchLexemeSubIds(Set<String> entityIds) {
        UpdateBuilder b = new UpdateBuilder(this.getLexemes);
        b.bindUris("entityList", entityIds, this.uris.entity());
        return this.rdfClient.selectToList(b.toString(), "lex", this.uris.entity());
    }

    @SuppressFBWarnings(value={"SLF4J_SIGN_ONLY_FORMAT"}, justification="We rely on that format.")
    private void verifyStatements(Set<String> entityIds, List<Statement> statements) throws QueryEvaluationException {
        log.debug("Verifying the update");
        UpdateBuilder bv = new UpdateBuilder(this.verify);
        bv.bindUri("schema:about", "http://schema.org/about");
        bv.bind("uris.statement", this.uris.statement());
        bv.bindUris("entityList", entityIds, this.uris.entity());
        bv.bindValues("allStatements", statements);
        TupleQueryResult result = this.rdfClient.query(bv.toString());
        if (result.hasNext()) {
            log.error("Update failed, we have extra data!");
            while (result.hasNext()) {
                BindingSet bindings = (BindingSet)result.next();
                Binding s = bindings.getBinding("s");
                Binding p = bindings.getBinding("p");
                Binding o = bindings.getBinding("o");
                log.error("{}\t{}\t{}", s.getValue().stringValue(), p.getValue().stringValue(), o.getValue().stringValue());
            }
            throw new FatalException("Update failed, bad old data in the store");
        }
        log.debug("Verification OK");
    }

    public int sync(String entityId, Collection<Statement> statements, Collection<String> valueList) {
        long start = System.currentTimeMillis();
        int modified = this.rdfClient.update(this.getSyncQuery(entityId, statements, valueList));
        log.debug("Updating {} took {} millis and modified {} statements", entityId, System.currentTimeMillis() - start, modified);
        return modified;
    }

    public int sync(String entityId, Collection<Statement> statements) {
        return this.sync(entityId, statements, null);
    }

    public Set<String> hasRevisions(Collection<Change> candidates) {
        UpdateBuilder b = new UpdateBuilder(this.getRevisions);
        StringBuilder values = new StringBuilder();
        for (Change entry : candidates) {
            values.append("( <").append(this.uris.entity()).append(entry.entityId()).append("> ").append(entry.revision()).append(" )\n");
        }
        b.bind("values", values.toString());
        b.bindUri("schema:version", "http://schema.org/version");
        return this.resultToSet(this.rdfClient.query(b.toString()), "s");
    }

    @SuppressFBWarnings(value={"VA_FORMAT_STRING_USES_NEWLINE"}, justification="we want to be platform independent here.")
    public boolean hasRevision(String entityId, long revision) {
        return this.rdfClient.ask(String.format(Locale.ROOT, "ASK {\n wd:%s schema:version ?v .\n  FILTER (?v >= %s)\n}", entityId, revision));
    }

    @SuppressFBWarnings(value={"PRMC_POSSIBLY_REDUNDANT_METHOD_CALLS"}, justification="prefix() is called with different StringBuilders")
    public Instant fetchLeftOffTime() {
        log.info("Checking for left off time from the updater");
        StringBuilder b = SchemaDotOrg.prefix(new StringBuilder());
        b.append("SELECT * WHERE { <").append(this.uris.root()).append("> schema:dateModified ?date }");
        Instant leftOffTime = this.dateFromQuery(b.toString());
        if (leftOffTime != null) {
            log.info("Found left off time from the updater");
            return leftOffTime;
        }
        log.info("Checking for left off time from the dump");
        b = Ontology.prefix(SchemaDotOrg.prefix(new StringBuilder()));
        b.append("SELECT * WHERE { ontology:Dump schema:dateModified ?date }");
        return this.dateFromQuery(b.toString());
    }

    public void updateLeftOffTime(Instant leftOffTime) {
        log.debug("Setting last updated time to {}", (Object)leftOffTime);
        UpdateBuilder b = new UpdateBuilder(this.updateLeftOffTimeBody);
        b.bindUri("root", this.uris.root());
        b.bindUri("dateModified", "http://schema.org/dateModified");
        b.bindValue("date", leftOffTime);
        this.rdfClient.update(b.toString());
    }

    private Instant dateFromQuery(String query) {
        TupleQueryResult result = this.rdfClient.query(query);
        try {
            if (!result.hasNext()) {
                return null;
            }
            Binding maxLastUpdate = ((BindingSet)result.next()).getBinding("date");
            if (maxLastUpdate == null) {
                return null;
            }
            XMLGregorianCalendar xmlCalendar = ((Literal)maxLastUpdate.getValue()).calendarValue();
            GregorianCalendar calendar = xmlCalendar.toGregorianCalendar();
            return calendar.getTime().toInstant();
        }
        catch (QueryEvaluationException e) {
            throw new FatalException("Error evaluating query", e);
        }
    }
}

