/*
 * Decompiled with CFR 0.152.
 */
package org.wikidata.query.rdf.tool.rdf;

import com.github.rholder.retry.Attempt;
import com.github.rholder.retry.RetryException;
import com.github.rholder.retry.RetryListener;
import com.github.rholder.retry.Retryer;
import com.github.rholder.retry.RetryerBuilder;
import com.github.rholder.retry.StopStrategies;
import com.github.rholder.retry.WaitStrategies;
import com.google.common.base.Charsets;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.Sets;
import com.google.common.io.Resources;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.HttpProxy;
import org.eclipse.jetty.client.ProxyConfiguration;
import org.eclipse.jetty.client.api.ContentProvider;
import org.eclipse.jetty.client.api.ContentResponse;
import org.eclipse.jetty.client.api.Request;
import org.eclipse.jetty.client.util.FormContentProvider;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.util.Fields;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;
import org.json.simple.parser.ParseException;
import org.openrdf.model.Literal;
import org.openrdf.model.Statement;
import org.openrdf.query.Binding;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryEvaluationException;
import org.openrdf.query.QueryResultHandler;
import org.openrdf.query.QueryResultHandlerException;
import org.openrdf.query.TupleQueryResult;
import org.openrdf.query.impl.TupleQueryResultBuilder;
import org.openrdf.query.resultio.QueryResultParseException;
import org.openrdf.query.resultio.binary.BinaryQueryResultParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wikidata.query.rdf.common.uri.Ontology;
import org.wikidata.query.rdf.common.uri.SchemaDotOrg;
import org.wikidata.query.rdf.common.uri.WikibaseUris;
import org.wikidata.query.rdf.tool.FilteredStatements;
import org.wikidata.query.rdf.tool.change.Change;
import org.wikidata.query.rdf.tool.exception.ContainedException;
import org.wikidata.query.rdf.tool.exception.FatalException;
import org.wikidata.query.rdf.tool.rdf.UpdateBuilder;

public class RdfRepository
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(RdfRepository.class);
    private static final TimeZone UTC = TimeZone.getTimeZone("UTC");
    private final HttpClient httpClient;
    private final URI uri;
    private final WikibaseUris uris;
    private final String syncBody;
    private final String msyncBody;
    private final String getValues;
    private final String getRefs;
    private final String cleanUnused;
    private final String updateLeftOffTimeBody;
    private final String getRevisions;
    private final String verify;
    private int maxRetries = 6;
    private int delay = 1000;
    private static final String HTTP_PROXY = "http.proxyHost";
    private static final String HTTP_PROXY_PORT = "http.proxyPort";
    public static final String TIMEOUT_PROPERTY = RdfRepository.class + ".timeout";
    private final int timeout;
    private final Retryer<ContentResponse> retryer;
    protected static final ResponseHandler<Integer> UPDATE_COUNT_RESPONSE = new UpdateCountResponse();
    protected static final ResponseHandler<TupleQueryResult> TUPLE_QUERY_RESPONSE = new TupleQueryResponse();
    protected static final ResponseHandler<Boolean> ASK_QUERY_RESPONSE = new AskQueryResponse();

    public RdfRepository(URI uri, WikibaseUris uris) {
        this.uri = uri;
        this.uris = uris;
        this.msyncBody = RdfRepository.loadBody("multiSync");
        this.syncBody = RdfRepository.loadBody("sync");
        this.updateLeftOffTimeBody = RdfRepository.loadBody("updateLeftOffTime");
        this.getValues = RdfRepository.loadBody("GetValues");
        this.getRefs = RdfRepository.loadBody("GetRefs");
        this.cleanUnused = RdfRepository.loadBody("CleanUnused");
        this.getRevisions = RdfRepository.loadBody("GetRevisions");
        this.verify = RdfRepository.loadBody("verify");
        this.timeout = Integer.parseInt(System.getProperty(TIMEOUT_PROPERTY, "-1"));
        this.httpClient = new HttpClient(new SslContextFactory(true));
        this.setupHttpClient();
        this.retryer = RetryerBuilder.newBuilder().retryIfExceptionOfType(TimeoutException.class).retryIfExceptionOfType(ExecutionException.class).retryIfExceptionOfType(IOException.class).retryIfRuntimeException().withWaitStrategy(WaitStrategies.exponentialWait((long)this.delay, (long)10L, (TimeUnit)TimeUnit.SECONDS)).withStopStrategy(StopStrategies.stopAfterAttempt((int)this.maxRetries)).withRetryListener(new RetryListener(){

            public <V> void onRetry(Attempt<V> attempt) {
                if (attempt.hasException()) {
                    log.info("HTTP request failed: {}, attempt {}, will {}", new Object[]{attempt.getExceptionCause(), attempt.getAttemptNumber(), attempt.getAttemptNumber() < (long)RdfRepository.this.maxRetries ? "retry" : "fail"});
                }
            }
        }).build();
    }

    private void setupHttpClient() {
        if (System.getProperty(HTTP_PROXY) != null && System.getProperty(HTTP_PROXY_PORT) != null) {
            ProxyConfiguration proxyConfig = this.httpClient.getProxyConfiguration();
            HttpProxy proxy = new HttpProxy(System.getProperty(HTTP_PROXY), Integer.parseInt(System.getProperty(HTTP_PROXY_PORT)));
            proxy.getExcludedAddresses().add("localhost");
            proxy.getExcludedAddresses().add("127.0.0.1");
            proxyConfig.getProxies().add(proxy);
        }
        try {
            this.httpClient.start();
        }
        catch (Exception e) {
            throw new RuntimeException("Unable to start HttpClient", e);
        }
    }

    @Override
    public void close() throws Exception {
        this.httpClient.stop();
    }

    public int getMaxRetries() {
        return this.maxRetries;
    }

    public RdfRepository setMaxRetries(int maxRetries) {
        this.maxRetries = maxRetries;
        return this;
    }

    public int getDelay() {
        return this.delay;
    }

    public RdfRepository setDelay(int delay) {
        this.delay = delay;
        return this;
    }

    private static String loadBody(String name) {
        URL url = Resources.getResource(RdfRepository.class, (String)("RdfRepository." + name + ".sparql"));
        try {
            return Resources.toString((URL)url, (Charset)Charsets.UTF_8);
        }
        catch (IOException e) {
            throw new FatalException("Can't load " + url, e);
        }
    }

    private Set<String> resultToSet(TupleQueryResult result, String binding) {
        HashSet<String> values = new HashSet<String>();
        try {
            while (result.hasNext()) {
                Binding value = ((BindingSet)result.next()).getBinding(binding);
                if (value == null) continue;
                values.add(value.getValue().stringValue());
            }
        }
        catch (QueryEvaluationException e) {
            throw new FatalException("Can't load results: " + (Object)((Object)e), e);
        }
        return values;
    }

    private ImmutableSetMultimap<String, String> resultToMap(TupleQueryResult result, String keyBinding, String valueBinding) {
        ImmutableSetMultimap.Builder values = ImmutableSetMultimap.builder();
        try {
            while (result.hasNext()) {
                BindingSet bindings = (BindingSet)result.next();
                Binding value = bindings.getBinding(valueBinding);
                Binding key = bindings.getBinding(keyBinding);
                if (value == null || key == null) continue;
                values.put((Object)key.getValue().stringValue(), (Object)value.getValue().stringValue());
            }
        }
        catch (QueryEvaluationException e) {
            throw new FatalException("Can't load results: " + (Object)((Object)e), e);
        }
        return values.build();
    }

    public ImmutableSetMultimap<String, String> getValues(Collection<String> entityIds) {
        UpdateBuilder b = new UpdateBuilder(this.getValues);
        b.bindUris("entityList", entityIds);
        b.bind("uris.value", this.uris.value());
        b.bind("uris.statement", this.uris.statement());
        b.bindUri("prov:wasDerivedFrom", "http://www.w3.org/ns/prov#wasDerivedFrom");
        return this.resultToMap(this.query(b.toString()), "entity", "s");
    }

    public ImmutableSetMultimap<String, String> getRefs(Collection<String> entityIds) {
        UpdateBuilder b = new UpdateBuilder(this.getRefs);
        b.bindUris("entityList", entityIds);
        b.bind("uris.statement", this.uris.statement());
        b.bindUri("prov:wasDerivedFrom", "http://www.w3.org/ns/prov#wasDerivedFrom");
        return this.resultToMap(this.query(b.toString()), "entity", "s");
    }

    public String getSyncQuery(String entityId, Collection<Statement> statements, Collection<String> valueList) {
        log.debug("Generating update for {}", (Object)entityId);
        UpdateBuilder b = new UpdateBuilder(this.syncBody);
        b.bindUri("entity:id", this.uris.entity() + entityId);
        b.bindUri("schema:about", "http://schema.org/about");
        b.bindUri("prov:wasDerivedFrom", "http://www.w3.org/ns/prov#wasDerivedFrom");
        b.bind("uris.value", this.uris.value());
        b.bind("uris.statement", this.uris.statement());
        b.bindStatements("insertStatements", statements);
        Collection<Statement> entityStatements = FilteredStatements.filtered(statements).withSubject(this.uris.entity() + entityId);
        b.bindValues("entityStatements", entityStatements);
        Collection<Statement> statementStatements = FilteredStatements.filtered(statements).withSubjectStarts(this.uris.statement());
        b.bindValues("statementStatements", statementStatements);
        HashSet<Statement> aboutStatements = new HashSet<Statement>(statements);
        aboutStatements.removeAll(entityStatements);
        aboutStatements.removeAll(statementStatements);
        aboutStatements.removeAll(FilteredStatements.filtered(statements).withSubjectStarts(this.uris.value()));
        aboutStatements.removeAll(FilteredStatements.filtered(statements).withSubjectStarts(this.uris.reference()));
        b.bindValues("aboutStatements", aboutStatements);
        if (valueList != null && !valueList.isEmpty()) {
            UpdateBuilder cleanup = new UpdateBuilder(this.cleanUnused);
            cleanup.bindUris("values", valueList);
            b.bind("cleanupQuery", cleanup.toString());
        } else {
            b.bind("cleanupQuery", "");
        }
        return b.toString();
    }

    public int syncFromChanges(Collection<Change> changes, boolean verifyResult) {
        if (changes.isEmpty()) {
            return 0;
        }
        UpdateBuilder b = new UpdateBuilder(this.msyncBody);
        b.bindUri("schema:about", "http://schema.org/about");
        b.bindUri("prov:wasDerivedFrom", "http://www.w3.org/ns/prov#wasDerivedFrom");
        b.bind("uris.value", this.uris.value());
        b.bind("uris.statement", this.uris.statement());
        HashSet entityIds = Sets.newHashSetWithExpectedSize((int)changes.size());
        ArrayList<Statement> insertStatements = new ArrayList<Statement>();
        ArrayList<Statement> entityStatements = new ArrayList<Statement>();
        HashSet<String> valueSet = new HashSet<String>();
        for (Change change : changes) {
            if (change.getStatements() == null) continue;
            entityIds.add(change.entityId());
            insertStatements.addAll(change.getStatements());
            entityStatements.addAll(FilteredStatements.filtered(change.getStatements()).withSubject(this.uris.entity() + change.entityId()));
            valueSet.addAll(change.getCleanupList());
        }
        if (entityIds.isEmpty()) {
            log.debug("Got no valid changes, we're done");
            return 0;
        }
        b.bindUris("entityList", entityIds, this.uris.entity());
        b.bindStatements("insertStatements", insertStatements);
        b.bindValues("entityStatements", entityStatements);
        Collection<Statement> statementStatements = FilteredStatements.filtered(insertStatements).withSubjectStarts(this.uris.statement());
        b.bindValues("statementStatements", statementStatements);
        HashSet<Statement> aboutStatements = new HashSet<Statement>(insertStatements);
        aboutStatements.removeAll(entityStatements);
        aboutStatements.removeAll(statementStatements);
        aboutStatements.removeAll(FilteredStatements.filtered(insertStatements).withSubjectStarts(this.uris.value()));
        aboutStatements.removeAll(FilteredStatements.filtered(insertStatements).withSubjectStarts(this.uris.reference()));
        b.bindValues("aboutStatements", aboutStatements);
        if (!valueSet.isEmpty()) {
            UpdateBuilder cleanup = new UpdateBuilder(this.cleanUnused);
            cleanup.bindUris("values", valueSet);
            b.bind("cleanupQuery", cleanup.toString());
        } else {
            b.bind("cleanupQuery", "");
        }
        long start = System.currentTimeMillis();
        int modified = this.execute("update", UPDATE_COUNT_RESPONSE, b.toString());
        log.debug("Update query took {} millis and modified {} statements", (Object)(System.currentTimeMillis() - start), (Object)modified);
        if (verifyResult) {
            try {
                this.verifyStatements(entityIds, insertStatements);
            }
            catch (QueryEvaluationException e) {
                throw new FatalException("Can't load verify results: " + (Object)((Object)e), e);
            }
        }
        return modified;
    }

    private void verifyStatements(Set<String> entityIds, List<Statement> statements) throws QueryEvaluationException {
        log.debug("Verifying the update");
        UpdateBuilder bv = new UpdateBuilder(this.verify);
        bv.bindUri("schema:about", "http://schema.org/about");
        bv.bind("uris.statement", this.uris.statement());
        bv.bindUris("entityList", entityIds, this.uris.entity());
        bv.bindValues("allStatements", statements);
        TupleQueryResult result = this.query(bv.toString());
        if (result.hasNext()) {
            log.error("Update failed, we have extra data!");
            while (result.hasNext()) {
                BindingSet bindings = (BindingSet)result.next();
                Binding s = bindings.getBinding("s");
                Binding p = bindings.getBinding("p");
                Binding o = bindings.getBinding("o");
                log.error("{}\t{}\t{}", new Object[]{s.getValue().stringValue(), p.getValue().stringValue(), o.getValue().stringValue()});
            }
            throw new FatalException("Update failed, bad old data in the store");
        }
        log.debug("Verification OK");
    }

    public int sync(String entityId, Collection<Statement> statements, Collection<String> valueList) {
        long start = System.currentTimeMillis();
        int modified = this.execute("update", UPDATE_COUNT_RESPONSE, this.getSyncQuery(entityId, statements, valueList));
        log.debug("Updating {} took {} millis and modified {} statements", new Object[]{entityId, System.currentTimeMillis() - start, modified});
        return modified;
    }

    public int syncQuery(String query) {
        long start = System.currentTimeMillis();
        int modified = this.execute("update", UPDATE_COUNT_RESPONSE, query);
        log.debug("Update query took {} millis and modified {} statements", (Object)(System.currentTimeMillis() - start), (Object)modified);
        return modified;
    }

    public int sync(String entityId, Collection<Statement> statements) {
        return this.sync(entityId, statements, null);
    }

    public Set<String> hasRevisions(Collection<Change> candidates) {
        UpdateBuilder b = new UpdateBuilder(this.getRevisions);
        StringBuilder values = new StringBuilder();
        for (Change entry : candidates) {
            values.append("( <").append(this.uris.entity()).append(entry.entityId()).append("> ").append(entry.revision()).append(" )\n");
        }
        b.bind("values", values.toString());
        b.bindUri("schema:version", "http://schema.org/version");
        return this.resultToSet(this.query(b.toString()), "s");
    }

    @SuppressFBWarnings(value={"VA_FORMAT_STRING_USES_NEWLINE"}, justification="we want to be platform independent here.")
    public boolean hasRevision(String entityId, long revision) {
        StringBuilder prefixes = new StringBuilder();
        prefixes.append("PREFIX schema: <").append("http://schema.org/").append(">\n");
        prefixes.append("PREFIX entity: <").append(this.uris.entity()).append(">\n");
        return this.ask(String.format(Locale.ROOT, "%sASK {\n  entity:%s schema:version ?v .\n  FILTER (?v >= %s)\n}", prefixes, entityId, revision));
    }

    @SuppressFBWarnings(value={"PRMC_POSSIBLY_REDUNDANT_METHOD_CALLS"}, justification="prefix() is called with different StringBuilders")
    public Date fetchLeftOffTime() {
        log.info("Checking for left off time from the updater");
        StringBuilder b = SchemaDotOrg.prefix((StringBuilder)new StringBuilder());
        b.append("SELECT * WHERE { <").append(this.uris.root()).append("> schema:dateModified ?date }");
        Date leftOffTime = this.dateFromQuery(b.toString());
        if (leftOffTime != null) {
            log.info("Found left off time from the updater");
            return leftOffTime;
        }
        log.info("Checking for left off time from the dump");
        b = Ontology.prefix((StringBuilder)SchemaDotOrg.prefix((StringBuilder)new StringBuilder()));
        b.append("SELECT * WHERE { ontology:Dump schema:dateModified ?date }");
        return this.dateFromQuery(b.toString());
    }

    public void updateLeftOffTime(Date leftOffTime) {
        log.debug("Setting last updated time to {}", (Object)leftOffTime);
        UpdateBuilder b = new UpdateBuilder(this.updateLeftOffTimeBody);
        b.bindUri("root", this.uris.root());
        b.bindUri("dateModified", "http://schema.org/dateModified");
        GregorianCalendar c = new GregorianCalendar(UTC, Locale.ROOT);
        c.setTime(leftOffTime);
        try {
            b.bindValue("date", DatatypeFactory.newInstance().newXMLGregorianCalendar(c));
        }
        catch (DatatypeConfigurationException e) {
            throw new FatalException("Holy cow datatype configuration exception on default datatype factory.  Seems like something really really strange.", e);
        }
        this.execute("update", UPDATE_COUNT_RESPONSE, b.toString());
    }

    public boolean ask(String sparql) {
        return this.execute("query", ASK_QUERY_RESPONSE, sparql);
    }

    public TupleQueryResult query(String sparql) {
        return this.execute("query", TUPLE_QUERY_RESPONSE, sparql);
    }

    private Request makeRequest(String type, String sparql, String accept) {
        Request post = this.httpClient.newRequest(this.uri);
        post.method(HttpMethod.POST);
        if (this.timeout > 0) {
            post.timeout((long)this.timeout, TimeUnit.SECONDS);
        }
        if (accept != null) {
            post.header("Accept", accept);
        }
        Fields fields = new Fields();
        fields.add(type, sparql);
        FormContentProvider form = new FormContentProvider(fields, Charsets.UTF_8);
        post.content((ContentProvider)form);
        return post;
    }

    protected <T> T execute(String type, ResponseHandler<T> responseHandler, String sparql) {
        log.debug("Running SPARQL: {}", (Object)sparql);
        long startQuery = System.currentTimeMillis();
        try {
            ContentResponse response = (ContentResponse)this.retryer.call(() -> this.makeRequest(type, sparql, responseHandler.acceptHeader()).send());
            if (response.getStatus() != 200) {
                throw new ContainedException("Non-200 response from triple store:  " + response + " body=\n" + this.responseBodyAsString(response));
            }
            log.debug("Completed in {} ms", (Object)(System.currentTimeMillis() - startQuery));
            return responseHandler.parse(response);
        }
        catch (RetryException | IOException | ExecutionException e) {
            throw new FatalException("Error updating triple store", e);
        }
    }

    protected String responseBodyAsString(ContentResponse response) throws IOException {
        return response.getContentAsString();
    }

    private Date dateFromQuery(String query) {
        TupleQueryResult result = this.query(query);
        try {
            if (!result.hasNext()) {
                return null;
            }
            Binding maxLastUpdate = ((BindingSet)result.next()).getBinding("date");
            if (maxLastUpdate == null) {
                return null;
            }
            XMLGregorianCalendar xmlCalendar = ((Literal)maxLastUpdate.getValue()).calendarValue();
            GregorianCalendar calendar = xmlCalendar.toGregorianCalendar();
            return calendar.getTime();
        }
        catch (QueryEvaluationException e) {
            throw new FatalException("Error evaluating query", e);
        }
    }

    private static class AskQueryResponse
    implements ResponseHandler<Boolean> {
        private AskQueryResponse() {
        }

        @Override
        public String acceptHeader() {
            return "application/json";
        }

        @Override
        public Boolean parse(ContentResponse entity) throws IOException {
            try {
                JSONObject response = (JSONObject)new JSONParser().parse(entity.getContentAsString());
                return (Boolean)response.get((Object)"boolean");
            }
            catch (ParseException e) {
                throw new IOException("Error parsing response", e);
            }
        }
    }

    private static class TupleQueryResponse
    implements ResponseHandler<TupleQueryResult> {
        private TupleQueryResponse() {
        }

        @Override
        public String acceptHeader() {
            return "application/x-binary-rdf-results-table";
        }

        @Override
        public TupleQueryResult parse(ContentResponse entity) throws IOException {
            BinaryQueryResultParser p = new BinaryQueryResultParser();
            TupleQueryResultBuilder collector = new TupleQueryResultBuilder();
            p.setQueryResultHandler((QueryResultHandler)collector);
            try {
                p.parseQueryResult((InputStream)new ByteArrayInputStream(entity.getContent()));
            }
            catch (IllegalStateException | QueryResultHandlerException | QueryResultParseException e) {
                throw new RuntimeException("Error parsing query", e);
            }
            return collector.getQueryResult();
        }
    }

    protected static class UpdateCountResponse
    implements ResponseHandler<Integer> {
        private static final Pattern ELAPSED_LINE = Pattern.compile("><p>totalElapsed=[^ ]+ elapsed=([^<]+)</p");
        private static final Pattern ELAPSED_LINE_CLAUSES = Pattern.compile("><p>totalElapsed=([^ ]+) elapsed=([^ ]+) whereClause=([^ ]+) deleteClause=([^ ]+) insertClause=([^ <]+)</p");
        private static final Pattern ELAPSED_LINE_FLUSH = Pattern.compile("><p>totalElapsed=([^ ]+) elapsed=([^ ]+) connFlush=([^ ]+) batchResolve=([^ ]+) whereClause=([^ ]+) deleteClause=([^ ]+) insertClause=([^ <]+)</p");
        private static final Pattern COMMIT_LINE = Pattern.compile("><hr><p>COMMIT: totalElapsed=([^ ]+) commitTime=[^ ]+ mutationCount=([^<]+)</p");
        private static final Pattern BULK_UPDATE_LINE = Pattern.compile("<\\?xml version=\"1.0\"\\?><data modified=\"(\\d+)\" milliseconds=\"(\\d+)\"/>");

        protected UpdateCountResponse() {
        }

        @Override
        public String acceptHeader() {
            return null;
        }

        @Override
        @SuppressFBWarnings(value={"PRMC_POSSIBLY_REDUNDANT_METHOD_CALLS"}, justification="more readable with 2 calls")
        public Integer parse(ContentResponse entity) throws IOException {
            Integer mutationCount = null;
            for (String line : entity.getContentAsString().split("\\r?\\n")) {
                Matcher m = ELAPSED_LINE_FLUSH.matcher(line);
                if (m.matches()) {
                    log.debug("total = {} elapsed = {} flush = {} batch = {} where = {} delete = {} insert = {}", new Object[]{m.group(1), m.group(2), m.group(3), m.group(4), m.group(5), m.group(6), m.group(7)});
                    continue;
                }
                m = ELAPSED_LINE_CLAUSES.matcher(line);
                if (m.matches()) {
                    log.debug("total = {} elapsed = {} where = {} delete = {} insert = {}", new Object[]{m.group(1), m.group(2), m.group(3), m.group(4), m.group(5)});
                    continue;
                }
                m = ELAPSED_LINE.matcher(line);
                if (m.matches()) {
                    log.debug("elapsed = {}", (Object)m.group(1));
                    continue;
                }
                m = COMMIT_LINE.matcher(line);
                if (m.matches()) {
                    log.debug("total = {} mutation count = {} ", (Object)m.group(1), (Object)m.group(2));
                    mutationCount = Integer.valueOf(m.group(2));
                    continue;
                }
                m = BULK_UPDATE_LINE.matcher(line);
                if (!m.matches()) continue;
                log.debug("bulk updated {} items in {} millis", (Object)m.group(1), (Object)m.group(2));
                mutationCount = Integer.valueOf(m.group(1));
            }
            if (mutationCount == null) {
                throw new IOException("Couldn't find the mutation count!");
            }
            return mutationCount;
        }
    }

    private static interface ResponseHandler<T> {
        public String acceptHeader();

        public T parse(ContentResponse var1) throws IOException;
    }
}

