/*
 * Decompiled with CFR 0.152.
 */
package org.wikidata.query.rdf.tool.wikibase;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.codahale.metrics.httpclient.InstrumentedHttpClientConnectionManager;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.InterruptedIOException;
import java.io.PushbackInputStream;
import java.io.Reader;
import java.net.SocketException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;
import org.apache.http.Consts;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.ServiceUnavailableRetryStrategy;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.client.IdleConnectionEvictor;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.HttpContext;
import org.openrdf.model.Statement;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.RDFHandler;
import org.openrdf.rio.RDFHandlerException;
import org.openrdf.rio.RDFParseException;
import org.openrdf.rio.RDFParser;
import org.openrdf.rio.Rio;
import org.openrdf.rio.helpers.StatementCollector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wikidata.query.rdf.tool.MapperUtils;
import org.wikidata.query.rdf.tool.change.Change;
import org.wikidata.query.rdf.tool.exception.BadParameterException;
import org.wikidata.query.rdf.tool.exception.ContainedException;
import org.wikidata.query.rdf.tool.exception.FatalException;
import org.wikidata.query.rdf.tool.exception.RetryableException;
import org.wikidata.query.rdf.tool.rdf.NormalizingRdfHandler;
import org.wikidata.query.rdf.tool.utils.NullStreamDumper;
import org.wikidata.query.rdf.tool.utils.StreamDumper;
import org.wikidata.query.rdf.tool.wikibase.Continue;
import org.wikidata.query.rdf.tool.wikibase.CsrfTokenResponse;
import org.wikidata.query.rdf.tool.wikibase.DeleteResponse;
import org.wikidata.query.rdf.tool.wikibase.EditRequest;
import org.wikidata.query.rdf.tool.wikibase.EditResponse;
import org.wikidata.query.rdf.tool.wikibase.RecentChangeResponse;
import org.wikidata.query.rdf.tool.wikibase.SearchResponse;
import org.wikidata.query.rdf.tool.wikibase.WikibaseResponse;

public class WikibaseRepository
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(WikibaseRepository.class);
    private static final String TIMEOUT_MILLIS = "5000";
    private static final String TIMEOUT_PROPERTY = WikibaseRepository.class.getName() + ".timeout";
    private static final int RETRIES = 3;
    private static final int RETRY_INTERVAL = 500;
    public static final String INPUT_DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss'Z'";
    public static final DateTimeFormatter INPUT_DATE_FORMATTER = new DateTimeFormatterBuilder().appendPattern("yyyy-MM-dd'T'HH:mm:ss'Z'").parseDefaulting(ChronoField.NANO_OF_SECOND, 0L).toFormatter().withZone(ZoneId.of("Z"));
    public static final DateTimeFormatter OUTPUT_DATE_FORMATTER = new DateTimeFormatterBuilder().appendPattern("yyyyMMddHHmmss").parseDefaulting(ChronoField.NANO_OF_SECOND, 0L).toFormatter().withZone(ZoneId.of("Z"));
    public static final String CHRONOLOGY_ID_HEADER = "MediaWiki-Chronology-Client-Id";
    private final CloseableHttpClient client;
    private final HttpClientConnectionManager connectionManager;
    private final int requestTimeout = Integer.parseInt(System.getProperty(TIMEOUT_PROPERTY, "5000"));
    private final RequestConfig configWithTimeout = RequestConfig.custom().setSocketTimeout(this.requestTimeout).setConnectTimeout(this.requestTimeout).setConnectionRequestTimeout(this.requestTimeout).build();
    private final Uris uris;
    private boolean collectConstraints;
    private Duration revisionCutoff;
    private final ObjectMapper mapper = MapperUtils.getObjectMapper();
    private final Timer rdfFetchTimer;
    private final Timer entityFetchTimer;
    private final Timer constraintFetchTimer;
    private final StreamDumper streamDumper;

    public WikibaseRepository(URI baseUrl, MetricRegistry metricRegistry) {
        this(new Uris(baseUrl), false, metricRegistry, new NullStreamDumper(), null);
    }

    public WikibaseRepository(String baseUrl, MetricRegistry metricRegistry) {
        this(Uris.fromString(baseUrl), false, metricRegistry, new NullStreamDumper(), null);
    }

    public WikibaseRepository(URI baseUrl, Set<Long> entityNamespaces, MetricRegistry metricRegistry) {
        this(new Uris(baseUrl), false, metricRegistry, new NullStreamDumper(), null);
        this.uris.setEntityNamespaces(entityNamespaces);
    }

    public WikibaseRepository(Uris uris, boolean collectConstraints, MetricRegistry metricRegistry, StreamDumper streamDumper, Duration revisionCutoff) {
        this.uris = uris;
        this.collectConstraints = collectConstraints;
        this.rdfFetchTimer = metricRegistry.timer("rdf-fetch-timer");
        this.entityFetchTimer = metricRegistry.timer("entity-fetch-timer");
        this.constraintFetchTimer = metricRegistry.timer("constraint-fetch-timer");
        this.connectionManager = WikibaseRepository.createConnectionManager(metricRegistry);
        this.client = WikibaseRepository.createHttpClient(this.connectionManager);
        this.streamDumper = streamDumper;
        this.revisionCutoff = revisionCutoff;
    }

    private static ServiceUnavailableRetryStrategy getRetryStrategy(final int max, final int interval) {
        return new ServiceUnavailableRetryStrategy(){

            public boolean retryRequest(HttpResponse response, int executionCount, HttpContext context) {
                return executionCount <= max && (response.getStatusLine().getStatusCode() == 503 || response.getStatusLine().getStatusCode() == 429);
            }

            public long getRetryInterval() {
                return interval;
            }
        };
    }

    private static HttpRequestRetryHandler getRetryHandler(int max) {
        return (exception, executionCount, context) -> {
            log.debug("Exception in attempt {}", (Object)executionCount, (Object)exception);
            if (executionCount >= max) {
                return false;
            }
            if (exception instanceof InterruptedIOException) {
                return true;
            }
            if (exception instanceof UnknownHostException) {
                return false;
            }
            if (exception instanceof SSLException) {
                return false;
            }
            HttpClientContext clientContext = HttpClientContext.adapt((HttpContext)context);
            HttpRequest request = clientContext.getRequest();
            return !(request instanceof HttpEntityEnclosingRequest);
        };
    }

    public RecentChangeResponse fetchRecentChangesByTime(Instant nextStartTime, int batchSize) throws RetryableException {
        return this.fetchRecentChanges(nextStartTime, null, batchSize);
    }

    public RecentChangeResponse fetchRecentChanges(Instant nextStartTime, Continue lastContinue, int batchSize) throws RetryableException {
        URI uri = this.uris.recentChanges(nextStartTime, lastContinue, batchSize);
        log.debug("Polling for changes from {}", (Object)uri);
        HttpGet request = new HttpGet(uri);
        request.setConfig(this.configWithTimeout);
        try {
            return this.checkApi(this.getJson((HttpRequestBase)request, RecentChangeResponse.class));
        }
        catch (SocketException | UnknownHostException e) {
            throw new RuntimeException(e);
        }
        catch (JsonParseException | JsonMappingException e) {
            throw new RuntimeException(e);
        }
        catch (IOException e) {
            throw new RetryableException("Error fetching recent changes", e);
        }
    }

    private InputStream getInputStream(HttpResponse response) throws IOException {
        PushbackInputStream in;
        int firstByte;
        HttpEntity entity = response.getEntity();
        if (entity != null && (firstByte = (in = new PushbackInputStream(entity.getContent())).read()) != -1) {
            in.unread(firstByte);
            return in;
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @SuppressFBWarnings(value={"CC_CYCLOMATIC_COMPLEXITY"})
    private void collectStatementsFromUrl(URI uri, StatementCollector collector, Timer timer, String chronologyId) throws RetryableException {
        RDFParser parser = Rio.createParser((RDFFormat)RDFFormat.TURTLE);
        parser.setRDFHandler((RDFHandler)new NormalizingRdfHandler((RDFHandler)collector));
        HttpGet request = new HttpGet(uri);
        if (chronologyId != null) {
            request.addHeader(CHRONOLOGY_ID_HEADER, chronologyId);
        }
        request.setConfig(this.configWithTimeout);
        log.debug("Fetching rdf from {}", (Object)uri);
        try (Timer.Context timerContext = timer.time();
             CloseableHttpResponse response = this.client.execute((HttpUriRequest)request);){
            if (response.getStatusLine().getStatusCode() == 404) {
                return;
            }
            if (response.getStatusLine().getStatusCode() == 204) {
                log.debug("No content, we're done");
                return;
            }
            if (response.getStatusLine().getStatusCode() == 400) {
                throw new BadParameterException("Status code 400 for " + uri);
            }
            if (response.getStatusLine().getStatusCode() >= 300) {
                throw new ContainedException("Unexpected status code fetching RDF for " + uri + ":  " + response.getStatusLine().getStatusCode());
            }
            try (InputStream in = this.streamDumper.wrap(this.getInputStream((HttpResponse)response));){
                if (in == null) {
                    log.debug("Empty response, we're done");
                    return;
                }
                parser.parse((Reader)new InputStreamReader(in, StandardCharsets.UTF_8), uri.toString());
                return;
            }
        }
        catch (SocketException | UnknownHostException | SSLHandshakeException e) {
            throw new RuntimeException(e);
        }
        catch (IOException e) {
            throw new RetryableException("Error fetching RDF for " + uri, e);
        }
        catch (RDFHandlerException | RDFParseException e) {
            throw new ContainedException("RDF parsing error for " + uri, e);
        }
    }

    private boolean isChangeRecent(Change change) {
        if (this.revisionCutoff == null || this.revisionCutoff.isZero()) {
            return false;
        }
        return change.timestamp() != null && change.timestamp().isAfter(Instant.now().minus(this.revisionCutoff));
    }

    public Collection<Statement> fetchRdfForEntity(Change entityChange) throws RetryableException {
        if (entityChange.revision() > 0L && this.isChangeRecent(entityChange)) {
            return this.fetchRdfForEntity(entityChange.entityId(), entityChange.revision(), entityChange.chronologyId());
        }
        return this.fetchRdfForEntity(entityChange.entityId(), -1L, null);
    }

    @VisibleForTesting
    public Collection<Statement> fetchRdfForEntity(String entityId) throws RetryableException {
        return this.fetchRdfForEntity(entityId, -1L, null);
    }

    public Collection<Statement> fetchRdfForEntity(String entityId, long revision, String chronologyId) throws RetryableException {
        Timer.Context timerContext = this.rdfFetchTimer.time();
        StatementCollector collector = new StatementCollector();
        try {
            this.collectStatementsFromUrl(this.uris.rdf(entityId, revision), collector, this.entityFetchTimer, chronologyId);
        }
        catch (BadParameterException ex) {
            if (revision >= 0L) {
                this.collectStatementsFromUrl(this.uris.rdf(entityId), collector, this.entityFetchTimer, chronologyId);
            }
            throw ex;
        }
        if (this.collectConstraints) {
            try {
                this.collectStatementsFromUrl(this.uris.constraints(entityId), collector, this.constraintFetchTimer, chronologyId);
            }
            catch (ContainedException ex) {
                log.info("Failed to load constraints: {}", (Object)ex.getMessage());
            }
        }
        log.debug("Done in {} ms", (Object)(timerContext.stop() / 1000000L));
        return collector.getStatements();
    }

    public String firstEntityIdForLabelStartingWith(String label, String language, String type) throws RetryableException {
        URI uri = this.uris.searchForLabel(label, language, type);
        log.debug("Searching for entity using {}", (Object)uri);
        try {
            SearchResponse result = this.checkApi(this.getJson((HttpRequestBase)new HttpGet(uri), SearchResponse.class));
            List<SearchResponse.SearchResult> resultList = result.getSearch();
            if (resultList.isEmpty()) {
                return null;
            }
            return resultList.get(0).getId();
        }
        catch (IOException e) {
            throw new RetryableException("Error searching for page", e);
        }
    }

    @VisibleForTesting
    public String setLabel(String entityId, String type, String label, String language) throws RetryableException {
        String datatype = type.equals("property") ? "string" : null;
        EditRequest data = new EditRequest(datatype, (Map<String, EditRequest.Label>)ImmutableMap.of((Object)language, (Object)new EditRequest.Label(language, label)));
        try {
            URI uri = this.uris.edit(entityId, type, this.mapper.writeValueAsString((Object)data));
            log.debug("Editing entity using {}", (Object)uri);
            EditResponse result = this.checkApi(this.getJson((HttpRequestBase)this.postWithToken(uri), EditResponse.class));
            return result.getEntity().getId();
        }
        catch (IOException e) {
            throw new RetryableException("Error adding page", e);
        }
    }

    public void delete(String entityId) throws RetryableException {
        URI uri = this.uris.delete(entityId);
        log.debug("Deleting entity {} using {}", (Object)entityId, (Object)uri);
        try {
            DeleteResponse result = this.checkApi(this.getJson((HttpRequestBase)this.postWithToken(uri), DeleteResponse.class));
            log.debug("Deleted: {}", (Object)result);
        }
        catch (IOException e) {
            throw new RetryableException("Error deleting page", e);
        }
    }

    private HttpPost postWithToken(URI uri) throws IOException {
        HttpPost request = new HttpPost(uri);
        ArrayList<BasicNameValuePair> entity = new ArrayList<BasicNameValuePair>();
        entity.add(new BasicNameValuePair("token", this.csrfToken()));
        request.setEntity((HttpEntity)new UrlEncodedFormEntity(entity, Consts.UTF_8));
        return request;
    }

    private String csrfToken() throws IOException {
        URI uri = this.uris.csrfToken();
        log.debug("Fetching csrf token from {}", (Object)uri);
        return this.getJson((HttpRequestBase)new HttpGet(uri), CsrfTokenResponse.class).getQuery().getTokens().getCsrfToken();
    }

    private <T extends WikibaseResponse> T getJson(HttpRequestBase request, Class<T> valueType) throws IOException {
        try (CloseableHttpResponse response = this.client.execute((HttpUriRequest)request);){
            WikibaseResponse wikibaseResponse = (WikibaseResponse)this.mapper.readValue(response.getEntity().getContent(), valueType);
            return (T)wikibaseResponse;
        }
    }

    private <T extends WikibaseResponse> T checkApi(T response) throws RetryableException {
        Object error = response.getError();
        if (error != null) {
            throw new RetryableException("Error result from Mediawiki:  " + error);
        }
        return response;
    }

    public boolean isEntityNamespace(long namespace) {
        return this.uris.isEntityNamespace(namespace);
    }

    public boolean isValidEntity(String name) {
        return name.matches("^[A-Za-z0-9:]+$");
    }

    @Override
    public void close() throws IOException {
        try {
            this.client.close();
        }
        finally {
            this.connectionManager.shutdown();
        }
    }

    private static CloseableHttpClient createHttpClient(HttpClientConnectionManager connectionManager) {
        return HttpClients.custom().setConnectionManager(connectionManager).setRetryHandler(WikibaseRepository.getRetryHandler(3)).setServiceUnavailableRetryStrategy(WikibaseRepository.getRetryStrategy(3, 500)).disableCookieManagement().setUserAgent(WikibaseRepository.getUserAgent()).build();
    }

    private static String getUserAgent() {
        return System.getProperty("http.userAgent", "Wikidata Query Service Updater");
    }

    private static InstrumentedHttpClientConnectionManager createConnectionManager(MetricRegistry registry) {
        InstrumentedHttpClientConnectionManager connectionManager = new InstrumentedHttpClientConnectionManager(registry);
        connectionManager.setDefaultMaxPerRoute(100);
        connectionManager.setMaxTotal(100);
        IdleConnectionEvictor connectionEvictor = new IdleConnectionEvictor((HttpClientConnectionManager)connectionManager, 1L, TimeUnit.SECONDS);
        connectionEvictor.start();
        return connectionManager;
    }

    @SuppressFBWarnings(value={"STT_STRING_PARSING_A_FIELD"}, justification="low priority to fix")
    public Change getChangeFromContinue(Continue nextContinue) {
        if (nextContinue == null) {
            return null;
        }
        String[] parts = nextContinue.getRcContinue().split("\\|");
        return new Change("DUMMY", -1L, OUTPUT_DATE_FORMATTER.parse((CharSequence)parts[0], Instant::from), Long.parseLong(parts[1]));
    }

    public Uris getUris() {
        return this.uris;
    }

    @VisibleForTesting
    public void setCollectConstraints(boolean collectConstraints) {
        this.collectConstraints = collectConstraints;
    }

    @VisibleForTesting
    public void setRevisionCutoff(Duration cutoff) {
        this.revisionCutoff = cutoff;
    }

    public void batchDone() {
        this.streamDumper.rotate();
    }

    @SuppressFBWarnings(value={"EI_EXPOSE_REP2", "MS_MUTABLE_ARRAY"}, justification="minor enough")
    public static class Uris {
        private static final String ENTITY_DATA_URL = "/wiki/Special:EntityData/";
        private static final String API_URL = "/w/api.php";
        public static final Set<Long> DEFAULT_ENTITY_NAMESPACES = ImmutableSet.of((Object)0L, (Object)120L);
        private Set<Long> entityNamespaces;
        private URI baseUrl;

        public Uris(URI baseUrl) {
            this(baseUrl, DEFAULT_ENTITY_NAMESPACES);
        }

        public Uris(URI baseUrl, Set<Long> entityNamespaces) {
            this.baseUrl = baseUrl;
            this.entityNamespaces = ImmutableSet.copyOf(entityNamespaces);
        }

        public static Uris fromString(String url) {
            try {
                return new Uris(new URI(url));
            }
            catch (URISyntaxException e) {
                throw new FatalException("Bad URL: " + url, e);
            }
        }

        public Uris setEntityNamespaces(Set<Long> entityNamespaces) {
            this.entityNamespaces = ImmutableSet.copyOf(entityNamespaces);
            return this;
        }

        public URI recentChanges(Instant startTime, Continue continueObject, int batchSize) {
            URIBuilder builder = this.apiBuilder();
            builder.addParameter("action", "query");
            builder.addParameter("list", "recentchanges");
            builder.addParameter("rcdir", "newer");
            builder.addParameter("rcprop", "title|ids|timestamp");
            builder.addParameter("rcnamespace", this.getEntityNamespacesString("|"));
            builder.addParameter("rclimit", Integer.toString(batchSize));
            if (continueObject == null) {
                builder.addParameter("continue", "");
                builder.addParameter("rcstart", startTime.toString());
            } else {
                builder.addParameter("continue", continueObject.getContinue());
                builder.addParameter("rccontinue", continueObject.getRcContinue());
            }
            return this.build(builder);
        }

        private URIBuilder entityURIBuilder(String entityId) {
            URIBuilder builder = this.builder();
            builder.setPath(this.baseUrl.getPath() + ENTITY_DATA_URL + entityId + ".ttl");
            builder.addParameter("flavor", "dump");
            return builder;
        }

        public URI rdf(String entityId, long revision) {
            if (revision > 0L) {
                return this.rdfRevision(entityId, revision);
            }
            return this.rdf(entityId);
        }

        public URI rdf(String entityId) {
            URIBuilder builder = this.entityURIBuilder(entityId);
            builder.addParameter("nocache", String.valueOf(Instant.now().toEpochMilli()));
            return this.build(builder);
        }

        public URI rdfRevision(String entityId, long revision) {
            URIBuilder builder = this.entityURIBuilder(entityId);
            builder.addParameter("revision", Long.toString(revision));
            return this.build(builder);
        }

        private URIBuilder constraintsURIBuilder(String entityId) {
            URIBuilder builder = this.builder();
            builder.setPath(this.baseUrl.getPath() + "/wiki/" + entityId);
            builder.addParameter("action", "constraintsrdf");
            return builder;
        }

        public URI constraints(String entityId) {
            URIBuilder builder = this.constraintsURIBuilder(entityId);
            builder.addParameter("nocache", String.valueOf(Instant.now().toEpochMilli()));
            return this.build(builder);
        }

        public URI csrfToken() {
            URIBuilder builder = this.apiBuilder();
            builder.setParameter("action", "query");
            builder.setParameter("meta", "tokens");
            builder.setParameter("continue", "");
            return this.build(builder);
        }

        public URI searchForLabel(String label, String language, String type) {
            URIBuilder builder = this.apiBuilder();
            builder.addParameter("action", "wbsearchentities");
            builder.addParameter("search", label);
            builder.addParameter("language", language);
            builder.addParameter("type", type);
            return this.build(builder);
        }

        public URI edit(String entityId, String newType, String data) {
            URIBuilder builder = this.apiBuilder();
            builder.addParameter("action", "wbeditentity");
            if (entityId != null) {
                builder.addParameter("id", entityId);
            } else {
                builder.addParameter("new", newType);
            }
            builder.addParameter("data", data);
            return this.build(builder);
        }

        public URI delete(String entityId) {
            URIBuilder builder = this.apiBuilder();
            builder.addParameter("action", "delete");
            builder.addParameter("title", entityId);
            return this.build(builder);
        }

        private URIBuilder apiBuilder() {
            URIBuilder builder = this.builder();
            builder.setPath(this.baseUrl.getPath() + API_URL);
            builder.addParameter("format", "json");
            return builder;
        }

        public URIBuilder builder() {
            return new URIBuilder(this.baseUrl);
        }

        private URI build(URIBuilder builder) {
            try {
                return builder.build();
            }
            catch (URISyntaxException e) {
                throw new FatalException("Unable to build url!?", e);
            }
        }

        public String getHost() {
            return this.baseUrl.getHost();
        }

        public boolean isEntityNamespace(long namespace) {
            return this.entityNamespaces.contains(namespace);
        }

        private String getEntityNamespacesString(String delimiter) {
            return this.entityNamespaces.stream().map(Object::toString).collect(Collectors.joining(delimiter));
        }
    }
}

