/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.californium.core.Utils;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Endpoint;
import org.eclipse.californium.core.network.EndpointManager;
import org.eclipse.californium.core.network.EndpointObserver;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.Matcher;
import org.eclipse.californium.core.network.Outbox;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.interceptors.MessageInterceptor;
import org.eclipse.californium.core.network.serialization.DataParser;
import org.eclipse.californium.core.network.serialization.Serializer;
import org.eclipse.californium.core.network.stack.CoapStack;
import org.eclipse.californium.core.server.MessageDeliverer;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.RawDataChannel;
import org.eclipse.californium.elements.UDPConnector;

public class CoapEndpoint
implements Endpoint {
    private static final Logger LOGGER = Logger.getLogger(CoapEndpoint.class.getCanonicalName());
    private final CoapStack coapstack;
    private final Connector connector;
    private final NetworkConfig config;
    private ScheduledExecutorService executor;
    private boolean started;
    private List<EndpointObserver> observers = new ArrayList<EndpointObserver>(0);
    private List<MessageInterceptor> interceptors = new ArrayList<MessageInterceptor>(0);
    private Matcher matcher;
    private Serializer serializer;

    public CoapEndpoint() {
        this(0);
    }

    public CoapEndpoint(int port) {
        this(new InetSocketAddress(port));
    }

    public CoapEndpoint(InetSocketAddress address) {
        this(address, NetworkConfig.getStandard());
    }

    public CoapEndpoint(NetworkConfig config) {
        this(new InetSocketAddress(0), config);
    }

    public CoapEndpoint(int port, NetworkConfig config) {
        this(new InetSocketAddress(port), config);
    }

    public CoapEndpoint(InetSocketAddress address, NetworkConfig config) {
        this(CoapEndpoint.createUDPConnector(address, config), config);
    }

    public CoapEndpoint(Connector connector, NetworkConfig config) {
        this.config = config;
        this.connector = connector;
        this.serializer = new Serializer();
        this.matcher = new Matcher(config);
        this.coapstack = new CoapStack(config, new OutboxImpl());
        this.connector.setRawDataReceiver((RawDataChannel)new InboxImpl());
    }

    private static Connector createUDPConnector(InetSocketAddress address, NetworkConfig config) {
        UDPConnector c = new UDPConnector(address);
        c.setReceiverThreadCount(config.getInt("NETWORK_STAGE_RECEIVER_THREAD_COUNT"));
        c.setSenderThreadCount(config.getInt("NETWORK_STAGE_SENDER_THREAD_COUNT"));
        c.setReceiveBufferSize(config.getInt("UDP_CONNECTOR_RECEIVE_BUFFER"));
        c.setSendBufferSize(config.getInt("UDP_CONNECTOR_SEND_BUFFER"));
        c.setReceiverPacketSize(config.getInt("UDP_CONNECTOR_DATAGRAM_SIZE"));
        return c;
    }

    @Override
    public synchronized void start() throws IOException {
        if (this.started) {
            LOGGER.log(Level.FINE, "Endpoint at " + this.getAddress().toString() + " is already started");
            return;
        }
        if (!this.coapstack.hasDeliverer()) {
            this.coapstack.setDeliverer(new EndpointManager.ClientMessageDeliverer());
        }
        if (this.executor == null) {
            LOGGER.config("Endpoint " + this.toString() + " requires an executor to start. Using default single-threaded daemon executor.");
            final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(new Utils.DaemonThreadFactory());
            this.setExecutor(executor);
            this.addObserver(new EndpointObserver(){

                @Override
                public void started(Endpoint endpoint) {
                }

                @Override
                public void stopped(Endpoint endpoint) {
                }

                @Override
                public void destroyed(Endpoint endpoint) {
                    executor.shutdown();
                }
            });
        }
        try {
            LOGGER.log(Level.INFO, "Starting endpoint at " + this.getAddress());
            this.started = true;
            this.matcher.start();
            this.connector.start();
            for (EndpointObserver obs : this.observers) {
                obs.started(this);
            }
            this.startExecutor();
        }
        catch (IOException e) {
            this.stop();
            throw e;
        }
    }

    private void startExecutor() {
        this.runInProtocolStage(new Runnable(){

            @Override
            public void run() {
            }
        });
    }

    @Override
    public synchronized void stop() {
        if (!this.started) {
            LOGGER.log(Level.INFO, "Endpoint at " + this.getAddress() + " is already stopped");
        } else {
            LOGGER.log(Level.INFO, "Stopping endpoint at address " + this.getAddress());
            this.started = false;
            this.connector.stop();
            this.matcher.stop();
            for (EndpointObserver obs : this.observers) {
                obs.stopped(this);
            }
            this.matcher.clear();
        }
    }

    @Override
    public synchronized void destroy() {
        LOGGER.log(Level.INFO, "Destroying endpoint at address " + this.getAddress());
        if (this.started) {
            this.stop();
        }
        this.connector.destroy();
        this.coapstack.destroy();
        for (EndpointObserver obs : this.observers) {
            obs.destroyed(this);
        }
    }

    @Override
    public void clear() {
        this.matcher.clear();
    }

    @Override
    public boolean isStarted() {
        return this.started;
    }

    @Override
    public synchronized void setExecutor(ScheduledExecutorService executor) {
        this.executor = executor;
        this.coapstack.setExecutor(executor);
        this.matcher.setExecutor(executor);
    }

    @Override
    public void addObserver(EndpointObserver obs) {
        this.observers.add(obs);
    }

    @Override
    public void removeObserver(EndpointObserver obs) {
        this.observers.remove(obs);
    }

    @Override
    public void addInterceptor(MessageInterceptor interceptor) {
        this.interceptors.add(interceptor);
    }

    @Override
    public void removeInterceptor(MessageInterceptor interceptor) {
        this.interceptors.remove(interceptor);
    }

    @Override
    public List<MessageInterceptor> getInterceptors() {
        return new ArrayList<MessageInterceptor>(this.interceptors);
    }

    @Override
    public void sendRequest(final Request request) {
        this.runInProtocolStage(new Runnable(){

            @Override
            public void run() {
                CoapEndpoint.this.coapstack.sendRequest(request);
            }
        });
    }

    @Override
    public void sendResponse(final Exchange exchange, final Response response) {
        if (exchange.hasCustomExecutor()) {
            this.runInProtocolStage(new Runnable(){

                @Override
                public void run() {
                    CoapEndpoint.this.coapstack.sendResponse(exchange, response);
                }
            });
        } else {
            this.coapstack.sendResponse(exchange, response);
        }
    }

    @Override
    public void sendEmptyMessage(Exchange exchange, EmptyMessage message) {
        this.coapstack.sendEmptyMessage(exchange, message);
    }

    @Override
    public void setMessageDeliverer(MessageDeliverer deliverer) {
        this.coapstack.setDeliverer(deliverer);
    }

    @Override
    public InetSocketAddress getAddress() {
        return this.connector.getAddress();
    }

    @Override
    public NetworkConfig getConfig() {
        return this.config;
    }

    private void runInProtocolStage(final Runnable task) {
        this.executor.execute(new Runnable(){

            @Override
            public void run() {
                try {
                    task.run();
                }
                catch (Throwable t) {
                    LOGGER.log(Level.SEVERE, "Exception in protocol stage thread: " + t.getMessage(), t);
                }
            }
        });
    }

    private class InboxImpl
    implements RawDataChannel {
        private InboxImpl() {
        }

        public void receiveData(final RawData raw) {
            if (raw.getAddress() == null) {
                throw new NullPointerException();
            }
            if (raw.getPort() == 0) {
                throw new NullPointerException();
            }
            Runnable task = new Runnable(){

                @Override
                public void run() {
                    InboxImpl.this.receiveMessage(raw);
                }
            };
            CoapEndpoint.this.runInProtocolStage(task);
        }

        private void receiveMessage(RawData raw) {
            DataParser parser = new DataParser(raw.getBytes());
            if (parser.isRequest()) {
                Exchange exchange;
                Request request;
                try {
                    request = parser.parseRequest();
                }
                catch (IllegalStateException e) {
                    StringBuffer log = new StringBuffer("message format error caused by ").append(raw.getInetSocketAddress());
                    if (!parser.isReply()) {
                        EmptyMessage rst = new EmptyMessage(CoAP.Type.RST);
                        rst.setMID(parser.getMID());
                        rst.setToken(new byte[0]);
                        rst.setDestination(raw.getAddress());
                        rst.setDestinationPort(raw.getPort());
                        for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                            interceptor.sendEmptyMessage(rst);
                        }
                        CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serialize(rst));
                        log.append(" and reset");
                    }
                    if (LOGGER.isLoggable(Level.INFO)) {
                        LOGGER.info(log.toString());
                    }
                    return;
                }
                request.setSource(raw.getAddress());
                request.setSourcePort(raw.getPort());
                request.setSenderIdentity(raw.getSenderIdentity());
                for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                    interceptor.receiveRequest(request);
                }
                if (!request.isCanceled() && (exchange = CoapEndpoint.this.matcher.receiveRequest(request)) != null) {
                    exchange.setEndpoint(CoapEndpoint.this);
                    CoapEndpoint.this.coapstack.receiveRequest(exchange, request);
                }
            } else if (parser.isResponse()) {
                Response response = parser.parseResponse();
                response.setSource(raw.getAddress());
                response.setSourcePort(raw.getPort());
                for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                    interceptor.receiveResponse(response);
                }
                if (!response.isCanceled()) {
                    Exchange exchange = CoapEndpoint.this.matcher.receiveResponse(response);
                    if (exchange != null) {
                        exchange.setEndpoint(CoapEndpoint.this);
                        response.setRTT(System.currentTimeMillis() - exchange.getTimestamp());
                        CoapEndpoint.this.coapstack.receiveResponse(exchange, response);
                    } else if (response.getType() != CoAP.Type.ACK) {
                        LOGGER.fine("Rejecting unmatchable response from " + raw.getInetSocketAddress());
                        this.reject(response);
                    }
                }
            } else if (parser.isEmpty()) {
                EmptyMessage message = parser.parseEmptyMessage();
                message.setSource(raw.getAddress());
                message.setSourcePort(raw.getPort());
                for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                    interceptor.receiveEmptyMessage(message);
                }
                if (!message.isCanceled()) {
                    if (message.getType() == CoAP.Type.CON || message.getType() == CoAP.Type.NON) {
                        LOGGER.info("Responding to ping by " + raw.getInetSocketAddress());
                        this.reject(message);
                    } else {
                        Exchange exchange = CoapEndpoint.this.matcher.receiveEmptyMessage(message);
                        if (exchange != null) {
                            exchange.setEndpoint(CoapEndpoint.this);
                            CoapEndpoint.this.coapstack.receiveEmptyMessage(exchange, message);
                        }
                    }
                }
            } else {
                LOGGER.finest("Silently ignoring non-CoAP message from " + raw.getInetSocketAddress());
            }
        }

        private void reject(Message message) {
            EmptyMessage rst = EmptyMessage.newRST(message);
            rst.setToken(new byte[0]);
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.sendEmptyMessage(rst);
            }
            if (!rst.isCanceled()) {
                CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serialize(rst));
            }
        }
    }

    private class OutboxImpl
    implements Outbox {
        private OutboxImpl() {
        }

        @Override
        public void sendRequest(Exchange exchange, Request request) {
            if (request.getDestination() == null) {
                throw new NullPointerException("Request has no destination address");
            }
            if (request.getDestinationPort() == 0) {
                throw new NullPointerException("Request has no destination port");
            }
            CoapEndpoint.this.matcher.sendRequest(exchange, request);
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.sendRequest(request);
            }
            if (!request.isCanceled()) {
                CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serialize(request));
            }
        }

        @Override
        public void sendResponse(Exchange exchange, Response response) {
            if (response.getDestination() == null) {
                throw new NullPointerException("Response has no destination address");
            }
            if (response.getDestinationPort() == 0) {
                throw new NullPointerException("Response has no destination port");
            }
            CoapEndpoint.this.matcher.sendResponse(exchange, response);
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.sendResponse(response);
            }
            if (!response.isCanceled()) {
                CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serialize(response));
            }
        }

        @Override
        public void sendEmptyMessage(Exchange exchange, EmptyMessage message) {
            if (message.getDestination() == null) {
                throw new NullPointerException("Message has no destination address");
            }
            if (message.getDestinationPort() == 0) {
                throw new NullPointerException("Message has no destination port");
            }
            CoapEndpoint.this.matcher.sendEmptyMessage(exchange, message);
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.sendEmptyMessage(message);
            }
            if (!message.isCanceled()) {
                CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serialize(message));
            }
        }
    }
}

