/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network;

import java.util.Iterator;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.ExchangeObserver;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.deduplication.Deduplicator;
import org.eclipse.californium.core.network.deduplication.DeduplicatorFactory;
import org.eclipse.californium.core.observe.ObserveRelation;

public class Matcher {
    private static final Logger LOGGER = Logger.getLogger(Matcher.class.getCanonicalName());
    private boolean started = false;
    private ExchangeObserver exchangeObserver = new ExchangeObserverImpl();
    private ScheduledExecutorService executor;
    private AtomicInteger currendMID;
    private int tokenSizeLimit;
    private ConcurrentHashMap<Exchange.KeyMID, Exchange> exchangesByMID = new ConcurrentHashMap();
    private ConcurrentHashMap<Exchange.KeyToken, Exchange> exchangesByToken = new ConcurrentHashMap();
    private ConcurrentHashMap<Exchange.KeyUri, Exchange> ongoingExchanges = new ConcurrentHashMap();
    private Deduplicator deduplicator;
    private Level healthStatusLevel;
    private int healthStatusInterval;

    public Matcher(NetworkConfig config) {
        DeduplicatorFactory factory = DeduplicatorFactory.getDeduplicatorFactory();
        this.deduplicator = factory.createDeduplicator(config);
        boolean randomMID = config.getBoolean("USE_RANDOM_MID_START");
        this.currendMID = randomMID ? new AtomicInteger(new Random().nextInt(65536)) : new AtomicInteger(0);
        this.tokenSizeLimit = config.getInt("TOKEN_SIZE_LIMIT");
        LOGGER.config("Matcher uses USE_RANDOM_MID_START=" + randomMID + " and TOKEN_SIZE_LIMIT=" + this.tokenSizeLimit);
        this.healthStatusLevel = Level.parse(config.getString("HEALTH_STATUS_PRINT_LEVEL"));
        this.healthStatusInterval = config.getInt("HEALTH_STATUS_INTERVAL");
    }

    public synchronized void start() {
        if (this.started) {
            return;
        }
        this.started = true;
        if (this.executor == null) {
            throw new IllegalStateException("Matcher has no executor to schedule exchange removal");
        }
        this.deduplicator.start();
        if (LOGGER.isLoggable(this.healthStatusLevel)) {
            this.executor.scheduleAtFixedRate(new Runnable(){

                @Override
                public void run() {
                    LOGGER.log(Matcher.this.healthStatusLevel, "Matcher state: " + Matcher.this.exchangesByMID.size() + " exchangesByMID, " + Matcher.this.exchangesByToken.size() + " exchangesByToken, " + Matcher.this.ongoingExchanges.size() + " ongoingExchanges");
                }
            }, this.healthStatusInterval, this.healthStatusInterval, TimeUnit.SECONDS);
        }
    }

    public synchronized void stop() {
        if (!this.started) {
            return;
        }
        this.started = false;
        this.deduplicator.stop();
        this.clear();
    }

    public synchronized void setExecutor(ScheduledExecutorService executor) {
        this.deduplicator.setExecutor(executor);
        this.executor = executor;
    }

    public void sendRequest(Exchange exchange, Request request) {
        Exchange.KeyToken idByToken;
        if (request.getMID() == -1) {
            request.setMID(this.currendMID.getAndIncrement() % 65536);
        }
        Exchange.KeyMID idByMID = new Exchange.KeyMID(request.getMID(), null, 0);
        if (request.getToken() == null) {
            byte[] token;
            while (this.exchangesByToken.get(idByToken = new Exchange.KeyToken(token = this.createNewToken())) != null) {
            }
            request.setToken(token);
        } else {
            idByToken = new Exchange.KeyToken(request.getToken());
            if (!(exchange.getFailedTransmissionCount() > 0 || request.getOptions().hasBlock1() || request.getOptions().hasBlock2() || request.getOptions().hasObserve() || this.exchangesByToken.get(idByToken) == null)) {
                LOGGER.warning("Manual token overrides existing open request: " + idByToken);
            }
        }
        exchange.setObserver(this.exchangeObserver);
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine("Stored open request by " + idByMID + ", " + idByToken);
        }
        this.exchangesByMID.put(idByMID, exchange);
        this.exchangesByToken.put(idByToken, exchange);
    }

    public void sendResponse(Exchange exchange, Response response) {
        ObserveRelation relation;
        if (response.getMID() == -1) {
            response.setMID(this.currendMID.getAndIncrement() % 65536);
        }
        response.setToken(exchange.getCurrentRequest().getToken());
        if ((response.getType() == CoAP.Type.CON || response.getType() == CoAP.Type.ACK) && (relation = exchange.getRelation()) != null) {
            this.removeNotificatoinsOf(relation);
        }
        if (response.getOptions().hasBlock2()) {
            Request request = exchange.getCurrentRequest();
            Exchange.KeyUri idByUri = new Exchange.KeyUri(request.getURI(), response.getDestination().getAddress(), response.getDestinationPort());
            if (exchange.getResponseBlockStatus() != null && !response.getOptions().hasObserve()) {
                if (this.ongoingExchanges.put(idByUri, exchange) == null) {
                    LOGGER.fine("Ongoing Block2 started late, storing " + idByUri + " for " + request);
                } else {
                    LOGGER.fine("Ongoing Block2 continued, storing " + idByUri + " for " + request);
                }
            } else {
                LOGGER.fine("Ongoing Block2 completed, cleaning up " + idByUri + " for " + request);
                this.ongoingExchanges.remove(idByUri);
            }
        }
        if (response.getType() == CoAP.Type.CON || response.getType() == CoAP.Type.NON) {
            Exchange.KeyMID idByMID = new Exchange.KeyMID(response.getMID(), null, 0);
            this.exchangesByMID.put(idByMID, exchange);
        }
        if (response.getType() != CoAP.Type.CON && response.isLast()) {
            exchange.setComplete();
        }
    }

    public void sendEmptyMessage(Exchange exchange, EmptyMessage message) {
        message.setToken(new byte[0]);
        if (message.getType() == CoAP.Type.RST && exchange != null) {
            exchange.setComplete();
        }
    }

    public Exchange receiveRequest(Request request) {
        Exchange ongoing;
        Exchange.KeyMID idByMID = new Exchange.KeyMID(request.getMID(), request.getSource().getAddress(), request.getSourcePort());
        if (!request.getOptions().hasBlock1() && !request.getOptions().hasBlock2()) {
            Exchange exchange = new Exchange(request, Exchange.Origin.REMOTE);
            Exchange previous = this.deduplicator.findPrevious(idByMID, exchange);
            if (previous == null) {
                exchange.setObserver(this.exchangeObserver);
                return exchange;
            }
            LOGGER.info("Duplicate request: " + request);
            request.setDuplicate(true);
            return previous;
        }
        Exchange.KeyUri idByUri = new Exchange.KeyUri(request.getURI(), request.getSource().getAddress(), request.getSourcePort());
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine("Looking up ongoing exchange for " + idByUri);
        }
        if ((ongoing = this.ongoingExchanges.get(idByUri)) != null) {
            Exchange prev = this.deduplicator.findPrevious(idByMID, ongoing);
            if (prev != null) {
                LOGGER.info("Duplicate ongoing request: " + request);
                request.setDuplicate(true);
            } else if (ongoing.getCurrentResponse() != null && ongoing.getCurrentResponse().getType() != CoAP.Type.ACK && !ongoing.getCurrentResponse().getOptions().hasObserve()) {
                idByMID = new Exchange.KeyMID(ongoing.getCurrentResponse().getMID(), null, 0);
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Ongoing exchange got new request, cleaning up " + idByMID);
                }
                this.exchangesByMID.remove(idByMID);
            }
            return ongoing;
        }
        Exchange exchange = new Exchange(request, Exchange.Origin.REMOTE);
        Exchange previous = this.deduplicator.findPrevious(idByMID, exchange);
        if (previous == null) {
            LOGGER.fine("New ongoing request, storing " + idByUri + " for " + request);
            exchange.setObserver(this.exchangeObserver);
            this.ongoingExchanges.put(idByUri, exchange);
            return exchange;
        }
        LOGGER.info("Duplicate initial request: " + request);
        request.setDuplicate(true);
        return previous;
    }

    public Exchange receiveResponse(Response response) {
        Exchange.KeyMID idByMID = response.getType() == CoAP.Type.ACK ? new Exchange.KeyMID(response.getMID(), null, 0) : new Exchange.KeyMID(response.getMID(), response.getSource().getAddress(), response.getSourcePort());
        Exchange.KeyToken idByToken = new Exchange.KeyToken(response.getToken());
        Exchange exchange = this.exchangesByToken.get(idByToken);
        if (exchange != null) {
            Exchange prev = this.deduplicator.findPrevious(idByMID, exchange);
            if (prev != null) {
                LOGGER.info("Duplicate response for open exchange: " + response);
                response.setDuplicate(true);
            } else {
                idByMID = new Exchange.KeyMID(exchange.getCurrentRequest().getMID(), null, 0);
                this.exchangesByMID.remove(idByMID);
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Closed open request with " + idByMID);
                }
            }
            if (response.getType() == CoAP.Type.ACK && exchange.getCurrentRequest().getMID() != response.getMID()) {
                LOGGER.warning("Possible MID reuse before lifetime end: " + response.getTokenString() + " expected MID " + exchange.getCurrentRequest().getMID() + " but received " + response.getMID());
            }
            return exchange;
        }
        if (response.getType() != CoAP.Type.ACK) {
            Exchange prev = this.deduplicator.find(idByMID);
            if (prev != null) {
                LOGGER.info("Duplicate response for completed exchange: " + response);
                response.setDuplicate(true);
                return prev;
            }
        } else {
            LOGGER.info("Ignoring unmatchable piggy-backed response from " + response.getSource() + ":" + response.getSourcePort() + ": " + response);
        }
        return null;
    }

    public Exchange receiveEmptyMessage(EmptyMessage message) {
        Exchange.KeyMID idByMID = new Exchange.KeyMID(message.getMID(), null, 0);
        Exchange exchange = this.exchangesByMID.get(idByMID);
        if (exchange != null) {
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("Exchange got reply: Cleaning up " + idByMID);
            }
            this.exchangesByMID.remove(idByMID);
            return exchange;
        }
        LOGGER.info("Ignoring unmatchable empty message from " + message.getSource() + ":" + message.getSourcePort() + ": " + message);
        return null;
    }

    public void clear() {
        this.exchangesByMID.clear();
        this.exchangesByToken.clear();
        this.ongoingExchanges.clear();
        this.deduplicator.clear();
    }

    private void removeNotificatoinsOf(ObserveRelation relation) {
        LOGGER.fine("Remove all remaining NON-notifications of observe relation");
        Iterator<Response> iterator = relation.getNotificationIterator();
        while (iterator.hasNext()) {
            Response previous = iterator.next();
            Exchange.KeyMID idByMID = new Exchange.KeyMID(previous.getMID(), null, 0);
            this.exchangesByMID.remove(idByMID);
            iterator.remove();
        }
    }

    private byte[] createNewToken() {
        Random random = new Random();
        byte[] token = new byte[random.nextInt(this.tokenSizeLimit) + 1];
        random.nextBytes(token);
        return token;
    }

    private class ExchangeObserverImpl
    implements ExchangeObserver {
        private ExchangeObserverImpl() {
        }

        @Override
        public void completed(Exchange exchange) {
            if (exchange.getOrigin() == Exchange.Origin.LOCAL) {
                Exchange.KeyMID idByMID = new Exchange.KeyMID(exchange.getCurrentRequest().getMID(), null, 0);
                Exchange.KeyToken idByToken = new Exchange.KeyToken(exchange.getCurrentRequest().getToken());
                Matcher.this.exchangesByToken.remove(idByToken);
                Matcher.this.exchangesByMID.remove(idByMID);
            } else {
                ObserveRelation relation;
                Request request;
                Response response = exchange.getCurrentResponse();
                if (response != null && response.getType() != CoAP.Type.ACK) {
                    Exchange.KeyMID midKey = new Exchange.KeyMID(response.getMID(), null, 0);
                    Matcher.this.exchangesByMID.remove(midKey);
                }
                if ((request = exchange.getCurrentRequest()) != null && (request.getOptions().hasBlock1() || response.getOptions().hasBlock2())) {
                    Exchange.KeyUri uriKey = new Exchange.KeyUri(request.getURI(), request.getSource().getAddress(), request.getSourcePort());
                    LOGGER.fine("Remote ongoing completed, cleaning up " + uriKey);
                    Matcher.this.ongoingExchanges.remove(uriKey);
                }
                if ((relation = exchange.getRelation()) != null) {
                    Matcher.this.removeNotificatoinsOf(relation);
                }
            }
        }
    }
}

