/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network.stack;

import java.util.Arrays;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.californium.core.coap.BlockOption;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.MessageObserverAdapter;
import org.eclipse.californium.core.coap.OptionSet;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.config.NetworkConfigObserverAdapter;
import org.eclipse.californium.core.network.stack.AbstractLayer;
import org.eclipse.californium.core.network.stack.BlockwiseStatus;

public class BlockwiseLayer
extends AbstractLayer {
    protected static final Logger LOGGER = Logger.getLogger(BlockwiseLayer.class.getCanonicalName());
    private int max_message_size;
    private int preferred_block_size;
    private int block_timeout;
    private final NetworkConfigObserverAdapter observer;
    private final NetworkConfig config;

    public BlockwiseLayer(NetworkConfig config) {
        this.config = config;
        this.max_message_size = config.getInt("MAX_MESSAGE_SIZE");
        this.preferred_block_size = config.getInt("PREFERRED_BLOCK_SIZE");
        this.block_timeout = config.getInt("BLOCKWISE_STATUS_LIFETIME");
        LOGGER.log(Level.CONFIG, "BlockwiseLayer uses MAX_MESSAGE_SIZE={0}, DEFAULT_BLOCK_SIZE={1} and BLOCKWISE_STATUS_LIFETIME={2}", new Object[]{this.max_message_size, this.preferred_block_size, this.block_timeout});
        this.observer = new NetworkConfigObserverAdapter(){

            @Override
            public void changed(String key, int value) {
                if ("MAX_MESSAGE_SIZE".equals(key)) {
                    BlockwiseLayer.this.max_message_size = value;
                } else if ("PREFERRED_BLOCK_SIZE".equals(key)) {
                    BlockwiseLayer.this.preferred_block_size = value;
                } else if ("BLOCKWISE_STATUS_LIFETIME".equals(key)) {
                    BlockwiseLayer.this.block_timeout = value;
                }
            }
        };
        this.config.addConfigObserver(this.observer);
    }

    @Override
    public void sendRequest(Exchange exchange, Request request) {
        if (request.getOptions().hasBlock2() && request.getOptions().getBlock2().getNum() > 0) {
            LOGGER.fine("Request carries explicit defined block2 option: create random access blockwise status");
            BlockwiseStatus status = new BlockwiseStatus(request.getOptions().getContentFormat());
            BlockOption block2 = request.getOptions().getBlock2();
            status.setCurrentSzx(block2.getSzx());
            status.setCurrentNum(block2.getNum());
            status.setRandomAccess(true);
            exchange.setResponseBlockStatus(status);
            super.sendRequest(exchange, request);
        } else if (this.requiresBlockwise(request)) {
            LOGGER.fine("Request payload " + request.getPayloadSize() + "/" + this.max_message_size + " requires Blockwise");
            BlockwiseStatus status = this.findRequestBlockStatus(exchange, request);
            Request block = BlockwiseLayer.getNextRequestBlock(request, status);
            exchange.setRequestBlockStatus(status);
            exchange.setCurrentRequest(block);
            super.sendRequest(exchange, block);
        } else {
            exchange.setCurrentRequest(request);
            super.sendRequest(exchange, request);
        }
    }

    @Override
    public void receiveRequest(Exchange exchange, Request request) {
        if (request.getOptions().hasBlock1()) {
            BlockOption block1 = request.getOptions().getBlock1();
            LOGGER.log(Level.FINE, "Request contains block1 option {0}", block1);
            BlockwiseStatus status = this.findRequestBlockStatus(exchange, request);
            if (block1.getNum() == 0 && status.getCurrentNum() > 0) {
                LOGGER.finer("Block1 num is 0, the client has restarted the blockwise transfer. Reset status.");
                exchange.setRequestBlockStatus(null);
                status = this.findRequestBlockStatus(exchange, request);
            }
            if (block1.getNum() == status.getCurrentNum()) {
                if (request.getOptions().getContentFormat() != status.getContentFormat()) {
                    Response error = Response.createResponse(request, CoAP.ResponseCode.REQUEST_ENTITY_INCOMPLETE);
                    error.getOptions().setBlock1(block1.getSzx(), block1.isM(), block1.getNum());
                    error.setPayload("Changed Content-Format");
                    exchange.setCurrentResponse(error);
                    super.sendResponse(exchange, error);
                    return;
                }
                status.addBlock(request.getPayload());
                status.setCurrentNum(status.getCurrentNum() + 1);
                if (block1.isM()) {
                    LOGGER.finest("There are more blocks to come. Acknowledge this block.");
                    Response piggybacked = Response.createResponse(request, CoAP.ResponseCode.CONTINUE);
                    piggybacked.getOptions().setBlock1(block1.getSzx(), true, block1.getNum());
                    piggybacked.setLast(false);
                    exchange.setCurrentResponse(piggybacked);
                    super.sendResponse(exchange, piggybacked);
                } else {
                    LOGGER.finer("This was the last block. Deliver request");
                    exchange.setBlock1ToAck(block1);
                    this.earlyBlock2Negotiation(exchange, request);
                    Request assembled = new Request(request.getCode());
                    assembled.setSenderIdentity(request.getSenderIdentity());
                    BlockwiseLayer.assembleMessage(status, assembled);
                    exchange.setRequest(assembled);
                    super.receiveRequest(exchange, assembled);
                }
            } else {
                LOGGER.warning("Wrong block number. Expected " + status.getCurrentNum() + " but received " + block1.getNum() + ". Respond with 4.08 (Request Entity Incomplete)");
                Response error = Response.createResponse(request, CoAP.ResponseCode.REQUEST_ENTITY_INCOMPLETE);
                error.getOptions().setBlock1(block1.getSzx(), block1.isM(), block1.getNum());
                error.setPayload("Wrong block number");
                exchange.setCurrentResponse(error);
                super.sendResponse(exchange, error);
            }
        } else if (exchange.getResponse() != null && request.getOptions().hasBlock2()) {
            BlockOption block2 = request.getOptions().getBlock2();
            Response response = exchange.getResponse();
            BlockwiseStatus status = this.findResponseBlockStatus(exchange, response);
            status.setCurrentNum(block2.getNum());
            status.setCurrentSzx(block2.getSzx());
            Response block = BlockwiseLayer.getNextResponseBlock(response, status);
            if (status.isComplete()) {
                LOGGER.fine("Ongoing is complete " + status);
                exchange.setResponseBlockStatus(null);
                exchange.setBlockCleanupHandle(null);
            } else {
                LOGGER.fine("Ongoing is continuing " + status);
            }
            exchange.setCurrentResponse(block);
            super.sendResponse(exchange, block);
        } else {
            this.earlyBlock2Negotiation(exchange, request);
            exchange.setRequest(request);
            super.receiveRequest(exchange, request);
        }
    }

    @Override
    public void sendResponse(Exchange exchange, Response response) {
        BlockOption block1 = exchange.getBlock1ToAck();
        if (block1 != null) {
            exchange.setBlock1ToAck(null);
        }
        if (this.requireBlockwise(exchange, response)) {
            LOGGER.log(Level.FINE, "Response payload {0}/{1} requires Blockwise", new Object[]{response.getPayloadSize(), this.max_message_size});
            BlockwiseStatus status = this.findResponseBlockStatus(exchange, response);
            Response block = BlockwiseLayer.getNextResponseBlock(response, status);
            if (block1 != null) {
                block.getOptions().setBlock1(block1);
            }
            if (status.isComplete()) {
                LOGGER.fine("Ongoing finished on first block " + status);
                exchange.setResponseBlockStatus(null);
                exchange.setBlockCleanupHandle(null);
            } else {
                LOGGER.fine("Ongoing started " + status);
            }
            exchange.setCurrentResponse(block);
            super.sendResponse(exchange, block);
        } else {
            if (block1 != null) {
                response.getOptions().setBlock1(block1);
            }
            exchange.setCurrentResponse(response);
            exchange.setBlockCleanupHandle(null);
            super.sendResponse(exchange, response);
        }
    }

    @Override
    public void receiveResponse(Exchange exchange, Response response) {
        BlockwiseStatus status;
        if (exchange.getRequest().isCanceled()) {
            if (response.getType() != CoAP.Type.ACK) {
                LOGGER.finer("Rejecting blockwise transfer for canceled Exchange");
                EmptyMessage rst = EmptyMessage.newRST(response);
                this.sendEmptyMessage(exchange, rst);
            }
            return;
        }
        if (!response.getOptions().hasBlock1() && !response.getOptions().hasBlock2()) {
            exchange.setResponse(response);
            super.receiveResponse(exchange, response);
            return;
        }
        if (response.getOptions().hasBlock1()) {
            BlockOption block1 = response.getOptions().getBlock1();
            LOGGER.log(Level.FINER, "Response acknowledges block {0}", block1);
            status = exchange.getRequestBlockStatus();
            if (!status.isComplete()) {
                int currentSize = 1 << 4 + status.getCurrentSzx();
                int nextNum = status.getCurrentNum() + currentSize / block1.getSize();
                LOGGER.finer("Sending next Block1 num=" + nextNum);
                status.setCurrentNum(nextNum);
                status.setCurrentSzx(block1.getSzx());
                Request nextBlock = BlockwiseLayer.getNextRequestBlock(exchange.getRequest(), status);
                nextBlock.setToken(response.getToken());
                exchange.setCurrentRequest(nextBlock);
                super.sendRequest(exchange, nextBlock);
            } else if (!response.getOptions().hasBlock2()) {
                super.receiveResponse(exchange, response);
            } else {
                LOGGER.finer("Block1 followed by Block2 transfer");
            }
        }
        if (response.getOptions().hasBlock2()) {
            BlockOption block2 = response.getOptions().getBlock2();
            status = this.findResponseBlockStatus(exchange, response);
            if (response.getOptions().hasObserve() && block2.getNum() == 0 && status.getCurrentNum() != 0) {
                if (response.getOptions().getObserve() > status.getObserve()) {
                    LOGGER.warning("Ongoing blockwise transfer reseted at num=" + status.getCurrentNum() + " by new notification: " + response);
                    exchange.setResponseBlockStatus(null);
                    status = this.findResponseBlockStatus(exchange, response);
                } else {
                    LOGGER.info("Ignoring old notification during ongoing blockwise transfer: " + response);
                    return;
                }
            }
            if (block2.getNum() == status.getCurrentNum() && (block2.getNum() == 0 || Arrays.equals(response.getToken(), exchange.getCurrentRequest().getToken()))) {
                status.addBlock(response.getPayload());
                if (response.getOptions().hasObserve()) {
                    status.setObserve(response.getOptions().getObserve());
                }
                if (status.isRandomAccess()) {
                    exchange.setResponse(response);
                    super.receiveResponse(exchange, response);
                } else if (block2.isM()) {
                    Request request = exchange.getRequest();
                    int num = block2.getNum() + 1;
                    int szx = block2.getSzx();
                    boolean m = false;
                    LOGGER.log(Level.FINER, "Requesting next Block2 num={0}", num);
                    Request block = new Request(request.getCode());
                    block.setType(request.getType());
                    block.setDestination(request.getDestination());
                    block.setDestinationPort(request.getDestinationPort());
                    if (!response.getOptions().hasObserve()) {
                        block.setToken(response.getToken());
                    }
                    block.setOptions(new OptionSet(request.getOptions()));
                    block.getOptions().removeObserve();
                    block.getOptions().setBlock2(szx, m, num);
                    block.addMessageObservers(request.getMessageObservers());
                    status.setCurrentNum(num);
                    exchange.setCurrentRequest(block);
                    super.sendRequest(exchange, block);
                } else {
                    LOGGER.finer("We have received all " + status.getBlockCount() + " blocks of the response. Assemble and deliver");
                    Response assembled = new Response(response.getCode());
                    BlockwiseLayer.assembleMessage(status, assembled);
                    assembled.setRTT(System.currentTimeMillis() - exchange.getTimestamp());
                    int observe = status.getObserve();
                    if (observe != -1) {
                        if (!response.getOptions().hasObserve()) {
                            exchange.completeCurrentRequest();
                        }
                        assembled.getOptions().setObserve(observe);
                        exchange.setResponseBlockStatus(null);
                    }
                    LOGGER.log(Level.FINE, "Assembled response: {0}", assembled);
                    exchange.setResponse(assembled);
                    super.receiveResponse(exchange, assembled);
                }
            } else {
                LOGGER.warning("Wrong block number. Expected " + status.getCurrentNum() + " but received " + block2.getNum() + ": " + response);
                if (response.getType() == CoAP.Type.CON) {
                    EmptyMessage rst = EmptyMessage.newRST(response);
                    super.sendEmptyMessage(exchange, rst);
                }
            }
        }
    }

    private void earlyBlock2Negotiation(Exchange exchange, Request request) {
        if (request.getOptions().hasBlock2()) {
            BlockOption block2 = request.getOptions().getBlock2();
            BlockwiseStatus status2 = new BlockwiseStatus(request.getOptions().getContentFormat(), block2.getNum(), block2.getSzx());
            LOGGER.fine("Request with early block negotiation " + block2 + ". Create and set new Block2 status: " + status2);
            exchange.setResponseBlockStatus(status2);
        }
    }

    private BlockwiseStatus findRequestBlockStatus(Exchange exchange, Request request) {
        BlockwiseStatus status = exchange.getRequestBlockStatus();
        if (status == null) {
            status = new BlockwiseStatus(request.getOptions().getContentFormat());
            status.setFirst(request);
            status.setCurrentSzx(BlockwiseLayer.computeSZX(this.preferred_block_size));
            exchange.setRequestBlockStatus(status);
            LOGGER.finer("There is no assembler status yet. Create and set new Block1 status: " + status);
        } else {
            LOGGER.finer("Current Block1 status: " + status);
        }
        this.prepareBlockCleanup(exchange);
        return status;
    }

    private BlockwiseStatus findResponseBlockStatus(Exchange exchange, Response response) {
        BlockwiseStatus status = exchange.getResponseBlockStatus();
        if (status == null) {
            status = new BlockwiseStatus(response.getOptions().getContentFormat());
            status.setCurrentSzx(BlockwiseLayer.computeSZX(this.preferred_block_size));
            status.setFirst(response);
            exchange.setResponseBlockStatus(status);
            LOGGER.finer("There is no blockwise status yet. Create and set new Block2 status: " + status);
        } else {
            LOGGER.finer("Current Block2 status: " + status);
        }
        this.prepareBlockCleanup(exchange);
        return status;
    }

    private static Request getNextRequestBlock(Request request, BlockwiseStatus status) {
        int num = status.getCurrentNum();
        int szx = status.getCurrentSzx();
        Request block = new Request(request.getCode());
        block.setType(request.getType());
        block.setDestination(request.getDestination());
        block.setDestinationPort(request.getDestinationPort());
        block.setOptions(new OptionSet(request.getOptions()));
        block.addMessageObservers(request.getMessageObservers());
        int currentSize = 1 << 4 + szx;
        int from = num * currentSize;
        int to = Math.min((num + 1) * currentSize, request.getPayloadSize());
        int length = to - from;
        byte[] blockPayload = new byte[length];
        System.arraycopy(request.getPayload(), from, blockPayload, 0, length);
        block.setPayload(blockPayload);
        boolean m = to < request.getPayloadSize();
        block.getOptions().setBlock1(szx, m, num);
        status.setComplete(!m);
        return block;
    }

    private static Response getNextResponseBlock(Response response, BlockwiseStatus status) {
        Response block;
        int szx = status.getCurrentSzx();
        int num = status.getCurrentNum();
        if (response.getOptions().hasObserve()) {
            block = response;
        } else {
            block = new Response(response.getCode());
            block.setDestination(response.getDestination());
            block.setDestinationPort(response.getDestinationPort());
            block.setOptions(new OptionSet(response.getOptions()));
            block.addMessageObserver(new TimeoutForwarder(response));
        }
        int payloadsize = response.getPayloadSize();
        int currentSize = 1 << 4 + szx;
        int from = num * currentSize;
        if (0 < payloadsize && from < payloadsize) {
            int to = Math.min((num + 1) * currentSize, response.getPayloadSize());
            int length = to - from;
            byte[] blockPayload = new byte[length];
            boolean m = to < response.getPayloadSize();
            block.getOptions().setBlock2(szx, m, num);
            System.arraycopy(response.getPayload(), from, blockPayload, 0, length);
            block.setPayload(blockPayload);
            block.setLast(!m && !response.getOptions().hasObserve());
            status.setComplete(!m);
        } else {
            block.getOptions().setBlock2(szx, false, num);
            block.setLast(true);
            status.setComplete(true);
        }
        return block;
    }

    private static void assembleMessage(BlockwiseStatus status, Message message) {
        message.setSource(status.getFirst().getSource());
        message.setSourcePort(status.getFirst().getSourcePort());
        message.setType(status.getFirst().getType());
        message.setMID(status.getFirst().getMID());
        message.setToken(status.getFirst().getToken());
        message.setOptions(new OptionSet(status.getFirst().getOptions()));
        int length = 0;
        for (byte[] block : status.getBlocks()) {
            length += block.length;
        }
        byte[] payload = new byte[length];
        int offset = 0;
        for (byte[] block : status.getBlocks()) {
            System.arraycopy(block, 0, payload, offset, block.length);
            offset += block.length;
        }
        message.setPayload(payload);
    }

    private boolean requiresBlockwise(Request request) {
        if (request.getCode() == CoAP.Code.PUT || request.getCode() == CoAP.Code.POST) {
            return request.getPayloadSize() > this.max_message_size;
        }
        return false;
    }

    private boolean requireBlockwise(Exchange exchange, Response response) {
        return response.getPayloadSize() > this.max_message_size || exchange.getResponseBlockStatus() != null;
    }

    private static int computeSZX(int blockSize) {
        return (int)(Math.log(blockSize) / Math.log(2.0)) - 4;
    }

    protected void prepareBlockCleanup(Exchange exchange) {
        if (this.executor.isShutdown()) {
            LOGGER.info("Endpoint is being destroyed: skipping block clean-up");
            return;
        }
        BlockCleanupTask task = new BlockCleanupTask(exchange);
        ScheduledFuture<?> f = this.executor.schedule(task, (long)this.block_timeout, TimeUnit.MILLISECONDS);
        exchange.setBlockCleanupHandle(f);
    }

    @Override
    public void destroy() {
        this.config.removeConfigObserver(this.observer);
    }

    public static class TimeoutForwarder
    extends MessageObserverAdapter {
        private final Message message;

        public TimeoutForwarder(Message message) {
            this.message = message;
        }

        @Override
        public void onTimeout() {
            this.message.setTimedOut(true);
        }
    }

    protected class BlockCleanupTask
    implements Runnable {
        private final Exchange exchange;

        public BlockCleanupTask(Exchange exchange) {
            this.exchange = exchange;
        }

        @Override
        public void run() {
            if (this.exchange.getRequest() == null) {
                LOGGER.info("Block1 transfer timed out: " + this.exchange.getCurrentRequest());
            } else {
                LOGGER.info("Block2 transfer timed out: " + this.exchange.getRequest());
            }
            this.exchange.setComplete();
        }
    }
}

