/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.channel;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinTask;
import java.util.concurrent.Future;
import java.util.stream.Collectors;
import org.jboss.logging.Logger;
import org.wildfly.channel.ArtifactCoordinate;
import org.wildfly.channel.ArtifactTransferException;
import org.wildfly.channel.Channel;
import org.wildfly.channel.ChannelImpl;
import org.wildfly.channel.ChannelManifest;
import org.wildfly.channel.ChannelRecorder;
import org.wildfly.channel.MavenArtifact;
import org.wildfly.channel.NoStreamFoundException;
import org.wildfly.channel.Repository;
import org.wildfly.channel.RuntimeChannel;
import org.wildfly.channel.UnresolvedMavenArtifactException;
import org.wildfly.channel.VersionResult;
import org.wildfly.channel.spi.MavenVersionsResolver;
import org.wildfly.channel.version.VersionMatcher;

public class ChannelSession
implements AutoCloseable {
    private static final Logger LOG = Logger.getLogger(ChannelSession.class);
    private static final int DEFAULT_SPLIT_ARTIFACT_PARALLELISM = 10;
    private final List<ChannelImpl> channels;
    private final ChannelRecorder recorder = new ChannelRecorder();
    private final MavenVersionsResolver combinedResolver;
    private final int versionResolutionParallelism;

    public ChannelSession(List<Channel> channelDefinitions, MavenVersionsResolver.Factory factory) {
        this(channelDefinitions, factory, 10);
    }

    public ChannelSession(List<Channel> channelDefinitions, MavenVersionsResolver.Factory factory, int versionResolutionParallelism) {
        Objects.requireNonNull(channelDefinitions);
        Objects.requireNonNull(factory);
        Set<Repository> repositories = channelDefinitions.stream().flatMap(c -> c.getRepositories().stream()).collect(Collectors.toSet());
        this.combinedResolver = factory.create(repositories);
        List<ChannelImpl> channelList = channelDefinitions.stream().map(ChannelImpl::new).collect(Collectors.toList());
        for (ChannelImpl channel : channelList) {
            channel.init(factory, channelList);
        }
        this.channels = channelList.stream().filter(c -> !c.isDependency()).collect(Collectors.toList());
        this.versionResolutionParallelism = versionResolutionParallelism;
        this.validateNoDuplicatedManifests();
    }

    public List<RuntimeChannel> getRuntimeChannels() {
        return this.channels.stream().map(c -> new RuntimeChannel(c.getResolvedChannelDefinition(), c.getManifest(), c.getBlocklist())).collect(Collectors.toList());
    }

    public MavenArtifact resolveMavenArtifact(String groupId, String artifactId, String extension, String classifier, String baseVersion) throws NoStreamFoundException, ArtifactTransferException {
        Objects.requireNonNull(groupId);
        Objects.requireNonNull(artifactId);
        ChannelImpl.ResolveLatestVersionResult result = this.findChannelWithLatestVersion(groupId, artifactId, extension, classifier, baseVersion);
        String latestVersion = result.version;
        ChannelImpl channel = result.channel;
        ChannelImpl.ResolveArtifactResult artifact = channel.resolveArtifact(groupId, artifactId, extension, classifier, latestVersion);
        this.recorder.recordStream(groupId, artifactId, latestVersion);
        return new MavenArtifact(groupId, artifactId, extension, classifier, latestVersion, artifact.file, artifact.channel.getResolvedChannelDefinition().getName());
    }

    public List<MavenArtifact> resolveMavenArtifacts(List<ArtifactCoordinate> coordinates) throws UnresolvedMavenArtifactException {
        Objects.requireNonNull(coordinates);
        Map<ChannelImpl, List<ArtifactCoordinate>> channelMap = this.splitArtifactsPerChannel(coordinates);
        ArrayList<MavenArtifact> res = new ArrayList<MavenArtifact>();
        for (ChannelImpl channel : channelMap.keySet()) {
            List<ArtifactCoordinate> requests = channelMap.get(channel);
            List<ChannelImpl.ResolveArtifactResult> resolveArtifactResults = channel.resolveArtifacts(requests);
            for (int i = 0; i < requests.size(); ++i) {
                ArtifactCoordinate request = requests.get(i);
                MavenArtifact resolvedArtifact = new MavenArtifact(request.getGroupId(), request.getArtifactId(), request.getExtension(), request.getClassifier(), request.getVersion(), resolveArtifactResults.get((int)i).file, resolveArtifactResults.get((int)i).channel.getResolvedChannelDefinition().getName());
                this.recorder.recordStream(resolvedArtifact.getGroupId(), resolvedArtifact.getArtifactId(), resolvedArtifact.getVersion());
                res.add(resolvedArtifact);
            }
        }
        return res;
    }

    public MavenArtifact resolveDirectMavenArtifact(String groupId, String artifactId, String extension, String classifier, String version) throws ArtifactTransferException {
        Objects.requireNonNull(groupId);
        Objects.requireNonNull(artifactId);
        Objects.requireNonNull(version);
        File file = this.combinedResolver.resolveArtifact(groupId, artifactId, extension, classifier, version);
        this.recorder.recordStream(groupId, artifactId, version);
        return new MavenArtifact(groupId, artifactId, extension, classifier, version, file);
    }

    public List<MavenArtifact> resolveDirectMavenArtifacts(List<ArtifactCoordinate> coordinates) throws ArtifactTransferException {
        coordinates.forEach(c -> {
            Objects.requireNonNull(c.getGroupId());
            Objects.requireNonNull(c.getArtifactId());
            Objects.requireNonNull(c.getVersion());
        });
        List<File> files = this.combinedResolver.resolveArtifacts(coordinates);
        ArrayList<MavenArtifact> res = new ArrayList<MavenArtifact>();
        for (int i = 0; i < coordinates.size(); ++i) {
            ArtifactCoordinate request = coordinates.get(i);
            MavenArtifact resolvedArtifact = new MavenArtifact(request.getGroupId(), request.getArtifactId(), request.getExtension(), request.getClassifier(), request.getVersion(), files.get(i));
            this.recorder.recordStream(resolvedArtifact.getGroupId(), resolvedArtifact.getArtifactId(), resolvedArtifact.getVersion());
            res.add(resolvedArtifact);
        }
        return res;
    }

    public VersionResult findLatestMavenArtifactVersion(String groupId, String artifactId, String extension, String classifier, String baseVersion) throws NoStreamFoundException {
        ChannelImpl.ResolveLatestVersionResult channelWithLatestVersion = this.findChannelWithLatestVersion(groupId, artifactId, extension, classifier, baseVersion);
        return new VersionResult(channelWithLatestVersion.version, channelWithLatestVersion.channel.getResolvedChannelDefinition().getName());
    }

    @Override
    public void close() {
        for (ChannelImpl channel : this.channels) {
            channel.close();
        }
        this.combinedResolver.close();
    }

    public ChannelManifest getRecordedChannel() {
        return this.recorder.getRecordedChannel();
    }

    private void validateNoDuplicatedManifests() {
        List manifestIds = this.channels.stream().map(c -> c.getManifest().getId()).filter(Objects::nonNull).collect(Collectors.toList());
        if (manifestIds.size() != new HashSet(manifestIds).size()) {
            throw new RuntimeException("The same manifest is provided by one or more channels");
        }
    }

    private ChannelImpl.ResolveLatestVersionResult findChannelWithLatestVersion(String groupId, String artifactId, String extension, String classifier, String baseVersion) throws NoStreamFoundException {
        Objects.requireNonNull(groupId);
        Objects.requireNonNull(artifactId);
        HashMap<String, ChannelImpl.ResolveLatestVersionResult> foundVersions = new HashMap<String, ChannelImpl.ResolveLatestVersionResult>();
        for (ChannelImpl channel : this.channels) {
            Optional<ChannelImpl.ResolveLatestVersionResult> result = channel.resolveLatestVersion(groupId, artifactId, extension, classifier, baseVersion);
            if (!result.isPresent()) continue;
            foundVersions.put(result.get().version, result.get());
        }
        Optional<String> foundLatestVersionInChannels = foundVersions.keySet().stream().max(VersionMatcher.COMPARATOR);
        return (ChannelImpl.ResolveLatestVersionResult)foundVersions.get(foundLatestVersionInChannels.orElseThrow(() -> {
            ArtifactCoordinate coord = new ArtifactCoordinate(groupId, artifactId, extension, classifier, "");
            Set<Repository> repositories = this.channels.stream().map(ChannelImpl::getResolvedChannelDefinition).flatMap(d -> d.getRepositories().stream()).collect(Collectors.toSet());
            throw new NoStreamFoundException(String.format("Can not resolve latest Maven artifact (no stream found) : %s:%s:%s:%s", groupId, artifactId, extension, classifier), Collections.singleton(coord), repositories);
        }));
    }

    private Map<ChannelImpl, List<ArtifactCoordinate>> splitArtifactsPerChannel(List<ArtifactCoordinate> coordinates) {
        long start = System.currentTimeMillis();
        HashMap<ChannelImpl, List<ArtifactCoordinate>> channelMap = new HashMap<ChannelImpl, List<ArtifactCoordinate>>();
        ForkJoinPool customThreadPool = new ForkJoinPool(this.versionResolutionParallelism);
        Future task = customThreadPool.submit(() -> coordinates.parallelStream().forEach(coord -> {
            ChannelImpl.ResolveLatestVersionResult result = this.findChannelWithLatestVersion(coord.getGroupId(), coord.getArtifactId(), coord.getExtension(), coord.getClassifier(), coord.getVersion());
            ArtifactCoordinate query = new ArtifactCoordinate(coord.getGroupId(), coord.getArtifactId(), coord.getExtension(), coord.getClassifier(), result.version);
            ChannelImpl channel = result.channel;
            ChannelSession channelSession = this;
            synchronized (channelSession) {
                if (!channelMap.containsKey(channel)) {
                    channelMap.put(channel, new ArrayList());
                }
                ((List)channelMap.get(channel)).add(query);
            }
        }));
        try {
            ((ForkJoinTask)task).get();
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof UnresolvedMavenArtifactException) {
                throw (UnresolvedMavenArtifactException)e.getCause();
            }
            throw new RuntimeException("Unable to resolve latest artifact versions.", e);
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Unable to resolve latest artifact versions: interrupted", e);
        }
        finally {
            customThreadPool.shutdown();
        }
        float total = (float)(System.currentTimeMillis() - start) / 1000.0f;
        LOG.debugf("Splitting artifacts per channels took %.2f seconds", (Object)Float.valueOf(total));
        return channelMap;
    }
}

