/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.channel.gpg;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.net.URI;
import java.net.URLConnection;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import org.bouncycastle.bcpg.ArmoredInputStream;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureList;
import org.bouncycastle.openpgp.PGPSignatureSubpacketVector;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.KeyFingerPrintCalculator;
import org.bouncycastle.openpgp.operator.PGPContentVerifierBuilderProvider;
import org.bouncycastle.openpgp.operator.bc.BcPGPContentVerifierBuilderProvider;
import org.bouncycastle.openpgp.operator.jcajce.JcaKeyFingerprintCalculator;
import org.jboss.logging.Logger;
import org.wildfly.channel.gpg.GpgKeystore;
import org.wildfly.channel.gpg.GpgSignatureValidatorListener;
import org.wildfly.channel.gpg.Keyserver;
import org.wildfly.channel.spi.ArtifactIdentifier;
import org.wildfly.channel.spi.SignatureResult;
import org.wildfly.channel.spi.SignatureValidator;

public class GpgSignatureValidator
implements SignatureValidator {
    private static final Logger LOG = Logger.getLogger(GpgSignatureValidator.class);
    private final GpgKeystore keystore;
    private final Keyserver keyserver;
    private GpgSignatureValidatorListener listener = new NoopListener();

    public GpgSignatureValidator(GpgKeystore keystore) {
        this(keystore, new Keyserver(Collections.emptyList()));
    }

    public GpgSignatureValidator(GpgKeystore keystore, Keyserver keyserver) {
        this.keystore = keystore;
        this.keyserver = keyserver;
    }

    public void addListener(GpgSignatureValidatorListener listener) {
        this.listener = listener;
    }

    public SignatureResult validateSignature(ArtifactIdentifier artifactId, InputStream artifactStream, InputStream signatureStream, List<String> gpgUrls) throws SignatureValidator.SignatureException {
        SignatureResult res;
        PGPPublicKey publicKey;
        PGPSignature pgpSignature;
        Objects.requireNonNull(artifactId);
        Objects.requireNonNull(artifactStream);
        Objects.requireNonNull(signatureStream);
        try {
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)"Reading the signature of artifact.");
            }
            pgpSignature = GpgSignatureValidator.readSignatureFile(signatureStream);
        }
        catch (IOException e) {
            throw new SignatureValidator.SignatureException("Could not find signature in provided signature file", (Throwable)e, SignatureResult.noSignature((ArtifactIdentifier)artifactId));
        }
        if (pgpSignature == null) {
            LOG.error((Object)"Could not read the signature in provided signature file");
            return SignatureResult.noSignature((ArtifactIdentifier)artifactId);
        }
        String keyID = GpgSignatureValidator.getKeyID(pgpSignature);
        if (LOG.isTraceEnabled()) {
            LOG.tracef("The signature was created using public key %s.", (Object)keyID);
        }
        if (this.keystore.get(keyID) != null) {
            if (LOG.isTraceEnabled()) {
                LOG.tracef("Using a public key %s was found in the local keystore.", (Object)keyID);
            }
            publicKey = this.keystore.get(keyID);
        } else {
            if (LOG.isTraceEnabled()) {
                LOG.tracef("Trying to download a public key %s from remote keyservers.", (Object)keyID);
            }
            List<Object> pgpPublicKeys = null;
            PGPPublicKey key = null;
            try {
                PGPPublicKeyRing keyRing = this.keyserver.downloadKey(keyID);
                if (keyRing != null) {
                    Iterator publicKeys = keyRing.getPublicKeys();
                    key = keyRing.getPublicKey(new BigInteger(keyID, 16).longValue());
                    pgpPublicKeys = new ArrayList<PGPPublicKey>();
                    while (publicKeys.hasNext()) {
                        pgpPublicKeys.add((PGPPublicKey)publicKeys.next());
                    }
                }
            }
            catch (IOException | PGPException e) {
                throw new SignatureValidator.SignatureException("Unable to parse the certificate downloaded from keyserver", e, SignatureResult.noMatchingCertificate((ArtifactIdentifier)artifactId, (String)keyID));
            }
            if (key == null) {
                for (String gpgUrl : gpgUrls) {
                    if (LOG.isTraceEnabled()) {
                        LOG.tracef("Trying to download a public key %s from channel defined URL %s.", (Object)keyID, (Object)gpgUrl);
                    }
                    try {
                        pgpPublicKeys = GpgSignatureValidator.downloadPublicKey(gpgUrl);
                    }
                    catch (IOException e) {
                        throw new SignatureValidator.SignatureException("Unable to parse the certificate downloaded from " + gpgUrl, (Throwable)e, SignatureResult.noMatchingCertificate((ArtifactIdentifier)artifactId, (String)keyID));
                    }
                    if (!pgpPublicKeys.stream().anyMatch(k -> k.getKeyID() == pgpSignature.getKeyID())) continue;
                    key = pgpPublicKeys.stream().filter(k -> k.getKeyID() == pgpSignature.getKeyID()).findFirst().get();
                    break;
                }
                if (key == null) {
                    if (LOG.isTraceEnabled()) {
                        LOG.tracef("A public key %s not found in the channel defined URLs.", (Object)keyID);
                    }
                    return SignatureResult.noMatchingCertificate((ArtifactIdentifier)artifactId, (String)keyID);
                }
            }
            if (this.keystore.add(pgpPublicKeys)) {
                if (LOG.isTraceEnabled()) {
                    LOG.tracef("Adding a public key %s to the local keystore.", (Object)keyID);
                }
                publicKey = key;
            } else {
                return SignatureResult.noMatchingCertificate((ArtifactIdentifier)artifactId, (String)keyID);
            }
        }
        if (LOG.isTraceEnabled()) {
            LOG.tracef("Checking if the public key %s is still valid.", (Object)artifactId);
        }
        if ((res = this.checkRevoked(artifactId, keyID, publicKey)).getResult() != SignatureResult.Result.OK) {
            return res;
        }
        res = GpgSignatureValidator.checkExpired(artifactId, publicKey, keyID);
        if (res.getResult() != SignatureResult.Result.OK) {
            return res;
        }
        if (LOG.isTraceEnabled()) {
            LOG.tracef("Verifying that artifact %s has been signed with public key %s.", (Object)artifactId, (Object)keyID);
        }
        try {
            pgpSignature.init((PGPContentVerifierBuilderProvider)new BcPGPContentVerifierBuilderProvider(), publicKey);
        }
        catch (PGPException e) {
            throw new SignatureValidator.SignatureException("Unable to verify the signature using key " + keyID, (Throwable)e, SignatureResult.invalid((ArtifactIdentifier)artifactId, (String)keyID));
        }
        SignatureResult result = GpgSignatureValidator.verifyFile(artifactId, artifactStream, pgpSignature);
        if (result.getResult() == SignatureResult.Result.OK) {
            this.listener.artifactSignatureCorrect(artifactId, publicKey);
        } else {
            this.listener.artifactSignatureInvalid(artifactId, publicKey);
        }
        return result;
    }

    private static SignatureResult checkExpired(ArtifactIdentifier artifactId, PGPPublicKey publicKey, String keyID) {
        if (LOG.isTraceEnabled()) {
            LOG.tracef("Checking if public key %s is not expired.", (Object)keyID);
        }
        if (publicKey.getValidSeconds() > 0L) {
            Instant expiry = Instant.from(publicKey.getCreationTime().toInstant().plus(publicKey.getValidSeconds(), ChronoUnit.SECONDS));
            if (LOG.isTraceEnabled()) {
                LOG.tracef("Public key %s expirates on %s.", (Object)keyID, (Object)expiry);
            }
            if (expiry.isBefore(Instant.now())) {
                return SignatureResult.expired((ArtifactIdentifier)artifactId, (String)keyID);
            }
        } else if (LOG.isTraceEnabled()) {
            LOG.tracef("Public key %s has no expiration.", (Object)keyID);
        }
        return SignatureResult.ok();
    }

    private SignatureResult checkRevoked(ArtifactIdentifier artifactId, String keyID, PGPPublicKey publicKey) {
        if (LOG.isTraceEnabled()) {
            LOG.tracef("Checking if public key %s has been revoked.", (Object)keyID);
        }
        if (publicKey.hasRevocation()) {
            if (LOG.isTraceEnabled()) {
                LOG.tracef("Public key %s has been revoked.", (Object)keyID);
            }
            return SignatureResult.revoked((ArtifactIdentifier)artifactId, (String)keyID, (String)GpgSignatureValidator.getRevocationReason(publicKey));
        }
        Iterator subKeys = publicKey.getSignaturesOfType(24);
        while (subKeys.hasNext()) {
            PGPSignature subKeySignature = (PGPSignature)subKeys.next();
            PGPPublicKey subKey = this.keystore.get(GpgSignatureValidator.getKeyID(subKeySignature));
            if (!subKey.hasRevocation()) continue;
            if (LOG.isTraceEnabled()) {
                LOG.tracef("Sub-key %s has been revoked.", (Object)Long.toHexString(subKey.getKeyID()).toUpperCase(Locale.ROOT));
            }
            return SignatureResult.revoked((ArtifactIdentifier)artifactId, (String)keyID, (String)GpgSignatureValidator.getRevocationReason(publicKey));
        }
        return SignatureResult.ok();
    }

    private static String getRevocationReason(PGPPublicKey publicKey) {
        Iterator keySignatures = publicKey.getSignaturesOfType(32);
        String revocationDescription = null;
        while (keySignatures.hasNext()) {
            PGPSignature sign = (PGPSignature)keySignatures.next();
            if (sign.getSignatureType() != 32) continue;
            PGPSignatureSubpacketVector hashedSubPackets = sign.getHashedSubPackets();
            revocationDescription = hashedSubPackets.getRevocationReason().getRevocationDescription();
        }
        return revocationDescription;
    }

    private static SignatureResult verifyFile(ArtifactIdentifier artifactSource, InputStream artifactStream, PGPSignature pgpSignature) throws SignatureValidator.SignatureException {
        byte[] data = new byte[1024];
        DataInputStream inputStream = null;
        try {
            int bytesRead;
            inputStream = new DataInputStream(new BufferedInputStream(artifactStream));
            while ((bytesRead = ((InputStream)inputStream).read(data, 0, 1024)) != -1) {
                pgpSignature.update(data, 0, bytesRead);
            }
            ((InputStream)inputStream).close();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        try {
            if (!pgpSignature.verify()) {
                return SignatureResult.invalid((ArtifactIdentifier)artifactSource, (String)GpgSignatureValidator.getKeyID(pgpSignature));
            }
            return SignatureResult.ok();
        }
        catch (PGPException e) {
            throw new SignatureValidator.SignatureException("Unable to verify the file signature", (Throwable)e, SignatureResult.invalid((ArtifactIdentifier)artifactSource, (String)GpgSignatureValidator.getKeyID(pgpSignature)));
        }
    }

    private static String getKeyID(PGPSignature pgpSignature) {
        return Long.toHexString(pgpSignature.getKeyID()).toUpperCase(Locale.ROOT);
    }

    private static PGPSignature readSignatureFile(InputStream signatureStream) throws IOException {
        PGPSignature pgpSignature = null;
        try (InputStream decoderStream = PGPUtil.getDecoderStream((InputStream)signatureStream);){
            PGPObjectFactory pgpObjectFactory = new PGPObjectFactory(decoderStream, (KeyFingerPrintCalculator)new JcaKeyFingerprintCalculator());
            Object o = pgpObjectFactory.nextObject();
            if (o instanceof PGPSignatureList) {
                PGPSignatureList signatureList = (PGPSignatureList)o;
                if (signatureList.isEmpty()) {
                    throw new RuntimeException("signatureList must not be empty");
                }
                pgpSignature = signatureList.get(0);
            } else if (o instanceof PGPSignature) {
                pgpSignature = (PGPSignature)o;
            }
        }
        return pgpSignature;
    }

    private static List<PGPPublicKey> downloadPublicKey(String signatureUrl) throws IOException {
        InputStream inputStream;
        URI uri = URI.create(signatureUrl);
        if (uri.getScheme().equals("classpath")) {
            if (LOG.isTraceEnabled()) {
                LOG.tracef("Resolving the public key from classpath %s.", (Object)uri);
            }
            String keyPath = uri.getSchemeSpecificPart();
            inputStream = GpgSignatureValidator.class.getClassLoader().getResourceAsStream(keyPath);
        } else {
            if (LOG.isTraceEnabled()) {
                LOG.tracef("Downloading the public key from %s.", (Object)uri);
            }
            URLConnection urlConnection = uri.toURL().openConnection();
            urlConnection.connect();
            inputStream = urlConnection.getInputStream();
        }
        try (ArmoredInputStream decoderStream = new ArmoredInputStream(inputStream);){
            PGPPublicKeyRing pgpPublicKeys = new PGPPublicKeyRing((InputStream)decoderStream, (KeyFingerPrintCalculator)new JcaKeyFingerprintCalculator());
            ArrayList<PGPPublicKey> res = new ArrayList<PGPPublicKey>();
            Iterator publicKeys = pgpPublicKeys.getPublicKeys();
            while (publicKeys.hasNext()) {
                res.add((PGPPublicKey)publicKeys.next());
            }
            ArrayList<PGPPublicKey> arrayList = res;
            return arrayList;
        }
    }

    private static class NoopListener
    implements GpgSignatureValidatorListener {
        private NoopListener() {
        }

        @Override
        public void artifactSignatureCorrect(ArtifactIdentifier artifact, PGPPublicKey publicKey) {
        }

        @Override
        public void artifactSignatureInvalid(ArtifactIdentifier artifact, PGPPublicKey publicKey) {
        }
    }
}

