/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.MoreExecutors;
import io.grpc.Attributes;
import io.grpc.CallOptions;
import io.grpc.ClientCall;
import io.grpc.ClientStreamTracer;
import io.grpc.Codec;
import io.grpc.Compressor;
import io.grpc.CompressorRegistry;
import io.grpc.Context;
import io.grpc.Contexts;
import io.grpc.Deadline;
import io.grpc.DecompressorRegistry;
import io.grpc.InternalConfigSelector;
import io.grpc.InternalDecompressorRegistry;
import io.grpc.Metadata;
import io.grpc.MethodDescriptor;
import io.grpc.Status;
import io.grpc.internal.CallTracer;
import io.grpc.internal.ClientStream;
import io.grpc.internal.ClientStreamListener;
import io.grpc.internal.ContextRunnable;
import io.grpc.internal.FailingClientStream;
import io.grpc.internal.GrpcUtil;
import io.grpc.internal.InsightBuilder;
import io.grpc.internal.LogExceptionRunnable;
import io.grpc.internal.ManagedChannelServiceConfig;
import io.grpc.internal.NoopClientStream;
import io.grpc.internal.RetriableStream;
import io.grpc.internal.SerializeReentrantCallsDirectExecutor;
import io.grpc.internal.SerializingExecutor;
import io.grpc.internal.StreamListener;
import io.perfmark.Link;
import io.perfmark.PerfMark;
import io.perfmark.Tag;
import jakarta.annotation.Nullable;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Locale;
import java.util.concurrent.CancellationException;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

final class ClientCallImpl<ReqT, RespT>
extends ClientCall<ReqT, RespT> {
    private static final Logger log = Logger.getLogger(ClientCallImpl.class.getName());
    private static final byte[] FULL_STREAM_DECOMPRESSION_ENCODINGS = "gzip".getBytes(Charset.forName("US-ASCII"));
    private final MethodDescriptor<ReqT, RespT> method;
    private final Tag tag;
    private final Executor callExecutor;
    private final boolean callExecutorIsDirect;
    private final CallTracer channelCallsTracer;
    private final Context context;
    private volatile ScheduledFuture<?> deadlineCancellationFuture;
    private final boolean unaryRequest;
    private CallOptions callOptions;
    private ClientStream stream;
    private volatile boolean cancelListenersShouldBeRemoved;
    private boolean cancelCalled;
    private boolean halfCloseCalled;
    private final ClientStreamProvider clientStreamProvider;
    private final ContextCancellationListener cancellationListener = new ContextCancellationListener();
    private final ScheduledExecutorService deadlineCancellationExecutor;
    private boolean fullStreamDecompression;
    private DecompressorRegistry decompressorRegistry = DecompressorRegistry.getDefaultInstance();
    private CompressorRegistry compressorRegistry = CompressorRegistry.getDefaultInstance();

    ClientCallImpl(MethodDescriptor<ReqT, RespT> method, Executor executor, CallOptions callOptions, ClientStreamProvider clientStreamProvider, ScheduledExecutorService deadlineCancellationExecutor, CallTracer channelCallsTracer, @Nullable InternalConfigSelector configSelector) {
        this.method = method;
        this.tag = PerfMark.createTag((String)method.getFullMethodName(), (long)System.identityHashCode((Object)this));
        if (executor == MoreExecutors.directExecutor()) {
            this.callExecutor = new SerializeReentrantCallsDirectExecutor();
            this.callExecutorIsDirect = true;
        } else {
            this.callExecutor = new SerializingExecutor(executor);
            this.callExecutorIsDirect = false;
        }
        this.channelCallsTracer = channelCallsTracer;
        this.context = Context.current();
        this.unaryRequest = method.getType() == MethodDescriptor.MethodType.UNARY || method.getType() == MethodDescriptor.MethodType.SERVER_STREAMING;
        this.callOptions = callOptions;
        this.clientStreamProvider = clientStreamProvider;
        this.deadlineCancellationExecutor = deadlineCancellationExecutor;
        PerfMark.event((String)"ClientCall.<init>", (Tag)this.tag);
    }

    ClientCallImpl<ReqT, RespT> setFullStreamDecompression(boolean fullStreamDecompression) {
        this.fullStreamDecompression = fullStreamDecompression;
        return this;
    }

    ClientCallImpl<ReqT, RespT> setDecompressorRegistry(DecompressorRegistry decompressorRegistry) {
        this.decompressorRegistry = decompressorRegistry;
        return this;
    }

    ClientCallImpl<ReqT, RespT> setCompressorRegistry(CompressorRegistry compressorRegistry) {
        this.compressorRegistry = compressorRegistry;
        return this;
    }

    @VisibleForTesting
    static void prepareHeaders(Metadata headers, DecompressorRegistry decompressorRegistry, Compressor compressor, boolean fullStreamDecompression) {
        headers.discardAll(GrpcUtil.CONTENT_LENGTH_KEY);
        headers.discardAll(GrpcUtil.MESSAGE_ENCODING_KEY);
        if (compressor != Codec.Identity.NONE) {
            headers.put(GrpcUtil.MESSAGE_ENCODING_KEY, (Object)compressor.getMessageEncoding());
        }
        headers.discardAll(GrpcUtil.MESSAGE_ACCEPT_ENCODING_KEY);
        byte[] advertisedEncodings = InternalDecompressorRegistry.getRawAdvertisedMessageEncodings((DecompressorRegistry)decompressorRegistry);
        if (advertisedEncodings.length != 0) {
            headers.put(GrpcUtil.MESSAGE_ACCEPT_ENCODING_KEY, (Object)advertisedEncodings);
        }
        headers.discardAll(GrpcUtil.CONTENT_ENCODING_KEY);
        headers.discardAll(GrpcUtil.CONTENT_ACCEPT_ENCODING_KEY);
        if (fullStreamDecompression) {
            headers.put(GrpcUtil.CONTENT_ACCEPT_ENCODING_KEY, (Object)FULL_STREAM_DECOMPRESSION_ENCODINGS);
        }
    }

    public void start(ClientCall.Listener<RespT> observer, Metadata headers) {
        PerfMark.startTask((String)"ClientCall.start", (Tag)this.tag);
        try {
            this.startInternal(observer, headers);
        }
        finally {
            PerfMark.stopTask((String)"ClientCall.start", (Tag)this.tag);
        }
    }

    private void startInternal(ClientCall.Listener<RespT> observer, Metadata headers) {
        boolean deadlineExceeded;
        Codec compressor;
        Preconditions.checkState((this.stream == null ? 1 : 0) != 0, (Object)"Already started");
        Preconditions.checkState((!this.cancelCalled ? 1 : 0) != 0, (Object)"call was cancelled");
        Preconditions.checkNotNull(observer, (Object)"observer");
        Preconditions.checkNotNull((Object)headers, (Object)"headers");
        if (this.context.isCancelled()) {
            this.stream = NoopClientStream.INSTANCE;
            final ClientCall.Listener<RespT> finalObserver = observer;
            class ClosedByContext
            extends ContextRunnable {
                ClosedByContext() {
                    super(ClientCallImpl.this.context);
                }

                @Override
                public void runInContext() {
                    ClientCallImpl.this.closeObserver(finalObserver, Contexts.statusFromCancelled((Context)ClientCallImpl.this.context), new Metadata());
                }
            }
            this.callExecutor.execute(new ClosedByContext());
            return;
        }
        this.applyMethodConfig();
        final String compressorName = this.callOptions.getCompressor();
        if (compressorName != null) {
            compressor = this.compressorRegistry.lookupCompressor(compressorName);
            if (compressor == null) {
                this.stream = NoopClientStream.INSTANCE;
                final ClientCall.Listener<RespT> finalObserver = observer;
                class ClosedByNotFoundCompressor
                extends ContextRunnable {
                    ClosedByNotFoundCompressor() {
                        super(ClientCallImpl.this.context);
                    }

                    @Override
                    public void runInContext() {
                        ClientCallImpl.this.closeObserver(finalObserver, Status.INTERNAL.withDescription(String.format("Unable to find compressor by name %s", compressorName)), new Metadata());
                    }
                }
                this.callExecutor.execute(new ClosedByNotFoundCompressor());
                return;
            }
        } else {
            compressor = Codec.Identity.NONE;
        }
        ClientCallImpl.prepareHeaders(headers, this.decompressorRegistry, (Compressor)compressor, this.fullStreamDecompression);
        Deadline effectiveDeadline = this.effectiveDeadline();
        boolean bl = deadlineExceeded = effectiveDeadline != null && effectiveDeadline.isExpired();
        if (!deadlineExceeded) {
            ClientCallImpl.logIfContextNarrowedTimeout(effectiveDeadline, this.context.getDeadline(), this.callOptions.getDeadline());
            this.stream = this.clientStreamProvider.newStream(this.method, this.callOptions, headers, this.context);
        } else {
            ClientStreamTracer[] tracers = GrpcUtil.getClientStreamTracers(this.callOptions, headers, 0, false);
            this.stream = new FailingClientStream(Status.DEADLINE_EXCEEDED.withDescription("ClientCall started after deadline exceeded: " + effectiveDeadline), tracers);
        }
        if (this.callExecutorIsDirect) {
            this.stream.optimizeForDirectExecutor();
        }
        if (this.callOptions.getAuthority() != null) {
            this.stream.setAuthority(this.callOptions.getAuthority());
        }
        if (this.callOptions.getMaxInboundMessageSize() != null) {
            this.stream.setMaxInboundMessageSize(this.callOptions.getMaxInboundMessageSize());
        }
        if (this.callOptions.getMaxOutboundMessageSize() != null) {
            this.stream.setMaxOutboundMessageSize(this.callOptions.getMaxOutboundMessageSize());
        }
        if (effectiveDeadline != null) {
            this.stream.setDeadline(effectiveDeadline);
        }
        this.stream.setCompressor((Compressor)compressor);
        if (this.fullStreamDecompression) {
            this.stream.setFullStreamDecompression(this.fullStreamDecompression);
        }
        this.stream.setDecompressorRegistry(this.decompressorRegistry);
        this.channelCallsTracer.reportCallStarted();
        this.stream.start(new ClientStreamListenerImpl(observer));
        this.context.addListener((Context.CancellationListener)this.cancellationListener, MoreExecutors.directExecutor());
        if (effectiveDeadline != null && !effectiveDeadline.equals((Object)this.context.getDeadline()) && this.deadlineCancellationExecutor != null) {
            this.deadlineCancellationFuture = this.startDeadlineTimer(effectiveDeadline);
        }
        if (this.cancelListenersShouldBeRemoved) {
            this.removeContextListenerAndCancelDeadlineFuture();
        }
    }

    private void applyMethodConfig() {
        Integer existingLimit;
        ManagedChannelServiceConfig.MethodInfo info = (ManagedChannelServiceConfig.MethodInfo)this.callOptions.getOption(ManagedChannelServiceConfig.MethodInfo.KEY);
        if (info == null) {
            return;
        }
        if (info.timeoutNanos != null) {
            Deadline newDeadline = Deadline.after((long)info.timeoutNanos, (TimeUnit)TimeUnit.NANOSECONDS);
            Deadline existingDeadline = this.callOptions.getDeadline();
            if (existingDeadline == null || newDeadline.compareTo(existingDeadline) < 0) {
                this.callOptions = this.callOptions.withDeadline(newDeadline);
            }
        }
        if (info.waitForReady != null) {
            CallOptions callOptions = this.callOptions = info.waitForReady != false ? this.callOptions.withWaitForReady() : this.callOptions.withoutWaitForReady();
        }
        if (info.maxInboundMessageSize != null) {
            existingLimit = this.callOptions.getMaxInboundMessageSize();
            this.callOptions = existingLimit != null ? this.callOptions.withMaxInboundMessageSize(Math.min(existingLimit, info.maxInboundMessageSize)) : this.callOptions.withMaxInboundMessageSize(info.maxInboundMessageSize.intValue());
        }
        if (info.maxOutboundMessageSize != null) {
            existingLimit = this.callOptions.getMaxOutboundMessageSize();
            this.callOptions = existingLimit != null ? this.callOptions.withMaxOutboundMessageSize(Math.min(existingLimit, info.maxOutboundMessageSize)) : this.callOptions.withMaxOutboundMessageSize(info.maxOutboundMessageSize.intValue());
        }
    }

    private static void logIfContextNarrowedTimeout(Deadline effectiveDeadline, @Nullable Deadline outerCallDeadline, @Nullable Deadline callDeadline) {
        if (!log.isLoggable(Level.FINE) || effectiveDeadline == null || !effectiveDeadline.equals((Object)outerCallDeadline)) {
            return;
        }
        long effectiveTimeout = Math.max(0L, effectiveDeadline.timeRemaining(TimeUnit.NANOSECONDS));
        StringBuilder builder = new StringBuilder(String.format(Locale.US, "Call timeout set to '%d' ns, due to context deadline.", effectiveTimeout));
        if (callDeadline == null) {
            builder.append(" Explicit call timeout was not set.");
        } else {
            long callTimeout = callDeadline.timeRemaining(TimeUnit.NANOSECONDS);
            builder.append(String.format(Locale.US, " Explicit call timeout was '%d' ns.", callTimeout));
        }
        log.fine(builder.toString());
    }

    private void removeContextListenerAndCancelDeadlineFuture() {
        this.context.removeListener((Context.CancellationListener)this.cancellationListener);
        ScheduledFuture<?> f = this.deadlineCancellationFuture;
        if (f != null) {
            f.cancel(false);
        }
    }

    private ScheduledFuture<?> startDeadlineTimer(Deadline deadline) {
        long remainingNanos = deadline.timeRemaining(TimeUnit.NANOSECONDS);
        return this.deadlineCancellationExecutor.schedule(new LogExceptionRunnable(new DeadlineTimer(remainingNanos)), remainingNanos, TimeUnit.NANOSECONDS);
    }

    @Nullable
    private Deadline effectiveDeadline() {
        return ClientCallImpl.min(this.callOptions.getDeadline(), this.context.getDeadline());
    }

    @Nullable
    private static Deadline min(@Nullable Deadline deadline0, @Nullable Deadline deadline1) {
        if (deadline0 == null) {
            return deadline1;
        }
        if (deadline1 == null) {
            return deadline0;
        }
        return deadline0.minimum(deadline1);
    }

    public void request(int numMessages) {
        PerfMark.startTask((String)"ClientCall.request", (Tag)this.tag);
        try {
            Preconditions.checkState((this.stream != null ? 1 : 0) != 0, (Object)"Not started");
            Preconditions.checkArgument((numMessages >= 0 ? 1 : 0) != 0, (Object)"Number requested must be non-negative");
            this.stream.request(numMessages);
        }
        finally {
            PerfMark.stopTask((String)"ClientCall.request", (Tag)this.tag);
        }
    }

    public void cancel(@Nullable String message, @Nullable Throwable cause) {
        PerfMark.startTask((String)"ClientCall.cancel", (Tag)this.tag);
        try {
            this.cancelInternal(message, cause);
        }
        finally {
            PerfMark.stopTask((String)"ClientCall.cancel", (Tag)this.tag);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cancelInternal(@Nullable String message, @Nullable Throwable cause) {
        if (message == null && cause == null) {
            cause = new CancellationException("Cancelled without a message or cause");
            log.log(Level.WARNING, "Cancelling without a message or cause is suboptimal", cause);
        }
        if (this.cancelCalled) {
            return;
        }
        this.cancelCalled = true;
        try {
            if (this.stream != null) {
                Status status = Status.CANCELLED;
                status = message != null ? status.withDescription(message) : status.withDescription("Call cancelled without message");
                if (cause != null) {
                    status = status.withCause(cause);
                }
                this.stream.cancel(status);
            }
        }
        finally {
            this.removeContextListenerAndCancelDeadlineFuture();
        }
    }

    public void halfClose() {
        PerfMark.startTask((String)"ClientCall.halfClose", (Tag)this.tag);
        try {
            this.halfCloseInternal();
        }
        finally {
            PerfMark.stopTask((String)"ClientCall.halfClose", (Tag)this.tag);
        }
    }

    private void halfCloseInternal() {
        Preconditions.checkState((this.stream != null ? 1 : 0) != 0, (Object)"Not started");
        Preconditions.checkState((!this.cancelCalled ? 1 : 0) != 0, (Object)"call was cancelled");
        Preconditions.checkState((!this.halfCloseCalled ? 1 : 0) != 0, (Object)"call already half-closed");
        this.halfCloseCalled = true;
        this.stream.halfClose();
    }

    public void sendMessage(ReqT message) {
        PerfMark.startTask((String)"ClientCall.sendMessage", (Tag)this.tag);
        try {
            this.sendMessageInternal(message);
        }
        finally {
            PerfMark.stopTask((String)"ClientCall.sendMessage", (Tag)this.tag);
        }
    }

    private void sendMessageInternal(ReqT message) {
        Preconditions.checkState((this.stream != null ? 1 : 0) != 0, (Object)"Not started");
        Preconditions.checkState((!this.cancelCalled ? 1 : 0) != 0, (Object)"call was cancelled");
        Preconditions.checkState((!this.halfCloseCalled ? 1 : 0) != 0, (Object)"call was half-closed");
        try {
            if (this.stream instanceof RetriableStream) {
                RetriableStream retriableStream = (RetriableStream)this.stream;
                retriableStream.sendMessage(message);
            } else {
                this.stream.writeMessage(this.method.streamRequest(message));
            }
        }
        catch (RuntimeException e) {
            this.stream.cancel(Status.CANCELLED.withCause((Throwable)e).withDescription("Failed to stream message"));
            return;
        }
        catch (Error e) {
            this.stream.cancel(Status.CANCELLED.withDescription("Client sendMessage() failed with Error"));
            throw e;
        }
        if (!this.unaryRequest) {
            this.stream.flush();
        }
    }

    public void setMessageCompression(boolean enabled) {
        Preconditions.checkState((this.stream != null ? 1 : 0) != 0, (Object)"Not started");
        this.stream.setMessageCompression(enabled);
    }

    public boolean isReady() {
        if (this.halfCloseCalled) {
            return false;
        }
        return this.stream.isReady();
    }

    public Attributes getAttributes() {
        if (this.stream != null) {
            return this.stream.getAttributes();
        }
        return Attributes.EMPTY;
    }

    private void closeObserver(ClientCall.Listener<RespT> observer, Status status, Metadata trailers) {
        observer.onClose(status, trailers);
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)((Object)this)).add("method", this.method).toString();
    }

    private class ClientStreamListenerImpl
    implements ClientStreamListener {
        private final ClientCall.Listener<RespT> observer;
        private Status exceptionStatus;

        public ClientStreamListenerImpl(ClientCall.Listener<RespT> observer) {
            this.observer = (ClientCall.Listener)Preconditions.checkNotNull(observer, (Object)"observer");
        }

        private void exceptionThrown(Status status) {
            this.exceptionStatus = status;
            ClientCallImpl.this.stream.cancel(status);
        }

        @Override
        public void headersRead(final Metadata headers) {
            PerfMark.startTask((String)"ClientStreamListener.headersRead", (Tag)ClientCallImpl.this.tag);
            final Link link = PerfMark.linkOut();
            try {
                final class HeadersRead
                extends ContextRunnable {
                    HeadersRead() {
                        super(ClientCallImpl.this.context);
                    }

                    @Override
                    public void runInContext() {
                        PerfMark.startTask((String)"ClientCall$Listener.headersRead", (Tag)ClientCallImpl.this.tag);
                        PerfMark.linkIn((Link)link);
                        try {
                            this.runInternal();
                        }
                        finally {
                            PerfMark.stopTask((String)"ClientCall$Listener.headersRead", (Tag)ClientCallImpl.this.tag);
                        }
                    }

                    private void runInternal() {
                        if (ClientStreamListenerImpl.this.exceptionStatus != null) {
                            return;
                        }
                        try {
                            ClientStreamListenerImpl.this.observer.onHeaders(headers);
                        }
                        catch (Throwable t) {
                            ClientStreamListenerImpl.this.exceptionThrown(Status.CANCELLED.withCause(t).withDescription("Failed to read headers"));
                        }
                    }
                }
                ClientCallImpl.this.callExecutor.execute(new HeadersRead());
            }
            finally {
                PerfMark.stopTask((String)"ClientStreamListener.headersRead", (Tag)ClientCallImpl.this.tag);
            }
        }

        @Override
        public void messagesAvailable(final StreamListener.MessageProducer producer) {
            PerfMark.startTask((String)"ClientStreamListener.messagesAvailable", (Tag)ClientCallImpl.this.tag);
            final Link link = PerfMark.linkOut();
            try {
                final class MessagesAvailable
                extends ContextRunnable {
                    MessagesAvailable() {
                        super(ClientCallImpl.this.context);
                    }

                    @Override
                    public void runInContext() {
                        PerfMark.startTask((String)"ClientCall$Listener.messagesAvailable", (Tag)ClientCallImpl.this.tag);
                        PerfMark.linkIn((Link)link);
                        try {
                            this.runInternal();
                        }
                        finally {
                            PerfMark.stopTask((String)"ClientCall$Listener.messagesAvailable", (Tag)ClientCallImpl.this.tag);
                        }
                    }

                    private void runInternal() {
                        if (ClientStreamListenerImpl.this.exceptionStatus != null) {
                            GrpcUtil.closeQuietly(producer);
                            return;
                        }
                        try {
                            InputStream message;
                            while ((message = producer.next()) != null) {
                                try {
                                    ClientStreamListenerImpl.this.observer.onMessage(ClientCallImpl.this.method.parseResponse(message));
                                }
                                catch (Throwable t) {
                                    GrpcUtil.closeQuietly(message);
                                    throw t;
                                }
                                message.close();
                            }
                        }
                        catch (Throwable t) {
                            GrpcUtil.closeQuietly(producer);
                            ClientStreamListenerImpl.this.exceptionThrown(Status.CANCELLED.withCause(t).withDescription("Failed to read message."));
                        }
                    }
                }
                ClientCallImpl.this.callExecutor.execute(new MessagesAvailable());
            }
            finally {
                PerfMark.stopTask((String)"ClientStreamListener.messagesAvailable", (Tag)ClientCallImpl.this.tag);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void closed(Status status, ClientStreamListener.RpcProgress rpcProgress, Metadata trailers) {
            PerfMark.startTask((String)"ClientStreamListener.closed", (Tag)ClientCallImpl.this.tag);
            try {
                this.closedInternal(status, rpcProgress, trailers);
            }
            finally {
                PerfMark.stopTask((String)"ClientStreamListener.closed", (Tag)ClientCallImpl.this.tag);
            }
        }

        private void closedInternal(Status status, ClientStreamListener.RpcProgress rpcProgress, Metadata trailers) {
            Deadline deadline = ClientCallImpl.this.effectiveDeadline();
            if (status.getCode() == Status.Code.CANCELLED && deadline != null && deadline.isExpired()) {
                InsightBuilder insight = new InsightBuilder();
                ClientCallImpl.this.stream.appendTimeoutInsight(insight);
                status = Status.DEADLINE_EXCEEDED.augmentDescription("ClientCall was cancelled at or after deadline. " + insight);
                trailers = new Metadata();
            }
            final Status savedStatus = status;
            final Metadata savedTrailers = trailers;
            final Link link = PerfMark.linkOut();
            final class StreamClosed
            extends ContextRunnable {
                StreamClosed() {
                    super(ClientCallImpl.this.context);
                }

                @Override
                public void runInContext() {
                    PerfMark.startTask((String)"ClientCall$Listener.onClose", (Tag)ClientCallImpl.this.tag);
                    PerfMark.linkIn((Link)link);
                    try {
                        this.runInternal();
                    }
                    finally {
                        PerfMark.stopTask((String)"ClientCall$Listener.onClose", (Tag)ClientCallImpl.this.tag);
                    }
                }

                private void runInternal() {
                    Status status = savedStatus;
                    Metadata trailers = savedTrailers;
                    if (ClientStreamListenerImpl.this.exceptionStatus != null) {
                        status = ClientStreamListenerImpl.this.exceptionStatus;
                        trailers = new Metadata();
                    }
                    ClientCallImpl.this.cancelListenersShouldBeRemoved = true;
                    try {
                        ClientCallImpl.this.closeObserver(ClientStreamListenerImpl.this.observer, status, trailers);
                    }
                    finally {
                        ClientCallImpl.this.removeContextListenerAndCancelDeadlineFuture();
                        ClientCallImpl.this.channelCallsTracer.reportCallEnded(status.isOk());
                    }
                }
            }
            ClientCallImpl.this.callExecutor.execute(new StreamClosed());
        }

        @Override
        public void onReady() {
            if (ClientCallImpl.this.method.getType().clientSendsOneMessage()) {
                return;
            }
            PerfMark.startTask((String)"ClientStreamListener.onReady", (Tag)ClientCallImpl.this.tag);
            final Link link = PerfMark.linkOut();
            try {
                final class StreamOnReady
                extends ContextRunnable {
                    StreamOnReady() {
                        super(ClientCallImpl.this.context);
                    }

                    @Override
                    public void runInContext() {
                        PerfMark.startTask((String)"ClientCall$Listener.onReady", (Tag)ClientCallImpl.this.tag);
                        PerfMark.linkIn((Link)link);
                        try {
                            this.runInternal();
                        }
                        finally {
                            PerfMark.stopTask((String)"ClientCall$Listener.onReady", (Tag)ClientCallImpl.this.tag);
                        }
                    }

                    private void runInternal() {
                        if (ClientStreamListenerImpl.this.exceptionStatus != null) {
                            return;
                        }
                        try {
                            ClientStreamListenerImpl.this.observer.onReady();
                        }
                        catch (Throwable t) {
                            ClientStreamListenerImpl.this.exceptionThrown(Status.CANCELLED.withCause(t).withDescription("Failed to call onReady."));
                        }
                    }
                }
                ClientCallImpl.this.callExecutor.execute(new StreamOnReady());
            }
            finally {
                PerfMark.stopTask((String)"ClientStreamListener.onReady", (Tag)ClientCallImpl.this.tag);
            }
        }
    }

    private class DeadlineTimer
    implements Runnable {
        private final long remainingNanos;

        DeadlineTimer(long remainingNanos) {
            this.remainingNanos = remainingNanos;
        }

        @Override
        public void run() {
            InsightBuilder insight = new InsightBuilder();
            ClientCallImpl.this.stream.appendTimeoutInsight(insight);
            long seconds = Math.abs(this.remainingNanos) / TimeUnit.SECONDS.toNanos(1L);
            long nanos = Math.abs(this.remainingNanos) % TimeUnit.SECONDS.toNanos(1L);
            StringBuilder buf = new StringBuilder();
            buf.append("deadline exceeded after ");
            if (this.remainingNanos < 0L) {
                buf.append('-');
            }
            buf.append(seconds);
            buf.append(String.format(Locale.US, ".%09d", nanos));
            buf.append("s. ");
            buf.append(insight);
            ClientCallImpl.this.stream.cancel(Status.DEADLINE_EXCEEDED.augmentDescription(buf.toString()));
        }
    }

    static interface ClientStreamProvider {
        public ClientStream newStream(MethodDescriptor<?, ?> var1, CallOptions var2, Metadata var3, Context var4);
    }

    private final class ContextCancellationListener
    implements Context.CancellationListener {
        private ContextCancellationListener() {
        }

        public void cancelled(Context context) {
            ClientCallImpl.this.stream.cancel(Contexts.statusFromCancelled((Context)context));
        }
    }
}

