/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2023 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wildfly.glow.cli.commands;

import java.nio.file.Path;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.TreeSet;
import org.jboss.galleon.api.config.GalleonFeaturePackConfig;
import org.jboss.galleon.api.config.GalleonProvisioningConfig;
import org.jboss.galleon.universe.FeaturePackLocation;
import org.wildfly.glow.Arguments;
import org.wildfly.glow.FeaturePacks;
import org.wildfly.glow.Layer;
import org.wildfly.glow.LayerMapping;
import org.wildfly.glow.deployment.openshift.api.Deployer;

import picocli.CommandLine;

@CommandLine.Command(
        name = Constants.SHOW_CONFIGURATION_COMMAND,
        sortOptions = true
)
public class ShowConfigurationCommand extends AbstractCommand {

    @CommandLine.Option(names = {Constants.CLOUD_OPTION_SHORT, Constants.CLOUD_OPTION})
    Optional<Boolean> cloud;

    @CommandLine.Option(names = {Constants.WILDFLY_PREVIEW_OPTION_SHORT, Constants.WILDFLY_PREVIEW_OPTION})
    Optional<Boolean> wildflyPreview;

    @CommandLine.Option(names = {Constants.SERVER_VERSION_OPTION_SHORT, Constants.SERVER_VERSION_OPTION}, paramLabel = Constants.SERVER_VERSION_OPTION_LABEL)
    Optional<String> wildflyServerVersion;

    @CommandLine.Option(names = Constants.INPUT_FEATURE_PACKS_FILE_OPTION, paramLabel = Constants.INPUT_FEATURE_PACKS_FILE_OPTION_LABEL)
    Optional<Path> provisioningXml;

    @Override
    public Integer call() throws Exception {
        print("Wildfly Glow is retrieving known provisioning configuration...");
        StringBuilder ocBuilder = new StringBuilder();
        ocBuilder.append("\nDeployers enabled when provisioning to OpenShift:\n");
        for (Deployer d : ServiceLoader.load(Deployer.class)) {
            ocBuilder.append("* " + d.getName() + ". Enabled when one of the following ");
            if (!d.getSupportedLayers().isEmpty()) {
                ocBuilder.append("layer(s) " + d.getSupportedLayers() + " is/are discovered.\n");
            } else {
                ocBuilder.append("add-on(s) " + d.getSupportedAddOns() + " is/are enabled.\n");
            }
        }
        print(ocBuilder.toString());

        String context = Arguments.BARE_METAL_EXECUTION_CONTEXT;
        if (cloud.orElse(false)) {
            context = Arguments.CLOUD_EXECUTION_CONTEXT;
        }
        String finalContext = context;
        boolean isLatest = wildflyServerVersion.isEmpty();
        String vers = wildflyServerVersion.isPresent() ? wildflyServerVersion.get() : FeaturePacks.getLatestVersion();
        CommandsUtils.ProvisioningConsumer consumer = new CommandsUtils.ProvisioningConsumer() {
            @Override
            public void consume(GalleonProvisioningConfig provisioning, Map<String, Layer> all,
                    LayerMapping mapping, Map<FeaturePackLocation.FPID, Set<FeaturePackLocation.ProducerSpec>> fpDependencies) throws Exception {
                String configStr = dumpConfiguration(fpDependencies, finalContext, vers, all,
                        mapping, provisioning, isLatest, wildflyPreview.orElse(false));
                print(configStr);
            }
        };
        CommandsUtils.buildProvisioning(consumer, context, provisioningXml.orElse(null), wildflyServerVersion.isEmpty(), context, wildflyPreview.orElse(false));

        return 0;
    }

    private static String dumpConfiguration(Map<FeaturePackLocation.FPID, Set<FeaturePackLocation.ProducerSpec>> fpDependencies,
            String context, String serverVersion, Map<String, Layer> allLayers,
            LayerMapping mapping, GalleonProvisioningConfig config, boolean isLatest, boolean techPreview) throws Exception {
        StringBuilder builder = new StringBuilder();
        builder.append("Execution context: ").append(context).append("\n");
        builder.append("Server version: ").append(serverVersion).append(isLatest ? " (latest)" : "").append("\n");
        builder.append("Tech Preview: ").append(techPreview).append("\n");
        Set<FeaturePackLocation.ProducerSpec> topLevel = new LinkedHashSet<>();
        for(GalleonFeaturePackConfig fp : config.getFeaturePackDeps()) {
            topLevel.add(fp.getLocation().getProducer());
        }
        for(GalleonFeaturePackConfig fp : config.getFeaturePackDeps()) {
            builder.append("\nFeature-pack: ").append("@|bold ").append(fp.getLocation().getFPID()).append("|@\n");
            builder.append("Contained layers: ");
            Set<String> layers = new TreeSet<>();
            Set<FeaturePackLocation.ProducerSpec> deps = fpDependencies.get(fp.getLocation().getFPID());
            for(Layer l : allLayers.values()) {
                if(l.getFeaturePacks().contains(fp.getLocation().getFPID())) {
                    layers.add(l.getName());
                }
                if(deps != null) {
                    for (FeaturePackLocation.ProducerSpec dep : deps) {
                        if (!topLevel.contains(dep)) {
                            for (FeaturePackLocation.FPID fpid : l.getFeaturePacks()) {
                                if (fpid.getProducer().equals(dep)) {
                                    layers.add(l.getName());
                                }
                            }
                        }
                    }
                }
            }
            topLevel.addAll(deps);
            builder.append(layers).append("\n");
        }
        return builder.toString();
    }
}
