/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.prospero.actions;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.wildfly.prospero.ProsperoLogger;
import org.wildfly.prospero.actions.GalleonHashesFileWalker;

class ApplyStageBackup
implements AutoCloseable {
    protected static final String BACKUP_FOLDER = ".update.old";
    private final Path backupRoot;
    private final Path serverRoot;
    private final Path candidateRoot;

    ApplyStageBackup(Path serverRoot, Path candidateRoot) throws IOException {
        this.serverRoot = serverRoot;
        this.candidateRoot = candidateRoot;
        this.backupRoot = serverRoot.resolve(BACKUP_FOLDER);
        if (ProsperoLogger.ROOT_LOGGER.isDebugEnabled()) {
            ProsperoLogger.ROOT_LOGGER.debug("Creating backup record in " + this.backupRoot);
        }
        if (!Files.exists(this.backupRoot, new LinkOption[0])) {
            if (ProsperoLogger.ROOT_LOGGER.isTraceEnabled()) {
                ProsperoLogger.ROOT_LOGGER.trace("Creating backup directory in " + this.backupRoot);
            }
            Files.createDirectories(this.backupRoot, new FileAttribute[0]);
        } else if (!Files.isDirectory(this.backupRoot, new LinkOption[0]) || !Files.isWritable(this.backupRoot)) {
            throw new RuntimeException(String.format("Unable to create backup in %s. It is not a directory or is not writable.", this.backupRoot));
        }
        File[] files = this.backupRoot.toFile().listFiles();
        if (files != null) {
            for (File file : files) {
                if (ProsperoLogger.ROOT_LOGGER.isTraceEnabled()) {
                    ProsperoLogger.ROOT_LOGGER.trace("Removing existing backup files: " + file);
                }
                FileUtils.forceDelete(file);
            }
        }
    }

    public void recordAll() throws IOException {
        ProsperoLogger.ROOT_LOGGER.debug("Starting building the update backup.");
        Path hashesRoot = this.serverRoot.resolve(".galleon").resolve("hashes");
        if (!Files.exists(hashesRoot, new LinkOption[0])) {
            ProsperoLogger.ROOT_LOGGER.warn("Unable to perform the backup: No Galleon Hashes record found.");
            return;
        }
        GalleonHashesFileWalker serverFS = new GalleonHashesFileWalker(this.serverRoot){

            @Override
            void visitFile(Path file) throws IOException {
                Path serverPath = ApplyStageBackup.this.serverRoot.resolve(file);
                Path backupPath = ApplyStageBackup.this.backupRoot.resolve(file);
                if (!Files.exists(serverPath, new LinkOption[0])) {
                    ProsperoLogger.ROOT_LOGGER.debug("Unable to find managed file: " + serverPath + ". File backup skipped.");
                } else {
                    ApplyStageBackup.backupFile(serverPath, backupPath);
                }
            }

            @Override
            void visitDirectory(Path relative) throws IOException {
                ProsperoLogger.ROOT_LOGGER.tracef("Creating a directory in the backup folder: %s", (Object)relative);
                Files.createDirectories(ApplyStageBackup.this.backupRoot.resolve(relative), new FileAttribute[0]);
            }
        };
        serverFS.walk();
        ProsperoLogger.ROOT_LOGGER.trace("Checking candidate folder for overwriting files.");
        GalleonHashesFileWalker candidateFS = new GalleonHashesFileWalker(this.candidateRoot){

            @Override
            void visitFile(Path file) throws IOException {
                Path serverFile = ApplyStageBackup.this.serverRoot.resolve(file);
                Path backupFile = ApplyStageBackup.this.backupRoot.resolve(file);
                if (Files.exists(serverFile, new LinkOption[0]) && !Files.exists(backupFile, new LinkOption[0])) {
                    if (!Files.isReadable(serverFile) || !Files.isWritable(serverFile)) {
                        throw new RuntimeException("The update is unable to modify file " + serverFile + " due to invalid file permissions.");
                    }
                    if (!Files.exists(backupFile.getParent(), new LinkOption[0])) {
                        ProsperoLogger.ROOT_LOGGER.tracef("Creating added directory based on the candidate folder:%s.", (Object)backupFile);
                        Files.createDirectories(backupFile.getParent(), new FileAttribute[0]);
                    }
                    ApplyStageBackup.backupFile(serverFile, backupFile);
                }
            }

            @Override
            void visitDirectory(Path dir) throws IOException {
            }
        };
        candidateFS.walk();
        if (Files.exists(this.serverRoot.resolve(".galleon"), new LinkOption[0])) {
            ProsperoLogger.ROOT_LOGGER.trace("Copying the Galleon provisioned state directory.");
            FileUtils.copyDirectory(this.serverRoot.resolve(".galleon").toFile(), this.backupRoot.resolve(".galleon").toFile());
        }
        if (Files.exists(this.serverRoot.resolve(".installation"), new LinkOption[0])) {
            ProsperoLogger.ROOT_LOGGER.trace("Copying the Prospero installation directory.");
            FileUtils.copyDirectory(this.serverRoot.resolve(".installation").toFile(), this.backupRoot.resolve(".installation").toFile());
        }
        ProsperoLogger.ROOT_LOGGER.debug("Finished building the update backup.");
    }

    private static void backupFile(Path serverPath, Path backupPath) throws IOException {
        ProsperoLogger.ROOT_LOGGER.tracef("Backing up file %s to %s.", (Object)serverPath, (Object)backupPath);
        try {
            Files.createLink(backupPath, serverPath);
        }
        catch (UnsupportedOperationException e) {
            Files.copy(serverPath, backupPath, new CopyOption[0]);
        }
    }

    @Override
    public void close() {
        FileUtils.deleteQuietly(this.backupRoot.toFile());
    }

    public void restore() throws IOException {
        if (this.backupRoot.toFile().listFiles() == null || this.backupRoot.toFile().listFiles().length == 0) {
            return;
        }
        if (ProsperoLogger.ROOT_LOGGER.isDebugEnabled()) {
            ProsperoLogger.ROOT_LOGGER.debug("Restoring server from the backup.");
        }
        if (!Files.exists(this.backupRoot, new LinkOption[0])) {
            throw new RuntimeException("Backup root doesn't exist.");
        }
        Files.walkFileTree(this.backupRoot, this.restoreModifiedFiles());
        Files.walkFileTree(this.serverRoot, this.deleteNewFiles());
    }

    private SimpleFileVisitor<Path> deleteNewFiles() {
        return new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                return ApplyStageBackup.this.ignoreUserManagedFiles(file, exc);
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (ApplyStageBackup.this.isUserProtectedFile(file)) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                Path relativePath = ApplyStageBackup.this.serverRoot.relativize(file);
                if (!Files.exists(ApplyStageBackup.this.backupRoot.resolve(relativePath), new LinkOption[0]) && Files.exists(ApplyStageBackup.this.candidateRoot.resolve(relativePath), new LinkOption[0])) {
                    if (ProsperoLogger.ROOT_LOGGER.isTraceEnabled()) {
                        ProsperoLogger.ROOT_LOGGER.trace("Removing added file " + relativePath);
                    }
                    Files.delete(file);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                Path relativePath = ApplyStageBackup.this.serverRoot.relativize(dir);
                if (!Files.exists(ApplyStageBackup.this.backupRoot.resolve(relativePath), new LinkOption[0]) && Files.exists(ApplyStageBackup.this.candidateRoot.resolve(relativePath), new LinkOption[0])) {
                    if (ProsperoLogger.ROOT_LOGGER.isTraceEnabled()) {
                        ProsperoLogger.ROOT_LOGGER.trace("Removing added directory " + relativePath);
                    }
                    Files.delete(dir);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (dir.equals(ApplyStageBackup.this.backupRoot) || dir.equals(ApplyStageBackup.this.serverRoot.resolve(".galleon")) || dir.equals(ApplyStageBackup.this.serverRoot.resolve(".installation"))) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                return FileVisitResult.CONTINUE;
            }
        };
    }

    private FileVisitResult ignoreUserManagedFiles(Path file, IOException exc) throws IOException {
        if (this.isUserProtectedFile(file)) {
            return FileVisitResult.SKIP_SUBTREE;
        }
        throw exc;
    }

    private boolean isUserProtectedFile(Path file) {
        Path relative = this.serverRoot.relativize(file);
        Path candidatePath = this.candidateRoot.resolve(relative);
        return !Files.isReadable(file) && !Files.exists(candidatePath, new LinkOption[0]);
    }

    private SimpleFileVisitor<Path> restoreModifiedFiles() {
        return new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Path targetFile;
                Path relativePath = ApplyStageBackup.this.backupRoot.relativize(file);
                Path parentDir = relativePath.getParent();
                if (parentDir != null && !Files.exists(ApplyStageBackup.this.serverRoot.resolve(parentDir), new LinkOption[0])) {
                    if (ProsperoLogger.ROOT_LOGGER.isTraceEnabled()) {
                        ProsperoLogger.ROOT_LOGGER.trace("Recreating removed directory " + parentDir);
                    }
                    Files.createDirectories(ApplyStageBackup.this.serverRoot.resolve(parentDir), new FileAttribute[0]);
                }
                if (ApplyStageBackup.fileChanged(file, targetFile = ApplyStageBackup.this.serverRoot.resolve(relativePath))) {
                    if (ProsperoLogger.ROOT_LOGGER.isTraceEnabled()) {
                        ProsperoLogger.ROOT_LOGGER.trace("Restoring changed file " + relativePath);
                    }
                    Files.copy(file, targetFile, StandardCopyOption.REPLACE_EXISTING);
                }
                return FileVisitResult.CONTINUE;
            }
        };
    }

    private static boolean fileChanged(Path file, Path targetFile) throws IOException {
        if (!Files.exists(targetFile, new LinkOption[0])) {
            return true;
        }
        try (FileInputStream fis1 = new FileInputStream(targetFile.toFile());){
            boolean bl;
            try (FileInputStream fis2 = new FileInputStream(file.toFile());){
                bl = !IOUtils.contentEquals(fis1, fis2);
            }
            return bl;
        }
    }
}

