/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.prospero.licenses;

import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Properties;
import java.util.stream.Collectors;
import org.jboss.logging.Logger;
import org.wildfly.prospero.licenses.License;

public class LicenseManager {
    private static final Logger logger = Logger.getLogger(LicenseManager.class);
    public static final String LICENSES_FOLDER = "licenses";
    protected static final String LICENSE_DEFINITION_NAME = "licenses";
    private static final String LICENSE_DEFINITION_EXTENSION = ".yaml";
    private static final String DEFAULT_LICENSE_DEFINITION = "licenses.yaml";
    protected static final String LICENSE_AGREEMENT_FILENAME = "license_accepted.properties";
    private final HashMap<String, List<License>> nameMap = new HashMap();

    public LicenseManager() {
        this(LicenseManager.getLicensesFile());
    }

    LicenseManager(URL licensesUrl) {
        if (licensesUrl == null) {
            logger.debug("No known repositories found");
        } else {
            logger.debug("Loading known provisioning configurations from: " + String.valueOf(licensesUrl));
            try {
                List<License> knownFeaturePacks = License.readLicenses(licensesUrl);
                for (License fp : knownFeaturePacks) {
                    if (!this.nameMap.containsKey(fp.getFpGav())) {
                        this.nameMap.put(fp.getFpGav(), new ArrayList());
                    }
                    this.nameMap.get(fp.getFpGav()).add(fp);
                }
            }
            catch (IOException e) {
                logger.warn("Failed to load licenses configurations from: " + String.valueOf(licensesUrl));
                logger.debug((Object)"Error parsing provisioning configurations:", e);
            }
        }
    }

    public List<License> getLicenses(List<String> fpls) {
        Objects.requireNonNull(fpls);
        return fpls.stream().filter(this.nameMap::containsKey).map(this.nameMap::get).flatMap(Collection::stream).collect(Collectors.toList());
    }

    public void recordAgreements(List<License> licenses, Path targetServer) throws IOException {
        Objects.requireNonNull(licenses);
        Objects.requireNonNull(targetServer);
        if (licenses.isEmpty()) {
            if (logger.isDebugEnabled()) {
                logger.debug("No licenses to save, skipping.");
            }
            return;
        }
        if (logger.isDebugEnabled()) {
            logger.debugf("Saving %d license agreements.", licenses.size());
        }
        String username = System.getProperty("user.name");
        LocalDateTime timestamp = LocalDateTime.now();
        Path licenseFolder = targetServer.resolve(".installation").resolve("licenses");
        if (!Files.exists(licenseFolder, new LinkOption[0])) {
            if (logger.isTraceEnabled()) {
                logger.trace("Creating license Folder " + String.valueOf(licenseFolder));
            }
            Files.createDirectory(licenseFolder, new FileAttribute[0]);
        }
        Path licenseAcceptFile = licenseFolder.resolve(LICENSE_AGREEMENT_FILENAME);
        try (FileOutputStream fos = new FileOutputStream(licenseAcceptFile.toFile());){
            Properties licenseApproveProperties = new Properties();
            licenseApproveProperties.setProperty("username", username);
            licenseApproveProperties.setProperty("timestamp", timestamp.toString());
            for (int i = 0; i < licenses.size(); ++i) {
                License license = licenses.get(i);
                LicenseManager.saveLicenseText(license, licenseFolder);
                licenseApproveProperties.setProperty("license." + i + ".name", license.getName());
                licenseApproveProperties.setProperty("license." + i + ".file", LicenseManager.toFileName(license));
            }
            if (logger.isTraceEnabled()) {
                logger.trace("Storing license agreement " + String.valueOf(licenseApproveProperties));
            }
            licenseApproveProperties.store(fos, "Agreements accepted during installation");
        }
    }

    private static URL getLicensesFile() {
        Locale locale = Locale.getDefault();
        URL licensesUrl = LicenseManager.class.getClassLoader().getResource("licenses_" + locale.getLanguage() + LICENSE_DEFINITION_EXTENSION);
        if (licensesUrl == null) {
            return LicenseManager.class.getClassLoader().getResource(DEFAULT_LICENSE_DEFINITION);
        }
        return licensesUrl;
    }

    private static void saveLicenseText(License license, Path licensesFolder) throws IOException {
        Path licenseFilePath = licensesFolder.resolve(LicenseManager.toFileName(license));
        if (logger.isTraceEnabled()) {
            logger.trace("Storing license text to: " + String.valueOf(licenseFilePath));
        }
        Files.writeString(licenseFilePath, (CharSequence)license.getText(), StandardOpenOption.CREATE);
    }

    private static String toFileName(License license) {
        return license.getName().toLowerCase(Locale.getDefault()).replace(' ', '-') + ".txt";
    }
}

