/*
 * Copyright 2022 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wildfly.prospero.api;

import org.apache.commons.lang3.StringUtils;
import org.jboss.galleon.Constants;
import org.jboss.galleon.config.ProvisioningConfig;
import org.wildfly.channel.Channel;
import org.wildfly.channel.ChannelManifest;
import org.apache.commons.io.FileUtils;
import org.wildfly.prospero.Messages;
import org.wildfly.prospero.model.ManifestVersionRecord;
import org.wildfly.prospero.api.exceptions.MetadataException;
import org.wildfly.prospero.installation.git.GitStorage;
import org.wildfly.prospero.metadata.ProsperoMetadataUtils;
import org.wildfly.prospero.model.ManifestYamlSupport;
import org.wildfly.prospero.model.ProsperoConfig;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.jboss.galleon.ProvisioningException;
import org.jboss.galleon.xml.ProvisioningXmlParser;
import org.wildfly.channel.Stream;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import static org.wildfly.prospero.metadata.ProsperoMetadataUtils.CURRENT_VERSION_FILE;

public class InstallationMetadata implements AutoCloseable {

    @Deprecated
    public static final String METADATA_DIR = ProsperoMetadataUtils.METADATA_DIR;
    @Deprecated
    public static final String MANIFEST_FILE_NAME = ProsperoMetadataUtils.MANIFEST_FILE_NAME;
    @Deprecated
    public static final String INSTALLER_CHANNELS_FILE_NAME = ProsperoMetadataUtils.INSTALLER_CHANNELS_FILE_NAME;
    public static final String PROVISIONING_FILE_NAME = "provisioning.xml";
    public static final String GALLEON_INSTALLATION_DIR = ".galleon";
    public static final String README_FILE_NAME = "README.txt";
    private static final String WARNING_MESSAGE = "WARNING: The files in .installation directory should be only edited by the provisioning tool.";
    private final Path manifestFile;
    private final Path channelsFile;
    private final Path readmeFile;
    private final Path provisioningFile;
    private final ProvisioningConfig galleonProvisioningConfig;
    private final GitStorage gitStorage;
    private final Path base;
    private final Optional<ManifestVersionRecord> manifestVersion;
    private ProsperoConfig prosperoConfig;
    private ChannelManifest manifest;

    /**
     * load the metadata of an existing installation. If the history is not available, it will be started.
     *
     * @param base
     * @return
     * @throws MetadataException
     */
    public static InstallationMetadata loadInstallation(Path base) throws MetadataException {
        final Path manifestFile = base.resolve(METADATA_DIR).resolve(InstallationMetadata.MANIFEST_FILE_NAME);

        ChannelManifest manifest;
        ProsperoConfig prosperoConfig;
        Optional<ManifestVersionRecord> currentVersion = Optional.empty();

        try {
            manifest = ManifestYamlSupport.parse(manifestFile.toFile());
        } catch (IOException e) {
            throw Messages.MESSAGES.unableToParseConfiguration(manifestFile, e);
        }
        try {
            prosperoConfig = ProsperoConfig.readConfig(base.resolve(METADATA_DIR));
        } catch (MetadataException e) {
            // re-wrap the exception to change the description
            throw Messages.MESSAGES.unableToParseConfiguration(base, e.getCause());
        }

        currentVersion = ManifestVersionRecord.read(base.resolve(ProsperoMetadataUtils.METADATA_DIR).resolve(CURRENT_VERSION_FILE));

        final GitStorage gitStorage = new GitStorage(base);
        final InstallationMetadata metadata = new InstallationMetadata(base, manifest, prosperoConfig, gitStorage, currentVersion);
        try {
            if (!gitStorage.isStarted()) {
                gitStorage.record();
            }
        } catch (IOException e) {
            throw Messages.MESSAGES.unableToCreateHistoryStorage(base.resolve(METADATA_DIR), e);
        }
        return metadata;
    }

    /**
     * create an in-memory installation metadata. No information is recorded until {@link InstallationMetadata#recordProvision(boolean)}
     * is called.
     *
     * @param base
     * @param manifest
     * @param prosperoConfig
     * @param currentVersions - manifest versions used to provision the installation
     * @return
     * @throws MetadataException
     */
    public static InstallationMetadata newInstallation(Path base, ChannelManifest manifest, ProsperoConfig prosperoConfig,
                                                       Optional<ManifestVersionRecord> currentVersions) throws MetadataException {
        return new InstallationMetadata(base, manifest, prosperoConfig, new GitStorage(base), currentVersions);
    }

    /**
     * read the metadata from an exported zip containing configuration files
     *
     * @param archiveLocation path to the exported zip
     * @return
     * @throws IOException
     * @throws MetadataException
     */
    public static InstallationMetadata fromMetadataBundle(Path archiveLocation) throws IOException, MetadataException {

        final Path tempDirectory = Files.createTempDirectory("installer-import");
        tempDirectory.toFile().deleteOnExit();
        try (ZipInputStream zis = new ZipInputStream(new FileInputStream(archiveLocation.toFile()))) {
            Path manifestFile = null;
            Path channelsFile = null;
            Path provisioningFile = null;
            ZipEntry entry;
            Files.createDirectory(tempDirectory.resolve(ProsperoMetadataUtils.METADATA_DIR));
            Files.createDirectory(tempDirectory.resolve(Constants.PROVISIONED_STATE_DIR));
            while ((entry = zis.getNextEntry()) != null) {

                if (entry.getName().equals(MANIFEST_FILE_NAME)) {
                    manifestFile = tempDirectory.resolve(ProsperoMetadataUtils.METADATA_DIR).resolve(ProsperoMetadataUtils.MANIFEST_FILE_NAME);
                    Files.copy(zis, manifestFile, StandardCopyOption.REPLACE_EXISTING);
                    manifestFile.toFile().deleteOnExit();
                }

                if (entry.getName().equals(INSTALLER_CHANNELS_FILE_NAME)) {
                    channelsFile = tempDirectory.resolve(ProsperoMetadataUtils.METADATA_DIR).resolve(ProsperoMetadataUtils.INSTALLER_CHANNELS_FILE_NAME);
                    Files.copy(zis, channelsFile, StandardCopyOption.REPLACE_EXISTING);
                    channelsFile.toFile().deleteOnExit();
                }

                if (entry.getName().equals(PROVISIONING_FILE_NAME)) {
                    provisioningFile = tempDirectory.resolve(Constants.PROVISIONED_STATE_DIR).resolve(Constants.PROVISIONING_XML);
                    Files.copy(zis, provisioningFile, StandardCopyOption.REPLACE_EXISTING);
                    provisioningFile.toFile().deleteOnExit();
                }
            }

            if (manifestFile == null || channelsFile == null || provisioningFile == null) {
                throw Messages.MESSAGES.incompleteMetadataBundle(archiveLocation);
            }
        }

        return InstallationMetadata.loadInstallation(tempDirectory);
    }

    protected InstallationMetadata(Path base, ChannelManifest manifest, ProsperoConfig prosperoConfig,
                                   GitStorage gitStorage, Optional<ManifestVersionRecord> currentVersions) throws MetadataException {
        this.base = base;
        this.gitStorage = gitStorage;
        this.manifestFile = base.resolve(METADATA_DIR).resolve(InstallationMetadata.MANIFEST_FILE_NAME);
        this.channelsFile = base.resolve(METADATA_DIR).resolve(InstallationMetadata.INSTALLER_CHANNELS_FILE_NAME);
        this.readmeFile = base.resolve(METADATA_DIR).resolve(InstallationMetadata.README_FILE_NAME);
        this.provisioningFile = base.resolve(GALLEON_INSTALLATION_DIR).resolve(InstallationMetadata.PROVISIONING_FILE_NAME);

        this.manifest = manifest;
        this.prosperoConfig = new ProsperoConfig(new ArrayList<>(prosperoConfig.getChannels()), prosperoConfig.getMavenOptions());

        final List<Channel> channels = prosperoConfig.getChannels();
        if (channels != null && channels.stream().filter(c-> StringUtils.isEmpty(c.getName())).findAny().isPresent()) {
            throw Messages.MESSAGES.emptyChannelName();
        }

        try {
            this.galleonProvisioningConfig = ProvisioningXmlParser.parse(provisioningFile);
        } catch (ProvisioningException e) {
            throw Messages.MESSAGES.unableToParseConfiguration(provisioningFile, e);
        }

        this.manifestVersion = currentVersions;
    }

    public Path exportMetadataBundle(Path location) throws IOException {
        final File file = location.toFile();

        try (ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(file))) {
            zos.putNextEntry(new ZipEntry(MANIFEST_FILE_NAME));
            try(FileInputStream fis = new FileInputStream(manifestFile.toFile())) {
                byte[] buffer = new byte[1024];
                int len;
                while ((len = fis.read(buffer)) > 0) {
                    zos.write(buffer, 0, len);
                }
            }
            zos.closeEntry();

            zos.putNextEntry(new ZipEntry(INSTALLER_CHANNELS_FILE_NAME));
            try(FileInputStream fis = new FileInputStream(channelsFile.toFile())) {
                byte[] buffer = new byte[1024];
                int len;
                while ((len = fis.read(buffer)) > 0) {
                    zos.write(buffer, 0, len);
                }
            }
            zos.closeEntry();

            zos.putNextEntry(new ZipEntry(PROVISIONING_FILE_NAME));
            try(FileInputStream fis = new FileInputStream(provisioningFile.toFile())) {
                byte[] buffer = new byte[1024];
                int len;
                while ((len = fis.read(buffer)) > 0) {
                    zos.write(buffer, 0, len);
                }
            }
            zos.closeEntry();
        }
        return file.toPath();
    }

    public ChannelManifest getManifest() {
        return manifest;
    }

    public ProvisioningConfig getGalleonProvisioningConfig() {
        return galleonProvisioningConfig;
    }

    public void recordProvision(boolean overrideProsperoConfig) throws MetadataException {
        recordProvision(overrideProsperoConfig, true);
    }

    public void recordProvision(boolean overrideProsperoConfig, boolean gitRecord) throws MetadataException {
        try {
            ManifestYamlSupport.write(this.manifest, this.manifestFile);
        } catch (IOException e) {
            throw Messages.MESSAGES.unableToSaveConfiguration(manifestFile, e);
        }
        // Add README.txt file to .installation directory to warn the files should not be edited.
        if (!Files.exists(readmeFile)) {
            try {
                FileUtils.writeStringToFile(new File(readmeFile.toString()), WARNING_MESSAGE , StandardCharsets.UTF_8);
            } catch (IOException e) {
                throw new MetadataException("Unable to create README.txt in installation", e);
            }
        }

        if (overrideProsperoConfig || !Files.exists(this.channelsFile)) {
            writeProsperoConfig();
        }

        if (manifestVersion.isPresent()) {
            ManifestVersionRecord.write(manifestVersion.get(), base.resolve(ProsperoMetadataUtils.METADATA_DIR).resolve(CURRENT_VERSION_FILE));
        }

        if (gitRecord) {
            gitStorage.record();
        }
    }

    private void writeProsperoConfig() throws MetadataException {
        try {
            getProsperoConfig().writeConfig(this.base.resolve(METADATA_DIR));
        } catch (IOException e) {
            throw Messages.MESSAGES.unableToSaveConfiguration(channelsFile, e);
        }
    }

    public List<SavedState> getRevisions() throws MetadataException {
        return gitStorage.getRevisions();
    }

    public InstallationMetadata getSavedState(SavedState savedState) throws MetadataException {
        // checkout previous version
        // record as rollback operation
        Path revert = null;
        try {
            revert = gitStorage.revert(savedState);

            // re-parse metadata
            return InstallationMetadata.loadInstallation(revert);
        } finally {
            gitStorage.reset();
            if (revert != null && Files.exists(revert)) {
                FileUtils.deleteQuietly(revert.toFile());
            }
        }
    }

    public InstallationChanges getChangesSince(SavedState savedState) throws MetadataException {
        return new InstallationChanges(gitStorage.getArtifactChanges(savedState), gitStorage.getChannelChanges(savedState));
    }

    public void setManifest(ChannelManifest resolvedChannel) {
        manifest = resolvedChannel;
    }

    public List<Artifact> getArtifacts() {
        return manifest.getStreams().stream().map(s-> streamToArtifact(s)).collect(Collectors.toList());
    }

    private DefaultArtifact streamToArtifact(Stream s) {
        return new DefaultArtifact(s.getGroupId(), s.getArtifactId(), "jar", s.getVersion());
    }

    public Artifact find(Artifact gav) {
        for (Stream stream : manifest.getStreams()) {
            if (stream.getGroupId().equals(gav.getGroupId()) && stream.getArtifactId().equals(gav.getArtifactId())) {
                return streamToArtifact(stream);
            }
        }
        return null;
    }

    public ProsperoConfig getProsperoConfig() {
        return prosperoConfig;
    }

    public void updateProsperoConfig(ProsperoConfig config) throws MetadataException {
        this.prosperoConfig = config;

        writeProsperoConfig();

        gitStorage.recordConfigChange();
    }

    public Optional<ManifestVersionRecord> getManifestVersions() throws IOException {
        return manifestVersion;
    }

    @Override
    public void close() {
        if (gitStorage != null) {
            try {
                gitStorage.close();
            } catch (Exception e) {
                // log and ignore
                Messages.MESSAGES.unableToCloseStore(e);
            }
        }
    }
}
