/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.prospero.licenses;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.io.FileUtils;
import org.jboss.logging.Logger;
import org.wildfly.prospero.licenses.License;
import org.wildfly.prospero.licenses.SortedProperties;

public class LicenseManager {
    private static final Logger logger = Logger.getLogger(LicenseManager.class);
    public static final String LICENSES_FOLDER = "licenses";
    protected static final String LICENSE_DEFINITION_NAME = "licenses";
    private static final String LICENSE_DEFINITION_EXTENSION = ".yaml";
    private static final String DEFAULT_LICENSE_DEFINITION = "licenses.yaml";
    protected static final String LICENSE_AGREEMENT_FILENAME = "license_accepted.properties";
    private final HashMap<String, List<License>> nameMap = new HashMap();

    public LicenseManager() {
        this(LicenseManager.getLicensesFile());
    }

    LicenseManager(URL licensesUrl) {
        if (licensesUrl == null) {
            logger.debug((Object)"No known repositories found");
        } else {
            logger.debug((Object)("Loading known provisioning configurations from: " + String.valueOf(licensesUrl)));
            try {
                List<License> knownFeaturePacks = License.readLicenses(licensesUrl);
                for (License fp : knownFeaturePacks) {
                    if (!this.nameMap.containsKey(fp.getFpGav())) {
                        this.nameMap.put(fp.getFpGav(), new ArrayList());
                    }
                    this.nameMap.get(fp.getFpGav()).add(fp);
                }
            }
            catch (IOException e) {
                logger.warn((Object)("Failed to load licenses configurations from: " + String.valueOf(licensesUrl)));
                logger.debug((Object)"Error parsing provisioning configurations:", (Throwable)e);
            }
        }
    }

    public List<License> getLicenses(Set<String> fpls) {
        Objects.requireNonNull(fpls);
        return fpls.stream().filter(this.nameMap::containsKey).map(this.nameMap::get).flatMap(Collection::stream).collect(Collectors.toList());
    }

    public void recordAgreements(List<License> licenses, Path targetServer) throws IOException {
        Objects.requireNonNull(licenses);
        Objects.requireNonNull(targetServer);
        if (licenses.isEmpty()) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)"No licenses to save, skipping.");
            }
            return;
        }
        if (logger.isDebugEnabled()) {
            logger.debugf("Saving %d license agreements.", licenses.size());
        }
        LocalDateTime timestamp = LocalDateTime.now();
        Path licenseFolder = targetServer.resolve(".installation").resolve("licenses");
        if (!Files.exists(licenseFolder, new LinkOption[0])) {
            if (logger.isTraceEnabled()) {
                logger.trace((Object)("Creating license Folder " + String.valueOf(licenseFolder)));
            }
            Files.createDirectory(licenseFolder, new FileAttribute[0]);
        }
        Path licenseAcceptFile = licenseFolder.resolve(LICENSE_AGREEMENT_FILENAME);
        SortedProperties licenseApproveProperties = new SortedProperties();
        if (Files.exists(licenseAcceptFile, new LinkOption[0])) {
            try (FileInputStream inStream = new FileInputStream(licenseAcceptFile.toFile());){
                licenseApproveProperties.load(inStream);
            }
        }
        Optional<Integer> first = licenseApproveProperties.keySet().stream().map(Object::toString).filter(s -> s.startsWith("license.") && s.endsWith(".name")).map(s -> s.replace("license.", "").replace(".name", "")).map(Integer::parseInt).max(Integer::compareTo).map(i -> i + 1);
        try (FileOutputStream fos = new FileOutputStream(licenseAcceptFile.toFile());){
            int nextLicenseIndex = first.orElse(0);
            for (License license : licenses) {
                LicenseManager.saveLicenseText(license, licenseFolder);
                licenseApproveProperties.setProperty("license." + nextLicenseIndex + ".name", license.getName());
                licenseApproveProperties.setProperty("license." + nextLicenseIndex + ".file", LicenseManager.toFileName(license));
                licenseApproveProperties.setProperty("license." + nextLicenseIndex + ".timestamp", timestamp.toString());
                ++nextLicenseIndex;
            }
            if (logger.isTraceEnabled()) {
                logger.trace((Object)("Storing license agreement " + String.valueOf(licenseApproveProperties)));
            }
            licenseApproveProperties.store(fos, "Agreements accepted during installation");
        }
    }

    public void copyIfExists(Path sourceServer, Path targetServer) throws IOException {
        Path licencesDir = Path.of(".installation", "licenses");
        if (!Files.exists(sourceServer.resolve(licencesDir), new LinkOption[0])) {
            return;
        }
        FileUtils.copyDirectory((File)sourceServer.resolve(licencesDir).toFile(), (File)targetServer.resolve(licencesDir).toFile());
    }

    private static URL getLicensesFile() {
        Locale locale = Locale.getDefault();
        URL licensesUrl = LicenseManager.class.getClassLoader().getResource("licenses_" + locale.getLanguage() + LICENSE_DEFINITION_EXTENSION);
        if (licensesUrl == null) {
            return LicenseManager.class.getClassLoader().getResource(DEFAULT_LICENSE_DEFINITION);
        }
        return licensesUrl;
    }

    private static void saveLicenseText(License license, Path licensesFolder) throws IOException {
        Path licenseFilePath = licensesFolder.resolve(LicenseManager.toFileName(license));
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Storing license text to: " + String.valueOf(licenseFilePath)));
        }
        Files.writeString(licenseFilePath, (CharSequence)license.getText(), StandardOpenOption.CREATE);
    }

    private static String toFileName(License license) {
        return license.getName().toLowerCase(Locale.getDefault()).replace(' ', '-') + ".txt";
    }
}

