/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2015 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wildfly.security.sasl.gs2;

import static org.wildfly.security._private.ElytronMessages.log;
import static org.wildfly.security.sasl.gs2.Gs2.*;

import java.io.IOException;
import java.util.Map;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.sasl.SaslClient;
import javax.security.sasl.SaslClientFactory;
import javax.security.sasl.SaslException;

import org.ietf.jgss.GSSException;
import org.ietf.jgss.GSSManager;
import org.kohsuke.MetaInfServices;
import org.wildfly.security.auth.callback.ChannelBindingCallback;
import org.wildfly.security.sasl.WildFlySasl;

/**
 * SaslClientFactory for the GS2 mechanism family as defined by RFC 5801.
 *
 * @author <a href="mailto:fjuma@redhat.com">Farah Juma</a>
 */
@MetaInfServices(value = SaslClientFactory.class)
public final class Gs2SaslClientFactory implements SaslClientFactory {

    public Gs2SaslClientFactory() {
        super();
    }

    public SaslClient createSaslClient(final String[] mechanisms, final String authorizationId, final String protocol,
            final String serverName, final Map<String, ?> props, final CallbackHandler cbh) throws SaslException {
        String name = null;
        boolean plus = false;
        final ChannelBindingCallback callback = new ChannelBindingCallback();
        try {
            cbh.handle(new Callback[] { callback });
        } catch (SaslException e) {
            throw e;
        } catch (IOException e) {
            throw log.saslFailedToDetermineChannelBindingStatus(e);
        } catch (UnsupportedCallbackException e) {
            // Ignored
        }
        GSSManager gssManager = GSSManager.getInstance();
        final String[] supportedMechanisms;
        try {
            supportedMechanisms = Gs2Util.getSupportedSaslNamesForMechanisms(gssManager.getMechs());
        } catch (GSSException e) {
            throw log.saslGettingSupportedMechanismsFailed(e);
        }
        final String bindingType = callback.getBindingType();
        final byte[] bindingData = callback.getBindingData();
        boolean bindingOk = (bindingType != null) && (bindingData != null);
        boolean bindingRequired = "true".equals(props.get(WildFlySasl.CHANNEL_BINDING_REQUIRED));
        for (String mechanism : mechanisms) {
            if (! Gs2Util.isIncluded(mechanism, supportedMechanisms)) continue;
            if (mechanism.endsWith(PLUS_SUFFIX)) {
                if (! bindingOk) continue;
                plus = true;
            }
            if (bindingRequired && ! plus) continue;
            name = mechanism;
            break;
        }
        if (name == null) return null;

        final Gs2SaslClient client = new Gs2SaslClient(name, protocol, serverName, cbh, authorizationId, props, gssManager, plus, bindingType, bindingData);
        client.init();
        return client;
    }

    public String[] getMechanismNames(final Map<String, ?> props) {
        String[] names;
        try {
            names = Gs2Util.getSupportedSaslNamesForMechanisms(GSSManager.getInstance().getMechs());
        } catch (GSSException e) {
            return WildFlySasl.NO_NAMES;
        }
        if (!"true".equals(props.get(WildFlySasl.MECHANISM_QUERY_ALL)) && "true".equals(props.get(WildFlySasl.CHANNEL_BINDING_REQUIRED))) {
            return Gs2Util.getPlusMechanisms(names);
        } else {
            return names;
        }
    }
}
