/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.auth.provider.jdbc.mapper;

import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.sql.ResultSet;
import java.sql.SQLException;
import org.wildfly.common.Assert;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.auth.provider.jdbc.KeyMapper;
import org.wildfly.security.auth.server.SupportLevel;
import org.wildfly.security.credential.Credential;
import org.wildfly.security.credential.PasswordCredential;
import org.wildfly.security.password.Password;
import org.wildfly.security.password.PasswordFactory;
import org.wildfly.security.password.interfaces.BCryptPassword;
import org.wildfly.security.password.interfaces.BSDUnixDESCryptPassword;
import org.wildfly.security.password.interfaces.ClearPassword;
import org.wildfly.security.password.interfaces.DigestPassword;
import org.wildfly.security.password.interfaces.SaltedSimpleDigestPassword;
import org.wildfly.security.password.interfaces.ScramDigestPassword;
import org.wildfly.security.password.interfaces.SimpleDigestPassword;
import org.wildfly.security.password.interfaces.SunUnixMD5CryptPassword;
import org.wildfly.security.password.interfaces.UnixDESCryptPassword;
import org.wildfly.security.password.interfaces.UnixMD5CryptPassword;
import org.wildfly.security.password.spec.ClearPasswordSpec;
import org.wildfly.security.password.spec.HashPasswordSpec;
import org.wildfly.security.password.spec.IteratedSaltedHashPasswordSpec;
import org.wildfly.security.password.spec.SaltedHashPasswordSpec;
import org.wildfly.security.password.util.ModularCrypt;
import org.wildfly.security.util.CodePointIterator;

public class PasswordKeyMapper
implements KeyMapper {
    private final int hash;
    private final String algorithm;
    private int salt = -1;
    private int iterationCount = -1;
    private final String credentialName;
    private final Class<?> passwordType;

    public PasswordKeyMapper(String credentialName, int hash) throws InvalidKeyException {
        Assert.checkNotNullParam((String)"credentialName", (Object)credentialName);
        Assert.checkMinimumParameter((String)"hash", (int)1, (int)hash);
        this.algorithm = this.toAlgorithm(credentialName);
        this.passwordType = this.toPasswordType(this.algorithm);
        this.hash = hash;
        this.credentialName = credentialName;
    }

    public PasswordKeyMapper(String credentialName, int hash, int salt) throws InvalidKeyException {
        this(credentialName, hash);
        Assert.checkMinimumParameter((String)"salt", (int)1, (int)salt);
        this.salt = salt;
    }

    public PasswordKeyMapper(String credentialName, int hash, int salt, int iterationCount) throws InvalidKeyException {
        this(credentialName, hash, salt);
        Assert.checkMinimumParameter((String)"iterationCount", (int)1, (int)iterationCount);
        this.iterationCount = iterationCount;
    }

    public PasswordKeyMapper(String credentialName, String algorithm, int hash) throws InvalidKeyException {
        Assert.checkNotNullParam((String)"credentialName", (Object)credentialName);
        Assert.checkNotNullParam((String)"algorithm", (Object)algorithm);
        Assert.checkMinimumParameter((String)"hash", (int)1, (int)hash);
        this.algorithm = algorithm;
        this.passwordType = this.toPasswordType(algorithm);
        this.hash = hash;
        this.credentialName = credentialName;
    }

    public PasswordKeyMapper(String credentialName, String algorithm, int hash, int salt) throws InvalidKeyException {
        this(credentialName, algorithm, hash);
        Assert.checkMinimumParameter((String)"salt", (int)1, (int)salt);
        this.salt = salt;
    }

    public PasswordKeyMapper(String credentialName, String algorithm, int hash, int salt, int iterationCount) throws InvalidKeyException {
        this(credentialName, algorithm, hash, salt);
        Assert.checkMinimumParameter((String)"iterationCount", (int)1, (int)iterationCount);
        this.iterationCount = iterationCount;
    }

    @Override
    public String getCredentialName() {
        return this.credentialName;
    }

    @Override
    public SupportLevel getCredentialSupport(ResultSet resultSet) {
        try {
            Credential map = this.map(resultSet);
            if (map != null) {
                return SupportLevel.SUPPORTED;
            }
            return SupportLevel.UNSUPPORTED;
        }
        catch (SQLException cause) {
            throw ElytronMessages.log.couldNotObtainCredentialWithCause(cause);
        }
    }

    public String getAlgorithm() {
        return this.algorithm;
    }

    public int getHash() {
        return this.hash;
    }

    public int getSalt() {
        return this.salt;
    }

    public int getIterationCount() {
        return this.iterationCount;
    }

    @Override
    public Credential map(ResultSet resultSet) throws SQLException {
        byte[] hash = null;
        byte[] salt = null;
        int iterationCount = 0;
        if (resultSet.next()) {
            hash = this.toByteArray(resultSet.getObject(this.getHash()));
            if (this.getSalt() > 0) {
                salt = this.toByteArray(resultSet.getObject(this.getSalt()));
            }
            if (this.getIterationCount() > 0) {
                iterationCount = resultSet.getInt(this.getIterationCount());
            }
        }
        if (hash != null) {
            PasswordFactory passwordFactory = this.getPasswordFactory(this.getAlgorithm());
            try {
                if (ClearPassword.class.equals(this.passwordType)) {
                    return new PasswordCredential(this.toClearPassword(hash, passwordFactory));
                }
                if (BCryptPassword.class.equals(this.passwordType)) {
                    return new PasswordCredential(this.toBcryptPassword(hash, salt, iterationCount, passwordFactory));
                }
                if (SaltedSimpleDigestPassword.class.equals(this.passwordType)) {
                    return new PasswordCredential(this.toSaltedSimpleDigestPassword(hash, salt, passwordFactory));
                }
                if (SimpleDigestPassword.class.equals(this.passwordType)) {
                    return new PasswordCredential(this.toSimpleDigestPassword(hash, passwordFactory));
                }
                if (ScramDigestPassword.class.equals(this.passwordType)) {
                    return new PasswordCredential(this.toScramDigestPassword(hash, salt, iterationCount, passwordFactory));
                }
            }
            catch (InvalidKeyException | InvalidKeySpecException e) {
                throw ElytronMessages.log.invalidPasswordKeySpecificationForAlgorithm(this.algorithm, e);
            }
        }
        return null;
    }

    private Password toBcryptPassword(byte[] hash, byte[] salt, int iterationCount, PasswordFactory passwordFactory) throws InvalidKeyException, InvalidKeySpecException {
        if (salt == null) {
            return passwordFactory.translate(ModularCrypt.decode(this.toCharArray(hash)));
        }
        return passwordFactory.generatePassword(new IteratedSaltedHashPasswordSpec(hash, salt, iterationCount));
    }

    private Password toScramDigestPassword(byte[] hash, byte[] salt, int iterationCount, PasswordFactory passwordFactory) throws InvalidKeySpecException {
        if (salt == null) {
            throw ElytronMessages.log.saltIsExpectedWhenCreatingPasswords(ScramDigestPassword.class.getSimpleName());
        }
        IteratedSaltedHashPasswordSpec saltedSimpleDigestPasswordSpec = new IteratedSaltedHashPasswordSpec(hash, salt, iterationCount);
        return passwordFactory.generatePassword(saltedSimpleDigestPasswordSpec);
    }

    private Password toSimpleDigestPassword(byte[] hash, PasswordFactory passwordFactory) throws InvalidKeySpecException {
        HashPasswordSpec hashPasswordSpec = new HashPasswordSpec(hash);
        return passwordFactory.generatePassword(hashPasswordSpec);
    }

    private Password toSaltedSimpleDigestPassword(byte[] hash, byte[] salt, PasswordFactory passwordFactory) throws InvalidKeySpecException {
        if (salt == null) {
            throw ElytronMessages.log.saltIsExpectedWhenCreatingPasswords(SaltedSimpleDigestPassword.class.getSimpleName());
        }
        SaltedHashPasswordSpec saltedSimpleDigestPasswordSpec = new SaltedHashPasswordSpec(hash, salt);
        return passwordFactory.generatePassword(saltedSimpleDigestPasswordSpec);
    }

    private Password toClearPassword(byte[] hash, PasswordFactory passwordFactory) throws InvalidKeySpecException {
        return passwordFactory.generatePassword(new ClearPasswordSpec(this.toCharArray(hash)));
    }

    private PasswordFactory getPasswordFactory(String algorithm) {
        try {
            return PasswordFactory.getInstance(algorithm);
        }
        catch (NoSuchAlgorithmException e) {
            throw ElytronMessages.log.couldNotObtainPasswordFactoryForAlgorithm(algorithm, e);
        }
    }

    private byte[] toByteArray(Object value) {
        if (String.class.isInstance(value)) {
            return value.toString().getBytes(StandardCharsets.UTF_8);
        }
        if (byte[].class.isInstance(value)) {
            return (byte[])value;
        }
        return new byte[0];
    }

    private char[] toCharArray(byte[] hash) {
        return CodePointIterator.ofUtf8Bytes(hash).drainToString().toCharArray();
    }

    private String toAlgorithm(String credentialName) throws InvalidKeyException {
        if (credentialName.endsWith("clear")) {
            return "clear";
        }
        if (credentialName.endsWith("bcrypt")) {
            return "bcrypt";
        }
        if (credentialName.endsWith("crypt-md5")) {
            return "crypt-md5";
        }
        if (credentialName.endsWith("sun-crypt-md5")) {
            return "sun-crypt-md5";
        }
        if (credentialName.endsWith("sun-crypt-md5-bare-salt")) {
            return "sun-crypt-md5-bare-salt";
        }
        if (credentialName.endsWith("crypt-sha-256")) {
            return "crypt-sha-256";
        }
        if (credentialName.endsWith("crypt-sha-512")) {
            return "crypt-sha-512";
        }
        if (credentialName.endsWith("digest-md5")) {
            return "digest-md5";
        }
        if (credentialName.endsWith("digest-sha")) {
            return "digest-sha";
        }
        if (credentialName.endsWith("digest-sha-256")) {
            return "digest-sha-256";
        }
        if (credentialName.endsWith("digest-sha-512")) {
            return "digest-sha-512";
        }
        if (credentialName.endsWith("simple-digest-md2")) {
            return "simple-digest-md2";
        }
        if (credentialName.endsWith("simple-digest-md5")) {
            return "simple-digest-md5";
        }
        if (credentialName.endsWith("simple-digest-sha-1")) {
            return "simple-digest-sha-1";
        }
        if (credentialName.endsWith("simple-digest-sha-256")) {
            return "simple-digest-sha-256";
        }
        if (credentialName.endsWith("simple-digest-sha-384")) {
            return "simple-digest-sha-384";
        }
        if (credentialName.endsWith("simple-digest-sha-512")) {
            return "simple-digest-sha-512";
        }
        if (credentialName.endsWith("password-salt-digest-md5")) {
            return "password-salt-digest-md5";
        }
        if (credentialName.endsWith("password-salt-digest-sha-1")) {
            return "password-salt-digest-sha-1";
        }
        if (credentialName.endsWith("password-salt-digest-sha-256")) {
            return "password-salt-digest-sha-256";
        }
        if (credentialName.endsWith("password-salt-digest-sha-384")) {
            return "password-salt-digest-sha-384";
        }
        if (credentialName.endsWith("password-salt-digest-sha-512")) {
            return "password-salt-digest-sha-512";
        }
        if (credentialName.endsWith("salt-password-digest-md5")) {
            return "salt-password-digest-md5";
        }
        if (credentialName.endsWith("salt-password-digest-sha-1")) {
            return "salt-password-digest-sha-1";
        }
        if (credentialName.endsWith("salt-password-digest-sha-256")) {
            return "salt-password-digest-sha-256";
        }
        if (credentialName.endsWith("salt-password-digest-sha-384")) {
            return "salt-password-digest-sha-384";
        }
        if (credentialName.endsWith("salt-password-digest-sha-512")) {
            return "salt-password-digest-sha-512";
        }
        if (credentialName.endsWith("crypt-des")) {
            return "crypt-des";
        }
        if (credentialName.endsWith("bsd-crypt-des")) {
            return "bsd-crypt-des";
        }
        if (credentialName.endsWith("scram-sha-1")) {
            return "scram-sha-1";
        }
        if (credentialName.endsWith("scram-sha-256")) {
            return "scram-sha-256";
        }
        throw ElytronMessages.log.couldNotResolveAlgorithmByCredentialName(credentialName);
    }

    private Class<?> toPasswordType(String algorithm) throws InvalidKeyException {
        switch (algorithm) {
            case "clear": {
                return ClearPassword.class;
            }
            case "bcrypt": {
                return BCryptPassword.class;
            }
            case "crypt-md5": {
                return UnixMD5CryptPassword.class;
            }
            case "sun-crypt-md5": 
            case "sun-crypt-md5-bare-salt": {
                return SunUnixMD5CryptPassword.class;
            }
            case "crypt-sha-256": 
            case "crypt-sha-512": {
                return ClearPassword.class;
            }
            case "digest-md5": 
            case "digest-sha": 
            case "digest-sha-256": 
            case "digest-sha-512": {
                return DigestPassword.class;
            }
            case "simple-digest-md2": 
            case "simple-digest-md5": 
            case "simple-digest-sha-1": 
            case "simple-digest-sha-256": 
            case "simple-digest-sha-384": 
            case "simple-digest-sha-512": {
                return SimpleDigestPassword.class;
            }
            case "password-salt-digest-md5": 
            case "password-salt-digest-sha-1": 
            case "password-salt-digest-sha-256": 
            case "password-salt-digest-sha-384": 
            case "password-salt-digest-sha-512": 
            case "salt-password-digest-md5": 
            case "salt-password-digest-sha-1": 
            case "salt-password-digest-sha-256": 
            case "salt-password-digest-sha-384": 
            case "salt-password-digest-sha-512": {
                return SaltedSimpleDigestPassword.class;
            }
            case "crypt-des": {
                return UnixDESCryptPassword.class;
            }
            case "bsd-crypt-des": {
                return BSDUnixDESCryptPassword.class;
            }
            case "scram-sha-1": 
            case "scram-sha-256": {
                return ScramDigestPassword.class;
            }
        }
        throw ElytronMessages.log.unknownPasswordTypeOrAlgorithm(algorithm);
    }
}

