/*
 * Licensed to the University Corporation for Advanced Internet Development, 
 * Inc. (UCAID) under one or more contributor license agreements.  See the 
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache 
 * License, Version 2.0 (the "License"); you may not use this file except in 
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.saml.saml1.core.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.util.XMLObjectChildrenList;
import org.opensaml.saml.saml1.core.Assertion;
import org.opensaml.saml.saml1.core.Response;
import org.opensaml.saml.saml1.core.Status;

/**
 * Implementation of the {@link org.opensaml.saml.saml1.core.Response} Object.
 */
public class ResponseImpl extends ResponseAbstractTypeImpl implements Response {

    /** Status associated with this element. */
    private Status status;

    /** List of all the Assertions. */
    private final XMLObjectChildrenList<Assertion> assertions;

    /**
     * Constructor.
     * 
     * @param namespaceURI the namespace the element is in
     * @param elementLocalName the local name of the XML element this Object represents
     * @param namespacePrefix the prefix for the given namespace
     */
    protected ResponseImpl(String namespaceURI, String elementLocalName, String namespacePrefix) {
        super(namespaceURI, elementLocalName, namespacePrefix);
        assertions = new XMLObjectChildrenList<Assertion>(this);
    }

    /** {@inheritDoc} */
    public List<Assertion> getAssertions() {
        return assertions;
    }

    /** {@inheritDoc} */
    public Status getStatus() {
        return status;
    }

    /** {@inheritDoc} */
    public void setStatus(Status s) {
        status = prepareForAssignment(status, s);
    }

    /** {@inheritDoc} */
    public List<XMLObject> getOrderedChildren() {
        ArrayList<XMLObject> children = new ArrayList<XMLObject>(1 + assertions.size());

        if (super.getOrderedChildren() != null) {
            children.addAll(super.getOrderedChildren());
        }

        if (status != null) {
            children.add(status);
        }

        children.addAll(assertions);

        if (children.size() == 0) {
            return null;
        }

        return Collections.unmodifiableList(children);
    }
}