package org.wildfly.swarm.config.infinispan.cache_container;

import org.wildfly.swarm.config.runtime.Addresses;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Map;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.infinispan.cache_container.PropertyConsumer;
import org.wildfly.swarm.config.infinispan.cache_container.PropertySupplier;
import org.wildfly.swarm.config.infinispan.cache_container.Property;
import org.wildfly.swarm.config.infinispan.cache_container.ThroughWriteConsumer;
import org.wildfly.swarm.config.infinispan.cache_container.ThroughWriteSupplier;
import org.wildfly.swarm.config.infinispan.cache_container.BehindWriteConsumer;
import org.wildfly.swarm.config.infinispan.cache_container.BehindWriteSupplier;
import org.wildfly.swarm.config.infinispan.cache_container.StringTableConsumer;
import org.wildfly.swarm.config.infinispan.cache_container.StringTableSupplier;
import org.wildfly.swarm.config.infinispan.cache_container.BinaryTableConsumer;
import org.wildfly.swarm.config.infinispan.cache_container.BinaryTableSupplier;
/**
 * The cache JDBC store configuration.
 */
@Addresses({
		"/subsystem=infinispan/cache-container=*/invalidation-cache=*/store=mixed-jdbc",
		"/subsystem=infinispan/cache-container=*/local-cache=*/store=mixed-jdbc",
		"/subsystem=infinispan/cache-container=*/replicated-cache=*/store=mixed-jdbc",
		"/subsystem=infinispan/cache-container=*/distributed-cache=*/store=mixed-jdbc"})
@ResourceType("store")
@Implicit
public class MixedJDBCStore<T extends MixedJDBCStore<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Long cacheLoaderLoads;
	private Long cacheLoaderMisses;
	private String dataSource;
	private String dialect;
	private Boolean fetchState;
	private Boolean passivation;
	private Boolean preload;
	private Map properties;
	private Boolean purge;
	private Boolean shared;
	private Boolean singleton;
	private MixedJDBCStoreResources subresources = new MixedJDBCStoreResources();

	public MixedJDBCStore() {
		this.key = "mixed-jdbc";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The number of cache loader node loads. May return null if the cache is
	 * not started.
	 */
	@ModelNodeBinding(detypedName = "cache-loader-loads")
	public Long cacheLoaderLoads() {
		return this.cacheLoaderLoads;
	}

	/**
	 * The number of cache loader node loads. May return null if the cache is
	 * not started.
	 */
	@SuppressWarnings("unchecked")
	public T cacheLoaderLoads(Long value) {
		Object oldValue = this.cacheLoaderLoads;
		this.cacheLoaderLoads = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("cacheLoaderLoads", oldValue, value);
		return (T) this;
	}

	/**
	 * The number of cache loader node misses. May return null if the cache is
	 * not started.
	 */
	@ModelNodeBinding(detypedName = "cache-loader-misses")
	public Long cacheLoaderMisses() {
		return this.cacheLoaderMisses;
	}

	/**
	 * The number of cache loader node misses. May return null if the cache is
	 * not started.
	 */
	@SuppressWarnings("unchecked")
	public T cacheLoaderMisses(Long value) {
		Object oldValue = this.cacheLoaderMisses;
		this.cacheLoaderMisses = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("cacheLoaderMisses", oldValue, value);
		return (T) this;
	}

	/**
	 * References the data source used to connect to this store.
	 */
	@ModelNodeBinding(detypedName = "data-source")
	public String dataSource() {
		return this.dataSource;
	}

	/**
	 * References the data source used to connect to this store.
	 */
	@SuppressWarnings("unchecked")
	public T dataSource(String value) {
		Object oldValue = this.dataSource;
		this.dataSource = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("dataSource", oldValue, value);
		return (T) this;
	}

	/**
	 * The dialect of this datastore.
	 */
	@ModelNodeBinding(detypedName = "dialect")
	public String dialect() {
		return this.dialect;
	}

	/**
	 * The dialect of this datastore.
	 */
	@SuppressWarnings("unchecked")
	public T dialect(String value) {
		Object oldValue = this.dialect;
		this.dialect = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("dialect", oldValue, value);
		return (T) this;
	}

	/**
	 * If true, fetch persistent state when joining a cluster. If multiple cache
	 * stores are chained, only one of them can have this property enabled.
	 */
	@ModelNodeBinding(detypedName = "fetch-state")
	public Boolean fetchState() {
		return this.fetchState;
	}

	/**
	 * If true, fetch persistent state when joining a cluster. If multiple cache
	 * stores are chained, only one of them can have this property enabled.
	 */
	@SuppressWarnings("unchecked")
	public T fetchState(Boolean value) {
		Object oldValue = this.fetchState;
		this.fetchState = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("fetchState", oldValue, value);
		return (T) this;
	}

	/**
	 * If true, data is only written to the cache store when it is evicted from
	 * memory, a phenomenon known as 'passivation'. Next time the data is
	 * requested, it will be 'activated' which means that data will be brought
	 * back to memory and removed from the persistent store. f false, the cache
	 * store contains a copy of the contents in memory, so writes to cache
	 * result in cache store writes. This essentially gives you a
	 * 'write-through' configuration.
	 */
	@ModelNodeBinding(detypedName = "passivation")
	public Boolean passivation() {
		return this.passivation;
	}

	/**
	 * If true, data is only written to the cache store when it is evicted from
	 * memory, a phenomenon known as 'passivation'. Next time the data is
	 * requested, it will be 'activated' which means that data will be brought
	 * back to memory and removed from the persistent store. f false, the cache
	 * store contains a copy of the contents in memory, so writes to cache
	 * result in cache store writes. This essentially gives you a
	 * 'write-through' configuration.
	 */
	@SuppressWarnings("unchecked")
	public T passivation(Boolean value) {
		Object oldValue = this.passivation;
		this.passivation = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("passivation", oldValue, value);
		return (T) this;
	}

	/**
	 * If true, when the cache starts, data stored in the cache store will be
	 * pre-loaded into memory. This is particularly useful when data in the
	 * cache store will be needed immediately after startup and you want to
	 * avoid cache operations being delayed as a result of loading this data
	 * lazily. Can be used to provide a 'warm-cache' on startup, however there
	 * is a performance penalty as startup time is affected by this process.
	 */
	@ModelNodeBinding(detypedName = "preload")
	public Boolean preload() {
		return this.preload;
	}

	/**
	 * If true, when the cache starts, data stored in the cache store will be
	 * pre-loaded into memory. This is particularly useful when data in the
	 * cache store will be needed immediately after startup and you want to
	 * avoid cache operations being delayed as a result of loading this data
	 * lazily. Can be used to provide a 'warm-cache' on startup, however there
	 * is a performance penalty as startup time is affected by this process.
	 */
	@SuppressWarnings("unchecked")
	public T preload(Boolean value) {
		Object oldValue = this.preload;
		this.preload = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("preload", oldValue, value);
		return (T) this;
	}

	/**
	 * A list of cache store properties.
	 */
	@ModelNodeBinding(detypedName = "properties")
	public Map properties() {
		return this.properties;
	}

	/**
	 * A list of cache store properties.
	 */
	@SuppressWarnings("unchecked")
	public T properties(Map value) {
		Object oldValue = this.properties;
		this.properties = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("properties", oldValue, value);
		return (T) this;
	}

	/**
	 * A list of cache store properties.
	 */
	public T property(String key, Object value) {
		if (this.properties == null) {
			this.properties = new java.util.HashMap<>();
		}
		this.properties.put(key, value);
		return (T) this;
	}

	/**
	 * If true, purges this cache store when it starts up.
	 */
	@ModelNodeBinding(detypedName = "purge")
	public Boolean purge() {
		return this.purge;
	}

	/**
	 * If true, purges this cache store when it starts up.
	 */
	@SuppressWarnings("unchecked")
	public T purge(Boolean value) {
		Object oldValue = this.purge;
		this.purge = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("purge", oldValue, value);
		return (T) this;
	}

	/**
	 * This setting should be set to true when multiple cache instances share
	 * the same cache store (e.g., multiple nodes in a cluster using a
	 * JDBC-based CacheStore pointing to the same, shared database.) Setting
	 * this to true avoids multiple cache instances writing the same
	 * modification multiple times. If enabled, only the node where the
	 * modification originated will write to the cache store. If disabled, each
	 * individual cache reacts to a potential remote update by storing the data
	 * to the cache store.
	 */
	@ModelNodeBinding(detypedName = "shared")
	public Boolean shared() {
		return this.shared;
	}

	/**
	 * This setting should be set to true when multiple cache instances share
	 * the same cache store (e.g., multiple nodes in a cluster using a
	 * JDBC-based CacheStore pointing to the same, shared database.) Setting
	 * this to true avoids multiple cache instances writing the same
	 * modification multiple times. If enabled, only the node where the
	 * modification originated will write to the cache store. If disabled, each
	 * individual cache reacts to a potential remote update by storing the data
	 * to the cache store.
	 */
	@SuppressWarnings("unchecked")
	public T shared(Boolean value) {
		Object oldValue = this.shared;
		this.shared = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("shared", oldValue, value);
		return (T) this;
	}

	/**
	 * If true, the singleton store cache store is enabled. SingletonStore is a
	 * delegating cache store used for situations when only one instance in a
	 * cluster should interact with the underlying store.
	 */
	@ModelNodeBinding(detypedName = "singleton")
	public Boolean singleton() {
		return this.singleton;
	}

	/**
	 * If true, the singleton store cache store is enabled. SingletonStore is a
	 * delegating cache store used for situations when only one instance in a
	 * cluster should interact with the underlying store.
	 */
	@SuppressWarnings("unchecked")
	public T singleton(Boolean value) {
		Object oldValue = this.singleton;
		this.singleton = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("singleton", oldValue, value);
		return (T) this;
	}

	public MixedJDBCStoreResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Property objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Property objects.
	 */
	@SuppressWarnings("unchecked")
	public T properties(List<Property> value) {
		this.subresources.properties = value;
		return (T) this;
	}

	/**
	 * Add the Property object to the list of subresources
	 * 
	 * @param value
	 *            The Property to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(Property value) {
		this.subresources.properties.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Property object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Property resource
	 * @param config
	 *            The PropertyConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(String childKey, PropertyConsumer consumer) {
		Property<?> child = new Property<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		property(child);
		return (T) this;
	}

	/**
	 * Create and configure a Property object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Property resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(String childKey) {
		property(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Property object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T property(PropertySupplier supplier) {
		property(supplier.get());
		return (T) this;
	}

	/**
	 * Configures a cache store as write-through.
	 */
	@SuppressWarnings("unchecked")
	public T throughWrite(ThroughWrite value) {
		this.subresources.throughWrite = value;
		return (T) this;
	}

	/**
	 * Configures a cache store as write-through.
	 */
	@SuppressWarnings("unchecked")
	public T throughWrite(ThroughWriteConsumer consumer) {
		ThroughWrite<?> child = new ThroughWrite<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.throughWrite = child;
		return (T) this;
	}

	/**
	 * Configures a cache store as write-through.
	 */
	@SuppressWarnings("unchecked")
	public T throughWrite() {
		ThroughWrite<?> child = new ThroughWrite<>();
		this.subresources.throughWrite = child;
		return (T) this;
	}

	/**
	 * Configures a cache store as write-through.
	 */
	@SuppressWarnings("unchecked")
	public T throughWrite(ThroughWriteSupplier supplier) {
		this.subresources.throughWrite = supplier.get();
		return (T) this;
	}

	/**
	 * Configures a cache store as write-behind instead of write-through.
	 */
	@SuppressWarnings("unchecked")
	public T behindWrite(BehindWrite value) {
		this.subresources.behindWrite = value;
		return (T) this;
	}

	/**
	 * Configures a cache store as write-behind instead of write-through.
	 */
	@SuppressWarnings("unchecked")
	public T behindWrite(BehindWriteConsumer consumer) {
		BehindWrite<?> child = new BehindWrite<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.behindWrite = child;
		return (T) this;
	}

	/**
	 * Configures a cache store as write-behind instead of write-through.
	 */
	@SuppressWarnings("unchecked")
	public T behindWrite() {
		BehindWrite<?> child = new BehindWrite<>();
		this.subresources.behindWrite = child;
		return (T) this;
	}

	/**
	 * Configures a cache store as write-behind instead of write-through.
	 */
	@SuppressWarnings("unchecked")
	public T behindWrite(BehindWriteSupplier supplier) {
		this.subresources.behindWrite = supplier.get();
		return (T) this;
	}

	/**
	 * Defines a table used to store cache entries whose keys can be expressed
	 * as strings.
	 */
	@SuppressWarnings("unchecked")
	public T stringTable(StringTable value) {
		this.subresources.stringTable = value;
		return (T) this;
	}

	/**
	 * Defines a table used to store cache entries whose keys can be expressed
	 * as strings.
	 */
	@SuppressWarnings("unchecked")
	public T stringTable(StringTableConsumer consumer) {
		StringTable<?> child = new StringTable<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.stringTable = child;
		return (T) this;
	}

	/**
	 * Defines a table used to store cache entries whose keys can be expressed
	 * as strings.
	 */
	@SuppressWarnings("unchecked")
	public T stringTable() {
		StringTable<?> child = new StringTable<>();
		this.subresources.stringTable = child;
		return (T) this;
	}

	/**
	 * Defines a table used to store cache entries whose keys can be expressed
	 * as strings.
	 */
	@SuppressWarnings("unchecked")
	public T stringTable(StringTableSupplier supplier) {
		this.subresources.stringTable = supplier.get();
		return (T) this;
	}

	/**
	 * Defines a table used to store cache entries whose keys cannot be
	 * expressed as strings.
	 */
	@SuppressWarnings("unchecked")
	public T binaryTable(BinaryTable value) {
		this.subresources.binaryTable = value;
		return (T) this;
	}

	/**
	 * Defines a table used to store cache entries whose keys cannot be
	 * expressed as strings.
	 */
	@SuppressWarnings("unchecked")
	public T binaryTable(BinaryTableConsumer consumer) {
		BinaryTable<?> child = new BinaryTable<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.binaryTable = child;
		return (T) this;
	}

	/**
	 * Defines a table used to store cache entries whose keys cannot be
	 * expressed as strings.
	 */
	@SuppressWarnings("unchecked")
	public T binaryTable() {
		BinaryTable<?> child = new BinaryTable<>();
		this.subresources.binaryTable = child;
		return (T) this;
	}

	/**
	 * Defines a table used to store cache entries whose keys cannot be
	 * expressed as strings.
	 */
	@SuppressWarnings("unchecked")
	public T binaryTable(BinaryTableSupplier supplier) {
		this.subresources.binaryTable = supplier.get();
		return (T) this;
	}

	/**
	 * Child mutators for MixedJDBCStore
	 */
	public static class MixedJDBCStoreResources {
		/**
		 * A cache store property with name and value.
		 */
		private List<Property> properties = new java.util.ArrayList<>();
		private ThroughWrite throughWrite;
		private BehindWrite behindWrite;
		private StringTable stringTable;
		private BinaryTable binaryTable;

		/**
		 * Get the list of Property resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Property> properties() {
			return this.properties;
		}

		/**
		 * Configures a cache store as write-through.
		 */
		@Subresource
		public ThroughWrite throughWrite() {
			return this.throughWrite;
		}

		/**
		 * Configures a cache store as write-behind instead of write-through.
		 */
		@Subresource
		public BehindWrite behindWrite() {
			return this.behindWrite;
		}

		/**
		 * Defines a table used to store cache entries whose keys can be
		 * expressed as strings.
		 */
		@Subresource
		public StringTable stringTable() {
			return this.stringTable;
		}

		/**
		 * Defines a table used to store cache entries whose keys cannot be
		 * expressed as strings.
		 */
		@Subresource
		public BinaryTable binaryTable() {
			return this.binaryTable;
		}
	}
}