package org.wildfly.swarm.config.messaging_activemq.server.ha_policy.shared_store_colocated;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * A messaging resource that allows you to configure High Availability for the
 * ActiveMQ server (the value of ha-policy can be live-only, replication-master,
 * replication-slave, or replication-colocated).
 */
@Address("/subsystem=messaging-activemq/server=*/ha-policy=shared-store-colocated/configuration=master")
@ResourceType("configuration")
@Implicit
public class MasterConfiguration<T extends MasterConfiguration<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Long failbackDelay;
	private Boolean failoverOnServerShutdown;

	public MasterConfiguration() {
		this.key = "master";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * delay to wait before failback occurs on (live's) restart
	 */
	@ModelNodeBinding(detypedName = "failback-delay")
	public Long failbackDelay() {
		return this.failbackDelay;
	}

	/**
	 * delay to wait before failback occurs on (live's) restart
	 */
	@SuppressWarnings("unchecked")
	public T failbackDelay(Long value) {
		Object oldValue = this.failbackDelay;
		this.failbackDelay = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("failbackDelay", oldValue, value);
		return (T) this;
	}

	/**
	 * Whether the server must failover when it is normally shutdown.
	 */
	@ModelNodeBinding(detypedName = "failover-on-server-shutdown")
	public Boolean failoverOnServerShutdown() {
		return this.failoverOnServerShutdown;
	}

	/**
	 * Whether the server must failover when it is normally shutdown.
	 */
	@SuppressWarnings("unchecked")
	public T failoverOnServerShutdown(Boolean value) {
		Object oldValue = this.failoverOnServerShutdown;
		this.failoverOnServerShutdown = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("failoverOnServerShutdown", oldValue,
					value);
		return (T) this;
	}
}