package org.wildfly.swarm.config.security.security_domain.authentication;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.security.security_domain.authentication.LoginModuleConsumer;
import org.wildfly.swarm.config.security.security_domain.authentication.LoginModuleSupplier;
import org.wildfly.swarm.config.security.security_domain.authentication.LoginModule;
/**
 * List of "named" login modules that are used by jaspi authentication modules.
 */
@Address("/subsystem=security/security-domain=*/authentication=jaspi/login-module-stack=*")
@ResourceType("login-module-stack")
public class LoginModuleStack<T extends LoginModuleStack<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private LoginModuleStackResources subresources = new LoginModuleStackResources();

	public LoginModuleStack(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public LoginModuleStackResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all LoginModule objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of LoginModule objects.
	 */
	@SuppressWarnings("unchecked")
	public T loginModules(List<LoginModule> value) {
		this.subresources.loginModules = value;
		return (T) this;
	}

	/**
	 * Add the LoginModule object to the list of subresources
	 * 
	 * @param value
	 *            The LoginModule to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T loginModule(LoginModule value) {
		this.subresources.loginModules.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a LoginModule object to the list of subresources
	 * 
	 * @param key
	 *            The key for the LoginModule resource
	 * @param config
	 *            The LoginModuleConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T loginModule(String childKey, LoginModuleConsumer consumer) {
		LoginModule<?> child = new LoginModule<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		loginModule(child);
		return (T) this;
	}

	/**
	 * Create and configure a LoginModule object to the list of subresources
	 * 
	 * @param key
	 *            The key for the LoginModule resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T loginModule(String childKey) {
		loginModule(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied LoginModule object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T loginModule(LoginModuleSupplier supplier) {
		loginModule(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for LoginModuleStack
	 */
	public static class LoginModuleStackResources {
		/**
		 * List of authentication modules
		 */
		private List<LoginModule> loginModules = new java.util.ArrayList<>();

		/**
		 * Get the list of LoginModule resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<LoginModule> loginModules() {
			return this.loginModules;
		}
	}
}