package org.wildfly.swarm.config.infinispan.cache_container;

import org.wildfly.swarm.config.runtime.Addresses;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * A backup site to which to replicate this cache.
 */
@Addresses({
		"/subsystem=infinispan/cache-container=*/replicated-cache=*/component=backups/backup=*",
		"/subsystem=infinispan/cache-container=*/distributed-cache=*/component=backups/backup=*"})
@ResourceType("backup")
public class Backup<T extends Backup<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Integer afterFailures;
	private Boolean enabled;
	private String failurePolicy;
	private Long minWait;
	private String strategy;
	private Long timeout;

	public Backup(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Indicates the number of failures after which this backup site should go
	 * offline.
	 */
	@ModelNodeBinding(detypedName = "after-failures")
	public Integer afterFailures() {
		return this.afterFailures;
	}

	/**
	 * Indicates the number of failures after which this backup site should go
	 * offline.
	 */
	@SuppressWarnings("unchecked")
	public T afterFailures(Integer value) {
		Object oldValue = this.afterFailures;
		this.afterFailures = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("afterFailures", oldValue, value);
		return (T) this;
	}

	/**
	 * Indicates whether or not this backup site is enabled.
	 */
	@ModelNodeBinding(detypedName = "enabled")
	public Boolean enabled() {
		return this.enabled;
	}

	/**
	 * Indicates whether or not this backup site is enabled.
	 */
	@SuppressWarnings("unchecked")
	public T enabled(Boolean value) {
		Object oldValue = this.enabled;
		this.enabled = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("enabled", oldValue, value);
		return (T) this;
	}

	/**
	 * The policy to follow when connectivity to the backup site fails.
	 */
	@ModelNodeBinding(detypedName = "failure-policy")
	public String failurePolicy() {
		return this.failurePolicy;
	}

	/**
	 * The policy to follow when connectivity to the backup site fails.
	 */
	@SuppressWarnings("unchecked")
	public T failurePolicy(String value) {
		Object oldValue = this.failurePolicy;
		this.failurePolicy = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("failurePolicy", oldValue, value);
		return (T) this;
	}

	/**
	 * Indicates the minimum time (in milliseconds) to wait after the max number
	 * of failures is reached, after which this backup site should go offline.
	 */
	@ModelNodeBinding(detypedName = "min-wait")
	public Long minWait() {
		return this.minWait;
	}

	/**
	 * Indicates the minimum time (in milliseconds) to wait after the max number
	 * of failures is reached, after which this backup site should go offline.
	 */
	@SuppressWarnings("unchecked")
	public T minWait(Long value) {
		Object oldValue = this.minWait;
		this.minWait = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("minWait", oldValue, value);
		return (T) this;
	}

	/**
	 * The backup strategy for this cache
	 */
	@ModelNodeBinding(detypedName = "strategy")
	public String strategy() {
		return this.strategy;
	}

	/**
	 * The backup strategy for this cache
	 */
	@SuppressWarnings("unchecked")
	public T strategy(String value) {
		Object oldValue = this.strategy;
		this.strategy = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("strategy", oldValue, value);
		return (T) this;
	}

	/**
	 * The timeout for replicating to the backup site.
	 */
	@ModelNodeBinding(detypedName = "timeout")
	public Long timeout() {
		return this.timeout;
	}

	/**
	 * The timeout for replicating to the backup site.
	 */
	@SuppressWarnings("unchecked")
	public T timeout(Long value) {
		Object oldValue = this.timeout;
		this.timeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("timeout", oldValue, value);
		return (T) this;
	}
}